C***********************************************************************
C                 copyright 2001, Amoco Production Company             *
C                             All Rights Reserved                      *
C                     an affiliate of BP America Inc.                  *
C***********************************************************************
c -----------------  Main Routine -----------------------

c     Program Changes:

c      - original written: April/95
c      - added least squares fitter: August/95

c     Program Description:

c***********************************************************************
c    llattrib3d -- A USP program to calculate attributes from angst3d data
c
c    Execute "llattrib3d -h" for self documentation.
c    Execute "llattrib3d -ha" for list of available recon attributes
c
c    attrib reads in USP datasets created by program "angst3d", and
c    outputs a USP dataset containing traces derived from user selected
c    attributed calculations.
c
c
c    This code was written by W.D. Woodruff based on the USP
c    code template for 2d processing, April 1995
c
c    NOTE:
c    afp, the "Amoco Fortran Preprocessor", is required to compile
c    this module.
c
c
c***********************************************************************

c get machine dependent parameters 

#include <f77/iounit.h>
#include <f77/lhdrsz.h>
#include <f77/sisdef.h>
#include <save_defs.h> 

c dimension standard USP variables 

      integer     itr ( SZLNHD )
     
      integer     nsamp, nsi, ntrc, nrec, nreco, iform, obytes
      integer     luin(70), luout, lbytes, nbytes, lbyout
      integer     irs, ire, ns, ne, argis

      real        dtsec
      real        tri ( SZLNHD )

      character   ntap(70)*255, atap*255, otap*255, name*10

      logical     verbos, query, dbug

c Program Specific 

      integer     luang, pdefaults
      integer     luparm, stakn, nxcor
      integer     attrnum, numangles, err1, nu
      integer     nptraces, ershifts
      integer     maxvcnt
      integer     deltat, totdt, tzero
      integer ifmt_RecNum,l_RecNum,ln_RecNum, RecNum
      integer ifmt_TrcNum,l_TrcNum,ln_TrcNum, TrcNum
      integer ifmt_DstSgn,l_DstSgn,ln_DstSgn, DstSgn
      integer ifmt_StaCor,l_StaCor,ln_StaCor, StaCor

      real    amn(63), amx(63)
      real    xdot, err2
      real    maxval

c zero crossing information arrays
      integer     zxary(SZLNHD), zxarycnt

c record-trace array
      real    rtrpt(SZLNHD,63)

c xcorrelation stacked trace array
      real    trstk(SZLNHD)

      character   ptap*255
      logical     OK, stkdone, posflg

c Initialize variables

      data name/"LLATTRIB3D"/
      stkcnt = 0
      ershifts = 0
      nptraces = 0
      pdefaults = 0
      err2 = 0.
      OK = .false.
      stkdone = .false.

c give command line help if requested

      query = ( argis ( '-?' ) .gt. 0 .or. argis ( '-h' ) .gt. 0 )
      if ( query )then
         call help()
         stop
      endif

      if ( argis ( '-ha' ) .gt. 0 )then
            call helpatt()
            stop
      endif

c open printout file

#include <f77/open.h>

c get command line input parameters

      call argstr( '-O', otap, ' ', ' ' )

      nu = 0
      do  i = 1, 70
          nu = nu + 1
          call argstr( '-N', ntap(i), ' ', ' ' )
          if (ntap(i) .eq. ' ') go to 1
      enddo
1     continue
      nu = nu -1
      if (nu .eq. 0) then
         write(LER,*)'Fatal Error in llattrib3d:'
         write(LER,*)'Must have at least 1 input file name'
         write(LER,*)'using -N[]'
         stop
      endif
 
      call argstr( '-A', atap, ' ', ' ' )
      if (atap(1:1) .eq. ' ') then
         write(LER,*)' '
         write(LER,*)'Fatal Error in llattrib3d:'
         write(LER,*)'Must supply angle cards using -A[]'
         stop
      endif
 
      call argstr( '-P', ptap, ' ', ' ' )
      if (ptap(1:1) .eq. ' ') then
         write(LER,*)' '
         write(LER,*)'No parameter file supplied.'
         write(LER,*)'Using the following default paramters:'
         write(LER,*)'  Number of traces in xcor stack P = 1'
         write(LER,*)'  Delta T (trace shift) time = 12msec'
         write(LER,*)'  Total portion of trace to xcor = 30msec'
         write(LER,*)'  Number of cross correlations = 3'
         write(LER,*)'  Start offset time for xcor = 0'
	 pdefaults = 1
      endif
 
      call argi4 ( '-rs', irs, 0 , 0  )
      call argi4 ( '-re', ire, 0 , 0  )
      call argi4 ( '-ns', ns, 0 , 0  )
      call argi4 ( '-ne', ne, 0 , 0  )
      call argi4 ( '-att', attrnum, 0 , 0  )
      verbos = (argis('-V') .gt. 0)
      dbug = (argis('-DB') .gt. 0)
 
c open input and output files

      do i = 1, nu
      	 call getln(luin(i), ntap(i),'r', 0)
      enddo

      call getln(luout, otap,'w', 1)
      call getln(luang, atap,'r', 0)

c  read input line header and save certain parameters
c       only the line header from the first file is used

      do i = nu, 1, -1
         call rtape(luin(i),itr,lbytes)
      enddo
      if(lbytes.eq.0)then
         write(LER,*)'llattrib3d: no header read from unit ',luin(1)
         write(LER,*)'FATAL'
         stop
      endif

c  read in angle cards
      err1 = 0
      open (unit=luang, file=atap, status='old', iostat=err1)
      if (err1 .ne. 0) then
	write(LER,*)'llattrib3d: Could not open'
	write(LER,*)'angle card file ',atap
        write(LER,*)'FATAL'
	goto 999
      endif

      err1 = 0
      call rdangs(amn, amx, numangles, luang, err1)
      if(err1 .ne. 0) then
	 goto 999
      endif

      if(nu .ne. numangles) then
         write(LER,*)'llattrib3d: number of angle pairs does not '
         write(LER,*)'equal number of specified input files.'
         write(LER,*)'FATAL'
	 goto 999
      endif

c  verify file exists, then read in xcor parameter cards

      err1 = 0
      inquire(file=ptap,exist=OK)

      if( OK ) then
        call getln(luparm, ptap,'r', 0)
        open (unit=luparm, file=ptap, status='old', iostat=err1)
      endif

      if ((err1 .ne. 0) .or. (.not. OK)) then
	write(LER,*)'llattrib3d: Could not open parameter file',ptap
        write(LER,*)'Using the following default paramters:'
        write(LER,*)'  Number of traces in xcor stack (stakn) = 1'
        write(LER,*)'  Delta T (trace shift) time (deltat) = 12msec'
        write(LER,*)'  Total portion of trace to xcor (totdt) = 30msec'
        write(LER,*)'  Number of cross correlations (nxcor) = 3'
        write(LER,*)'  Start offset time for xcor (tzero) = 0'
	pdefaults = 1
        if( OK ) call lbclos ( luparm )
      endif

      if (pdefaults .ne. 1) then
        err1 = 0
	call rparms(stakn,deltat,totdt,nxcor,tzero,luparm,err1)
        call lbclos ( luparm )
      endif

      if (err1 .ne. 0) then
	write(LER,*)'llattrib3d: Error reading parameter file ',ptap
        write(LER,*)'Using the following default parameters:'
        write(LER,*)'  Number of traces in xcor stack (stakn) = 1'
        write(LER,*)'  Delta T (trace shift) time (deltat) = 12msec'
        write(LER,*)'  Total portion of trace to xcor (totdt) = 30msec'
        write(LER,*)'  Number of cross correlations (nxcor) = 3'
        write(LER,*)'  Start offset time for xcor (tzero) = 0'
	pdefaults = 1
      endif

c set xcor defaults if unable to get user defined parameters
c  (also set in rparms subroutine on out of limits parameters)
      if (pdefaults .eq. 1) then
	 stakn = 1
	 deltat = 12
	 totdt = 30
	 nxcor = 3
	 tzero = 0
      endif

      if (stakn .gt. nu) then
	write(LER,*)'llattrib3d: Value for stakn greater than '
        write(LER,*)'number of input angle stacks. Using stakn = ',nu
        stakn = nu
      endif

      call saver(itr, 'NumSmp', nsamp, LINHED)
      call saver(itr, 'SmpInt', nsi  , LINHED)
      call saver(itr, 'NumTrc', ntrc , LINHED)
      call saver(itr, 'NumRec', nrec , LINHED)
      call saver(itr, 'Format', iform, LINHED)

c define pointers to header words required by your routine

      call savelu('RecNum',ifmt_RecNum,l_RecNum,ln_RecNum,TRACEHEADER)
      call savelu('TrcNum',ifmt_TrcNum,l_TrcNum,ln_TrcNum,TRACEHEADER)
      call savelu('DstSgn',ifmt_DstSgn,l_DstSgn,ln_DstSgn,TRACEHEADER)
      call savelu('StaCor',ifmt_StaCor,l_StaCor,ln_StaCor,TRACEHEADER)

c update historical line header and print to printout file 

      call hlhprt (itr, lbytes, name, 4, LERR)

c handle millisecond sample interval if present

      if ( nsi .gt. 32 ) then
         dtmsec = .001 * float(nsi)
         dtsec = real(nsi)/1000000.
      else
         dtmsec = float(nsi)
         dtsec = real(nsi)/1000.
      endif

c check user supplied boundary conditions and set defaults

      if ( irs .eq. 0 ) irs = 1
      if ( ire .eq. 0 .or. ire .gt. nrec ) ire = nrec

      if ( ns .eq. 0 ) ns = 1
      if ( ne .eq. 0 .or. ne .gt. ntrc ) ne = ntrc

      nreco = ire - irs + 1
      ntrco = ne - ns + 1

c modify line header to reflect actual record configuration output
c NOTE: in this case the sample limits ist and iend are used to 
c       limit processing only.   All data samples are actually passed.

      call savew(itr, 'NumRec', nreco, LINHED)
      call savew(itr, 'NumTrc', ntrco, LINHED)

c number output bytes

      obytes = SZTRHD + SZSMPD * nsamp 

c save out hlh and line header

      call savhlh  ( itr, lbytes, lbyout )
      call wrtape ( luout, itr, lbyout )

c verbose output of all pertinent information before processing begins

      if ( verbos ) then
        call verbal( nsamp, nsi, ntrc, nrec, iform, attrnum,
     1    atap, ntap, nu, otap, irs, ire, ns, ne, amn, amx, numangles,
     2    ptap,stakn,deltat,totdt,nxcor,tzero)
      endif

c BEGIN PROCESSING 

c initialize xcor stack memory
      	call vclr ( trstk, 1, nsamp )

      if ( dbug ) then
        write(LER,*)'Using the following parameters:'
        write(LER,*)'  Number of traces in xcor stack (stakn) = ',stakn
        write(LER,*)'  Delta T (trace shift) time (deltat) = ',deltat
        write(LER,*)'  Total portion of trace to xcor (totdt) = ',totdt
        write(LER,*)'  Number of cross correlations (nxcor) = ',nxcor
        write(LER,*)'  Start offset time for xcor (tzero) = ',tzero
      endif

c re-entry point after cross correlation stacks are built
 100  continue

c initialize memory
         
      do i = 1, nu
      	call vclr ( rtrpt(1,i), 1, nsamp )
      enddo


c skip unwanted input records

      do i = 1, nu
      	call recskp ( 1, irs-1, luin(i), ntrc, itr )
      enddo

c JJ = record counter, KK = trace counter,
c   II = input file counter

c do records
      DO JJ = irs, ire
 
c skip to start trace
 
       do i = 1, nu
         call trcskp ( JJ, 1, ns-1, luin(i), ntrc, itr )
       enddo

c do traces
       DO KK = ns, ne

c do buckets
         DO II = 1, nu

            nbytes = 0
            call rtape( luin(II), itr, nbytes)

c if end of data encountered (nbytes=0) then bail out

            if(nbytes .eq. 0) then
               write(LERR,*)'Premature EOF on input at:'
               write(LERR,*)'  rec= ',JJ,'  trace= ',KK
               write(LERR,*)'  input file = ',ntap(II)
               go to 999
            endif

c get required trace header information

            call saver2( itr, ifmt_RecNum, l_RecNum, ln_RecNum, 
     :           RecNum, TRACEHEADER )

            call saver2( itr, ifmt_TrcNum, l_TrcNum, ln_TrcNum, 
     :           TrcNum, TRACEHEADER )

            call saver2( itr, ifmt_DstSgn, l_DstSgn, ln_DstSgn, 
     :           DstSgn, TRACEHEADER )

            call saver2( itr, ifmt_StaCor, l_StaCor, ln_StaCor, 
     :           StaCor, TRACEHEADER )

c process only live traces

            if ( StaCor .ne. 30000) then

c Put your subroutine here [remember to declare any arguments you need
c over and above those already declared]
c load trace portion of itr[] to real array tri[]

 
             call vmov ( itr(ITHWP1), 1, tri, 1, nsamp )
c  move data from tri to rtrpt(II)
c     for attribute computation; save trace1 header for output

             call vmov (tri, 1, rtrpt(1,II), 1,  nsamp)
               
c
c     Check for invalid number of angle stacks from ANGST:
c	2p = 1 or 2 angle stacks from ANGST
c	3p = 3 angle stacks from ANGST
c

             call vclr ( tri, 1, nsamp )
	     err1 = 0

c       check for a trace from each angle bucket
             if ( II .eq. nu ) then

c       verify stack traces have been built
		if ( STKDONE ) then

        if ( dbug ) then
               write(LER,*)'Processing rec= ',JJ,'  trace= ',KK
        endif

c	cross correlate and shift trace data as needed
		  call llshift(tzero,deltat,totdt,nxcor,rtrpt,
     1				trstk,nsi,nu,ershifts,dbug)

c	apply maximum amplitude to trace lobes
		  do i = 1, nu
		    do n = 1, zxarycnt-1
			maxval = 0.
			call maxmgv(rtrpt(zxary(n),i),1,maxval,
     1			       maxvcnt,zxary(n+1)-zxary(n))
			if (rtrpt(zxary(n),i).ge.0.) then
			  if(n .eq. (zxarycnt-1)) then
			   call vfill(maxval,rtrpt(zxary(n),i),1,
     1			        (zxary(n+1)-zxary(n)+1))
			  else
			   call vfill(maxval,rtrpt(zxary(n),i),1,
     1				zxary(n+1)-zxary(n))
			  endif
			else
			   maxval = (maxval * (-1.0))
			  if(n .eq. (zxarycnt-1)) then
			   call vfill(maxval,rtrpt(zxary(n),i),1,
     1			        (zxary(n+1)-zxary(n)+1))
			  else
			   call vfill(maxval,rtrpt(zxary(n),i),1,
     1				zxary(n+1)-zxary(n))
			  endif
			endif
		    enddo
		  enddo

c	keep a running total on the number of traces processed
		  nptraces = nptraces + nu

        if ( dbug ) then
               write(LER,*)'traces processed =',nptraces
        endif

c       single angle bucket calculations are not allowed

                  if(numangles .eq. 1) then
                     write(LERR,*)' '
            write(LERR,*) 'llattrib3d:Invalid number of angle buckets ('
                     write(LERR,*) numangles,') specified.'
                     write(LERR,*)' '
                     write(LER ,*) ' FATAL'
		     goto 999

c       with two angles, only two bucket attributes allowed

                  else if(numangles .eq. 2) then
                     call att2p3d (amn, amx, numangles, attrnum,
     :                    dtsec,nsamp,rtrpt,tri,err1)
		     if (err1 .gt. 0) then
			write(LER ,*) ' FATAL'
			goto 999
		     endif

c       with three angles, only three bucket attributes calculated here

                  else if((numangles .eq. 3) .and.
     :		     ((attrnum .eq. 3) .or. (attrnum .eq. 8) .or. 
     :               (attrnum .eq. 9) .or. (attrnum .eq. 12) .or.
     :               (attrnum .eq. 13) .or. (attrnum .eq. 14) .or.
     :               (attrnum .eq. 19) .or. (attrnum .eq. 22) .or.
     :               (attrnum .eq. 24) .or. (attrnum .eq. 25))) then
                       call att3p3d (amn, amx, numangles, attrnum,
     :                    dtsec,nsamp,rtrpt,tri,err1)
		     if (err1 .gt. 0) then
			write(LER ,*) ' FATAL'
			goto 999
		     endif

c       with more than 3 angles or calculation of 2 param attributes
c	with three angle buckets, the B0, B1, and B2 are calculated
c	via a least squares curve fit routine

                  else 
                     call lsfit (amn, amx, numangles, attrnum,
     :                    dtsec,nsamp,rtrpt,tri,err1)
		     if (err1 .gt. 0) then
			write(LER ,*) ' FATAL'
			goto 999
                     endif
                  endif
               
                  call vmov ( tri, 1, itr(ITHWP1), 1, nsamp )
                  call dotpr  (tri, 1, tri, 1, xdot, nsamp)
                  if (xdot .lt. 1.e-30) then
                     call savew2(itr,ifmt_StaCor,l_StaCor,ln_StaCor,
     1                    30000  , TRACEHEADER)
                  endif

c  	stack traces are built here
		else
        	  if ((dbug).and.(JJ.eq.1).and.(KK.eq.1)) then
               	  	write(LER,*)'Creating stack trace.'
        	  endif
		  do i = 1, stakn
		     do n = 1, nsamp
			trstk(n)=trstk(n)+(rtrpt(n,i)/float(stakn))
		     enddo
		  enddo
		  stkdone = .true.

c		  detect and log zero crossings in stacked trace
                  if (trstk(1).ge. 0.) then
		  	posflg = .true.
		  else
			posflg = .false.
		  endif
		  zxary(1) = 1
		  zxarycnt = 2
                  do n = 2, nsamp
			if ((posflg).and.(trstk(n).lt.0.)) then
				zxary(zxarycnt) = n
				zxarycnt = zxarycnt + 1
				posflg = .false.
c       	     if ( dbug ) then
c              	      write(LER,*)'Zero crossing at sample ',n
c       	     endif
  	         elseif ((.not.posflg).and.(trstk(n).ge.0.)) then
				zxary(zxarycnt) = n
				zxarycnt = zxarycnt + 1
				posflg = .true.
c       	     if ( dbug ) then
c              	      write(LER,*)'Zero crossing at sample ',n
c       	     endif
			endif	
                  enddo
        	  if ( dbug ) then
               	      write(LER,*)'Stack trace done.'
        	  endif

c		  rewind to start, then advance past line header
		  do i = nu, 1, -1
		      call rwd(luin(i))
	   	      call rtape(luin(i),itr,lbytes)
		  enddo
 		  goto 100
		endif

             endif

            endif
 
c enddo buckets
         ENDDO

c write output data
            
            call savew2( itr, ifmt_RecNum, l_RecNum, ln_RecNum, 
     :           RecNum, TRACEHEADER )

            call savew2( itr, ifmt_TrcNum, l_TrcNum, ln_TrcNum, 
     :           TrcNum, TRACEHEADER )

            call wrtape (luout, itr, obytes)
 
c enddo traces
        ENDDO

c skip to end of record

        do i = 1, nu
           call trcskp ( JJ, ne+1, ntrc, luin(i), ntrc, itr )
        enddo

c enddo records
      ENDDO

c close data files 

      do i = 1, nu
      	call lbclos ( luin(i) )
      enddo
      call lbclos ( luout )
      call lbclos ( luang )

c write out cross correlation error info and termination statement
      err2 = float(ershifts)/(float(nxcor)*float(nptraces))
      write(LERR,*)'llattrib3d: Trace shift error =',err2
      write(LER,*)'llattrib3d: Trace shift error =',err2

      write(LERR,*)'llattrib3d: Normal Termination'
      write(LER,*)'llattrib3d: Normal Termination'
      stop

 999  continue

      do i = 1, nu
      	call lbclos ( luin(i) )
      enddo
      call lbclos ( luout )
      call lbclos ( luang )
      write(LERR,*)'llattrib3d: ABNORMAL Termination'
      write(LER,*)'llattrib3d: ABNORMAL Termination'
      stop
      end

c -----------------  Subroutine -----------------------
c
c
c **** Subroutine rdangs  ****
c reads in min_max angle pairs from a file
c
      subroutine rdangs(amn, amx, numangles, luang, err1)

#include <f77/iounit.h>

      integer    numangles, luang, ri, i, err1, EOFflag
      character  card*80
      real       amn(*), amx(*)

	err1 = 0
	EOFflag = 0
	numangles = 1

 10	continue
	read(luang,20,end=40)card,(amn(i),amx(i),
     *      i=numangles,numangles+6),ri
 20	format(a80,t1,5x,7(f5.0,f5.0),t76,i5)

c  	write out angl cards
	write(LERR,*)'Input angle card: '
	write(LERR,*)card

	goto 50
 40	EOFflag = 1

 50	continue

c       correct the number of angle pairs counter
	do i=1, 7
  	   if((amn(numangles).ge.0).and.(amx(numangles).gt.0))then
		numangles = numangles + 1

c  	   check for unequal number of min_max angles
	   elseif((amn(numangles).gt.0).and.
     *                 (amx(numangles).eq.0))then
		   write(LER,*)'llattrib3d: Need maximum angle'
		   write(LER,*)'for minimum angle ',amn(numangles)
		   write(LER,*)'FATAL'
		err1 = 1
		goto 90

c  	   check for zero min and max angles
	   elseif((amn(numangles).eq.0).and.
     *                 (amx(numangles).eq.0))then
		goto 60
	   endif
	enddo

 60	continue

c	check for too many angles
	if((numangles-1).gt.45) then
		write(LER,*)'llattrib3d: Too many angle pairs.'
		write(LER,*)'Maximum number of angles = 45.'
		write(LER,*)'FATAL'
		err1 = 1
		goto 90
	endif
c  	check for 9angl card
	if ((card(1:5).ne.'9angl').and.(EOFflag.ne.1)) goto 10

c  	decrement numangles to show correct angle count
	numangles = numangles - 1

 90	continue
	return
	end
c
c
c **** Subroutine verbal  ****
c prints out program input and output data
c
      subroutine verbal(nsamp, nsi, ntrc, nrec, iform,
     1  attrnum,atap,ntap,nu,otap,irs,ire,ns,ne,amn,amx,numangles,
     2  ptap,stakn,deltat,totdt,nxcor,tzero)

#include <f77/iounit.h>

      integer     nsamp, nsi, ntrc, nrec, irs, ire
      integer     nu, numangles, ns, ne
      integer     stakn, nxcor
      integer     attrnum
      integer     deltat, totdt, tzero
      real        amn(*),amx(*)
      character   ntap(70)*(*), otap*(*), atap*(*), ptap*(*)
            write(LERR,*)' '
            write(LERR,*)' line header values after default check '
            write(LERR,*) ' # of samples/trace =  ', nsamp
            write(LERR,*) ' sample interval    =  ', nsi
            write(LERR,*) ' traces per record  =  ', ntrc
            write(LERR,*) ' records per line   =  ', nrec
            write(LERR,*) ' format of data     =  ', iform
            write(LERR,*)' '
            write(LERR,*)' input parameters '
            write(LERR,*) ' number of input files =  ', nu
            write(LERR,*) ' angle data set name =  ', atap
	do i = 1 , nu
            write(LERR,*) ' input data set name ',i,' =  ',ntap(i)
	enddo
            write(LERR,*) ' output data set name =  ', otap
            write(LERR,*) ' start record number =  ', irs
            write(LERR,*) ' last record number =  ', ire
            write(LERR,*) ' start trace number =  ', ns
            write(LERR,*) ' last trace number =  ', ne
            write(LERR,*) ' attribute number =  ', attrnum
            write(LERR,*)' '
            write(LERR,*)' '
       	    write(LERR,'(a   )')' Minimum Angles:'
            call writer(amn,numangles,1,LERR)
            write(LERR,*)' '
            write(LERR,'(a   )')' Maximum Angles:'
            call writer(amx,numangles,1,LERR)
            write(LERR,*)' '
            write(LERR,*)' parameter card file name =  ', ptap
            write(LERR,*)' number of traces in xcor stack  = ',stakn
            write(LERR,*)' delta T (trace shift) time = ',deltat
            write(LERR,*)' total portion of trace to xcor = ',totdt
            write(LERR,*)' number of cross correlations = ',nxcor
            write(LERR,*)' start offset time for xcor = ',tzero
            write(LERR,*)' '
            write(LERR,*)' '
      return
      end
      subroutine help()
#include <f77/iounit.h>
         write(LER,*)
     :'***************************************************************'
      write(LER,*) ' '
      write(LER,*)
     :'llattrib3d: Attribute calculator for angst3d data.'
      write(LER,*) ' '
      write(LER,*)
     :'Execute llattrib3d by typing '
      write(LER,*)
     :'        "llattrib3d" and the program parameters.'
      write(LER,*)
     :'Note that each parameter is proceeded by -a where "a" is '
      write(LER,*)
     :'a character(s) corresponding to some parameter.'
      write(LER,*)
     :'Enter the following parameters, or use the default values'
      write(LER,*)' '
        write(LER,*)
     :' -A [atap]    (no default)         : input angle card file'
        write(LER,*)
     :' -P [ptap]    (no default)         : xcor parameter card file'
        write(LER,*)
     :' -N [ntap]    (no default)         : input data file names'
        write(LER,*) ' '
        write(LER,*)
     :' -O [otap]    (no default; stdout if not specified)'
        write(LER,*)
     :'                                   : output data file name'
        write(LER,*) ' '
        write(LER,*)
     :' -rs[irs]     (default = first)    : start record number'
        write(LER,*)
     :' -re[ire]     (default = last)     : end record number'
        write(LER,*) ' '
        write(LER,*)
     :' -ns[ns]      (default = first)    : start trace number'
        write(LER,*)
     :' -ne[ne]      (default = last)     : end trace number'
        write(LER,*) ' '
        write(LER,*)
     :' -att[attrnum]  (no default)       : attribute number'
        write(LER,*) ' '
        write(LER,*)
     :' -V                                : verbose printout'
        write(LER,*)
     :'usage:    llattrib3d  -A[atap] -N[ntap1] -N[ntap2} -N[ntap3}'
        write(LER,*)
     :'      -P[ptap] -O[otap] -rs[irs] -re[ire] -att[attrnum] [-V]'
        write(LER,*) ' '
        write(LER,*) ' '
        write(LER,*)
     :'Self doc:  llattrib3d  -h'
        write(LER,*)
     :'Available list of attributes: llattrib3d  -ha'
        write(LER,*) ' '
        write(LER,*)
     :'USAGE NOTES:  '
        write(LER,*)
     :'          Input angle data from angst3d should be '
        write(LER,*)
     :'          specified in the input angle card file '
        write(LER,*)
     :'          as min1,max1,min2,max2, etc. (maximum = 63) '
        write(LER,*) ' '
        write(LER,*)
     :'          Input data from angst3d must be organized'
        write(LER,*)
     :'          as n stacked volumes for the n angles,'
        write(LER,*)
     :'          distributed one angle per file, and specified'
        write(LER,*)
     :'          in command line with small angle stack in first'
        write(LER,*)
     :'          file, middle angle stack in second file, etc.'
        write(LER,*) ' '
        write(LER,*)
     :'          The data shift parameter file is fixed format, '
        write(LER,*)
     :'          with the parameter name starting in column 1, '
        write(LER,*)
     :'          and its value starting in column 10.  The '
        write(LER,*)
     :'          parameters specified in this file along with '
        write(LER,*)
     :'          their default values follow:'
        write(LER,*) ' '
        write(LER,*)
     :'                                 1 '
        write(LER,*)
     :'               column   1        0 '
        write(LER,*) ' '
        write(LER,*)
     :'                        stakn    1 '
        write(LER,*)
     :'                        deltat   12.0 '
        write(LER,*)
     :'                        totdt    30.0 '
        write(LER,*)
     :'                        nxcor    3 '
        write(LER,*)
     :'                        tzero    0.0 '
        write(LER,*) ' '
        write(LER,*)
     :'          "stakn" is the number of traces stacked together '
        write(LER,*)
     :'          that will be used by the cross correlation routine. '
        write(LER,*)
     :'          "deltat" is the time in milliseconds through '
        write(LER,*)
     :'          which the data will be shifted. '
        write(LER,*)
     :'          "totdt" is the total time in milliseconds through '
        write(LER,*)
     :'          which each section is cross correlated. '
        write(LER,*)
     :'          "nxcor" is the number of sections of each trace '
        write(LER,*)
     :'          to be cross correlated. '
        write(LER,*)
     :'          "tzero" is the start time in milliseconds for the '
        write(LER,*)
     :'          first section to be cross correlated '
        write(LER,*) ' '
        write(LER,*)
     :'***************************************************************'
      return
      end
c
c
c **** Subroutine help  ****
c prints out usage instructions
c
      subroutine helpatt()
#include <f77/iounit.h>
         write(LER,*)
     :'***************************************************************'
      write(LER,*) ' '
      write(LER,*)
     :'llattrib3d: Attribute calculator for angst3d data.'
      write(LER,*) ' '
      write(LER,*)
     :'The following recon attributes are currently available:'
      write(LER,*)' '
        write(LER,*)
     :'1) B0 Note: Needs a two or three angle bucket input from angst.'

        write(LER,*)
     :'2) B1 Note: Needs a two or three angle bucket input from angst.'
        write(LER,*)
     :'         with angles out to a minimum of 15 degrees.'

        write(LER,*)
     :'3) B2 Note: Needs a two or three angle bucket input from angst'
        write(LER,*)
     :'         with angles out to a minimum of 37 degrees.'

        write(LER,*)
     :'4) Bz=SIGN(B0)*B1 Note: Needs a two or three angle bucket input'
        write(LER,*)
     :'         from angst with angles out to a minimum of 15 degrees.'

        write(LER,*)
     :'5) Bp=B0*B1 Note: Needs a two or three angle bucket input'
        write(LER,*)
     :'         from angst with angles out to a minimum of 15 degrees.'

        write(LER,*)
     :'6) Br=B1/B0 Note: Needs a two or three angle bucket input'
        write(LER,*)
     :'         from angst with angles out to a minimum of 15 degrees.'

        write(LER,*)
     :'7) Middle angle stack - Near angle stack   Note: A two or'
        write(LER,*)
     :'         three angle bucket input from angst needed.'

        write(LER,*)
     :'8) Large angle stack - Near angle stack'
        write(LER,*)
     :'         Note: A three angle bucket input needed from angst.'

        write(LER,*)
     :'9) Large angle stack - Middle angle stack'
        write(LER,*)
     :'         Note: A three angle bucket input needed from angst.'

        write(LER,*)
     :'10) Zero crossing angle in degrees.   Note: A two or'
        write(LER,*)
     :'         three angle bucket input from angst needed.'

        write(LER,*)
     :'11) Restricted Gradient'
        write(LER,*)
     :'       Two angle buckets: EE(middle angle)-EE(small angle)'
        write(LER,*)
     :'       Three angle buckets: EE(large angle)-EE(small angle)'
        write(LER,*)
     :'         '

        write(LER,*)
     :'12) dVp/Vp   Note: Needs a two or three angle bucket input'
        write(LER,*)
     :'       from angst with angles out to a minimum of 37 degrees.'

        write(LER,*)
     :'13) dVs/Vs   Note: Needs a two or three angle bucket input'
        write(LER,*)
     :'       from angst with angles out to a minimum of 37 degrees.'

        write(LER,*)
     :'14) dRho/Rho  Note: Needs a two or three angle bucket input'
        write(LER,*)
     :'       from angst with angles out to a minimum of 37 degrees.'

        write(LER,*)
     :'15) dZp/Zp  Note: Needs a two angle bucket input'
        write(LER,*)
     :'       from angst with angles out to a maximum of 30 degrees.'

        write(LER,*)
     :'16) dZs/Zs  Note: Needs a two angle bucket input'
        write(LER,*)
     :'       from angst with angles out to a maximum of 30 degrees.'

        write(LER,*)
     :'17) d(Vp/Vs)/(Vp/Vs)  Note: Needs a two angle bucket input'
        write(LER,*)
     :'       from angst with angles out to a maximum of 30 degrees.'

        write(LER,*)
     :'18) dF    Shear Residual'
        write(LER,*)
     :'       '

        write(LER,*)
     :'19) 1/B2  Contact Event  Note: Needs a three angle bucket input'
        write(LER,*)
     :'       from angst with angles out to a minimum of 37 degrees.'

        write(LER,*)
     :'20) Energy Envelope(small angle)  Note: Needs at least one '
        write(LER,*)
     :'       angle bucket input from ANGST.'
 
        write(LER,*)
     :'21) Energy Envelope(mid angle)  Note: Needs at least two '
        write(LER,*)
     :'       angle buckets input from ANGST.'
 
        write(LER,*)
     :'22) Energy Envelope(large angle)  Note: Needs a three '
        write(LER,*)
     :'       angle bucket input from ANGST.'

        write(LER,*)
     :'23) Energy Envelope(mid angle) - Energy Envelope(small angle)'
        write(LER,*)
     :'       Note:  Needs a two angle bucket input from ANGST.'
 
        write(LER,*)
     :'24) Energy Envelope(large angle) - Energy Envelope(small angle)'
        write(LER,*)
     :'       Note:  Needs a three angle bucket input from ANGST.'
 
        write(LER,*)
     :'25) Enhanced Restricted Gradient.'
        write(LER,*)
     :'       (EE(large angle) - EE(small angle)) * EE(large angle)'
        write(LER,*)
     :'       Note:  Needs a three angle bucket input from ANGST.'

        write(LER,*)
     :'***************************************************************'
      return
      end
