C***********************************************************************
C                 copyright 2001, Amoco Production Company             *
C                             All Rights Reserved                      *
C                     an affiliate of BP America Inc.                  *
C***********************************************************************
c *********************************************************************
c
c  SUBROUTINE LSFIT (amn, amx, numangles, attrnum, dtsec, nsamp,
c                       rtrpt, arrayout, err)
c
c    This subroutine processes attributes for 3 parameter angst data
c
c    "amn" is the minimum angle data array as used in ANGST3D
c    "amx" is the maximum angle data array as used in ANGST3D
c    "numangles" is the number of angle buckets input from ANGST3D
c    "attrnum" is the attribute (by number) to be calculated
c    "dtsec" is the sample interval in seconds
c    "nsamp" is the number of samples in each angle stack
c    "rtrpt" is the array of data traces input from ANGST3D
c    "arrayout" is the attribute data calculated from the three
c               input angle data stack traces 
c    "err" is an error flag
c
c**********************************************************************
 
      subroutine lsfit (amn, amx, numangles, attrnum, dtsec,
     1     nsamp, rtrpt, arrayout, err)
#include <f77/lhdrsz.h>
  
      external  tpoly
      integer   numangles
      integer   attrnum, nsamp, err
      integer   ch, np, i, j, ma
      real      amn(*), amx(*)
      real      rtod, dtor, small, dtsec
      real      quadp, quadm, quada, quadb
      real      min(63), max(63), meanangs(63)
      real      sig(63), y(63), a(3)
      real      u (SZLNHD)
      real      v (SZLNHD)
      real      w (SZLNHD)
      real      b0 (SZLNHD)
      real      b1 (SZLNHD)
      real      b2 (SZLNHD)
      real      Q (SZLNHD)
      real      wkarray1 (SZLNHD)
      real      wkarray2 (SZLNHD)
      real      rtrpt (SZLNHD,63)
      real      arrayout (nsamp)
 
c load "small" value
c
      small = 1.e-15
c
c load degree/radian conversion factors
c
      dtor = .0174533
      rtod = 57.29578

c
c    Clear output array to zeros
c
      call vclr (arrayout, 1, nsamp)
      err = 0

c
c-----
c
c    Convert all min and max input angles to radians,
c	and calculate the mean angle for each bucket
c

      do 10 j = 1, numangles
         min(j) = dtor * amn(j)
         max(j) = dtor * amx(j)
 	 meanangs(j) = min(j) + ((max(j)-min(j))/2)
 10     continue

c
c-----
c
c
c   Calculate b0s, b1s, and b2s using least squares routine
c
c
c     The following are the input parms for the svdfit routine:
c	meanangs = array of mean angle bucket values (one for each angle bucket)
c	y = array of data points (one from each angle bucket) at sample number j
c	sig = standard of deviation for each data point
c	numangles = number of data points to which the curve will be fitted
c	a = array containing returned b0s, b1s, and b2s
c	ma = number of coeffcients to calculate
c	u, v, and w = work arrays for the svdfit routine
c	np = order of fit for the polynomial
c	ch = chi square constant for polyfit routine
c	tpoly = subroutine where basis functions are calculated
c

	  if((attrnum .eq. 1) .or. (attrnum .eq. 2) .or.
     :       (attrnum .eq. 4) .or. (attrnum .eq. 5) .or.
     :       (attrnum .eq. 6) .or. (attrnum .eq. 7) .or.
     :       (attrnum .eq. 10) .or. (attrnum .eq. 11) .or.
     :       (attrnum .eq. 15) .or. (attrnum .eq. 16) .or.
     :       (attrnum .eq. 17) .or. (attrnum .eq. 18) .or.
     :       (attrnum .eq. 20) .or. (attrnum .eq. 21) .or.
     :       (attrnum .eq. 23)) then
		ma = 2
	  else  
	 	ma = 3
	  endif

	  do 50 j = 1, numangles 
		sig(j) = 1
 50 	  continue

	  np = 6
	  ch = 0.

	  do 100 j = 1, nsamp
		call vclr(a,1,3)
		call vclr(u,1,SZLNHD)
		call vclr(v,1,SZLNHD)
		call vclr(w,1,SZLNHD)

	        do 110 i = 1, numangles
		   y(i) = rtrpt(j,i)
 110 	        continue

 	        call svdfit(meanangs,y,sig,numangles,a,ma,u,v,w,
     :	        	numangles,np,ch,tpoly,min,max)

                b0(j) = a(1)
                b1(j) = a(2)
                b2(j) = a(3)
 100 	  continue
 
c
c-----
c
c   Attribute calculations follow
c

c
c   If attrnum = 1, output b0s only
c
	if(attrnum .eq. 1) then
      		call vmov (b0, 1, arrayout, 1, nsamp)
		return
	endif

c
c   If attrnum = 2, output b1s only
c
	if(attrnum .eq. 2) then
      		call vmov (b1, 1, arrayout, 1, nsamp)
		return
	endif

c
c   If attrnum = 3, output b2s only
c
	if(attrnum .eq. 3) then
      		call vmov (b2, 1, arrayout, 1, nsamp)
                return
        endif

c
c   If attrnum = 4, output Bz=SIGN(B0)*B1
c
	if(attrnum .eq. 4) then
	  do 400 j = 1, nsamp 
		if(b0(j).lt.0.0) then
			arrayout(j) = -1.0 * b1(j)
		else
			arrayout(j) = b1(j)
		endif
 400 	  continue
	  return
	endif

c
c   If attrnum = 5, output Bp=B0*B1
c
        if(attrnum .eq. 5) then
          do 500 j = 1, nsamp
		if (b0(j) .le. small .or. b1(j) .le. small) then
                        arrayout(j) = 0
                else
                        arrayout(j) = b0(j) * b1(j)
                endif
 500      continue
          return
        endif

c
c   If attrnum = 6, output Br=B1/B0
c
        if(attrnum .eq. 6) then
          do 600 j = 1, nsamp
                if (b0(j) .eq. 0 .or. b1(j) .eq. 0) then
                        arrayout(j) = 0
                else
                        arrayout(j) = b1(j) / b0(j)
                endif
 600      continue
          return
        endif

c
c   If attrnum = 7, output st2 - st1
c
        if(attrnum .eq. 7) then
          do 700 j = 1, nsamp
                arrayout(j) = rtrpt(j,2) - rtrpt(j,1)
 700      continue
          return
        endif
 
c
c   If attrnum = 8, output st3 - st1
c
        if(attrnum .eq. 8) then
          do 800 j = 1, nsamp
                arrayout(j) = rtrpt(j,3) - rtrpt(j,1)
 800      continue
          return
        endif
 
c
c   If attrnum = 9, output st3 - st2
c
        if(attrnum .eq. 9) then
          do 900 j = 1, nsamp
                arrayout(j) = rtrpt(j,3) - rtrpt(j,2)
 900      continue
          return
        endif
 
c
c   If attrnum = 10, output in degrees the smaller of 
c		(-B1+sqrt((B1*B1)-4*(B2+2*B1/3)*B0))/(2*(B2+2*B1/3))
c	    or  (-B1-sqrt((B1*B1)-4*(B2+2*B1/3)*B0))/(2*(B2+2*B1/3))
c
        if(attrnum .eq. 10) then
          do 1000 j = 1, nsamp
		if(b1(j).le.small .or. b2(j).le.small) then
		  arrayout(j) = 0.0
		else
		  quada = b2(j) + ((2 * b1(j))/3)
		  quadb = (b1(j) * b1(j)) - (4 * quada * b0(j))
		  if(quadb .lt. 0.0) then
			arrayout(j) = 0.0
		  else
		       quadp = (-b1(j)+sqrt(quadb))/(2*quada)
		       quadm = (-b1(j)-sqrt(quadb))/(2*quada)
		       arrayout(j) = 0.0
                       if (quadp .le. quadm) then
                          if (quadp .ge. 0) then
				arrayout(j) = rtod * quadp
                          elseif (quadm .ge. 0) then
				arrayout(j) = rtod * quadm
                          endif
                       elseif (quadm .lt. quadp) then
                          if (quadm .ge. 0) then
				arrayout(j) = rtod * quadm
                          elseif (quadp .ge. 0) then
				arrayout(j) = rtod * quadp
                          endif
                       endif
		  endif
		endif
 1000     continue
          return
        endif
 
c
c   If attrnum = 11, output Restricted Gradient
c                       EE(large)-EE(small)
c
        if(attrnum .eq. 11) then
          call asig2(rtrpt(1,3),Q,dtsec,nsamp,3,wkarray1)
          call asig2(rtrpt(1,1),Q,dtsec,nsamp,3,wkarray2)
          do 1100 j = 1, nsamp
                arrayout(j) = wkarray1(j) - wkarray2(j)
 1100     continue
	  return
        endif
 
c
c   If attrnum = 12, output dvp/vp = 2B1+2B2
c
        if(attrnum .eq. 12) then
          do 1200 j = 1, nsamp
                arrayout(j) = (2 * b1(j)) + (2 * b2(j))
 1200     continue
          return
        endif
 
c
c   If attrnum = 13, output dvs/vs = B1-B0+2B2
c
        if(attrnum .eq. 13) then
          do 1300 j = 1, nsamp
                arrayout(j) = b1(j) - b0(j) + (2 * b2(j))
 1300     continue
          return
        endif
 
c
c   If attrnum = 14, output drho/rho = 2[B0-B1-B2]
c
        if(attrnum .eq. 14) then
          do 1400 j = 1, nsamp
                arrayout(j) = 2 * (b0(j) - b1(j) - b2(j))
 1400     continue
          return
        endif
 
c
c   If attrnum = 15, output 2*B0
c
        if(attrnum .eq. 15) then
          do 1500 j = 1, nsamp
                arrayout(j) = 2*b0(j)
 1500     continue
          return
        endif
 
c
c   If attrnum = 16, output B0-B1
c
        if(attrnum .eq. 16) then
          do 1600 j = 1, nsamp
                arrayout(j) = b0(j) - b1(j)
 1600     continue
          return
        endif
 
c
c   If attrnum = 17, output B0+B1
c
        if(attrnum .eq. 17) then
          do 1700 j = 1, nsamp
                arrayout(j) = b0(j) + b1(j)
 1700     continue
          return
        endif
 
c
c   If attrnum = 18, output 0.58*B0(j)+1.42*B1(j)+1.84*B2
c
        if(attrnum .eq. 18) then
          do 1800 j = 1, nsamp
                arrayout(j) = 0.58*b0(j)+1.42*b1(j)+1.84*b2(j)
 1800     continue
          return
        endif
 
c
c   If attrnum = 19, output 100000*(1/B2)
c
        if(attrnum .eq. 19) then
          do 1900 j = 1, nsamp
                if (b2(j) .eq. 0) then
                        arrayout(j) = 0
                else
                	arrayout(j) = 100000*(1.0/b2(j))
		endif
 1900     continue
          return
        endif
 
c
c   If attrnum = 20, output EE(small angle)
c
        if(attrnum .eq. 20) then
          call asig2(rtrpt(1,1),Q,dtsec,nsamp,3,arrayout)
          return
        endif
 
c
c   If attrnum = 21, output EE(mid angle)
c
        if(attrnum .eq. 21) then
          call asig2(rtrpt(1,2),Q,dtsec,nsamp,3,arrayout)
          return
        endif
 
c
c   If attrnum = 22, output EE(large angle)
c
        if(attrnum .eq. 22) then
          call asig2(rtrpt(1,3),Q,dtsec,nsamp,3,arrayout)
          return
        endif
 
c
c   If attrnum = 23, output EE(mid angle)-EE(small angle)
c
        if(attrnum .eq. 23) then
          call asig2(rtrpt(1,2),Q,dtsec,nsamp,3,wkarray1)
          call asig2(rtrpt(1,1),Q,dtsec,nsamp,3,wkarray2)
          do 2300 j = 1, nsamp
                arrayout(j) = wkarray1(j) - wkarray2(j)
 2300     continue
          return
        endif
 
c
c   If attrnum = 24, output EE(large angle)-EE(small angle)
c
        if(attrnum .eq. 24) then
          call asig2(rtrpt(1,3),Q,dtsec,nsamp,3,wkarray1)
          call asig2(rtrpt(1,1),Q,dtsec,nsamp,3,wkarray2)
          do 2400 j = 1, nsamp
                arrayout(j) = wkarray1(j) - wkarray2(j)
 2400     continue
          return
        endif

c
c   If attrnum = 25, output Enhanced Restricted Gradient
c                       (EE(large)-EE(small))*EE(large)
c
        if(attrnum .eq. 25) then
          call asig2(rtrpt(1,3),Q,dtsec,nsamp,3,wkarray1)
          call asig2(rtrpt(1,1),Q,dtsec,nsamp,3,wkarray2)
          do 2500 j = 1, nsamp
                arrayout(j)=(wkarray1(j)-wkarray2(j))*wkarray1(j)
 2500     continue
          return
        endif
 
c
c   If attrnum < 1 or attrnum > 25, return error code = 1
c
        if(attrnum .lt. 1 .or. attrnum .gt. 25) then
                write(LERR,*)' '
                write(LERR,*) ' Requested attribute not available. '
                write(LERR,*)' '
		err = 1
		return
	endif
      write(LERR,*)' '
      write(LERR,*) ' Invalid attribute condition.'
      write(LERR,*)' '
      err = 1
      return
      end

c*********************************************************************
c
c    Least Squares subroutines follow
c
c*********************************************************************

c
C
c*********************************************************************
c
c  subroutine SVDFIT
c
**********************************************************************
C
      SUBROUTINE SVDFIT(X,Y,SIG,NDATA,A,MA,U,V,W,MP,NP,CHISQ,tpoly,
     :			   tmin,tmax)

      integer ndata, ma, mp, np, chisq
      PARAMETER(NMAX=4096,MMAX=3,TOL=1.E-5)
      DIMENSION X(*),Y(*),SIG(*),A(*),V(NP,NP),
     :    U(MP,NP),W(NP),B(NMAX),AFUNC(MMAX),
     :    tmin(*), tmax(*)
      DO 20 I=1,NDATA
        call tpoly(tmin(i),tmax(i),afunc)
        TMP=1./SIG(I)
        DO 10 J=1,MA
          U(I,J)=AFUNC(J)*TMP
   10   CONTINUE
        B(I)=Y(I)*TMP
   20 CONTINUE
      CALL SVDCMP(U,NDATA,MA,MP,NP,W,V)
      WMAX=0.
      DO 30 J=1,MA
        IF(W(J).GT.WMAX)WMAX=W(J)
   30 CONTINUE
      THRESH=TOL*WMAX
      DO 40 J=1,MA
        IF(W(J).LT.THRESH)W(J)=0.
   40 CONTINUE
      CALL SVBKSB(U,W,V,NDATA,MA,MP,NP,B,A)
      CHISQ=0.
      DO 60 I=1,NDATA
        call tpoly(tmin(i),tmax(i),afunc)
        SUM=0.
        DO 50 J=1,MA
          SUM=SUM+A(J)*AFUNC(J)
   50   CONTINUE
        CHISQ=CHISQ+((Y(I)-SUM)/SIG(I))**2
   60 CONTINUE
      RETURN
      END
c
c
C
c*********************************************************************
c
c  subroutine SVBKSB
c
**********************************************************************
C
c
      SUBROUTINE SVBKSB(U,W,V,M,N,MP,NP,B,X)
      PARAMETER (NMAX=100)
      DIMENSION U(MP,NP),W(NP),V(NP,NP),B(MP),X(NP),TMP(NMAX)
      DO 20 J=1,N
        S=0.
        IF(W(J).NE.0.)THEN
          DO 10 I=1,M
            S=S+U(I,J)*B(I)
   10     CONTINUE
          S=S/W(J)
        ENDIF
        TMP(J)=S
   20 CONTINUE
      DO 40 J=1,N
        S=0.
        DO 30 JJ=1,N
          S=S+V(J,JJ)*TMP(JJ)
   30   CONTINUE
        X(J)=S
   40 CONTINUE
      RETURN
      END
c
c
C
c*********************************************************************
c
c  subroutine SVDCMP
c
**********************************************************************
C
c
      SUBROUTINE SVDCMP(A,M,N,MP,NP,W,V)
      PARAMETER (NMAX=100)
      DIMENSION A(MP,NP),W(NP),V(NP,NP),RV1(NMAX)
      G=0.0
      SCALE=0.0
      ANORM=0.0
      DO 140 I=1,N
        L=I+1
        RV1(I)=SCALE*G
        G=0.0
        S=0.0
        SCALE=0.0
        IF (I.LE.M) THEN
          DO 10 K=I,M
            SCALE=SCALE+ABS(A(K,I))
   10     CONTINUE
          IF (SCALE.NE.0.0) THEN
            DO 20 K=I,M
              A(K,I)=A(K,I)/SCALE
              S=S+A(K,I)*A(K,I)
   20       CONTINUE
            F=A(I,I)
            G=-SIGN(SQRT(S),F)
            H=F*G-S
            A(I,I)=F-G
            IF (I.NE.N) THEN
              DO 50 J=L,N
                S=0.0
                DO 30 K=I,M
                  S=S+A(K,I)*A(K,J)
   30           CONTINUE
                F=S/H
                DO 40 K=I,M
                  A(K,J)=A(K,J)+F*A(K,I)
   40           CONTINUE
   50         CONTINUE
            ENDIF
            DO 60 K= I,M
              A(K,I)=SCALE*A(K,I)
   60       CONTINUE
          ENDIF
        ENDIF
        W(I)=SCALE *G
        G=0.0
        S=0.0
        SCALE=0.0
        IF ((I.LE.M).AND.(I.NE.N)) THEN
          DO 70 K=L,N
            SCALE=SCALE+ABS(A(I,K))
   70     CONTINUE
          IF (SCALE.NE.0.0) THEN
            DO 80 K=L,N
              A(I,K)=A(I,K)/SCALE
              S=S+A(I,K)*A(I,K)
   80       CONTINUE
            F=A(I,L)
            G=-SIGN(SQRT(S),F)
            H=F*G-S
            A(I,L)=F-G
            DO 90 K=L,N
              RV1(K)=A(I,K)/H
   90       CONTINUE
            IF (I.NE.M) THEN
              DO 120 J=L,M
                S=0.0
                DO 100 K=L,N
                  S=S+A(J,K)*A(I,K)
  100           CONTINUE
                DO 110 K=L,N
                  A(J,K)=A(J,K)+S*RV1(K)
  110           CONTINUE
  120         CONTINUE
            ENDIF
            DO 130 K=L,N
              A(I,K)=SCALE*A(I,K)
  130       CONTINUE
          ENDIF
        ENDIF
        ANORM=MAX(ANORM,(ABS(W(I))+ABS(RV1(I))))
  140 CONTINUE
      DO 200 I=N,1,-1
        IF (I.LT.N) THEN
          IF (G.NE.0.0) THEN
            DO 150 J=L,N
              V(J,I)=(A(I,J)/A(I,L))/G
  150       CONTINUE
            DO 180 J=L,N
              S=0.0
              DO 160 K=L,N
                S=S+A(I,K)*V(K,J)
  160         CONTINUE
              DO 170 K=L,N
                V(K,J)=V(K,J)+S*V(K,I)
  170         CONTINUE
  180       CONTINUE
          ENDIF
          DO 190 J=L,N
            V(I,J)=0.0
            V(J,I)=0.0
  190     CONTINUE
        ENDIF
        V(I,I)=1.0
        G=RV1(I)
        L=I
  200 CONTINUE
      DO 270 I=N,1,-1
        L=I+1
        G=W(I)
        IF (I.LT.N) THEN
          DO 210 J=L,N
            A(I,J)=0.0
  210     CONTINUE
        ENDIF
        IF (G.NE.0.0) THEN
          G=1.0/G
          IF (I.NE.N) THEN
            DO 240 J=L,N
              S=0.0
              DO 220 K=L,M
                S=S+A(K,I)*A(K,J)
  220         CONTINUE
              F=(S/A(I,I))*G
              DO 230 K=I,M
                A(K,J)=A(K,J)+F*A(K,I)
  230         CONTINUE
  240       CONTINUE
          ENDIF
          DO 250 J=I,M
            A(J,I)=A(J,I)*G
  250     CONTINUE
        ELSE
          DO 260 J= I,M
            A(J,I)=0.0
  260     CONTINUE
        ENDIF
        A(I,I)=A(I,I)+1.0
  270 CONTINUE
      DO 390 K=N,1,-1
        DO 370 ITS=1,50
          DO 280 L=K,1,-1
            NM=L-1
            IF ((ABS(RV1(L))+ANORM).EQ.ANORM)  GO TO 320
            IF ((ABS(W(NM))+ANORM).EQ.ANORM)  GO TO 290
  280     CONTINUE
  290     C=0.0
          S=1.0
          DO 310 I=L,K
            F=S*RV1(I)
            IF ((ABS(F)+ANORM).NE.ANORM) THEN
              G=W(I)
              H=SQRT(F*F+G*G)
              W(I)=H
              H=1.0/H
              C= (G*H)
              S=-(F*H)
              DO 300 J=1,M
                Y=A(J,NM)
                Z=A(J,I)
                A(J,NM)=(Y*C)+(Z*S)
                A(J,I)=-(Y*S)+(Z*C)
  300         CONTINUE
            ENDIF
  310     CONTINUE
  320     Z=W(K)
          IF (L.EQ.K) THEN
            IF (Z.LT.0.0) THEN
              W(K)=-Z
              DO 330 J=1,N
                V(J,K)=-V(J,K)
  330         CONTINUE
            ENDIF
            GO TO 380
          ENDIF
C         IF (ITS.EQ.50) PAUSE 'NO CONVERGENCE IN 30 ITERATIONS'
          X=W(L)
          NM=K-1
          Y=W(NM)
          G=RV1(NM)
          H=RV1(K)
          F=((Y-Z)*(Y+Z)+(G-H)*(G+H))/(2.0*H*Y)
          G=SQRT(F*F+1.0)
          F=((X-Z)*(X+Z)+H*((Y/(F+SIGN(G,F)))-H))/X
          C=1.0
          S=1.0
          DO 360 J=L,NM
            I=J+1
            G=RV1(I)
            Y=W(I)
            H=S*G
            G=C*G
            Z=SQRT(F*F+H*H)
            RV1(J)=Z
            C=F/Z
            S=H/Z
            F= (X*C)+(G*S)
            G=-(X*S)+(G*C)
            H=Y*S
            Y=Y*C
            DO 340 NM=1,N
              X=V(NM,J)
              Z=V(NM,I)
              V(NM,J)= (X*C)+(Z*S)
              V(NM,I)=-(X*S)+(Z*C)
  340       CONTINUE
            Z=SQRT(F*F+H*H)
            W(J)=Z
            IF (Z.NE.0.0) THEN
              Z=1.0/Z
              C=F*Z
              S=H*Z
            ENDIF
            F= (C*G)+(S*Y)
            X=-(S*G)+(C*Y)
            DO 350 NM=1,M
              Y=A(NM,J)
              Z=A(NM,I)
              A(NM,J)= (Y*C)+(Z*S)
              A(NM,I)=-(Y*S)+(Z*C)
  350       CONTINUE
  360     CONTINUE
          RV1(L)=0.0
          RV1(K)=F
          W(K)=X
  370   CONTINUE
  380   CONTINUE
  390 CONTINUE
      RETURN
      END
c
c
C
c*********************************************************************
c
c  subroutine tpoly
c
**********************************************************************
C
C     subroutine for creating the curve fit coefficients
c
      subroutine tpoly(tmin,tmax,p)

      real tmin, tmax
      real p(*)

      p(1) = 1.
      p(2) = ((tan(tmax)-tan(tmin))/(tmax-tmin))-1.
      p(3) = ((tan(tmax)-tan(tmin))/(tmax-tmin))
     : +(((sin(tmax)*cos(tmax))-(sin(tmin)*cos(tmin)))/(2.*(tmax-tmin)))
     : -(3./2.)
       RETURN
       END
