C***********************************************************************
C                 copyright 2001, Amoco Production Company             *
C                             All Rights Reserved                      *
C                     an affiliate of BP America Inc.                  *
C***********************************************************************
c**********************************************************************c
c
c lmpicks reads seismic trace data from an input file,
c reads an ascii file of Landmark picks, extracts the pick time,
c stuffs the difference between this and some user defined datum, and
c writes the results to an output file

c It is assumed that the development location has afp - Amoco Fortran
c Preprocessor
c
c
c**********************************************************************c
c
c     declare variables
c
c-----
c    get machine dependent parameters
c    these includes are picked up and expanded into the code by afp

#include <f77/iounit.h>
#include <f77/lhdrsz.h>
#include <f77/sisdef.h>
c-----
 
c-----
c    SZLNHD is a value obtained from lhdrsz.h
c-----
      integer itr ( SZLNHD )
      integer     nsamp, nsi, ntrc, nrec, iform, obytes
      integer     luin , luout, lbytes, nbytes, lbyout
      integer     irs,ire,ns,ne
      integer     lines

      real        scl, t0

      character   iswd*6

c-----
c    SZSMPM is a value obtained from lhdrsz.h
c-----
      integer     argis
      integer DstSgn, ifmt_DstSgn, l_DstSgn, ln_DstSgn
      integer StaCor, ifmt_StaCor, l_StaCor, ln_StaCor
      integer RecNum, ifmt_RecNum, l_RecNum, ln_RecNum
      integer TrcNum, ifmt_TrcNum, l_TrcNum, ln_TrcNum
      integer StaWrd, ifmt_StaWrd, l_StaWrd, ln_StaWrd
      integer SrcLoc, ifmt_SrcLoc, l_SrcLoc, ln_SrcLoc
      integer SoPtNm, ifmt_SoPtNm, l_SoPtNm, ln_SoPtNm

      real        tri ( SZSMPM ), sp ( 10000 ), shift ( 10000 )

      character   ntap * 255, otap * 255, ptap * 255, name*7
      logical     verbos, query, samps
 
      data lbytes / 0 /
      data nbytes / 0 /
      data name/'LMPICKS'/
 
c-----
c     read program parameters from command line card image file
c-----
      query = ( argis ( '-?' ) .gt. 0 .or. argis ( '-h' ) .gt. 0)
      if ( query )then
            call help()
            stop
      endif
 
c-----
c     open printout files
c-----
#include <f77/open.h>
 
      call gcmdln(ntap,otap,ptap,ns,ne,irs,ire,samps,padval,
     1             trc1, sht1, trc2, sht2,scl,t0,iswd,verbos)

      open ( lun, file = ptap, status = 'old' )
c-----
c     get logical unit numbers for input and output of seismic data

c     input values are strings ntap & otap (i.e. names of files), and the
c     read 'r' or write 'w' designations of these files, and the default
c     logical unit numbers if ntap & otap are blank strings, i.e. ntap = " "
c     in which case:
c     0 = default stdin
c     1 = default stdout
c     Note: default values other that 0 & 1 may be used

c     output values are the logical unit numbers accessing these disk files. if
c     these values are less than 0 it means there was a fatal error in trying to
c     open these files
c-----
      call getln(luin , ntap,'r', 0)
      call getln(luout, otap,'w', 1)

c-----
c     read line header of inputa DSN (rtape reads data into vector "itr"
c     lbytes is the number of bytes actually read
c-----
      call rtape  ( luin, itr, lbytes)
      if(lbytes .eq. 0) then
         write(LOT,*)'LMPICKS: no header read from unit ',luin
         write(LOT,*)'FATAL'
         stop
      endif

c------
c     save certain parameters

c     to get and/or put LINE header values please use saver/savew
c     since that way code will always be portable to whatever machine
c     we maintain usp on
c     saver/w refer to header words using mnemonics rather than position

c     LINHED is a value in the include file <f77/sisdef.h> that refers to the
c     lineheader
c
c     see saver/w manual pages
c------
      call saver(itr, 'NumSmp', nsamp, LINHED)
      call saver(itr, 'SmpInt', nsi  , LINHED)
      call saver(itr, 'NumTrc', ntrc , LINHED)
      call saver(itr, 'NumRec', nrec , LINHED)
      call saver(itr, 'Format', iform, LINHED)
      call saver(itr, 'UnitSc', unitsc, LINHED)
      if (unitsc .eq. 0.0) then
          write(LERR,*)'********************************************'
          write(LERR,*)'WARNING: sample unit scaler in LH = ',unitsc
          write(LERR,*)'         will set to .001 (millisec default)'
          write(LERR,*)'********************************************'
          unitsc = .001
          call savew(itr, 'UnitSc', unitsc, LINHED)
      endif

      call savelu('DstSgn',ifmt_DstSgn, l_DstSgn, ln_DstSgn, 
     :     TRACEHEADER)
      call savelu('StaCor',ifmt_StaCor, l_StaCor, ln_StaCor, 
     :     TRACEHEADER)
      call savelu('RecNum',ifmt_RecNum, l_RecNum, ln_RecNum, 
     :     TRACEHEADER)
      call savelu('TrcNum',ifmt_TrcNum, l_TrcNum, ln_TrcNum, 
     :     TRACEHEADER)
      call savelu(iswrd, ifmt_StaWrd, l_StaWrd, ln_StaWrd,
     :     TRACEHEADER)
      call savelu('SrcLoc', ifmt_SrcLoc, l_SrcLoc, ln_SrcLoc,
     :     TRACEHEADER)
      call savelu('SoPtNm', ifmt_SoPtNm, l_SoPtNm, ln_SoPtNm,
     :     TRACEHEADER)

c------
c     hlhprt prints out the historical line header of length lbytes AND

c     hlhprt takes "name", in this case 4 characters long and stuffs this
c     into the modified historical line header and returns the NEW length
c     of the line header in lbytes
c------
      call hlhprt (itr, lbytes, name, 7, LERR)

c-----
c     ensure that command line values are compatible with data set
c     (i.e. start/end traces; start/end records).  This guards against
c     zero start values or values that are greater than those specified
c     in the line header
c-----
      call cmdchk(ns,ne,irs,ire,ntrc,nrec)

c-----
c     modify line header to reflect actual number of traces output
c-----
      nrecc=ire - irs+1
      call savew(itr, 'NumRec', nrecc, LINHED)
      jtr=ne-ns+1
      call savew(itr, 'NumTrc', jtr  , LINHED)

c----------------------
c  number output bytes
      obytes = SZTRHD + nsamp * SZSMPD

c----------------------
c  inject command line into
c  historical LH:
c  inputs are current line header in vector "itr" and length lbytes (bytes)
c  outputs are line header modified by insertion of the command line into
c  the historical LH and the modified length of the new LH (lbyout)

      call savhlh(itr,lbytes,lbyout)
c----------------------

c------
c     write to unit number luout lbyout bytes contained in vector itr
c------
      call wrtape ( luout, itr, lbyout   )

c-----
c     verbose output of all pertinent information before
c     processing begins
c-----
      if( verbos ) then
            call verbal(nsamp, nsi, ntrc, nrec, iform,
     1                  scl,t0,iswd,ptap,ntap,otap,
     2                  trc1, sht1, trc2, sht2, samps)
      end if

c--------------------------------------------------
c  compute sample interval in secs
c  take care of micro secs if necessary

      dt = real (nsi) * unitsc

c--------------------------------------------------

c-----
c     read picks, put in proper trace positions, & fix some
c     indexing

      ntot = nrec * ntrc
      call pickrd (lun,ntot,lines,sp,shift,trc1,sht1,trc2,sht2,padval)

      
c-----
c-----
c     BEGIN PROCESSING
c     read trace, put in shifts & other indices
c-----
c-----
c     skip unwanted records
c-----
      call recskp(1,irs-1,luin,ntrc,itr)

c-----
c     process desired trace records
c-----
      ic = 0
      do 1000 jj = irs, ire
 
c----------------------
c  skip to start trace
            call trcskp(jj,1,ns-1,luin,ntrc,itr)
c----------------------
 
            do 1001  kk = ns, ne

                  nbytes = 0
                  call rtape( luin, itr, nbytes)
c------
c     if end of data encountered (nbytes=0) then bail out
c     Note:  if you're processing records you might really want
c     to branch to the processing part rather than bailing out
c------
                  if(nbytes .eq. 0) then
                     write(LERR,*)'End of file on input:'
                     write(LERR,*)'  rec= ',jj,'  trace= ',kk
                     go to 999
                  endif

                  call saver2 ( itr, ifmt_DstSgn, l_DstSgn, ln_DstSgn, 
     :                 DstSgn, TRACEHEADER )
                  call saver2 ( itr, ifmt_StaCor, l_StaCor, ln_StaCor, 
     :                 StaCor, TRACEHEADER )
                  call saver2 ( itr, ifmt_RecNum, l_RecNum, ln_RecNum, 
     :                 RecNum, TRACEHEADER )
                  call saver2 ( itr, ifmt_TrcNum, l_TrcNum, ln_TrcNum, 
     :                 TrcNum, TRACEHEADER )

                  if (StaCor .eq. 30000) then
                     call vclr (tri,1,nsamp)
                  endif

                  ic = ic + 1
                  if (samps) then
                     StaWrd = (t0 - shift(ic))/nsi
                  else
                     StaWrd = (t0 - shift(ic))
                  endif

c write out header entries as required

                  call savew2 ( itr, ifmt_StaWrd, l_StaWrd, ln_StaWrd, 
     :                 StaWrd, TRACEHEADER )
                  SrcLoc = ifix(sp(ic) * 10.)
                  call savew2 ( itr, ifmt_SrcLoc, l_SrcLoc, ln_SrcLoc, 
     :                 SrcLoc, TRACEHEADER )
                  SoPtNm = sp(ic)
                  call savew2 ( itr, ifmt_SoPtNm, l_SoPtNm, ln_SoPtNm, 
     :                 SoPtNm, TRACEHEADER )

                  if (verbos)
     1            write(LERR,*)'Rec= ',RecNum,' trc= ',TrcNum,
     2                         ' time datum= ',t0,' pick= ',shift(ic),
     3                         ' static= ',StaWrd

                  call wrtape (luout, itr, obytes)


1001        continue

c----------------------
c  skip to end of record
            call trcskp(jj,ne+1,ntrc,luin,ntrc,itr)
c----------------------

 1000       continue

  999 continue

c-----
c     close data files
c     flush data left
c     in output buffer
c     NOTE: if the output buffer is not closed you can sometimes end up
c     with missing data
c-----
      call lbclos ( luin )
      call lbclos ( luout )

            write(LERR,*)'end of lmpicks, processed',nrec,' record(s)',
     :               ' with ',ntrc, ' traces'
      end
 
C***********************************************************************
      subroutine help
#include <f77/iounit.h>
         write(LER,*)
     :'***************************************************************'
        write(LER,*) ' '
        write(LER,*)
     :'lmpicks reads a LandMark pick file and stuffs the pick time'
        write(LER,*)
     :'into a static word, one pick per trace'
        write(LER,*)
     :'see manual pages for details ( online by typing uman lmpicks )'
        write(LER,*)' '
        write(LER,*)
     :'execute lmpicks by typing prgm and the of program parameters'
        write(LER,*)
     :'note that each parameter is proceeded by -a where "a" is '
        write(LER,*)
     :'a character(s) corresponding to some parameter.'
        write(LER,*)
     :'users enter the following parameters, or use the default values'
        write(LER,*)' '
        write(LER,*)
     :' -N [ntap]    (no default)         : input data file name'
        write(LER,*)
     :' -O [otap]    (no default)         : output data file name'
        write(LER,*)
     :' -P [ptap]    (no default)         : LandMark pick file name'
        write(LER,*)' '
        write(LER,*)
     :' -ns[ns]      (default = first)    : start trace number'
        write(LER,*)
     :' -ne[ne]      (default = last)     : end trace number'
        write(LER,*)
     :' -rs[irs]     (default = first)    : start record number'
        write(LER,*)
     :' -ne[ire]     (default = last)     : end record number'
        write(LER,*) ' '
        write(LER,*)
     :' -sw[iswd] (default = StaCor)      :  static Mnemonic'
        write(LER,*)
     :' -p[padval]  (default = -15999)    :  scale pick times'
        write(LER,*)
     :' -td[t0]   (default = 0)           :  time datum (ms)'
        write(LER,*)
     :' -sc[scl]  (default = 1.0)         :  scale pick times'
        write(LER,*)
     :' -ft[trc1] (default = 1.0)         :  first trace #'
        write(LER,*)
     :' -fs[sht1] (default = 1.0)         :  shot point at trc1'
        write(LER,*)
     :' -st[trc2] (default = 2.0)         :  second trace #'
        write(LER,*)
     :' -ss[sht2] (default = 2.0)         :  shot point at trc2'
        write(LER,*) ' '
        write(LER,*)
     :' -S  include on command line to output shift in samples'
        write(LER,*)
     :' -V  include on command line if verbose printout is desired'
        write(LER,*)
     :'usage:   lmpicks -N[ntap] -O[otap] -P[ptap] -ns[ns] -ne[ne]'
        write(LER,*)
     :'                  -rs[irs] -re[ire] -sw[iswd] -sc[] -td[t0]'
        write(LER,*)
     :'                  -ft[trc1] -fs[sht1] -st[trc2] -ss[sht2] -S -V'
        write(LER,*) ' '
        write(LER,*)
     :'***************************************************************'
      return
      end
 
C***********************************************************************
      subroutine gcmdln(ntap,otap,ptap,ns,ne,irs,ire,samps,padval,
     1                  trc1, sht1, trc2, sht2,scl,t0,iswd,verbos)
c-----
c     get command arguments
c
c     ntap  - C*255    input file name
c     otap  - C*255    output file name
c     ptap  - C*255   landmark pick file name
c     iswd  - C*6     static word
c     trc1  - R*4     first trace #
c     sht1  - R*4     first shot point associated w. trc1
c     trc2  - R*4     second trace #
c     sht2  - R*4     second shot point associated w. trc1
c     t0    - R*4     time datum
c     scl   - R*4     scale pick times
c     ns    - I*4      starting trace index
c     ne    - I*4      ending trace index
c     irs   - I*4      starting record index
c     ire   - I*4      ending record index
c     samps   L        verbose output or not
c     verbos  L        verbose output or not
c-----
#include <f77/iounit.h>
      character   ntap*(*), otap*(*), ptap*(*)
      integer     ns, ne, irs, ire 
      real        t0, scl, trc1, sht1, trc2, sht2
      character   iswd*(*)
      logical     verbos, samps
      integer     argis
 
c-------
c     import values from the command line using keys, e.g. -N
c     to which are immediately attached the users values.

c     For example program lmpicks might be invoked in the following way:

c     lmpicks  -Nxyz -Oabc

c     in which case xyz is a string (the name of the input data set)
c     which will be imported into lmpicks and associated with the variable
c     "ntap"

c     see manual pages on the argument handler routines
c     for the meanings of these functions.  Briefly though
c     we can import from the command line strings, integers, reals,
c     double precision reals, and check the existence of command line
c     keys.  The last 2 fields are the value to be assigned the variable
c     (1) if ONLY the key is present (no value attached to it), or
c     (2) if NO key & no value are present
c-------
            call argstr( '-N', ntap, ' ', ' ' )
            call argstr( '-O', otap, ' ', ' ' )
            call argstr( '-P', ptap, ' ', ' ' )
            if (ptap .eq. ' ') then
               write(LERR,*)'You must enter a LandMark pick file name'
               write(LERR,*)'using the -P[] command line entry'
               write(LERR,*)'re-run'
               stop
            endif
            call argi4 ( '-ns', ns ,   0  ,  0    )
            call argi4 ( '-ne', ne ,   0  ,  0    )
            call argi4 ( '-rs', irs ,   0  ,  0    )
            call argi4 ( '-re', ire ,   0  ,  0    )
            call argstr ( '-sw', iswd,  'StaCor', 'StaCor' )
            call argr4 ( '-td',  t0 ,  0.,  0.)
            call argr4 ( '-sc',  scl,  1.,  1.)
            call argr4 ( '-ft',  trc1,  1.,  1.)
            call argr4 ( '-fs',  sht1,  1.,  1.)
            call argr4 ( '-st',  trc2,  2.,  2.)
            call argr4 ( '-ss',  sht2,  2.,  2.)
            call argr4 ( '-p',  padval,  -15999.,  -15999.)
            samps  =   (argis('-S') .gt. 0)
            verbos =   (argis('-V') .gt. 0)

c-------
c     Note:  argis() = 1 if the key is present
c            argis() = 0 if it isn't
 
      return
      end
 
      subroutine verbal(nsamp, nsi, ntrc, nrec, iform,
     1                  scl,t0,iswd,ptap,ntap,otap,
     2                  trc1, sht1, trc2, sht2,samps)
c-----
c     verbose output of processing parameters
c
#include <f77/iounit.h>

      integer     nsamp, nsi, ntrc, nrec 
      real        t0,trc1, sht1, trc2, sht2
      character   ntap*(*), otap*(*), ptap*(*),iswd*(*) 
      logical     samps
 
            write(LERR,*)' '
            write(LERR,*)' line header values after default check '
            write(LERR,*) ' # of samples/trace =  ', nsamp
            write(LERR,*) ' sample interval    =  ', nsi
            write(LERR,*) ' traces per record  =  ', ntrc
            write(LERR,*) ' records per line   =  ', nrec
            write(LERR,*) ' format of data     =  ', iform
            write(LERR,*) ' Static Mnemonic    =  ', iswd
            write(LERR,*) ' Time to shift traces=  ',t0
            write(LERR,*) ' First Trace #       =  ', trc1
            write(LERR,*) ' First shot Point #  =  ', sht1
            write(LERR,*) ' Second Trace #      =  ', trc2
            write(LERR,*) ' Second shot Point # =  ', sht2
            write(LERR,*) ' Time pick scale factor  =  ',scl
            write(LERR,*) ' LandMark pick file name =  ', ptap
            write(LERR,*) ' input data set name =  ', ntap
            write(LERR,*) ' output data set name=  ', otap
            if (samps) then
              write(LERR,*) 'Output shift in units of samples'
            else
              write(LERR,*) 'Output shift in units of time(ms)'
            endif
            write(LERR,*)' '
            write(LERR,*)' '
 
      return
      end
 
