C***********************************************************************
C                 copyright 2001, Amoco Production Company             *
C                             All Rights Reserved                      *
C                     an affiliate of BP America Inc.                  *
C***********************************************************************
C**********************************************************************C 
C 
C     PROGRAM MODULE Matchf 
C 
C**********************************************************************C 
c 
c	Matchf does a phase matched filter rejection of ground roll
c-----
#include <save_defs.h>
#include <f77/iounit.h>
#include <f77/lhdrsz.h>
#include <f77/sisdef.h>

	integer    itr (SZLNHD )
	integer    lhed (SZLNHD )
	real       tri (SZSMPM)
	integer    nsamp, nsi, ntrc, nrec, iform
	integer    luin, luout,lbytes,nbytes,lbyout
	integer    rs, re, ns, ne
	character  ntap*256, otap*256, name*6
	logical    verbos
        logical    query
	integer    argis

	integer    niter
	real*4     uu,ul,fu,fl,t0,dist,dt,tt0,xx0
	character  namev*40
#include <f77/pid.h>

c	equivalence( itr(129), tri(1) )
	equivalence( itr(  1), lhed(1) )
	data lbytes/ 0 /, nbytes / 0 /, name/'MATCHF'/
c-----
c-----
c
c	read program paramters from command line 
c
c-----
	query = ( argis('-?').gt.0 .OR. argis('-h').gt.0 )
	if( query ) then
		call help()
		stop
	endif

c----------------------------------------
c  open printout file
#include <f77/open.h>
c----------------------------------------

	call gcmdln(ntap,otap,rs,re,ns,ne,
     1		niter,uu,ul,fu,fl,namev,verbos,xx0,tt0)
c-----
c	get logical unit numbers for input and output
c-----
	call getln(luin , ntap,'r', 0)
c	if(otap(1:1).ne.' ')then
		call getln(luout, otap,'w', 1)
c	else
c		luout = 1
c	endif
c------
c	read line header of input
c	save certain parameters
c-----
	call rtape ( luin, itr, lbytes)
        if(lbytes .eq. 0) then
           write(LERR,*)'MATCHF: no header read on unit ',ntap
           write(LERR,*)'FATAL'
           write(LERR,*)'Check existence of file & rerun'
           stop
        endif
	call saver(itr, 'NumSmp', nsamp, LINHED)
	call saver(itr, 'SmpInt', nsi  , LINHED)
	call saver(itr, 'NumTrc', ntrc , LINHED)
	call saver(itr, 'NumRec', nrec , LINHED)
	call saver(itr, 'Format', iform, LINHED)
	call saver(itr, 'UnitFl', iunit, LINHED)
      call saver(itr, 'UnitSc', unitsc, LINHED)
      if (unitsc .eq. 0.0) then
          write(LERR,*)'********************************************'
          write(LERR,*)'WARNING: sample unit scaler in LH = ',unitsc
          write(LERR,*)'         will set to .001 (millisec default)'
          write(LERR,*)'********************************************'
          unitsc = .001
          call savew(itr, 'UnitSc', unitsc, LINHED)
      endif

      call savelu('TrcNum',ifmt_TrcNum,l_TrcNum,ln_TrcNum,TRACEHEADER)
      call savelu('RecNum',ifmt_RecNum,l_RecNum,ln_RecNum,TRACEHEADER)
      call savelu('SrcLoc',ifmt_SrcLoc,l_SrcLoc,ln_SrcLoc,TRACEHEADER)
      call savelu('RecInd',ifmt_RecInd,l_RecInd,ln_RecInd,TRACEHEADER)
      call savelu('DphInd',ifmt_DphInd,l_DphInd,ln_DphInd,TRACEHEADER)
      call savelu('DstSgn',ifmt_DstSgn,l_DstSgn,ln_DstSgn,TRACEHEADER)
      call savelu('DstUsg',ifmt_DstUsg,l_DstUsg,ln_DstUsg,TRACEHEADER)
      call savelu('StaCor',ifmt_StaCor,l_StaCor,ln_StaCor,TRACEHEADER)
      call savelu('SoPtNm',ifmt_SoPtNm,l_SoPtNm,ln_SoPtNm,TRACEHEADER)

c-----
c	ensure that command line values are compatible with data set
c-----
	call cmdchk(ns,ne,rs,re,ntrc,nrec)
c-----
c     modify line header to reflect actual number of traces output
c-----
	nrecc = re - rs+1
	call savew(itr, 'NumRec', nrecc, LINHED)
	jtr   = ne-ns+1
	call savew(itr, 'NumTrc', jtr  , LINHED)
	call savhlh(itr,lbytes,lbyout)
	call wrtape ( luout, itr, lbyout                 )

c-----
c	verbose output here
c-----
c	if(verbos)then
		call verbal(ntap,otap,
     1		rs,re,ns,ne,
     2		nsamp,nsi,ntrc,nrec,iform,
     3		niter,uu,ul,fu,fl,namev,xx0,tt0)
c	end if
c-----
c	start processing
c-----
c	skip unwanted records
c-----
	call recskp(1,rs-1,luin,ntrc,itr)

c-----
c	convert t0 to secs & dt to secs
c-----
	    dt = real(NSI) * unitsc
            tt0 = tt0 * unitsc

c-----
c	process desired trace records
c-----
	do 1000 jj=rs,re
		k=0

		call trcskp(jj,1,ns-1,luin,ntrc,itr)

		do 1001 kk=ns,ne
			call rtape (luin, itr, nbytes)

                  if(nbytes .eq. 0) then
                     write(LERR,*)'End of file on input:'
                     write(LERR,*)'  rec= ',jj,'  trace= ',kk
                     go to 999
                  endif
                  call vmov (itr(ITHWP1), 1, tri, 1, nsamp)

                  call saver2(itr,ifmt_StaCor,l_StaCor, ln_StaCor,
     1                        ist  , TRACEHEADER)
                  call saver2(itr,ifmt_DstSgn,l_DstSgn, ln_DstSgn,
     1                        idist, TRACEHEADER)

                        idist = iabs(idist)
			dist = idist
			t0=0.0
			if(ist .ne. 30000)then
			call pmtch(tri,nsamp,niter,verbos,uu,ul,fu,fl,
     1				t0-tt0,dist-xx0,dt,namev)
                        else
                             call vclr (tri, 1, nsamp)
			endif

                  call vmov (tri, 1, itr(ITHWP1), 1, nsamp)
                  call wrtape(luout,itr,nbytes)
 1001		continue

		call trcskp(jj,ne+1,ntrc,luin,ntrc,itr)

 1000 	continue

999     continue

	call lbclos(luout)
	call lbclos(luin )
	end

	subroutine help
#include <f77/iounit.h>
	   write(LER,*)
     :'***************************************************************'
	write(LER,*)' '
	write(LER,*)
     :'Execute Matchf by typing Matchf and the following arguments'
        write(LER,*)' '
        write(LER,*)
     :' -N [ntap]    (default stdin)     : Input data file name'
        write(LER,*)
     :' -O [otap]    (default none)      : Output data file name'
           write(LER,*)
     :' -rs [rs],    (default=1)         : beginning record number'
           write(LER,*)
     :' -re [re],    (default=last)      : ending record number'
           write(LER,*)
     :' -ns[ns],	(default=1)      : starting trace number'
	write(LER,*)
     :' -ne[ne]		(default=all)    : ending trace number'
	write(LER,*)
     :' -V		(default = false): verbose output'
	write(LER,*)
     :' -uu[uu]       (default=4000)     : upper group velocity'
	write(LER,*)
     :' -ul[ul]       (default=1000)     : lower group velocity'
	write(LER,*)
     :' -fu[fu]       (default=5)        : frequency at uu'
	write(LER,*)
     :' -fl[fu]       (efault=20)        : frequency at ul'
	write(LER,*)
     :' -niter[niter] ((default=3)       : number of iterations'
	write(LER,*)
     :' -v[namev]     (default=null)     : dispersion file'
	write(LER,*)
     :' -x0[x0]       (default=0.0)      : distance shift'
	write(LER,*)
     :' -t0[t0]       (default=0.0)      : time shift'
	   write(LER,*)
     :'Usage: Matchf -N[ntap] -re[re] -rs[rs] -ns[ns] -ne[ne] ',
     :'-uu[uu] -fu[fu] -ul[ul] -fl[fl] -v[namev] -x0[x0] -t0[t0]'
	   write(LER,*)
     :'***************************************************************'
	return
	end

	subroutine gcmdln(ntap,otap,rs,re,ns,ne,
     1		niter,uu,ul,fu,fl,namev,verbos,x0,t0)
c-----
c	get command arguments
c
c	ntap	C*100	- input file name
c	otap	C*100	- output file name
c	ns	I*4	- starting trace index
c	ne	I*4	- ending trace index
c	rs	I*4	- starting record index
c	re	I*4	- ending trace index
c	niter	I*4	- number of iterations
c	uu	R*4	- upper group velocity
c	ul	R*4	- lower group velocity
c	fu	R*4	- frequency at upper group velocity
c	fl	R*4	- frequency at lower group velocity
c	namev	C*40	- dispersion file name
c	verbos	L	- verbos flag
c	x0	R*4	- distance shift
c	t0	R*4	- time shift
c-----
	character ntap*(*), otap*(*)
	integer*4 ns, ne,rs,re
	real*4 uu,ul,fu,fl,x0,t0
	character namev*(*)
	integer*4 niter
	logical verbos

		call argstr('-N', ntap, ' ', ' ' )
		call argstr('-O', otap, ' ', ' ' )
	        call argi4 ('-rs',rs,1,1)
	        call argi4 ('-re',re,0,0)
		call argi4 ('-ns',ns,1,1)
		call argi4 ('-ne',ne,0,0)
		call argr4 ('-ul',ul,1000.0,1000.0)
		call argr4 ('-uu',uu,4000.0,4000.0)
		call argi4 ('-niter',niter,3,3)
		call argr4('-fu',fu,5.0,5.0)
		call argr4('-fl',fl,20.0,20.0)
		call argr4('-t0',t0,0.0,0.0)
		call argr4('-x0',x0,0.0,0.0)
		call argstr('-v', namev, ' ', ' ')
		verbos = ( argis ('-V') .gt. 0 )
	return
	end

	subroutine verbal(ntap,otap,
     1		rs,re,ns,ne,
     2		nsamp,nsi,ntrc,nrec,iform,
     3		niter,uu,ul,fu,fl,namev,x0,t0)
c-----
c	get command arguments
c
c	ntap	C*100	- input file name
c	otap	C*100	- output file name
c	ns	I*4	- starting trace index
c	ne	I*4	- ending trace index
c	rs	I*4	- starting record index
c	re	I*4	- ending trace index
c	nsamp	I*4	- number of samples
c	nsi	I*4	- sample interval in ms
c	ntrc	I*4	- number of traces
c	nrec	I*4	- number of records
c	iform	I*4	- format code
c	niter	I*4	- number of iterations per trace
c	uu	R*4	- upper group velocity
c	fu	R*4	- frequency at upper group velocity
c	ul	R*4	- lower group velocity
c	fl	R*4	- frequency at lower group velocity
c	namev	C*40	- name of dispersion file
c	x0	R*4	- distance shift
c	t0	R*4	- time shift
c-----
#include <f77/iounit.h>
	character ntap*(*), otap*(*)
	integer*4 ns, ne,rs,re
	integer*4 nsamp,nsi,ntrc,nrec,iform
	integer*4 niter
	real*4 uu,ul,fu,fl,x0,t0
	character namev*(*)
	integer*4 lgstr

	write(LERR,*)' Values read from file command line'
	ii=lgstr(ntap,100)
	write(LERR,*)' Input  data file                        =  ',
     1		ntap(1:ii)
	ii=lgstr(otap,100)
	write(LERR,*)' Output data file                        =  ',
     1		otap(1:ii)
	write(LERR,*)' First record to process                 = ',rs
	write(LERR,*)' Last record to process                  = ',re
	write(LERR,*)' Starting trace number                   = ',ns
	write(LERR,*)' Ending   trace number                   = ',ne
	write(LERR,*)' Number of Samples/Trace                 = ',nsamp
	write(LERR,*)' Sample Interval (ms)                    = ',nsi
	write(LERR,*)' Traces per Record                       = ',ntrc
	write(LERR,*)' Records per Line                        = ',nrec
	write(LERR,*)' Format of Data                          = ',iform
	write(LERR,*)' Upper Group velocity                    = ',uu
	write(LERR,*)' Frequency at Upper Group Velocity       = ',fu
	write(LERR,*)' Lower Group velocity                    = ',ul
	write(LERR,*)' Frequency at Lower Group Velocity       = ',fl
        write(LERR,*)' Niter                                   = ',niter
	write(LERR,*)' Distance shift                          = ',x0
	write(LERR,*)' Time shift                              = ',t0
	if(namev.ne.' ')then
	ii=lgstr(namev,40)
	write(LERR,*)' Dispersion file                         = ',
     1		namev(1:ii)
	endif
	return
	end



	function lgstr(str,n)
c-----
c	function to find the length of a string
c	this will only be used with file system path names
c	thus the first blank 
c	indicates the end of the string
c-----
	character*(*) str
	integer*4 lgstr
	do 5010 i=1,n
		lgstr = i
		if(str(i:i).eq.' ')goto 100
 5010 	continue
  100   continue
        return
	end

	subroutine pmtch(tser,npts,niter,verby,uu,ul,fu,fl,
     1		tt0,ddist,ddt,namev)
c-----
c	main driver for phase matched rejection
c
c	tser	- R*4	time series array
c	npts	- I*4	number of points in original time series
c	niter	- I*4	number of iterations to perform
c	verby	- L	verbose output
c	uu	- R*4	group velocity upper limit
c	ul	- R*4	group velocity window lower limit
c	fu	- R*4	frequency at upper group velocity
c	fl	- R*4	frequency at lower group velocity
c	tt0	- R*4	time of initial sample in seconds
c	ddist	- R*4	distance of trace
c	ddt	- R*4	sampling interval in seconds
c	namev	- C*40	name of surf(IV) dispersion file
c-----
	implicit complex (z)
#include <f77/iounit.h>
	parameter(NP=8192,NP2=4096)
	dimension x(NP2),y(NP2),yf(NP2),yg(NP2),c(NP2)
	common/tdum/x,y,yf,yg,c,df,n,n21,t0,dist,ns
	common /pfplot/ paf(NP),waf(NP),npf,nb
	common /tvar/ zc(NP),z0(NP),dt,ntv,nti1,nti2,ntj1,ntj2
	logical verby
	integer*4 niter
	real*4 uu,ul,fu,fl,w,tt0,ddist,ddt
	real*4 tser(*)
	character namev*(*)
c-----
c	we now create a windowed time series
c	which we will work with
c------
c	set up number of points for FFT
c------
	call npow2(npts,n,n21)
c-----
c	put some parameters into the common block
c----
	dist = ddist
	dt = ddt
	t0 = tt0
	df = 1./(n*dt)
	do 100 i=1,n
		if(i.le.npts)then
			zc(i)   = cmplx(tser(i),0.0)
		else
			zc(i)   = cmplx(0.0,0.0)
		endif
  100	continue
c-----
c	if a dispersion file is present, make use of
c	it to define the dispersion limits
c	this will return som information here
c	and will communicate with vlstrt through
c	the common block
c-----
	call vlinit(namev,ul,uu,fl,fu,f0,c0)
c-----
c	window original time series so that only the signal within
c	this group velocity window is processed. The windowing
c	is flat between the velocity band and a Parzen taper
c	20 time samples on either side
c-----
	i1 = (dist/uu -t0)/dt +1
	i2 = (dist/ul -t0)/dt +1
	if(i1.gt.i2)then
		idum = i1
		i1 = i2
		i2 = idum
	endif
c-----
c	If the window defined by i1 and i2 lies outside the
c	original data set, do no further processing.
c	This will improve the speed of processing since not
c	every trace need be processed
c-----
	if(i2.lt.1 .or. i1.gt.npts)return
c-----
c	extend window one period on either side
c-----
	i1 = i1 - 1.0/(fu*dt)
	i2 = i2 + 1.0/(fl*dt)
	if(i1.lt.1)i1=1
	if(i2.gt.n)i2=n
	if(i1.gt.i2)then
		idum = i1
		i1 = i2
		i2 = idum
	endif
	call band(zc,n+n,i1,i2,20,0)
	call four(zc  ,n,-1,dt,df)
c-----
c	call subroutine to readin initial group velocity estimates
c	and then to compute first estimate of phase velocity curves
c-----
	call vlstrt(uu,ul,fu,fl,w,f0,c0)
c-----
c	phase match filter time series
c-----
	call pmatch(niter)
c-----
c	convert filtered spectra 
c	spectra to time series
c-----
 	call timser()
c-----
c	at this point zc is the time series
c	we window the phase matched filtered series 
c-----
	call band(zc,n+n,i1,i2,20,0)
	do 200 i=1,npts
		tser(i) = tser(i) - real(zc(i))
		zc(i) = cmplx(tser(i),0.0)
  200	continue
	return
	end

	subroutine npow2(nsamp,npts,npts21)
c-----
c	Given nsamp, find npts >= nsamp such that npts is a power of 2
c-----	
	integer*4 nsamp, npts, npts21
	npts = 1
 1000	continue
	npts = 2*npts
	if(npts.lt.nsamp)goto 1000
	npts21 = npts/2 + 1
	return
	end
 
	subroutine band(z,n,i1,i2,nb,nt)
		parameter(NP=8192,NP2=4096)
		complex z(NP)
		common/iband/ib1,ib2,ib3,ib4
c-----
c	apply a parzen window to the complex function z
c
c	z	- C*8	complex array to be windowed
c	n 	- I*4	number of points in the z array
c	i1,i2	- I*4	passband of windowing function in units
c			of array index
c			the original array is unchanged between i1 and i2
c	nb 	- I*4	taper width for parzen window about i1, i2
c			the array is zeroed for indices less than i1 - nb + 1
c			and for indices greater than    i2 + nb - 1
c	nt	- I*4	= 0 assume spectra input, and only window
c			the positive array frequencies, e.g.,
c			the first half of the array. This is
c			in fact not a true parzen window, but
c			we use parzen half-windows to taper the
c			band edges
c			= 1 true parzen window about the center of the array
c			used here for the pseudo-autocorrelation functions
c-----
c-----
c     define center point, corners, and check for valid 
c     choices of corners
c-----
	if(nt.eq.0)then
		m = n / 2 + 1
	else
		m = n
	endif
	if(nb.lt.2)then
		kl = 2
	else
		kl = nb
	endif
	ib1 = i1 - kl + 1
	if(ib1.lt.1)ib1=1
	ib2 = i1
	ib3 = i2
	ib4 = i2 + kl -1
	if(ib4.gt.m)ib4=m
c-----
	do 100 i = 1 , m
		if(i.lt.ib1)then
			z(i) = cmplx(0.0,0.0)
		elseif(i.ge.ib1 .and. i.lt.ib2)then
			fac = fparz(ib2-i,ib2-ib1)
			z(i) = fac * z(i)
		elseif(i.ge.ib2 .and. i.lt.ib3)then
			z(i) = z(i)
		elseif(i.ge.ib3 .and. i.lt.ib4)then
			fac = fparz(i-ib3,ib4-ib3)
			z(i) = fac * z(i)
		else
			z(i) = cmplx(0.0,0.0)
		endif
  100   continue
	return
	end

	function fparz(i,iw)
c-----
c	parzen windowing function
c
c	iw	- I*4	window halfwidth
c	i	- I*4	index within window
c			i = 0 corresponds to the passband = 1
c-----
		fi = i
		fiw = iw
		rat = abs(fi/fiw)
		if(2*i .lt. iw)then
			fparz = 1.0 -6.*rat*rat*(1.-rat)
		else
			fparz = 2.*(1.0-rat)**3
		endif
	return
	end

	subroutine unwrap(y,dw,i2)
c-----
c	unwrap phase from complex spectrum zc().
c	and return unwrapped phase in y()
c
c	y()	- R*4	array of unwrapped phase as a function of frequency
c	dw	- R*4	frequency sampling interval
c	i2	- I*4	upper limit of frequencies for processing
c	
c-----
	parameter(NP=8192,NP2=4096)
	real*4 y(NP2)
	complex zc,z0
	common /tvar/ zc(NP),z0(NP),dt,ntv,nti1,nti2,ntj1,ntj2
c-----
c	assume phase = 0.0 at zero frequency
c-----
		y(1)=0.0
		y(2)=atan2(aimag(zc(2)),real(zc(2)))
		y(3)=atan2(aimag(zc(3)),real(zc(3)))
c-----
c	calculate d(phase)/df
c-----
		do 30 i=4,i2
			j=i+1
			if(i.eq.i2)j=i
			cz0=cabs(zc(j))
			cz1=cabs(zc(j-1))
			cz2=cabs(zc(j-2))
			u1=real(zc(j))/cz0
			u2=real(zc(j-1))/cz1
			u3=real(zc(j-2))/cz2
			v1=aimag(zc(j))/cz0
			v2=aimag(zc(j-1))/cz1
			v3=aimag(zc(j-2))/cz2
			x=u2*(v1-v3)-v2*(u1-u3)
			if(i.eq.i2) then
				x=u1*(3.*v1-4.*v2+v3)-v1*
     1					(3.*u1-4.*u2+u3)
			endif
			y(i)=x/(2.*dw)
   30		continue
c-----
c	numerically integrate to get unwrapped phases
c	no checks are made by comparing the mod(unwrapped phase) to
c	atanf() phase from original spectrum
c-----
		yt=y(3)
		xt=0.
		do 120 i=4,i2
			yt=(y(i)+xt)*dw/2.+yt
			xt=y(i)
			y(i)=yt
  120		continue
c-----
c	force original spectrum to be zero
c-----
		zc(1)=cmplx(0.0,0.0)
	return
	end

	subroutine bfit(x,y,c,m,nfd)
c-----
c	use spline coefficients to define
c	fit to data
c
c	x	- array of abscissae
c	y	- array to be filled at x values
c	c	- array of spline coefficients
c	m	- number of points in spline
c	nfd	- 0 evaluate spline value at a point
c		- 1 estimate derivative at point
c		- 2 estimate integral at point
c-----
	parameter(NP2=4096)
	dimension x(NP2),y(NP2),c(NP2)
	common/bsp/b(NP2),q(4),h
c-----
c	definition of arithmetic statement functions
c-----
		p1(t)=.25*t**2*t
		p2(t)=-(1.-t)**2*(1.+t)*.75+1.
		cp1(t)=t**4/16.
		cp2(t)=t*(1./4.+t*(3./8.+t*(1./4.-3./16.*t)))
		dp1(t)=.75*t**2
		dp2(t)=.75+t*(1.5-2.25*t)
c-----
c	begin processing
c-----
		sum=0.
		suma=0.
		ifl=0
		jt=1
		do 110 i=1,m
   80			if(x(i).le.b(jt+1)) go to 90
				if(nfd.ge.2) then
					ifl=0
					fa=-(c(jt)+11.*c(jt+1))*h/16.
					fb=(11.*c(jt+2)+c(jt+3))*h/16.
					sum=sum+fb-fa
				endif
				jt=jt+1
			go to 80
   90			u=(x(i)-b(jt))/h
			if(nfd.ne.1 .and. nfd.ne.2)then
				q(1)=p1(1.-u)
				q(2)=p2(1.-u)
				q(3)=p2(u)
				q(4)=p1(u)
			elseif(nfd .eq. 1)then
				q(1)=-dp1(1.-u)/h
				q(2)=-dp2(1.-u)/h
				q(3)=dp2(u)/h
				q(4)=dp1(u)/h
			elseif(nfd .eq. 2)then
				if(ifl.ne.1) then
					ifl=1
					fa=-(c(jt)+11.*c(jt+1))*h/16.
					suma=sum-fa
				endif
				q(1)=-cp1(1.-u)*h
				q(2)=-cp2(1.-u)*h
				q(3)=cp2(u)*h
				q(4)=cp1(u)*h
			endif
			yfit=0.0
			do 100 l=1,4
				ic=jt-1+l
				yfit=yfit+c(ic)*q(l)
  100			continue
			y(i)=yfit+suma
  110		continue
	return
	end

	subroutine spline(x,y,c,m,nbp)
c-----
c	x	- R*4	array of abscissa
c	y	- R*4	array of ordinates
c	c	- R*4	array of cubic spline coefficients
c	m	- I*4	number of points in data
c	nbp	- I*4	number of break points, e.g., 2 = one spline
c			segment used, 3 = two segments
c-----
	parameter(NP=8192,NP2=4096)
	dimension x(NP2),y(NP2),g(NP2,5),c(NP2)
	common/bsp/b(NP2),q(4),h
c-----
c	arithmetic statment functions
c-----
	p1(t)=.25*t**2*t
	p2(t)=-(1.-t)**2*(1.+t)*.75+1.
c-----
c	begin processing
c-----
		zero=0.
		mdg=NP2
		nband=4
c-----
c	safety check on the number of spline segments
c-----
		if(nbp.gt.m-2) nbp=m-4
		if(nbp.lt.2) nbp=2
		nc=nbp+2
		b(1)=x(1)
		b(nbp)=x(m)
		h=(b(nbp)-b(1))/float(nbp-1)
		if(nbp.gt.2) then
			do 20 i=3,nbp
  				b(i-1)=b(i-2)+h
   20			continue
		endif
		ir=1
		ip=1
		i=1
		jt=1
  40		mt=0
  50			continue
				if(x(i).gt.b(jt+1)) go to 60
				u=(x(i)-b(jt))/h
				ig=ir+mt
				g(ig,1)=p1(1.-u)
				g(ig,2)=p2(1.-u)
				g(ig,3)=p2(u)
				g(ig,4)=p1(u)
				g(ig,5)=y(i)
				mt=mt+1
				if(i.eq.m) go to 60
				i=i+1
			go to 50
  60			continue
			call bndacc(g,mdg,nband,ip,ir,mt,jt)
			if(i.eq.m) go to 70
			jt=jt+1
		go to 40
  70		continue
		call bndsol(1,g,mdg,nband,ip,ir,c,nc,rnorm)
	return
	end

      subroutine bndacc(g,mdg,nb,ip,ir,mt,jt)
c-----
c     c.l. lawson and r.j. hanson, jet propulsion laboratory
c       sequential algorithm for banded least squares problem.
c     accumulation phase.   for solution phase use bndsol.
c
c     the calling program must set ir=1 and ip=1 before the
c       first call to bndacc for a new case.
c
c     the second subscript of g() must be dimensioned at least
c       nb+1 in the calling program.
c-----
      parameter(NP=8192,NP2=4096)
      dimension g(NP2,5)
      zero=0.
      nbp1=nb+1
      if(mt.le.0) return
      if(jt.eq.ip) go to 70
      if(jt.le.ir) go to 30
      do 10 i=1,mt
      ig1=jt+mt-i
      ig2=ir+mt-i
      do 10 j=1,nbp1
  10  g(ig1,j)=g(ig2,j)
      ie=jt-ir
      do 20 i=1,ie
      ig=ir+i-1
      do 20 j=1,nbp1
  20  g(ig,j)=zero
      ir=jt
  30  mu=min0(nb-1,ir-ip-1)
      if(mu.eq.0) go to 60
      do 50 l=1,mu
      k=min0(l,jt-ip)
      lp1=l+1
      ig=ip+l
      do 40 i=lp1,nb
      jg=i-k
  40  g(ig,jg)=g(ig,i)
      do 50 i=1,k
      jg=nbp1-i
  50  g(ig,jg)=zero
  60  ip=jt
  70  mh=ir+mt-ip
      kh=min0(nbp1,mh)
      do 80 i=1,kh
      call h12(1,i,max0(i+1,ir-ip+1),mh,g(ip,i),
     $           1,rho,g(ip,i+1),1,mdg,nbp1-i)
   80 continue
      ir=ip+kh
      if(kh.lt.nbp1) go to 100
      do 90 i=1,nb
  90  g(ir-1,i)=zero
 100  continue
      return
      end

      subroutine bndsol(mode,g,mdg,nb,ip,ir,x,n,rnorm)
c-----
c     c.l. lawson and r.j. hanson, jet propulsion laboratory
c     sequential solution of a banded least squares problem.
c     solution phase.  for the accumulation phase us bndacc.
c
c     the second subscript of g() must be dimensioned at
c     least nb+1 in the calling program
c-----
#include <f77/iounit.h>
      parameter(NP=8192,NP2=4096)
      dimension g(NP2,5),x(NP2)
      zero=0.
      rnorm=zero
      go to (10,90,50), mode
  10  do 20 j=1,n
  20  x(j)=g(j,nb+1)
      rsq=zero
      np1=n+1
      irm1=ir-1
      if(NP1.gt.irm1) go to 40
      do 30 j=np1,irm1
  30  rsq=rsq+g(j,nb+1)**2
      rnorm=sqrt(rsq)
  40  continue
  50  do 80 ii=1,n
      i=n+1-ii
      s=zero
      l=max0(0,i-ip)
      if(i.eq.n) go to 70
      ie=min0(n+1-i,nb)
      do 60 j=2,ie
      jg=j+l
      ix=i-1+j
  60  s=s+g(i,jg)*x(ix)
  70  if(g(i,l+1)) 80,130,80
  80  x(i)=(x(i)-s)/g(i,l+1)
      return
  90  do 120 j=1,n
      s=zero
      if(j.eq.1) go to 110
      i1=max0(1,j-nb+1)
      i2=j-1
      do 100 i=i1,i2
      l=j-i+1+max0(0,i-ip)
 100  s=s+x(i)*g(i,l)
 110  l=max0(0,j-ip)
      if(g(j,l+1)) 120,130,120
 120  x(j)=(x(j)-s)/g(j,l+1)
      return
 130  write(LOT,*)'zero diagonal term in bndsol'
c-----
c     cleanup
c-----
c-----
c     terminate program
c-----
      stop
      end

      subroutine h12 (mode,lpivot,l1,m,u,iue,up,c,ice,icv,ncv)
c-----
c     construction and/or application of a single
c     householder transformation..     q = i+u*(u**t)/b
c
c     mode    = 1 or 2   to select algorithm h1 or h2.
c     lpivot is the index of the pivot element.
c     l1,m   if l1 .le. m   the transformation will be constructed to
c            zero elements indexed from l1 through m.   if l1 gt. m
c            the subroutine does an identity transformation.
c     u(),iuw,up    on entry to h1 u() contains the pivot vector.
c                   iue is the storage increment between elements.
c                                       on exit from h1 u() and up
c                   contain quantities defining the vector u of the
c                   householder transformation.   on entry to h2 u()
c                   and up should contain quantities previously computed
c                   by h1.  these will not be modified by h2.
c     c()    on entry to h1 or h2 c() contains a matrix which will be
c            regarded as a set of vectors to which the householder
c            transformation is to be applied.  on exit c() contains the
c            set of transformed vectors.
c     ice    storage increment between elements of vectors in c().
c     icv    storage increment between vectors in c().
c     ncv    number of vectors in c() to be transformed. if ncv .le. 0
c            no operations will be done on c().
c-----
      parameter(NP=8192,NP2=4096)
      dimension u(iue,m),c(5*NP2)
      double precision sm,b
      one=1.
c
      if (0.ge.lpivot.or.lpivot.ge.l1.or.l1.gt.m) return
      cl=abs(u(1,lpivot))
      if (mode.eq.2) go to 60
c            ****** construct the transformation. ******
          do 10 j=l1,m
  10      cl=amax1(abs(u(1,j)),cl)
      if (cl) 130,130,20
  20  clinv=one/cl
      sm=(dble(u(1,lpivot))*clinv)**2
          do 30 j=l1,m
  30      sm=sm+(dble(u(1,j))*clinv)**2
c             convert dble. prec. sm to sngl. prec. sm1
      sm1=sm
      cl=cl*sqrt(sm1)
      if (u(1,lpivot)) 50,50,40
  40  cl=-cl
  50  up=u(1,lpivot)-cl
      u(1,lpivot)=cl
      go to 70
c     ****** apply the transformation 1+u*(u**t)/b to c. ******
c
  60  if (cl) 130,130,70
  70  if (ncv.le.0) return
      b=dble(up)*u(1,lpivot)
c          b  must be nonpositive here.  if b=0., return.
c
      if (b) 80,130,130
  80  b=one/b
      i2=1-icv+ice*(lpivot-1)
      incr=ice*(l1-lpivot)
          do 120 j=1,ncv
          i2=i2+icv
          i3=i2+incr
          i4=i3
          sm=c(i2)*dble(up)
              do 90 i=l1,m
              sm=sm+c(i3)*dble(u(1,i))
  90          i3=i3+ice
          if (sm) 100,120,100
 100      sm=sm*b
          c(i2)=c(i2)+sm*dble(up)
              do 110 i=l1,m
              c(i4)=c(i4)+sm*dble(u(1,i))
 110          i4=i4+ice
 120      continue
 130      return
          end
	
	subroutine inter(sx,sy,x,y,x0,dx,num)
c-----
c
c	sx	- R*4	input x values unevenly spaced
c	sy	- R*4	input y values
c	x	- R*4	output evenly spaced abscissa
c	y	- R*4	output ordinates
c	x0	- R*4	desired first abscissa
c	dx	- R*4	desired abscissa interval
c	num	- I*4	number of points in (sx,sy)
c			on return this is the number of points
c			in (x,y)
c-----
	parameter(NP=8192,NP2=4096)
	real*4 sx(NP2),sy(NP2),x(NP2),y(NP2),x0,dx
	real*4 xx(4),yy(4)
	integer*4 num
		x(1)=x0
		i=1
c-----
c	use linear interpolation at first interval
c-----
  113		continue
			if (sx(2).lt.x(i)) go to 114
			s1=(x(i)-sx(1))/(sx(2)-sx(1))
			y(i)=s1*(sy(2)-sy(1))+sy(1)
			i=i+1
			x(i)=x(i-1)+dx
		go to 113
c-----
c	use piecewise cubic interpolation in between
c-----
  114		continue
		n1=num-2
		do 120 j=2,n1
			j1=j-1
			do 118 k=1,4
				xx(k)=sx(j1+k-1)
				yy(k)=sy(j1+k-1)
  118			continue
  119			continue
				if (sx(j+1).lt.x(i)) go to 120
  220				call pccp(xx,yy,x(i),y(i))
				i=i+1
				x(i)=x(i-1)+dx
			go to 119
  120		continue
c-----
c	use linear interpolation over last data interval
c-----
  121		continue
			if (sx(num).lt.x(i)) go to 122
			s1=(x(i)-sx(num-1))/(sx(num)-sx(num-1))
			y(i)=s1*(sy(num)-sy(num-1))+sy(num-1)
			i=i+1
			x(i)=x(i-1)+dx
		go to 121
  122		num=i-1
	return
	end
	
        subroutine pccp (x,y,xd,yd)
c-----
c     pccp performs piecewise continuous cubic polynomial interpolation.
c     the method requires two x and y points on each side of position xd
c     where the value of yd is determined. weighted averages of the slopes
c     are determined at the knots (positions x(2) and x(3)), and the slopes
c     and the values x(2), y(2), x(3), y(3) are used to determine yd which
c     corresponds to x position xd which falls between x(2) and x(3).
c     references
c     wiggins,r.a.,bull. seism. soc. am.,v.66,p2077-2081,1976.
c-----
        real*4 x(4),y(4)
	real*4 xd, yd
        eps=0.001
c-----
c     determine slopes at x(2) and x(3)
c-----
               sx2=(y(2)-y(1))/(x(2)-x(1))
               sx3=(y(3)-y(2))/(x(3)-x(2))
               sx4=(y(4)-y(3))/(x(4)-x(3))
c-----
c     weight slopes.
c-----
               w1=1.0/amax1(abs(sx2),eps)
               w2=1.0/amax1(abs(sx3),eps)
               w3=1.0/amax1(abs(sx4),eps)
               s2=(w2*sx3+w3*sx4)/(w2+w3)
               s1=(w1*sx2+w2*sx3)/(w1+w2)
c-----
c     evaluate polynomial at x=xd
c     with slopes given at x(2) and x(3)
c-----
               p0=y(2)
               p1=s1
               p2=(3.0*(y(3)-y(2))/(x(3)-x(2))-2.0*s1-s2)/(x(3)-x(2))
               p3=(s1+s2-2.0*(y(3)-y(2))/(x(3)-x(2)))/((x(3)-x(2))**2)
               xmxd=xd-x(2)
               yd = p0 + xmxd*(p1 + xmxd*(p2 + xmxd*p3 ) )
        return
        end

	subroutine vlinit(namev,ul,uu,fl,fu,f0,c0)
c-----
c	read contents of the file 'namev' if it exists
c	return lower and upper group velocity limits
c	pass information to vlstrt concerning the dispersion curves
c-----
	parameter(NP=8192,NP2=4096)
	real*4 ul,uu
	character namev*(*)
	dimension x(NP2),y(NP2),yf(NP2),yg(NP2),c(NP2)
	common/tdum/x,y,yf,yg,c,df,n,n21,t0,dist,ns
	common /dpplot/ 
     $                ml,mh,m
c-----
c	check to see if a dispersion file has been specified
c-----
	if(namev.eq.' ')then
c-----
c	generate a pseudo dispersion curve
c	assuming the phase velocity slowness is the mean of
c	the group velocity slownesses
c-----
		m=20
		vi =  0.5*(1./uu + 1./ul)
		fc = 0.5*(fu + fl)
		dff = (fu-fl)/(m-1)
		duu = (1./uu - 1./ul)/(m-1)
		do 26 i=1,m
			f = fl + (i-1)*dff
			u = 1./ul + (i-1)*duu
			x(i) = 1./f
			y(i) = 1./u
   26		continue
		f0=fc
		c0 = 1./vi
	else
c-----
c	determine if phase velocity data are present
c-----
		c0 = -1.0
		f0 = -1.0
		moffst=0
		call getvel(x,y,1,m,moffst,namev)
		if(m.gt.1)then
			call fitvel(m,i1,i2,nbp)
c-----
c	now x is in units of angular frequency
c	do a spline fit to get group velocity from
c	phase velocity
c-----
			nfd = 1
			call bfit(x,yg,c,m,nfd)
			do 100 i=1,m
				uvel = yf(i)/(1.0- x(i)*yg(i)/yf(i))
				x(i)=6.2831853/x(i)
				y(i)=uvel
  100			continue
c-----
c	get a reference phase velocity point
c-----
			c0 = yf(m/2)
			f0 = 1./x(m/2)
		endif
		moffst=m
c-----
c	if phase velocity data are present
c	spline fit and compute corresponding group velocity data
c-----
		call getvel(x,y,2,m,moffst,namev)
		call sort(x,y,m)
		call maxmin(y,m,ul,uu)
c-----
c	if there is no phase velocity data use simple
c	subterfuge
c-----
		vi =  0.5*(1./uu + 1./ul)
		fc = 0.5*(fu + fl)
		c0 = 1./vi
	endif
c-----
c	for safety force uu to be thehigher velocity
c-----
	if(ul.gt.uu)then
		tmp=uu
		uu=ul
		ul=tmp
		tmp=fu
		fu=fl
		fl=tmp
	endif
	return
	end

	subroutine maxmin(y,m,ymin,ymax)
c-----
c	determine extremal values of an array
c
c	y	- R*4	array of values
c	m	- I*4	number of points in array
c	ymin	- R*4	minimum value
c	ymax	- R*4	maximum value
c-----
	real*4 y(*), ymin, ymax
	integer*4 m
		ymax = -1.0e+38
		ymin = -ymax
		do 100 i=1,m
			if(y(i).gt.ymax)ymax=y(i)
			if(y(i).lt.ymin)ymin=y(i)
  100		continue
	return
	end

	subroutine getvel(x,y,iphgp,m,moffst,namev)
c-----
c	read unit 2, containing dispersion information
c	returning either phase or group velocities
c
c	x	- R*4	array of periods
c	y	- R*4	array of velocity values
c	iphgp	- I*4	=1 phase velocity
c			=2 group velocity
c	m	- I*4	number of velocity points returned
c	moffst	- I*4	offset into arrays at which to
c			add additional data
c	namev	- C*	name of velocity file
c-----
#include <f77/iounit.h>
	real*4 x(*),y(*)
	integer*4 iphgp,m,moffst
	character namev*(*)
		open(2,file=namev,status='old')
		rewind 2
c------
c	input initial group velocity measurements using surf(IV)
c	the file can only contain one wave type and one mode
c-----
		read(2,*)iunit,ifrpr
		ilrold = 0
		imdold = 0
		m=moffst
   25		continue
			read(2,*,end=100,err=100)ilr,icvg,imd,prfr,vel
			if(icvg.ne.iphgp)goto 25
			m = m + 1
			y(m) = vel
			if(ifrpr.eq.0)then
				x(m) = prfr
			else
				x(m) = 1./prfr
			endif
			if(m.gt.(moffst+1))then
				if(imdold.ne.imd )then
					write(LOT,*)'MULTIMODE DATA',
     1						' NOT PERMITTED'
					ierr = 1
				elseif(ilrold.ne.ilr)then
					write(LOT,*)'MIXED LOVE-RAYL',
     1						' NOT PERMITTED'
					ierr = 1
				else
					ierr = 0
				endif
			else
				ilrold = ilr
				imdold = imd
			endif
			if(ierr .eq. 1)stop
			goto 25
  100		continue
		close(2)
	return
	end
	
	subroutine vlstrt(uu,ul,fu,fl,w,f0,c0)
	implicit complex (z)
#include <f77/iounit.h>
	parameter(NP=8192,NP2=4096)
	real*4 uu,ul,fu,fl,w
	dimension x(NP2),y(NP2),yf(NP2),yg(NP2),c(NP2)
	common/tdum/x,y,yf,yg,c,df,n,n21,t0,dist,ns
	common /dpplot/ 
     $                ml,mh,m
	common /pfplot/ paf(NP),waf(NP),npf,nb
	common /tvar/ zc(NP),z0(NP),dt,ntv,nti1,nti2,ntj1,ntj2
	common/vwind/i1,i2,xmx,ii1,ii2
	common/iband/ib1,ib2,ib3,ib4
	tupi = 2.* 3.141592654
	dw = tupi*df
c-----
c	now sort the period-group velocity in order of
c	increasing period
c
c	obtain xmx, indicating the longest period of interest
c	this is very important for band passing the signal
c-----
	call sort(x,y,m)
	xmx=x(m)
	nb=int(1.5*xmx/dt) + 1
	call fitvel(m,i1,i2,nbp)
c-----
c	enter band edges for spectrum
c
c	ns - number of array points for band edge (default = 10)
c	nt - set to zero for positive spectrum band filter
c-----
	nt=0
	n21=n/2+1
	ns=10
	call band(zc,n,i1,i2,ns,nt)
c-----
c	extend group velocity data by a linear extension of the
c	data endpoints slownesses (this is done to minimize edge effects).
c
c	ii1 - lower passband limit (in terms of frequency array location)
c	ii2 - upper passband limit
c	i1 - extended lower passband limit
c	i2 - extended upper passband limit
c	m  - number of group velocity points in extended passband
c	ml - lower passband limit (in terms of period array location)
c	mh - upper passband limit
c-----
	j1=i1-ns+2
	j2=i2+ns-2
	if(j1.lt.2) j1=2
	if(j2.gt.n21) j2=n21-1
	ml=i1-j1
	mh=j2-i2
	sl=1./yf(2)-1./yf(1)
	sh=1./yf(m)-1./yf(m-1)
	yl=yf(1)
	yh=yf(m)
	do 200 i=1,m
		j=m-i+1
		x(j+ml)=x(j)
		yf(j+ml)=yf(j)
 200	continue
	do 210 i=1,ml
		dx=float(i-1-ml)
		x(i)=x(ml+1)+dx*dw
		yf(i)=1./(sl*dx+1./yl)
 210	continue
	do 220 i=1,mh
		dx=float(i)
		x(m+ml+i)=x(m+ml)+dx*dw
		yf(m+ml+i)=1./(sh*dx+1./yh)
 220	continue
	mm=m
	m=m+ml+mh
	ml=ml+1
	mh=m-mh
	mll=ml
	ml=m-mh+1
	mh=m-mll+1
	ii1=i1
	i1=j1
	ii2=i2
	i2=j2
	ntv=n
	nti1=i1
	nti2=i2
	ntj1=ii1
	ntj2=ii2
c-----
c	end of band extension
c-----
	do 230 i=1,m
		y(i)=dist/yf(i)-t0
  230	continue
c-----
c	refit group velocity spline with extended data
c-----
	call spline(x,y,c,m,nbp)
	call bfit(x,yf,c,m,nfd)
	do 260 i=1,m
		y(i1+i-1)=yf(i)
  260	continue
c
c     integrate spline fit of group delay for phase estimate
c
c     nfd=2 for spline integral
c
      nfd=2
      call bfit(x,yf,c,m,nfd)
c
c     calculate integration constant (ctg) based on reference
c     period and phase velocity (c0,f0)
c
      w=tupi*f0
      ic=int(w/dw)+1
      if(ic.gt.i2) ic=i2
      if(ic.lt.i1) ic=i1
      ic=ic-i1+1
      ctg=x(ic)*(dist/c0-t0)-yf(ic)
      do 270 i=1,m
  270 yf(i)=yf(i)+ctg
      j=1
      do 300 i=i1,i2
      x(i)=yf(j)
      yg(j)=float(i-1)*dw
  300 j=j+1
c-----
c     fill in negative frequency component of filtered spectrum
c-----
	do 325 i=2,n21
		j = n + 2 - i
		zc(j) = conjg(zc(i))
  325	continue
	return
	end

	subroutine fitvel(m,i1,i2,nbp)
c-----
c	take raw dispersion data and least squares spline fit it
c
c	m	- I*4	number of points in dispersion array
c	i1	- I*4	minimum fourier frequency
c	i2	- I*4	maximum fourier frequency
c	nbp	- I*4	number of breakpoints in spline
c-----
	parameter(NP=8192,NP2=4096)
	dimension x(NP2),y(NP2),yf(NP2),yg(NP2),c(NP2)
	common/tdum/x,y,yf,yg,c,df,n,n21,t0,dist,ns
		tupi=6.283185308
		dw = tupi*df
		nbp=m
c-----
c	fit wiggin's spline interpolator to original group velocity
c	data in order to calculate group velocity in terms of frequency
c	sampled increments
c-----
		nfd=0
		do 170 i=1,m
			j=m-i+1
			yg(j)=tupi/x(i)
			yf(j)=y(i)
  170		continue
		ntw=0
		i1=int((yg(1)/dw)*1.00001)+1
		w1=float(i1-1)*dw
		call inter(yg,yf,x,y,w1,dw,m)
c-----
c	inter changes m, use this to update i2
c-----
		i2=i1+m-1
c-----
c	output of wiggin's spline in x,y.  find least squares cubic
c	spline coefficients to x,y (spline) and store in c.  fit spline
c	to y (bfit) and store in yf.
c-----
c	nbp - number of breakpoints
c	nfd - set to zero for function fit
c-----
		if(nbp.lt.0) nbp=m-4
		call spline(x,y,c,m,nbp)
		call bfit(x,yf,c,m,nfd)
	return
	end

      subroutine sort(x,y,no)
c-----
c     This provides a sort of items
c     x(i) is the array to be sorted
c     which has associated with it an y(i) array
c     no is the number of points to be sorted
c     After the sort the x(i),y(i) pairs will be ordered
c     from the least to the largest
c-----
      real*4 x(1),y(1)
      mo=no
    2 if(mo-15)21,21,23
   21 if(mo-1)29,29,22
   22 mo=2*(mo/4)+1
      goto 24
   23 mo=2*(mo/8)+1
   24 ko=no-mo
      jo=1
   25 i=jo
   26 if(x(i)-x(i+mo))28,28,27
   27 temp=x(i)
      x(i)=x(i+mo)
      x(i+mo)=temp
      yemp=y(i)
      y(i)=y(i+mo)
      y(i+mo)=yemp
      i=i-mo
      if(i-1)28,26,26
   28 jo=jo+1
      if(jo-ko)25,25,2
   29 return
      end

	subroutine pmatch(niter)
	implicit complex (z)
#include <f77/iounit.h>
	parameter(NP=8192,NP2=4096)
	dimension x(NP2),y(NP2),yf(NP2),yg(NP2),c(NP2)
	common/tdum/x,y,yf,yg,c,df,n,n21,t0,dist,ns
	common /dpplot/ 
     $		ml,mh,m
	common /pfplot/ paf(NP),waf(NP),npf,nb
	common /tvar/ zc(NP),z0(NP),dt,ntv,nti1,nti2,ntj1,ntj2
	common/vwind/i1,i2,xmx,ii1,ii2
	integer*4 niter
c-----
c	calculate pseudo-autocorrelation function (zc) by removing
c	estimate of phase (x) from observed spectrum (z0)
c-----
	tupi = 2.* 3.141592654
	dw = tupi*df
	ts=float(n)*dt/2.
	iter=0
c-----
c	save copy of original spectrum
c----
	do 339 i=1,n
		z0(i) = zc(i)
  339	continue
c-----
c	undisperse waveform, and then time shift so
c	that zero lag will appear at the center of the
c	time series
c-----
  340	continue
		zc(1)=cmplx(0.,0.)
		do 345 i=2,n21
			tp=ts*(i-1)*dw
			zr=cexp(cmplx(0.,x(i)-tp))
			zc(i)=z0(i)*zr
			zc(n+2-i)=conjg(zc(i))
  345		continue
		zc(n21)=cmplx(real(zc(n21)),0.)
c-----
c	fourier transform zc() to time domain (ifr=1)
c-----
		call four(zc,n,+1,dt,df)
c-----
c	time window the pseudo-autocorrelation function
c	with a Parzen window. 
c	when done use a cosine taper instead
c-----
		if(iter.lt.niter)then
			nt=1
			call band(zc,n,n21,n21,nb,nt)
		else
			nbx = int(2.0*xmx/dt) + 1
			call cwind(zc,n,n21,nbx)
		endif
c-----
c	fourier transform back to frequency domain and reshift
c	zero lag back to its correct position.
c-----
		call four(zc,n,-1,dt,df)
		zc(1)=cmplx(0.,0.)
		do 450 i=2,n21
			tp=ts*(i-1)*dw
			zp=cexp(cmplx(0.,tp))
			zc(i)=zc(i)*zp
			zc(n+2-i)=conjg(zc(i))
  450		continue
		zc(n21) = cmplx(real(zc(n21)),0.0)
c-----
c	exit program here if at last iteration
c-----
		if(iter.eq.niter)return
		iter = iter + 1
c-----
c	unwrap the phase spectrum by numerical differentiation and
c	integration.  check the subroutine for the location of
c	plotting the group delay error and interpolating it for
c	getting rid of "spikes" due to spectral nulls.
c
c	unwrapped phase is returned in (yf).
c-----
		call unwrap(yf,dw,i2)
		j=1
		do 500 i=i1,i2
			yf(j)=yf(i)
			j=j+1
  500		continue
c-----
c	enter number of breakpoints for least squares spline fit
c	to the unwrapped phase.
c-----
		nfd=0
		nbp=5
		if(nbp.lt.0) nbp=m-4
		call spline(yg,yf,c,m,nbp)
c-----
c	permanently update phase (x) and group delay (y) file
c-----
		nfd=0
		call bfit(yg,yf,c,m,nfd)
		do 550 j=i1,i2
			x(j)=x(j)-yf(j-i1+1)
  550		continue
c-----
c     start next iteration
c-----
		go to 340
	end

	subroutine cwind(zc,n,n21,nb)
c-----
c	cosine window a band nb points wide about
c	the center of the array
c
c	zc	- C*8	complex array
c	n	- I*4	number of points in array
c	n21	- I*4	index of Nyquist frequency
c	nb	- I*4	bandwidth of taper region
c-----
	complex zc(*)
	integer*4 n, n21, nb
		ll = n21 - nb
		lr = n21 + nb
		if(ll.lt.1)ll=1
		if(lr.gt.n)lr=n
		do 100 i=1,n
			if(i.lt.ll.or.i.gt.lr)then
				fac = 0.0
			else
				rw = real(i-n21)/real(nb)
				rw = rw*1.57079633
				fac = cos(rw)
			endif
			zc(i) = zc(i) * fac
  100		continue
	return
	end

	subroutine timser
c-----
c	window spectra
c	disperse, convert to time series
c-----
	implicit complex (z)
	parameter(NP=8192,NP2=4096)
	dimension x(NP2),y(NP2),yf(NP2),yg(NP2),c(NP2)
	common/tdum/x,y,yf,yg,c,df,n,n21,t0,dist,ns
	common /tvar/ zc(NP),z0(NP),dt,ntv,nti1,nti2,ntj1,ntj2
	common/vwind/i1,i2,xmx,ii1,ii2
		nt=0
		call band(zc,n,ii1,ii2,ns,nt)
		do 100 i=1,n21
			if(i.ge.i1 .and. i.le.i2)then
				zr=cexp(cmplx(0.,-x(i)))
				zc(i)=zc(i)*zr
			endif
			if(i.gt.1)zc(n+2-i)=conjg(zc(i))
  100		continue
		call four(  zc,n,1,dt,df)
	return
	end

