C***********************************************************************
C                 copyright 2001, Amoco Production Company             *
C                             All Rights Reserved                      *
C                     an affiliate of BP America Inc.                  *
C***********************************************************************
c**********************************************************************c
c      radon3d performs 3d dip filtering in the radon transform
c      (tau,p,q) domain using a precomputed operator.
c**********************************************************************c
c______________________________________________________________________
c      Kurt J. Marfurt (Amoco Production Research, Tulsa, OK, USA)
c      August 16, 1994.
c______________________________________________________________________
c
c     declare variables
c
#include <f77/iounit.h>
#include <f77/lhdrsz.h>
#include <f77/sisdef.h>
c______________________________________________________________________
c     parameters needed for dynamic memory allocation routine 'galloc'.
c______________________________________________________________________
      parameter (maxs=0 000 002)
      dimension s(maxs)
      pointer   (pntrs,s)
c______________________________________________________________________
      parameter (maxomega=8192)
      parameter (maxp=200,maxq=200)             
      parameter (maxmx=51,maxmy=51,maxrec=2001)
      parameter (minswgt=0,maxswgt=100)
      parameter (minawgt=-360,maxawgt=+360)
      real      wgtpq(maxp*maxq)
      real      plive(maxp*maxq)
      real      qlive(maxp*maxq)
      real      slive(maxp*maxq)
      integer   jpqlive(maxp*maxq)
      real      swgt(minswgt:maxswgt)
      real      awgt(minawgt:maxawgt)
      real      pwgt(-maxswgt:maxswgt)
      real      qwgt(-maxswgt:maxswgt)
c
      parameter   (pi=3.1415926,twopi=2.*pi)
      parameter   (ztol=.001)
      parameter   (eps2=1.e-20,rnull=-.001)  
      integer     iypointer(-maxmy:maxrec+maxmy)     
      integer     kypointer(0:maxrec)     
      integer     locate(2,(2*maxmy+1)*(2*maxmx+1))
      real        wgtxy((2*maxmy+1)*(2*maxmx+1))
      real        x(2*maxmx+1)
      real        y(2*maxmy+1)
      real        xwgt(2*maxmx+1)
      real        ywgt(2*maxmy+1)
      real        p(2*maxp+1)
      real        q(2*maxq+1)
      real        omega(maxomega)
c
      integer     hbegin
      integer     startline,endline,starttrace,endtrace
      integer     sheader(SZLNHD)
      real        cputim(20),waltim(20)
#include <f77/pid.h>
#include <save_defs.h>
c
      character*200  file_in,file_out
      character*6    domain
      character*7    name
      logical        verbose, query  ,depth_section,eof,reject
      logical        in_line_filtering
      logical        interpolate
      logical        fwd_drt,rev_drt,lsq_rev
      logical        toeplitz,svd,rhofilt
      integer        argis
      integer        stderr
      logical        padded
      logical        true_dip
 
      data           name/'RADON3D'/
c
      stderr=ler
c_______________________________________________________________
c     initialize timing arrays.
c_______________________________________________________________
      call vclr(cputim,1,20)
      call vclr(waltim,1,20)
      call timstr(vtot,wtot)
      call timstr(va,wa)
c_______________________________________________________________
c     read program parameters from command line card image file
c_______________________________________________________________
      query=(argis( '-?' ) .gt. 0 )
      if(query) then
         call help(ler)
         call exit(0)
      endif
c_______________________________________________________________
c     open printout files
c_______________________________________________________________
#include <f77/open.h>
c_______________________________________________________________
c     read i/o data set names from the command line.
c_______________________________________________________________
      call argstr('-N',file_in, ' ', ' ')
      call argstr('-O',file_out, ' ', ' ')
c_______________________________________________________________
c     get logical unit numbers for input and output
c_______________________________________________________________
      call getln(luin,file_in,'r',0)
      call getln(luout, file_out,'w', 1)
c_______________________________________________________________
c     read line header of input
c     save certain parameters
c_______________________________________________________________
      lbytes=0
      call rtape(luin,sheader,lbytes)
      if(lbytes .eq. 0) then
         write(lerr,*)'RADON3D: no header read from unit ',luin
         call exit(0)
      endif
      call hlhprt(sheader,lbytes,name,len(name),lerr)
c_______________________________________________________________________
c     pull relevent values from line header.
c_______________________________________________________________________
      call saver(sheader,'NumSmp',nsamp_in,LINEHEADER)
      call saver(sheader,'SmpInt',nsi,LINEHEADER)
      call saver(sheader,'NumRec',nrec,LINEHEADER)
      call saver(sheader,'NumTrc',ntr,LINEHEADER)
      call saver(sheader,'ILClIn',dxh,LINEHEADER)
      call saver(sheader,'CLClIn',dyh,LINEHEADER)
      call saver(sheader,'AziIln',xazimh,LINEHEADER)
      call saver(sheader,'AziXln',yazimh,LINEHEADER)
      call saver(sheader,'Dz1000',idz_1000,LINEHEADER)
      call saver(sheader,'TmMsFS',tstart_orig,LINEHEADER)
      call saver(sheader,'TmMsSl',jtstart_orig,LINEHEADER)
      call saver(sheader,'TmSlIn',jtslice_orig,LINEHEADER)
      call saver(sheader,'Nx_Pad',nxpad,LINEHEADER)
      call saver(sheader, 'UnitSc', unitsc, LINEHEADER)
      if (unitsc .eq. 0.0) then
          write(LERR,*)'********************************************'
          write(LERR,*)'WARNING: sample unit scaler in LH = ',unitsc
          write(LERR,*)'         will set to .001 (millisec default)'
          write(LERR,*)'********************************************'
          unitsc = .001
          call savew(sheader, 'UnitSc', unitsc, LINEHEADER)
      endif
c
      dzh=.001*idz_1000
      dt = float(nsi) * unitsc
      dtmsec = 1000 * dt
      write(0,*) 'dt,dtmsec ',dt,dtmsec

      fwd_drt=(argis( '-F' ) .gt. 0 )
      rev_drt=(argis( '-R' ) .gt. 0 )
      lsq_rev=(argis( '-LSQR' ) .gt. 0 )
      interpolate=(argis( '-I' ) .gt. 0 )
      if(rev_drt) then
c_______________________________________________________________
c        read in key parameters from previous forward transform.
c_______________________________________________________________
         call saver(sheader,'CrwNam',domain,LINEHEADER)
         if(domain .ne. 'tau_pq') then
            write(lerr,*) 'error in running -R option'
            write(lerr,*) 'domain = tau_pq not found in lineheader'
     1                    //' word CrwNam'
            write(lerr,*) 'probable cause: input data not generated'
     1                    //' by program radon3d with -F option' 
            call exit(1666)
         endif
         call saver(sheader,'OrNTRC',ntrace_out,LINEHEADER)
         call saver(sheader,'OrNREC',nline_out,LINEHEADER)
         call saver(sheader,'MutVel',smax,LINEHEADER)
         call saver(sheader,'MnSPEl',mx,LINEHEADER)
         call saver(sheader,'MxSPEl',my,LINEHEADER)
         call saver(sheader,'MnGrEl',jf1,LINEHEADER)
         call saver(sheader,'MxGrEl',jf4,LINEHEADER)
         f1=jf1
         f4=jf4
         nx=ntrace_out-1
         ny=nline_out-1
         dx=dxh
         dy=dyh
      else
         nx=ntr-1
         ny=nrec-1
      endif

      lenhed=ITRWRD
      hbegin=1-lenhed
c_______________________________________________________________
c     read in additional command line parameters.
c_______________________________________________________________
      call cmdlin(s1,s2,s3,s4,smax,
     1            a1,a2,a3,a4,f1,f2,f3,f4,fs,df,fnyquist,
     2            t1,t2,t3,t4,nsamp_in,nt,dtmsec,istart,iend,
     3            nfft,dx,dy,dz,dsamp,dmsamp,dxh,dyh,dzh,prew,
     4            mx,my,wx,wy,xtaper,ytaper,reject,verbose,
     5            fwd_drt,rev_drt,depth_section,lerr,nx,ny,
     6            xazim,yazim,xazimh,yazimh,toeplitz,svd,rhofilt,
     7            startline,endline,starttrace,endtrace,dang,
     8            pmin,pmax,qmin,qmax,true_dip,
     9            p1,p2,p3,p4,q1,q2,q3,q4,
     a            nxinterp,nyinterp,dxi,dyi,interpolate,
     b            kximin,kximax,kyimin,kyimax)

      pnyquist=1./(.001*f4*dx)
      qnyquist=1./(.001*f4*dy)
      if(nxpad .gt. 0) then
         padded=.true.
         if(mx .gt. nxpad) then
            write(lerr,*) 'error in radon3d!'
            write(lerr,*) 'mx = ',mx,' exceeds nxpad = ',nxpad,
     1                 ' read from line header!'
            write(lerr,*) 'examine command lines to radon3d and splitr!'
         endif
      else
         padded=.false.
         nxpad=0
      endif
      if(padded) then
c______________________________________________________________________
c        input data have been padded each with nx_pad extra traces by
c        routine 'splitr' or equivalent.
c        it is the user's responsibility to make sure that the
c        nxpad following -nxpad in routine 'splitr' is .ge. to
c        the mx used in this application.
c______________________________________________________________________
         nx=nx-(2*nxpad)
      endif
      nt=iend-istart+1
c_____________________________________________________________________
c     calculate the p and q that critically sample the moveout.
c     fref needs to be in cycles/msec, cycles/m, cycles/ft to
c     correspond to the units of p.
c_____________________________________________________________________
      fref=.5*(f3+f4)
      periodmin=1./(.001*fref)
      if(mx .eq.  0) then
         dp=0.
         np=1
         pmin=0.
         pmax=0.
      else
         dp=periodmin/(dang*mx*dx)
         np=nint((pmax-pmin)/dp)+1
         if(np .gt. 1) then
            dp=(pmax-pmin)/(np-1)
         endif
      endif
      if(my .eq. 0) then
         dq=0.
         nq=1
         qmin=0.
         qmax=0.
      else
         dq=periodmin/(dang*my*dy)
         nq=nint((qmax-qmin)/dq)+1
         if(nq .gt. 1) then
            dq=(qmax-qmin)/(nq-1)
         endif
      endif
      if(mx .gt. 0) then
         in_line_filtering=.true.
      else
         in_line_filtering=.false.
      endif
      nxw=2*mx+1
      nyw=2*my+1
      nxwfine=2*mx*nxinterp+1
      nywfine=2*my*nyinterp+1
      nline_out=(endline-startline)*nyinterp+1
      ntrace_out=(endtrace-starttrace)*nxinterp+1

      if(np .gt. maxp) then
         write(lerr,*) 'np = ',np,'  exceeds maxp =',maxp
         write(lerr,*) 'check command line arguments!'
         call exitfu(1666)
      endif
      if(nq .gt. maxq) then
         write(lerr,*) 'nq = ',nq,'  exceeds maxq =',maxq
         write(lerr,*) 'check command line arguments!'
         call exitfu(1666)
      endif
      if(mx .gt. maxmx) then
         write(lerr,*) 'mx = ',mx,' exceeds maxmx = ',maxmx
         write(lerr,*) 'check command line arguments!'
         call exitfu(1666)
      endif
      if(my .gt. maxmy) then
         write(lerr,*) 'my = ',my,' exceeds maxmy = ',maxmy
         write(lerr,*) 'check command line arguments!'
         call exitfu(1666)
      endif
      call getgrid(p,q,x,y,locate,mx,my,np,nq,dx,dy,dp,dq,
     1             pmin,qmin,jcenter,lerr)
c
      fh=f4
      fl=f1
      ifl=nint(fl/df)
      ifh=nint(fh/df)
      ifs=nint(fs/df)
      nf=ifh-ifl+1
c______________________________________________________________________
c     calculate omega in cycles/ms:
c______________________________________________________________________
      domega=.001*2.*pi*df
      do 12000 jomega=ifl,ifh
       omega(jomega)=(jomega+ifs)*domega
12000 continue
c______________________________________________________________________
c     calculate buffer requirements for nested mixed radix ffts.
c     cray scilib routine rfftmlt runs twice as fast as mathadvantage
c     routine rnfftm.
c
c     buffer   mathadvantage rnfftm        cray scilib rfftmlt
c
c     itab      nfft+34                      2*nfft
c     rtab      3(nfft/2)+13                 2*nfft
c     work      18*maxtrace                  4*nfft*maxtrace
c______________________________________________________________________
      maxtrace=max(nx*nxinterp+2*mx+1,np*nq,nxw*nyw)
      if(szsmpd .eq. 4) then
         lenitab=nfft+34
         lenrtab=3*(nfft/2)+13
         lenwork=18*maxtrace
      else
         lenitab=max(2*nfft,nfft+34)
         lenrtab=max(2*nfft,3*(nfft/2)+13)
         lenwork=max(4*(maxtrace+1)*nfft,18*maxtrace)
      endif
c
      nbytes_out=(nsamp_in+lenhed)*szsmpd
      nt_orig=nsamp_in-1  
      lentr2=(lenhed+(nt+1))*szsmpd
c
      write(lerr,*)
      write(lerr,'(a,t50,i5,a)') 'input seismic file name',
     1                         luin,file_in
      write(lerr,'(a,t50,i5,a)') 'output file name',
     1                         luout,file_out
      write(lerr,'(a,t50,i5)') 'number of input samples',nsamp_in
      write(lerr,'(a,t50,i5)') 'number of traces',ntr  
      write(lerr,'(a,t50,i5)') 'number of records',nrec 
      if(depth_section) then
         write(lerr,*) 'slownesses in m/m or ft/ft'
      else
         write(lerr,*) 'slownesses in msec/m or msec/ft'
      endif
      if(true_dip) then
         write(lerr,'(a,t50,1pe12.3)') 'smax',smax
         write(lerr,'(a,t50,1pe12.3)') 's1',s1 
         write(lerr,'(a,t50,1pe12.3)') 's2',s2    
         write(lerr,'(a,t50,1pe12.3)') 's3',s3    
         write(lerr,'(a,t50,1pe12.3)') 's4',s4    
         write(lerr,'(a,t50,1pe12.3)') 'a1',a1 
         write(lerr,'(a,t50,1pe12.3)') 'a2',a2    
         write(lerr,'(a,t50,1pe12.3)') 'a3',a3    
         write(lerr,'(a,t50,1pe12.3)') 'a4',a4    
      else
         write(lerr,'(a,t50,1pe12.3)') 'pmin',pmin
         write(lerr,'(a,t50,1pe12.3)') 'pmax',pmax
         write(lerr,'(a,t50,1pe12.3)') 'p1',p1 
         write(lerr,'(a,t50,1pe12.3)') 'p2',p2    
         write(lerr,'(a,t50,1pe12.3)') 'p3',p3    
         write(lerr,'(a,t50,1pe12.3)') 'p4',p4    
         write(lerr,'(a,t50,1pe12.3)') 'qmin',qmin
         write(lerr,'(a,t50,1pe12.3)') 'qmax',qmax
         write(lerr,'(a,t50,1pe12.3)') 'q1',q1 
         write(lerr,'(a,t50,1pe12.3)') 'q2',q2    
         write(lerr,'(a,t50,1pe12.3)') 'q3',q3    
         write(lerr,'(a,t50,1pe12.3)') 'q4',q4    
      endif
      write(lerr,'(a,t50,f12.6)') 't1, start time (msec)',t1 
      write(lerr,'(a,t50,f12.6)') 't2 (msec)',t2    
      write(lerr,'(a,t50,f12.6)') 't3 (msec)',t3    
      write(lerr,'(a,t50,f12.6)') 't4, end time (msec)',t4    
      write(lerr,'(a,t50,i5)') 'start time (samples)',istart
      write(lerr,'(a,t50,i5)') 'end time (samples)',iend  
      write(lerr,'(a,t50,i5)') 'nt',nt,'nt_orig',nt_orig  
      write(lerr,'(a,t50,i5)') 'fft length (samples)',nfft
      write(lerr,'(a,t50,1pe12.3)') 
     1             'sample interval dmsamp (msec,m,ft)',dmsamp
      write(lerr,'(a,t50,1pe12.3)') 
     1             'sample interval dsamp (sec,km,kft)',dsamp
      write(lerr,'(a,t50,1pe12.3)') 'f1 (Hz, cycles/km, cycles/kft)',
     1                             f1                
      write(lerr,'(a,t50,1pe12.3)') 'f2 (Hz, cycles/km, cycles/kft)',
     1                             f2                
      write(lerr,'(a,t50,1pe12.3)') 'f3 (Hz, cycles/km, cycles/kft)',
     1                             f3                
      write(lerr,'(a,t50,1pe12.3)') 'f4 (Hz, cycles/km, cycles/kft)',
     1                             f4                
      write(lerr,'(a,t50,1pe12.3)') 'fs (Hz, cycles/km, cycles/kft)',
     1                             fs                
      write(lerr,'(a,t50,1pe12.3)') 'fnyquist '//
     1                     '(Hz, cycles/km, cycles/kft)',fnyquist
      write(lerr,'(a,t40,1pe12.3)') 'pnyquist (at f4) ',pnyquist,
     1            ' q nyquist (at f4) ',qnyquist
      write(lerr,'(a,t50,1pe12.3)') 
     1       'df (Hz, cycles/km, cycles/kft)',df
      write(lerr,'(a,t50,1pe12.3)') 
     1      'domega (radians/msec, radians/m, radians/ft)',domega
      write(lerr,'(a,t50,i5)') 'first frequency (samples)',ifl
      write(lerr,'(a,t50,i5)') 'last frequency (samples)',ifh
      write(lerr,'(a,t50,i5)') 'total frequencies (samples)',nf
      write(lerr,'(a,t50,f12.6)') 'inline trace spacing dx (m)',dx   
      write(lerr,'(a,t50,f12.6)') 'interpolated inline spacing'
     1                 //' dxi (m)',dxi
      write(lerr,'(a,t50,f12.6)') 'crossline spacing dy (m)',
     1                          dy   
      write(lerr,'(a,t50,f12.6)') 'interpolated crossline spacing '
     1                   //' dyi (m)',dyi   
      write(lerr,'(a,t50,f12.6)') 'inline half window wx (m)',wx
      write(lerr,'(a,t50,f12.6)') 'crossline half window wy (m)',wy
      write(lerr,'(a,t40,f12.6)') 'azimuth of in line (x) axis '
     1             //'(degrees) ',xazim
      write(lerr,'(a,t40,f12.6)') 'azimuth of cross line (y) axis '
     1             //'(degrees) ',yazim
      write(lerr,'(a,t50,i5)') 'mx',mx,'my',my,'nxw',nxw,'nyw',nyw
      write(lerr,'(a,t50,i5)') 'nxpad',nxpad
      write(lerr,'(a,t50,i5)') 'startline',startline,
     1         'endline',endline,'starttrace',starttrace,
     2         'endtrace',endtrace
      write(lerr,'(a,t40,i5)') 'nxinterp',nxinterp,
     1                         'nyinterp',nyinterp,
     2                         'kximin',kximin,'kximax',kximax,
     3                         'kyimin',kyimin,'kyimax',kyimax
      write(lerr,'(a,t40,i5)') 'ntrace_out',ntrace_out,
     1                         'nline_out',nline_out
      write(lerr,'(a,t50,i5)') 'np',np,'nq',nq
      write(lerr,'(a,t50,i5)') 'lentr2',lentr2,'lntrhd',lntrhd,
     1                         'lenhed',lenhed,'ITRWRD',ITRWRD,
     1                         'nbytes_out',nbytes_out           
c
      write(lerr,'(a,t50,l5)') 'reject data between velocity/azim?',
     1                          reject
      write(lerr,'(a,t50,l5)') 'input depth section?',depth_section
      write(lerr,'(a,t50,l5)') 'forward DRT only?',fwd_drt
      write(lerr,'(a,t50,l5)') 'reverse DRT only?',rev_drt
      write(lerr,'(a,t50,l5)') 'least square reverse transform?',
     1                           lsq_rev         
      write(lerr,'(a,t50,l5)') 'conventional rho filtered transform?',
     1                          rhofilt
      write(lerr,'(a,t50,l5)') 'invert using SVD?',svd       
      write(lerr,'(a,t50,l5)') 'invert using Toeplitz structure?',
     1                           toeplitz
      write(lerr,'(a,t50,l5)') 'verbose output?',verbose     
      write(lerr,*)' '
      write(lerr,*) 'center point at node = ',jcenter
      if(true_dip) then
c_____________________________________________________________
c        calculate true dip/azimuth tapers.
c_____________________________________________________________
         ds=2.*smax/(maxswgt-minswgt)  
         nswgt=maxswgt-minswgt+1
         call gttapr(swgt,s1,s2,s3,s4,ds,minswgt,maxswgt)
         da=1.
         call aztapr(awgt,a1,a2,a3,a4,minawgt,maxawgt)
         if(verbose) then
            write(lerr,*) 's1,s2,s3,s4,smax ',s1,s2,s3,s4,smax
            write(lerr,'(a5,2a12)') 'k','s(k)','swgt(k)' 
            write(lerr,'(i5,2f12.6)') 
     1                    (k,k*ds,swgt(k),k=minswgt,maxswgt)
            write(lerr,'(a5,a12)') 'k','awgt(k)'
            write(lerr,'(i5,f12.6)') (k,awgt(k),k=-180,+180)
         endif
      else
c_____________________________________________________________
c        calculate apparent dip tapers.         
c_____________________________________________________________
         da=1.
         ds=2.*smax/(maxswgt-minswgt)
         nswgt=2*maxswgt+1
         call gttapr(pwgt,p1,p2,p3,p4,ds,-maxswgt,maxswgt)
         call gttapr(qwgt,q1,q2,q3,q4,ds,-maxswgt,maxswgt)
         if(verbose) then
            write(lerr,*) 'p1,p2,p3,p4 ',p1,p2,p3,p4
            write(lerr,*) 'q1,q2,q3,q4 ',q1,q2,q3,q4
            write(lerr,'(a5,4a12)') 'k','p(k)','pwgt(k)','q(k)',
     1                     'qwgt(k)' 
            write(lerr,'(i5,4f12.6)')
     1         (k,k*ds,pwgt(k),k*ds,qwgt(k),k=-maxswgt,maxswgt)
         endif

      endif
c_____________________________________________________________
c     calculate spatial tapers.             
c_____________________________________________________________
      xmin=-mx*dx
      xmax=+mx*dx
      x1=xmin-dx 
      x2=xmin+xtaper
      x3=xmax-xtaper
      x4=xmax+dx
      call gttapr(xwgt,x1,x2,x3,x4,dx,-mx,+mx)               
      ymin=-my*dy
      ymax=+my*dy
      y1=ymin-dy
      y2=ymin+ytaper
      y3=ymax-ytaper
      y4=ymax+dy
      call gttapr(ywgt,y1,y2,y3,y4,dy,-my,+my)
      call getwxy(wgtxy,locate,xwgt,ywgt,mx,my,lerr)
c_____________________________________________________________________
c     calculate mutes as a function of angle.
c_____________________________________________________________________
      call getmute(wgtpq,jpqlive,p,q,swgt,awgt,ds,da,np,nq,
     1             minswgt,maxswgt,minawgt,maxawgt,lerr,
     2             xazim,yazim,clockwise,verbose,npqlive,
     3             plive,qlive,slive,pwgt,qwgt,true_dip)
      write(lerr,*) 'npqlive = ',npqlive
      write(lerr,*) 'np*nq   = ',np*nq
      write(ler,*) 'npqlive = ',npqlive
      write(ler,*) 'np*nq   = ',np*nq
c______________________________________________________________________
c     calculate memory requirements
c______________________________________________________________________
      write(lerr,'(///,80a1)') ('_',i=1,80)
      write(lerr,'(A)')'storage map of major arrays'
      write(lerr,'(80a1)') ('_',i=1,80)
C
      l_free=1
      if(fwd_drt) then
         lenrf=2*npqlive*nxw*nyw*nf
         ifh_rf=ifh
      else
         lenrf=2*npqlive*nxw*nyw
         ifh_rf=ifl 
      endif
      if(rev_drt) then
         lenrr=2*npqlive*nf*nxinterp*nyinterp
         ifh_rr=ifh 
      else
         lenrr=2*npqlive*nxinterp*nyinterp
         ifh_rr=ifl 
      endif
      lenudata=max(nx+2*mx+1,np*nq)*nfft*nxinterp
      lensvd=max(np,nxw,nq,nyw)+1
      lenra=2*nf*nxw*nyw*nxinterp*nyinterp
      lenuout=2*nf*(nx+1)*nxinterp*nyinterp
      maxpq=max(np,nq)
      maxxypq=max(np,nq,nxw,nyw)
      maxuv=max(np*nxw,nq*nyw)
c____________________________________________________________________
      write(lerr,'(a20,3a10)')'variable name','begin','end','length'
      call mapmem('uin',l_uin,l_free,2*nf*(nx+2*mx+1)*(2*my+1),lerr)
      call mapmem('uorig',l_uorig,l_free,
     1               (nt_orig+1)*(nx+1)*(my+1),lerr)
      call mapmem('udata',l_udata,l_free,lenudata,lerr)
      call mapmem('live',l_live,l_free,(2*my+1)*(nx+1+2*mx),lerr)
      call mapmem('tracebuf',l_tracebuf,l_free,(nt_orig+1+lenhed),lerr)
      call mapmem('trheader',l_trheader,l_free,
     1                             lenhed*(nx+1)*(2*my+1),lerr)
      call mapmem('uout',l_uout,l_free,lenuout,lerr)
      call mapmem('utaup',l_utaup,l_free,2*nf*np*nq,lerr)
      call mapmem('map',l_map,l_free,nxwfine*nywfine,lerr)
c
      call mapmem('rx1d',l_rx1d,l_free,2*nxw*np,lerr)
      call mapmem('rx1dc',l_rx1dc,l_free,2*nxw*np,lerr)
      call mapmem('rx1dinv',l_rx1dinv,l_free,2*nxw*np,lerr)
      call mapmem('rx2d',l_rx2d,l_free,2*nxw*nyw*nyw*np,lerr)
      call mapmem('rx2dinv',l_rx2dinv,l_free,2*nxw*nyw*nyw*np,lerr)
      call mapmem('rx1dfine',l_rx1dfine,l_free,2*nxwfine*np,lerr)
      call mapmem('rx2dfine',l_rx2dfine,l_free,
     1             2*nxwfine*nywfine*nywfine*np,lerr)
c
      call mapmem('ry1d',l_ry1d,l_free,2*nyw*nq,lerr)
      call mapmem('ry1dc',l_ry1dc,l_free,2*nyw*nq,lerr)
      call mapmem('ry1dinv',l_ry1dinv,l_free,2*nyw*nq,lerr)
      call mapmem('ry2d',l_ry2d,l_free,2*nyw*np*nq*np,lerr)
      call mapmem('ry2dinv',l_ry2dinv,l_free,2*nyw*np*nq*np,lerr)
      call mapmem('ry1dfine',l_ry1dfine,l_free,2*nywfine*nq,lerr)
      call mapmem('ry2dfine',l_ry2dfine,l_free,
     1             2*nywfine*np*nq*np,lerr)
c
      call mapmem('r2',l_r2,l_free,2*maxpq*maxpq,lerr)
      call mapmem('z',l_z,l_free,2*maxpq,lerr)
      call mapmem('iperm',l_iperm,l_free,maxpq,lerr)
      call mapmem('rf',l_rf,l_free,lenrf,lerr)
      call mapmem('rr',l_rr,l_free,lenrr,lerr)
      call mapmem('ra',l_ra,l_free,lenra,lerr)
      call mapmem('sigma',l_sigma,l_free,2*lensvd,lerr)
      call mapmem('re',l_re,l_free,2*lensvd,lerr)
      call mapmem('ru',l_ru,l_free,2*maxuv,lerr)
      call mapmem('rv',l_rv,l_free,2*maxuv,lerr)
      call mapmem('rwork',l_rwork,l_free,2*maxxypq,lerr)
c
      call mapmem('rtabf',l_rtabf,l_free,lenrtab,lerr)
      call mapmem('itabf',l_itabf,l_free,lenitab,lerr)
      call mapmem('rtabi',l_rtabi,l_free,lenrtab,lerr)
      call mapmem('itabi',l_itabi,l_free,lenitab,lerr)
      call mapmem('work',l_work,l_free,lenwork,lerr)
      call mapmem('twgt',l_twgt,l_free,nt+1,lerr)
      call mapmem('omtwgt',l_omtwgt,l_free,nt+1,lerr)
      call mapmem('fwgt',l_fwgt,l_free,nf,lerr)
c___________________________________________________________________
C     allocate dynamic memory.
C_______________________________________________________________________
      lens=l_free-1
      write(ler,'(//,a)') 'allocate dynamic memory for RADON3D: '
      write(ler,*) 1.e-6*lens,' Mwords'
      write(ler,*) 1.e-6*lens*szsmpd,' Mbytes'
      write(lerr,'(a20,10x,i10)') 'total vector length',lens
      call galloc(pntrs,lens*szsmpd,ierrcd,0)
      if(ierrcd .ne. 0 ) then
         write(lerr,*)'galloc memory allocation error from main'
         write(lerr,*)'ierrcd = ',ierrcd
         write(lerr,*)
         write(lerr,*)'probable cause: too much memory requested!'
         write(lerr,*)'go check your seismic headers and command'//
     1                ' line arguments!'
         write(lerr,*)'memory can be decreased by using the -M option'
         write(lerr,*)
         write(lerr,*)'program RADON3D aborted'
         call exit(101)
      endif
c
c_____________________________________________________________
c     calculate temporal tapers.
c_____________________________________________________________
      call gttapr(s(l_twgt),t1,t2,t3,t4,dmsamp,istart,iend)
      call getcomp(s(l_twgt),s(l_omtwgt),istart,iend)
c_____________________________________________________________
c     calculate frequency tapers.
c_____________________________________________________________
      call gttapr(s(l_fwgt),f1,f2,f3,f4,df,ifl,ifh)
c_______________________________________________________________________
c     modify line header to reflect actual number of traces output
c_______________________________________________________________________
      call savhlh(sheader,lbytes,lbyout)
      if(padded .and. in_line_filtering) then
c______________________________________________________________________
c        We are going to reduce the output by Nx_Pad traces.  We
c        now need to set this header value to 0.
c______________________________________________________________________
         newpad = 0
         call savew(sheader,'Nx_Pad',newpad,LINEHEADER)
      endif
      if(fwd_drt) then
c_______________________________________________________________________
c        modify line header such that data can be reconstructed
c        when running with the -R option without
c        explicitly typing in all the parameters again.
c_______________________________________________________________________
         idz_1000=1000.*dz
         jf1=nint(f1)
         jf4=nint(f4)
         domain='tau_pq'
         call savew(sheader,'OrNTRC',ntrace_out,LINEHEADER)
         call savew(sheader,'OrNREC',nline_out,LINEHEADER)
         call savew(sheader,'NumTrc',npqlive,LINEHEADER)
         call savew(sheader,'NumRec',nline_out*ntrace_out,LINEHEADER)
         call savew(sheader,'ILClIn',dx,LINEHEADER)
         call savew(sheader,'CLClIn',dy,LINEHEADER)
         call savew(sheader,'Dz1000',idz_1000,LINEHEADER)
         call savew(sheader,'CrwNam',domain,LINEHEADER)
         call savew(sheader,'MnSPEl',mx,LINEHEADER)
         call savew(sheader,'MxSPEl',my,LINEHEADER)
         call savew(sheader,'MnGrEl',jf1,LINEHEADER)
         call savew(sheader,'MxGrEl',jf4,LINEHEADER)
      else
         call savew(sheader,'NumTrc',ntrace_out,LINEHEADER)
         call savew(sheader,'NumRec',nline_out,LINEHEADER)
      endif
      call wrtape(luout,sheader,lbyout)
      call timend(cputim(1),va,vb,waltim(1),wa,wb)

      call savelu('TrcNum',ifmt_TrcNum,l_TrcNum,ln_TrcNum,TRACEHEADER)
      call savelu('RecNum',ifmt_RecNum,l_RecNum,ln_RecNum,TRACEHEADER)
      call savelu('DstSgn',ifmt_DstSgn,l_DstSgn,ln_DstSgn,TRACEHEADER)
      call savelu('StaCor',ifmt_StaCor,l_StaCor,ln_StaCor,TRACEHEADER)
      call savelu('DphInd',ifmt_DphInd,l_DphInd,ln_DphInd,TRACEHEADER)
      call savelu('LinInd',ifmt_LinInd,l_LinInd,ln_LinInd,TRACEHEADER)
c_____________________________________________________________   
c     precompute the (tau,p,q) forward transform [rf]
c     precompute the (tau,p,q) reverse transform [rr]
c     precompute the filter operator matrix [ra]=[rr]*[wgtpq]*[rf]
c_____________________________________________________________   
      call timstr(va,wa)
      write(ler,*) 'begin calculation of transform matrix'  
      call fwdlsq(s(l_rx1d),s(l_rx2d),s(l_rx1dinv),s(l_rx2dinv),
     1          s(l_ry1d),s(l_ry2d),s(l_ry1dinv),s(l_ry2dinv),
     2          s(l_rx1dc),s(l_ry1dc),s(l_r2),s(l_z),s(l_iperm),
     3   s(l_rx1dfine),s(l_ry1dfine),s(l_rx2dfine),s(l_ry2dfine),
     3          s(l_rf),s(l_rr),s(l_ra),toeplitz,svd,rhofilt,
     4          s(l_sigma),s(l_re),s(l_ru),s(l_rv),s(l_rwork),
     5          x,y,p,q,nxw,nyw,np,nq,dx,dy,dp,dq,
     6          wgtpq,jpqlive,npqlive,omega(ifl),
     7          ifl,ifh,ifh_rf,ifh_rr,reject,
     8          prew,lerr,ler,fwd_drt,rev_drt,wgtxy,
     9          kximin,kximax,xmin,dxi,mx,nxinterp,nxwfine,
     a          kyimin,kyimax,ymin,dyi,my,nyinterp,nywfine,
     b          s(l_map))
      write(ler,*) 'transform matrix calculated'
      call timend(cputim(2),va,vb,waltim(2),wa,wb)
      if(fwd_drt) then
         call fwddrt(s(l_uin),s(l_live),s(l_utaup),s(l_udata),
     1            s(l_uorig),s(l_rf),s(l_tracebuf),s(l_trheader),
     2               ifl,ifh,cputim,waltim,mx,my,np,nq,
     3               iypointer,jy,ny,locate,nt,nx,nxw,nyw,
     4               luin,luout,lerr,jrec,jcenter,
     5               l_StaCor,eof,lenhed,lntrhd,nfft,
     6               s(l_rtabf),s(l_itabf),s(l_rtabi),s(l_fwgt),
     7               s(l_itabi),s(l_work),s(l_twgt),s(l_omtwgt),
     8               lenrtab,lenitab,lenwork,ler,nxpad,
     9               istart,iend,nt_orig,nbytes_out,interpolate,
     a               l_LinInd,l_DphInd,jpqlive,npqlive,
     b               ifmt_LinInd,ln_LinInd,ifmt_DphInd,ln_DphInd,
     c               ifmt_StaCor,ln_StaCor,reject,kypointer,
     d               starttrace,endtrace,startline,endline,
     e               plive,qlive,slive,dmsamp)
      elseif(rev_drt) then
c_____________________________________________________________   
c        read in (tau,p,q) data and reverse transform to generate
c        (t,x,y) data.
c_____________________________________________________________   
         call revdrt(s(l_utaup),s(l_live),s(l_uout),s(l_udata),
     1               s(l_uorig),s(l_rr),s(l_tracebuf),s(l_trheader),
     2               ifl,ifh,cputim,waltim,mx,my,np,nq,
     3               iypointer,jy,ny,locate,nt,nx,nxw,nyw,
     4               luin,luout,lerr,jrec,jcenter,
     5               l_StaCor,eof,lenhed,lntrhd,nfft,
     6               s(l_rtabf),s(l_itabf),s(l_rtabi),s(l_fwgt),
     7               s(l_itabi),s(l_work),s(l_twgt),s(l_omtwgt),
     8               lenrtab,lenitab,lenwork,ler,nxpad,
     9               istart,iend,nt_orig,nbytes_out,
     a               l_LinInd,l_DphInd,jpqlive,npqlive,
     b               ifmt_LinInd,ln_LinInd,ifmt_DphInd,ln_DphInd,
     c               ifmt_StaCor,ln_StaCor,reject,kypointer,
     d               starttrace,endtrace,startline,endline,
     e               kximin,kximax,kyimin,kyimax,
     b               ifmt_RecNum,l_RecNum,ln_RecNum,
     c               ifmt_TrcNum,l_TrcNum,ln_TrcNum,TRACEHEADER)
      else
c_____________________________________________________________   
c        filter the data by lumping the forward drt, muting, and
c        reverse drt into one matrix operation.
c_____________________________________________________________   
         call filtdrt(s(l_uin),s(l_live),s(l_uout),s(l_udata),
     1         s(l_uorig),s(l_ra),s(l_tracebuf),s(l_trheader),
     2                ifl,ifh,cputim,waltim,mx,my,
     3                iypointer,jy,ny,locate,nt,nx,nxw,nyw,
     4                luin,luout,lerr,jrec,
     5                l_StaCor,eof,lenhed,lntrhd,nfft,
     6                s(l_rtabf),s(l_itabf),s(l_rtabi),s(l_fwgt),
     7                s(l_itabi),s(l_work),s(l_twgt),s(l_omtwgt),
     8                lenrtab,lenitab,lenwork,ler,nxpad,
     9                istart,iend,nt_orig,nbytes_out,interpolate,
     a                l_LinInd,l_DphInd,
     b                ifmt_LinInd,ln_LinInd,ifmt_DphInd,ln_DphInd,
     c                ifmt_StaCor,ln_StaCor,reject,kypointer,
     d                starttrace,endtrace,startline,endline,
     e                kximin,kximax,kyimin,kyimax,
     b               ifmt_RecNum,l_RecNum,ln_RecNum,
     c               ifmt_TrcNum,l_TrcNum,ln_TrcNum,TRACEHEADER)
      endif
      if(eof) go to 99000
c_____________________________________________________________   
c     write out timing statistics.
c_____________________________________________________________   
      call timend(cputim(20),vtot,vb,waltim(20),wtot,wb)
      write(lerr,'(A30,2A15,/)') 'routine','cpu time','wall time'
      write(lerr,'(A30,2f15.3)')
     1         'read input',cputim(1),waltim(1),
     2         'precompute operator',cputim(2),waltim(2),
     3         'filter data',cputim(3),waltim(3),
     4         'calculate tau-p transform',cputim(4),waltim(4),
     5         'write tau-p transform',cputim(5),waltim(5),
     6         'write filtered data',cputim(10),waltim(10),
     7         'total',cputim(20),waltim(20)
c_____________________________________________________________   
c     close data files
c_____________________________________________________________   
      call lbclos(luin)
      call lbclos(luout)

      write(ler,*)'normal completetion. routine RADON3D'           
      write(lerr,*)'normal completetion. routine RADON3D'           
      close(lerr)
      call exit(0)
99000 write(ler,*) 'end of file encountered when reading line jy = ',
     1              jy,' of ',ny,' from unit luin ',luin
      call exit(666)
c
      end
      subroutine help(ler)

      write(ler,*)
     1'**************************************************************'
      write(ler,*)'3D radon transform'                                 
      write(ler,*)
     1'**************************************************************'
      write(ler,*)'execute radon3d by typing radon3d and list of'
     1                   //' program parameters.'
      write(ler,*)
     1'note that each parameter is proceeded by -a where "a" is '
      write(ler,*)
     1'a character(s) correlation to some parameter.'
      write(ler,*)
     1'users enter the following parameters, or use the default values'
      write(ler,*)' '
        write(ler,*)
     1' -N [file_in]  (stdin)      : input seismic file name'          
        write(ler,*)
     1' -O [file_out]  (stdout)   : output filtered file name'        
        write(ler,*)' '
        write(ler,*)
     1' -ildm       (ILClIn)        : in line trace spacing'        
        write(ler,*)
     1' -cldm       (CLClIn)        : cross line trace spacing'        
        write(ler,*)
     1' -dz         (Dz1000)        : depth sample unit (indicates'
     2                               //' depth section)'
        write(ler,*)
     1' -ilhw       (5*ildm)        : in half length of '     
     2              //'rectangular analysis window (m)'
        write(ler,*)
     1' -clhw       (5*cldm)        : cross line half width '
     2           //'rectangular analysis window (m)'
        write(ler,*)
     1' -f1[f1]   (0.)      :  begin roll in frequency taper '
     2                   //'weights (Hz, cycles/km, cycles/kft) '
        write(ler,*)
     1' -f2[f2]   (f1)      :  end roll in frequency taper '
     2                   //'weights (Hz, cycles/km, cycles/kft) '
        write(ler,*)
     1' -f3[f3]   (f4)      :  begin roll off frequency taper '
     2                   //'weights (Hz, cycles/km, cycles/kft) '
        write(ler,*)
     1' -f4[f4]   (Nyquist) :  end roll off frequency taper'
     2                   //'weights (Hz, cycles/km, cycles/kft) '
        write(ler,*)
     1' -t1[t1]   (first samp): begin roll in temporal (depth) '
     2                   //'taper weights (msec, m, ft)'                   
        write(ler,*)
     1' -t2[t2]   (t1)      :  end roll in temporal (depth) '
     2                   //'taper weights (msec, m, ft)'                   
        write(ler,*)
     1' -t3[t3]   (t4)      :  begin roll off temporal (depth) '
     2                   //'taper weights (msec, m, ft)'                   
        write(ler,*)
     1' -t4[t4]   (last samp) :  end roll temporal (depth) '
     2                   //'taper weights (msec, m, ft)'                   
        write(ler,*)
     1' -ilazim    (AziIln)   : in line azimuth in degrees'
     2                //' (azimuth of increasing trace numbers)'
        write(ler,*)
     1' -clazim    (AziCln)   : cross line azimuth in degrees'
     2                //' (azimuth of increasing line numbers) '
     2                //' axis (increasing line numbers, degrees)'
        write(ler,*)
     1' -startline (1)       : first line to be processed'     
        write(ler,*)
     1' -endline   (last)    : last line to be processed'     
        write(ler,*)
     1' -starttrace (1)      : first trace to be processed'    
        write(ler,*)
     1' -endtrace   (last)   : last trace to be processed'    
        write(ler,*)
     1' -prew[white]   (5.0) :  prewhitening for inverse (%)'
        write(ler,*)
        write(ler,*) 'Parameterization using apparent dips'
        write(ler,*)
        write(ler,*)
     1' -pmin       (0.0)            : minimum inline (x) dip'
     2       //' (msec/m, msec/ft, m/m, ft/ft)'
        write(ler,*)
     1' -pmax       (0.0)            : maximum inline (x) dip'
     2       //' (msec/m, msec/ft, m/m, ft/ft)'
        write(ler,*)
     1' -qmin       (0.0)            : minimum crossline (y) dip'
     2       //' (msec/m, msec/ft, m/m, ft/ft)'
        write(ler,*)
     1' -qmax       (0.0)            : maximum crossline (y) dip'
     2       //' (msec/m, msec/ft, m/m, ft/ft)'
        write(ler,*)
     1' -p1         (pmin)           : begin roll in of inline'
     2       //'filter weights (msec/m, msec/ft, m/m, ft/ft)'
        write(ler,*)
     1' -p2         (p1)             : end roll in of inline'
     2       //'filter weights (msec/m, msec/ft, m/m, ft/ft)'
        write(ler,*)
     1' -p3         (p4)             : begin roll out of inline'
     2       //'filter weights (msec/m, msec/ft, m/m, ft/ft)'
        write(ler,*)
     1' -p4         (qmax)           : end roll out of crossline'
     2       //'filter weights (msec/m, msec/ft, m/m, ft/ft)'
        write(ler,*)
     1' -q1         (qmin)           : begin roll in of crossline'
     2       //'filter weights (msec/m, msec/ft, m/m, ft/ft)'
        write(ler,*)
     1' -q2         (q1)             : end roll in of crossline'
     2       //'filter weights (msec/m, msec/ft, m/m, ft/ft)'
        write(ler,*)
     1' -q3         (q4)             : begin roll out of crossline'
     2       //'filter weights (msec/m, msec/ft, m/m, ft/ft)'
        write(ler,*)
     1' -q4         (qmax)           : end roll out of crossline'
     2       //'filter weights (msec/m, msec/ft, m/m, ft/ft)'
        write(ler,*)
        write(ler,*) 'Parameterization using true dip/azimuth'
        write(ler,*)
        write(ler,*)
     1' -smax       (0.0)            : maximum true dip'
     2       //' (msec/m, msec/ft, m/m, ft/ft)'
        write(ler,*)
     1' -s1         (0.0)            : begin roll in moveout'
     2       //'filter weights (msec/m, msec/ft, m/m, ft/ft)'
        write(ler,*)
     1' -s2         (s1)             : end roll in moveout  '
     2       //'filter weights (msec/m, msec/ft, m/m, ft/ft)'
        write(ler,*)
     1' -s3         (s4)             : begin roll out moveout  '
     2       //'filter weights (msec/m, msec/ft, m/m, ft/ft)'
        write(ler,*)
     1' -s4         (smax)           : end roll out moveout  '
     2       //'filter weights (msec/m, msec/ft, m/m, ft/ft)'
        write(ler,*)
     1' -a1         (+180.)          : begin roll in azimuth  '
     2                    //'filter weights (degrees)'
        write(ler,*)
     1' -a2         (a1)             : end roll in azimuth  '
     2                    //'filter weights (degrees)'
        write(ler,*)
     1' -a3         (a4)             : begin roll out azimuth  '
     2                    //'filter weights (degrees)'
        write(ler,*)
     1' -a4         (-180.)          : end roll out azimuth  '
     2                    //'filter weights (degrees)'
        write(ler,*)
     1' -ilinterp   (1)              : in line interpolation rate'
        write(ler,*)
     1' -clinterp   (1)              : cross line interpolation rate'
        write(ler,*)
     1' -F                    : if present, perform forward DRT'
     2                      //' only (Warning! large output files!)'
        write(ler,*)
     1' -R                    : if present, perform mute and reverse'
     2                      //' DRT only'                                 
        write(ler,*)
     1' -reject               : if present, reject modeled data '
     2                      //' between velocity and azimuth limits'
        write(ler,*)
     1' -rho                  : if present, use the less accurate'
     2               //' rho filter vs. the discrete Radon transform'
        write(ler,*)
     1' -V                    : if present, verbose printout'
       write(ler,*)' '
       write(ler,*)'usage:'
       write(ler,*)
       write(ler,*)'radon3d -N[file_in] -O[file_out] '
       write(ler,*)'        -ildm[] -cldm[] -dz[] -smax[]'
       write(ler,*)'        -ilhw[] -clhw[] -ilazim[] -clazim[]'
       write(ler,*)'        -ilinterp[] -clinterp[]'
       write(ler,*)'        -t1[] -t2[] -t3[] -t4[]'
       write(ler,*)'        -f1[] -f2[] -f3[] -f4[]'
       write(ler,*)'        -startline[] -endline[]' 
       write(ler,*)'        -starttrace[] -endtrace[]' 
       write(ler,*)'        -pmin[] -pmax[] -qmin[] -qmax[]'
       write(ler,*)'        -p1[] -p2[] -p3[] -p4[]'
       write(ler,*)'        -q1[] -q2[] -q3[] -q4[]'
       write(ler,*)'        -smax[]'
       write(ler,*)'        -s1[] -s2[] -s3[] -s4[]'
       write(ler,*)'        -a1[] -a2[] -a3[] -a4[]'
       write(ler,*)'        -prew[] -[F,R,rho,reject,V]'
       write(ler,*)' '
       write(ler,*)

      return
      end
 

