/***********************************************************************
 *                copyright 2001, Amoco Production Company             *
 *                            All Rights Reserved                      *
 *                    an affiliate of BP America Inc.                  *
 ***********************************************************************/
/*
  Modified by:  Cory Hoelting
  Original code by:  Chi-hsin Wu
  Date last modified:  July 29, 1996
  Purpose:  Used to compute the forward index and the
            backward index.  These are used by disparity
            and bethe during segmentation.
*/

#include <stdio.h>
#include <math.h>
#include <string.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/uio.h>
#include <stdlib.h>

/* USP stuff */
#include <iounit.h>
#include <lhdrsz.h>
#include <save_defs.h>
#include <cu_defs.h>
#include <sisio.h>

/*#define   NSEED   56374452 */     /* seed for generating random graph   */
#define     NSEED   81933945

/*function declaration...@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@*/
void randomgraph2d_help();
void allocation();
void generate();
void write_neighbor_sys();

/*extern function declaration...@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@*/
extern void srand48();
extern double drand48();
extern long lrand48();

/* global variable for program name, used in error messages */
char prog_name[] = "randomgraph2d";


int main(argc, argv)
int argc;
char *argv[];
{
 int ffp, bfp;  /* file descriptors for the forward and backward index files */
 int pipe1, pipe2;

 char for_file[256], bac_file[256];

 char host[10], blank9[10];
 int ikp;          /* flag set to 1 if running under ikp */
 int one;

 int ierr = 0;

 int *forward;     /* array that stores forward indices */
 int *backward;    /* array that stores backward indices */

 int ROW, COL, COARSE;
 int LROW,LCOL;    /* number of rows and columns in the label matrix */
 int N;            /* total number of label sites, i.e., N=LROW*LCOL */
 int HWW;          /* half window width */
 int WW;           /* window width */


  /* Help? */
  if (C_ARGIS("-?", &argc, argv) ||
      C_ARGIS("-h", &argc, argv) ||
      C_ARGIS("-help", &argc, argv))
    {
     randomgraph2d_help();
     exit(0);
    }

  /* check to see if we are running under ikp */
  strcpy(host, "         ");
  strcpy(blank9, "         ");
  ikpchk_(host, strlen(host));
  if (strcmp(host, blank9))
     ikp = 1;
  else
     ikp = 0;

  /* display usage part of help if there was */
  /* only one argument on the command line */ 
  if (argc == 1 && !ikp)
    {
     fprintf(stderr, 
       "Usage:  randomgraph2d -numtrc <number_of_traces>\n\
                    -numsmp <number_of_samples>\n\
                    [-coarse <coarseness>] [-halfwin <half_win_size>]\n\
                    -fmat <fifn> -bmat <bifn>\n\
                    [-h -? -help]\n");
     exit(0);
    }

  /* set up the default file descriptors for ikp */
  bfp = 3;
  pipe1 = 1;
  pipe2 = 3;
  one = 1;
 
  /* process the command line arguments */
  C_ARGSTR("-fmat", for_file, "", "", &argc, argv);
  if (strlen(for_file) < 1)
    {
     if (!ikp)
       {
        fprintf(stderr, "%s: error: the command line argument ", prog_name);
        fprintf(stderr, "-fmat <fifn> is required.\n");
        ierr = 1; 
       }
     else   /* running in ikp */
        if (pipcnt_(&pipe1, &one) > 0)
           sisfdfit_(&ffp, &pipe1);
    }
  else
    {
     ffp = open(for_file, O_WRONLY | O_CREAT | O_TRUNC, 0644);
     if (ffp == -1)
       {
        fprintf(stderr, "%s: error: cannot open the forward index file %s\n",
                prog_name, for_file);
        ierr = 1; 
       }
    }
 
  C_ARGSTR("-bmat", bac_file, "", "", &argc, argv);
  if (strlen(bac_file) < 1)
    {
     if (!ikp)
       {
        fprintf(stderr, "%s: error: the command line argument ", prog_name);
        fprintf(stderr, "-bmat <bifn> is required.\n");
        ierr = 1;
       }
     else
        if (pipcnt_(&pipe2, &one) > 0)
           sisfdfit_(&bfp, &pipe2);
    }
  else
    {
     bfp = open(bac_file, O_WRONLY | O_CREAT | O_TRUNC, 0644);
     if (bfp == -1)
       {
        fprintf(stderr, "%s: error: cannot open the backward index file %s\n",
                prog_name, bac_file);
        ierr = 1;
       }
    }
 
  C_ARGI4("-numtrc", &ROW, -1, -1, &argc, argv);
  if (ROW < 1)
    {
     fprintf(stderr,
             "%s: error: the command line argument -numtrc ", prog_name);
     fprintf(stderr, "<number_of_traces> is required and must be positive.\n");
     ierr = 1; 
    }
 
  C_ARGI4("-numsmp", &COL, -1, -1, &argc, argv);
  if (COL < 1)
    {
     fprintf(stderr, "%s: error: the command line argument -numsmp ",prog_name);
     fprintf(stderr, "<number_of_samples> is required and must be ");
     fprintf(stderr, "positive.\n");
     ierr = 1;
    }

  C_ARGI4("-coarse", &COARSE, 9, 9, &argc, argv);
  if (COARSE < 1)
    {
     fprintf(stderr, "%s: error: the command line argument -coarse ",prog_name);
     fprintf(stderr, "<coarseness> must be positive.\n");
     ierr = 1;
    }

  C_ARGI4("-halfwin", &HWW, 7, 7, &argc, argv);
  if (HWW < 1)
    {
     fprintf(stderr, "%s: error:  the command line argument ", prog_name);
     fprintf(stderr, "-halfwin <half_win_size> must be positive.\n");
     ierr = 1;
    }

  /* exit if the error condition has been set */ 
  if (ierr) exit(1);

  WW = (2 * HWW) + 1;
  LROW=(ROW-WW)/COARSE+1;
  LCOL=(COL-WW)/COARSE+1;
  N=LROW*LCOL;

  allocation(&forward, &backward, N);
  generate(forward, backward, N);
  write_neighbor_sys(forward, N, ffp);
  write_neighbor_sys(backward, N, bfp);

  close(ffp);
  close(bfp);

  /* Normal completion */
  fprintf(stderr, "%s:  normal completion\n", prog_name);
  return (0);
}/*main*/


/* allocate memory                                                            */
void allocation(forward, backward, N)
int ** forward;
int ** backward;
int N;
{
  *forward =  (int *)malloc(N*sizeof(int));
  *backward = (int *)malloc(N*sizeof(int));

}/*allocation*/


/* create the forward index and backward index which */
/* are used to generate the random neighbors in disparity and bethe */
void generate(forward, backward, N)
int * forward;
int * backward;
int N;
{
  int *checker;
  int i,j,n,ri,rj;
  int rand_site;

  srand48(NSEED);
  
  checker = (int*)malloc(N*sizeof(int));
  for (i=0; i<N; i++)
    checker[i]=0;

  for (i=0; i<N; i++) {
    do {
      rand_site=lrand48()%N;
    } while (checker[rand_site]);
    forward[i]=rand_site;
    backward[rand_site]=i;
    checker[rand_site]=1;
  }  

 free(checker);

}/*generate*/


/*write image to the file fname*/
void write_neighbor_sys(img, N, fp)
int *img;
int N;
int fp; /* output file descriptor */
{
  write(fp, (char *) img, sizeof(int) * N);

}/*write_neighbor_sys*/

  
void randomgraph2d_help()
 {
  fprintf(stderr, "\
  This program constructs the forward index and backward index used\n\
  to create a random graph neighborhood system.  The forward index\n\
  and backward index generated in this program are used by the program\n\
  disparity and the the program bethe.\n\
  \n\
  The programs disparity and bethe, use a method based on De Bruijn's\n\
  graph to generate random neighbors in the random graph neighborhood\n\
  system. For more information see Christine Graffigne, Experiments\n\
  in Texture Analysis and Segmentation, Ph.D.  dissertation, p. 48,\n\
  Division Appli. Math., Brown Univ., 1987.\n\
  \n\
  Usage:  randomgraph2d -numtrc <number_of_traces> -numsmp <number_of_samples>\n\
                       [-coarse <coarseness>] [-halfwin <half_win_size>]\n\
                       -fmat <fifn> -bmat <bifn>\n\
                       [-? -h -help]\n\
  \n\
  -numtrc <number_of_traces> - number of traces in the file containing\n\
                               the data lattice. The data lattice is the\n\
                               input to the segmentation process.\n\
  \n\
  -numsmp <number_of_samples> - number of samples per trace in the file\n\
                                containing the data lattice.  The data\n\
                                lattice is the input to the segmentation\n\
                                process.\n\
  \n\
  -coarse <coarseness> - inverse of the resolution at which the\n\
                         segmentation is performed.  That is, the ratio\n\
                         of the number of samples/traces in the data\n\
                         lattice to the number of samples/traces in the\n\
                         label lattice. The label lattice is the output\n\
                         from any segmentation optimizer, such as the\n\
                         program bethe.  For highest resolution\n\
                         set the coarseness to 1.\n\
                         (Default: 9)\n\
  \n\
  -halfwin <half_win_size> - half window width used to determine\n\
                             the width of the texture analysis window\n\
                             over which the cdf's are computed in the\n\
                             program disparity.  The width of the texture\n\
                             analysis window is given by\n\
                             2 * <half_win_size> + 1.\n\
                             (Default: 7)\n\
  \n\
  -fmat <fifn> - the output file containing the forward index\n\
  \n\
  -bmat <bifn> - the output file containing the backward index\n\
  \n\
  -? -h -help - Help.\n\
");
}
