c***********************************************************************
c                                                                      *
c                 copyright 2001, Amoco Production Company             *
c                             All Rights Reserved                      *
c                     an affiliate of BP America Inc.                  *
c                                                                      *
c***********************************************************************
c  routine:       alftms                                               *
c  routine type:  function  real                                       *
c  purpose:                                                            *
c      implement the alfa trimmed mean algorithm, for stacking.        *
c      a statistical count is kept of the number of times the data     *
c      value from each input array element is included in the          *
c      stack.                                                          *
c                                                                      *
c  entry points:                                                       *
c      alftms  real  (luprt,alphap,nfold,ntlive,b,livetr,istat,ipntr,  *
c                     tabul8)                                          *
c  arguments:                                                          *
c      luprt   integer       i           -                             *
c      alphap  real          i           -                             *
c      nfold   integer       i           -                             *
c      ntlive  integer       i           -                             *
c      b       real     ??iou*  (nfold) -                              *
c      livetr  logical  ??iou*  (nfold) -                              *
c      istat   integer  ??iou*  (nfold) -                              *
c      ipntr   integer  ??iou*  (nfold) -                              *
c      tabul8  logical  ??iou*           -                             *
c       +------------------------------------------------------+       *
c       |               development information                |       *
c       +------------------------------------------------------+       *
c  author:   bill done                          origin date: 86/09/19  *
c  language: fortran 77                  date last compiled: 86/09/19  *
c       +------------------------------------------------------+       *
c       |                 external environment                 |       *
c       +------------------------------------------------------+       *
c  routines called:                                                    *
c      srtqkp -                                                        *
c  intrinsic functions called:                                         *
c      float   real    -                                               *
c      mod     generic -                                               *
c  files:                                                              *
c      luprt  ( output sequential ) -                                  *
c  common:           none                                              *
c  stop codes:       none                                              *
c       +------------------------------------------------------+       *
c       |             other documentation details              |       *
c       +------------------------------------------------------+       *
c  error handling:                                                     *
c      if nfold or ntlive are less than 1, the function value is       *
c      set to zero.                                                    *
c                                                                      *
c  general description:                                                *
c      implement the alpha trimmed mean algorithm for finding the      *
c      representative value of nfold points in vector b.  parameter    *
c      alpha determines the number of extreme points in the ordered    *
c      elements of b which will not be included in the average.        *
c      (100 - alphap) per cent of the points on the two extremes       *
c      of the median are discarded from the average.                   *
c                                                                      *
c      this results in a variable interval around the median within    *
c      which the elements of b are average to determine the trimmed    *
c      mean.  the number of elements used is constant (except in the   *
c      case of dead traces).                                           *
c                                                                      *
c  revised by:  bill done                     revision date: 86/09/19  *
c      modified to eliminate need to sort dead trace values to rear    *
c      of b vector.                                                    *
c                                                                      *
c       +------------------------------------------------------+       *
c       |                 analysis information                 |       *
c       +------------------------------------------------------+       *
c  nonstandard features:   none detected                               *
c*******************   end of documentation package   ******************
c***********************************************************************
c
      real function alftms (luprt , alphap, nfold , ntlive , b, livetr,
     *                      istat , ipntr , tabul8)
c
      logical livetr(nfold), oddntl, tabul8
      integer istat(nfold), ipntr(nfold)
      real b(nfold), median
      data iprint/0/
c
c     convert alphap from per cent to decimal.  determine the
c     number of data values to be excluded from the sum based
c     on nfold, the maximum possible value for ntlive.  print
c     this information on the first entry to the function.
c
      if (iprint .eq. 0) then
         iprint = 1
         nmd2 = nfold/2
         nmd2p1 = nmd2 + 1
         alpha = alphap/100.
         if (mod(nfold,2) .eq. 1) then
            nex = (1.0 - alpha)*(nmd2p1 - 1)
          else
            nex = (1.0 - alpha)*(nmd2 - 1)
         endif
         write (luprt,1200) nfold, nex, alphap, nmd2, nmd2p1
 1200    format(/' ALFTMS:  nfold = ',i5,3x,'nex = ',i5,3x,
     *          'alpha = ',f6.1/9x,'nfold/2 = ',i5,3x,
     *          '(nfold/2)+1 = ',i5/)
      endif
c
c     set alftms to zero if nfold < 1 or ntlive < 1
c
      if ((nfold .lt. 1) .or. (ntlive .lt. 1)) then
         alftms = 0.0
         return
      endif
c
c     set alftms to b(nfold) if nfold = 1.
c
      if (nfold .eq. 1) then
         alftms = b(nfold)
         if (tabul8) istat(nfold) = istat(nfold) + 1
         return
      endif
c
c     sort the ntlive elements of b, which are at the front of b.
c
      call srtqkp(ntlive,b,nfold,ipntr)
c
c     find center index of array b (if ntlive is odd, nd2p1 is center;
c     if ntlive is even, nd2 and nd2p1 are the center 2 values)
c
      nd2 = ntlive/2
      nd2p1 = nd2 + 1
c
c     find the median value of the data in b
c
      oddntl = mod(ntlive,2) .eq. 1
      if (oddntl) then
         median = b(nd2p1)
       else
         median = 0.5*(b(nd2) + b(nd2p1))
      endif
c
c     determine the number of the ntlive data values to be
c     excluded from the sum.  this is the alpha trim.
c
      if (oddntl) then
         nexcld = (1.0 - alpha)*(nd2p1 - 1)
       else
         nexcld = (1.0 - alpha)*(nd2 - 1)
      endif
c
c     sum the elements of b which have values in the
c     k = nexcld+1 to ntlive-nexcld range.  at least one value
c     (median) is guaranteed to be in this interval, even
c     if alpha = 0.0.
c
      alftms = 0.0
      do 200 k = nexcld+1, ntlive-nexcld
         alftms = alftms + b(k)
  200 continue
      nadd = ntlive - 2*nexcld
      alftms = alftms/float(nadd)
c
c     if this time step is to be tabulated (tabul8 = .true.),
c     then the statistics vector is updated.  the pointer vector
c     is used to update the element in istat corresponding to
c     the data value being stacking in b.
c
      if (tabul8) then
         do 300 k = nexcld+1, ntlive-nexcld
            istat(ipntr(k)) = istat(ipntr(k)) + 1
  300    continue
      endif
      return
      end
