C***********************************************************************
C                 copyright 2001, Amoco Production Company             *
C                             All Rights Reserved                      *
C                     an affiliate of BP America Inc.                  *
C***********************************************************************
c
c Author:  Kelly D. Crawford
c Date:    09/11/96
c Purpose: Takes frequency output from spec plus scale file from spec
c          and generates a scaled spec volume.
c
      program scspec
      implicit none

      integer jerr

c get machine dependent parameters
#include <fu_defs.h>
#include <f77/iounit.h>
#include <f77/lhdrsz.h>
#include <f77/sisdef.h>
#include <save_defs.h>

      integer line_header(SZLNHD), lenhed
      character infile*(255), outfile*(255)
      character scale_file*(256), name*(6)
 
      integer lbytes, lbyout

      real peak, record
      logical live
      pointer (memadr_peak, peak(2))
      pointer (memadr_record, record(2))
      pointer (memadr_live, live(2))
      integer peak_size, record_size, live_size, total_size

      integer errcd, errcds, abort, ierr
      integer argis

      integer NumSmp, NumTrc, NumRec
      integer ifmt_StaCor, l_StaCor, ln_StaCor

      integer luin, luscale, luout
      real prew
      logical intflg, maxflg
 
c Initialize variables
      data name/'SCSPEC'/
      data abort/0/
      data luscale/81/
 
c give command line help if requested
      if (argis ('-?') .gt. 0 .or. argis ('-h') .gt. 0) then
         call scspec_help(LER)
         stop
      endif
 
c let us know if someone actually uses this thing
      call tattle("scspec")

c open printout file
#include <f77/open.h>
 
c get command line input parameters
      call cmdln(infile, scale_file, outfile, prew, name,
     1           intflg, maxflg, LER, LERR)

c open input data, input scale and output file
      call getln(luin,  infile,  'r', 0)
      call getln(luout, outfile, 'w', 1)
      open(unit=luscale, file=scale_file, iostat=errcd)
      if (errcd .ne. 0) then
         write(LERR,*)'Cannot locate scale file: ',scale_file
         write(LERR,*)'Exiting scspec'
         write(LER,*)'Cannot locate scale file: ',scale_file
         write(LER,*)'Exiting scspec'
         stop
      endif

c read input line header and grab some parameters
      call rtape(luin, line_header, lbytes)
      if (lbytes .eq. 0) then
         write(LER,*)'SCSPEC: no header read from unit ',luin
         write(LER,*)'FATAL'
         stop
      endif
      call saver(line_header, 'NumSmp', NumSmp, LINEHEADER)
      call saver(line_header, 'NumTrc', NumTrc, LINEHEADER)
      call saver(line_header, 'NumRec', NumRec, LINEHEADER)

c save out hlh and line header
      call savhlh(line_header, lbytes, lbyout)
      call wrtape(luout, line_header, lbyout)

c set up pointers to header mnemonics StaCor, RecNum and TrcNum
      call savelu('StaCor', ifmt_StaCor, l_StaCor, ln_StaCor,
     1            TRACEHEADER)

c constants
      lenhed = ITRWRD
      lbyout = SZTRHD + (NumSmp * SZSMPD)

c output parameters
      write(LERR,*) 'Parameters:'
      write(LERR,*) 'infile = ', infile
      write(LERR,*) 'scale_file = ', scale_file
      write(LERR,*) 'outfile = ', outfile
      write(LERR,*) 'prew = ', prew
      write(LERR,*) 'integer output? = ', intflg
      write(LERR,*) 'scale to max? =', maxflg

c determine sizes
      peak_size = NumSmp * SZSMPD
      record_size = ((NumSmp + ITRWRD) * SZSMPD) * NumTrc
      live_size = SZSMPD * NumTrc
      total_size = peak_size + record_size + live_size

c allocate memory
      errcds = 0
      call galloc(memadr_peak, peak_size, errcd, abort)
      errcds = errcds + errcd
      call galloc(memadr_record, record_size, errcd, abort)
      errcds = errcds + errcd
      call galloc(memadr_live, live_size, errcd, abort)
      errcds = errcds + errcd

      if ( errcds .ne. 0 ) then
         write(LERR,*)' '
         write(LERR,*)'Unable to allocate workspace:'
         write(LERR,*) total_size, '  bytes'
         write(LERR,*)' '
         write(LER,*)' '
         write(LER,*)'Unable to allocate workspace:'
         write(LER,*) total_size, '  bytes'
         write(LER,*)' '
         call lbclos(luin)
         close(luscale)
         call lbclos(luout)
         write(LERR,*)'SCSPEC: ABNORMAL Termination'
         write(LER,*)'SCSPEC: ABNORMAL Termination'
         stop
      else
         write(LERR,*) ' '
         write(LERR,*) 'Allocating workspace:'
         write(LERR,*) 'peak = ', peak_size
         write(LERR,*) 'record = ', record_size
         write(LERR,*) 'live = ', live_size
         write(LERR,*) 'total = ', total_size
      endif

c initialize memory
      call vclr(peak, 1, peak_size/SZSMPD)
      call vclr(record, 1, record_size/SZSMPD)
      call vclr(live, 1, live_size/SZSMPD)

c do the work
      call process(peak, record, live,
     1             NumRec, NumTrc, NumSmp, prew, intflg,
     3             maxflg, luin, luscale, luout, lenhed, lbyout,
     4             ifmt_StaCor, l_StaCor, ln_StaCor,
     5             ierr, LER, LERR)

c check error code
      if (ierr .ne. 0) then
         write(LERR,*)'SCSPEC: Unsuccessful completion'
         write(LER,*)'SCSPEC: Unsuccessful completion'
      else
         write(LERR,*)'SCSPEC: Successful completion'
         write(LER,*)'SCSPEC: Successful completion'
      endif

c close all files
      call lbclos(luin)
      close(luscale)
      call lbclos(luout)

c exit
      call exit(0)
      end

      subroutine process(peak, record, live,
     1                   NumRec, NumTrc, NumSmp, prew, intflg,
     3                   maxflg, luin, luscale, luout, lenhed, lbyout,
     4                   ifmt_StaCor, l_StaCor, ln_StaCor,
     5                   ierr, LER, LERR)
      implicit none

      integer NumRec, NumTrc, NumSmp
      integer luin, luscale, luout, lenhed, lbyout
      integer ifmt_StaCor, l_StaCor, ln_StaCor
      integer LER, LERR
      real peak(NumSmp)
      real rf, ra, maxpeak, eps, prew, fac, off
      logical live(NumTrc), intflg, maxflg
      real record(-lenhed+1:NumSmp, NumTrc)

      integer irec, itrc, ismp, ifreq, ierr, io
      logical newfmt

      ierr = 0

c There have been two versions of the spec_scale_file.  The first
c puts out in the format:
c
c frequency max_amplitude
c rms_amplitude
c frequency max_amplitude
c rms_amplitude
c ...
c
c The second is the more compact:
c
c frequency max_amplitude rms_amplitude
c frequency max_amplitude rms_amplitude
c ...
c
c By simply counting the number of lines in the scale file we can tell
c which version we have.  Then we can rewind the file and read in the
c data.
      read(luscale,*) ismp
      read(luscale,*)
      io = 0
10    if (io .eq. 0) then
         ifreq = ifreq + 1
         read(luscale,*,iostat=io)
c        Make sure iostat is working ok...some compilers, well you know...
         if (ifreq .gt. 3*ismp) io = -1
         goto 10
      endif
      if (ifreq .ge. 2*ismp) then
         newfmt = .false.
         write(LERR,*)'Using old spec_scale_file format'
      else
         newfmt = .true.
      endif
      rewind(luscale)

c read in the spec coeffecients (ignore first two lines)
      read(luscale,*) ismp
      read(luscale,*)
      if (ismp .ne. NumSmp) then
c        ismp must match NumSmp or spec_scale_file is out of date!
         write(LER,*)'spec_scale_file does not match input data!'
         write(LER,*)'   ismp = ', ismp
         write(LER,*)'   NumSmp = ', NumSmp
         write(LER,*)'Check to make sure this is the correct'
         write(LER,*)'scale file before proceeding.'
         write(LERR,*)'spec_scale_file does not match input data!'
         write(LERR,*)'   ismp = ', ismp
         write(LERR,*)'   NumSmp = ', NumSmp
         write(LERR,*)'Check to make sure this is the correct'
         write(LERR,*)'scale file before proceeding.'
         call exitfu(-10)
      endif
      maxpeak = -99999.0
      do ifreq = 1, ismp
         if (newfmt) then
            read(luscale,*) rf, peak(ifreq), ra
         else
            read(luscale,*) rf, peak(ifreq)
            read(luscale,*) ra
         endif
         if (peak(ifreq) .gt. maxpeak) maxpeak = peak(ifreq)
      enddo

c Prewhiten the peak values
      eps = prew * maxpeak
      do ifreq = 1, ismp
         peak(ifreq) = peak(ifreq) + eps
      enddo

c Set interpolation parms
      if (intflg) then
          fac = 255.0
          off = -128.0
      elseif (maxflg) then
          fac = maxpeak
          off = 0.0
      else
          fac = 1.0
          off = 0.0
      endif

c read in each record, scale it and write it out
      do irec = 1, NumRec
c        Read the next record
         call read_record(record, live, NumTrc, NumSmp,
     1                    luin, LERR, lenhed,
     2                    ifmt_StaCor, l_StaCor, ln_StaCor, ierr)
         if (ierr .ne. 0) return

c        Scale it
         do itrc = 1, NumTrc
            if (live(itrc)) then
               do ifreq = 1, ismp
                  rf = record(ifreq, itrc) / peak(ifreq)
                  record(ifreq, itrc) = rf * fac + off
               enddo
            endif
         enddo

c        Write out the scaled record
         call write_record(record, NumTrc, NumSmp,
     1                     luout, lenhed, lbyout)
      enddo

      end

      subroutine cmdln(infile, scale_file, outfile, prew, name,
     1                 intflg, maxflg, LER, LERR)
      implicit none
      character*(*) infile, scale_file, outfile, name
      real prew
      integer LER, LERR
      logical argis, intflg, maxflg

      call argstr('-N', infile, ' ', ' ')
      call argstr('-scale', scale_file,
     1            'spec_scale_file', 'spec_scale_file')
      call argstr('-O', outfile, ' ', ' ')
      call argr4('-prew', prew, 0.0, 0.0)

      intflg = argis('-int')
      maxflg = argis('-max')

c check for extraneous arguments and abort if found
      call xtrarg(name, LER, .FALSE., .FALSE.)
      call xtrarg(name, LERR, .FALSE., .TRUE.)
      end

      subroutine scspec_help(LER)
      implicit none
      integer LER
      write(LER,*)' '
      write(LER,*)'===================================================='
      write(LER,*)' '
      write(LER,*)' Command Line Arguments for scspec:'
      write(LER,*)' '
      write(LER,*)' For a more detailed description of these parameters'
      write(LER,*)' see the online man page using uman, xman or xuspman'
      write(LER,*)' '
      write(LER,*)'-N[]     -- input data set                 (stdin)'
      write(LER,*)'-scale[] -- scale info from spec (spec_scale_file)'
      write(LER,*)'-O[]     -- output data set               (stdout)'
      write(LER,*)'-prew[]  -- prewhitener                      (0.0)'
      write(LER,*)'-int     -- produce output from -128..+127  (0..1)'
      write(LER,*)'-max     -- scale to max of input data (0..1)'
      write(LER,*)' '
      write(LER,*)'Usage:'
      write(LER,*)'       scspec -N[] -scale[] -O[] -prew[] -int -max'
      write(LER,*)' '
      write(LER,*)'===================================================='
      end
