C***********************************************************************
C                 copyright 2001, Amoco Production Company             *
C                             All Rights Reserved                      *
C                     an affiliate of BP America Inc.                  *
C***********************************************************************
      program stat3d	
		
      implicit none
		
c-----------------------------------------------------------------------
c Kelly D. Crawford 	04/14/97
c
c Bertram Kaufhold 	05/01/98	Modification
c
c This programm was written specifically to smooth the p and q outputs 
c from c2p6 so they can be used as input to c3p6.
c
c The simple operation being done is to obtain the arithmetic mean of a
c 3-d running subcube which moves within a 3-d data cube. 
c Given a window halfsize of iap x jap x kap defined around a center
c point M(im,jm,km), the formula for computing each point in the output is:
c
c                     +-                                  -+
c                   1 | im+iap   jm+jap   km+kap           |
c    M(im,jm,km) =  - |  sum      sum      sum     S(i,j,k)|
c                   N |i=im-iap j=jm-jap k=km-kap          |
c                     +-                                  -+
c
c where N is (2*iap+1)*(2*jap+1)*(2*kap+1).  So we sum all the samples
c in the entire window, and divide by the total number of samples.
c
c For each trace read, compute all possible sums over the window length.
c Now we no longer have to sum in the z direction (sample_sums).
c After an entire record has been read in, using the z sums, compute all
c possible sums in the x direction (trace_sums).  Now we need only add
c and drop these trace sums as we read in each record (see process.F.)
c
c                      z-direction
c		       index i -> number of samples
c		       height  -> traces
c			
c		       :                  y-direction
c		       :                . index k -> number of records
c		       :             .    width   -> cross line
c		       :          .
c		       :       .
c		       :    . 
c		       : .                   x-direction
c		       :...................> index j -> number of traces
c					      length  -> in line
c
c  ii = height analysis window size 
c  jj = length analysis window size
c  kk = width  analysis window size
c  iap = half height analysis window size
c  jap = half length analysis window size
c  kap = half width  analysis window size
c  im  = center point of height analysis window
c  jm  = center point of length analysis window
c  km  = center point of width  analysis window 
c-----------------------------------------------------------------------

c get machine dependent parameters 

#include <f77/iounit.h>
#include <f77/lhdrsz.h>
#include <f77/sisdef.h>
#include <save_defs.h> 

c dimension standard USP variables 
c SZLNHD is defined in lhdrsz.h. It provides room to store a trace 
c (including both trace header and trace data). SZLNHD can also be
c used to store the line header (big enough to store line header).
c SZLNHD => line file size = (SZTRHD + SZSAMP + MAXSMP)/2
c SZTRHD => size of trace header in pipes in byte
c SZSAMP => size of floating point sample in byte
c MAXSMP => ?????

      integer     itr (SZLNHD)
      integer     luin , luout, lbytes, lbyout, lenhed
      integer     argis, jerr
      real        tstart, tend, tskip
      integer     istart, iend, iskip, isamp
      integer     jtstart, jtslice, junits
      character   ntap*(255), otap*(255), name*(6)
      logical     query, fdead

c Program Specific - dynamic memory variables

      real cputim(10), waltim(10)
      real v10, vtot10, w10, wtot10

      integer errcd, errcds, abort
      integer u_size, uout_size, uhdr_size, tsum_size
      integer ssum_size, recip_size, live_size
      integer liverec_size, livesum_size 
      integer i_mult_num_size, i_mult_asg_size
      integer livesum_num_size, livesum_asg_size, total_size
      integer u_size_bytes, uout_size_bytes
      integer uhdr_size_bytes, tsum_size_bytes, ssum_size_bytes
      integer recip_size_bytes, live_size_bytes, liverec_size_bytes
      integer livesum_size_bytes, total_size_bytes
      integer i_mult_num_size_bytes, i_mult_asg_size_bytes 
      integer livesum_num_size_bytes, livesum_asg_size_bytes
      real TmMsFS, ILClIn, CLClIn
      real apheight, aplength, apwidth, ildm, cldm
      real dtmsec
      integer SmpInt, NumSmp, NumTrc, NumRec, Nx_Pad
      integer ii, jj, kk, iap, jap, kap, im, jm, km

c ----------------------------------------------------------------------

c Uncomment the following line and adjust the array sizes in the else
c portion of the ifndef block below if you want to do bounds checking.
c #define STATIC_ALLOCATION_VERSION

#ifndef STATIC_ALLOCATION_VERSION

      double precision tsum, ssum
      real u, uout, uhdr, recip 
      integer live, liverec, livesum, i_mult_num, i_mult_asg 
      integer livesum_num, livesum_asg  
      pointer (memadr_u,       u(2))
      pointer (memadr_uout,    uout(2))
      pointer (memadr_uhdr,    uhdr(2))
      pointer (memadr_tsum,    tsum(2))
      pointer (memadr_ssum,    ssum(2))
      pointer (memadr_recip,   recip(2))
      pointer (memadr_live,    live(2))
      pointer (memadr_liverec, liverec(2))
      pointer (memadr_livesum, livesum(2))
      pointer (memadr_i_mult_num, i_mult_num(2))
      pointer (memadr_i_mult_asg, i_mult_asg(2))
      pointer (memadr_livesum_num, livesum_num(2))
      pointer (memadr_livesum_asg, livesum_asg(2))
#else

c     This is just for bounds checking.  Adjust these arrays to fit the
c     test dataset before running.  Note: These numbers are currently
c     random!

      real u(7878), uout(90), uhdr(1170), recip(1232)
      double precision tsum(2626), ssum(19392)
      integer live(1000), liverec(1000), livesum(1000), i_mult_num(10)
      integer i_mult_asg(10), livesum_num(10), livesum_asg(1000)

#endif

c Program Specific static memory variables

      integer ifmt_StaCor, l_StaCor, ln_StaCor

c Initialize variables

      data abort/0/
      data name/'STAT3D'/ 

c start the overall timer

      call timstr(v10,w10)

c give command line help if requested

      query = (argis ('-?') .gt. 0 .or. argis ('-h') .gt. 0)
      if (query)then
         call help()
         stop
      endif

c Let us know someone is using this thing

      call tattle('stat3d')

c open printout file

#include <f77/open.h>

c-----------------------------------------------------------------------
c get command line input parameters

      call cmdln(name, ntap, otap,
     1           tstart, tskip, tend,
     2           apheight, aplength, apwidth,
     3           ildm, cldm, fdead)

c-----------------------------------------------------------------------
c open input and output files
c Synopsis: getln(luin , ntap, mode, defval) => get logical unit number 
c luin  => logical unit number of file opened
c ntap  => string of file to be opened
c mode  => 
c         'r' - open for read
c         'w' - create for writing
c         'a' - open for writing at end of file or create for writing
c         'a+' - open for read/write position at end
c         'r+' - open for read/write position at beginning
c         'w+' - open for read/write position at beginning, truncate
c defval=> default value of ntap if == ' '

      call getln(luin , ntap,'r', 0)
      call getln(luout, otap,'w', 1)

c-----------------------------------------------------------------------
c read input line header and save certain parameters
c Synopsis: rtape(unit, buffer, length)
c unit   => unit number to read from
c buffer => buffer to read data into
c length => size of record read in bytes

      call rtape(luin,itr,lbytes)
      if(lbytes.eq.0)then
         write(LER,*)'STAT3D: no header read from unit ',luin
         write(LER,*)'FATAL'
         stop
      endif

c-----------------------------------------------------------------------
c obtain line header information 
c 'NumSmp' => Number of samples per trace
c 'SmpInt' => Sample interval 
c 'NumTrc' => Number of traces per record
c 'NumRec' => Number of records per line
c 'Nx_Pad' => Number of padded traces per record
c 'ILClIn' => In line cell increment (in line spacing)
c 'CLClIn' => Cross line cell increment (cross line spacing)
c 'TmMsFS' => Time in miliseconds of first sample

      call saver(itr, 'NumSmp', NumSmp, LINEHEADER)
      call saver(itr, 'SmpInt', SmpInt, LINEHEADER)
      call saver(itr, 'NumTrc', NumTrc, LINEHEADER)
      call saver(itr, 'NumRec', NumRec, LINEHEADER)
      call saver(itr, 'Nx_Pad', Nx_Pad, LINEHEADER)
      call saver(itr, 'ILClIn', ILClIn, LINEHEADER)
      call saver(itr, 'CLClIn', CLClIn, LINEHEADER)
      call saver(itr, 'TmMsFS', TmMsFS, LINEHEADER)

c check if the in line spacing and the cross line spacing are set to
c their default values, i.e., check if they are not one of the arguments
c listed in the command line. If they were specified in the command line
c then use this specifications to overwrite the header in line- and
c cross line spacing information

      if (ildm .ne. 0) ILClIn = ildm
      if (cldm .ne. 0) CLClIn = cldm

c check if the in line- and/or cross line spacing are set to zero. If
c they are then they have to be specified in the command line
c Note: LER specifies the terminal as output unit (unbuffered)
c       LERR specifies the print file as output unit (buffered)

      if (ILClIn .eq. 0 .or. CLClIn .eq. 0) then
         write(LER,*)'STAT3D: Error -- -ildm and -cldm values'
         write(LER,*)'were not in the line header.  Please'
         write(LER,*)'specify them explicitly on the command line'
         write(LERR,*)'STAT3D: Error -- -ildm and -cldm values'
         write(LERR,*)'were not in the line header.  Please'
         write(LERR,*)'specify them explicitly on the command line'
         call exitfu(-1)
      endif

c-----------------------------------------------------------------------
c calculate the sample increment in msec.
c 'T_unit' => Time Units (0=milliseconds, 1=microseconds)
c USP convention is that if SmpInt > 16, it is microseconds

      call saver(itr, 'T_Unit', junits, LINEHEADER)
      if(junits .eq. 1 .or. SmpInt .gt. 16) then
         junits = 1
         call savew(itr, 'T_Unit', junits, LINEHEADER)
         dtmsec = .001 * SmpInt
      else
         dtmsec = SmpInt
      endif

c-----------------------------------------------------------------------
c check tstart, tend, tskip specified in command line to derive 
c sample ranges 
c If no command arguments for tstart, tskip and tend have been defined
c then the default values for istart, iskip and iend are set to

      istart = 1
      iskip  = 1		
c     iend   = NumSmp

c Note: for negative entries of tstart, tskip or tend we also assign
c the same default values for istart, iskip and iend as mentioned above 


      if (tstart .gt. TmMsFS) then
         istart = ((tstart-TmMsFS) / dtmsec) + 1
         if ((istart-1)*dtmsec+TmMsFS .lt. tstart) istart=istart+1
         if (istart .gt. NumSmp) istart = NumSmp
      endif
      if (tskip .gt. 0) then
         iskip = (tskip / dtmsec) 
         if (iskip .lt. 1) iskip = 1
         if (iskip .gt. NumSmp) iskip = NumSmp
      endif
      if (tend .gt. TmMsFS) then      
         iend = ((tend-TmMsFS) / dtmsec) +1 
         if (iend .lt. 1) iend = 1
         if (iend .gt. NumSmp) iend = NumSmp
         iend = istart + iskip * ((iend-istart)/iskip)
      else
         iend = NumSmp
         iend = istart + iskip * ((iend-istart)/iskip)
      endif	

      isamp = (iend - istart)/iskip + 1 
      
c-----------------------------------------------------------------------
c Recalculate tstart, tend and tskip so that they match the actual 
c sampling grid of the time traces used in the application
c 'TmMsFS' 	=> 	Time in ms of first sample
c tstart 	= 	time start point
c tskip  	= 	time skip
c tend   	= 	time end point 

      tstart = TmMsFS + (istart-1) * dtmsec
      tskip  = dtmsec * iskip            
      tend   = TmMsFS + (iend-1) * dtmsec 

c check if tstart <= tend, otherwise abort and print error message 

      if (tstart .gt. tend) then
         write(LER,*)'STAT3D: !!!!!!!!!!  Error !!!!!!!!!!!'
         write(LER,*)'-tstart argument greater than -tend argument'
         write(LER,*)'     Please modify them on command line'
         write(LERR,*)'STAT3D: !!!!!!!!!!  Error !!!!!!!!!!!'
         write(LERR,*)'-tstart argument greater than -tend argument'
         write(LERR,*)'     Please modify them on command line'
         call exitfu(-1)
      endif

c-----------------------------------------------------------------------
c Compute window sizes
c the default values of apheight, aplength and apwidth are defined in 
c the command subroutine cmdln and are set to 0.0

c if apheight is not defined as an arguments in the command line
c or it is <= tskip  -> assign it to the new sample interval tskip
                                               
      if (apheight .eq. 0.0) then
         apheight = 3 * tskip
      else if (apheight .ne. 0.0 .and. apheight .le. tskip) then
         apheight = tskip
      endif
      iap = int(apheight / tskip)

c if half height analysis window size is smaller than one 
c -> assign it to one

      apheight = tskip * iap    
      			
c if the height analysis window size is larger then the existing number
c of samples per trace then set the former equal to the latter

      ii = iap + iap + 1	
      if (ii .gt. isamp) then	 

c the height analysis window size has to be odd. If it is not decrement
c by one    				      				 
      			
         ii = isamp

c check if ii > 2, otherwise abort and print error message 

         if (ii .le. 2) then
            write(LER,*)'STAT3D: !!!!!!!!!!!!!  Error !!!!!!!!!!!!!!'
            write(LER,*)'Due to your command line specifications for'
            write(LER,*)'-tstart, -tskip and -tend there are less than'
            write(LER,*)'3 sample points left per trace for processing'
            write(LERR,*)'STAT3D: !!!!!!!!!!!!!  Error !!!!!!!!!!!!!!'
            write(LERR,*)'Due to your command line specifications for'
            write(LERR,*)'-tstart, -tskip and -tend there are less than'
            write(LERR,*)'3 sample points left per trace for processing'
            call exitfu(-1)
         endif

         if (mod(ii,2) .eq. 0) ii = ii - 1       
         iap = ii / 2 		
         apheight = tskip * iap    
      endif
      im = iap + 1

c if aplength is not defined as an arguments in the command line or it
c is <= ILClIn -> assign it to the in line cell increment (ILClIn)
                                               
      if (aplength .eq. 0.0) then
         aplength = 3 * ILClIn
      else if (aplength .ne. 0.0 .and. aplength .le. ILClIn) then
         aplength = ILClIn 
      endif
      jap = int(aplength / ILClIn) 

c if half length analysis window size is smaller than one 
c -> assign it to one
                 
      aplength = ILClIn * jap   

c if the length analysis window size is larger then the existing number
c of traces per record then set the former equal to the latter
      				
      jj = jap + jap + 1
      if (jap .gt. NumTrc) then	
      				
c the length analysis window size has to be odd. If it is not decrement
c by one    				
      			
         jj = NumTrc
         
c check if jj > 2, otherwise abort and print error message 

         if (jj .le. 2) then
            write(LER,*)'STAT3D: !!!!!!!!!!!!!  Error !!!!!!!!!!!!!!'
            write(LER,*)'There are less than 3 traces available in'
            write(LER,*)'           x-direction'
            write(LERR,*)'STAT3D: !!!!!!!!!!!!!  Error !!!!!!!!!!!!!!'
            write(LERR,*)'There are less than 3 traces available in'
            write(LERR,*)'           x-direction'  
            call exitfu(-1)
         endif

         if (mod(jj,2) .eq. 0) jj = jj - 1				         			
         jap = jj / 2	 
         aplength = ILClIn * jap   
      endif
      jm = jap + 1

c Note: The half length analysis window size (jap) should never be 
c bigger than the number of padded traces (Nx_Pad) if you plan to 
c use the output of the stat3d routine to merge splitted records back
c together. Otherwise the first and last jap-Nx_Pad traces of the stat3d
c output file do not correspond to the correct values of the overall    
c record. Output warning message if this is the case!

      if (jap .gt. Nx_Pad) then
         write(LER,*)'STAT3D: !!!!!!!!!! Warning !!!!!!!!!!!'
         write(LER,*)'  The half lenght analysis window size (jap) is'
         write(LER,*)'greater than the number of padded traces (Nx_Pad)'
         write(LER,*)'  !!!!Merging will not give correct results!!!!' 
         write(LERR,*)'STAT3D: !!!!!!!!!! Warning !!!!!!!!!!!'
         write(LERR,*)'  The half lenght analysis window size (jap) is'
        write(LERR,*)'greater than the number of padded traces (Nx_Pad)'
         write(LERR,*)'  !!!!Merging will not give correct results!!!!' 
      endif

c if apwidth is not defined as an arguments in the command line or it
c is <= CLClIn -> assign it to the cross line cell increment (CLClIn)
                                            
      if (apwidth .eq. 0.0) then
         apwidth = 3 * CLClIn
      elseif (apwidth .ne. 0.0 .and. apwidth .le. CLClIn) then
         apwidth = CLClIn 
      endif
      kap = int(apwidth / CLClIn)            

c if half width analysis window size is smaller than one 
c -> assign it to one

      apwidth = CLClIn * kap 

c if the width analysis window size is larger then the existing number 
c of records per data set then set the former equal to the latter
      			
      kk = kap + kap + 1
      if (kap .gt. NumRec) then
      				 
c the width analysis window size has to be odd, if it is not decrement
c by one
      				
         kk = NumRec

c check if kk > 2, otherwise abort and print error message 

         if (kk .le. 2) then
            write(LER,*)'STAT3D: !!!!!!!!!!!!!  Error !!!!!!!!!!!!!!'
            write(LER,*)'There are less than 3 traces available in'
            write(LER,*)'           y-direction'
            write(LERR,*)'STAT3D: !!!!!!!!!!!!!  Error !!!!!!!!!!!!!!'
            write(LERR,*)'There are less than 3 traces available in'
            write(LERR,*)'           y-direction'  
            call exitfu(-1)
         endif

         if (mod(kk,2) .eq. 0) kk = kk - 1         			         			
         kap = kk / 2
         apwidth = CLClIn * kap 
      endif
      km = kap + 1
      
c-----------------------------------------------------------------------
c update the historical part of the line header  

      call hlhprt(itr, lbytes, name, 6, LERR)

c-----------------------------------------------------------------------
c if we have constrained the sample ranges, set header values accordingly

      if (istart .gt. 1 .or. iend .lt. NumSmp) then
         call savew(itr, 'NumSmp', isamp, LINEHEADER)
         if (istart .gt. 1) then
            call savew(itr, 'TmMsFS', tstart, LINEHEADER)
         endif
      endif
      if (iskip .gt. 1) then 
         call savew(itr, 'SmpInt', SmpInt*iskip, LINEHEADER)
      endif

c we will strip all Nx_Pad'ed traces in the output, so
c adjust the number of traces (in the lineheader) if Nx_Pad > 0

      if (Nx_Pad .gt. 0) then
         call savew(itr, 'NumTrc', NumTrc-(Nx_Pad*2), LINEHEADER)
      endif

c Add stuff specifically for c3p6
c 'TmMSSl' => Time in milliseconds of first slice
c 'TmSlIn' => Time slice increment in milliseconds

      jtstart = nint(tstart)
      jtslice = nint(tskip)
      call savew(itr, 'TmMsSl', jtstart, LINEHEADER)
      call savew(itr, 'TmSlIn', jtslice, LINEHEADER)

c save out hlh and line header

      call savhlh(itr, lbytes, lbyout)
      call wrtape(luout, itr, lbyout)

c set up pointers to header mnemonic StaCor

      call savelu('StaCor', ifmt_StaCor, l_StaCor, ln_StaCor,
     1     TRACEHEADER)

c-----------------------------------------------------------------------
c dump some stats to the print file

      write(lerr,*)
      write(lerr,'(a,t40,i5,t50,a)') 'input file name',
     1                         luin,ntap
      write(lerr,'(a,t40,i5,t50,a)') 'output file name',
     1                         luout,otap

      write(lerr,'(a,t40,i5)') 'input sample interval',SmpInt
      write(lerr,'(a,t40,i5)') 'input number of samples',NumSmp
      write(lerr,'(a,t40,f12.6)') 'input:time in ms of first sample',
     1                             TmMsFS
      write(lerr,'(a,t40,i5)') 'number of traces',NumTrc
      write(lerr,'(a,t40,i5)') 'number of records',NumRec
      write(lerr,'(a,t40,f12.6)') 'inline trace spacing', ILClIn
      write(lerr,'(a,t40,f12.6)') 'crossline trace spacing', CLClIn

      write(lerr,'(a,t40,f12.6)') 'apheight',apheight
      write(lerr,'(a,t40,f12.6)') 'aplength',aplength
      write(lerr,'(a,t40,f12.6)') 'apwidth',apwidth

      write(lerr,'(a,t40,i5)') 'ii',ii,'jj',jj,'kk',kk
      write(lerr,'(a,t40,i5)') 'im',im,'jm',jm,'km',km
      write(lerr,'(a,t40,i5)') 'output number of samples',isamp
      write(lerr,'(a,t40,f12.6)') 'tstart',tstart
      write(lerr,'(a,t40,f12.6)') 'tskip',tskip 
      write(lerr,'(a,t40,f12.6)') 'tend',tend

      write(lerr,'(a,t40,i5)') 'nxpad',Nx_Pad
      write(lerr,'(a,t40,i5)') 'fdead',fdead

c-----------------------------------------------------------------------
c dynamic memory allocation: look in /home/usp/include/c/lhdrsz.h
c SZSMPD => size of sample in pipe (in bytes) in bytes
c ITRWRD => number of full words (SZSMPD-bytes) in trace header

      u_size             	= NumSmp + ITRWRD
      u_size_bytes       	= u_size * SZSMPD

      uout_size          	= isamp * NumTrc
      uout_size_bytes    	= uout_size * SZSMPD

      uhdr_size          	= NumTrc * (kk+1) * ITRWRD
      uhdr_size_bytes    	= uhdr_size * SZSMPD

      ssum_size          	= isamp * NumTrc
      ssum_size_bytes    	= ssum_size * SZSMPD * 2

      tsum_size          	= isamp * NumTrc * (kk+1)
      tsum_size_bytes   	 = tsum_size * SZSMPD * 2

      recip_size         	= im * kk * jj
      recip_size_bytes   	= recip_size * SZSMPD

      live_size          	= NumTrc * (kk+1)
      live_size_bytes    	= live_size * SZSMPD

      liverec_size       	= NumTrc
      liverec_size_bytes 	= live_size * SZSMPD

      livesum_size       	= NumTrc
      livesum_size_bytes 	= livesum_size * SZSMPD

      i_mult_num_size		= im
      i_mult_num_size_bytes	= i_mult_num_size * SZSMPD
      
      i_mult_asg_size    	= isamp
      i_mult_asg_size_bytes 	= i_mult_asg_size * SZSMPD

      livesum_num_size		= NumTrc
      livesum_num_size_bytes	= livesum_num_size * SZSMPD

      livesum_asg_size		= NumTrc
      livesum_asg_size_bytes	= livesum_asg_size * SZSMPD 

      total_size       = u_size + uout_size + tsum_size+
     1                   uhdr_size + ssum_size + recip_size+
     2                   live_size + liverec_size + livesum_size+
     3                   i_mult_num_size + i_mult_asg_size + 
     4                   livesum_num_size + livesum_asg_size
      total_size_bytes = u_size_bytes + uout_size_bytes+
     1                   tsum_size_bytes + uhdr_size_bytes+
     2                   ssum_size_bytes + recip_size_bytes+
     3                   live_size_bytes + liverec_size_bytes+
     4                   livesum_size_bytes + i_mult_num_size_bytes+
     5                   i_mult_asg_size_bytes + livesum_num_size_bytes+
     6                   livesum_asg_size_bytes

      errcds = 0
      
#ifndef STATIC_ALLOCATION_VERSION

      call galloc(memadr_u,       u_size_bytes,     errcd, abort)
      errcds = errcds + errcd
      call galloc(memadr_uout,    uout_size_bytes,  errcd, abort)
      errcds = errcds + errcd
      call galloc(memadr_uhdr,    uhdr_size_bytes,  errcd, abort)
      errcds = errcds + errcd
      call galloc(memadr_ssum,    ssum_size_bytes,  errcd, abort)
      errcds = errcds + errcd
      call galloc(memadr_tsum,    tsum_size_bytes,  errcd, abort)
      errcds = errcds + errcd
      call galloc(memadr_recip,   recip_size_bytes, errcd, abort)
      errcds = errcds + errcd
      call galloc(memadr_live,    live_size_bytes,  errcd, abort)
      errcds = errcds + errcd
      call galloc(memadr_liverec, liverec_size_bytes,  errcd, abort)
      errcds = errcds + errcd
      call galloc(memadr_livesum, livesum_size_bytes,  errcd, abort)
      errcds = errcds + errcd
      call galloc(memadr_i_mult_num, i_mult_num_size_bytes,
     1            errcd, abort)
      errcds = errcds + errcd
      call galloc(memadr_i_mult_asg, i_mult_asg_size_bytes,
     1            errcd, abort)
      errcds = errcds + errcd
      call galloc(memadr_livesum_num, livesum_num_size_bytes,
     1            errcd, abort)
      errcds = errcds + errcd
      call galloc(memadr_livesum_asg, livesum_asg_size_bytes,
     1            errcd, abort)
      errcds = errcds + errcd

#endif

c-----------------------------------------------------------------------
      if ( errcds .ne. 0 ) then
         write(LERR,*)' '
         write(LERR,*)'Unable to allocate workspace:'
         write(LERR,*) total_size,       '  words'
         write(LERR,*) total_size_bytes, '  bytes'
         write(LERR,*)' '
         write(LER,*)' '
         write(LER,*)'Unable to allocate workspace:'
         write(LER,*) total_size,       '  words'
         write(LER,*) total_size_bytes, '  bytes'
         write(LER,*)' '

c-----------------------------------------------------------------------
c close the units with unit number luin and luout
         call lbclos(luin)
         call lbclos(luout)
         
         write(LERR,*)'STAT3D: ABNORMAL Termination'
         write(LER,*)'STAT3D: ABNORMAL Termination'
         stop
      else
         write(LERR,*)' '
         write(LERR,*)'Allocating workspace:'
         write(LERR,*) 'u       	= ',u_size_bytes
         write(LERR,*) 'uout    	= ',uout_size_bytes
         write(LERR,*) 'uhdr    	= ',uhdr_size_bytes
         write(LERR,*) 'ssum    	= ',ssum_size_bytes
         write(LERR,*) 'tsum    	= ',tsum_size_bytes
         write(LERR,*) 'recip   	= ',recip_size_bytes
         write(LERR,*) 'live    	= ',live_size_bytes
         write(LERR,*) 'liverec 	= ',liverec_size_bytes
         write(LERR,*) 'livesum 	= ',livesum_size_bytes
         write(LERR,*) 'i_mult_num 	= ',i_mult_num_size_bytes
         write(LERR,*) 'i_mult_asg 	= ',i_mult_asg_size_bytes
         write(LERR,*) 'livesum_num 	= ',livesum_num_size_bytes 
         write(LERR,*) 'livesum_asg 	= ',livesum_asg_size_bytes
         write(LERR,*) 'total   	= ',total_size,       ' words'
         write(LERR,*) 'total   	= ',total_size_bytes, ' bytes'
         write(LERR,*)' '
      endif

c-----------------------------------------------------------------------
c initialize memory to zero 
c Synopsis vclr(c, ic, n) look in /home/usp/Tomography/src/cmd/femodel/vclr.
c c  => vector which will be set to zero (0.0)
c n  => integer input element countf
c ic => integer input stride

      call vclr(u,       	1, u_size)
      call vclr(uout,    	1, uout_size)
      call vclr(uhdr,    	1, uhdr_size)
      call vclr(ssum,    	1, ssum_size)
      call vclr(tsum,    	1, tsum_size)
      call vclr(recip,   	1, recip_size)
      call vclr(live,    	1, live_size)
      call vclr(liverec, 	1, liverec_size)
      call vclr(livesum, 	1, livesum_size)
      call vclr(i_mult_num, 	1, i_mult_num_size)
      call vclr(i_mult_asg, 	1, i_mult_asg_size)
      call vclr(livesum_num, 	1, livesum_num_size)
      call vclr(livesum_asg, 	1, livesum_asg_size)

c length of a trace header
     
      lenhed = ITRWRD

c calculate the number of output bytes 
c SZTRHD => size of trace header in pipe in bytes
c SZSMPD => size of sample in pipe in bytes

      lbyout = SZTRHD + (isamp * SZSMPD)

c-----------------------------------------------------------------------      						
c call main processing loop

      call process(u, uout, uhdr, tsum, ssum, recip, live,
     1             liverec, livesum, i_mult_num, i_mult_asg, 
     2             livesum_num, livesum_asg, ii, jj, kk, im, 
     3             jm, km, isamp, istart, iskip, iend, NumSmp, NumTrc,
     4             NumRec, lenhed, lbyout, Nx_Pad, luin, luout, ler,
     5             lerr, cputim, waltim,
     6             ifmt_StaCor, l_StaCor, ln_StaCor, fdead)

c-----------------------------------------------------------------------
c write out timing statistics

      call timend(cputim(10),v10,vtot10,waltim(10),w10,wtot10)
      write(lerr,'(A30,2A15,/)') 'routine','cpu time','wall time'
      write(lerr,'(A30,2f15.3)')
     1         'read input',cputim(1),waltim(1),
     1         'smooth',cputim(2),waltim(2),
     1         'write output',cputim(3),waltim(3),
     1         'total',cputim(10),waltim(10)

c close data files 

      call lbclos(luin)
      call lbclos(luout)

      write(LERR,*)'STAT3D: Normal Termination'
      write(LER,*) 'STAT3D: Normal Termination'
      stop

      end
      
c-----------------------------------------------------------------------
      subroutine help()
c provide terse online help
#include <f77/iounit.h>

      write(LER,*)' '
      write(LER,*)'===================================================='
      write(LER,*)' '
      write(LER,*)' Command Line Arguments for stat3d:'
      write(LER,*)' '
      write(LER,*)' For a more detailed description of these parameters'
      write(LER,*)' see the online man page using uman, xman or xuspman'
      write(LER,*)' '
      write(LER,*)'-N[]        -- input file                    (stdin)'
      write(LER,*)'-AMEAN[]    -- output file                  (stdout)'
      write(LER,*)'-apheight[] -- window half height (ms)       (tskip)'
      write(LER,*)'-aplength[] -- window half length (m)       (ILClIn)'
      write(LER,*)'-apwidth[]  -- window half width (m)        (CLClIn)'
      write(LER,*)'-ildm[]     -- inline trace spacing         (ILClIn)'
      write(LER,*)'-cldm[]     -- crossline trace spacing      (CLClIn)'
      write(LER,*)'-tstart[]   -- start time (ms)        (first sample)'
      write(LER,*)'-tend[]     -- end time (ms)           (last sample)'
      write(LER,*)'-tskip[]    -- incremental time skip (ms)   (SmpInt)'
      write(LER,*)'-fdead      -- fill dead traces with means'     
      write(LER,*)' '
      write(LER,*)'Usage:'
      write(LER,*)'       stat3d -N[] -AMEAN[] '
      write(LER,*)'              -apheight[] -aplength[] -apwidth[]'
      write(LER,*)'              -ildm[] -cldm[] '
      write(LER,*)'              -tstart[] -tend[] -tskip[]'
      write(LER,*)'              -fdead'
      write(LER,*)'===================================================='
      
      return
      end

c-----------------------------------------------------------------------
c pick up command line arguments 

      subroutine cmdln(name, ntap, otap, 
     1                 tstart, tskip, tend,
     2                 apheight, aplength, apwidth,
     3                 ildm, cldm, fdead)

#include <f77/iounit.h>

      integer    argis
      character  name*(*), ntap*(*), otap*(*)
      real       tstart, tskip, tend
      real       apheight, aplength, apwidth
      real       ildm, cldm
      logical    fdead

c     Documented options

      call argstr('-N', ntap, ' ', ' ') 
      call argstr('-AMEAN', otap, ' ', ' ') 
      call argr4('-aplength', aplength, 0.0, 0.0)
      call argr4('-apwidth',  apwidth,  0.0, 0.0)
      call argr4('-apheight', apheight, 0.0, 0.0)

      call argr4('-ildm', ildm, 0.0, 0.0)
      call argr4('-cldm', cldm, 0.0, 0.0)

      fdead   = (argis('-fdead') .gt. 0)
      
      call argr4('-tstart', tstart, 0.0, 0.0)
      call argr4('-tend',   tend,   0.0, 0.0)
      call argr4('-tskip',  tskip,  0.0, 0.0)

c check for extraneous arguments and abort if found

      call xtrarg(name, ler, .FALSE., .FALSE.)
      call xtrarg(name, lerr, .FALSE., .TRUE.)

      return
      end
