C***********************************************************************
C                 copyright 2001, Amoco Production Company             *
C                             All Rights Reserved                      *
C                     an affiliate of BP America Inc.                  *
C***********************************************************************
       program swfilt3d
c*********************************************************************c
c swfilt3d reads in 3D (t,x,y) seismic data set, calculates a running 
c window (tau,p,q) transform, weights the result as a function of the 
c the semblance, and inverts back to (t,x,y).
c**********************************************************************c
c______________________________________________________________________
c          Kurt J. Marfurt (Amoco Production Research, Tulsa, OK, USA)
c          April 28, 1995.
c______________________________________________________________________
c
c______________________________________________________________________
c     include USP header files:
c______________________________________________________________________
#include <f77/iounit.h>
#include <f77/lhdrsz.h>
#include <f77/sisdef.h>
#include <f77/pid.h>
#include <save_defs.h>
c______________________________________________________________________
c     parameters needed for dynamic memory allocation routine 'galloc'.
c______________________________________________________________________
      parameter (maxs=0 000 002)
      dimension s(maxs)
      pointer   (pntrs,s)
c______________________________________________________________________
      parameter (maxomega=8192)
      parameter (maxangs=1000,maxannuli=12)         
      parameter (maxmx=51,maxmy=51,maxrec=6001)
      parameter (pi=3.1415926,twopi=2.*pi)
      parameter (maxp=200,maxq=200)
      parameter (minswgt=0,maxswgt=100)
      parameter (minawgt=-360,maxawgt=+360)
      parameter (iws=-2,iwe=+3)
c
      real        x(2*maxmx+1)
      real        y(2*maxmy+1)
      real        xwgt(2*maxmx+1)
      real        ywgt(2*maxmy+1)
      real        p(maxp),q(maxq)
      real        ptheta(maxangs),phi(maxangs)
      integer     iypointer(-maxmy:maxrec+maxmy)     
      integer     locate(2,(2*maxmy+1)*(2*maxmx+1))
      real        wgtxy((2*maxmy+1)*(2*maxmx+1))
      real        omega(0:maxomega)
      real        wgtpq(maxp*maxq)
      integer     jpqlive(maxp*maxq)
      integer     jplive(maxp*maxq)
      integer     jqlive(maxp*maxq)
      real        swgt(minswgt:maxswgt)
      real        awgt(minawgt:maxawgt)
      real        pwgt(-maxswgt:maxswgt)
      real        qwgt(-maxswgt:maxswgt)

      integer     hbegin
      integer     sheader(SZLNHD)
      real        cputim(20),waltim(20)
c
      character*200  file_in,file_out
      character*8    name
      logical        verbose,query,depth_section,eof,IKP,padded
      logical        svd,toeplitz,rho_filt,reject,true_dip
      logical        clockwise
      logical        readswath         
      logical        interpolate       
      logical        in_line_filtering
      integer        argis
      integer        stdin,stdout,stderr
      integer    startline,endline,starttrace,endtrace
      character*9    host,blank9
      integer        pipe(5)

      common /thdr/ ifmt_TrcNum,l_TrcNum,ln_TrcNum,
     1              ifmt_RecNum,l_RecNum,ln_RecNum,
     2              ifmt_SrcLoc,l_SrcLoc,ln_SrcLoc,
     3              ifmt_DphInd,l_DphInd,ln_DphInd,
     4              ifmt_LinInd,l_LinInd,ln_LinInd,
     5              ifmt_StaCor,l_StaCor,ln_StaCor
 

      data           name/'SWFILT3D'/
      data           host/'         '/
      data           blank9/'         '/
      data           pipe/3,4,5,6,7/
      data           luhls/88/
      data           undefined/-99999./
c_______________________________________________________________
c     check to see if we are running under IKP.
c_______________________________________________________________
      call ikpchk(host)
      if(host .ne. blank9) then
         IKP=.true.
      else
         IKP=.false.
      endif
c_______________________________________________________________
c     initialize timing arrays.
c_______________________________________________________________
      call vclr(cputim,1,20)
      call vclr(waltim,1,20)
      call timstr(vtot,wtot)
      stdin=LIN
      stdout=LOT
      stderr=LER
c_______________________________________________________________
c     read program parameters from command line card image file
c_______________________________________________________________
      query=(argis( '-?' ) .gt. 0 )
      if(query) then
         call help(ler)
         call exitfu(0)
      endif
c_______________________________________________________________
c     open printout files
c_______________________________________________________________
#include <f77/open.h>
c_______________________________________________________________
c     get i/o data set names
c_______________________________________________________________
      call argstr('-N',file_in, ' ', ' ')
      call argstr('-O',file_out,' ', ' ')
c_______________________________________________________________
c     open input and output files.
c_______________________________________________________________
      call getln(luin,file_in,'r',0)
      call getln(luout,file_out,'w',1)
c_______________________________________________________________
c     read line header of input
c_______________________________________________________________
      lbytes=0
      call rtape(luin,sheader,lbytes)
      if(lbytes .eq. 0) then
         write(lerr,*)'SWFILT3D: no header read from unit ',luin
         write(ler,*)'SWFILT3D: no header read from unit ',luin
         call exitfu(1666)
      endif
      lenhed=ITRWRD
      hbegin=1-lenhed
c_______________________________________________________________________
c     print historical line header.           
c_______________________________________________________________________
      call hlhprt (sheader,lbytes,name,len(name),lerr)
c_______________________________________________________________________
c     pull relevent values from line header.
c_______________________________________________________________________
      call saver(sheader,'NumSmp',nsamp_in,LINEHEADER)
      call saver(sheader,'SmpInt',nsi,LINEHEADER)
      call saver(sheader,'NumRec',nrec,LINEHEADER)
      call saver(sheader,'NumTrc',ntr,LINEHEADER)
      call saver(sheader,'TmMsFS',tstart_orig,LINEHEADER)
      call saver(sheader,'TmMsSl',jtstart_orig,LINEHEADER)
      call saver(sheader,'TmSlIn',jtslice_orig,LINEHEADER)
      call saver(sheader,'Dz1000',idz_1000,LINEHEADER)
      call saver(sheader,'ILClIn',dxh,LINEHEADER)
      call saver(sheader,'CLClIn',dyh,LINEHEADER)
      call saver(sheader,'AziIln',xazimh,LINEHEADER)
      call saver(sheader,'AziXln',yazimh,LINEHEADER)
      call saver(sheader,'Nx_Pad',nxpad,LINEHEADER)
      call saver(sheader, 'UnitSc', unitsc, LINEHEADER)
      if (unitsc .eq. 0.0) then
          write(LERR,*)'********************************************'
          write(LERR,*)'WARNING: sample unit scaler in LH = ',unitsc
          write(LERR,*)'         will set to .001 (millisec default)'
          write(LERR,*)'********************************************'
          unitsc = .001
          call savew(sheader, 'UnitSc', unitsc, LINEHEADER)
      endif
      dt = float(nsi) * unitsc
      dtmsec = 1000 * dt
      dzh=.001*idz_1000
      ny=nrec-1

      if(nxpad .gt. 0) then
         padded=.true.
      else
         padded=.false.
         nxpad=0
      endif

      if(padded) then
c______________________________________________________________________
c        input data have been padded each with nx_pad extra traces by
c        routine 'splitr' or equivalent.
c        it is the user's responsibility to make sure that the 
c        nxpad following -nxpad in routine 'splitr' is identical to 
c        the mx used in this application.
c______________________________________________________________________
         nx=ntr-1-(2*nxpad)           
      else
         nx=ntr-1
      endif
c_______________________________________________________________________
c     read other parameters from the command line.
c_______________________________________________________________________
      call cmdlin(s1,s2,s3,s4,smax,tstart_orig,
     1            a1,a2,a3,a4,f1,f2,f3,f4,fs,df,fnyquist,
     2            t1,t2,t3,t4,nsamp_in,dtmsec,istart,iend,nfft,
     3            dx,dy,dz,dsamp,dmsamp,dxh,dyh,dzh,prew,
     4            mx,my,xtaper,ytaper,reject,verbose,
     5            fwd_drt,rev_drt,depth_section,lerr,
     6            xazim,yazim,xazimh,yazimh,toeplitz,svd,rho_filt,
     7            semb_window,ntfine,eps2,sigma1,sigma2,
     8            nxinterp,nyinterp,dxi,dyi,interpolate,
     9            kximin,kximax,kyimin,kyimax,
     a            pmin,pmax,qmin,qmax,true_dip,nx,ny,
     b            p1,p2,p3,p4,q1,q2,q3,q4,ler,
     c            startline,endline,starttrace,endtrace)

      if(padded .and. mx .gt. nxpad) then
         write(lerr,*) 'error in swfilt3d!'
         write(lerr,*) 'mx = ',mx,' exceeds nxpad = ',nxpad,
     1                 ' read from line header!'
         write(lerr,*) 
     1      'examine command lines to swfilt3d and splitr!'
      endif

      pref=.5*(pmax+pmin)
      qref=.5*(qmax+qmin)
      prange=(pmax-pmin)
      qrange=(qmax-qmin)
      pnyquist=1./(.001*f4*dx)
      qnyquist=1./(.001*f4*dy)
      power=0.
      nt_orig=nsamp_in-1  
      if(depth_section) then
         dsamp=dz
      else
         dsamp=dtmsec
      endif
      if(mx .gt. 0) then
         in_line_filtering=.true.
      else
         in_line_filtering=.false.
      endif
      xmin=-mx*dx
      xmax=+mx*dx
      ymin=-my*dy
      ymax=+my*dy
      dfine=dsamp/ntfine
      ierror=0
      if(semb_window .lt. 0.) semb_window=5.*dsamp 
      dt=.001*dsamp
      tstart=t1
      tend=t4
      istart=nint((tstart-tstart_orig)/dsamp) 
      iend=nint((tend-tstart_orig)/dsamp) 
      istart=max(0,istart)
      iend=min(nsamp_in-1,iend)
      nt=(iend-istart+1)
      ltsemb=nint(semb_window/dsamp) 
      lentaper=nint(ttaper/dsamp) 
      if(iend .lt. istart) then
         write(lerr,*) 'error in SWFILT3D!'
         write(lerr,*) 'istart exceeds iend !'
         write(lerr,*) 'tstart,istart ',tstart,istart
         write(lerr,*) 'tend,  iend   ',tend,iend       
         call exit(6666)
      endif
      if(nrec .gt. maxrec) then
         write(lerr,*) 'error in SWFILT3D!'
         write(lerr,*) 'too many seismic records!'                 
         write(lerr,*) 'nrec = ',nrec,' exceeds maxrec = ',maxrec 
         write(lerr,*) 'call Marfurt at APR and have him do some'
     1                 //' memory management voodoo!'
         close(lerr)
         write(ler,*) 'error in SWFILT3D!'
         write(ler,*) 'too many seismic records!'                 
         write(ler,*) 'nrec = ',nrec,' exceeds maxrec = ',maxrec 
         write(ler,*) 'call Marfurt at APR and have him do some'
         call exit(7666)
      endif

      nbytes_out=(nt_orig+1+lenhed)*szsmpd
      ifl=nint(f1/df)
      ifl=max(ifl,1)
      ifh=nint(f4/df)
      ifh=min(ifh,nfft/2-1)
      fref=.5*(f3+f4)
c___________________________________________________________________
c     omega will be measured in radians/ms.
C_______________________________________________________________________
      domega=.001*2.*pi*df
      nf=ifh-ifl+1
c___________________________________________________________________
c     precompute the frequencies.
C_______________________________________________________________________
      do 12000 jomega=ifl,ifh
       omega(jomega)=(jomega)*domega
12000 continue
c
      call getgrid(p,q,x,y,locate,mx,my,nangs,maxangs,
     1             dx,dy,dp,dq,np,nq,jcenter,lerr,fref,
     2             npoint,rad,pmin,pmax,qmin,qmax,pref,qref,
     3             ptheta,phi,xazim,yazim,clockwise)
      nxw=2*mx+1
      nyw=2*my+1
      nxwfine=2*mx*nxinterp+1
      nywfine=2*my*nyinterp+1
      nline_out=(endline-startline)*nyinterp+1
      ntrace_out=(endtrace-starttrace)*nxinterp+1
      delaymax=max(pmax*xmax,qmax*ymax)
      delaymin=min(pmin*xmax,qmin*ymax)
      idelaymax=delaymax/dsamp+1
      idelaymin=delaymin/dsamp+1
      kstart=max(istart-ltsemb+idelaymin+iws,0)
      kend=min(iend+ltsemb+idelaymax+iwe,nt_orig)
      write(lerr,*)
      write(lerr,'(a,t40,i5,t50,a)') 'input seismic file name',
     1                         luin,file_in
      write(lerr,'(a,t40,i5,t50,a)') 'output multi-attribute file name',
     1                         luout,file_out
      write(lerr,'(a,t40,i5)') 'number of input samples',nsamp_in
      write(lerr,'(a,t40,i5)') 'number of samples in fft',nfft   
      write(lerr,'(a,t40,i5)') 'number of frequencies',nf      
      write(lerr,'(a,t40,i5)') 'number of output samples',nt
      write(lerr,'(a,t40,i5)') 'number of traces',ntr  
      write(lerr,'(a,t40,i5)') 'number of records',nrec 
      write(lerr,'(a,t40,i5)') 'interpolation to finer time sample',
     1                          ntfine
      write(lerr,'(a,t40,i5)') 'number of search angles',nangs
      write(lerr,'(a,t40,f12.6)') 'original start time (msec)',
     1                             tstart_orig
      write(lerr,'(a,t40,f12.6)') 'output start time (msec)',
     1                             tstart
      write(lerr,'(a,t40,f12.6)') 'output end time (msec)',
     1                             tend
      write(lerr,'(a,t40,f12.6)') 'minimum delay time (msec)',delaymin
      write(lerr,'(a,t40,f12.6)') 'maximum delay time (msec)',delaymax
      write(lerr,'(a,t40,i5)') 'output start time (samples)',istart
      write(lerr,'(a,t40,i5)') 'output end time (samples)',iend  
      write(lerr,'(a,t40,i5)') 'minimum delay time (samples)',
     1                           idelaymin
      write(lerr,'(a,t40,i5)') 'maximum delay time (samples)',
     1                           idelaymax
      write(lerr,'(a,t40,i5)') 'integration window (samples)',ltsemb
      write(lerr,'(a,t40,i5)') 'input data begin time (samples)',kstart
      write(lerr,'(a,t40,i5)') 'input data end time (samples)',kend
      write(lerr,'(a,t40,f12.6)') 'input data sample interval (msec)', 
     1                         dsamp 
      write(lerr,'(a,t40,f12.6)') 'fine sample interval (msec)', 
     1                         dfine           
      write(lerr,'(a,t40,f12.6)') 'sample interval (sec)', dt
      write(lerr,'(a,t40,f12.6)') 
     1       'input inline trace spacing dx (m)',dx   
      write(lerr,'(a,t40,f12.6)') 
     1       'input crossline trace spacing dy (m)',dy   
      write(lerr,'(a,t40,f12.6)') 
     1       'output interpolated inline trace spacing dxi (m)',dxi
      write(lerr,'(a,t40,f12.6)') 
     1       'output interpolated crossline trace spacing dyi (m)',dyi   
      if(rad .eq. 0.) then
         write(lerr,'(a,t40,f12.6)') 'analysis window mx*dx (m) ',mx*dx
         write(lerr,'(a,t40,f12.6)') 'analysis window my*dy (m) ',my*dy
      else
         write(lerr,'(a,t40,f12.6)') 'analysis window radius (m) ',rad
      endif
      if(depth_section) then
         write(lerr,'(a,t40,f12.6)') 'depth interval dz (m)',dz   
      else
         write(lerr,'(a,t40,f12.6)') 'time  interval dt (sec)',dt   
      endif
      if(true_dip) then
         write(lerr,'(a,t50,f12.6)') 'smax',smax
         write(lerr,'(a,t50,f12.6)') 's1',s1
         write(lerr,'(a,t50,f12.6)') 's2',s2
         write(lerr,'(a,t50,f12.6)') 's3',s3
         write(lerr,'(a,t50,f12.6)') 's4',s4
         write(lerr,'(a,t50,f12.6)') 'a1',a1
         write(lerr,'(a,t50,f12.6)') 'a2',a2
         write(lerr,'(a,t50,f12.6)') 'a3',a3
         write(lerr,'(a,t50,f12.6)') 'a4',a4
      else
         write(lerr,'(a,t50,f12.6)') 'pmin',pmin
         write(lerr,'(a,t50,f12.6)') 'pmax',pmax
         write(lerr,'(a,t50,f12.6)') 'p1',p1
         write(lerr,'(a,t50,f12.6)') 'p2',p2
         write(lerr,'(a,t50,f12.6)') 'p3',p3
         write(lerr,'(a,t50,f12.6)') 'p4',p4
         write(lerr,'(a,t50,f12.6)') 'qmin',qmin
         write(lerr,'(a,t50,f12.6)') 'qmax',qmax
         write(lerr,'(a,t50,f12.6)') 'q1',q1
         write(lerr,'(a,t50,f12.6)') 'q2',q2
         write(lerr,'(a,t50,f12.6)') 'q3',q3
         write(lerr,'(a,t50,f12.6)') 'q4',q4
      endif
      write(lerr,'(a,t50,f12.6)') 't1, start time (msec)',t1
      write(lerr,'(a,t50,f12.6)') 't2 (msec)',t2
      write(lerr,'(a,t50,f12.6)') 't3 (msec)',t3
      write(lerr,'(a,t50,f12.6)') 't4, end time (msec)',t4
      write(lerr,'(a,t50,i5)') 'start time (samples)',istart
      write(lerr,'(a,t50,i5)') 'end time (samples)',iend
      write(lerr,'(a,t50,i5)') 'nt',nt,'nt_orig',nt_orig
      write(lerr,'(a,t50,i5)') 'fft length (samples)',nfft
      write(lerr,'(a,t50,f12.6)')
     1             'sample interval dmsamp (msec,m,ft)',dmsamp
      write(lerr,'(a,t50,f12.6)')
     1             'sample interval dsamp (sec,km,kft)',dsamp


      write(lerr,'(a,t40,f12.6)') 'azimuth of in line (x) axis '
     1             //'(degrees) ',xazim
      write(lerr,'(a,t40,f12.6)') 'azimuth of cross line (y) axis '
     1             //'(degrees) ',yazim
      
      write(lerr,'(a,t40,f12.6)') 'df',df             
      write(lerr,'(a,t40,f12.6)') 'f1',f1,'f2',f2,'f3',f3,'f4',f4,
     1                            'fref',fref,'fnyquist',fnyquist         
      write(lerr,'(a,t40,f12.6)') 'pnyquist (at f4) ',pnyquist,
     1            ' q nyquist (at f4) ',qnyquist 
      write(lerr,'(a,t40,f12.6)') 'rho filter power ',power
      write(lerr,'(a,t40,f12.6)') 'dsamp',dsamp
      write(lerr,'(a,t40,i5)') 'ifl',ifl,'ifh',ifh,'nf',nf,'nfft',nfft       
      write(lerr,'(a,t40,i5)') 'ntr',ntr,'nx',nx,'ny',ny  
      write(lerr,'(a,t40,i5)') 'nxpad',nxpad,'mx',mx,'my',my,
     1                         'npoint',npoint
      write(lerr,'(a,t50,i5)') 'startline',startline,
     1         'endline',endline,'starttrace',starttrace,
     2         'endtrace',endtrace
      write(lerr,'(a,t40,l5)') 'spatial interpolation?',interpolate   
      write(lerr,'(a,t40,i5)') 'nxinterp',nxinterp,
     1                         'nyinterp',nyinterp,
     2                         'kximin',kximin,'kximax',kximax,
     3                         'kyimin',kyimin,'kyimax',kyimax
      write(lerr,'(a,t40,i5)') 'ntrace_out',ntrace_out,
     1                         'nline_out',nline_out
      write(lerr,'(a,t40,i5)') 'nbytes_out',nbytes_out           
      write(lerr,'(a,t40,f12.6)') 'sigma1',sigma1,'sigma2',sigma2         
c
      write(lerr,'(a,t50,f12.6)') 
     1    'minimum inline dip  (ms/m; ms/ft ;m/m; ft/ft) ',pmin
      write(lerr,'(a,t50,f12.6)') 
     1     'maximum inline dip  (ms/m; ms/ft ;m/m; ft/ft) ',pmax
      write(lerr,'(a,t50,f12.6)') 
     1    'minimum crossline dip  (ms/m; ms/ft ;m/m; ft/ft) ',qmin
      write(lerr,'(a,t50,f12.6)') 
     1    'maximum crossline dip  (ms/m; ms/ft ;m/m; ft/ft) ',qmax
      write(lerr,'(a,t40,l5)') 'clockwise acquisition grid?',clockwise
      write(lerr,'(a,t40,l5)') 'reject data between s1-s4,a1-a4?',reject
      write(lerr,'(a,t40,l5)') 'input depth section?',depth_section
      write(lerr,'(a,t40,l5)') 'running under IKP?',IKP
      write(lerr,'(a,t40,l5)') 'verbose output?',verbose     
      write(lerr,*)' '
      write(lerr,'(a,t50,l5)') 'invert using SVD?',svd
      write(lerr,'(a,t50,l5)') 'invert using Toeplitz structure?',
     1                           toeplitz
      write(lerr,'(a,t50,l5)') 'verbose output?',verbose
      write(lerr,*)' '
      write(lerr,*) 'center point at node = ',jcenter
      da=1.
      if(true_dip) then
c_____________________________________________________________
c        calculate true dip/azimuth tapers.
c_____________________________________________________________
         ds=2.*smax/(maxswgt-minswgt)
         nswgt=maxswgt-minswgt+1
         call gttapr(swgt,s1,s2,s3,s4,ds,minswgt,maxswgt)
         call aztapr(awgt,a1,a2,a3,a4,minawgt,maxawgt)
         if(verbose) then
            write(lerr,*) 's1,s2,s3,s4,smax ',s1,s2,s3,s4,smax
            write(lerr,'(a5,2a12)') 'k','s(k)','swgt(k)'
            write(lerr,'(i5,2f12.6)')
     1                    (k,k*ds,swgt(k),k=minswgt,maxswgt)
            write(lerr,'(a5,a12)') 'k','awgt(k)'
            write(lerr,'(i5,f12.6)') (k,awgt(k),k=-180,+180)
         endif
      else
c_____________________________________________________________
c        calculate apparent dip tapers.
c_____________________________________________________________
         smax=max(abs(pmin),abs(pmax),abs(qmin),abs(qmax))
         ds=smax/maxswgt
         nswgt=2*maxswgt+1
         call gttapr(pwgt,p1,p2,p3,p4,ds,-maxswgt,maxswgt)
         call gttapr(qwgt,q1,q2,q3,q4,ds,-maxswgt,maxswgt)
         if(verbose) then
            write(lerr,*) 'p1,p2,p3,p4 ',p1,p2,p3,p4
            write(lerr,*) 'q1,q2,q3,q4 ',q1,q2,q3,q4
            write(lerr,'(a5,4a12)') 'k','p(k)','pwgt(k)','q(k)',
     1                     'qwgt(k)'
            write(lerr,'(i5,4f12.6)')
     1         (k,k*ds,pwgt(k),k*ds,qwgt(k),k=-maxswgt,maxswgt)
         endif
      endif
c_____________________________________________________________
c     calculate spatial tapers.
c_____________________________________________________________
      x1=xmin-dx
      x2=xmin+xtaper
      x3=xmax-xtaper
      x4=xmax+dx
      call gttapr(xwgt,x1,x2,x3,x4,dx,-mx,+mx)
      y1=ymin-dy
      y2=ymin+ytaper
      y3=ymax-ytaper
      y4=ymax+dy
      call gttapr(ywgt,y1,y2,y3,y4,dy,-my,+my)
      call getwxy(wgtxy,locate,xwgt,ywgt,mx,my,lerr)
c_____________________________________________________________________
c     calculate mutes as a function of angle.
c_____________________________________________________________________
      call getmute(wgtpq,jpqlive,p,q,swgt,awgt,ds,da,np,nq,
     1             minswgt,maxswgt,minawgt,maxawgt,lerr,
     2             xazim,yazim,clockwise,verbose,npqlive,
     3             pwgt,qwgt,true_dip,jplive,jqlive)

      write(lerr,*) 'npqlive = ',npqlive
      write(lerr,*) 'np*nq   = ',np*nq
      write(ler,*) 'npqlive = ',npqlive
      write(ler,*) 'np*nq   = ',np*nq
c______________________________________________________________________
c     calculate memory requirements
c______________________________________________________________________
      lenrr=2*npqlive*nf*nxinterp*nyinterp
      lensvd=max(np,nxw,nq,nyw)+1
      maxxypq=max(np,nq,nxw,nyw)
      maxuv=max(np*nxw,nq*nyw)
      write(lerr,'(///,80a1)') ('_',i=1,80)
      write(lerr,'(A)')'storage map of major arrays'
      write(lerr,'(80a1)') ('_',i=1,80)
C
      l_free=1
c____________________________________________________________________
      write(lerr,'(a20,3a10)')'variable name','begin','end','length'
      call mapmem('uin',l_uin,l_free,(nx+2*mx+1)*(5+nt_orig+1),lerr)
      call mapmem('utxfine',l_utxfine,l_free,
     1             (nx+2*mx+1)*(2*my+1)*(nt_orig+1)*ntfine,lerr)
      call mapmem('utxfine2',l_utxfine2,l_free,
     1             (2*mx+1)*(2*my+1)*(nt_orig+1)*ntfine,lerr)
      call mapmem('live',l_live,l_free,(2*my+1)*(nx+1+2*mx),lerr)
      call mapmem('tracebuf',l_tracebuf,l_free,(nt_orig+1+lenhed),lerr)
      call mapmem('trheader',l_trheader,l_free,
     1                             lenhed*(nx+1)*(2*my+1),lerr)
      call mapmem('uout',l_uout,l_free,
     1            nt*(nx+1)*nxinterp*nyinterp,lerr)
      call mapmem('usemb',l_usemb,l_free,(nt_orig+1)*nangs,lerr)
      call mapmem('wsemb',l_wsemb,l_free,(nt_orig+1)*nangs,lerr)
      call mapmem('utaupq',l_utaupq,l_free,(5+nt_orig+1)*nangs,lerr)
      call mapmem('utaupqfine',l_utaupqfine,l_free,
     1                     (nt_orig+1)*nangs*ntfine,lerr)
      call mapmem('unum',l_unum,l_free,(nt_orig+1)*nangs,lerr)
      call mapmem('unumsum',l_unumsum,l_free,(nt_orig+1)*nangs,lerr)
      call mapmem('udenom',l_udenom,l_free,(nt_orig+1)*nangs,lerr)
      call mapmem('udenomsum',l_udenomsum,l_free,(nt_orig+1)*nangs,lerr)
      call mapmem('jpdelay',l_jpdelay,l_free,2*nangs*npoint,lerr)
      call mapmem('jpadvance',l_jpadvance,l_free,
     1                 2*nangs*nxinterp*nyinterp,lerr)
      call mapmem('w',l_w,l_free,6*(ntfine+1),lerr)
      call mapmem('uomegaxy',l_uomegaxy,l_free,nfft,lerr)
      call mapmem('uomegapq',l_uomegapq,l_free,nfft*npqlive,lerr)
c___________________________________________________________________
C     arrays needed to calculate and apply the rho filter.
C_______________________________________________________________________
      call mapmem('rho',l_rho,l_free,nfft/2,lerr)
      call mapmem('fwgt',l_fwgt,l_free,nfft/2,lerr)
      call mapmem('rbuf',l_rbuf,l_free,nfft,lerr)
      call mapmem('cbuf',l_cbuf,l_free,nfft,lerr)
c___________________________________________________________________
C     transforms needed for the (omega,p,q) reverse transform matrix.
C_______________________________________________________________________
      call mapmem('rx1d',l_rx1d,l_free,2*nxw*np,lerr)
      call mapmem('rx1dc',l_rx1dc,l_free,2*nxw*np,lerr)
      call mapmem('rx1dinv',l_rx1dinv,l_free,2*np*np,lerr)
      call mapmem('rx2dinv',l_rx2dinv,l_free,2*np*nq*nq*np,lerr)
c
      call mapmem('rx1dfine',l_rx1dfine,l_free,2*nxwfine*np,lerr)
      call mapmem('rx2dfine',l_rx2dfine,l_free,2*nxwfine*nywfine*nq*np,
     1             lerr)
c
      call mapmem('ry1d',l_ry1d,l_free,2*nyw*nq,lerr)
      call mapmem('ry1dc',l_ry1dc,l_free,2*nyw*nq,lerr)
      call mapmem('ry1dinv',l_ry1dinv,l_free,2*nq*nq,lerr)
      call mapmem('ry2dinv',l_ry2dinv,l_free,2*nq*np*nq*np,lerr)
c 
      call mapmem('ry1dfine',l_ry1dfine,l_free,2*nywfine*nq,lerr)
      call mapmem('ry2dfine',l_ry2dfine,l_free,2*nywfine*np*nq*np,lerr)
c
      call mapmem('r2',l_r2,l_free,2*maxxypq*maxxypq,lerr)
      call mapmem('z',l_z,l_free,2*maxxypq,lerr)
      call mapmem('iperm',l_iperm,l_free,maxxypq,lerr)
      call mapmem('rr',l_rr,l_free,lenrr,lerr)
      call mapmem('r2rinv',l_r2rinv,l_free,2*npqlive*npqlive,lerr)
      call mapmem('rt',l_rt,l_free,2*npqlive*nxwfine*nywfine,lerr)
      call mapmem('rv',l_rv,l_free,2*maxuv,lerr)
      call mapmem('map',l_map,l_free,nxwfine*nywfine,lerr)
c___________________________________________________________________
C     allocate dynamic memory.
C_______________________________________________________________________
      lens=l_free-1
      write(ler,'(//,a)') 'allocate dynamic memory for SWFILT3D: '
      write(ler,*) 1.e-6*lens,' Mwords'
      write(ler,*) 1.e-6*lens*szsmpd,' Mbytes'
      write(lerr,'(a20,10x,i10)') 'total vector length',lens
      call galloc(pntrs,lens*szsmpd,ierrcd,0)
      if(ierrcd .ne. 0 ) then
         write(lerr,*)'galloc memory allocation error from main'
         write(lerr,*)'ierrcd = ',ierrcd
         write(lerr,*)
         write(lerr,*)'probable cause: too much memory requested!'
         write(lerr,*)'go check your seismic headers and command'//
     1                ' line arguments!'
         write(lerr,*)
         write(lerr,*)'program SWFILT3D aborted'
c
         write(ler,*)'galloc memory allocation error from main'
         write(ler,*)'ierrcd = ',ierrcd
         write(ler,*)
         write(ler,*)'probable cause: too much memory requested!'
         write(ler,*)'go check your seismic headers and command'//
     1                ' line arguments!'
         write(ler,*)
         write(ler,*)'program SWFILT3D aborted'
         call exit(101)
      endif
C_______________________________________________________________________
c     modify line header to reflect actual number of traces output
c     determine if the attributes will go to separate files or
c     one large files with attribute lines written back to back.
C_______________________________________________________________________
      call savhlh(sheader,lbytes,lbyout)
      jtstart=nint(tstart)
      jtend=nint(tend)
      jtslice=nint(dsamp) 
      call savew(sheader,'TmMsFS',tstart,LINEHEADER)
      call savew(sheader,'TmMsSl',jtstart,LINEHEADER)
      call savew(sheader,'TmSlIn',jtslice,LINEHEADER)
      call savew(sheader,'NumSmp',nt_orig+1,LINEHEADER)
      write(0,*) 'nt_orig, nt_orig+1 ',nt_orig, nt_orig+1 
      if(padded .and. in_line_filtering) then
c______________________________________________________________________
c        We are going to reduce the output by Nx_Pad traces.  We
c        now need to set this header value to 0.
c______________________________________________________________________
         newpad = 0
         call savew(sheader,'Nx_Pad',newpad,LINEHEADER)
      endif
      call savew(sheader,'NumTrc',ntrace_out,LINEHEADER)
      call savew(sheader,'NumRec',nline_out,LINEHEADER)
c
      call wrtape(luout,sheader,lbyout)
      call savelu('TrcNum',ifmt_TrcNum,l_TrcNum,ln_TrcNum,TRACEHEADER)
      call savelu('RecNum',ifmt_RecNum,l_RecNum,ln_RecNum,TRACEHEADER)
      call savelu('StaCor',ifmt_StaCor,l_StaCor,ln_StaCor,TRACEHEADER)
      call savelu('DphInd',ifmt_DphInd,l_DphInd,ln_DphInd,TRACEHEADER)
      call savelu('LinInd',ifmt_LinInd,l_LinInd,ln_LinInd,TRACEHEADER)
      call savelu('SrcLoc',ifmt_SrcLoc,l_SrcLoc,ln_SrcLoc,TRACEHEADER)
C_______________________________________________________________________
C     calculate 6 point lagrange interpolation weights for ntfine
c     possible interpolation divisions.
C_______________________________________________________________________
      call getw6(s(l_w),ntfine)
c_____________________________________________________________   
c     precompute the (tau,p,q) delays and associated ptheta,phi values.
c_____________________________________________________________   
      call getdelay(s(l_jpdelay),s(l_jpadvance),locate,
     1              p,q,npoint,dx,dy,dxi,dyi,
     2              kximin,kximax,kyimin,kyimax,
     3              jpqlive,npqlive,nangs,dfine,ntfine,
     4              np,nq,jplive,jqlive,verbose,lerr)
c_____________________________________________________________
c     calculate frequency tapers.
c_____________________________________________________________
      call gttapr(s(l_fwgt),f1,f2,f3,f4,df,ifl,ifh) 
c_____________________________________________________________
c     precompute the (omega,p,q) reverse Radon transform [rr]
c_____________________________________________________________
      call timstr(v1,w1)
      write(0,*) 'before getrr. ifl,ifh = ',ifl,ifh
      call getrr(s(l_rx1d),s(l_rx1dinv),s(l_rx2dinv),
     1          s(l_ry1d),s(l_ry1dinv),s(l_ry2dinv),
     2          s(l_rx1dc),s(l_ry1dc),s(l_r2),s(l_z),s(l_iperm),
     3  s(l_rx1dfine),s(l_rx2dfine),s(l_ry1dfine),s(l_ry2dfine),
     4          s(l_r2rinv),s(l_rt),s(l_rr),toeplitz,s(l_rv),
     5          x,y,p,q,nxw,nyw,np,nq,dx,dy,dp,dq,
     6          wgtpq,jpqlive,npqlive,omega(ifl),jcenter,
     7          ifl,ifh,s(l_rho),s(l_fwgt),
     a          prew,lerr,ler,
     9          kximin,kximax,xmin,dxi,mx,nxinterp,nxwfine,
     a          kyimin,kyimax,ymin,dyi,my,nyinterp,nywfine,
     b          s(l_map))

      call timend(cputim(19),v1,v2,waltim(19),w1,w2)
c
      scale=0.
      jrec=0
      jybegin=0
      jyend=ny
c_____________________________________________________________   
c     fill in the data swath uin needed for processing.
c_____________________________________________________________   
      jrec=0
      time_left=0.
      readswath=.true.
      do 80000 jy=0,endline           
       if(readswath) then
c_____________________________________________________________   
c         read in an entire swath of data (-my:+my) for processing.
c_____________________________________________________________   
          do 70000 iy=-my,+my
           iypointer(jy+iy)=iy
           if(iy .lt. 0 .or. iy .gt. ny) then
c_____________________________________________________________   
c             no records exist. set all traces to be dead.
c_____________________________________________________________   
              call deadrec(s(l_live),iypointer(jy+iy),my,mx,nx)
           else
c_____________________________________________________________   
c             read in the next record.    
c_____________________________________________________________   
              jrec=jrec+1
              call rdgather(s(l_tracebuf),s(l_tracebuf),
     1                     s(l_trheader),s(l_uin),s(l_utxfine),
     2                     s(l_w),ntfine,s(l_live),luin,nx,
     3                     l_StaCor,eof,iypointer(jy+iy),my,mx,
     4                     lntrhd,lenhed,nt_orig,lerr,l_DphInd,
     5                     kstart,kend,scale,cputim,waltim,nxpad,
     7                     ifmt_DphInd,ln_DphInd,ifmt_StaCor,ln_StaCor)
              if(eof) go to 99000
           endif
70000     continue
          readswath=.false.
      else
c_____________________________________________________________
c         roll the indices to the swath of data in y.
c_____________________________________________________________
          iypointer(jy+my)=iypointer(jy-my-1)
          if(jy+my .gt. ny) then
c_____________________________________________________________
c            no more records exist. set all traces to be dead.
c_____________________________________________________________
             call deadrec(s(l_live),iypointer(jy+my),my,mx,nx)
          else
c_____________________________________________________________
c            read in the next record.
c_____________________________________________________________
             jrec=jrec+1
             call rdgather(s(l_tracebuf),s(l_tracebuf),
     1                     s(l_trheader),s(l_uin),s(l_utxfine),
     2                     s(l_w),ntfine,s(l_live),luin,nx,
     3                     l_StaCor,eof,iypointer(jy+my),my,mx,
     4                     lntrhd,lenhed,nt_orig,lerr,l_DphInd,
     5                     kstart,kend,scale,cputim,waltim,nxpad,
     8                     ifmt_DphInd,ln_DphInd,ifmt_StaCor,ln_StaCor)
             if(eof) go to 99000
          endif
      endif
      if(jy .lt. startline) go to 80000
       if(jy .lt. my) then
          write(ler,*) 'process line ',jy,' of ',ny
       elseif(jy .eq. my) then
          call timstr(v0,w0)
          write(ler,*) 'process line ',jy,' of ',ny
       else
          write(ler,'(a,i5,a,i5,a,i5,a,i5,a,i5,a)')
     1    'process line ',jy,' of ',ny,
     2    ' time to completion: ',nhour,' hr',
     3      nmin,' min',nsec,' sec'
       endif
c_____________________________________________________________   
c      data swath is now full. roll through in the y direction,
c      each time adding the next seismic line and dropping the
c      oldest.
c_____________________________________________________________   
      call process(s(l_utxfine),s(l_utxfine2),s(l_live),ntfine,
     1             s(l_w),s(l_wsemb),s(l_uout),                 
     2             s(l_usemb),s(l_utaupq),s(l_utaupqfine),
     3             s(l_unum),s(l_udenom),
     4             s(l_unumsum),s(l_udenomsum),
     5             s(l_jpdelay),s(l_jpadvance),npqlive,nangs,
     6             lenhed,cputim,waltim,
     7             iypointer(-my),jy,ny,locate,nt_orig,nx,npoint,
     8             ltsemb,mx,my,nt,
     9             lerr,jrec,kstart,kend,istart,iend,
     a             kximin,kximax,kyimin,kyimax,interpolate,
     b             eps2,sigma1,sigma2,wgtpq,
     c             s(l_rho),s(l_cbuf),s(l_rbuf),nfft,ifl,ifh,
     d             s(l_uomegaxy),s(l_uomegapq),s(l_rr),rho_filt,
     e             starttrace,endtrace,startline,endline)

c_____________________________________________________________   
c      write out the results.      
c_____________________________________________________________   
       call timstr(v1,w1)
       call wrgather(s(l_uout),s(l_utxfine),s(l_trheader),
     1               s(l_tracebuf),s(l_tracebuf),
     2               luout,nx,iypointer(jy),jy,ny,
     3               my,nt_orig,kximin,kximax,kyimin,kyimax,
     4               istart,iend,lenhed,hbegin,nbytes_out,
     5               scale,l_DphInd,ifmt_DphInd,ln_DphInd,
     6               mx,ntfine,reject)

       call timend(cputim(10),v1,v2,waltim(10),w1,w2)
       if(jy .ge .my) then
c_____________________________________________________________   
c         calculate average wall time for completely
c         filled lines.
c_____________________________________________________________   
          call timstr(vcurr,wcurr)                 
          time_per_line=(wcurr-w0)/(jy-my+1)
c_____________________________________________________________   
c         predict wall time to completion.
c_____________________________________________________________   
          time_left=(ny-jy)*time_per_line
          nhour=time_left/3600.
          nmin=(time_left-3600.*nhour)/60.
          nsec=time_left-3600.*nhour-60.*nmin
       endif
80000 continue
c_____________________________________________________________   
c     write out timing statistics.
c_____________________________________________________________   
      call timend(cputim(20),vtot,v2,waltim(20),wtot,w2)
      write(lerr,'(A30,2A15,/)') 'routine','cpu time','wall time'
      write(lerr,'(A30,2f15.3)')
     1         'precompute inverse DRT',cputim(19),waltim(19),
     1         'read input',cputim(1),waltim(1),
     2         'scale data',cputim(2),waltim(2),
     3         'time interpolate u(t,x,y)',cputim(3),waltim(3),
     4         'square',cputim(9),waltim(9),
     3         'U(tau,p,q) and semblance',cputim(4),waltim(4),
     3         'running window integration',cputim(5),waltim(5),
     3         'semblance weights',cputim(15),waltim(15),
     3         'time interpolate U(tau,p,q) ',cputim(13),waltim(13),
     3         'reverse taupq transform ',cputim(14),waltim(14),
     3         'reverse DRT transform ',cputim(17),waltim(17),
     6         'write output',cputim(10),waltim(10),
     7         'total',cputim(20),waltim(20)
c_____________________________________________________________   
c     close data files
c_____________________________________________________________   
      call lbclos(luout)
      call lbclos(luin)

      write(ler,*)'normal completetion. routine SWFILT3D'           
      write(lerr,*)'normal completetion. routine SWFILT3D'           
      close(lerr)
      call exitfu(0)
99000 write(ler,*) 'end of file encountered when reading line jy = ',
     1              jy,' of ',ny,' from unit luin ',luin
      call exitfu(666)
c
      end
      subroutine help(ler)

      write(ler,*)
     1'**************************************************************'
      write(ler,*)'3D seismic attributes using a semblance algorithm'
      write(ler,*)
     1'**************************************************************'
      write(ler,*)'execute swfilt3d by typing swfilt3d and list of'
     1                   //' program parameters.'
      write(ler,*)
     1'note that each parameter is proceeded by -a where "a" is '
      write(ler,*)
     1'a character(s) correlation to some parameter.'
      write(ler,*)
     1'users enter the following parameters, or use the default values'
      write(ler,*)' '
        write(ler,*)
     1' -N [file_in]  (stdin)      : input seismic file name'          
        write(ler,*)
     1' -O [file_out]  (stdout)    : output filtered file name'        
        write(ler,*)
     1' -tstart[tstart] (first sample) : starting time in msec'         
        write(ler,*)
     1' -tend[tend]   (last  sample) : ending   time in msec'         
        write(ler,*)
     1' -w [semb_window] (2 samples): running time integration'
     2                               //' window in msec'
        write(ler,*)
     1' -ildm(dx)   (ILClIn)        : in line cell dimension '
     2                                 //'in m (ft)'        
        write(ler,*)
     1' -cldm       (CLClIn)        : cross line cell dimension ' 
     2                                 //'in m (ft)'        
        write(ler,*)
     1' -dz         (0.)            : depth sample unit (dz > 0 '
     2                     //'indicates depth section)'
        write(ler,*)
     1' -ilhw       (1)             : in line half aperture '
     2              //'width in m.'
        write(ler,*)
     1' -clhw       (1)             : cross line half aperture '
     2              //'width in m.'
        write(ler,*)
     1' -ilazim     (AziIln)        : in line azimuth in degrees'
     2                //' (azimuth of increasing trace numbers)'
        write(ler,*)
     1' -clazim     (AziCln)         : cross line azimuth in degrees'           
     2                //' (azimuth of increasing line numbers) '    
        write(ler,*)
     1' -ilinterp   (1)              : in line interpolation rate'              
        write(ler,*)
     1' -clinterp   (1)              : cross line interpolation rate' 
        write(ler,*)
     1' -startline (1)       : first line to be processed'
        write(ler,*)
     1' -endline   (last)    : last line to be processed'
        write(ler,*)
     1' -starttrace (1)      : first trace to be processed'
        write(ler,*)
     1' -endtrace   (last)   : last trace to be processed'
        write(ler,*) 
     1'             (30)            : reference frequency in '
     2                //' cycles/km (cycles/kft) for depth data'
        write(ler,*)
     1' -sigma1     (.05)           : mute events with semblance'
     2                                 //' less than sigma1 '      
        write(ler,*)
     1' -sigma2     (.10)           : pass events with semblance'
     2                                 //' greater than sigma1 '      
        write(ler,*)
     1' -f1[f1]   (0.)      :  begin roll in frequency taper '
     2                   //'weights (Hz, cycles/km, cycles/kft) '
        write(ler,*)
     1' -f2[f2]   (f1)      :  end roll in frequency taper '
     2                   //'weights (Hz, cycles/km, cycles/kft) '
        write(ler,*)
     1' -f3[f3]   (f4)      :  begin roll off frequency taper '
     2                   //'weights (Hz, cycles/km, cycles/kft) '
        write(ler,*)
     1' -f4[f4]   (Nyquist) :  end roll off frequency taper'
     2                   //'weights (Hz, cycles/km, cycles/kft) '
        write(ler,*)
     1' -t1[t1]   (first samp): begin roll in temporal (depth) '
     2                   //'taper weights (msec, m, ft)'
        write(ler,*)
     1' -t2[t2]   (t1)      :  end roll in temporal (depth) '
     2                   //'taper weights (msec, m, ft)'
        write(ler,*)
     1' -t3[t3]   (t4)      :  begin roll off temporal (depth) '
     2                   //'taper weights (msec, m, ft)'
        write(ler,*)
     1' -t4[t4]   (last samp) :  end roll temporal (depth) '
     2                   //'taper weights (msec, m, ft)'
        write(ler,*)
        write(ler,*) 'Parameterization using apparent dips'
        write(ler,*)
        write(ler,*)
     1' -pmin       (0.0)            : minimum inline (x) dip'
     2       //' (msec/m, msec/ft, m/m, ft/ft)'
        write(ler,*)
     1' -pmax       (0.0)            : maximum inline (x) dip'
     2       //' (msec/m, msec/ft, m/m, ft/ft)'
        write(ler,*)
     1' -qmin       (0.0)            : minimum crossline (y) dip'
     2       //' (msec/m, msec/ft, m/m, ft/ft)'
        write(ler,*)
     1' -qmax       (0.0)            : maximum crossline (y) dip'
     2       //' (msec/m, msec/ft, m/m, ft/ft)'
        write(ler,*)
     1' -p1         (pmin)           : begin roll in of inline'
     2       //'filter weights (msec/m, msec/ft, m/m, ft/ft)'
        write(ler,*)
     1' -p2         (p1)             : end roll in of inline'
     2       //'filter weights (msec/m, msec/ft, m/m, ft/ft)'
        write(ler,*)
     1' -p3         (p4)             : begin roll out of inline'
     2       //'filter weights (msec/m, msec/ft, m/m, ft/ft)'
        write(ler,*)
     1' -p4         (qmax)           : end roll out of crossline'
     2       //'filter weights (msec/m, msec/ft, m/m, ft/ft)'
        write(ler,*)
     1' -q1         (qmin)           : begin roll in of crossline'
     2       //'filter weights (msec/m, msec/ft, m/m, ft/ft)'
        write(ler,*)
     1' -q2         (q1)             : end roll in of crossline'
     2       //'filter weights (msec/m, msec/ft, m/m, ft/ft)'
        write(ler,*)
     1' -q3         (q4)             : begin roll out of crossline'
     2       //'filter weights (msec/m, msec/ft, m/m, ft/ft)'
        write(ler,*)
     1' -q4         (qmax)           : end roll out of crossline'
     2       //'filter weights (msec/m, msec/ft, m/m, ft/ft)'
        write(ler,*)
        write(ler,*) 'Parameterization using true dip/azimuth'
        write(ler,*)
        write(ler,*)
     1' -smax       (0.0)            : maximum true dip'
     2       //' (msec/m, msec/ft, m/m, ft/ft)'
        write(ler,*)
     1' -s1         (0.0)            : begin roll in moveout'
     2       //'filter weights (msec/m, msec/ft, m/m, ft/ft)'
        write(ler,*)
     1' -s2         (s1)             : end roll in moveout  '
     2       //'filter weights (msec/m, msec/ft, m/m, ft/ft)'
        write(ler,*)
     1' -s3         (s4)             : begin roll out moveout  '
     2       //'filter weights (msec/m, msec/ft, m/m, ft/ft)'
        write(ler,*)
     1' -s4         (smax)           : end roll out moveout  '
     2       //'filter weights (msec/m, msec/ft, m/m, ft/ft)'
        write(ler,*)
     1' -a1         (+180.)          : begin roll in azimuth  '
     2                    //'filter weights (degrees)'
        write(ler,*)
     1' -a2         (a1)             : end roll in azimuth  '
     2                    //'filter weights (degrees)'
        write(ler,*)
     1' -a3         (a4)             : begin roll out azimuth  '
     2                    //'filter weights (degrees)'
        write(ler,*)
     1' -a4         (-180.)          : end roll out azimuth  '
     2                    //'filter weights (degrees)'
        write(ler,*)
     1' -reject               : if present, reject dips between'
     2                     //' s1-s4, a1-a4 '
        write(ler,*)
     1' -V                    : if present, verbose printout'
       write(ler,*)' '
       write(ler,*)'usage:'
       write(ler,*)
       write(ler,*)'swfilt3d -N[file_in] -O[file_out]'
       write(ler,*)'         -ildm[] -cldm[] -ilazim[] -clazim[]'
       write(ler,*)'         -ilhw[] -clhw[] -dz[] -smax[]'
       write(ler,*)'         -ilinterp[] -clinterp[]'
       write(ler,*)'         -sigma1[] -sigma2[] -w[]' 
       write(ler,*)'         -t1[] -t2[] -t3[] -t4[]'
       write(ler,*)'         -f1[] -f2[] -f3[] -f4[]'
       write(ler,*)'         -startline[] -endline[]'
       write(ler,*)'         -starttrace[] -endtrace[]'
       write(ler,*)'         -pmin[] -pmax[] -qmin[] -qmax[]'
       write(ler,*)'         -p1[] -p2[] -p3[] -p4[]'
       write(ler,*)'         -q1[] -q2[] -q3[] -q4[]'
       write(ler,*)'         -smax[]'
       write(ler,*)'         -s1[] -s2[] -s3[] -s4[]'
       write(ler,*)'         -a1[] -a2[] -a3[] -a4[]'
       write(ler,*)'         -[reject,rho,V]'
       write(ler,*)' '
       write(ler,*)

      return
      end
