C***********************************************************************
C                 copyright 2001, Amoco Production Company             *
C                             All Rights Reserved                      *
C                     an affiliate of BP America Inc.                  *
C***********************************************************************
      subroutine ti2dr ( rt, rv, nn, MaxTime, MaxDepth )

c ti2dr converts an input INTERVAL velocity function
c in TIME to RMS in DEPTH using Dix Equation. The algorithm used is:
c
c vrms = sqrt[sum(1,nn)[vint(i)**2*TimeInterval(i)]/sum(1,nn)TimeInterval(i)]
c
c from Applied Geophysics, pp271.
c
c it is assumed that:
c
c  rt() = 2way TIME in seconds
c  rv() = INTERVAL velocity in units/second
c  velocity is referenced to the bottom of the interval
c  first time sample is at time zero 
c
c on output rv() = RMS velocity in units/second
c on output rt() = DEPTH in units
c
c MaxTime is maximum time on input dataset
c MaxDepth is maximum required depth on output dataset

#include <f77/lhdrsz.h>

c declare variables passed from calling routine 

      integer  nn 

      real     rt(*), rv(*), MaxTime, MaxDepth 

c declare variables unique to subroutine

      integer i

      real vrms(SZLNHD), Numerator
      real TimeInterval, CummulativeTime, depth(SZLNHD)

c initialize memory
      
      call vclr( vrms, 1, SZLNHD )
      call vclr( depth, 1, SZLNHD )

c first RMS velocity  equals first INTERVAL velocity

      vrms(1) = rv(1)
      depth(1) = rv(1) * rt(1) / 2.

c initialize Numerator and CummulativeTime

      Numerator = rv(1)**2 * rt(1) / 2.
      CummulativeTime = rt(1) / 2.

      DO i=2,nn
         TimeInterval = ( rt(i) - rt(i-1) ) / 2.
         Numerator = Numerator + rv(i)**2 * TimeInterval 
         CummulativeTime = CummulativeTime + TimeInterval
         vrms(i) = sqrt( Numerator / CummulativeTime )
         depth(i) = depth(i-1) + TimeInterval * rv(i)
      ENDDO

c form vrms for last sample if required

      IF ( depth(nn) .lt. MaxDepth ) then

c can only do this if we have enough time data on input

         if ( rt(nn) .lt. MaxTime ) then
            TimeInterval = ( MaxTime - rt(nn) ) / 2.
            Numerator = Numerator + rv(nn)**2 * TimeInterval
            CummulativeTime = CummulativeTime + TimeInterval

c add one to the number of elements in input function

            nn = nn + 1
            vrms(nn) = sqrt ( Numerator / CummulativeTime )

c assign depth for new velocity table entry

            depth(nn) = depth(nn - 1) +
     :           ( ( MaxTime - rt(nn-1) ) / 2. ) * rv(nn - 1)
         endif

      ENDIF

      call vmov ( vrms, 1, rv, 1, nn)
      call vmov ( depth, 1, rt, 1, nn)

      return
      end
