C***********************************************************************
C                 copyright 2001, Amoco Production Company             *
C                             All Rights Reserved                      *
C                     an affiliate of BP America Inc.                  *
C***********************************************************************
c -----------------  vstak Main Routine -----------------------

c     Program Changes:

c      - added panel option and did general clean up for clarity
c        of purpose: Mar21/96 P. Garossino

c     Program Description:

c      - read in data from two separate datasets, each having the same number
c        of samples per trace.  Stack the two traces together using on of
c        the following options:
c 
c            1. O = N1 + amp * N2
c
c            2. O = N2 where N1 = 0.0
c
c            3. O = N1**2 + [amp * N2]**2
c
c            4. O = sqrt [ N1**2 + {amp * N2}**2 ]

c        The N1 and N2 datasets may have either the same number of records and
c        traces per record or the N2 dataset may be a single trace record if 
c        using the panel option.  In this case the single N2 trace will be 
c        stacked to all traces in the input N1 record.  If the panel option is
c        not used the the N2 dataset must have the same number of traces per
c        record as the input dataset.

c get machine dependent parameters 

      implicit none

#include <save_defs.h>
#include <f77/iounit.h>
#include <f77/lhdrsz.h>
#include <f77/sisdef.h>

c declare standard USP variables

      integer     lhdr1( SZLNHD ), lhdr2( SZLNHD )
      integer     itr1, itr2
      integer     nsamp, nsi, ntrc, nrec, iform
      integer     luout, lbytes, nbytes, obytes, nbytes2
      integer     luin1, luin2, lbytes1, lbytes2, ns, ne, irs, ire 
      integer     lbyout
      integer     argis, ntrco, nreco
      integer     iend, ist, nsampo, JJ, KK, jerr

      real  tri_N1, tri_N2

      character  name * 5,  ntap1* 255, ntap2 * 255, otap * 255

      logical  verbos

c declare variables used by this routine

      integer pipe, idummy, ii
      integer ifmt_RecNum, l_RecNum, ln_RecNum, RecNum
      integer ifmt_StaCor, l_StaCor, ln_StaCor, StaCor1, StaCor2
      integer nsamp2, nsi2, ntrc2, nrec2, on_mute, off_mute
      integer iabort, ierr1, ierr2, ierr3, ierr4, ierr5
 
      real  amp, tri_out

      pointer (mem_itr1, itr1(2))
      pointer (mem_itr2, itr2(2))
      pointer (mem_tri_N1, tri_N1(2))
      pointer (mem_tri_N2, tri_N2(2))
      pointer (mem_tri_out, tri_out(2))

      logical vector, over, sum_of_squares, normalize, splice, panel
      logical bigger, smaller

c initialize variables

      data name /'VSTAK'/
      data luout  / 2 /
      data lbytes / 0 /
      data nbytes / 0 /
      data obytes / 0 /
      data verbos/.false./
      data vector/.false./
      data pipe/3/
      data normalize/.false./
      data splice/.false./
      data panel /.false./
      data bigger /.false./
      data smaller /.false./

c get online help if requested

      if ( argis('-?') .gt. 0 .or. 
     :     argis('-h') .gt. 0 .or. 
     :     argis('-help') .gt. 0 ) then
          call help()
          stop
      endif

c open printout file

#include <f77/open.h>

c parse command line parameters

      call cmdln ( ntap1, ntap2, otap, ist, iend, ns, ne, irs, ire, 
     :     amp, over, vector, verbos, sum_of_squares, normalize, splice,
     :     panel, bigger, smaller )

c open -N1 dataset

      call getln( luin1, ntap1, 'r', 0)

c debug
      write(ler,*)' luin1 =',luin1
c debug
      if (luin1 .lt. 0) then
         write(LERR,*)'Cannot open N1 dataset', ntap1
         write(LERR,*)'Check spelling / existence and rerun'
         write(LERR,*)'FATAL'
         write(LER,*)' '
         write(LER,*)'VSTAK: Cannot open N1 dataset', ntap1
         write(LER,*)'       Check spelling / existence and rerun'
         write(LER,*)'FATAL'
         stop
      endif

      IF ( .not. over ) then

c open -N2 dataset and check to see if in IKP, if so open a socket

         if ( ntap2(1:1) .eq. ' ' ) then
             write(LERR,*)'VSTAK assumed to be running inside ikp'

c pipe is defined above to be 3 as that is the socket number used for
c the -N2 port in IKP

             call sisfdfit (luin2, pipe)
             ns  = 0
             ne  = 0
             irs = 1
             ire = 0
         else
             call getln( luin2, ntap2, 'r', 0)
         endif

         if(luin2 .lt. 0) then
            write(LERR,*)'Cannot open N2 dataset', ntap2
            write(LERR,*)'Check spelling / existence and rerun'
            write(LERR,*)'FATAL'
            write(LER,*)' '
            write(LER,*)'VSTAK: Cannot open N2 dataset', ntap2
            write(LER,*)'       Check spelling / existence and rerun'
            write(LER,*)'FATAL'
            stop
         endif

c open output dataset

         call getln( luout, otap, 'w', 1)

      ELSE

c here -N2 and -O are the same so we will not need to open the
c -O dataset separately.  We will be overprinting the -N2 dataset
c which of course means that we will have to be able to back up
c on this logical unit so therefore it CANNOT be a pipe.

         if(otap .eq. ' ') then
            write(LERR,*)' '
            write(LERR,*)' Output data set must be a named file'
            write(LERR,*)' FATAL'
            write(LER,*)' '
            write(LER,*)'VSTAK: Output dataset must be a named file'
            write(LER,*)'FATAL'
            stop
         endif
         call lbopen( luin2, ntap2, 'a+' )
         luout = luin2

      ENDIF

      if ( ntap2 .eq. otap ) then
         call sislgbuf ( luin1, 'off' )
         call sislgbuf ( luout, 'off' )
      endif

c read the line header from N1 dataset

      lbytes1 = 0
      call rtape ( luin1, lhdr1, lbytes1 )
      if(lbytes1 .eq. 0) then
         write(LERR,*)'VSTAK: no header read on unit ',ntap1
         write(LERR,*)'FATAL'
         write(LERR,*)'Check existence of file & rerun'
         write(LER,*)'VSTAK: no header read on unit ',ntap1
         write(LER,*)'FATAL'
         write(LER,*)' '
         stop
      endif

c  get global parameters from N1 lineheader

      call saver( lhdr1, 'NumSmp', nsamp , LINHED )
      call saver( lhdr1, 'SmpInt', nsi   , LINHED )
      call saver( lhdr1, 'NumTrc', ntrc  , LINHED )
      call saver( lhdr1, 'NumRec', nrec  , LINHED )
      call saver( lhdr1, 'Format', iform , LINHED )

c set up pointers to trace header values to be used in this routine

      call savelu('RecNum',ifmt_RecNum,l_RecNum,ln_RecNum,TRACEHEADER)
      call savelu('StaCor',ifmt_StaCor,l_StaCor,ln_StaCor,TRACEHEADER)

c print historical line header from N1 dataset to printout file

      call hlhprt (lhdr1, lbytes1, name, 5, LERR)

c read the line header from N2 dataset

      lbytes2 = 0
      call rtape ( luin2, lhdr2, lbytes2 )

      IF ( .not. over ) THEN

         if ( lbytes2 .eq. 0 ) then
            write(LERR,*)'VSTAK: no header read on unit ',ntap2
            write(LERR,*)'FATAL'
            write(LERR,*)'Check existence of file & rerun'
            write(LER,*)'VSTAK: no header read on unit ',ntap2
            write(LER,*)'FATAL'
            write(LER,*)' '
            stop
         endif

c  get global parameters from N2 lineheader

         call saver( lhdr2, 'NumSmp', nsamp2 , LINHED )
         call saver( lhdr2, 'SmpInt', nsi2   , LINHED )
         call saver( lhdr2, 'NumTrc', ntrc2  , LINHED )
         call saver( lhdr2, 'NumRec', nrec2  , LINHED )

         if ( panel ) then

            if ( ntrc2 .ne. 1 ) then
               write(LERR,*)' '
               write(LERR,*)'VSTAK: The N2 dataset must have single '
               write(LERR,*)'       trace records when using the panel'
               write(LERR,*)'       option. Yours has ',ntrc2,' trace'
               write(LERR,*)'       record.'
               write(LERR,*)'FATAL'
               write(LER,*)' '
               write(LER,*)'VSTAK: The N2 dataset must have single '
               write(LER,*)'       trace records when using the panel'
               write(LER,*)'       option. Yours has ',ntrc2,' trace'
               write(LER,*)'       record.'
               write(LER,*)'FATAL'
               stop
            endif

         else

           if ( ntrc2 .ne. ntrc ) then
               write(LERR,*)' '
               write(LERR,*)'VSTAK: The N2 dataset must have the same '
               write(LERR,*)'       traces / record as the N1 dataset'
               write(LERR,*)'FATAL'
               write(LER,*)' '
               write(LER,*)'VSTAK: The N2 dataset must have the same '
               write(LER,*)'       traces / record as the N1 dataset'
               write(LER,*)'FATAL'
               stop
            endif
         endif

         if ( nsi2 .ne. nsi ) then
            write(LERR,*)' '
            write(LERR,*)'VSTAK: The N2 dataset must have the same '
            write(LERR,*)'       sample interval as the N1 dataset.'
            write(LERR,*)'FATAL'
            write(LER,*)' '
            write(LER,*)'VSTAK: The N2 dataset must have the same '
            write(LER,*)'       sample interval as the N1 dataset.'
            write(LER,*)'FATAL'
            stop
         endif

         if ( nsamp2 .lt. nsamp ) then
            write(LERR,*)' '
            write(LERR,*)'VSTAK: The N2 dataset must have at least  '
            write(LERR,*)'       the same number of samples as the N1'
            write(LERR,*)'       dataset.'
            write(LERR,*)'FATAL'
            write(LER,*)' '
            write(LER,*)'VSTAK: The N2 dataset must have at least  '
            write(LER,*)'       the same number of samples as the N1'
            write(LER,*)'       dataset.'
            write(LER,*)'FATAL'
            stop
         endif

         if ( nrec2 .lt. nrec ) then
            write(LERR,*)' '
            write(LERR,*)'VSTAK: The N2 dataset must have at least '
            write(LERR,*)'       the same number of records as the '
            write(LERR,*)'       N1 dataset.'
            write(LERR,*)'FATAL'
            write(LER,*)' '
            write(LER,*)'VSTAK: The N2 dataset must have at least '
            write(LER,*)'       the same number of records as the '
            write(LER,*)'       N1 dataset.'
            write(LER,*)'FATAL'
            stop
         endif

      ELSE

         write(LERR,*)'Overwriting -N2 Dataset'
         call rwd(luout)
         if ( lbytes2 .eq. 0 ) then

c in this case the N2 and O datasets are the same and  do not yet
c exist.  We will have to move the N1 lineheader to N2

            call move (1,lhdr2,lhdr1,lbytes1)
            nsamp2 = nsamp
            nsi2 = nsi
            ntrc2 = ntrc
            nrec2 = nrec
         else

c  get global parameters from N2 lineheader

            call saver( lhdr2, 'NumSmp', nsamp2 , LINHED )
            call saver( lhdr2, 'SmpInt', nsi2   , LINHED )
            call saver( lhdr2, 'NumTrc', ntrc2  , LINHED )
            call saver( lhdr2, 'NumRec', nrec2  , LINHED )
         endif
      ENDIF

      if( nsamp .gt. SZLNHD ) then
         write(LERR,*)'Too many samples in traces -- FATAL'
         write(LERR,*)'window the input data & rerun'
         write(LER,*)'VSTAK: '
         write(LER,*)'Too many samples in traces '
         write(LER,*)'window the input data & rerun'
         write(LER,*)'FATAL'
         stop
      endif

c check default values; update header

      call cmdchk( ns, ne, irs, ire, ntrc, nrec )
 
      ntrco = ne - ns + 1
      call savew( lhdr1, 'NumTrc', ntrco  , LINHED )

      nreco = ire - irs + 1
      call savew( lhdr1, 'NumRec', nreco, LINHED )

      iend = iend / nsi + .5
      ist = ist / nsi
      if ( ist .le. 1) ist = 1
      if ( iend .eq. 0 ) iend = nsamp
      if ( iend .gt. nsamp ) iend = nsamp
      nsampo = iend - ist + 1
      call savew( lhdr1, 'NumSmp', nsampo, LINHED )

      obytes = SZTRHD + SZSMPD * nsampo

      if (.not. over) then
         call savhlh( lhdr1, lbytes1, lbyout )
         call wrtape ( luout, lhdr1, lbyout )
      else
         if (lbytes2 .eq. 0) then
            lbyout = lbytes1
            call wrtape ( luout, lhdr2, lbyout )
         else
            lbyout = lbytes2
            call wrtape ( luout, lhdr2, lbyout )
         endif
      endif

c print control parameters to printout file

      call verbal( ntap1, ntap2, otap, nsamp, nsi, ntrc, nrec, 
     :     ist, iend, iform, luin1, luin2, luout, nreco, ntrco, nsampo, 
     :     nsamp2, nsi2, nrec2, ntrc2, irs, ire, ns, ne, over, vector,  
     :     sum_of_squares, panel, splice, amp, normalize, bigger, 
     :     smaller, verbos)

c allocate arrays for trace data

      iabort = 0

      call galloc (mem_itr1, SZTRHD + (nsamp * SZSMPD), ierr1, iabort)

      call galloc (mem_itr2, SZTRHD + (nsamp2 * SZSMPD), ierr2, iabort)

      call galloc (mem_tri_out, SZTRHD+(nsampo*SZSMPD), ierr3, iabort)

      call galloc (mem_tri_N1, (nsampo * SZSMPD), ierr4, iabort)

      call galloc (mem_tri_N2, (nsampo * SZSMPD), ierr5, iabort)
        
      if (ierr1 .ne. 0 .or. ierr2 .ne. 0  .or. ierr3 .ne. 0 .or.
     :    ierr4 .ne. 0 .or. ierr5 .ne. 0) then
          write (LERR,*) name, 
     :                 ' : Unable to allocate workspace'
          write (LERR,*) SZTRHD + (nsamp * SZSMPD),' bytes'
          write (LERR,*) SZTRHD + (nsamp2 * SZSMPD),' bytes'
          write (LERR,*) SZTRHD + (nsampo * SZSMPD),' bytes'
          write (LERR,*) (nsampo * SZSMPD),' bytes'
          write (LERR,*) (nsampo * SZSMPD),' bytes'
          write (LERR,*) 'FATAL'
          write (LERR,*) '**********************************'
          write (LERR,*)
          write (LER,*) name, 
     :                 ' : Unable to allocate workspace'
          write (LER,*) SZTRHD + (nsamp * SZSMPD),' bytes'
          write (LER,*) SZTRHD + (nsamp2 * SZSMPD),' bytes'
          write (LER,*) SZTRHD + (nsampo * SZSMPD),' bytes'
          write (LER,*) (nsampo * SZSMPD),' bytes'
          write (LER,*) (nsampo * SZSMPD),' bytes'
          write (LER,*) 'FATAL'
          write (LER,*) '**********************************'
          stop
      endif
      write (LERR,*) name, 
     :               ' : Allocating workspace'
      write (LERR,*) SZTRHD + (nsamp * SZSMPD),' bytes'
      write (LERR,*) SZTRHD + (nsamp2 * SZSMPD),' bytes'
      write (LERR,*) SZTRHD + (nsampo * SZSMPD),' bytes'
      write (LERR,*) (nsampo * SZSMPD),' bytes'
      write (LERR,*) (nsampo * SZSMPD),' bytes'
      write (LERR,*) '************************************'
      write (LERR,*)

      call vclr (itr1, 1, ITRWRD + nsamp)
      call vclr (itr2, 1, ITRWRD + nsamp2)
      call vclr (tri_out, 1, ITRWRD + nsampo)
      call vclr ( tri_N1, 1, nsampo )
      call vclr ( tri_N2, 1, nsampo )

c skip to start record in each dataset if required

      call recskp ( 1, irs-1, luin1, ntrc, itr1 )
      if ( .not. over .and. .not. panel ) 
     :     call recskp ( 1, irs-1, luin2, ntrc, itr2 )
      if ( panel ) call recskp ( 1, irs-1, luin2, ntrc2, itr2 )

      DO JJ = irs, ire

c skip to start trace if necessary

         if ( ns .gt. 1 ) then
            call trcskp ( JJ, 1, ns-1, luin1, ntrc, itr1 )
            if ( .not. over .and. .not. panel ) 
     :           call trcskp ( JJ, 1, ns-1, luin2, ntrc, itr2 )
         endif

         if ( panel ) then
            
c read in trace to be stacked with entire output record

            nbytes2 = 0
            call rtape  ( luin2 , itr2, nbytes2 )
            if(nbytes2 .eq. 0) then
               write(LERR,*)'End of file on input N2 dataset:'
               write(LERR,*)'  rec= ',JJ,'  trace= ',KK
               go to 999
            endif
            call saver2(itr2,ifmt_StaCor,l_StaCor, ln_StaCor,
     1           StaCor2, TRACEHEADER)

            if ( StaCor2 .ne. 30000 ) then
               call vmov ( itr2(ITHWP1+ist-1), 1, tri_N2, 1, nsampo )
            else
               call vclr( tri_N2, 1, nsampo )
            endif
         endif

         DO KK = ns, ne

c read in N1 trace

            nbytes = 0
            call rtape ( luin1 , itr1, nbytes )
            if(nbytes .eq. 0) then
               write(LERR,*)'End of file on input DSN 1:'
               write(LERR,*)'  rec= ',JJ,'  trace= ',KK
               go to 999
            endif

            call saver2(itr1,ifmt_StaCor,l_StaCor, ln_StaCor,
     1           StaCor1, TRACEHEADER)
            call saver2(itr1,ifmt_RecNum,l_RecNum, ln_RecNum,
     1           RecNum    , TRACEHEADER)

            if ( StaCor1 .ne. 30000 ) then
               call vmov ( itr1(ITHWP1+ist-1), 1, tri_N1, 1, nsampo )
            else
               call vclr ( tri_N1, 1, nsampo)
            endif

            if ( .not. panel ) then

c only read an N2 trace if NOT doing the panel option and only if
c there is a dataset attached.  If N2 and O datasets are the same name
c is not necessarily the case that it contains data yet.

               nbytes2 = 0
               if(lbytes2 .ne. 0) then
                  call rtape  ( luin2 , itr2, nbytes2 )
                  if(nbytes2 .eq. 0) then
                     write(LERR,*)'End of file on input DSN 2:'
                     write(LERR,*)'  rec= ',JJ,'  trace= ',KK
                     go to 999
                  endif

                  call vmov (itr2(ITHWP1+ist-1), 1, tri_N2, 1, nsampo)
               else
                  call vclr(tri_N2,1,nsampo)
               endif

               call saver2(itr2,ifmt_StaCor,l_StaCor, ln_StaCor,
     1              StaCor2, TRACEHEADER)
               if(StaCor2 .eq. 30000) call vclr(tri_N2,1,nsampo)

            endif
 
            if ( verbos .and. KK .eq. ns ) then
               write(LERR,*)'Output Record ',RecNum
            endif

            if (StaCor1.eq.30000 .AND. StaCor2.eq.30000) then
               call savew2(itr1,ifmt_StaCor,l_StaCor, ln_StaCor,
     1              30000   , TRACEHEADER)
            elseif (StaCor1.eq.30000 .AND. StaCor2.ne.30000) then

               if ( .not. panel ) then

c put trace header from N2 onto N1 unless we are in panel mode.

                  call move ( 1, itr1, itr2, SZTRHD )
               endif
                  
            endif

c clear output trace

            call vclr (tri_out, 1, nsampo )

            if ( panel ) then

c detect on and off mutes based  on N1 dataset for panel option

               call bd_detmut ( tri_N1, on_mute, nsampo, 0)
               call bd_detmut ( tri_N1, off_mute, nsampo, 1)
            endif

            if ( splice ) then

c splice in -N1 dataset if non-zero

               do ii = 1, nsampo
                  if ( abs( tri_N1(ii) ) .gt. 1.e-31 ) then
                     tri_out(ii) = tri_N1(ii)
                  else
                     tri_out(ii) = amp * tri_N2(ii)
                  endif
               enddo

            elseif ( bigger ) then

c output the larger of N1 or amp*N2, do the check based on magnitude but
c output the signed sample value.

               do ii = 1, nsampo
                  if ( abs( tri_N1(ii) ) .gt. abs(amp*tri_N2(ii) ) ) 
     :                 then
                     tri_out(ii) = tri_N1(ii)
                  else
                     tri_out(ii) = amp * tri_N2(ii)
                  endif
               enddo

            elseif ( smaller ) then

c output the smaller of N1 or amp*N2, do the check based on magnitude but
c output the signed sample value.

               do ii = 1, nsampo
                  if ( abs( tri_N1(ii) ) .lt. abs(amp*tri_N2(ii) ) )
     :                 then
                     tri_out(ii) = tri_N1(ii)
                  else
                     tri_out(ii) = amp * tri_N2(ii)
                  endif
               enddo

            elseif ( sum_of_squares ) then

c sum_of_squares option

               do ii = 1, nsampo
                  tri_out(ii) = ( amp * tri_N2(ii) )**2 
     :                 + ( tri_N1(ii) )**2
               enddo

            elseif ( vector ) then

c sqrt of sum_of_squares option

               do ii = 1, nsampo
                  tri_out(ii) = sqrt ( ( amp * tri_N2(ii) )**2 
     :                 + ( tri_N1(ii) )**2 )
               enddo

            else

               do ii = 1, nsampo
                  tri_out(ii) = amp * tri_N2(ii) + tri_N1(ii)
               enddo

            endif

c normalize output when both input samples are live if requested, of course
c we  don't want to do this in splice mode as values outside the splice 
c zone would be divided by 2 when we didn't want that to happen.

            if ( normalize ) then

               do ii=1,nsampo

                  if ( abs( tri_N1(ii) )  .gt. 1.e-31 .and. 
     :                 abs( tri_N2(ii) )  .gt. 1.e-31 ) then

                     tri_out(ii) = tri_out(ii) / 2.
                  endif
               enddo

            endif

            if (over) then

               if ( lbytes2 .ne. 0 ) then

c must back up one trace before writing the output over the
c pre-existing dataset for this trace

                  call bkspt ( luin2, 1,  idummy )
               endif
            endif 

            if ( panel ) then

c restore mute based on N1 dataset for panel option

               call bd_resmut ( tri_out, on_mute, nsampo, 0 )
               call bd_resmut ( tri_out, off_mute, nsampo, 1 )
            endif

            call vmov ( tri_out, 1, itr1(ITHWP1), 1, nsampo )
            call wrtape( luout, itr1, obytes )
 
         ENDDO

c skip from current trace to end of record

         if ( ne .lt. ntrc) then
            call trcskp ( JJ, ne+1, ntrc, luin1, ntrc, itr1 )
            if ( .not. over .and. .not. panel )
     :           call trcskp ( JJ, ne+1, ntrc, luin2, ntrc, itr2 )
         endif

      ENDDO

c normal termination

      call lbclos(luin1)
      if ( .not. over ) call lbclos(luin2)
      call lbclos(luout)

      write(LERR,*)' processed ',nreco,' records of ',ntrco,' traces'
      write(LERR,*)' Normal Termination'
      write(LER,*)'vstak: Normal Termination'
      stop

 999  continue

c abnormal termination
      
      call lbclos(luin1)
      if(.not.over)call lbclos(luin2)
      call lbclos(luout)

      write(LERR,*)' Abnormal Termination'
      write(LER,*)'vstak: Abnormal Termination'
      stop
      end

c Subroutines, help, cmdln, verbal

      subroutine help

#include <f77/iounit.h>

        write(LER,*)' '
        write(LER,*)'Command Line Arguments for VSTAK: vertical stack'
        write(LER,*)' '
        write(LER,*)'Input....................................... (def)'
        write(LER,*)' '
        write(LER,*)'-N1[ntap1] -- input data set name 1        (stdin)'
        write(LER,*)'-N2[ntap2] -- input data set name 2         (none)'
        write(LER,*)'              if ntap2 = otap then contents of '
        write(LER,*)'              ntap1 will be added to otap'
        write(LER,*)'-O[otap]   -- output data set name'
        write(LER,*)'-s[ist]    -- start time                    (0 ms)'
        write(LER,*)'-e[iend]   -- end time                 (last samp)'
        write(LER,*)'-ns[ns]    -- start trace number        (first tr)'
        write(LER,*)'-ne[ne]    -- end trace number           (last tr)'
        write(LER,*)'-rs[irs]   -- start record             (first rec)'
        write(LER,*)'-re[ire]   -- end record                (last rec)'
        write(LER,*)'-amp[amp]  -- weight:  ntap1 + amp*ntap2    ( 1.0)'
        write(LER,*)'-norm      -- (N1 + amp*N2)/2 if both are live'
        write(LER,*)'-splice    -- merge only if one sample is live and'
        write(LER,*)'              the other zero'
        write(LER,*)'-panel     -- traces from N2  will be stacked with'
        write(LER,*)'              all traces of record from N1'
        write(LER,*)'-bigger    -- output sample from N1 if abs(N1)'
        write(LER,*)'              is greater than abs(amp*N2) else '
        write(LER,*)'              output sample from amp*N2.'
        write(LER,*)'-smaller   -- output sample from N1 if abs(N1)'
        write(LER,*)'              is less than abs(amp*N2) else '
        write(LER,*)'              output sample from amp*N2.'
        write(LER,*)'-S         -- vector square: N1**2 & amp*N2**2'
        write(LER,*)'-R         -- vector square root:  N1**2 & amp*N2**
     :2'
        write(LER,*)'-V         -- verbos printout'
        write(LER,*)' '
        write(LER,*)'Usage:'
        write(LER,*)'      vstak -N1[] -N2[] -O[] -s[] -e[] -ns[] -ne[]'
        write(LER,*)'            -rs[] -re[] -amp[] -panel -norm -V '
        write(LER,*)'            [-S -R -splice -bigger -smaller ]'
        write(LER,*)' '

      return
      end

c-----
c     get command arguments
c
c     ntap1 - C*255  input file name
c     ntap2 - C*255  input file name
c     otap  - C*255  output file name
c     ist   - I      start trace
c    iend   - I      stop end trace
c     ns   - I      start trace
c     ne   - I      stop trace
c    irs   - I      start record
c    ire   - I      stop end record
c    over   - L      overwrite output file
c    vector - L      sqroot of sum of squares
c    sum_of_squares    - L      sum of squares
C    normalize - L   normalize output
c    splice  -L      splice output
c    panel  -L       panel option
c    verbos - L      verbose output or not
c-----
      subroutine cmdln ( ntap1, ntap2, otap, ist, iend, ns, ne, irs, 
     :     ire, amp, over, vector, verbos, sum_of_squares, normalize, 
     :     splice, panel, bigger, smaller )

#include <f77/iounit.h>

c declare variable passed from calling routine

      integer      argis, ist, iend, ns, ne, irs, ire

      real         amp

      character    ntap1*(*), ntap2*(*), otap*(*)

      logical  vector, verbos, over, sum_of_squares, normalize
      logical  splice, panel, bigger, smaller

      call argr4('-amp',amp,1.,1.)

      bigger = ( argis( '-bigger' ) .gt. 0 )
      
      call argi4('-e',iend,0,0) 

      normalize = ( argis( '-norm' ) .gt. 0 )
      call argstr('-N1',ntap1,' ',' ')
      call argstr('-N2',ntap2,' ',' ')
      call argi4('-ne',ne,0,0)
      call argi4('-ns',ns,0,0)

      call argstr('-O',otap,' ',' ')

      panel = ( argis( '-panel' ) .gt. 0 )

      call argi4('-rs',irs,1,1)
      call argi4('-re',ire,0,0)
      vector = ( argis( '-R' ) .gt. 0 )

      smaller = ( argis( '-smaller' ) .gt. 0 )
      splice = ( argis( '-splice' ) .gt. 0 )
      sum_of_squares    = ( argis( '-S' ) .gt. 0 )
      call argi4('-s',ist,1,1) 

      verbos = ( argis( '-V' ) .gt. 0 )

      over = .false.

      if(ntap2 .eq. otap .AND. otap(1:1) .ne. ' ') over = .true.
      if(ntap1 .eq. otap .AND. otap(1:1) .ne. ' ') then
        write(LERR,*)' Cannot overwrite N1 data set'
        write(LERR,*)' If you wish do do an overwrite, set N2 & O to '
        write(LERR,*)' be the same.  N1 will then be vertically staked'
        write(LERR,*)' with whatever is in the output or with zeroes '
        write(LERR,*)' if the output does not yet exist'
        stop
      endif

      if(over) then
         if(ist .ne. 1) then
            write(LERR,*)
     :  'start time must default for overwrite of N2 data set'
            stop
         endif
         if(iend .ne. 0) then
            write(LERR,*)
     :  'end time must default for overwrite of N2 data set'
            stop
         endif
         if(ns .ne. 0) then
            write(LERR,*)
     :  'start trace no. must default for overwrite of N2 data set'
            stop
         endif
         if(ne .ne. 0) then
            write(LERR,*)
     :  'end trace no. must default for overwrite of N2 data set'
            stop
         endif
         if(irs .ne. 1) then
            write(LERR,*)
     :  'start record no. must default for overwrite of N2 data set'
            stop
         endif
         if(ire .ne. 0) then
            write(LERR,*)
     :  'end record no. must default for overwrite of N2 data set'
            stop
         endif
      endif

      return
      end
      
      subroutine verbal ( ntap1, ntap2, otap, nsamp, nsi, ntrc, nrec, 
     :     ist, iend, iform, luin1, luin2, luout, nreco, ntrco, nsampo, 
     :     nsamp2, nsi2, nrec2, ntrc2, irs, ire, ns, ne, over, vector,  
     :     sum_of_squares, panel, splice, amp, normalize, bigger, 
     :     smaller, verbos)

#include <f77/iounit.h>

c declare variables passed from calling routine

      integer nsamp, nsi, ntrc, nrec, ist, iend, iform, luin1, luin2
      integer luout, nreco, ntrco, nsampo, nsamp2, nsi2, nrec2, ntrc2
      integer irs, ire, ns, ne

      real amp

      character ntap1*(*), ntap2*(*), otap*(*)

      logical over, vector, sum_of_squares, panel, splice, verbos
      logical normalize, bigger, smaller

      write(LERR,*)' '
      write(LERR,*)'========================================== '
      write(LERR,*)' '
      write(LERR,*)' N1 Dataset Line Header Parameters'
      write(LERR,*)' '
      length = lenth(ntap1)
      if (length .gt. 0) then
        write(LERR,*) ' Filename           =  ',ntap1(1:length)
      else
        write(LERR,*) ' Filename           =  stdin'
      endif
      write(LERR,*) ' Samples / Trace    =  ', nsamp
      write(LERR,*) ' Sample Interval    =  ', nsi  
      write(LERR,*) ' Traces per Record  =  ', ntrc 
      write(LERR,*) ' Records per Line   =  ', nrec 
      write(LERR,*) ' Format of Data     =  ', iform
      write(LERR,*)' '
      write(LERR,*)' N2 Dataset Line Header Parameters'
      write(LERR,*)' '
      length = lenth(ntap2)
      if (length .gt. 0) then
        write(LERR,*) ' Filename           =  ',ntap2
      else
        write(LERR,*) ' Filename           =  pipe'
      endif
      write(LERR,*) ' Samples / Trace    =  ', nsamp2
      write(LERR,*) ' Sample Interval    =  ', nsi2  
      write(LERR,*) ' Traces per Record  =  ', ntrc2 
      write(LERR,*) ' Records per Line   =  ', nrec2 
      write(LERR,*) ' Format of Data     =  ', iform
      write(LERR,*)' '
      write(LERR,*)' Fortran Units Used'
      write(LERR,*)' '
      write(LERR,*) ' N1 unit            =  ', luin1
      write(LERR,*) ' N2 unit            =  ', luin2
      write(LERR,*) ' O  unit            =  ', luout
      write(LERR,*)' '
      write(LERR,*)' Output Dataset Parameters '
      write(LERR,*)' '
      length = lenth(otap)
      if (length .gt. 0) then
        write(LERR,*) ' Filename           =  ', otap
      else
        write(LERR,*) ' Filename           =  stdout'
      endif
      write(LERR,*) ' Samples / Trace    =  ', nsampo
      write(LERR,*) ' Sample Interval    =  ', nsi  
      write(LERR,*) ' Traces per Record  =  ', ntrco 
      write(LERR,*) ' Records per Line   =  ', nreco
      write(LERR,*) ' Format of Data     =  ', iform
      write(LERR,*)' '
      write(LERR,*)' Command Line Parameters '
      write(LERR,*)' '
      write(LERR,*) ' start record          =  ', irs 
      write(LERR,*) ' end record            =  ', irs 
      write(LERR,*) ' start trace           =  ', ns
      write(LERR,*) ' end trace             =  ', ne
      write(LERR,*) ' start sample          =  ', ist
      write(LERR,*) ' end sample            =  ', iend
      write(LERR,*) ' Scale factor for N2=  ',amp
      write(LERR,*)' '
      if ( verbos )  write(LERR,*) ' verbose printout requested'
      write(LERR,*)' '
      write(LERR,*)' Vstak Option Chosen '
      write(LERR,*)' '
      if (bigger) 
     :     write(LERR,*)'output larger of abs(N1) or abs(amp*N2)'
      if (smaller) 
     :     write(LERR,*)'output smaller of abs(N1) or abs(amp*N2)'
      if (sum_of_squares) write(LERR,*)'Do sum of squares option'
      if (sum_of_squares) write(LERR,*)'Do sum of squares option'

      if (vector) write(LERR,*)'Do square root of sum of squares option'

      if (splice) write(LERR,*)'Do splice of ',ntap2,' into zero values
     : of ', ntap1

      if (panel) write(LERR,*)'Do panel option '

      if (normalize) write(LERR,*)'Do normalization '

      if ( .not. sum_of_squares .and. 
     :     .not. vector .and. 
     :     .not. splice .and. 
     :     .not. panel ) then
         write(LERR,*)' Do Default vstak '
      endif
      write(LERR,*)' '
      write(LERR,*)' '
      write(LERR,*)'========================================== '
      write(LERR,*)' '
      
      return
      end
