/***********************************************************************
 *                copyright 2001, Amoco Production Company             *
 *                            All Rights Reserved                      *
 *                    an affiliate of BP America Inc.                  *
 ***********************************************************************/
/*
  This file contains the functions for
  displaying the seismic data.
*/


#include <time.h>
#include "xdisplayP.h"
#include "cross1.h"
#include "cross1_invert.h"
#include "prototyping.h"
/* DWN added for C_SAVER */
#include "ut_defs.h"


#define NUM_PARAMETER 9

#define X_AXIS_PAD 75 /* DWN 04/17/98 */
#define Y_AXIS_PAD 75 /* DWN 04/17/98 */

extern int v_index;
extern Boolean body_flag;
extern Boolean show_parameters;
extern Widget *v_parameter;
extern Widget draw_vel;

Pixmap pix;

static void expose_zooms(long);
static void mark_vel(long, long);
static void draw_label(int, long, long);
static CB_PROTO( expose_imageCB )

static Widget draw_area=NULL;
static int red_cursor_global=0,green_cursor_global=0,blue_cursor_global=0;

static const int	AxisLabelXoffset = 12;	/* RJM: added */

/*
  This function returns the next 
  available display id.
*/
int next_image(void)
{
  long id;	/* RJM: handle 64-bit architectures */
  struct tm *t;
  time_t second;
  static int sequence=0;

				/* Search for a free id. */
  for (id=1; id<MAX_IMAGES-1; id++) 
    if (xdisplay[id].init==0) {
      sequence++;		/* Sequence number of window creation. */

/*    xdisplay[id].window = (Widget)NULL;
      xdisplay[id].form = (Widget)NULL;
      xdisplay[id].menu = (Widget)NULL;
      xdisplay[id].menubar = (Widget)NULL;
      xdisplay[id].control = (Widget)NULL;
      xdisplay[id].scroll = (Widget)NULL;
      xdisplay[id].top = (Widget)NULL;
      xdisplay[id].gc = (GC)NULL;
      xdisplay[id].image = (XImage*)NULL;
      xdisplay[id].cmap = (Colormap)NULL;
      xdisplay[id].parent = 0;
      xdisplay[id].grid_flag = False;
      xdisplay[id].solid_flag = False;
      xdisplay[id].segment_flag = False;
      xdisplay[id].overlay_flag = False;
      xdisplay[id].label_flag = False;
      xdisplay[id].number_flag = False;
      xdisplay[id].lines_flag=True;
      xdisplay[id].auto_flag = False;
      xdisplay[id].text_flag = False;
      xdisplay[id].record_flag = False;
      xdisplay[id].axis_flag = True;
      xdisplay[id].dual_flag = False;
      xdisplay[id].segment_toggle = (Widget)NULL;
      xdisplay[id].number_label = (Widget)NULL;
      xdisplay[id].number_text = (Widget)NULL;
      xdisplay[id].hide_flag = False;
      xdisplay[id].flip_flag = False;
      xdisplay[id].upside_flag = False;
      xdisplay[id].swap_flag = False;
      xdisplay[id].label = (Widget)NULL;
      xdisplay[id].values = (Widget)NULL;
      xdisplay[id].scalar_offset = (Widget)NULL;
      xdisplay[id].grid_x = 0;
      xdisplay[id].grid_y = 0;
      xdisplay[id].grid_color = white;
      xdisplay[id].pick_x = 0.0;
      xdisplay[id].pick_y = 0.0;
      xdisplay[id].offset = 0;
      xdisplay[id].scalar = 0.0;
      for( i = 0; i<3; i++ )
        xdisplay[id].axis_units[i] = 0.0;
      for( i = 0; i<3; i++ )
        xdisplay[id].axis_offset[i] = 0.0;
      xdisplay[id].mode = 0;
      xdisplay[id].mode_label = (Widget)NULL;
      xdisplay[id].global = 0;
      xdisplay[id].sort = 0;
      xdisplay[id].middle = 0;
      xdisplay[id].pick_color = white;
      for( i = 0; i<10; i++ )
        xdisplay[id].groups[i] = 0;
      strcpy( xdisplay[id].filename, "" );
      strcpy( xdisplay[id].hostname, "" );
      xdisplay[id].background = white;
      strcpy( xdisplay[id].comment, "" );
*/

/*    time(&second);*/		/* Get current time. */
      second=time((time_t*)NULL);		/* RJM. Get current time. */
      t=localtime(&second);
      sprintf(xdisplay[id].time,"%02d:%02d",t->tm_hour,t->tm_min);
      sprintf(xdisplay[id].date,"%02d/%02d",t->tm_mon+1,t->tm_mday);
      strcpy(xdisplay[id].data1,trace_data1); /* Set initial trace header. */
      strcpy(xdisplay[id].data2,trace_data2); /* Set initial trace header. */
      strcpy(xdisplay[id].data3,trace_data3); /* Set initial trace header. */
      if (verbose)
	fprintf(stderr,"Image %d\n",id);


				/* Mark id as being used. */
      xdisplay[id].init=-2;	
      xdisplay[id].inter_parent=0;

				/* Add sequence number to window. */
      xdisplay[id].sequence=sequence;

				/* Window type is defaulted to image.*/
      strcpy(xdisplay[id].type,"Image");

				/* Segment lines are drawn. */
      xdisplay[id].lines_flag=True;

      return(id);
    }

				/* Too many windows are being used. */
  error_message("Too many images are being used. Delete some to continue.");
  return(id);
}


/*
  This function sets up the 
  scrolling and drawing areas.
*/
void xdisplay_image(long id, Widget window, Widget w, int width, int height)
{
				/* Store toplevel and form widget ids. */
  xdisplay[id].window=window;
  xdisplay[id].form=w; 

  xdisplay[id].scroll=XmCreateScrolledWindowVa(w,"scroll_display",
		       XmNtopAttachment,XmATTACH_WIDGET,
		       XmNtopWidget,xdisplay[id].menu,
		       XmNbottomAttachment,XmATTACH_FORM,
		       XmNleftAttachment,XmATTACH_FORM,
		       XmNrightAttachment,XmATTACH_FORM,
		       XmNwidth,width,
		       XmNheight,height,
		       XmNscrollingPolicy,XmAUTOMATIC,
		       NULL);

				/* Add pixels for axis. */
  draw_area=XmCreateDrawingAreaVa(xdisplay[id].scroll,"draw",
			  XmNwidth,xdisplay[id].image->width+X_AXIS_PAD,
			  XmNheight,xdisplay[id].image->height+Y_AXIS_PAD,
			  NULL);
  XtAddCallback(draw_area,XmNexposeCallback,expose_imageCB,(XtPointer)id);
  
                                     /* Set up event handlers. */
  XtAddEventHandler(draw_area,ButtonPressMask,False,button_pushEH,(XtPointer)id);
  XtAddEventHandler(draw_area,ButtonReleaseMask,False,button_releaseEH,(XtPointer)id);
  XtAddEventHandler(draw_area,PointerMotionMask,False,moving_mouseEH,(XtPointer)id);

/*XtAddCallback(window,XmNdestroyCallback,free_imageCB,(XtPointer)id); RJM: DO NOT DO THIS WAY SINCE free_imageCB references widgets already destroyed via destroyCB a la delete_windowCB!!! */
}


/* 
  This function sets the initial
  GC for the drawing area. It also
  handles all expose events.
*/
static void expose_imageCB(Widget w, XtPointer client_data, XtPointer call_data)
{
  XmDrawingAreaCallbackStruct	*CBst = (XmDrawingAreaCallbackStruct*)call_data;
  XColor color[256], whiteColor, blackColor;
  int x,y,width,height;
  int n,window_count=0;
  long id,id_anim;	/* RJM: handle 64-bit architectures */
  XGCValues values;
  XExposeEvent *expose=(XExposeEvent *)CBst->event;
				
  /* Get x, y, width, and height of exposed area. */
  x=expose->x;
  y=expose->y;
  width=expose->width;
  height=expose->height;

  id=(long)client_data;

  if (xdisplay[id].init==-1) {
				
    /* Check if window is an animation. */
    if (strcmp(xdisplay[id].filename,"Animation")==0) {
      id_anim=animate_list[step_count];
      expose_animation(id,id_anim);
    } else {
      /* Update window image. */
      XPutImage(display,xdisplay[id].top,xdisplay[id].gc,
		xdisplay[id].image,x,y,x,y,width,height);
      expose_zooms(id);		/* Expose dash line on parent of zoom. */
      expose_picks(id);		/* Expose any picks. */
      expose_grid(id);		/* Expose the grid. */
      mark_vel(id,id);
      if (xdisplay[id].axis_flag) draw_axis(id,id); /* Draw axis. */
      show_labels(id,id);	/* Show any labels. */
    }

    return;
  } 

  xdisplay[id].init=-1;	/* Execute once per window. */

  xdisplay[id].top=XtWindow(w);

				/* Get default colormap. */
  for (n=0; n<256; n++) {
    color[n].pixel=n;
    color[n].flags=DoRed|DoGreen|DoBlue;
  }
  XQueryColors(display,DefaultColormap(display,screen),color,256);

				/* Create colormap space. */
  /* NOTE: XCreateColormap window arg2 is only used to determine the screen */
  xdisplay[id].cmap=XCreateColormap(display,xdisplay[id].top,visual,AllocAll);

				/* Creates the colors for the image. */
  for (n=start_color; n<=255; n++) {
    color[n].red=(int)raster_color_red[id][n]*256;
    color[n].green=(int)raster_color_green[id][n]*256;
    color[n].blue=(int)raster_color_blue[id][n]*256;
    color[n].pixel=n;
    color[n].flags=DoRed|DoGreen|DoBlue;
  }

				/* Creates the colors for the picks. */
  for (n=start_pick_color-4; n<=end_pick_color; n++) {
    color[n].red=(int)raster_color_red[id][n]*256;
    color[n].green=(int)raster_color_green[id][n]*256;
    color[n].blue=(int)raster_color_blue[id][n]*256;
    color[n].pixel=n;
    color[n].flags=DoRed|DoGreen|DoBlue;
  }

  XStoreColors(display,xdisplay[id].cmap,color,256);

  /* RJM: use appropriate method for allocating black/white */
  if( !XParseColor( display, xdisplay[id].cmap, "white", &whiteColor ) )
  {	/* don't have white in current cmap */
     n = start_pick_color-5;
     color[n].red = color[n].green = color[n].blue = 255 * 256;
     color[n].pixel=n;
     color[n].flags=DoRed|DoGreen|DoBlue;
     white = n;
     (void)fprintf( stderr, "white reassigned to color slot %d\n", white );
  }

  if( !XParseColor( display, xdisplay[id].cmap, "black", &blackColor ) )
  {	/* don't have black in current cmap */
     n = start_pick_color-6;
     color[n].red = color[n].green = color[n].blue = 0 * 256;
     color[n].pixel=n;
     color[n].flags=DoRed|DoGreen|DoBlue;
     black = n;
     (void)fprintf( stderr, "black reassigned to color slot %d\n", black );
  }

  /* Set the window to the new colormap. */
  XSetWindowColormap(display,XtWindow(xdisplay[id].window),xdisplay[id].cmap);
  /* added sun in case redirect disp from sun to sgi */
#if defined(__sgi) || defined(sun) || defined(linux)	
  /* RJM: drawing area must be set to cmap */
  XSetWindowColormap(display,xdisplay[id].top,xdisplay[id].cmap); 

  /* velocity model builder window */
  if( xdisplay[id].mode == 18 || xdisplay[id].mode == 19 )	
    /* RJM: drawing area must be set to cmap */
    XSetWindowColormap(display,XtWindow(draw_vel),xdisplay[id].cmap); 
  /* velocity painter window */
  else if( xdisplay[id].mode == 28 )	
    /* RJM: drawing area must be set to cmap */
    XSetWindowColormap(display,XtWindow(draw_vel),xdisplay[id].cmap); 
#endif

  values.foreground = black;
  values.background = white;
  values.line_width=0;
  values.font=XLoadFont(display,"9x15");

				/* Create GC for the window. */
  xdisplay[id].gc = XCreateGC( display, xdisplay[id].top, 
			      GCForeground|GCBackground|GCLineWidth|GCFont,
			      &values );

				/* Draw the image. */
  XPutImage(display,xdisplay[id].top,xdisplay[id].gc,
	    xdisplay[id].image,x,y,x,y,width,height);

				/* Create cross hair cursor. */
  set_cursor_color(id,red_cursor_global,green_cursor_global,blue_cursor_global);
		
  /* RJM: THIS ASSUMES these colors are in xdisplay[id].cmap!!! */

  if (xdisplay[id].axis_flag) 
    draw_axis(id,id);

  expose_picks(id);
  expose_image(xdisplay[id].parent);

  contrast_bias(id,id);

				
  /* Check how many windows are being used by xsd. */
  for (n=1; n<MAX_IMAGES-1; n++) if (xdisplay[n].init==-1) window_count++;

  /* Every 40 window pop up a warning.
     DWN disabled at request of USP team, but still check max movies and traces
     if (((window_count%40)==0)&&(window_count!=0)) {
        sprintf(buf,"Your (%d) windows are consuming a lot of memory. It's advisable to delete some.", window_count );
        msg_message(buf);
     }
  */
}


/* 
  This function ask the user 
  for the cursor color.
*/
void cursor_colorCB(Widget w, XtPointer client_data, XtPointer call_data)
{
  long id;	/* RJM: handle 64-bit architectures */
  Widget prompt;
  ClientData *message;

  id=(long)client_data;

  prompt=XmCreatePromptDialogVa(w,"Cursor Color",
				XmNdialogTitle,XMstr("Cursor Color"),
				XmNselectionLabelString,
				XMstr("Input Red, Green, and Blue (0 to 255)"),
				NULL);
  XmRemoveTabGroup(prompt);
  XmAddTabGroup(XmSelectionBoxGetChild(prompt,XmDIALOG_TEXT));
  XmAddTabGroup(XmSelectionBoxGetChild(prompt,XmDIALOG_HELP_BUTTON));

  message=(ClientData *)malloc(sizeof(ClientData));
  message->w=prompt;
  message->id=id;
  XtAddCallback(prompt,XmNokCallback,set_cursor_colorCB,message);
  XtAddCallback(prompt,XmNcancelCallback,destroyCB,prompt);
  XtAddCallback(prompt,XmNhelpCallback,helpCB,"cursor.help");
}


/* 
  This function gets the color 
  from the prompt dialog.
*/
void set_cursor_colorCB(Widget w, XtPointer client_data, XtPointer call_data)
{
  XmSelectionBoxCallbackStruct	*CBst = (XmSelectionBoxCallbackStruct*)call_data;
  ClientData *message=(ClientData *)client_data; 
  int red=0,green=0,blue=0;
  long id, n;	/* RJM: handle 64-bit architectures */
  char *str;

  id=message->id;

  XmStringGetLtoR(CBst->value,XmSTRING_DEFAULT_CHARSET,&str);
  if (str!=NULL)
  {
    sscanf(str,"%d %d %d",&red,&green,&blue);
    XtFree(str);
  }

  if (red>255) red=255;
  if (green>255) green=255;
  if (blue>255) blue=255;

  red=red<<8;
  green=green<<8;
  blue=blue<<8;


				/* Set the cursor color. */
  if (id==0) {
    red_cursor_global=red;
    blue_cursor_global=blue;
    green_cursor_global=green;
    for (n=1; n<MAX_IMAGES; n++)
      if (xdisplay[n].init==-1)
	set_cursor_color((long)n,red,green,blue);
  } else {
    set_cursor_color(id,red,green,blue);
  }

  if( message->w != NULL )
    XtDestroyWidget( message->w );
  free(message);
}


/* 
  This function sets the cursor color.
*/
void set_cursor_color(long id, int red, int green, int blue)
{
  static XColor fg;

  fg.pixel=0;
  fg.red=red;
  fg.green=green;
  fg.blue=blue;

  XDefineCursor(display,xdisplay[id].top,
		XCreatePixmapCursor(display,
			XCreateBitmapFromData(display,xdisplay[id].top,
				(char*)cross1_bits,cross1_width,cross1_height),
			XCreateBitmapFromData(display,xdisplay[id].top,
				(char*)cross1_bits,cross1_width,cross1_height),
		&fg,&fg,cross1_x_hot,cross1_y_hot));
}


/*
  This function draw axis next
  to the seismic data.
*/
void draw_axis(long id, long id_anim)
{
  int width, height, x, x1, y, y1, temp, old_rec, old_x, old_y;
  struct values_struct values;
  XGCValues value_GC;

  if (xdisplay[id].init!=-1  ||  xdisplay[id].hide_flag)
    return;

  width = xdisplay[id_anim].image->width;
  height = xdisplay[id_anim].image->height;

  XClearArea(display,xdisplay[id].top,
             0,height,width+X_AXIS_PAD,height+Y_AXIS_PAD,False);
  XClearArea(display,xdisplay[id].top,
             width,0,width+X_AXIS_PAD,height+Y_AXIS_PAD,False);

  value_GC.line_width=0;
  value_GC.font=XLoadFont(display,"9x15");

  XChangeGC( display, xdisplay[id].gc,
			      GCLineWidth|GCFont,&value_GC );

  XSetForeground(display,xdisplay[id].gc,black);
  XDrawLine(display,xdisplay[id].top,xdisplay[id].gc,
	    0,height,width,height);	/* horizontal axis */
  XDrawLine(display,xdisplay[id].top,xdisplay[id].gc,
	    width,0,width,height);	/* vertical axis */

  if (xdisplay[id_anim].mode==26) {
    digline(0,height,width,height,black,1);
    digline(-1,-1,width,-1,black,1);
    digline(width,0,width,height,black,1);
    digline(-1,0,-1,height,black,1);
  }

  for (x=25; x<width; x+=50) {	/* horizontal axis intermediate tic marks */
    XDrawLine(display,xdisplay[id].top,xdisplay[id].gc,
	      x,height,x,height+5);
    if (xdisplay[id_anim].mode==26)
      digline(x,height,x,height+5,black,1);
  }
  for (x1=5; x1<=45; x1+=5)	/* horizontal axis (9) minor tic marks */
    for (x=x1; x<=width; x+=50) {
      XDrawLine(display,xdisplay[id].top,xdisplay[id].gc,
		x,height,x,height+2);
      if (xdisplay[id_anim].mode==26)
	digline(x,height,x,height+2,black,1);
    }

  for (x=0; x<width; x+=50) {	/* horizontal axis major tic marks/labels */
    XDrawLine(display,xdisplay[id].top,xdisplay[id].gc,
	      x,height,x,height+10);
    if (xdisplay[id_anim].mode==26)
      digline(x,height,x,height+10,black,1);
    temp=(int)(xdisplay[id_anim].pick_x*x);
    values=get_values(temp,0,id_anim);
    values.record=xdisplay[id_anim].axis_units[0]*values.record+
      xdisplay[id_anim].axis_offset[0];
    values.trace=xdisplay[id_anim].axis_units[1]*values.trace+
      xdisplay[id_anim].axis_offset[1];
    values.sample=xdisplay[id_anim].axis_units[2]*values.sample+
      xdisplay[id_anim].axis_offset[2];

    /* DWN add new support for annotating header values */
    if (xdisplay[id_anim].axis_style==False) {

      if (ssamp[id_anim]==esamp[id_anim])
        if( xdisplay[id_anim].swap_flag )
          sprintf(buf,"%4d", values.record);
        else
          sprintf(buf,"%4d", values.trace );
      else
        sprintf(buf,"%4d", values.trace );

      if( x-10 >= 0 ) /* RJM: added to be consistent with expose_labels below */
        XDrawString(display,xdisplay[id].top,xdisplay[id].gc,x-10,height+22,
  		  buf,strlen(buf));
      if (xdisplay[id_anim].mode==26)
        if( x-10 >= 0 ) /* RJM: added, else seg. fault */
          expose_labels(id,buf,"9x15bold",x-10,height+22,black,0,1,True,False);

    } else {
      int i;
      int data1,trace_offset,id_parent,twidth;

      id_parent=xdisplay[id_anim].parent;

          /* Compute twidth. */
      twidth=(((erec[id_parent]-srec[id_parent])/irec[id_parent])+1)*
        (((etrace[id_parent]-strace[id_parent])/itrace[id_parent])+1);

          /* Find trace_offset for trace header. */
      if (xdisplay[id_anim].sort)
        trace_offset=256*((values.record-srec[id_parent])/irec[id_parent])+
          (((values.trace-strace[id_parent])/itrace[id_parent])*
           ceil((float)(erec[id_parent]-srec[id_parent]+1)/
           (float)irec[id_parent])*256);
      else
        trace_offset=256*((values.trace-strace[id_parent])/itrace[id_parent])+
          (((values.record-srec[id_parent])/irec[id_parent])*
           ceil((float)(etrace[id_parent]-strace[id_parent]+1)/
          (float)itrace[id_parent])*256);

          /* Make sure trace_offset is valid. */
      if (trace_offset>256*twidth)
        trace_offset=256*(twidth-1);

      /* DWN cycle through three selected header values for annotation */
      for (i=0; i<3; i++) {

            /* Check if the data was a time slice. */
        if (ssamp[id_parent]!=esamp[id_parent]) {
          if (i==0)
            C_SAVER((int*)(traceH[id_parent]+trace_offset),
                    xdisplay[id_anim].data1,&data1,1);
          if (i==1)
            C_SAVER((int*)(traceH[id_parent]+trace_offset),
                    xdisplay[id_anim].data2,&data1,1);
          if (i==2)
            C_SAVER((int*)(traceH[id_parent]+trace_offset),
                    xdisplay[id_anim].data3,&data1,1);
        } else {
                  /* No trace header available for time slice. */
          data1=0;
        }

        sprintf(buf,"%d",data1);

        if( x-10 >= 0 ) {
          XDrawString(display,xdisplay[id].top,xdisplay[id].gc,
                      x-10,height+(i+1)*22,buf,strlen(buf));
          if (xdisplay[id_anim].mode==26)
            expose_labels(id,buf,"9x15bold",x-10,height+(i+1)*22,
                          black,0,1,True,False);
        }
      }
    }
  }

  /* DWN if in header annotation style, label the header names */
  if (xdisplay[id_anim].axis_style==True) {
      XDrawString(display,xdisplay[id].top,xdisplay[id].gc,width+15,height+22,
                  xdisplay[id_anim].data1,strlen(xdisplay[id_anim].data1));
      XDrawString(display,xdisplay[id].top,xdisplay[id].gc,width+15,height+44,
                  xdisplay[id_anim].data2,strlen(xdisplay[id_anim].data2));
      XDrawString(display,xdisplay[id].top,xdisplay[id].gc,width+15,height+66,
                  xdisplay[id_anim].data3,strlen(xdisplay[id_anim].data3));
      if (xdisplay[id_anim].mode==26) {
        expose_labels(id,xdisplay[id_anim].data1,"9x15bold",width+15,height+22,
                      black,0,1,True,False);
        expose_labels(id,xdisplay[id_anim].data2,"9x15bold",width+15,height+44,
                      black,0,1,True,False);
        expose_labels(id,xdisplay[id_anim].data3,"9x15bold",width+15,height+66,
                      black,0,1,True,False);
      }
  }


/* DWN in header annotation mode, skip this whole block of code */
if (xdisplay[id_anim].axis_style==False) {
  if (srec[id_anim]!=erec[id_anim]) {	/* plot record numbers */
    values=get_values(0,0,id);
    values.record=xdisplay[id_anim].axis_units[0]*values.record+
      xdisplay[id_anim].axis_offset[0];
    values.trace=xdisplay[id_anim].axis_units[1]*values.trace+
      xdisplay[id_anim].axis_offset[1];
    values.sample=xdisplay[id_anim].axis_units[2]*values.sample+
      xdisplay[id_anim].axis_offset[2];
    old_rec=values.record;
    if( xdisplay[id_anim].swap_flag )	/* RJM added */
    {
      old_y=1;
      for (y=1; y<height; y++) {
        temp=(int)(xdisplay[id_anim].pick_y*y);
        values=get_values(0,temp,id);
        values.record=xdisplay[id_anim].axis_units[0]*values.record+
		xdisplay[id_anim].axis_offset[0];
        values.trace=xdisplay[id_anim].axis_units[1]*values.trace+
		xdisplay[id_anim].axis_offset[1];
        values.sample=xdisplay[id_anim].axis_units[2]*values.sample+
		xdisplay[id_anim].axis_offset[2];
        if ((values.record!=old_rec)&&(xdisplay[id_anim].sort==0)) {
	  old_rec=values.record;
	  sprintf(buf,"R %d",values.record);
	  if( y>old_y+30) {
            if( y-10 >= 0 ) /* RJM: added for consistency */
	      XDrawString(display,xdisplay[id].top,xdisplay[id].gc,y-10,width+28,
		        buf,strlen(buf));
	    XDrawLine(display,xdisplay[id].top,xdisplay[id].gc,
		      width+27,y,width+37,y);
	    if (xdisplay[id_anim].mode==26) {
              if( y-10 >= 0 ) /* RJM: added else seg. faults */
	        expose_labels(id,buf,"9x15bold",width+28,y-10,black,0,1,True,False);
	      digline(width+27,y,width+37,y,black,1);
	    }
	    old_y=y;
          }
        } else if ((values.record==1)&&(xdisplay[id_anim].sort==1)) {
	  sprintf(buf,"R %d",values.record);
	  if (y>old_y+30) {
            if( y-10 >= 0 ) /* RJM: added for consistency */
	      XDrawString(display,xdisplay[id].top,xdisplay[id].gc,
			width+28,y-10,buf,strlen(buf));
	    XDrawLine(display,xdisplay[id].top,xdisplay[id].gc,
		      width+27,y,width+37,y);
	    if (xdisplay[id_anim].mode==26) {
              if( y-10 >= 0 ) /* RJM: added else seg. faults */
	        expose_labels(id,buf,"9x15bold",width+28,y-10,black,0,1,True,False);
	      digline(width+27,y,width+37,y,black,1);
	    }
	    old_y=y;
	  }
	}
      }
    }
    else
    {
      old_x=1;
      for (x=1; x<width; x++) {
        temp=(int)(xdisplay[id_anim].pick_x*x);
        values=get_values(temp,0,id);
        values.record=xdisplay[id_anim].axis_units[0]*values.record+
	xdisplay[id_anim].axis_offset[0];
        values.trace=xdisplay[id_anim].axis_units[1]*values.trace+
	xdisplay[id_anim].axis_offset[1];
        values.sample=xdisplay[id_anim].axis_units[2]*values.sample+
	xdisplay[id_anim].axis_offset[2];
        if ((values.record!=old_rec)&&(xdisplay[id_anim].sort==0)) {
	  old_rec=values.record;
	  sprintf(buf,"R   %d",values.record);
	  if (x>old_x+30) {
            if( x-10 >= 0 ) /* RJM: added for consistency */
	      XDrawString(display,xdisplay[id].top,xdisplay[id].gc,x-10,height+47,
		        buf,strlen(buf));
	    XDrawLine(display,xdisplay[id].top,xdisplay[id].gc,
		      x,height+27,x,height+37);
	    if (xdisplay[id_anim].mode==26) {
              if( x-10 >= 0 ) /* RJM: added else seg. faults */
	        expose_labels(id,buf,"9x15bold",x-10,height+47,black,0,1,True,False);
	      digline(x,height+27,x,height+37,black,1);
	    }
	    old_x=x;
	  }
        } else if ((values.record==1)&&(xdisplay[id_anim].sort==1)) {
	  sprintf(buf,"R %d",values.record);
	  if (x>old_x+30) {
            if( x-10 >= 0 ) /* RJM: added for consistency */
	      XDrawString(display,xdisplay[id].top,xdisplay[id].gc,x-10,
			height+47,buf,strlen(buf));
	    XDrawLine(display,xdisplay[id].top,xdisplay[id].gc,
		      x,height+27,x,height+37);
	    if (xdisplay[id_anim].mode==26) {
              if( x-10 >= 0 ) /* RJM: added else seg. faults */
	        expose_labels(id,buf,"9x15bold",x-10,height+47,black,0,1,True,False);
	      digline(x,height+27,x,height+37,black,1);
	    }
	    old_x=x;
	  }
	}
      }
    }
  }
}

  for (y=25; y<height; y+=50) {	/* vertical axis intermediate tic marks */
    XDrawLine(display,xdisplay[id].top,xdisplay[id].gc,
	      width,y,width+5,y);
    if (xdisplay[id_anim].mode==26) {
      digline(width,y,width+5,y,black,1);
      digline(-1,y,-5,y,black,1);
    }
  }

  for (y1=5; y1<=45; y1+=5)	/* vertical axis (9) minor tic marks */
    for (y=y1; y<height; y+=50) {
      XDrawLine(display,xdisplay[id].top,xdisplay[id].gc,
	      width,y,width+2,y);
      if (xdisplay[id_anim].mode==26) {
	digline(width,y,width+2,y,black,1);
	digline(-1,y,-3,y,black,1);
      }
    }

  XDrawLine(display,xdisplay[id].top,xdisplay[id].gc,
	    width,0,width+10,0);	/* vertical axis topmost major tic */
  if (xdisplay[id_anim].mode==26) {
    digline(width,0,width+10,0,black,1);
    digline(-1,0,-11,0,black,1);
  }
  values=get_values(0,0,id);
  values.record=xdisplay[id_anim].axis_units[0]*values.record+
    xdisplay[id_anim].axis_offset[0];
  values.trace=xdisplay[id_anim].axis_units[1]*values.trace+
    xdisplay[id_anim].axis_offset[1];
  values.sample=xdisplay[id_anim].axis_units[2]*values.sample+
    xdisplay[id_anim].axis_offset[2];
  if( ssamp[id_anim]==esamp[id_anim]  && !xdisplay[id_anim].swap_flag )
    sprintf(buf,"%-4d", values.record);
  else
    sprintf(buf,"%-4d", values.sample);
  XDrawString(display,xdisplay[id].top,xdisplay[id].gc,width+AxisLabelXoffset,
		AxisLabelXoffset, buf,strlen(buf));
  if (xdisplay[id_anim].mode==26) {
    expose_labels(id,buf,"9x15bold",width+AxisLabelXoffset,AxisLabelXoffset,black,
		0,0,True,False);
    expose_labels(id,buf,"9x15bold",-AxisLabelXoffset,AxisLabelXoffset,black,
		2,0,True,False);
  }
  for (y=50; y<height; y+=50) {	/* vertical axis major tic marks/labels */
    XDrawLine(display,xdisplay[id].top,xdisplay[id].gc,
	      width,y,width+10,y);
    if (xdisplay[id_anim].mode==26) {
      digline(width,y,width+10,y,black,1);
      digline(-1,y,-11,y,black,1);
    }
    temp=(int)(xdisplay[id_anim].pick_y*y);
    values=get_values(0,temp,id);
    values.record=xdisplay[id_anim].axis_units[0]*values.record+
      xdisplay[id_anim].axis_offset[0];
    values.trace=xdisplay[id_anim].axis_units[1]*values.trace+
      xdisplay[id_anim].axis_offset[1];
    values.sample=xdisplay[id_anim].axis_units[2]*values.sample+
      xdisplay[id_anim].axis_offset[2];
    if( ssamp[id_anim]==esamp[id_anim]  && !xdisplay[id_anim].swap_flag )
      sprintf(buf,"%-4d", values.record);
    else
      sprintf(buf,"%-4d", values.sample);
    XDrawString(display,xdisplay[id].top,xdisplay[id].gc,width+AxisLabelXoffset,
		y+5,buf,strlen(buf));
    if (xdisplay[id_anim].mode==26) {
      expose_labels(id,buf,"9x15bold",width+AxisLabelXoffset,y+5,black,
		0,0,True,False);
      expose_labels(id,buf,"9x15bold",-AxisLabelXoffset,y+5,black,2,0,True,False);
    }
  }

  value_GC.font=XLoadFont(display,"9x15");

  XChangeGC( display, xdisplay[id].gc,GCFont,&value_GC );
}


/*
  This function redraws the whole
  raster image.
*/
void expose_image(long id)
{
  long id_anim;
  Dimension width,height;
  XWindowAttributes win_attr;

				/* Check if window id exists. */
  if (xdisplay[id].init!=-1)
    return;

				/* Check if expose is in 
				   an animation window. */
  if (strcmp(xdisplay[id].filename,"Animation")==0) {
    id_anim=animate_list[step_count];
    expose_animation(id,id_anim);
    return;
  }
				/* Check if window id is hidden. */
  if (xdisplay[id].hide_flag)
    return;

  if ((xdisplay[id].mode==18)||(xdisplay[id].mode==19))
    expand_vel(id);

				/* Find out the part of the image
				   that is being display (scrolling) */
  XGetWindowAttributes(display,xdisplay[id].top,&win_attr);

  XtVaGetValues(xdisplay[id].scroll,
		XmNwidth,&width,
		XmNheight,&height,
		NULL);

  XPutImage(display,xdisplay[id].top,xdisplay[id].gc,
	    xdisplay[id].image,abs(win_attr.x),abs(win_attr.y),
	    abs(win_attr.x),abs(win_attr.y),width,height);

  show_labels(id,id);
  expose_picks(id);
  expose_zooms(id);
  expose_grid(id);
  mark_vel(id,id);

  if (xdisplay[id].axis_flag)	/* RJM: added */
    draw_axis(id, id);
  show_labels(id,id);		/* RJM: added */
}



/*
  This function redraws the images
  in all the windows (with the same
  parent id).
*/
void expose_all_images(void)
{
  int n;
  long id_anim;	/* RJM: handle 64-bit architectures */

				/* Loop through all the window ids. */
  for (n=0; n<MAX_IMAGES; n++)
    if ((xdisplay[n].init==-1)&&(!xdisplay[n].hide_flag)) {
      id_anim=n;
      if (strcmp(xdisplay[n].filename,"Animation")==0)
	id_anim=animate_list[step_count];
      XPutImage(display,xdisplay[n].top,xdisplay[n].gc,
		xdisplay[id_anim].image,0,0,0,0,
		xdisplay[id_anim].image->width,
		xdisplay[id_anim].image->height);
      show_labels(n,id_anim);
      expose_zooms(n);
      expose_grid(n);
      mark_vel(n,id_anim);
    }
}



/*
  This function draws the labels
  for the particular window id.
*/
void show_labels(long id, long id_anim)
{
  int n;
  XGCValues value_GC;

  if (xdisplay[id_anim].init!=-1
  ||  xdisplay[id_anim].hide_flag
  ||  xdisplay[id_anim].text_flag)
    return;
				/* Loop through the label ids. */
  for (n=0; n<MAX_LABELS; n++)
				/* If a label exits. */
    if ((labels[n].id==id)&&(labels[n].init==-1)) {
      value_GC.line_width=0;
      value_GC.font=XLoadFont(display,labels[n].font);
      
      XChangeGC( display, xdisplay[id].gc,
				  GCLineWidth|GCFont,&value_GC );
      XSetForeground(display,xdisplay[id].gc,labels[n].color);
      if (labels[n].type==1)
	XSetBackground(display,xdisplay[id].gc,black);
      if (labels[n].type==2)
	XSetBackground(display,xdisplay[id].gc,white);
      if (labels[n].type==0) {
	XDrawString(display,xdisplay[id].top,xdisplay[id].gc,
		    labels[n].x,labels[n].y,labels[n].string,
		    strlen(labels[n].string));
	if (xdisplay[id_anim].mode==26)
	  expose_labels(id,labels[n].string,labels[n].font,
			labels[n].x,labels[n].y,labels[n].color,0,0,True,False);
      } else {
	XDrawImageString(display,xdisplay[id].top,xdisplay[id].gc,
			 labels[n].x,labels[n].y,labels[n].string,
			 strlen(labels[n].string));
	if (xdisplay[id_anim].mode==26)
	  if (labels[n].type==1)
	    expose_labels_image(id,labels[n].string,labels[n].font,
				labels[n].x,labels[n].y,labels[n].color,black,
				0,0,True,False);
	  else
	    expose_labels_image(id,labels[n].string,labels[n].font,
				labels[n].x,labels[n].y,labels[n].color,white,
				0,0,True,False);
      }
    }
  value_GC.font=XLoadFont(display,"9x15");
  
  XChangeGC( display, xdisplay[id].gc,GCFont,&value_GC );
}



  
/*
  This function redraws all the picks
  for window id.
*/
void expose_picks(long id)
{
  if (xdisplay[id].init!=-1
  ||  xdisplay[id].hide_flag)
    return;

  expose_picks_animation(id,id); 
}



/*
  This function draws the boxes
  of the parent window indicating
  what portion of the window was
  zoomed.
*/
static void expose_zooms(long id)
{
  int x1,y1,x2,y2;
  long id1;	/* RJM: handle 64-bit architectures */
  Boolean old_record_flag;
  struct location_struct location;

  if (xdisplay[id].init!=-1
  ||  xdisplay[id].hide_flag)
    return;

				/* Reset record number so the correct x
                                   and y will be return for zooming. */
  old_record_flag=xdisplay[id].record_flag;
  xdisplay[id].record_flag=True;

  if (xdisplay[id].parent==id) {
    XSetForeground(display,xdisplay[id].gc,black);
    XSetLineAttributes(display,xdisplay[id].gc,0,
		       LineOnOffDash,CapNotLast,JoinRound);
    for (id1=0; id1<MAX_IMAGES; id1++)
      if ((xdisplay[id1].parent==id)&&(id1!=id)&&(xdisplay[id1].init==-1)) {
	location=get_location(srec[id1],strace[id1],ssamp[id1],id);
	x1=(int)location.x/xdisplay[id].pick_x;
	y1=(int)location.y/xdisplay[id].pick_y;
	location=get_location(erec[id1],etrace[id1],esamp[id1],id);
	x2=(int)location.x/xdisplay[id].pick_x;
	y2=(int)location.y/xdisplay[id].pick_y;
	XDrawRectangle(display,xdisplay[id].top,xdisplay[id].gc,
		  x1,y1,(x2-x1),(y2-y1));
      }
    XSetLineAttributes(display,xdisplay[id].gc,line_width,
		       LineSolid,CapNotLast,JoinRound);
  }
                                /* Restore the record flag. */
  xdisplay[id].record_flag=old_record_flag;
}



/*
  This function redraws all the picks
  for window id for animation.
*/
void expose_picks_animation(long id, long id_anim)
{
  int n;
  long id_parent;	/* RJM: handle 64-bit architectures */

  if (xdisplay[id_anim].init==0
  ||  xdisplay[id].init!=-1
  ||  xdisplay[id].hide_flag)
    return;

  id_parent=xdisplay[id_anim].parent;

  show_labels(id,id_anim);

				/* Loop throught the pick ids. */
  for (n=0; n<MAX_PICKS; n++)
				/* Check if pick exists and is visible. */
    if (picks[n].init==-1) {
      if (((visible_segment(id_parent,picks[n].group)==0)&&
	   !xdisplay[id].overlay_flag)||
	  ((visible_segment(id_parent,picks[n].group))&&
	   !xdisplay[id].segment_flag)) {
	XSetForeground(display,xdisplay[id].gc,picks[n].color);
	draw_label(n,id,id_anim);

	draw_picks(n,id,id_anim);
      }
    }
  
  if (xdisplay[id_anim].axis_flag) 
    draw_axis(id,id_anim);
}


/*
  This function redraws all but one picks
  for window id.
*/
void expose_some_picks(long id,long id_anim,int current_pick)
{
  int n;
  long id_parent;	/* RJM: handle 64-bit architectures */

  if (xdisplay[id].init!=-1
  ||  xdisplay[id].hide_flag)
    return;

  XPutImage(display,xdisplay[id].top,xdisplay[id].gc,
	    xdisplay[id_anim].image,0,0,0,0,xdisplay[id_anim].image->width,
	    xdisplay[id_anim].image->height);
  
  show_labels(id,id_anim);
  expose_zooms(id);

  id_parent=xdisplay[id_anim].parent;

				/* Loop through all the pick ids. */
  for (n=0; n<MAX_PICKS; n++)
    if (picks[n].init==-1) {
      if (((visible_segment(id_parent,picks[n].group)==0)&&
	   !xdisplay[id_anim].overlay_flag)||
	  ((visible_segment(id_parent,picks[n].group))&&
	   !xdisplay[id_anim].segment_flag)) {
	XSetForeground(display,xdisplay[id].gc,picks[n].color);
	draw_label(n,id,id_anim);

				/* Check if pick id is the current one. */
	if (n!=current_pick)
	  draw_picks(n,id,id_anim);
      }
    }
  if (xdisplay[id_anim].axis_flag) 
    draw_axis(id,id_anim);
}



/*
  This function redraws all the picks.
*/
void expose_all_picks(void)
{
  int n;
  long id;	/* RJM: handle 64-bit architectures */

				/* Loop through the window ids. */
  for (id=0; id<MAX_IMAGES; id++)  {
    if (strcmp(xdisplay[id].filename,"Animation")!=0)
      if ((xdisplay[id].init==-1)&&!xdisplay[id].hide_flag) {
				/* Loop through the pick ids. */
	for (n=0; n<MAX_PICKS; n++)
	  if (picks[n].init==-1) {
	    if (((visible_segment(id,picks[n].group)==0)&&
		 !xdisplay[id].overlay_flag)||
		((visible_segment(id,picks[n].group))&&
		 !xdisplay[id].segment_flag)) {
	      XSetForeground(display,xdisplay[id].gc,picks[n].color);
	      draw_label(n,id,id);
	      
	      draw_picks(n,id,id);
	    }
	  }
	if (xdisplay[id].axis_flag) 
	  draw_axis(id,id);
      }
  }
}



/*
  This function draws the grid.
*/
void expose_grid(long id)
{
  int x,y;

  if (strcmp(xdisplay[id].filename,"Animation")!=0) {
    if (xdisplay[id].grid_flag) {
      if (xdisplay[id].solid_flag) {
	XSetLineAttributes(display,xdisplay[id].gc,0,
			   LineSolid,CapNotLast,JoinRound);
      } else {
	XSetLineAttributes(display,xdisplay[id].gc,0,
			   LineOnOffDash,CapNotLast,JoinRound);
      }
      XSetForeground(display,xdisplay[id].gc,xdisplay[id].grid_color);
      
      for (x=0; x<xdisplay[id].image->width; x+=xdisplay[id].grid_x) {
	XDrawLine(display,xdisplay[id].top,xdisplay[id].gc,
		  x,0,x,xdisplay[id].image->height);
	if (xdisplay[id].mode==26)
	  digline(x,0,x,xdisplay[id].image->height,xdisplay[id].grid_color,1);
      }
      
      for (y=0; y<xdisplay[id].image->height; y+=xdisplay[id].grid_y) {
	XDrawLine(display,xdisplay[id].top,xdisplay[id].gc,
		  0,y,xdisplay[id].image->width,y);
	if (xdisplay[id].mode==26)
	  digline(0,y,xdisplay[id].image->width,y,xdisplay[id].grid_color,1);
      }
      
      XSetLineAttributes(display,xdisplay[id].gc,0,
			 LineSolid,CapNotLast,JoinRound);
    }
  }  
  XSetForeground(display,xdisplay[id].gc,black);
}



/*
  This function draw the picks.
*/
void draw_picks(int n, long id, long id_anim)
{
  int j,m,x1,y1,x2,y2,y,size,size1;
  Boolean flag=False;
  int x_model1,y_model1,x_model2,y_model2,tmp_width;
  struct location_struct location;

/* fprintf( stderr, "display.c: draw_picks:\n" ); */
  XSetLineAttributes(display,xdisplay[id].gc,line_width,
                     LineSolid,CapNotLast,JoinRound);

				/* Compute size for nodes. */
  if (line_width<=1)
    size=4;
  else
    size=4+line_width;
  size1=size/2;

				/* Loop through pick ids. */
/*if(picks[n].id != id_anim )
fprintf( stderr,"NOTE: draw_picks: picks[n].id(%d) != id_anim(%d)\n", picks[n].id, id_anim);
*/
/* fprintf( stderr, "display.c: draw_picks: picks[%d].size=%d\n", n, picks[n].size); */
  for (m=1; m<picks[n].size; m++) {
    /* get location of pick m-1 */
				/* If window is a sort then place the 
				   picks in the middle of the sort panel. */
    if ((xdisplay[id_anim].sort)&&(xdisplay[id_anim].middle)) {
/*    location=get_location(((erec[id_anim]-srec[id_anim]+1)/2)+srec[id_anim],*(picks[n].trace+m-1),*(picks[n].sample+m-1),id_anim); RJM */
      location=get_location(((erec[id_anim]-srec[id_anim]+1)/2)+srec[id_anim],picks[n].trace[m-1],picks[n].sample[m-1],id_anim?id_anim:picks[n].id);

    } else {
				/* Check if record number is honored. */
      if (xdisplay[id].record_flag) {
/*	location=get_location(*(picks[n].record+m-1),*(picks[n].trace+m-1),*(picks[n].sample+m-1),id_anim); RJM */
	location=get_location(picks[n].record[m-1],picks[n].trace[m-1],picks[n].sample[m-1],id_anim?id_anim:picks[n].id);
/* fprintf( stderr, "display.c: draw_picks: pick %d, record honored, location=%d,%d\n", m-1, location.x, location.y); */
      } else {
/*	location=get_location(srec[id_anim],*(picks[n].trace+m-1),*(picks[n].sample+m-1),id_anim); RJM */
	location=get_location(srec[id_anim],picks[n].trace[m-1],picks[n].sample[m-1],id_anim?id_anim:picks[n].id);
/* fprintf( stderr, "display.c: draw_picks: pick %d, record not honored, location=%d,%d\n", m, location.x, location.y); */
      }
    }

				/* Scale x and y according to zoom. */
    x1=(int)location.x/xdisplay[id_anim].pick_x;
    y1=(int)location.y/xdisplay[id_anim].pick_y;
/* fprintf( stderr, "display.c: draw_picks: x1,y1= %d,%d\n", x1, y1 ); */

    
    /* get location of pick m */
				/* If window is a sort then place the 
				   picks in the middle of the sort panel. */
    if ((xdisplay[id_anim].sort)&&(xdisplay[id_anim].middle)) {
/*    location=get_location(((erec[id_anim]-srec[id_anim])/2)+srec[id_anim],*(picks[n].trace+m),*(picks[n].sample+m),id_anim); RJM */
      location=get_location(((erec[id_anim]-srec[id_anim])/2)+srec[id_anim],picks[n].trace[m],picks[n].sample[m],id_anim?id_anim:picks[n].id);
    } else {
				/* Check if record number is honored. */
      if (xdisplay[id].record_flag) {
/*	location=get_location(*(picks[n].record+m),*(picks[n].trace+m),*(picks[n].sample+m),id_anim); RJM */
	location=get_location(picks[n].record[m],picks[n].trace[m],picks[n].sample[m],id_anim?id_anim:picks[n].id);
/* fprintf( stderr, "display.c: draw_picks: pick %d, record honored, location=%d,%d\n", m, location.x, location.y); */
      } else {
/*	location=get_location(srec[id_anim],*(picks[n].trace+m),*(picks[n].sample+m),id_anim); RJM */
	location=get_location(srec[id_anim],picks[n].trace[m],picks[n].sample[m],id_anim?id_anim:picks[n].id);
/* fprintf( stderr, "display.c: draw_picks: pick %d, record not honored, location=%d,%d\n", m, location.x, location.y); */
      }
    }

				/* Scale x and y according to zoom. */
    x2=(int)location.x/xdisplay[id_anim].pick_x;
    y2=(int)location.y/xdisplay[id_anim].pick_y;
/* fprintf( stderr, "display.c: draw_picks: x2,y2= %d,%d\n", x2, y2 ); */

				/* Check if record number is honored. */
    if (xdisplay[id].record_flag) {
      flag=True;
				/* Loop through record number to see 
				   if the picks are actually visible. */
/* fprintf( stderr, "display.c: draw_picks: checking if record of pick matches record range\n" ); */
/* fprintf( stderr, "display.c: draw_picks: srec[id_anim]=%d, erec[id_anim]=%d, irec[id_anim]=%d\n", srec[id_anim], erec[id_anim], irec[id_anim] ); */
      for (j=srec[id_anim]; j<=erec[id_anim]; j+=irec[id_anim]) 
/*	if (*(picks[n].record+m)==j) RJM */
	if (picks[n].record[m]==j)
	  flag=False;
    }

    if ((!flag)||(xdisplay[id_anim].sort)) {
      if (((xdisplay[id_anim].mode==18)||(xdisplay[id_anim].mode==19))&&
	  (picks[n].group==0)) {
				/* If in model mode draw digital line. */
	x_model1=x1/fill_skip; y_model1=y1/fill_skip;
	x_model2=x2/fill_skip; y_model2=y2/fill_skip;
	if ((x_model1<model_width)||(x_model2<model_width)) {
	  if (x_model1>model_width-1) x_model1=model_width-1;
	  if (y_model1>model_height-1) y_model1=model_height-1;
	  if (x_model2>model_width-1) x_model2=model_width-1;
	  if (y_model2>model_height-1) y_model2=model_height-1;
	  digline(x_model1,y_model1,x_model2,y_model2,black,0);
	}
      } else
	if (xdisplay[id].lines_flag) { /* Draw segment. */
/* fprintf( stderr, "display.c: draw_picks: drawing line from x1 to x2\n" ); */
	  XDrawLine(display,xdisplay[id].top,xdisplay[id].gc,
		    x1,y1,x2,y2);
	  if (xdisplay[id_anim].mode==26) {
	    digline(x1,y1,x2,y2,picks[n].color,1);
	    for (tmp_width=1; tmp_width<line_width; tmp_width++) { 
	      if (abs(y2-y1)>abs(x2-x1))
		digline(x1+tmp_width,y1,x2+tmp_width,y2,picks[n].color,1);
	      else
		digline(x1,y1+tmp_width,x2,y2+tmp_width,picks[n].color,1);
	    }
	  }
	} else {		/* Draw atleast the nodes. */
	  XFillArc(display,xdisplay[id].top,xdisplay[id].gc,
		   x1-size1,y1-size1,size,size,0,23040);
	  if (xdisplay[id_anim].mode==26) {
	    pixelprint(x1,y1,picks[n].color);
	    if (y1>0)
	      pixelprint(x1,y1-1,picks[n].color);
	    if (y1<xdisplay[id_anim].image->height)
	      pixelprint(x1,y1+1,picks[n].color);
	    if (x1>0)
	      pixelprint(x1-1,y1,picks[n].color);
	    if (x1<xdisplay[id_anim].image->width)
	      pixelprint(x1+1,y1,picks[n].color);
	  }
	}
      
				/* If in edit mode draw nodes. */
      if ((xdisplay[id].mode>1)&&(picks[n].group==0)&&
	  (xdisplay[id].mode!=18)&&(xdisplay[id].mode!=19))
	XFillArc(display,xdisplay[id].top,xdisplay[id].gc,
		 x1-size1,y1-size1,size,size,0,23040);
    } else {
      flag=False;
    }
  }

				/* Check if record number is honored. */
  if (xdisplay[id].record_flag) {
    flag=True;
				/* Loop through record number to see 
				   if the picks are actually visible. */
    for (j=srec[id_anim]; j<=erec[id_anim]; j+=irec[id_anim]) 
/*    if (*(picks[n].record+m-1)==j) RJM */
      if (picks[n].record[m-1]==j)
	flag=False;
  }
  if (((!flag)||(xdisplay[id_anim].sort))&&(picks[n].size!=1)) {
				/* If in edit mode draw nodes. */
    if ((xdisplay[id].mode>1)&&(picks[n].group==0)&&
	(xdisplay[id].mode!=18)&&(xdisplay[id].mode!=19))
      XFillArc(display,xdisplay[id].top,xdisplay[id].gc,
	       x2-size1,y2-size1,size,size,0,23040);

    if (!xdisplay[id_anim].lines_flag) /* Draw atleast the nodes. */
      XFillArc(display,xdisplay[id].top,xdisplay[id].gc,
	       x2-size1,y2-size1,size,size,0,23040);

    if (xdisplay[id_anim].mode==26) {
      pixelprint(x2,y2,picks[n].color);
      if (y2>0)
	pixelprint(x2,y2-1,picks[n].color);
      if (y2<xdisplay[id_anim].image->height)
	pixelprint(x2,y2+1,picks[n].color);
      if (x2>0)
	pixelprint(x2-1,y2,picks[n].color);
      if (x2<xdisplay[id_anim].image->width)
	pixelprint(x2+1,y2,picks[n].color);
    }
  }

  if (picks[n].size==1) {
    m=1;
    if ((xdisplay[id_anim].sort)&&(xdisplay[id_anim].middle)) {
/*    location=get_location(((erec[id_anim]-srec[id_anim]+1)/2)+srec[id_anim],*(picks[n].trace+m-1),*(picks[n].sample+m-1),id_anim); RJM */
      location=get_location(((erec[id_anim]-srec[id_anim]+1)/2)+srec[id_anim],picks[n].trace[m-1],picks[n].sample[m-1],id_anim?id_anim:picks[n].id);
    } else {
				/* Check if record number is honored. */
      if (xdisplay[id].record_flag)
      {
/*	location=get_location(*(picks[n].record+m-1),*(picks[n].trace+m-1),*(picks[n].sample+m-1),id_anim); RJM */
	location=get_location(picks[n].record[m-1],picks[n].trace[m-1],picks[n].sample[m-1],id_anim?id_anim:picks[n].id);
      }
      else
      {
/*	location=get_location(srec[id_anim],*(picks[n].trace+m-1),*(picks[n].sample+m-1),id_anim); RJM */
	location=get_location(srec[id_anim],picks[n].trace[m-1],picks[n].sample[m-1],id_anim?id_anim:picks[n].id);
      }
    }

				/* Scale x and y according to zoom. */
    x1=(int)location.x/xdisplay[id_anim].pick_x;
    y1=(int)location.y/xdisplay[id_anim].pick_y;

    XFillArc(display,xdisplay[id].top,xdisplay[id].gc,
	     x1-size1,y1-size1,size,size,0,23040);

    if (xdisplay[id_anim].mode==26) {
      pixelprint(x1,y1,picks[n].color);
      if (y1>0)
	pixelprint(x1,y1-1,picks[n].color);
      if (y1<xdisplay[id_anim].image->height)
	pixelprint(x1,y1+1,picks[n].color);
      if (x1>0)
	pixelprint(x1-1,y1,picks[n].color);
      if (x1<xdisplay[id_anim].image->width)
	pixelprint(x1+1,y1,picks[n].color);
    }

    x2=x1-size1; y2=y1-size1;
    x1+=size1; y1+=size1;
    if (x2<0) x2=0;
    if (y2<0) y2=0;
    if (x1>=xdisplay[id_anim].image->width)
      x1=xdisplay[id_anim].image->width-1;
    if (y1>=xdisplay[id_anim].image->height) 
      y1=xdisplay[id_anim].image->height-1;
    if (xdisplay[id_anim].mode==26) {
/*      digline(x1,y1,x2,y1,picks[n].color,1);
      digline(x2,y1,x2,y2,picks[n].color,1);
      digline(x2,y2,x1,y2,picks[n].color,1);
      digline(x1,y2,x1,y1,picks[n].color,1); */
      for (y=y2; y<y1; y++)                  /* Solid box. */
	digline(x1,y,x2,y,picks[n].color,1);
    }
  }

				/* Reset line attributes. */
  XSetLineAttributes(display,xdisplay[id].gc,0,
                     LineSolid,CapNotLast,JoinRound);
}


/*
  Mark velocity reference points.
*/
static void mark_vel(long id, long id_anim)
{
  int x1,y1,x2,y2;
  int current_index,x,y,x_tmp;

				/* If in the animation with then
				   check if picks are hidden. */
  if (xdisplay[id].segment_flag)
    return;
  
  XSetForeground(display,xdisplay[id].gc,black);

  if ((xdisplay[id_anim].mode>1)&&(xdisplay[id_anim].mode!=18)&&(xdisplay[id_anim].mode!=19))
    for (current_index=0; current_index<v_index; current_index++) {
      if (xdisplay[id_anim].flip_flag) {
  	x_tmp=(int)((etrace[id_anim]-strace[id_anim])/itrace[id_anim])*itrace[id_anim]+1;
	x=x_tmp-(model_body[current_index].x/xdisplay[id_anim].pick_x) ;
      } else {
	x=model_body[current_index].x/xdisplay[id_anim].pick_x;
      }
      y=model_body[current_index].y/xdisplay[id_anim].pick_y;
      
      XDrawLine(display,xdisplay[id].top,xdisplay[id].gc,
		x-5,y,x+5,y);
      XDrawLine(display,xdisplay[id].top,xdisplay[id].gc,
		x,y-5,x,y+5);
    }
  
  if (xdisplay[id_anim].mode==18) {
    if (show_parameters)
      for (current_index=0; current_index<v_index; current_index++) {
	if (xdisplay[id_anim].flip_flag) {
  	  x_tmp=(int)((etrace[id_anim]-strace[id_anim])/itrace[id_anim])*itrace[id_anim]+1;
	  x=x_tmp-(model_body[current_index].x/xdisplay[id_anim].pick_x) ;
	} else {
	  x=model_body[current_index].x/xdisplay[id_anim].pick_x;
	}
	y=model_body[current_index].y/xdisplay[id_anim].pick_y;
	
	XDrawLine(display,xdisplay[id].top,xdisplay[id].gc,
		  x-5,y,x+5,y);
	XDrawLine(display,xdisplay[id].top,xdisplay[id].gc,
		  x,y-5,x,y+5);
      }
    if (body_flag) {
      for (current_index=0; current_index<v_index; current_index++) 
	if (XtIsManaged(v_parameter[current_index*NUM_PARAMETER])) {
	  if (xdisplay[id_anim].flip_flag) {
  	    x_tmp=(int)((etrace[id_anim]-strace[id_anim])/itrace[id_anim])*itrace[id_anim]+1;
	    x1=x_tmp-(model_body[current_index].x1/xdisplay[id_anim].pick_x);
	  } else {
	    x1=model_body[current_index].x1/xdisplay[id_anim].pick_x;
	  }
	  y1=model_body[current_index].y1/xdisplay[id_anim].pick_y;
	  if (xdisplay[id_anim].flip_flag) {
  	    x_tmp=(int)((etrace[id_anim]-strace[id_anim])/itrace[id_anim])*itrace[id_anim]+1;
	    x2=x_tmp-(model_body[current_index].x2/xdisplay[id_anim].pick_x);
	  } else {
	    x2=model_body[current_index].x2/xdisplay[id_anim].pick_x;
	  }
	  y2=model_body[current_index].y2/xdisplay[id_anim].pick_y;
	  
	  if (((model_body[current_index].x1>0)||
	      (model_body[current_index].y1>0))&&
	      ((model_body[current_index].x2>0)||
	       (model_body[current_index].y2>0))) {
	    XDrawLine(display,xdisplay[id].top,xdisplay[id].gc,
		      x1,y1,x2,y2);
	  }
	  if ((model_body[current_index].x1>0)||
	      (model_body[current_index].y1>0)) {
	    XDrawArc(display,xdisplay[id].top,xdisplay[id].gc,
		     x1-4,y1-4,8,8,0,23040);
	  }
	  if ((model_body[current_index].x2>0)||
	      (model_body[current_index].y2>0)) {
	    XDrawArc(display,xdisplay[id].top,xdisplay[id].gc,
		     x2-4,y2-4,8,8,0,23040);
	  }
	  if (xdisplay[id_anim].flip_flag) {
  	    x_tmp=(int)((etrace[id_anim]-strace[id_anim])/itrace[id_anim])*itrace[id_anim]+1;
	    x1=x_tmp-(model_body[current_index].x/xdisplay[id_anim].pick_x);
	  } else {
	    x1=model_body[current_index].x/xdisplay[id_anim].pick_x;
	  }
	  y1=model_body[current_index].y/xdisplay[id_anim].pick_y;
	  
	  XDrawLine(display,xdisplay[id].top,xdisplay[id].gc,
		    x1-5,y1,x1+5,y1);
	  XDrawLine(display,xdisplay[id].top,xdisplay[id].gc,
		    x1,y1-5,x1,y1+5);
	}
    }
  }
}


/*
  This function draws the labels
  for the picks.
*/
static void draw_label(int n, long id, long id_anim)
{
  int x1,y1;
  struct location_struct values;

  if ((!xdisplay[id_anim].label_flag)&&(!xdisplay[id_anim].number_flag))
    return;

				/* If in the animation with then
				   check if picks are hidden. */
  if (xdisplay[id].segment_flag)
    return;

				/* Get location of pick n. */
/*if(picks[n].id != id_anim )
fprintf( stderr,"NOTE: draw_label: picks[n].id(%d) != id_anim(%d)\n", picks[n].id, id_anim);
*/
/*values=get_location(*(picks[n].record),*(picks[n].trace),*(picks[n].sample),id_anim); RJM */
  values=get_location(picks[n].record[0],picks[n].trace[0],picks[n].sample[0],id_anim?id_anim:picks[n].id);

  x1=(int)(values.x+picks[n].offset_x)/xdisplay[id_anim].pick_x;
  y1=(int)(values.y+picks[n].offset_y)/xdisplay[id_anim].pick_y;

#ifdef DEBUG
  fprintf( stderr, "n=%d, picks[n].record[0]=%d, picks[n].trace[0]=%d, picks[n].sample[0]=%d, values.x,y=%d,%d, x1=%d, y1=%d\n", picks[n].record[0], picks[n].trace[0], picks[n].sample[0], values.x, values.y, x1, y1 );
#endif

  if( x1+AxisLabelXoffset < 0		/* RJM: added */
  ||  y1 < 0
  ||  x1 > xdisplay[id_anim].image->width
  ||  y1 > xdisplay[id_anim].image->height )
    return;

				/* Check if label is too close 
				   to the right border. */
  if (x1>(xdisplay[id_anim].image->width-30)) 
    x1=xdisplay[id_anim].image->width-30;

				/* Check if label is too close
				   to the top of the border. */
  if ((y1-20)<0)
    y1=20;

				/* Check the label type. */
  if ((xdisplay[id_anim].label_flag)&&(xdisplay[id_anim].number_flag)) {
    sprintf(buf,"(%d) %s",n+1,picks[n].label);
  } else {
    if (xdisplay[id_anim].label_flag) {
      sprintf(buf,"%s",picks[n].label);
    } else {
      if (xdisplay[id_anim].number_flag) {
	sprintf(buf,"(%d)",n+1);
      }
    }
  }

				/* Draw the label. */
  if (xdisplay[id_anim].mode==26) {
    expose_labels(id,buf,"9x15bold",x1,y1,picks[n].color,0,0,True,False);
  } else {
    XDrawString(display,xdisplay[id].top,xdisplay[id].gc,x1,y1,
		buf,strlen(buf));
  }
}



/*
  This function draw the pixmap
  and update the picks when in
  the edit pick mode of xsd.
*/
void draw_pix(long id, long id_anim, int current_segment)
{
  int j,m,n,x1,y1,x2,y2,size,size1;
  Boolean flag=False;
  struct location_struct location;

  XSetLineAttributes(display,xdisplay[id].gc,line_width,
                     LineSolid,CapNotLast,JoinRound);

				/* Compute the size of the pick nodes. */
  if (line_width<=1)
    size=4;
  else
    size=4+line_width;
  size1=size/2;

  XPutImage(display,pix,xdisplay[id].gc,xdisplay[id_anim].image,0,0,0,0,
	    xdisplay[id_anim].image->width,xdisplay[id_anim].image->height);

				/* Loop through the picks. */
  for (n=0; n<MAX_PICKS; n++)
    if (picks[n].init==-1) {
      if (((visible_segment(id,picks[n].group)==0)&&
	   !(xdisplay[id_anim].overlay_flag))||
	  ((visible_segment(id,picks[n].group))&&
	   !(xdisplay[id_anim].segment_flag))) {
	if (n==current_segment) {
	  XSetLineAttributes(display,xdisplay[id].gc,0,
			     LineSolid,CapNotLast,JoinRound);
	  XSetForeground(display,xdisplay[id].gc,white);
	} else
	  XSetForeground(display,xdisplay[id].gc,picks[n].color);
	flag=False;
/*if(picks[n].id != id_anim )
fprintf( stderr,"NOTE: draw_pix: picks[n].id(%d) != id_anim(%d)\n", picks[n].id, id_anim);
*/
	for (m=1; m<picks[n].size; m++) {
	  
	  if ((xdisplay[id_anim].sort)&&(xdisplay[id_anim].middle))
/*	    location=get_location((erec[id_anim]-srec[id_anim])/2,*(picks[n].trace+m-1),*(picks[n].sample+m-1),id_anim); RJM */
	    location=get_location((erec[id_anim]-srec[id_anim])/2,picks[n].trace[m-1],picks[n].sample[m-1],id_anim?id_anim:picks[n].id);
	  else
/*	    location=get_location(*(picks[n].record+m-1),*(picks[n].trace+m-1),*(picks[n].sample+m-1),id_anim); RJM */
	    location=get_location(picks[n].record[m-1],picks[n].trace[m-1],picks[n].sample[m-1],id_anim?id_anim:picks[n].id);
	  x1=(int)location.x/xdisplay[id_anim].pick_x;
	  y1=(int)location.y/xdisplay[id_anim].pick_y;
	  if ((xdisplay[id_anim].sort)&&(xdisplay[id_anim].middle))
/*	    location=get_location((erec[id_anim]-srec[id_anim])/2,*(picks[n].trace+m),*(picks[n].sample+m),id_anim); RJM */
	    location=get_location((erec[id_anim]-srec[id_anim])/2,picks[n].trace[m],picks[n].sample[m],id_anim?id_anim:picks[n].id);
	  else
/*	    location=get_location(*(picks[n].record+m),*(picks[n].trace+m),*(picks[n].sample+m),id_anim); RJM */
	    location=get_location(picks[n].record[m],picks[n].trace[m],picks[n].sample[m],id_anim?id_anim:picks[n].id);
	  x2=(int)location.x/xdisplay[id_anim].pick_x;
	  y2=(int)location.y/xdisplay[id_anim].pick_y;
	  
	  if ((*(picks[n].trace+m-1)<strace[id_anim])||(*(picks[n].trace+m-1)>etrace[id_anim])||
	      (*(picks[n].trace+m)<strace[id_anim])||(*(picks[n].trace+m)>etrace[id_anim]))
	    flag=True;
	  if ((*(picks[n].sample+m-1)<ssamp[id_anim])||(*(picks[n].sample+m-1)>esamp[id_anim])|| 
	      (*(picks[n].sample+m)<ssamp[id_anim])||(*(picks[n].sample+m)>esamp[id_anim]))
	    flag=True;
	  if (xdisplay[id_anim].record_flag) {
	    flag=True;
	    for (j=srec[id_anim]; j<=erec[id_anim]; j+=irec[id_anim]) 
	      if (*(picks[n].record+m)==j)
		flag=False;
	  }
	  if (!flag) {
	    XDrawLine(display,pix,xdisplay[id].gc,
		      x1,y1,x2,y2);
	    if (xdisplay[id_anim].mode>1)
	      XFillArc(display,pix,xdisplay[id].gc,
		       x1-size1,y1-size1,size,size,0,23040);
	  } else {
	    flag=False;
	  }
	}
	
	if ((*(picks[n].trace+m-1)<strace[id_anim])||(*(picks[n].trace+m-1)>etrace[id_anim])) 
	  flag=True;
	if ((*(picks[n].sample+m-1)<ssamp[id_anim])||(*(picks[n].sample+m-1)>esamp[id_anim])) 
	  flag=True;
	if (xdisplay[id_anim].record_flag) {
	  flag=True;
	  for (j=srec[id_anim]; j<=erec[id_anim]; j+=irec[id_anim]) 
	    if (*(picks[n].record+m-1)==j)
	      flag=False;
	}
	if (!flag) {
	  if (xdisplay[id_anim].mode>1)
	    XFillArc(display,pix,xdisplay[id].gc,
		     x2-size1,y2-size1,size,size,0,23040);
	}
      
	if (picks[n].size==1) {
	  m=1;
	  if ((xdisplay[id_anim].sort)&&(xdisplay[id_anim].middle)) {
/*	    location=get_location(((erec[id_anim]-srec[id_anim]+1)/2)+srec[id_anim],*(picks[n].trace+m-1),*(picks[n].sample+m-1),id_anim); RJM */
	    location=get_location(((erec[id_anim]-srec[id_anim]+1)/2)+srec[id_anim],picks[n].trace[m-1],picks[n].sample[m-1],id_anim?id_anim:picks[n].id);
	  } else {
				/* Check if record number is honored. */
	    if (xdisplay[id_anim].record_flag) {
/*	      location=get_location(*(picks[n].record+m-1),*(picks[n].trace+m-1),*(picks[n].sample+m-1),id_anim); RJM */
	      location=get_location(picks[n].record[m-1],picks[n].trace[m-1],picks[n].sample[m-1],id_anim?id_anim:picks[n].id);
	    } else {
/*	      location=get_location(srec[id_anim],*(picks[n].trace+m-1),*(picks[n].sample+m-1),id_anim); RJM */
	      location=get_location(srec[id_anim],picks[n].trace[m-1],picks[n].sample[m-1],id_anim?id_anim:picks[n].id);
	    }
	  }
	  
				/* Scale x and y according to zoom. */
	  x1=(int)location.x/xdisplay[id_anim].pick_x;
	  y1=(int)location.y/xdisplay[id_anim].pick_y;
	  
	  XFillArc(display,pix,xdisplay[id].gc,
		   x1-size1,y1-size1,size,size,0,23040);
	}


	if (n==current_segment) {
	  XSetLineAttributes(display,xdisplay[id].gc,line_width, 
			     LineSolid,CapNotLast,JoinRound);
	}
	
      }
    }

  XSetLineAttributes(display,xdisplay[id].gc,0,
                     LineSolid,CapNotLast,JoinRound);
}
