/***********************************************************************
 *                copyright 2001, Amoco Production Company             *
 *                            All Rights Reserved                      *
 *                    an affiliate of BP America Inc.                  *
 ***********************************************************************/
/*
	Name:		picks_to_agm
	Purpose:	write picks to an AGM file.
	Author:		Bob Mars
	Date:		9/5/91
*/


#include <stdio.h>	/* cuserid */
#include <unistd.h>	/* access */
#include "std.h"
#include "agm.h"
#include "agm_io.h"
#include "agm_datatypes.h"
#include "agm_collection_types.h"
#include "stringutils.h"
#include "miscutils.h"	/* todays_date */


/* #define TEST */
/* #define DEBUG */


int picks_to_agm( Tuple **curves, unsigned *nptspercurve, unsigned ncurves,
		String modelname, Units *units, String fn, Boolean override)
{
	extern int	agm( unsigned, ... );
	extern int	agm_io( unsigned, ... );
	extern void	curv3dminmax( Tuple **, unsigned, unsigned*, Tuple* );

	Id		xsection_collectionid;
	Id		collectionid, segmentid;
	Tuple		minmax[2];
	Factors		factors;
	Location	location;
	char		expirydate[11+1];
	String		tmp, name;
	unsigned int	curve, n;
	int		year, rc, i;

#ifdef DEBUG
	int j;

	(void)printf( "picks_to_agm: ncurves=%u\n", ncurves );
	for( i = 0; i < ncurves;  i++ ) {
		(void)printf( "\tnptspercurve[%u]=%u\n", nptspercurve[i] );
		for( j = 0;  j < nptspercurve[i];  j++ )
			(void)printf( "\t\t%f, %f, %f\n", curves[i][j].x,
					curves[i][j].y, curves[i][j].z );
	}
#endif

	/* clear out any other models sitting in agm memory (from agm_to_picks)
		as these also end up getting written out
	*/
	rc = agm( GetNumberModels, &n );
	for( i=0; i<n; i++ )
	{
		rc = agm( GetIndexedModelName, i+1, &name );	/* nth model starts at 1 */
		if( 0 == rc )
			(void)agm( DeleteModel, name );
	}

	if( (rc = agm( CreateModel, modelname )) != 0 )
		return rc;

	(void)agm( PutComments, "Model created by picks_to_agm." );

	if( (rc = agm( PutUserId, cuserid( NULL ) )) != 0 )
		return rc;
	if( (rc = agm( PutOwnerName, owner_name()) ) != 0 )
		return rc;
	(void)strcpy( expirydate, todays_date() );
	year = atoi( &expirydate[7] );   itoa( year+1, &expirydate[7] );
	if( (rc = agm( PutExpiryDate, expirydate )) != 0 )
		return rc;
	if( (rc = agm( PutModelDescription, "Model made from picks." )) != 0)
		return rc;

	if( (rc = agm( PutUnits, units )) != 0 )
		return rc;
	factors.x = factors.y = factors.z = 1.0;
	if( (rc = agm( PutFactors, &factors )) != 0 )
		return rc;
	curv3dminmax( curves, ncurves, nptspercurve, minmax );
	if( (rc = agm( PutExtents, minmax )) != 0 )
		return rc;

	location.x = location.y = location.z = 0.0;
	if( (rc = agm( PutLocation, &location )) != 0 )
		return rc;
	if( (rc = agm( PutZoffset, 0.0 )) != 0 )
		return rc;
	if( (rc = agm( PutAzimuth, 0.0 )) != 0 )
		return rc;
	if( (rc = agm( PutLatitude, 0.0 )) != 0 )
		return rc;
	if( (rc = agm( PutLongitude, 0.0 )) != 0 )
		return rc;
	if( (rc = agm( PutProjectionSystem, "  000000" )) != 0 )
		return rc;

	if( (rc = agm( CreateVersion, 1L )) != 0 )
		return rc;

	if( (rc = agm( PutVersionParent, 0L )) != 0 )
		return rc;
	if( (rc = agm( PutVersionCreationDate, todays_date() )) != 0 )
		return rc;
	if( (rc = agm( PutVersionCreationTime, current_time() )) != 0 )
		return rc;
	if( (rc = agm( PutVersionLastUpDate, todays_date() )) != 0 )
		return rc;
	if( (rc = agm( PutVersionLastUpTime, current_time() )) != 0 )
		return rc;
	if( (rc = agm( PutVersionDescription, "One and only." )) != 0 )
		return rc;

	if( (rc = agm( PutVersionHistory, "Adam was my father." )) != 0 )
		return rc;

	if( (rc = agm( CreateCollection, NULL )) != 0 )
		return rc;
	if( (rc = agm( PutCollectionType, CrossSection )) != 0 )
		return rc;
	if( (rc = agm( PutCollectionParent, (Id)0 )) != 0 )
		return rc;  /* no parent */
	if( (rc = agm( PutCollectionDescription, "2-D cross-section" )) != 0)
		return rc;
	if( (rc = agm( GetCollectionId, &xsection_collectionid )) != 0 )
		return rc;


	for( curve = 0;  curve < ncurves;  curve++ )  {
		if( (rc = agm( CreateCollection, NULL )) != 0 )
			break;
		if( (rc = agm( PutCollectionType, Horizon )) != 0 )
			break;
		if( (rc = agm( PutCollectionParent, xsection_collectionid ))
			!= 0 )
			break;
		if( (rc = agm( PutCollectionDescription,
		     tmp = concat( "Curve ", itostr( curve+1 ), NULL ) )) != 0){
			free( (char *)tmp );
			break;
		}
		free( (char *)tmp );
		if( (rc = agm( GetCollectionId, &collectionid )) != 0 )
			break;

		if( (rc = agm( CreateSegment, NULL )) != 0 )
			break;
		if( (rc = agm( PutSegment, curves[curve],
				nptspercurve[curve] )) != 0 )
			break;
		if( (rc = agm( GetSegmentId, &segmentid )) != 0 )
			break;
		if( (rc = agm( PutCollectionSegmentAssociation,
				collectionid, segmentid )) != 0 )
			break;
	}

	if( rc != 0 )
		return rc;

	if( override )	/* remove fn */
		if( access( fn, R_OK ) == 0 )	/* file exists */
			unlink( fn );

	(void)agm_io( PutFileName, fn );
	if( (rc = agm_io( Action, Write )) != 0 )
		return rc;

	(void)agm( DeleteModel, modelname );

	return 0;
}
#ifdef TEST


void main()
{
	static Tuple	curve1[] = { {0., 0., 0.}, {1., 1., 1.}, {2., 2., 2.} };
	static Tuple	curve2[] = { {0., 0., 0.}, {1., 1., 1.} };
	static Tuple	curve3[] = { {3., 3., 3.}, {5., 5., 5.} };
	static Tuple	curve4[] = { {4., 4., 4.}, {6., 6., 6.}, {7., 7.,7.} };
	static Tuple	*curves[] = { curve1, curve2, curve3, curve4 };
	static int	nptspercurve[] = { 3, 2, 2, 3 };
	static int	ncurves = 4;
	static Units	units = { "feet", "feet", "feet" };
	static String	modelname = "Bob", fn = "Bob.agm", message = NULL;


	if( picks_to_agm( curves, nptspercurve, ncurves, modelname, units, fn,
				TRUE ) != 0 ) {
		(void)agm( GetMessage, &message );
		(void)fprintf( stderr, "%s\n", message );
	} else
		(void)printf( "Success.\n" );
}
#endif
