/***********************************************************************
 *                copyright 2001, Amoco Production Company             *
 *                            All Rights Reserved                      *
 *                    an affiliate of BP America Inc.                  *
 ***********************************************************************/
/* unzip.c -- decompress buffers in gzip format.
 * Taken from the GNU gzip freeware.
 */
#include <ctype.h>
#include <errno.h>
#include <sys/types.h>
#include "zip.h"

#ifdef TESTZIP
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#endif

/* Local variables */

static long ocnt;               /* # bytes in output buffer */
static long osize;              /* size of output buffer */
static uch *obuf;               /* output buffer */

static unsigned inptr;  	/* index of next byte in inbuf*/
static unsigned insize;       	/* size of input buffer */
static uch *inbuf;              /* input buffer */


/* =====================================================================
 * Unzip in to out.  This routine works on both gzip and pkzip files.
 *
 * IN assertions: the buffer inbuf contains already the beginning of
 *   the compressed data, from offsets inptr to insize-1 included.
 *   The magic header has already been checked. The output buffer is 
 *   cleared.
 */
int unzip(int *buflen, uch **buffer)
{
    int n, rc;
    int method;		/* compression method */
    ulg orig_len = 0;  	/* original uncompressed length */
    char magic[2]; 	/* magic header */
    uch buf[4];   	/* buffer for uncompressed size */

    inptr  = 0;
    insize = *buflen;
    inbuf  = *buffer;
    
    ocnt  = 0;
    osize = 10000;
    obuf  = (uch*)malloc(osize);

    /* get header */
    
    magic[0] = (char)get_byte();
    magic[1] = (char)get_byte();
    method = (int)get_byte();
    for (n=0;n<4;n++) buf[n] = (uch)get_byte(); 
    orig_len = LG(buf);
    
    if (memcmp(magic, GZIP_MAGIC, 2) != 0) 
    {
       ocnt = -1;
       fprintf(stderr, "unzip error: bad MAGIC #!\n");
    }
    else if (method != DEFLATED) 
    {
       ocnt = -1;
       fprintf(stderr, "unzip error: unknown method %d!\n", method);
    }

    /* Decompress */

    if (ocnt >= 0) rc = inflate();
    else rc = 0;

    if (rc == 3) 
    {
       ocnt = -1;
       fprintf(stderr, "unzip error: out of memory!\n");
    } 
    else if (rc != 0) 
    {
       ocnt = -1;
       fprintf(stderr, 
       		"unzip error: invalid compressed data format!\n");
    }

    /* Validate decompression */
    else if (ocnt > 0 && orig_len != (ulg)ocnt) 
    {
       ocnt = -1;
       fprintf(stderr, 
       		"unzip error: invalid compressed data length!\n");
    }
    
    if (ocnt > 0)
    {
       free((char*)*buffer);
       *buflen = ocnt;
       *buffer = obuf;
    }
    else
    {
       free((char*)obuf);
    }
    
    return(ocnt);
}

/* =====================================================================
 * Get the next input byte. 
 */
uch get_byte()
{
    if (inptr < insize) return(inbuf[inptr++]);
    return((uch)0);
}

/* =====================================================================
 * Write the output window window[0..outcnt-1] and bytes_out.
 * (Used for the decompressed data only.)
 */
void flush_window(uch *window, unsigned outcnt)
{
    if (outcnt == 0) return;

    if (ocnt + outcnt > osize)
    {
       osize = ocnt + outcnt + 10000;
       obuf = (uch*)realloc(obuf, osize);
    }
    memcpy((char*)&obuf[ocnt], (char*)window, (size_t)outcnt);
    ocnt += outcnt;
}


#ifdef TESTZIP

main(int argc, char **argv)
{
   int fd, n, m;
   long nbuf;
   uch *buf;
   
   fd = open(argv[1], O_RDONLY);
   nbuf = lseek(fd, (off_t)0, SEEK_END);
   lseek(fd, (off_t)0, SEEK_SET);
   
   buf = (uch*)malloc(nbuf);
   n = 0;
   while (n<nbuf)
   {
      m = read(fd, &buf[n], nbuf-n);
      if (m <= 0)
      {
         perror("read");
         exit(0);
      }
      n += m;
   }
   
   unzip(&nbuf, &buf);
   printf("unzip: %d==>%d\n", n, nbuf);
   
   fd = open("tst.unzip", O_WRONLY|O_CREAT);
   n = 0;
   while(n<nbuf)
   {
      m = write(fd, &buf[n], nbuf-n);
      if (m <= 0)
      {
         perror("write");
         exit(0);
      }
      n += m;
   }
}

#endif
