/***********************************************************************
 *                copyright 2001, Amoco Production Company             *
 *                            All Rights Reserved                      *
 *                    an affiliate of BP America Inc.                  *
 ***********************************************************************/
/* ---------------------------------------------------------------------------

	Package	:	sisio
	File	:	bkspt.c
	Author	:	Dean Kopesky
	Company	:	Minnesota Supercomputer Center, Inc.


	BKSPT backspaces the unit the specified number of records (line header
	and traces).  A negative record count is an error.  Backspacing before
	BOF will position the pointer at BOF.  This subroutine does not work
	on format L.

	FORTRAN:
	
	CALL BKSPT( unit, records, dummy )

	unit	input	integer	Unit to backspace.
	records	input	integer	Number of records to backspace.
	dummy	dummy	any	Dummy parameter for compatiblity with an
				older library.

	C:
	
	bkspt( unit, records );

	unit	input	int	Unit to backspace.
	records	input	int	Number of records to backspace.


	$Header: /m/s1/dmk/xlate/sisio/RCS/bkspt.c,v 4.3 91/09/11 08:38:10 dmk Exp $

	$Log:	bkspt.c,v $
	Revision 4.3  91/09/11  08:38:10  dmk
	MODSET: 4.3
	Changed entry point names to macros.
	
	Revision 4.1  90/07/13  09:15:33  dmk
	MODSET: 4.1
	Added check for format L.  Changed SUN to sun.
	
	Revision 3.1  89/09/27  09:52:44  dmk
	MODSET: 3.1
	Added C entry point.  No dummy parameter on that one.
	
	Revision 2.1  88/07/12  15:34:33  dmk
	MODSET: 2.1
	Added lower case entry point for SUN.
	
	Revision 1.2  88/06/27  09:37:10  dmk
	MODSET: 1.2
	Will now flush the buffer if large buffers are enabled, rather than
	flagging an error.  Will now use _sisfit[ unit ].cwlen rather than
	recalculating cwlen.
	
	Revision 1.1  88/06/24  11:55:32  dmk
	Initial revision
	
--------------------------------------------------------------------------- */


#include	"sisio.h"
#include	<sys/types.h>
#include	<unistd.h>

#define	MODULE	"BKSPT"


void F_BKSPT( unit, records, dummy )

	int *	unit;
	int *	records;
	char *	dummy;

{
	C_BKSPT( *unit, *records );
}


void	C_BKSPT( unit, records )

	int	unit;
	int	records;

{
	int	ferror;

	/*
		Initialize iff necessary, check unit for range, check for
		outstanding error, check to see if unit is open.  Check for
		format L.
	*/

	INITIALIZE;
	CHECKRANGE(MODULE,unit);
	CHECKERROR(MODULE,unit);
	CHECKOPEN(MODULE,unit);

	if ( _sisfit[ unit ].format == 'L' )
	{
		_error( MODULE, unit, E_NOTONL );
		return;
	}

	/*
		Check for large buffering.  If so, flush the buffer.
	*/

	if ( _sisfit[ unit ].largebuf )
	{
		ferror = _flushit( unit );
		if ( ferror < 0 )
		{
			_error( MODULE, unit, -ferror );
			return;
		}
	}

	/*
		Check records:  if zero, return; if negative, error.
	*/

	if ( records == 0 ) return;

	if ( records < 0 )
	{
		_error( MODULE, unit, E_NEGREC );
		return;
	}

	/*
		If backspacing past BOF, just rewind.
	*/

	if ( _sisfit[ unit ].pointer - records <= 0 )
	{
		if ( lseek( _sisfit[ unit ].filedes, 0L, 0 ) != 0 )
		{
			_error( MODULE, unit, E_LSEEK );
			return;
		}
		_sisfit[ unit ].pointer = 0;
		return;
	}

	/*
		Otherwise, fall through to here.  If trace length is
		unknown, flag an error.  If trace length is known, lseek.
	*/

	if ( _sisfit[ unit ].trclen == 0 )
	{
		_error( MODULE, unit, E_NOTRCLEN );
		return;
	}

#ifndef __convex__
	if ( lseek( _sisfit[ unit ].filedes, ( long ) - ( records * (
		_sisfit[ unit ].trclen + _sisfit[ unit ].cwlen ) ), 1 ) < 0 )
#else
	if ( lseek64( _sisfit[ unit ].filedes, ( off64_t ) - ( records * (
		_sisfit[ unit ].trclen + _sisfit[ unit ].cwlen ) ), SEEK_CUR ) < 0 )
#endif
	{
		_error( MODULE, unit, E_LSEEK );
		return;
	}

	_sisfit[ unit ].pointer -= records;
}
