C***********************************************************************
C                                                                      *
c                 copyright 2001, Amoco Production Company             *
c                             All Rights Reserved                      *
c                     an affiliate of BP America Inc.                  *
C***********************************************************************
C***********************************************************************
C                                                                      *
C  ROUTINE:       VZRDVM                                               *
C  ROUTINE TYPE:  SUBROUTINE                                           *
C  PURPOSE:                                                            *
C       VZRDVM2 reads a V(z) velocity model from a file.               *
C                                                                      *
C  ENTRY POINTS:                                                       *
C      VZRDVM2  (IFLG,LUVEL,LUPRT,MAXSEG,STRCH,DZOUT,TT2WAY,NZSEG,     *
C                VREFL,VINCI,ZSTEP,IERR)                               *
C  ARGUMENTS:                                                          *
C      IFLG    INTEGER  ??IOU*      -                                  *
C      LUVEL   INTEGER  ??IOU*      -                                  *
C      LUPRT   INTEGER  ??IOU*      -                                  *
C      MAXSEG  INTEGER  ??IOU*      -                                  *
C      STRCH   REAL     ??IOU*      -                                  *
C      DZOUT   REAL     ??IOU*      -                                  *
C      TT2WAY  REAL     ??IOU*      -                                  *
C      NZSEG   INTEGER  ??IOU*      -                                  *
C      VREFL   REAL     ??IOU*  (*) -                                  *
C      VINCI   REAL     ??IOU*  (*) -                                  *
C      ZSTEP   REAL     ??IOU*  (*) -                                  *
C      IERR    INTEGER  ??IOU*      -                                  *
C       +------------------------------------------------------+       *
C       |               DEVELOPMENT INFORMATION                |       *
C       +------------------------------------------------------+       *
C  AUTHOR:   ???                                ORIGIN DATE: 97/02/13  *
C  LANGUAGE: FORTRAN 77                  DATE LAST COMPILED: 97/02/18  *
C       FORTRAN 77                                                     *
C                                                                      *
C  HISTORY:                                                            *
C       ORIGINAL                NOV 92          R.D. COLEMAN, CETech   *
C       Revised           June 2, 1993          Mary Ann Thornton      *
C               Subroutine was changed to allow the input velocity     *
C               cards to be in either of 2 allowable formats as        *
C               shown below.                                           *
C                                                                      *
C  CALLING FORMAT:                                                     *
C       CALL VZRDVM2( IFLG, LUVEL, LUPRT, MAXSEG, STRCH, DZOUT,        *
C                     TT2WAY, NZSEG, VREFL, VINCI, ZSTEP, IERR )       *
C                                                                      *
C  PARAMETERS:                                                         *
C       IFLG    INTEGER INPUT SCALAR                                   *
C               Function flag:                                         *
C                  IFLG  = 0, read velocities for prestack  migration. *
C                  IFLG != 0, read velocities for poststack migration. *
C                                                                      *
C       LUVEL   INTEGER INPUT SCALAR                                   *
C               Logical unit number of velocity file.                  *
C                                                                      *
C       LUPRT   INTEGER INPUT SCALAR                                   *
C               Logical unit number for print file.  If LUERR < 0,     *
C               then VZRDVM2 executes in silent mode; i.e., nothing    *
C               is written to the print file..                         *
C                                                                      *
C       MAXSEG  INTEGER INPUT SCALAR                                   *
C               Maximum number of Z segments allowed; i.e., the physica*
C               length of the output vectors.                          *
C                                                                      *
C       STRCH   REAL INPUT SCALAR                                      *
C               Velocity stretch factor.                               *
C                                                                      *
C       DZOUT   REAL INPUT SCALAR                                      *
C               Detla Z for output image.                              *
C                                                                      *
C       TT2WAY  REAL OUTPUT SCALAR                                     *
C               Two way travel time (in seconds).                      *
C                                                                      *
C       NZSEG   INTEGER OUTPUT SCALAR                                  *
C               Number of Z segments.                                  *
C                                                                      *
C       VREFL   REAL OUTPUT VECTOR OF LENGTH NZSEG                     *
C               Contains the reflected velocity for each segment.      *
C                                                                      *
C       VINCI   REAL OUTPUT VECTOR OF LENGTH NZSEG                     *
C               Contains the incident velocity for each segment, if    *
C               IFLG = 0.  Otherwise, contains zeros.                  *
C                                                                      *
C       ZSTEP   REAL OUTPUT VECTOR OF LENGTH NZSEG                     *
C               Contains the depth of each segment.                    *
C                                                                      *
C       IERR    INTEGER OUTPUT SCALAR                                  *
C               Completion code:                                       *
C                  IERR = 0, normal completion                         *
C                       = 1, unexpected end-of-file                    *
C                       = 2, data conversion error                     *
C                       = 3, cards not in order                        *
C                       = 4, invalid data value                        *
C                       = 5, NZSEG <= 0                                *
C                       = 6, NZSEG >  MAXSEG                           *
C                                                                      *
C       +------------------------------------------------------+       *
C       |                 EXTERNAL ENVIRONMENT                 |       *
C       +------------------------------------------------------+       *
C  ROUTINES CALLED:  NONE                                              *
C  INTRINSIC FUNCTIONS CALLED:                                         *
C      AINT    GENERIC -                                               *
C  FILES:                                                              *
C       Velocity card file.  This file will accept 2 formats:          *
C       (1) A velocity card file consists of:                          *
C             a NOVEL (number of velocities) template card,            *
C             a NOVEL data card,                                       *
C             a velocity template card,                                *
C             and NOVEL ( = NZSEG ) velocity data cards.               *
C                                                                      *
C           The NOVEL template card must contain the literal string    *
C           'NOVEL' in columns 1 through 5.  The NOVEL data card contai*
C           the number of velocities (NOVEL = NZSEG) and is read in    *
C           (10X, I10) format.  The velocity template card must contain*
C           'VELOCITIES' in columns 1 through 10.  The each velocity da*
C           card contains the following:                               *
C                                                                      *
C              Columns  Format                                         *
C                1-10    I10    Sequence number (must be in sequential *
C                               order beginning with 1)                *
C               11-20    F10.0  Reflected velocity for segment         *
C               21-30    F10.0  Prestack: Incident  velocity for segmen*
C                        10X    Poststack: Ignored                     *
C               31-40    F10.0  Depth at end of segment                *
C                                                                      *
C       (2) The 2nd velocity card file format consists of:             *
C             a MODEL (number of velocities) template/velocity card,   *
C             a velocity template card,                                *
C             and MODEL ( = NZSEG ) velocity data cards.               *
C                                                                      *
C           The MODEL template/velocity card must contain the literal s*
C           'MODEL' in columns 1 through 5.  The MODEL data card contai*
C           the number of velocities ( = NZSEG) and is reread under    *
C           (10X, I10) format control.                                 *
C                                                                      *
C           See examples of both formats below.                        *
C                                                                      *
C      CARD   ( INPUT  INTERNAL   ) -                                  *
C      LUPRT  ( OUTPUT SEQUENTIAL ) -                                  *
C      LUVEL  ( INPUT  SEQUENTIAL ) -                                  *
C  DESCRIPTION:                                                        *
C       VZRDVM2 reads the velocity file of either format shown above an*
C       checks it for validity.  The velocities and end depths are     *
C       scaled by the stretch factor, STRCH.  If the input value of    *
C       STRCH is less than or equal to zero, then no scaling takes     *
C       place.  The image of the velocity file is written to LUPRT     *
C       (if LUPRT > 0) with the "stretched" velocities and end depths  *
C       replacing the original.  The end depth of each segment is      *
C       rounded up to the next highest integer multiple of DZOUT; i.e.,*
C       ZEND(i) = DZOUT * AINT( ZEND(i) / DZOUT + 1.0 ).  The program  *
C       then computes the depth of each segment (i.e., the end depth   *
C       minus the previous end depth) and the minimum two way travel   *
C       time.  If the first velocity card contains an end depth less   *
C       than or equal to zero, then that card is ignored.              *
C                                                                      *
C       For post-stack migration (IFLG != 0), the incident velocity is *
C       not read; instead, it is set equal to the reflected velocity fo*
C       internal use and a value of zero is returned in VINCI.         *
C                                                                      *
C  EXAMPLE:                                                            *
C       Velocity file:                                                 *
C                                                                      *
C                        1111111111222222222233333333334               *
C       Column: 1234567890123456789012345678901234567890               *
C                                                                      *
C         (1)                                                          *
C               NOVEL     # VELOCITY                                   *
C               NOVEL              4                                   *
C               VELOCITIES REFLECTED  INCIDENT END DEPTH               *
C                        1      5000      4000      2000               *
C                        2      6000      6000      5000               *
C                        3      7500      7500      8000               *
C                        4      8000      8000     12000               *
C         (2)                                                          *
C               MODEL              4                                   *
C               VELOCITIES REFLECTED  INCIDENT END DEPTH               *
C                        1      5000      4000      2000               *
C                        2      6000      6000      5000               *
C                        3      7500      7500      8000               *
C                        4      8000      8000     12000               *
C                                                                      *
C       Inputs:                                                        *
C               IFLG   =    0                                          *
C               LUVEL  =   29                                          *
C               LUPRT  =   26                                          *
C               MAXSEG = 1000                                          *
C               STRCH  =    1.0                                        *
C               DZOUT  =   20.0                                        *
C                                                                      *
C       Outputs:                                                       *
C               TT2WAY = 3.709                                         *
C               NZSEG  = 4                                             *
C               VREFL  = ( 5000.0, 6000.0, 7500.0, 8000.0 )            *
C               VINCI  = ( 4000.0, 6000.0, 7500.0, 8000.0 )            *
C               ZSTEP  = ( 2020.0, 3000.0, 3000.0, 4000.0 )            *
C               IERR   = 0                                             *
C                                                                      *
C  SUBPROGRAMS CALLED:                                                 *
C       None.                                                          *
C                                                                      *
C  ERROR CONDITIONS:                                                   *
C       If an error is detected by VZRDVM2, then an error message is   *
C       written to LUPRT (if LUPRT >= 0), IERR is set to the appropriat*
C       error code (see above), and the routine is aborted.            *
C                                                                      *
C--------------------------------------------------------------------- *
C                                                                      *
C  COMMON:           NONE                                              *
C  STOP CODES:       NONE                                              *
C       +------------------------------------------------------+       *
C       |             OTHER DOCUMENTATION DETAILS              |       *
C       +------------------------------------------------------+       *
C  ERROR HANDLING:  ???                                                *
C  GENERAL DESCRIPTION:  ???                                           *
C  REVISED BY:  ???                           REVISION DATE: ?Y/?M/?D  *
C       +------------------------------------------------------+       *
C       |                 ANALYSIS INFORMATION                 |       *
C       +------------------------------------------------------+       *
C  NONSTANDARD FEATURES:   NONE DETECTED                               *
C*******************   END OF DOCUMENTATION PACKAGE   ******************
C***********************************************************************
C*****************  ITEMS DELETED FROM DOCPACK  ************************
C  =============================== DATE: 97/02/13 ==================   *
C NAME: VZRDVM2   READ V(Z) VELOCITY MODEL             REV 1.0  NOV 92 *
C********************  END OF DELETED ITEMS  ***************************
C***********************************************************************
      SUBROUTINE VZRDVM2( IFLG, LUVEL, LUPRT, MAXSEG, STRCH, DZOUT,
     &                    TT2WAY, NZSEG, VREFL, VINCI, ZSTEP, IERR )
C
C  SUBROUTINE ARGUMENTS
C
      INTEGER IFLG, LUVEL, LUPRT, MAXSEG, NZSEG, IERR
      REAL    STRCH, DZOUT, TT2WAY, VREFL(*), VINCI(*), ZSTEP(*)
C
C  LOCAL VARIABLES
C
      CHARACTER*80 CARD
      INTEGER IZSEG, JZSEG, KZSEG
      REAL    DZ, VR, VI, STRETCH, ZEND, ZLAST
C
C-----------------------------------------------------------------------
C
  900 FORMAT (A80)
  901 FORMAT ( 10X, I10)
  902 FORMAT ( I10, 3F10.0)
  903 FORMAT ( I10, F10.0, 10X, F10.0 )
  911 FORMAT (/' ', 'NOVEL     # VELOCITY')
  912 FORMAT (/' ', 'VELOCITIES REFLECTED  INCIDENT END DEPTH')
  913 FORMAT (/' ', 'VELOCITIES REFLECTED           END DEPTH')
  921 FORMAT ( ' ', 'NOVEL     ', I10)
  931 FORMAT (/' ', 'MODEL     ', I10)
  922 FORMAT ( ' ', I10, 3F10.1)
  923 FORMAT ( ' ', I10, F10.1, 10X, F10.1 )
  924 FORMAT (/' ', 'VELOCITIES AND DEPTHS ARE STRETCHED BY', F6.3)
  991 FORMAT (/' ', '***** VZRDVM2 ERROR - UNEXPECTED END-OF-FILE')
  992 FORMAT (/' ', '***** VZRDVM2 ERROR - DATA CONVERSION ERROR'/
     &         ' ', '      INPUT CARD: ', A80)
  993 FORMAT (/' ', '***** VZRDVM2 ERROR - INVALID DATA CARD ORDER')
  994 FORMAT (/' ', '***** VZRDVM2 ERROR - INVALID DATA VALUE')
  995 FORMAT (/' ', '***** VZRDVM2 ERROR - INVALID NUMBER OF ',
     &                                     'VELOCITIES')
  996 FORMAT (/' ', '***** VZRDVM2 ERROR - TOO MANY VELOCITIES')
C
C-----------------------------------------------------------------------
C
      IF( STRCH .GT. 0. ) THEN
         STRETCH = STRCH
      ELSE
         STRETCH = 1.0
      ENDIF
C
C  READ NOVEL/MODEL (NUMBER OF VELOCITIES) CARD
C
  100 CONTINUE
         READ (LUVEL, 900, END=810) CARD
 
         IF( CARD(1:5) .EQ. 'NOVEL' ) THEN
            READ (LUVEL, 901, END=810) NZSEG
            IF( LUPRT .GE. 0 ) THEN
              WRITE (LUPRT, 911)
              WRITE (LUPRT, 921) NZSEG
            ENDIF
         ELSE IF( CARD(1:5) .EQ. 'MODEL' ) THEN
               READ (CARD, 901, END=810) NZSEG
               IF( LUPRT .GE. 0 ) THEN
                 WRITE (LUPRT, 931) NZSEG
               ENDIF
         ELSE
            GO TO 100
         ENDIF
C
      IF( NZSEG .LE.      0 ) GO TO 850
      IF( NZSEG .GT. MAXSEG ) GO TO 860
C
C  READ VELOCITY CARDS
C
  200 CONTINUE
         READ (LUVEL, 900, END=810) CARD
         IF( CARD(1:10) .NE. 'VELOCITIES' ) GO TO 200
C
      IF( LUPRT .GE. 0 .AND. IFLG .EQ. 0 ) WRITE (LUPRT, 912)
      IF( LUPRT .GE. 0 .AND. IFLG .NE. 0 ) WRITE (LUPRT, 913)
C
      KZSEG  = 0
      TT2WAY = 0.0
      ZLAST  = 0.0
      DO 210 IZSEG = 1, NZSEG
         VI = 0
C
         READ (LUVEL, 900, END=810) CARD
         IF( IFLG .EQ. 0 ) READ (CARD, 902, ERR=820) JZSEG, VR, VI, ZEND
         IF( IFLG .NE. 0 ) READ (CARD, 903, ERR=820) JZSEG, VR,     ZEND
C
         IF( VI .LE. 0.0 ) VI = VR
C
         VR   = VR   * STRETCH
         VI   = VI   * STRETCH
         ZEND = ZEND * STRETCH
C
         IF( LUPRT .GE. 0 ) THEN
            IF( IFLG .EQ. 0 ) THEN
               WRITE (LUPRT, 922) JZSEG, VR, VI, ZEND
            ELSE
               WRITE (LUPRT, 923) JZSEG, VR,     ZEND
            ENDIF
         ENDIF
C
         IF( JZSEG .NE. IZSEG ) GO TO 830
         IF( JZSEG .EQ. 1 .AND. ZEND .LE. 0.0 ) GO TO 210
C
         ZEND  = DZOUT * AINT( ZEND / DZOUT + 1.0 )
         DZ    = ZEND - ZLAST
         ZLAST = ZEND
C
         IF( VR.LE.0.0 .OR. VI.LE.0.0 .OR. DZ.LE.0.0 ) GO TO 840
C
         KZSEG        = KZSEG + 1
         TT2WAY       = TT2WAY + DZ / VR + DZ / VI
         ZSTEP(KZSEG) = DZ
         VREFL(KZSEG) = VR
         IF( IFLG .EQ. 0 ) THEN
            VINCI(KZSEG) = VI
         ELSE
            VINCI(KZSEG) = 0.0
         ENDIF
  210 CONTINUE
      NZSEG = KZSEG
C
      IF( STRETCH.NE.1.0 .AND. LUPRT.GT.0 ) WRITE (LUPRT, 924) STRETCH
C
C  NORMAL COMPLETION
C
      IERR = 0
      RETURN
C
C  ERROR EXITS
C
  810 CONTINUE
      IF( LUPRT .GE. 0 ) WRITE (LUPRT, 991)
      IERR = 1
      RETURN
C
  820 CONTINUE
      IF( LUPRT .GE. 0 ) WRITE (LUPRT, 992)
      IERR = 2
      RETURN
C
  830 CONTINUE
      IF( LUPRT .GE. 0 ) WRITE (LUPRT, 993)
      IERR = 3
      RETURN
C
  840 CONTINUE
      IF( LUPRT .GE. 0 ) WRITE (LUPRT, 994)
      IERR = 4
      RETURN
C
  850 CONTINUE
      IF( LUPRT .GE. 0 ) WRITE (LUPRT, 995)
      IERR = 5
      RETURN
C
  860 CONTINUE
      IF( LUPRT .GE. 0 ) WRITE (LUPRT, 996)
      IERR = 6
      RETURN
C
      END
