C***********************************************************************
C                                                                      *
c                 copyright 2001, Amoco Production Company             *
c                             All Rights Reserved                      *
c                     an affiliate of BP America Inc.                  *
C***********************************************************************
C***********************************************************************
C                                                                      *
C  ROUTINE:       VZVMOD                                               *
C  ROUTINE TYPE:  SUBROUTINE                                           *
C  PURPOSE:                                                            *
C       VZVMOD computes a V(z) velocity model from velocities and      *
C       Z steps.                                                       *
C                                                                      *
C  ENTRY POINTS:                                                       *
C      VZVMOD  (IFLG,FMAX,NZSEG,VREFL,VINCI,ZSTEP,NZT,IZSNZ,ZSDZ,      *
C               ZSSLOR,ZSSLOI)                                         *
C  ARGUMENTS:                                                          *
C      IFLG    INTEGER  ??IOU*          -                              *
C      FMAX    REAL     ??IOU*          -                              *
C      NZSEG   INTEGER  ??IOU*          -                              *
C      VREFL   REAL     ??IOU*  (NZSEG) -                              *
C      VINCI   REAL     ??IOU*  (NZSEG) -                              *
C      ZSTEP   REAL     ??IOU*  (NZSEG) -                              *
C      NZT     INTEGER  ??IOU*          -                              *
C      IZSNZ   INTEGER  ??IOU*  (NZSEG) -                              *
C      ZSDZ    REAL     ??IOU*  (NZSEG) -                              *
C      ZSSLOR  REAL     ??IOU*  (NZSEG) -                              *
C      ZSSLOI  REAL     ??IOU*  (NZSEG) -                              *
C       +------------------------------------------------------+       *
C       |               DEVELOPMENT INFORMATION                |       *
C       +------------------------------------------------------+       *
C  AUTHOR:   ???                                ORIGIN DATE: 92/07/22  *
C  LANGUAGE: FORTRAN 77                  DATE LAST COMPILED: 97/02/18  *
C       FORTRAN 77                                                     *
C                                                                      *
C  HISTORY:                                                            *
C       ORIGINAL                NOV 91          R.D. COLEMAN, CETech   *
C                                                                      *
C  CALLING FORMAT:                                                     *
C       CALL VZVMOD( IFLG, FMAX, NZSEG, VREFL, VINCI, ZSTEP,           *
C      &             NZT, IZSNZ, ZSDZ, ZSSLOR, ZSSLOI )                *
C                                                                      *
C  PARAMETERS:                                                         *
C       IFLG    INTEGER INPUT SCALAR                                   *
C               Function flag:                                         *
C                  IFLG  = 0, compute model for prestack  migration.   *
C                  IFLG != 0, compute model for poststack migration.   *
C                                                                      *
C       FMAX    REAL INPUT SCALAR                                      *
C               Maximum frequency to be migrated (in Hertz).           *
C                                                                      *
C       NZSEG   INTEGER INPUT SCALAR                                   *
C               Number of Z segments.                                  *
C                                                                      *
C       VREFL   REAL INPUT VECTOR OF LENGTH NZSEG                      *
C               Contains the reflected velocity for each segment.      *
C                                                                      *
C       VINCI   REAL INPUT VECTOR OF LENGTH NZSEG                      *
C               Contains the incident velocity for each segment, if    *
C               IFLG = 0.  Otherwise, contains zeros.                  *
C                                                                      *
C       ZSTEP   REAL INPUT VECTOR OF LENGTH NZSEG                      *
C               Contains the depth of each segment.                    *
C                                                                      *
C       NZT     INTEGER OUTPUT SCALAR                                  *
C               Total number of z's (NZT = sum(IZSNZ(i), i = 1, NZS).  *
C                                                                      *
C       IZSNZ   INTEGER OUTPUT VECTOR OF LENGTH NZSEG                  *
C               IZSNZ(i) = number of z's for the i-th z segment.       *
C                                                                      *
C       ZSDZ    REAL OUTPUT VECTOR OF LENGTH NZSEG                     *
C               Contains the delta z for each segment.                 *
C                                                                      *
C       ZSSLOR  REAL OUTPUT VECTOR OF LENGTH NZSEG                     *
C               Contains the reflected slowness for each segment, if   *
C               IFLG = 0. Otherwise, contains twice the slowness.      *
C                                                                      *
C       ZSSLOI  REAL OUTPUT VECTOR OF LENGTH NZSEG                     *
C               Contains the incident slowness for each segment, if    *
C               IFLG = 0.  Otherwise, contains zeros.                  *
C                                                                      *
C  DESCRIPTION:                                                        *
C       Computes the delta z for each z segment ( ZSDZ(i) ) and the    *
C       corresponding number of delta z steps within each z segment    *
C       ( IZSNZ(i) ) such that ZSDZ(i) * IZSNZ(i) = ZSTEP(i) and ZSDZ(i*
C       is less than or equal to a quarter of the minimum wave length  *
C       (i.e., the wave length at the maximum frequency).  Also,       *
C       computes the reflected and incident slowness' (i.e., the       *
C       reciprocal of the velocities).  If the model is for poststack  *
C       migration, then the reflected slowness is doubled and the      *
C       incident slowness is set to zero.                              *
C                                                                      *
C  SUBPROGRAMS CALLED:                                                 *
C       None.                                                          *
C                                                                      *
C  ERROR CONDITIONS:                                                   *
C       None.                                                          *
C                                                                      *
C--------------------------------------------------------------------- *
C                                                                      *
C       +------------------------------------------------------+       *
C       |                 EXTERNAL ENVIRONMENT                 |       *
C       +------------------------------------------------------+       *
C  ROUTINES CALLED:  NONE                                              *
C  INTRINSIC FUNCTIONS CALLED:                                         *
C      IFIX    INTEGER -                                               *
C      FLOAT   REAL    -                                               *
C  FILES:            NONE                                              *
C  COMMON:           NONE                                              *
C  STOP CODES:       NONE                                              *
C       +------------------------------------------------------+       *
C       |             OTHER DOCUMENTATION DETAILS              |       *
C       +------------------------------------------------------+       *
C  ERROR HANDLING:  ???                                                *
C  GENERAL DESCRIPTION:  ???                                           *
C  REVISED BY:  ???                           REVISION DATE: ?Y/?M/?D  *
C       +------------------------------------------------------+       *
C       |                 ANALYSIS INFORMATION                 |       *
C       +------------------------------------------------------+       *
C  NONSTANDARD FEATURES:   NONE DETECTED                               *
C*******************   END OF DOCUMENTATION PACKAGE   ******************
C***********************************************************************
C*****************  ITEMS DELETED FROM DOCPACK  ************************
C  =============================== DATE: 92/07/22 ==================   *
C NAME: VZVMOD    COMPUTE V(Z) VELOCITY MODEL          REV 1.0  NOV 91 *
C********************  END OF DELETED ITEMS  ***************************
C***********************************************************************
      SUBROUTINE VZVMOD( IFLG, FMAX, NZSEG, VREFL, VINCI, ZSTEP,
     &                   NZT, IZSNZ, ZSDZ, ZSSLOR, ZSSLOI )
C
C  SUBROUTINE ARGUMENTS
C
      INTEGER IFLG, NZSEG, NZT, IZSNZ(NZSEG)
      REAL    FMAX, VREFL(NZSEG), VINCI(NZSEG), ZSTEP(NZSEG),
     &        ZSDZ(NZSEG), ZSSLOR(NZSEG), ZSSLOI(NZSEG)
C
C  LOCAL VARIABLES
C
      INTEGER IZSEG, MZ
      REAL    AVEL, DZMAX, STEP, SLOR, SLOI
C
C-----------------------------------------------------------------------
C
      NZT = 0
      DO 110 IZSEG = 1, NZSEG
         STEP = ZSTEP(IZSEG)
         SLOR = 1.0 / VREFL(IZSEG)
         IF( IFLG .EQ. 0 ) THEN
            SLOI = 1.0 / VINCI(IZSEG)
         ELSE
            SLOI = SLOR
         ENDIF
C
         AVEL  = 2.0 / ( SLOR + SLOI )
         DZMAX = 0.25 * AVEL / FMAX
         MZ    = IFIX( STEP / DZMAX )
         IF( MZ * DZMAX .LT. STEP ) MZ = MZ + 1
C
         NZT = NZT + MZ
         IZSNZ (IZSEG) = MZ
         ZSDZ  (IZSEG) = STEP / FLOAT( MZ )
C
         IF( IFLG .EQ. 0 ) THEN
            ZSSLOR(IZSEG) = SLOR
            ZSSLOI(IZSEG) = SLOI
         ELSE
            ZSSLOR(IZSEG) = SLOR + SLOI
            ZSSLOI(IZSEG) = 0.0
         ENDIF
  110 CONTINUE
C
      RETURN
      END
