C***********************************************************************
C                 copyright 2001, Amoco Production Company             *
C                             All Rights Reserved                      *
C                     an affiliate of BP America Inc.                  *
C***********************************************************************
c
c The maximum number of components that we have room allocated
c for in the line header.
c
#define MCLLen	16


c
c Search range for "extra" components is from 1 to this for both
c Source and Receiver.
c
#define COMPRANGE	3


c
c The maximum size of an in-core transpose buffer, in bytes. Bigger
c than this and it won't try to allocate it.
c
#ifndef MAXTBUFSZ
#define MAXTBUFSZ	4194304
#endif


#define STDINNO		0


      subroutine uspinput (mode, clparam, wclparam, dptw,
     1   ncomp, ncompmax, complist, allcomps,
     1   ikpoffs, ikpsock, seekalot, seekstat,
     1   un1, un2, un3, ud1, uf2, uf3, upt,
     1   lheader, lhedlength, trlength, trdataoff, rinfile,
     1   icon, ierror)

c******************************************************************
c******************************************************************
c******************************************************************
c
c Find all the necessary inputs, verify they exist, process
c their headers, and figure out how we're going to read what
c we need from them.
c
c******************************************************************
c
c Uspinput has an extensive argument list:
c
c mode (read / write mode):
c 'r'	open an already existing file for reading only
c 'rw'  open an already existing file for reading and writing
c 'ra'  open an already existing file for reading, writing, and
c       appending. Output will start at the end of the file
c       (although nothing will prevent you seeking back to the
c       beginning and overwriting the original traces if you wish).
c
c  Pass through will be disabled if the mode is 'rw' or 'ra'.
c
c
c clparam ("Input file name command line parameter"): this string
c is appended to '-N' and looked for on the command line to find
c an input file name for this input. If clparam is whitespace,
c then uspinput will look for '-N' as is usual practice for scalar
c USP programs that only need a single input.
c
c If an input file name is specified, then uspinput will attempt
c to open that file for input. If the file is not there, and the
c input is not being opened for scalar (non-MC) access, then uspinput
c will look for multiple subscripted files according to Leon's MC
c filename convention.
c
c If no input file name is specified on the command line, the place
c uspinput looks for input depends on whether the program is running
c under IKP or not. If not in IKP, the only possible fallback is to
c standard input. If in IKP, the fallback is to the "IKP single input"
c socket number specified by ikpsock (which is quite likely 0 meaning
c standard input, but doesn't have to be). If that is not connected
c or grounded, and the input is not scalar, then the final fallback
c is to one socket per component, with the socket number for each
c component being given by the formula
c (10 * Source Comp No + Receiver Comp No + ikpoffs).
c
c If you specify the same "clparam" for different inputs, uspinput
c will check the command line for repeat occurrences of the same
c argument. (For example, "-N file1 -N file2 -N file3".) Note this is
c very different from the behavior for "wclparam" (see below).
c
c
c wclparam ("Windowing command line parameter"): This string is appended
c to '-ns', '-ne', '-rs', '-re', '-wi', and '-pt' and the result looked
c for on the command line to determine how to "window" the data. If
c wclparam is entirely whitespace you'll get the usual USP defaults.
c '-ns' and '-ne' are the standard start trace, end trace in a gather
c parameters; '-rs' and '-re' are the standard start gather, end gather
c in a line parameters. Note the dimensions are in terms of the MC
c dataset; the "multicomponent" dimension is assumed to already be
c "projected out". This is the way the windowing parameters HAVE to
c work, otherwise the same window parameters would specify different
c parts of the data depending on what input format the MC was read in
c (i.e. Leon style, single input with MC axis fast, intermediate,
c or slow).
c
c If you specify wclparam for multiple input files, uspinput will only
c look for the associated windowing parameters on the command line ONCE.
c After that it will re-use whatever values it found before. (The idea
c is that if you specify one '-rs', say, you'll want it to apply to
c ALL the associated '-N' input files, no matter how many there happen
c to be.)
c
c
c dptw ("default pass through or window"): In the absence of anything
c on the command line, 0 means don't pass through unused data, 1 means
c do pass it through unchanged. '-wi(wclparam suffix)' on the command
c line overrides, and means to truncate anyway. '-pt(wclparam suffix)'
c overrides, and means to pass through anyway. If dptw is -1, that
c means don't even LOOK on the command line for any windowing
c parameters; the user program wants to handle ALL such
c windowing / pass-through etc details for this file for itself.
c
c
c ncomp is the number of components wanted. On return, ncomp is set to
c the number of components found. (This only happens if allcomps is
c not 0; see the allcomps entry below.) For scalar input, ncomp should
c be 1 (and see "complist" below).
c
c If "allcomps" is not 0, then ncomps can be 0. That means no
c components are being _specifically_ requested; up to ncompmax
c components that can be "found" will be returned, and ncomp updated
c appropriately. (Note ncomp will never be _smaller_ on return than
c it was on entry. If components specifically asked for were not found
c it's an error!)
c
c
c ncompmax is the maximum number of components that can be returned.
c
c
c complist is an array of dimension (2,ncompmax) specifying for each
c component:
c 1:  Source comp #
c 2:  Rec comp #
c For SCALAR input, set the source component number of the single
c component to -1; the IKP socket number to use for this scalar
c input inside IKP will be in "ikpsock" (see below).
c
c
c allcomps is either 0 or 1. 0 means "only tell me about the
c components I asked for". 1 means "tell me about all the components
c you could find". If allcomps is 1, then ncomp will be set equal to
c the number of components found, and complist will be updated to
c include the additional components that were found beyond the ones
c asked for. (The ncomp ones asked for will always be in the first
c ncomp slots, in the exact order the user program asked for them.)
c
c
c ikpoffs ("IKP socket offset") is a number to add to the standard
c socket numbers (10 * Source Comp No + Receiver Comp No + ikpoffs)
c for IKP purposes for this file. Usually ikpoffs will just be 0;
c the purpose of this parameter is to allow multiple multicomponent
c file input from inside IKP.
c
c
c ikpsock is the ikp socket number to look for a single input on.
c If this is not grounded, then all the input will be looked for
c here instead of in multiple sockets. Usually ikpsock will be 0,
c meaning standard input.
c
c
c seekalot guides uspinput in deciding how to buffer the data.
c
c A value of 0 means "this program will mostly plod through the data
c reading in order (alternated perhaps with passes through the data
c writing in order)". If you're going to mostly plod with only the
c occasional long seek, that would also count as 0. (Basically, if
c you would have left the sisio buffers on yourself, then seekalot
c should be 0.)
c
c uspseek is smart enough not to flush sisio's buffers unnecessarily,
c so don't worry about short forward seeks on reading: if that's
c all you're going to do, then seekalot can also be 0, because
c usprtrace will avoid the buffer flush by reading the intervening
c traces and discarding them for you. (Repeated short forward seeks
c on writing, however, count as bona-fide disruptive seeks. In that
c case seekalot should probably be 1.)
c
c A value of 1 means "this program will jump around within a gather
c a lot, but not often between different gathers". If the program
c is going to furiously alternate reading and writing nearby traces
c (for example, read a trace, write it back out, read the next trace,
c write it back out, etc) that also counts as 1. (This mode will
c turn off the sisio large buffers unless a gather will be contained
c within the span of uspio's in-core transpose buffer. In that case
c it leaves them on.)
c
c If the program will jump at random all over the data, even between
c different gathers and lines, then seekalot should be 2. This will
c always turn off sisio large buffers.
c
c
c seekstat can be 0, 1, or 2.
c
c 0 means "feel free to seek all you want".
c
c 1 means "you are reading or writing to a pipe or socket, so
c seeking backwards is not allowed". (It's OK to seek forwards.
c If you're reading it will read the intervening traces and
c discard them; if you're writing it will write null traces for
c you.)
c
c 2 means "pass-through is enabled so no seeking is allowed at all".
c (If pass through is enabled, just calling uspseek will generate
c an error.)
c
c
c un1, un2, un3, ud1 ("u" for "user") are as per SEPlib notation
c conventions:
c  un1 = number of samples in a trace (NumSmp),
c  ud1 = sampling rate (SmpInt)
c  un2 = number of traces in a gather (NumTrc)
c  un3 = number of gathers in a line  (NumRec)
c
c
c uf2, uf3 tell the program the trace number and gather number,
c respectively, of the first trace the user program sees. (These are
c not from any header; they are simply counted from the beginning of
c the file starting with 1, 2, 3, etc.) The user program may need
c this information to modify values from the line header specifying
c values like initial trace offsets, etc; they may no longer be
c correct after windowing.
c
c
c upt tells the user program whether the output will be windowed
c (0) or passed through (1). The user program may need to know
c this for modifying values in an outgoing line header. (The standard
c USP dimensional line-header keywords "NumTrc" and "NumRec" will
c automatically be set appropriately by "uspoutput"; the user program
c can't do that for itself because the correct values to use will
c depend on how the MC dimension is scrambled into the other
c dimensions, as specified by the line header keyword "MCTrSp". That
c information may not be available to the user program.)
c
c 
c lheader is the line header going with the first component
c (if the different components come from multiple files)
c or the line header of the input dataset
c (if the different components come from one MC file).
c
c
c lhedlength is the length in bytes of the line header.
c
c
c trlength is the length in full words (floats) of a single trace,
c including the trace header. You'll need to allocate
c trlength * sizeof(float) * the number of components (returned ncomp)
c bytes to store a single MC trace.
c
c
c trdataoff is the offset in full words (floats) from the start of
c the trace to the start of the trace data.
c
c rinfile is the "root input file name", which you may find useful
c if you want to construct a filename for auxilliary output. Note
c if there was no '-N' field on the command line uspinput will be
c forced to invent a name. If the input is from standard input, the
c name will be 'stdin'. If from a single IKP socket it will be of
c the form 'IKP.sN', where N is the socket number. If MC input
c through one or more IKP sockets, the name will be of the form
c 'IKP.iN', where N is the socket offset.
c
c icon is the "connection number", the number you'll have to use
c to refer to this connection later.
c
c
c ierror will be incremented once for each error that was found. (For
c example, it will be incremented once for each component that isn't
c found.) Unless ierror is zero this input cannot be used, and
c attempting to do so will result in an error.
c
c
c******************************************************************
c******************************************************************
c******************************************************************

#include <save_defs.h>
#include <f77/iounit.h>
#include <f77/lhdrsz.h>
#include <f77/sisdef.h>

#ifdef SUNSYSTEM
      implicit   none
#endif
      character  mode*(*), clparam*(*), wclparam*(*)
      integer    dptw
      integer    ncomp, ncompmax
      integer    complist(2,ncompmax), allcomps
      integer    ikpoffs, ikpsock, seekalot, seekstat
      integer    un1, un2, un3, uf2, uf3, upt
      real       ud1
      integer    lheader (1)
      integer    lhedlength, trlength, trdataoff
      character  rinfile*(*)
      integer    icon
      integer    ierror


      character  infile * 100, mcinfile * 100
      character  tempstring * 100, finamsv * 100
      character  estr * 100
      character  cmode * 10
      character  namstr * 120
      integer    ns, ne, irs, ire
      integer    luin, ncomp1, mcomp
      integer    ierr, istat
      integer    icomp, icomp2, ipipeno, mcinlen
      integer    ikp, iseek, ikpsinngr
      integer    ifile, jcomp, icount, numcmp, mctrsp
      integer    ifmt_MCList, l_MCList, ln_MCList
      integer    comps(MCLLen), luinsv
      pointer    (pluinsv, luinsv(2))
      integer    isrc, irec, dcomps(COMPRANGE,COMPRANGE)
      integer    toend


      integer    stat, statb(13)
      integer    ludsk, lutty, sisbufsz
      integer    argis, pipchk, usprlhdr, uspseek
      integer    in_ikp, nblen, usptba, uspsta, uspcmcmp
      integer    slength

      integer    ileft, iright

c
c    Scratch space to hold an unused header
c
      integer    slheader(SZLNHD)
      integer    slhedlength

      character  rname*(*)
      parameter (rname = 'uspinput()')

c******************************************************************
c
c Load all the global file information, saved in a common block
c
c******************************************************************
#include "uspinfo.h"



      ierror = 0


      write (LERR, *) '***************',
     1'****************************************************************'

      write (LERR, *)
      write (LERR, *) 'Summary of uspinput() calling parameters:'
      write (LERR, *)

      slength = nblen(mode)
      if (slength .eq. 0) then
          slength = 1
      endif
      write (LERR, *) 'mode=  ', '''', mode(1:slength), ''''

      slength = nblen(clparam)
      if (slength .eq. 0) then
          slength = 1
      endif
      write (LERR, *) 'clparam=  ', '''', clparam(1:slength), ''''

      slength = nblen(wclparam)
      if (slength .eq. 0) then
          slength = 1
      endif
      write (LERR, *) 'wclparam=  ', '''', wclparam(1:slength), ''''
 
      write (LERR, *) 'dptw=', dptw
      write (LERR, *) 'ncomp=', ncomp
      write (LERR, *) 'ncompmax=', ncompmax
      write (LERR, *) 'allcomps=', allcomps
      write (LERR, *) 'ikpoffs=', ikpoffs
      write (LERR, *) 'ikpsock=', ikpsock
      write (LERR, *) 'seekalot=', seekalot
 
      write (LERR, *)
      write (LERR, *)



c******************************************************************
c
c Check for various fatal bonehead errors
c
c******************************************************************

      if (curcon .lt. 0) then
       write (LERR, *) name(1:nblen(name)), ' ', rname,
     1              ': uspioinit() must be called before uspinput()!'
#ifdef SUNSYSTEM
       call flush(LERR)
#endif
       write (LER , *) name(1:nblen(name)), ' ', rname,
     1              ': uspioinit() must be called before uspinput()!'
       write (LER , *) name(1:nblen(name)),
     1              ': FATAL'
       stop
      endif


c
c    Increment the file number counter
c
      curcon = curcon + 1


      if (curcon .gt. maxfile) then
       write (LERR, *) name(1:nblen(name)), ' ', rname,
     1              ': More files opened than promised to uspioinit().'
#ifdef SUNSYSTEM
       call flush(LERR)
#endif
       write (LER , *) name(1:nblen(name)), ' ', rname,
     1              ': More files opened than promised to uspioinit().'
       write (LER , *) name(1:nblen(name)),
     1              ': FATAL'
       stop
      endif

      if (ncomp .gt. ncompmax) then
       write (LERR, *) name(1:nblen(name)), ' ', rname,
     1 ': More components requested (', ncomp,
     1 ') than room for (', ncompmax, ').'
#ifdef SUNSYSTEM
       call flush(LERR)
#endif
       write (LER , *) name(1:nblen(name)), ' ', rname,
     1 ': More components requested (', ncomp,
     1 ') than room for (', ncompmax, ').'
       write (LER , *) name(1:nblen(name)),
     1              ': FATAL'
       stop
      endif

      if (ncompmax .gt. maxcomp) then
       write (LERR, *) name(1:nblen(name)), ' ', rname,
     1 ': More components requested (', ncompmax,
     1 ') than max allowed by uspioinit() (', maxcomp, ').'
#ifdef SUNSYSTEM
       call flush(LERR)
#endif
       write (LER , *) name(1:nblen(name)), ' ', rname,
     1 ': More components requested (', ncompmax,
     1 ') than max allowed by uspioinit() (', maxcomp, ').'
       write (LER , *) name(1:nblen(name)),
     1              ': FATAL'
       stop
      endif

      if (ncomp .lt. 0 .or. (ncomp .eq. 0 .and. allcomps .eq. 0)) then
       write (LERR, *) name(1:nblen(name)), ' ', rname,
     1 ': Requested number of components (', ncomp, ') is invalid.'
#ifdef SUNSYSTEM
       call flush(LERR)
#endif
       write (LER , *) name(1:nblen(name)), ' ', rname,
     1 ': Requested number of components (', ncomp, ') is invalid.'
       write (LER , *) name(1:nblen(name)), ' ', rname,
     1              ': FATAL'
       stop
      endif

      if (ikpsock .lt. 0) then
       write (LERR, *) name(1:nblen(name)), ' ', rname,
     1 ': Invalid ikp single-input socket number (', ikpsock, ').'
#ifdef SUNSYSTEM
       call flush(LERR)
#endif
       write (LER , *) name(1:nblen(name)), ' ', rname,
     1 ': Invalid ikp single-input socket number (', ikpsock, ').'
       write (LER , *) name(1:nblen(name)), ' ', rname,
     1              ': FATAL'
       stop
      endif

      if (dptw .ne. DPTW_IGNORE .and.
     1    dptw .ne. DPTW_WINDOW .and.
     1    dptw .ne. DPTW_PASS) then
       write (LERR, *) name(1:nblen(name)), ' ', rname,
     1 ': Unknown default pass-through or window parameter (',
     1 dptw, ').'
#ifdef SUNSYSTEM
       call flush(LERR)
#endif
       write (LER , *) name(1:nblen(name)), ' ', rname,
     1 ': Unknown default pass-through or window parameter (',
     1 dptw, ').'
       write (LER , *) name(1:nblen(name)), ' ', rname,
     1              ': FATAL'
       stop
      endif

c
c    Make a copy of "mode" into "cmode"
c    (we may need to make it longer or rewrite it, you see)
c    and set the mode.
c

      toend = 0

      if (mode .eq. ' ' .or. mode .eq. 'r') then

          cmode = 'r'

          write (LERR, *) name(1:nblen(name)), ' ', rname,
     1    ': Input number ', curcon,
     1    ' is being opened for reading only.'

          canread(curcon) = 1

      else if (mode .eq. 'rw' .or. mode .eq. 'w' .or.
     1                                     mode .eq. 'wr') then

          cmode = 'r+'

          write (LERR, *) name(1:nblen(name)), ' ', rname,
     1    ': Input number ', curcon,
     1    ' is being opened for both reading and WRITING.'

          canread(curcon) = 1
          canwrite(curcon) = 1

      else if (mode .eq. 'ra' .or. mode .eq. 'ar' .or.
     1                                     mode .eq. 'a') then

          cmode = 'r+'

          write (LERR, *) name(1:nblen(name)), ' ', rname,
     1    ': Input number ', curcon,
     1    ' is being opened for reading, WRITING, and APPENDING.'

          canread(curcon) = 1
          canwrite(curcon) = 1
          append(curcon) = 1

c    After opening the file immediately position the pointer at
c    the end.

          toend = 1

      else

          write (LERR, *) name(1:nblen(name)), ' ', rname,
     1    ': Unknown read/write mode ''', mode(1:nblen(mode)), '''.'
#ifdef SUNSYSTEM
          call flush(LERR)
#endif
          write (LER , *) name(1:nblen(name)), ' ', rname,
     1    ': Unknown read/write mode ''', mode(1:nblen(mode)), '''.'
          write (LER , *) name(1:nblen(name)), ' ', rname,
     1    ': FATAL'
          stop

      endif



c******************************************************************
c
c Slightly less fatal bonehead errors and general commentary
c
c******************************************************************


c
c If there is only one component, and the source component number
c is -1, then it's a SCALAR trace.
c
      if (ncomp .eq. 1 .and.
     1    complist(COMPSRC,1) .eq. SCALAR) then

          write (LERR, *) name(1:nblen(name)), ' ', rname,
     1    ': Input number ', curcon,
     1    ' is scalar.'

c
c For consistency, set the receiver number to be the same.
c
          complist(COMPREC,1) = SCALAR

      else

c
c Otherwise, it's a multicomponent trace and the source and
c receiver component numbers must be between 1 and 3, inclusive.
c
          if (allcomps .eq. 0) then
              write (LERR, *) name(1:nblen(name)), ' ', rname,
     1        ': Input number ', curcon,
     1        ' has ', ncomp, ' component(s):'
          else
              if (ncomp .gt. 0) then
                  write (LERR, *) name(1:nblen(name)), ' ', rname,
     1            ': Input number ', curcon,
     1            ' begins with these ', ncomp, ' component(s):'
              else
                  write (LERR, *) name(1:nblen(name)), ' ', rname,
     1            ': Input number ', curcon,
     1            ' has an unspecified number of unknown components.'
              endif
          endif

          do icomp = 1, ncomp

              write (LERR, *) '        ',
     1        'Comp No. ', icomp, 
     1        ',   Src Comp. = ', complist(COMPSRC,icomp),
     1        ',   Rec Comp. = ', complist(COMPREC,icomp), ' .'

              if (complist(COMPSRC,icomp) .lt. 1 .or.
     1            complist(COMPSRC,icomp) .gt. COMPRANGE .or.
     1            complist(COMPREC,icomp) .lt. 1 .or.
     1            complist(COMPREC,icomp) .gt. COMPRANGE) then

                  write (LERR, *) name(1:nblen(name)), ' ', rname,
     1            ': Component numbers must be between 1 and ',
     1            COMPRANGE, '.'
#ifdef SUNSYSTEM
                  call flush(LERR)
#endif

                  write (LER , *) name(1:nblen(name)), ' ', rname,
     1            ': Component numbers must be between 1 and ',
     1            COMPRANGE, '.'

                  ierror = ierror + 1
              endif

c
c Also check that we don't have to return the same component
c multiple times. That would cause unnecessary complications
c when seeking later on.
c
              do jcomp = 1, icomp-1
                  if ((complist(COMPSRC,icomp) .eq.
     1                complist(COMPSRC,jcomp)) .and.
     1                (complist(COMPREC,icomp) .eq.
     1                complist(COMPREC,jcomp))) then

                      write (LERR, *) name(1:nblen(name)), ' ', rname,
     1                ': Component ', icomp,
     1                ' repeats component ', jcomp, '.'
#ifdef SUNSYSTEM
                      call flush(LERR)
#endif

                      write (LER , *) name(1:nblen(name)), ' ', rname,
     1                ': Component ', icomp,
     1                ' repeats component ', jcomp, '.'

                      ierror = ierror + 1
                  endif
              enddo

          enddo

      endif

      write (LERR, *)


      if (ierror .gt. 0) then
          goto 667
      endif


c*******************************************************************
c
c Save any other passed information we'll need later
c
c*******************************************************************


      if (nblen(clparam) .gt. 0) then
          namstr = clparam(1:nblen(clparam))
      else
          namstr = ' '
      endif


      nc(curcon) = ncomp
      icon = curcon

c
c The first component may be used to hold info about the single input
c file, even if no components at all are _specifically_ requested
c (ncomp = 0).
c
      ncomp1 = ncomp
      if (ncomp1 .lt. 1) then
          ncomp1 = 1
      endif

c
c This array is used to keep track of what components have been
c found. Initially, none have.
c
      do isrc = 1, COMPRANGE
          do irec = 1, COMPRANGE
              dcomps(irec,isrc) = 0
          enddo
      enddo

c
c We're required to find all the ones specifically asked
c for, so go back and mark those as taken.
c
      do icomp = 1, ncomp

          isrc = complist(COMPSRC,icomp)
          irec = complist(COMPREC,icomp)
          dcomps(irec,isrc) = 1

          LCOMP(COMPSRC,icomp,curcon) =
     1                      isrc
          LCOMP(COMPREC,icomp,curcon) =
     1                      irec
      enddo

c
c    Keep track of whether we can seek on all components
c    of this MC file.
c
      iseek = 1



c********************************************************************
c
c Collect some background information about our current environment.
c
c********************************************************************

c
c    Look for input file name '-Nsuffix', where clparam gives the
c    suffix. Spaces mean just look for '-N'. (Fortran doesn't allow
c    null strings, so we have to do a bit of kluging.)
c

      tempstring = '-N'//namstr
 
#ifdef DEBUG
      write (LERR, *) 'Searching command line for ',
     1    '''', tempstring(1:nblen(tempstring)), '''.'
#endif
 
      call argstr (tempstring(1:nblen(tempstring)), infile, ' ', ' ')

#ifdef DEBUG
      if (infile .eq. ' ') then
          write (LERR, *) 'Did not find it.'
      else
          write (LERR, *) 'Found ', '''',
     1        infile(1:nblen(infile)), '''.'
      endif
#endif


c
c    Copy across the file name found. rinfile is the root input
c    file name. (If it's stdin update it now. If it's an IKP socket,
c    we'll update it later.)
c
      rinfile = infile
      if (rinfile .eq. ' ') then
          rinfile = 'stdin'
      endif
 

c
c    Are we in IKP?
c
      ikp = in_ikp()
      ikpsinngr = 1

      if (ikp .ne. 0) then
          write (LERR, *) name(1:nblen(name)), ' ', rname,
     1                ': running under the control of IKP.'

c
c    In IKP-style multiple-socket input, the corresponding
c    single input (usually will be stdin, but can be any
c    socket number) should be connected to ground. If we've
c    found an input file name already, though, don't bother.
c
 
              if (infile .eq. ' ' .and. ikpsock .ge. 0) then
                  ikpsinngr = pipchk(ikpsock, istat)
              else
                  ikpsinngr = 0
              endif
 
          if (ikpsinngr .gt. 0) then
              if (ikpsock .eq. STDINNO) then
                  write (LERR, *) name(1:nblen(name)), ' ', rname,
     1            ': Single input (stdin) is connected.'
              else
                  write (LERR, *) name(1:nblen(name)), ' ', rname,
     1            ': Single input (socket ', ikpsock, ') is connected.'
              endif
          endif

      endif


c********************************************************************
c
c This next part is a mess of intertwined program flows!
c Here are all the different kinds of input that are supported:
c
c  [1.] IKP style, with one socket per component, with components
c       distinguished by "canonical socket numbering". (Canonical
c       socket numbering puts the Source Component number in the
c       10's place, the Receiver Component number in the 1's place,
c       with an additional 5 for output instead of input.)
c  [2.] IKP, but with the input all coming through one socket.
c  [3.] "Leon" style: one file per component, with the components
c       identified by filename suffix.
c  [4.] "Compatible" style: all components in one file,
c       with components distinguished via line header information
c       (at this point we don't yet have to worry about whether
c       the data is ordered component fast or component slow).
c       It's called "compatible" because "scalar" USP programs
c       will be able to do many sorts of reasonable processing
c       on such datasets.
c  [5.] Scalar (1 component, with source component number -1).
c
c   For inputs 2, 4, and 5 there are two sub-cases to consider: input
c   from standard in, or input from elsewhere (a named file or
c   an IKP socket).
c
c********************************************************************


      if (ikp .ne. 0 .and.
     1    infile .eq. ' ' .and.
     1    ikpsinngr .eq. 0 .and.
     1    (ncomp .ne. 1 .or. complist(COMPSRC, 1) .ne. SCALAR)
     1                                                        ) then
c--------------------------------------------------------------------
c    [1.] IKP-style multiple-socket input case
c
c    IKP mode is set, the associated single input is grounded,
c    and there isn't an input filename specified by -Nsomething
c    on the command line.
c--------------------------------------------------------------------
       multiin(curcon) = MI_MULT


       write (LERR, *) name(1:nblen(name)), ' ', rname,
     1 ': Using IKP-style input with ', ikpoffs, ' socket offset.'
#ifdef SUNSYSTEM
       call flush(LERR)
#endif

c
c     Construct a suitable "root input file name". Put the ikpoffs
c     into the name in case there are multiple IKP inputs.
c     We have to do a little extra work because Fortran has no
c     built-in way of left justifying and we won't allow a space
c     in the output file name.
c
       rinfile = 'IKP.i'

       write (tempstring, '(i8)') ikpoffs
       iright = nblen(tempstring)
       ileft = iright
       do icount = 1, iright
           if (tempstring(icount:icount) .ne. ' ') then
               ileft = icount
               goto 111
           endif
       enddo
 111   continue

       if (iright .gt. 0) then
           rinfile(6:6+(iright-ileft)) = tempstring(ileft:iright)
       endif


c
c    Look for all the requested components.
c

       do icomp = 1, ncomp
c
c    MC convention: If X is the source component number (1-3)
c    and Y is the receiver component number (1-3) then
c    unit number XY (as two digits) is the standard place to
c    read that component from. For more flexibility, however,
c    we also add an offset to the socket number. This is to allow
c    multiple multi-component files to be used, for example.
c
           isrc = complist(COMPSRC,icomp)
           irec = complist(COMPREC,icomp)

           ipipeno = isrc * 10 + irec + ikpoffs


c
c    If it's grounded (or not connected at all!) don't try to
c    open it.
c
           if (pipchk(ipipeno, istat) .gt. 0) then
               call sisfdfit (luin, ipipeno)
               estr = '(sisfdfit error)'
           else
               luin = -1
               if (istat .gt. 0) then
                   estr = '(grounded)'
               else
                   estr = '(unconnected)'
               endif
           endif


           if (luin .lt. 0) then
               write (LER , *) name(1:nblen(name)), ' ', rname,
     1         ': Error opening component ', complist(COMPSRC, icomp),
     1         complist(COMPREC, icomp),
     1         ', socket ', ipipeno, ' ', estr(1:nblen(estr))
               write (LERR, *) name(1:nblen(name)), ' ', rname,
     1         ': Error opening component ', complist(COMPSRC, icomp),
     1         complist(COMPREC, icomp),
     1         ', socket ', ipipeno, ' ', estr(1:nblen(estr))
#ifdef SUNSYSTEM
               call flush(LERR)
#endif
 
               LUIOMC(icomp,curcon) = -1
               ierror = ierror + 1
       write(FINAME(icomp,curcon),*)
     1         'IKP spigot ', ipipeno, ' ', estr(1:nblen(estr))

           else

               LUIOMC(icomp,curcon) = luin
       write(FINAME(icomp,curcon),*)
     1         'IKP spigot ', ipipeno
c
c    If any of them aren't a disk, then we can't seek
c
               if (ludsk(luin) .eq. 0) then
                   iseek = 0
               endif
           endif

       enddo


       if (ierror .eq. 0 .and. allcomps .ne. 0) then
c
c    Now look to see if there are any EXTRA connected spigots
c    to find out there.
c
           icomp = ncomp

           do isrc = 1, COMPRANGE
               do irec = 1, COMPRANGE
c
c    Skip it if we've already done it.
c
                   if (dcomps(irec,isrc) .eq. 0) then

                       ipipeno = isrc * 10 + irec + ikpoffs

                       if (pipchk(ipipeno, istat) .gt. 0) then
                           call sisfdfit (luin, ipipeno)
                       else
                           luin = -1
                       endif
 
 
                       if (luin .ge. 0) then
c
c    We found one! Add it to the list.
c
                           if (icomp .ge. ncompmax) then
c
c    Uh oh, no room left for this one.
c
                               write (LER , *)
     1                             name(1:nblen(name)), ' ', rname,
     1 ': No space for found comp. ', isrc, irec, ',',
     1 ' IKP spigot ', ipipeno, '.'
                               write (LERR, *)
     1                             name(1:nblen(name)), ' ', rname,
     1 ': No space for found comp. ', isrc, irec, ',',
     1 ' IKP spigot ', ipipeno, '.'
#ifdef SUNSYSTEM
                               call flush(LERR)
#endif
                           else
                               icomp = icomp + 1

       LUIOMC(icomp,curcon) = luin
       write(FINAME(icomp,curcon),*)
     1                         'IKP spigot ', ipipeno, ' (found)'
c
c    If any of them aren't a disk, then we can't seek
c
                               if (ludsk(luin) .eq. 0) then
                                   iseek = 0
                               endif

                               dcomps(irec,isrc) = 1
                               complist(COMPSRC,icomp) = isrc
                               complist(COMPREC,icomp) = irec
       LCOMP(COMPSRC,icomp,curcon) =
     1                         complist(COMPSRC,icomp)
       LCOMP(COMPREC,icomp,curcon) =
     1                         complist(COMPREC,icomp)
                           endif
                       endif

                   endif
               enddo
           enddo

           ncomp = icomp
           nc(curcon) = ncomp

           if (ncomp .eq. 0) then
               write (LER , *) name(1:nblen(name)), ' ', rname,
     1         ': No components found.'
               write (LERR, *) name(1:nblen(name)), ' ', rname,
     1         ': No components found.'
#ifdef SUNSYSTEM
               call flush(LERR)
#endif
               ierror = ierror + 1
           endif

       endif


      else if (ikp .ne. 0 .and.
     1         infile .eq. ' ' .and.
     1         ikpsinngr .gt. 0 .and. ikpsock .ne. STDINNO) then
c--------------------------------------------------------------------
c    [2.] IKP-style single-socket input case
c
c    IKP mode is set, there isn't an input filename specified
c    by -Nsomething on the command line, and the associated
c    single input is NOT grounded, nor is it missing a connector
c    altogether. We should read from the single-input IKP socket.
c    Note if the input is standard in, we will instead handle
c    it in the catch-all case below; no special IKP processing
c    is required for that case.
c--------------------------------------------------------------------
       multiin(curcon) = MI_PIPE

       write (LERR, *) name(1:nblen(name)), ' ', rname,
     1 ': Using IKP-style single-socket input for socket ',
     1 ikpsock, '.'
#ifdef SUNSYSTEM
       call flush(LERR)
#endif

c
c     Construct a suitable "root input file name". Put the ikpsock
c     into the name in case there are multiple IKP inputs.
c     We have to do a little extra work because Fortran has no
c     built-in way of left justifying and we won't allow a space
c     in the output file name.
c
       rinfile = 'IKP.s'

       write (tempstring, '(i8)') ikpsock
       iright = nblen(tempstring)
       ileft = iright
       do icount = 1, iright
           if (tempstring(icount:icount) .ne. ' ') then
               ileft = icount
               goto 112
           endif
       enddo
 112   continue

       if (iright .gt. 0) then
           rinfile(6:6+(iright-ileft)) = tempstring(ileft:iright)
       endif

c
c    If we got here, there must be a non-grounded input file
c    to read from. So no need to check again.
c
       ipipeno = ikpsock
       call sisfdfit (luin, ipipeno)

       if (luin .lt. 0) then

           write (LER , *) name(1:nblen(name)), ' ', rname,
     1     ': Error from sisfdfit opening socket number ', ipipeno, '.'
           write (LERR, *) name(1:nblen(name)), ' ', rname,
     1     ': Error from sisfdfit opening socket number ', ipipeno, '.'
#ifdef SUNSYSTEM
           call flush(LERR)
#endif
 
           do icomp = 1, ncomp1
       LUIOMC(icomp,curcon) = -1
       write(FINAME(icomp,curcon),*)
     1         'IKP spigot ', ipipeno, ' (sisfdfit error)'
           enddo

           ierror = ierror + 1

       else

           do icomp = 1, ncomp1
       LUIOMC(icomp,curcon) = luin
       write(FINAME(icomp,curcon),*)
     1         'IKP spigot ', ipipeno
           enddo

c
c    If it's not a disk, then we can't seek
c
           if (ludsk(luin) .eq. 0) then
               iseek = 0
           endif
       endif


      else
c--------------------------------------------------------------------
c    NOT the IKP-style case.
c
c    See if the single input file exists: attempt to "stat" it.
c    (Note we can't simply call getln and look for an error message;
c    if getln can't open a file it aborts the entire program!)
c
c--------------------------------------------------------------------

       if (ikp .ne. 0) then
           write (LERR, *)  name(1:nblen(name)), ' ', rname,
     1     ': Falling through to non-IKP-style input.'
       endif

c
c    Don't try to stat standard input (we don't need to anyway;
c    standard input always exists).
c
       ierr = 0
       statb(8) = 1
       if (infile .ne. ' ') then
           ierr = stat (infile, statb)
       endif
 
       if (ierr .ne. 0) then
           if (ncomp .ne. 1 .or. complist(COMPSRC, 1) .ne. SCALAR) then
c---------------------------------------------------------------------
c
c    We didn't find that file, but perhaps this is
c    "Leon-style" input, with each component in a separate
c    file? So don't give up yet.
c
c---------------------------------------------------------------------
               write (LERR, *)  name(1:nblen(name)), ' ', rname,
     1                          ': Could not find ',
     2                          infile(1:nblen(infile)), ';',
     3                          ' checking for subscripted files.'


c--------------------------------------------------------------------
c     [3.] Leon-style Multiple Input File case
c--------------------------------------------------------------------
               multiin(curcon) = MI_MULT

c
c    That file wasn't there... see if Leon-style subscripted ones are.
c
               mcinfile = infile
               mcinlen = nblen(mcinfile)
               mcinfile(mcinlen+1:mcinlen+1) = '.'
 
               do icomp = 1, ncomp
 
                   write(mcinfile(mcinlen+2:mcinlen+2),'(i1)')
     1                   complist(COMPSRC, icomp)
                   write(mcinfile(mcinlen+3:mcinlen+3),'(i1)')
     1                   complist(COMPREC, icomp)
 
                   statb(8) = 1
                   ierr = stat (mcinfile, statb )
  
                   if (ierr .ne. 0) then
                       luin = -1
                   else
                       call getln(luin , mcinfile, cmode, -1)
                   endif
 
 
                   if (luin .lt. 0) then
                       write (LERR, *)
     1                      name(1:nblen(name)), ' ', rname,
     1                      ': Could not open ',
     1                      mcinfile(1:nblen(mcinfile)), ', ',
     1                      ' Component ', complist(COMPSRC, icomp),
     1                      complist(COMPREC, icomp)
#ifdef SUNSYSTEM
                       call flush(LERR)
#endif
                       write (LER , *)
     1                       name(1:nblen(name)), ' ', rname,
     1                       ': Could not open ',
     1                       mcinfile(1:nblen(mcinfile)), ', ',
     1                       ' Component ', complist(COMPSRC, icomp),
     1                       complist(COMPREC, icomp)
 
       LUIOMC(icomp,curcon) = -1
       write(FINAME(icomp,curcon),*)
     1                      mcinfile(1:nblen(mcinfile)),
     1                      ' ', '(error)'
                       ierror = ierror + 1

                   else

c
c    If it's a zero-length file that doesn't count!
c
                       if (statb(8) .eq. 0) then
                           write (LERR, *)
     1                       name(1:nblen(name)), ' ', rname,
     1                       ': Zero length file ',
     1                       mcinfile(1:nblen(mcinfile)), ', ',
     1                       ' Component ', complist(COMPSRC, icomp),
     1                       complist(COMPREC, icomp)
#ifdef SUNSYSTEM
                           call flush(LERR)
#endif
                           write (LER , *)
     1                        name(1:nblen(name)), ' ', rname,
     1                        ': Zero length file ',
     1                        mcinfile(1:nblen(mcinfile)), ', ',
     1                        ' Component ', complist(COMPSRC, icomp),
     1                        complist(COMPREC, icomp)

                           luin = -1
       LUIOMC(icomp,curcon) = -1
       write(FINAME(icomp,curcon),*)
     1                       mcinfile(1:nblen(mcinfile)),
     1                       ' ', '(zero length)'
                           ierror = ierror + 1

                       else

       LUIOMC(icomp,curcon) = luin
       write(FINAME(icomp,curcon),*)
     1                          mcinfile(1:nblen(mcinfile))
c
c    If any of them aren't a disk, then we can't seek
c
                           if (ludsk(luin) .eq. 0) then
                               iseek = 0
                           endif
                       endif

                   endif
 
               enddo


               if (ierror .eq. 0 .and. allcomps .ne. 0) then
c
c    Now look to see if there are any EXTRA files matching the
c    pattern out there.
c
                   icomp = ncomp

                   do isrc = 1, COMPRANGE
                       do irec = 1, COMPRANGE
c
c    Skip it if we've already done it.
c
                           if (dcomps(irec,isrc) .eq. 0) then

                               write(mcinfile(mcinlen+2:mcinlen+2),
     1                               '(i1)') isrc
                               write(mcinfile(mcinlen+3:mcinlen+3),
     1                               '(i1)') irec
 
                               statb(8) = 1
                               ierr = stat (mcinfile, statb )
  
                               if (ierr .ne. 0) then
                                   luin = -1
                               else
                                   call getln(luin, mcinfile, cmode, -1)
                               endif
 
 
                               if (luin .ge. 0 .and.
     1                                           statb(8) .ne. 0) then
c
c    We found one! Add it to the list.
c
                                   if (icomp .ge. ncompmax) then
c
c    Uh oh, no room left for this one.
c
                                       write (LER , *)
     1                                 name(1:nblen(name)), ' ', rname,
     1 ': No space for found comp. ', isrc, irec, ',',
     1 ' file ''', mcinfile(1:nblen(mcinfile)), '''.'
                                       write (LERR, *)
     1                                 name(1:nblen(name)), ' ', rname,
     1 ': No space for found comp. ', isrc, irec, ',',
     1 ' file ''', mcinfile(1:nblen(mcinfile)), '''.'
#ifdef SUNSYSTEM
                                       call flush(LERR)
#endif
                                   else

                                       icomp = icomp + 1


       LUIOMC(icomp,curcon) = luin
       write(FINAME(icomp,curcon),*)
     1                                 mcinfile(1:nblen(mcinfile)),
     1                                 ' (found)'
c
c    If any of them aren't a disk, then we can't seek
c
                                       if (ludsk(luin) .eq. 0) then
                                           iseek = 0
                                       endif


                                       dcomps(irec,isrc) = 1
                                       complist(COMPSRC,icomp) = isrc
                                       complist(COMPREC,icomp) = irec
       LCOMP(COMPSRC,icomp,curcon) =
     1                                 complist(COMPSRC,icomp)
       LCOMP(COMPREC,icomp,curcon) =
     1                                 complist(COMPREC,icomp)

                                   endif

                               endif
                           endif

                       enddo
                   enddo

                   ncomp = icomp
                   nc(curcon) = ncomp


                   if (ncomp .eq. 0) then
                       write (LER , *) name(1:nblen(name)), ' ', rname,
     1                 ': No components found.'
                       write (LERR, *) name(1:nblen(name)), ' ', rname,
     1                 ': No components found.'
#ifdef SUNSYSTEM
                       call flush(LERR)
#endif
                       ierror = ierror + 1
                   endif

               endif

           else
c---------------------------------------------------------------------
c
c    However, if there is only ONE COMPONENT, and its source
c    component number is -1, this is the SCALAR case. In
c    that case if we didn't find the file it's an error!
c
c---------------------------------------------------------------------
               write (LERR, *)  name(1:nblen(name)), ' ', rname,
     1                          ': Could not find scalar input ', '''',
     2                          infile(1:nblen(infile)), '''.'
#ifdef SUNSYSTEM
               call flush(LERR)
#endif

               write (LER , *)  name(1:nblen(name)), ' ', rname,
     1                          ': Could not find scalar input ', '''',
     2                          infile(1:nblen(infile)), '''.'

               luin = -1
               multiin(curcon) = MI_PIPE
       LUIOMC(1,curcon) = luin

       write(FINAME(1,curcon),*)
     1         infile(1:nblen(infile)), ' (not found)'

               ierror = ierror + 1

           endif


       else
c----------------------------------------------------------------------
c
c     [4.], [5.]
c     (and [2.] also, when IKP single input is stdin)
c
c     The Single Input File case:
c
c     We FOUND the input file!!!
c     (Note standard input is always connected to SOMETHING!)
c
c     We've got a single input file to read from. If "infile"
c     is blank, the input file must be standard in. (The case
c     of a single input IKP socket not from standard in was
c     already handled in another case above.)
c
c----------------------------------------------------------------------

           if (ncomp .ne. 1 .or. complist(COMPSRC, 1) .ne. SCALAR) then
c
c [4.] MC case
c
               if (infile .ne. ' ') then
                   write (LERR, *) name(1:nblen(name)), ' ', rname,
     1                           ': Found ''',
     2                           infile(1:nblen(infile)), ''';',
     3                           ' attempting to open it for MC I/O.'
               else
                   if (ikp .eq. 0) then
                        write (LERR, *) name(1:nblen(name)), ' ', rname,
     1                  ': Attempting to open stdin for MC input.'
                   else
                        write (LERR, *) name(1:nblen(name)), ' ', rname,
     1                  ': Opening stdin for MC input, instead of ',
     1                  'multiple IKP sockets.'
                   endif
               endif
           else
c
c [5.] Scalar case
c
               if (infile .ne. ' ') then
                   write (LERR, *) name(1:nblen(name)), ' ', rname,
     1                      ': Found ''',
     2                      infile(1:nblen(infile)), ''';',
     3                      ' attempting to open it for scalar I/O.'
               else
                    write (LERR, *) name(1:nblen(name)), ' ', rname,
     1              ': Attempting to open stdin for scalar input.'
               endif
           endif

#ifdef SUNSYSTEM
           call flush(LERR)
#endif

c
c [4.] and [5.] together
c

           estr = ' '

c
c    Now that we know it's there, let's try to connect to it.
c
           call getln(luin , infile, cmode, STDINNO)

c
c    Just in case the user goofed and didn't connect standard input
c    to anything... let's check and make sure it's not connected to
c    their terminal!
c
           if (luin .ge. 0) then
               if (lutty(luin) .ne. 0) then
                   write (LERR, *) name(1:nblen(name)), ' ', rname,
     1             ': This input is connected to a terminal!'
#ifdef SUNSYSTEM
                   call flush(LERR)
#endif
                   write (LER , *) name(1:nblen(name)), ' ', rname,
     1       ': You are attempting to read usp data from a terminal!'

                   estr = '(a terminal)'
                   ierror = ierror + 1

               else if (luin. eq. 0 .and.
     1                  ikp .ne. 0 .and. ikpsinngr .eq. 0) then
c
c    Another possible error:
c    if in IKP and we're reading from stdin (possibly because
c    the requested input is scalar), make sure it's not grounded.
c
                   write (LERR, *) name(1:nblen(name)), ' ', rname,
     1             ': This input is connected to /dev/null!'
#ifdef SUNSYSTEM
                   call flush(LERR)
#endif
                   write (LER , *) name(1:nblen(name)), ' ', rname,
     1       ': You are attempting to read usp data from /dev/null!'

                   estr = '(/dev/null)'
                   ierror = ierror + 1

               else if (statb(8) .eq. 0) then
c
c    And yet another possible error:
c    the file is zero bytes long.
c
                   write (LERR, *) name(1:nblen(name)), ' ', rname,
     1             ': This input file is empty!'
#ifdef SUNSYSTEM
                   call flush(LERR)
#endif
                   write (LER , *) name(1:nblen(name)), ' ', rname,
     1     ': You are attempting to read usp data from an empty file!'

                   estr = '(zero length)'
                   ierror = ierror + 1

               endif
           else
c
c    We returned from getln with an error. (Somebody must have
c    turned off the automatic abort on not being able to open
c    a file! Normally if getln has problems, it will abort the
c    whole program right then and there.)
c
               write (LERR, *) name(1:nblen(name)), ' ', rname,
     1         ': Error from getln().'
#ifdef SUNSYSTEM
               call flush(LERR)
#endif
               write (LER , *) name(1:nblen(name)), ' ', rname,
     1         ': Error from getln().'

               estr = '(getln error)'
               ierror = ierror + 1

           endif

c
c    See whether we can seek or not
c
           if (luin .ge. 0) then
               if (ludsk(luin) .eq. 0) then
                   iseek = 0
               endif
           endif

c
c    For now, set everything up according to the assumption that
c    we won't be able to open the file multiple times. (We will
c    try to "upgrade" to MI_MOPEN later.)
c
           multiin(curcon) = MI_PIPE

           do icomp = 1, ncomp1

       LUIOMC(icomp,curcon) = luin

               if (estr .eq. ' ') then
                   if (infile .ne. ' ') then
       write(FINAME(icomp,curcon),*)
     1                 infile(1:nblen(infile))
                   else
       write(FINAME(icomp,curcon),*)
     1                 'stdin'
                   endif
               else
                   if (infile .ne. ' ') then
       write(FINAME(icomp,curcon),*)
     1                 infile(1:nblen(infile)), ' ', estr(1:nblen(estr))
                   else
       write(FINAME(icomp,curcon),*)
     1                 'stdin', ' ', estr(1:nblen(estr))
                   endif
               endif

           enddo


       endif
      endif



c********************************************************************
c********************************************************************
c
c    WHEW! We're done tracking down all the inputs!!!!!
c
c    Now we've got them all, do some double-checking on them.
c
c********************************************************************
c********************************************************************


c
c    Call uspmic to see if we're trying to read more than one input
c    file from the same logical unit... that can't possibly work.
c    It will also check to see if we're actually able to read and
c    write as required for our current file. If we can't, it will
c    update the values of "canread" and "canwrite" accordingly.
c    (It can also increment ierror, depending on a switch in the code,
c    in which case it's a fatal error.)
c

      call uspmic(ierror)

      if (ierror .eq. 0 .and.
     1                iseek .eq. 0 .and. append(curcon) .ne. 0) then

          ierror = ierror + 1

          write (LERR, *) name(1:nblen(name)), ' ', rname,
     1    ': Sorry, must be able to seek to append to a file.'
#ifdef SUNSYSTEM
          call flush(LERR)
#endif
          write (LER , *) name(1:nblen(name)), ' ', rname,
     1    ': Sorry, must be able to seek to append to a file.'

      endif

c********************************************************************
c
c Just give up and return if we had any serious errors.
c
c********************************************************************

      write (LERR,*)

      if (ierror .gt. 0) then
          write (LERR, *) name(1:nblen(name)), ' ', rname,
     1    ': ', ierror,
     1    ' serious error(s) occurred opening one or more files.'
          write (LERR, *) name(1:nblen(name)), ' ', rname,
     1    ': Giving up on opening input number ', curcon, '.'

          goto 666
      endif


      luseek(curcon) = iseek

      if (iseek .eq. 0) then
           write (LERR, *) name(1:nblen(name)), ' ', rname,
     1                     ': No seeking backwards on input ', curcon
      else
           write (LERR, *) name(1:nblen(name)), ' ', rname,
     1                     ': We can seek on input ', curcon
      endif



c**********************************************************************
c
c See if we need to turn the large SISIO buffers off for efficiency.
c
c For now, if seekalot is either 1 or 2 then turn off large buffers.
c NOTE! We may want to change this if the logic in sisio for combining
c seeks and large buffers is improved. ( ??????  !!!!!! )
c Note there is one other place further down (when we multiply open
c a single file) where we also check the value of seekalot and decide
c whether to turn off buffering or not.
c There is also a place further down where if seekalot is 1 and the
c transpose mode is IO_MEMBUF, then we turn buffering back ON again!
c (After having read the line header.)
c
c**********************************************************************

      if (seekalot .ne. 0) then

          write (LERR, *) name(1:nblen(name)), ' ', rname,
     1        ': Turning off large buffers for input ', curcon

          if (multiin(curcon) .eq. MI_PIPE .and.
     1                                       nc(curcon) .gt. 0) then
              mcomp = 1
          else
              mcomp = nc(curcon)
          endif

          do icomp = 1, mcomp

              if (
     1 LUIOMC(icomp,curcon)
     1                             .ge. 0) then

                  call sislgbuf(
     1 LUIOMC(icomp,curcon)
     1                                  , 'off')
              endif
          enddo

      endif

c**********************************************************************
c
c    Set the trace header length and trace data offset.
c
c    For now, we will set these here in the old hard-wired way.
c    EVERYWHERE ELSE we should refer only to trhlen(curcon), so
c    later on it will be easy to read the trace header length from
c    the line header. (In the "Process line header(s)" block of
c    code which immediately follows this one.)
c
c**********************************************************************

      trhlen(curcon) = ITRWRD
      trdataoff = trhlen(curcon)



c*******************************************************************
c*******************************************************************
c
c Process line header(s).
c
c*******************************************************************
c*******************************************************************

c
c Set default numcmp (number of components in the input file) and
c mctrsp (how the components are layed out in the input file):
c
c    Assume the input file is Scalar until we discover otherwise!
c
      numcmp = 0
      mctrsp = MC_SCALAR


      if (multiin(curcon) .eq. MI_PIPE .or.
     1    multiin(curcon) .eq. MI_MOPEN) then
c*******************************************************************
c
c All the cases that require reading only one file:
c
c   IKP reading from a single socket
c   MC input from a single file, whether multiply opened or not
c   Scalar input
c
c*******************************************************************

c
c There is only one line header to read for this input. Read it.
c
          icomp = 1
          if (0 .eq. usprlhdr(luin,
     1                        icomp, lheader, lhedlength, ierror)) then
              if (ncomp .ne. 1 .or.
     1            complist(COMPSRC, 1) .ne. SCALAR) then

c************************************
c
c MC from one input file case
c
c************************************

c
c    How many components are there in this file?
c
                  call saver(lheader, 'NumCmp', numcmp, LINEHEADER)

                  if (numcmp .gt. MCLLen) then
                    write (LERR, *) name(1:nblen(name)), ' ', rname,
     1              ': Too many components (', numcmp, ') in ', '''',
     1 FINAME(icomp,curcon)
     1 (1:nblen(FINAME(icomp,curcon))),
     1 '''.'
                    write (LERR, *) name(1:nblen(name)), ' ', rname,
     1              ': Truncating to ', MCLLen, '.'

#ifdef SUNSYSTEM
                    call flush(LERR)
#endif
                    write (LER , *) name(1:nblen(name)), ' ', rname,
     1              ': Too many components (', numcmp, ') in ', '''',
     1 FINAME(icomp,curcon)
     1 (1:nblen(FINAME(icomp,curcon))),
     1 '''.'

                    numcmp = MCLLen
                  endif


c
c    If there are components in this file, then find out what
c    components they are.
c
                  if (numcmp .gt. 0) then

                      call savelu('MCList', ifmt_MCList, l_MCList,
     1                         ln_MCList, LINEHEADER)

                      call saver2(lheader, ifmt_MCList, l_MCList,
     1                           numcmp, comps, LINEHEADER)

                  endif

c
c    Find out how the dataset is layed out and see where the
c    components we want are among the ones present.
c
                  call saver(lheader, 'MCTrSp', mctrsp, LINEHEADER)
                  call uspmctrsp(mctrsp, numcmp, ierror)

                  if (ierror .eq. 0) then

                      finamsv =
     1 FINAME(1,curcon)

                      do icomp = 1, ncomp
                          ierr = uspcmcmp(numcmp, comps, icomp,
     1                            complist(COMPSRC,icomp),
     1                            complist(COMPREC,icomp),
     1                            1, finamsv, complist, ncompmax, luin,
     1                            ierror)
                      enddo

                  endif

                  if (ierror .eq. 0 .and. allcomps .ne. 0 .and.
     1            (ncomp .ne. 1 .or. complist(COMPSRC, 1) .ne. SCALAR)
     1                                                         ) then
c
c    Now look to see what EXTRA components are there. We could
c    just take whatever's there, but for compatibility with how we
c    handled the multiple input-file case we'll do it this way.
c    (The idea being that what components you get and what order
c    they arrive in shouldn't depend _at all_ on the style of input!)
c
                      icomp = ncomp

                      do isrc = 1, COMPRANGE
                          do irec = 1, COMPRANGE
c
c    Skip it if we've already done it.
c
                              if (dcomps(irec,isrc) .eq. 0) then

                                  ierr = 0
                                  if (0 .eq. uspcmcmp(numcmp, comps,
     1                            icomp+1, isrc, irec,
     1                            0, finamsv, complist, ncompmax, luin,
     1                            ierr)) then

c
c    We found one! Add it to the list.
c
                                      dcomps(irec,isrc) = 1
                                      icomp = icomp + 1

                                  endif
                              endif

                          enddo
                      enddo

                      ncomp = icomp
                      nc(curcon) = ncomp

                      if (ncomp .eq. 0) then
                          write (LER , *)
     1                    name(1:nblen(name)), ' ', rname,
     1                    ': No components found.'
                          write (LERR, *)
     1                    name(1:nblen(name)), ' ', rname,
     1                    ': No components found.'
#ifdef SUNSYSTEM
                          call flush(LERR)
#endif
                          ierror = ierror + 1

                      endif

                  endif

c
c    numcmp = 0 means it was scalar input, not that it was empty!
c    Scalar input has one (unspecified, matches anything) component.
c    Now that we've matched it, set numcmp appropriately instead of
c    carrying along a special case.
c
                  if (numcmp .eq. 0) then
                      numcmp = 1
                  endif


                  if (ierror .eq. 0) then
c
c    Find the optimal order to read the components we need.
c
                      icount = 0
                      do icomp = 1, numcmp
                          do jcomp = 1, ncomp
                              if (
     1 LUCN(jcomp,curcon)
     1                                              .eq. icomp) then
                                  icount = icount + 1
       LUORDER(icount,curcon) = jcomp
                                  goto 200
                              endif
                          enddo
 200                      continue
                      enddo


c
c This shouldn't happen! But just in case someone else later
c changes the code, we'll check for it.
c
                      if (icount .ne. ncomp) then
                          ierror = ierror + 1
                          write (LERR, *)
     1                    name(1:nblen(name)), ' ', rname,
     1                    ': Not all components found.'

#ifdef SUNSYSTEM
                          call flush(LERR)
#endif
                          write (LER , *)
     1                    name(1:nblen(name)), ' ', rname,
     1                    ': Not all components found.'
                      endif

                  endif

c
c If we are appending, don't allow any windowing, including windowing
c over the component axis.
c

                  if (ierror .eq. 0 .and.
     1                    append(curcon) .ne. 0 .and.
     1                    ncomp .ne. numcmp) then

                          ierror = ierror + 1

                          write (LERR, *)
     1                    name(1:nblen(name)), ' ', rname,
     1                    ': Sorry, cannot window over the component',
     1                    ' axis when appending.'

#ifdef SUNSYSTEM
                          call flush(LERR)
#endif
                          write (LER , *)
     1                    name(1:nblen(name)), ' ', rname,
     1                    ': Sorry, cannot window over the component',
     1                    ' axis when appending.'

                  endif


c***********************************************************************
c
c Now let's figure out how we're going to transpose this single MC file!
c
c Here are the possibilities, in order of preference:
c
c    lustatus    multiin
c ------------------------------------------------------------------
c 1) IO_OK,      MI_PIPE:  It doesn't need to be transposed at all!
c 2) IO_SKIP,    MI_PIPE:  We can transpose it by ignoring some
c                          parts (skipping only forward)
c 3) IO_MEMBUF,  MI_PIPE:  We can transpose it in a memory buffer
c 4) IO_SKIP,    MI_MOPEN: We can open the same file several times
c 5) IO_SEEK,    MI_PIPE:  We can transpose it by seeking on disk
c 6) IO_NOK             :  Uh oh. The user is going to be disappointed
c
c***********************************************************************


c
c (1)
c Check for an important special case: the traces happen to be
c in exactly the order we need to read them (perhaps after rearranging
c what order we want to read things in!). In this happy circumstance
c we can dispense with seeking, buffering, worrying, etc, and simply
c use the traces as they come. (Since we don't allow repeated input
c components, all we have to check for is that the number of components
c there is equal to the number of components asked for, and we didn't
c lack any.)
c
                  if (ierror .eq. 0 .and.
     1                ncomp .eq. numcmp .and.
     1                lun1(curcon) .eq. 1) then

                      lustatus(curcon) = IO_OK

                  endif


c
c (2)
c Well, perhaps we can still guarantee that all required seeks
c will be _forward_ ones. Either the data is component fast (but there
c are more components there than we are going to use), or the data is
c not component fast but we only need one component.
c
                  if (ierror .eq. 0 .and.
     1                lustatus(curcon) .eq. IO_NOK .and.
     1                (lun1(curcon) .eq. 1 .or. ncomp .eq. 1)) then

                      lustatus(curcon) = IO_SKIP

                  endif


c
c (3)
c Try setting up a transpose buffer in memory.
c Make sure the transpose area required is of reasonable size, and the
c user isn't planning on jumping around between gathers much.
c (If seeking isn't possible, then try it this way even if they are
c going to jump around a lot, as it's their only remaining possibility!)
c

                  if (ierror .eq. 0 .and.
     1                lustatus(curcon) .eq. IO_NOK .and.
     1                lun1(curcon)*lunc(curcon)*
     1                (n1(curcon)+trhlen(curcon))*SZSMPD
     1                .le. MAXTBUFSZ
     1                .and.
     1                (seekalot .le. 1 .or. luseek(curcon) .eq. 0)
     1                                                        ) then


                      if (usptba(curcon,
     1                     lun1(curcon)*lunc(curcon)*
     1                     (n1(curcon)+trhlen(curcon))*SZSMPD,
     1                     maxfile, ierror) .ne. 0) then

                          write(LERR,*)
     1                    name(1:nblen(name)), ' ', rname,
     1                    ': Unable to allocate ',
     1                    lun1(curcon)*lunc(curcon)*
     1                    (n1(curcon)+trhlen(curcon))*SZSMPD,
     1                    ' bytes for transpose buffer.'
#ifdef SUNSYSTEM
                          call flush(LERR)
#endif
                      else

                          write(LERR,*)
     1                    name(1:nblen(name)), ' ', rname,
     1                    ': allocating ',
     1                    lun1(curcon)*lunc(curcon)*
     1                    (n1(curcon)+trhlen(curcon))*SZSMPD,
     1                    ' bytes,'
                          write(LERR,*) '               ',
     1                    lun1(curcon)*lunc(curcon), ' traces ',
     1                    'for transpose buffer ',
     1                    curcon, '.'

                          lustatus(curcon) = IO_MEMBUF

c
c If seekalot is "1", then we should turn the buffers back on again.
c Seeking within a gather will be handled within the transpose
c buffer, so won't matter to sisio, so the buffers can be left on.
c
                          if (seekalot .eq. 1) then

                              write (LERR, *)
     1                         name(1:nblen(name)), ' ', rname,
     1  ': Turning large buffers on again for input ', curcon

                              call sislgbuf(luin, 'on')

                          endif

                      endif

                  endif


c
c (4)
c If we can open the same input file multiple times, try doing that.
c There are no outstanding errors
c We haven't found a better way of doing it yet
c We need to read more than one component
c We have a file name for it
c We can seek on it
c The component axis is not fast
c The user isn't planning on seeking around between gathers a lot
c

c
c Unfortunately buggy SISIO screws this case up for writing, so for now
c only allow it if canwrite(curcon) is zero (meaning we can't write).
c ??????
c

                  if (canwrite(curcon) .eq. 0 .and. ierror .eq. 0 .and.
     1               lustatus(curcon) .eq. IO_NOK .and.
     1               ncomp .gt. 1  .and.
     1               infile .ne. ' ' .and.
     1               luseek(curcon) .ne. 0 .and.
     1               lun1(curcon) .ne. 1 .and.
     1               seekalot .le. 1) then

                      call galloc (pluinsv, ncomp * SZSMPD, ierr, 1)

c
c The file is already open once; use that one for the first component.
c

                      icomp = 1
                      luinsv(icomp) = luin

                      do icomp = 2, ncomp

                          call getln(luinsv(icomp) , infile, cmode, -1)

                          if (luinsv(icomp) .lt. 0) then

                              write (LERR, *)
     1                        name(1:nblen(name)), ' ', rname,
     1                        ': Unable to reopen ''',
     1                        infile(1:nblen(infile)), '''.'
#ifdef SUNSYSTEM
                              call flush(LERR)
#endif
                              write (LER , *)
     1                        name(1:nblen(name)), ' ', rname,
     1                        ': Unable to reopen ''',
     1                        infile(1:nblen(infile)), '''.'

c
c    Be neat and close the ones we just opened, since it turns out
c    we can't use them after all.
c
                              do icomp2 = 2, icomp - 1
                                  call lbclos (luinsv(icomp2))
                              enddo

                              goto 220

                          else

c
c We managed to open it again! Set the buffering on this new stream,
c same as we already did for the first.
c
                              if (seekalot .ne. 0) then
                                  call sislgbuf(luinsv(icomp), 'off')
                              endif

c
c Read past the line header. Double check everything is consistent
c just to be sure.
c
                              if (0 .ne. usprlhdr(luinsv(icomp),
     1                            icomp, slheader, slhedlength,
     1                                                      ierr)) then
c
c Note usprlhdr will have already printed an error message for us.
c
                                  goto 220
                              endif
                          endif
                      enddo

c
c Yup, we managed to open the file multiple times.
c
                      do icomp = 2, ncomp
       LUIOMC(icomp,curcon) =
     1                                            luinsv(icomp)
                      enddo

                      multiin(curcon) = MI_MOPEN
                      lustatus(curcon) = IO_SKIP


c*******************************************************************
c
c NOTE! NOTE! NOTE! NOTE!   ???????????????  !!!!!!!!!!!!!!!!!
c
c For future efficiency, at this point in the code we should
c explicitly SET the buffer size for EACH of the multiple inputs
c in sisio (including the first) to correspond to the size:
c
c       lun1 * (trlen * SZSMPD + sizeof(a greenword))
c
c *UNLESS*, of course, the buffers were already smaller than that;
c in that case leave them alone.
c
c That will be the "natural buffer size" for the dataset. Any larger
c buffers than that would be wasted (smaller ones would be OK, though).
c Unfortunately as of this time there is no provision in sisio for
c setting the buffer size from user code.
c
c*******************************************************************


220                   continue

                      call gfree (pluinsv)
                  endif


c
c (5)
c If we can seek, try doing that as a last resort.
c Probably this means that "seekalot" is 2, meaning the user is
c planning on seeking _between different gathers_ a lot. If that's
c the case, it's pointless to try any fancy buffering; we should
c just seek to get each individual component trace as we need them.
c
                  if (ierror .eq. 0 .and.
     1                lustatus(curcon) .eq. IO_NOK .and.
     1                luseek(curcon) .ne. 0) then

                      lustatus(curcon) = IO_SEEK

c
c Turn off buffers in SISIO. They behave very ungracefully
c when there are lots of seeks, repeatedly flushing buffers again
c and again. (Buffers are even flushed when the seek is a no-op!)
c

                      write (LERR, *)
     1                    name(1:nblen(name)), ' ', rname,
     1  ': Ensuring large buffers are off for input ', curcon

                      call sislgbuf(luin, 'off')

                  endif


c
c (6)
c Uh oh! We've run out of options!
c
                  if (ierror .eq. 0 .and.
     1                            lustatus(curcon) .eq. IO_NOK) then

                      write(LERR,*) name(1:nblen(name)), ' ', rname,
     1                ': Unable to find a way to transpose input ',
     1                curcon, '.'
#ifdef SUNSYSTEM
                      call flush(LERR)
#endif
                      write(LER ,*) name(1:nblen(name)), ' ', rname,
     1                ': Unable to find a way to transpose input ',
     1                curcon, '.'

                      ierror = ierror + 1

                  endif


              else

c************************************************************
c
c Scalar input from a single file case:
c    Just take whatever's there; we don't care! We're scalar!
c
c************************************************************

                  icomp = 1
                  LUCN(icomp,curcon) = 1
                  lunc(curcon) = 1
                  lun1(curcon) = 1
                  LUORDER(icomp,curcon) = 1
                  lustatus(curcon) = IO_OK

              endif
          endif

      else

c*******************************************************************
c
c The multiple single-component files case. (This one is easy in
c comparison!)
c
c*******************************************************************

c
c Read ONE line header first. We'll determine the input dataset
c size, etc, from this one. Then we'll check all the others
c against this one for dimensional consistency.
c

          icomp = 1
          if (0 .eq. usprlhdr(
     1 LUIOMC(icomp,curcon),
     1                      icomp, lheader, lhedlength, ierror)) then
              call uspc1cmp(lheader,
     1                       complist(COMPSRC,icomp),
     1                       complist(COMPREC,icomp),
     1                       icomp,
     1                       ierror)
          endif

c
c If we got that one OK, and there are multiple files for this
c input, loop over all the components reading the line header for
c each one.
c

          if (ierror .eq. 0) then
              do icomp = 2, nc(curcon)
                  if (0 .eq. usprlhdr(
     1 LUIOMC(icomp,curcon),
     1                          icomp, slheader, slhedlength, ierror))
     1                then
                          call uspc1cmp(slheader,
     1                               complist(COMPSRC,icomp),
     1                               complist(COMPREC,icomp),
     1                               icomp,
     1                               ierror)
                  endif
              enddo
          endif


          do icomp = 1, nc(curcon)
c
c    In this case each component is in a separate file.
c
              LUCN(icomp,curcon) = 1
              LUORDER(icomp,curcon) = icomp

          enddo

          lun1(curcon) = 1
          lunc(curcon) = 1
          lustatus(curcon) = IO_OK


      endif


c***********************************************************************
c
c Did we get past the stage of reading the line header OK?
c
c***********************************************************************

      if (ierror .gt. 0) then
          write (LERR, *) name(1:nblen(name)), ' ', rname,
     1    ': ', ierror,
     1    ' serious error(s) occurred processing line header(s).'
          write (LERR, *) name(1:nblen(name)), ' ', rname,
     1    ': Giving up on opening input number ', curcon, '.'
#ifdef SUNSYSTEM
          call flush(LERR)
#endif
          goto 666
      endif



c*******************************************************************
c
c Find WINDOWED dataset dimensions and pass-through status
c
c*******************************************************************

c
c    Increment the "number of times we checked the command line"
c    counter. We need a separate counter for this because it's
c    possible the opening this file will fail later on, and the
c    file won't actually get opened and curcon will get decremented
c    back again... but we still will have irrevocably read the command
c    line nevertheless!
c
      curclcon = curclcon + 1

      if (curclcon .gt. maxfile) then
       write (LERR, *) name(1:nblen(name)), ' ', rname,
     1              ': More files checked than promised to uspioinit().'
#ifdef SUNSYSTEM
       call flush(LERR)
#endif
       write (LER , *) name(1:nblen(name)), ' ', rname,
     1              ': More files checked than promised to uspioinit().'
       write (LER , *) name(1:nblen(name)),
     1              ': FATAL'
       stop
      endif


      whichopen(curclcon) = 1

      if (dptw .eq. DPTW_IGNORE) then
              winstr(curclcon) = 'XXX IGNORE! IGNORE! XXX'
      else
          if (nblen(wclparam) .gt. 0) then
              winstr(curclcon) = wclparam(1:nblen(wclparam))
          else
              winstr(curclcon) = ' '
          endif
      endif

c
c    Don't even LOOK at the command line if the user program
c    asked us not to.
c
      if (dptw .eq. DPTW_IGNORE) then
          ns = 0
          ne = 0
          irs = 0
          ire = 0
          clpass(curclcon) = -2
          goto 100
      endif

c
c    Check to see if this one has already been read
c    for a previous input file.
c
      do ifile = 1, curclcon - 1
          if (whichopen(ifile) .eq. 1 .and.
     1               clpass(ifile) .gt. -2 .and.
     1               winstr(ifile) .eq. winstr(curclcon)) then
c
c    It has; use the numbers we read that time and jump over
c    the part where we check the command line. (Remember
c    the command-line parsing routines only let you read any
c    given field in the command line ONCE!)
c
              ns = clns(ifile)
              ne = clne(ifile)
              irs = clrs(ifile)
              ire = clre(ifile)
              clpass(curclcon) = clpass(ifile)
              goto 100
          endif
      enddo

c
c    We haven't looked for this one before. Go look for it on
c    the command line. We have to use a temporary variable
c    because FORTRAN doesn't understand zero-length strings.
c

      tempstring = '-ns'//winstr(curclcon)
      call argi4 (tempstring(1:nblen(tempstring)), ns,  0,  0)
      tempstring = '-ne'//winstr(curclcon)
      call argi4 (tempstring(1:nblen(tempstring)), ne,  0,  0)
      tempstring = '-rs'//winstr(curclcon)
      call argi4 (tempstring(1:nblen(tempstring)), irs, 0,  0)
      tempstring = '-re'//winstr(curclcon)
      call argi4 (tempstring(1:nblen(tempstring)), ire, 0,  0)

      clpass(curclcon) = -1
      tempstring = '-wi'//winstr(curclcon)
      if (argis (tempstring(1:nblen(tempstring))) .gt. 0) then
          clpass(curclcon) = DPTW_WINDOW
      endif
      tempstring = '-pt'//winstr(curclcon)
      if (argis (tempstring(1:nblen(tempstring))) .gt. 0) then
          clpass(curclcon) = DPTW_PASS
      endif

 100  continue
c
c    Save what we got in case we need to refer to it again later
c
      clns(curclcon) = ns
      clne(curclcon) = ne
      clrs(curclcon) = irs
      clre(curclcon) = ire


c***********************************************************************
c
c    Figure out whether to pass through or not for this data, and how
c    to pack the windowed MC dimensions in.
c
c***********************************************************************

      if (cmode .eq. 'r+') then

c    If reading and WRITING, then don't pass through.

          ipass(curcon) = DPTW_WINDOW

      else if (clpass(curclcon) .gt. -1) then

c    Otherwise, if there was something on the command line use that.

          ipass(curcon) = clpass(curclcon)

      else if (dptw .ne. DPTW_IGNORE) then

c    Nothing on the command line? Then use the default ...

          ipass(curcon) = dptw

      else

c    ... UNLESS we're supposed to ignore all windowing! In that case
c    turn off pass through. (And there won't be any windowing, either.)

          ipass(curcon) = DPTW_WINDOW

      endif



      if ((multiin(curcon) .eq. MI_PIPE .or.
     1    multiin(curcon) .eq. MI_MOPEN) .and.
     1    numcmp .gt. 1) then
c
c    Find the dimensions of the input dataset taking into account
c    the MC layout specified by mctrsp.
c
          if (mctrsp .eq. MC_TRACE) then

              vn2(curcon) = n2(curcon) / numcmp
              vn3(curcon) = n3(curcon)

          else if (mctrsp .eq. MC_LINE .or. mctrsp .eq. MC_GATHER) then

              vn2(curcon) = n2(curcon)
              vn3(curcon) = n3(curcon) / numcmp

              if (mctrsp .eq. MC_LINE .and. append(curcon) .ne. 0) then
c
c    Uh oh! We're trying to append to a file that can't be appended to.
c
                  ierror = ierror + 1

                  write (LERR, *) name(1:nblen(name)), ' ', rname,
     1            ': Sorry, cannot append to',
     1            ' single-component line (MCTrSp=3) MC file ', '''',
     1 FINAME(1,curcon)
     1 (1:nblen(FINAME(1,curcon))),
     1            '''.'
#ifdef SUNSYSTEM
                  call flush(LERR)
#endif
                  write (LER , *) name(1:nblen(name)), ' ', rname,
     1            ': Sorry, cannot append to',
     1            ' single-component line (MCTrSp=3) MC file ', '''',
     1 FINAME(1,curcon)
     1 (1:nblen(FINAME(1,curcon))),
     1            '''.'

              endif

          else

              write (LERR, *) name(1:nblen(name)), ' ', rname,
     1        ': Invalid mctrsp not previously caught;',
     1        ' THIS SHOULD NOT BE ABLE TO HAPPEN!'
#ifdef SUNSYSTEM
              call flush(LERR)
#endif
              write (LER , *) name(1:nblen(name)), ' ', rname,
     1        ': Invalid mctrsp not previously caught;',
     1        ' THIS SHOULD NOT BE ABLE TO HAPPEN!'
              stop

          endif

      else
c
c    We aren't trying to pack the multicomponent dimension into a
c    single file; the MC dimensions are the same as the scalar USP
c    dimensions.
c
          vn2(curcon) = n2(curcon)
          vn3(curcon) = n3(curcon)

      endif

c
c    Ensure that the command-line values for starting and ending trace,
c    record, etc, are compatible with the MC dimensions of the data set.
c    Command-line parameters are modified to fit as necessary.
c    By default ns, ne, irs, and ire are all zero;
c    zero or negative values are changed to the defaults (to use the
c    entire dataset) by cmdchk.
c    Note cmdchk is a standard USP library routine.
c
 
      call cmdchk(ns,ne,irs,ire,vn2(curcon),vn3(curcon))

c
c    wn2, wn3, wf2, wf3: specify MC windowed subset of dataset
c
      wf2(curcon) = ns
      wn2(curcon) = ne - ns + 1
      wf3(curcon) = irs
      wn3(curcon) = ire - irs + 1

c
c    The only dimensions the user program will hear about.
c
      un1 = n1(curcon)
      ud1 = d1(curcon)
      un2 = wn2(curcon)
      uf2 = wf2(curcon)
      un3 = wn3(curcon)
      uf3 = wf3(curcon)

      write (LERR, *)
      write (LERR, *) '   For input ', curcon,
     1 ':   ns = ', wf2(curcon),
     1 ',  ne = ', wn2(curcon) + wf2(curcon) - 1,
     1 ',  rs = ', wf3(curcon),
     1 ',  re = ', wn3(curcon) + wf3(curcon) - 1

      write (LERR, *)
      write (LERR, *) '   Input dataset ''Cube'' dimensions:'
      write (LERR, *) '         NumSmp = ', n1(curcon)
      if (numcmp .gt. 0) then
          write (LERR, *) '         NumCmp = ', numcmp
      endif
      write (LERR, *) '         NumTrc = ', vn2(curcon)
      write (LERR, *) '         NumRec = ', vn3(curcon)


c
c    Flag all the cases where the input and output dimensions are
c    different:
c
c    A) Some components are being windowed out of a MC dataset
c    B) We're reading an MC dataset as a scalar one, ignoring the
c       component axis
c    C) We're windowing over the NumTrc or NumRec dimensions
c

      if  ( ((ncomp .ne. 1 .or. complist(COMPSRC,1) .ne. SCALAR) .and.
     1        (ncomp .ne. numcmp)) .or.
     1      ((ncomp .eq. 1 .and. complist(COMPSRC,1) .eq. SCALAR) .and.
     1        (numcmp .gt. 0)) .or.
     1      wn2(curcon) .ne. vn2(curcon) .or.
     1      wn3(curcon) .ne. vn3(curcon) ) then

          write (LERR, *)
          write (LERR, *) '   Visible dataset ''Cube'' dimensions:'
          write (LERR, *) '         NumSmp = ', n1(curcon)
          if (ncomp .ne. 1 .or.
     1        complist(COMPSRC,1) .ne. SCALAR) then
              write (LERR, *) '         NumCmp = ', ncomp
          endif
          write (LERR, *) '         NumTrc = ', wn2(curcon)
          write (LERR, *) '         NumRec = ', wn3(curcon)

      endif


c
c    Is there any windowing over the NumTrc or NumRec dimensions
c    going on?
c

      if (wn2(curcon) .ne. vn2(curcon) .or.
     1    wn3(curcon) .ne. vn3(curcon)) then

c
c    Yes, there is windowing. Check to see that we're not trying to
c    simultaneously window and append!
c

          if (append(curcon) .ne. 0) then

              ierror = ierror + 1

              write (LERR, *) name(1:nblen(name)), ' ', rname,
     1        ': Sorry, cannot append to windowed files.'
#ifdef SUNSYSTEM
              call flush(LERR)
#endif
              write (LER , *) name(1:nblen(name)), ' ', rname,
     1        ': Sorry, cannot append to windowed files.'

          else

              write (LERR, *)
              if (ipass(curcon) .eq. DPTW_PASS) then
                  write (LERR, *)
     1           '   Data outside the window will be passed through.'
              else
                  write (LERR, *)
     1           '   Data outside the window will be skipped.'
              endif

          endif

      else

c
c    No windowing; just set ipass to 0.
c
          ipass(curcon) = DPTW_WINDOW

          if (dptw .eq. DPTW_IGNORE) then

              write (LERR, *)
              write (LERR, *)
     1    '   User routines will be solely responsible for windowing.'

          endif

      endif


      if (ipass(curcon) .ne. 0 .and. canwrite(curcon) .ne. 0) then

          ierror = ierror + 1

          write (LERR, *) name(1:nblen(name)), ' ', rname,
     1    ': Sorry, cannot write to passed-through input.'
#ifdef SUNSYSTEM
          call flush(LERR)
#endif
          write (LER , *) name(1:nblen(name)), ' ', rname,
     1    ': Sorry, cannot write to passed-through input.'

      endif

c
c    Let the user program know whether windowing or pass-through
c    are going on, in case they care.
c
      upt = ipass(curcon)

c
c    Let the user know whether they can seek or not.
c

      if (ipass(curcon) .ne. 0) then
          seekstat = 2
      else
          if (luseek(curcon) .eq. 0) then
              seekstat = 1
          else
              seekstat = 0
          endif
      endif

c
c    Save input mctrsp so we can use the same default for output that
c    we got for input.
c
      inmctrsp(curcon) = mctrsp



c***********************************************************************
c
c Did we get past the stage of figuring out windowing OK?
c
c***********************************************************************

      if (ierror .gt. 0) then
          write (LERR, *) name(1:nblen(name)), ' ', rname,
     1    ': ', ierror,
     1    ' serious error(s) occurred during',
     1    ' windowing/pass-through initialization.'
          write (LERR, *) name(1:nblen(name)), ' ', rname,
     1    ': Giving up on opening input number ', curcon, '.'
#ifdef SUNSYSTEM
          call flush(LERR)
#endif
          goto 666
      endif


c
c    Set the trace length (including header)
c
      trlen(curcon) = trhlen(curcon) + n1(curcon)
      trlength = trlen(curcon)

      write (LERR,*)
      write (LERR, *) name(1:nblen(name)), ' ', rname,
     1   ': Line header length for connection ', curcon,
     1   ' is ', lhedlength, ' bytes.'
      write (LERR, *) name(1:nblen(name)), ' ', rname,
     1   ': Trace header length for connection ', curcon,
     1   ' is ', trhlen(curcon), ' full words.'
      write (LERR, *) name(1:nblen(name)), ' ', rname,
     1   ': Total trace length for connection ', curcon,
     1   ' is ', trlen(curcon), ' full words.'

c
c    This file already exists, so set the "amount that already exists"
c    to show the entire thing as already existing. (These variables are
c    primarily for keeping track of files we're creating by writing,
c    so we can properly handle the case when a seek is beyond
c    the current end of the file.)
c
      lumax(curcon) = n2(curcon) * n3(curcon)
      luumax(curcon) = wn2(curcon) * wn3(curcon)

c
c    If there is a trace transpose buffer, then THIS variable keeps
c    track of how long the file actually is on disk (versus how long
c    it appears to be to the user, who can only access the file through
c    the transpose I/O buffer)
c
      trbendpoint(curcon) = lumax(curcon)

c
c    This variable tells us how big we expect the file to be at the
c    time we close it. If we aren't appending, it determines the limit
c    of how far it's legal to write.
c
      luulim(curcon) = wn2(curcon) * wn3(curcon)

c
c    Make sure we have sufficient temporary space to store
c    the longest trace we can encounter
c
      if (maxtrlen .lt. ncomp * trlen(curcon) * SZSMPD) then
          maxtrlen = ncomp * trlen(curcon) * SZSMPD

          if (uspsta(maxtrlen, pscrtrace, pnulltrace) .ne. 0) then

              write(LERR,*) name(1:nblen(name)), ' ', rname,
     1        ': Unable to allocate ', maxtrlen,
     1        ' bytes for scratch trace buffer.'
#ifdef SUNSYSTEM
              call flush(LERR)
#endif
              write(LER ,*) name(1:nblen(name)), ' ', rname,
     1        ': Unable to allocate ', maxtrlen,
     1        ' bytes for scratch trace buffer.'

              ierror = ierror + 1
              goto 666

          endif

      endif


      lubufsz(curcon) = sisbufsz(
     1 LUIOMC(1,curcon))

      write(LERR,*)

      if (lubufsz(curcon) .gt. 0) then
          write(LERR,*) name(1:nblen(name)), ' ', rname,
     1    ': SIS IO buffers of ', lubufsz(curcon), ' bytes.'
      else
          write(LERR,*) name(1:nblen(name)), ' ', rname,
     1    ': SIS IO is unbuffered.'
      endif


c*******************************************************************
c
c    We're done!
c
c*******************************************************************

 666  continue

c
c If they asked to start at the end of the file, seek to the end.
c
      if (ierror .eq. 0 .and. toend .ne. 0) then

          if (uspseek(curcon,0) .ne. 0) then

              write(LERR,*) name(1:nblen(name)), ' ', rname,
     1        ': Unable to seek to the end of the file.'
#ifdef SUNSYSTEM
              call flush(LERR)
#endif
              write(LER ,*) name(1:nblen(name)), ' ', rname,
     1        ': Unable to seek to the end of the file.'

              ierror = ierror + 1

          else

              write(LERR,*) name(1:nblen(name)), ' ', rname,
     1        ': Seeking to the current EOF.'

          endif

      endif


c
c    If there has been an error, make sure they can't try to use
c    this connection.
c
      if (ierror .ne. 0) then
          lustatus(curcon) = IO_NOK
      endif

c
c Summarize the results for their edification.
c
      write (LERR,*)
      call uspinsum(curcon)


 667  continue

      write (LERR, *) 

      if (ierror .eq. 0) then

          write (LERR, *) name(1:nblen(name)), ' ', rname,
     1    ': Connection ', curcon, ' is now open and ready for use.'
          write (LERR, *)
     1'++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++'

      else

c
c    Close all open files associated with this botched connection
c
          write (LERR, *) name(1:nblen(name)), ' ', rname,
     1    ': Closing down partially opened connection.'

          call uspclall (curcon)

c
c    Reset all the uspinfo structures to their initial clean state,
c    to remove all traces this embarrassing input ever happened.
c
          call uspcleanslate (curcon)
          curcon = curcon - 1
          icon = -1

          write (LERR, *)
     1'----------------------------------------------------------------'

      endif

      write (LERR, *) 
#ifdef SUNSYSTEM
      call flush(LERR)
#endif

      return
      end
