C***********************************************************************
C                 copyright 2001, Amoco Production Company             *
C                             All Rights Reserved                      *
C                     an affiliate of BP America Inc.                  *
C***********************************************************************
c
c The maximum number of components that we have room allocated
c for in the line header.
c
#define MCLLen	16


c
c Search range for "extra" components is from 1 to this for both
c Source and Receiver.
c
#define COMPRANGE	3


c
c The maximum size of an in-core transpose buffer, in bytes. Bigger
c than this and it won't try to allocate it.
c
#ifndef MAXTBUFSZ
#define MAXTBUFSZ       4194304
#endif


c
c To keep default input and output IKP numbers from
c colliding.
c
#define IKP_OUT_OFFSET	5


c
c Format "3" is the USP format everyone uses.
c
#define USUAL_USP_FORMAT	3


#define STDOUTNO	1

      subroutine uspoutput (mode, clparam, wclparam,
     1   ncomp, complist, ucinput, dmctrsp,
     1   ikpoffs, ikpsock, seekalot, seekstat,
     1   un1, un2, un3, ud1,
     1   lheader, lhedlength, trlength, roufile,
     1   icon, ierror)

c******************************************************************
c******************************************************************
c******************************************************************
c
c Create an output file from scratch.
c
c******************************************************************
c
c Uspoutput has an extensive argument list:
c
c mode (read / write mode):
c 'w'	create a new file for writing only
c 'rw'  create a new file for reading and writing
c 'ra'  create a new file for reading and writing and appending,
c       meaning that you don't have to specify the final file
c       size up front. After you're done the line header will
c       automatically be updated to reflect the final size.
c       You DO have to write out an integral number of gathers.
c
c
c clparam ("Output file name command line parameter"): this string
c is appended to '-O' and looked for on the command line to find
c an output file name for this output. If clparam is whitespace,
c then uspoutput will look for '-O' as is usual practice for scalar
c USP programs that only need a single output.
c
c If no output file name is specified on the command line, the place
c uspoutput next looks depends on whether the program is running
c under IKP or not. If not in IKP, the only possible fallback is to
c standard output. If in IKP, the fallback is to the "IKP single output"
c socket number specified by the argument "ikpsock" (this is quite
c likely 1, meaning standard output, but doesn't have to be).
c
c If you specify the same "clparam" for different outputs, uspoutput
c will check the command line for repeated occurrences of the same
c argument. (For example, "-O file1 -O file2 -O file3".) Note this is
c very different from the behavior for "wclparam" (see below).
c
c If the "multi-component trace spacing" mctrsp is set to 0, possibly
c by the user specifying "-mc 0" on the command line, then the file
c name is used as a template: a .IJ ending is attached, where I is the
c source component number (1 to 3) and J is the receiver component
c number (1 to 3).
c
c If the user specifies "-mc 0" but there is no filename template, then
c the output will be into a single MC file (with single-component
c gathers) and the "-mc 0" will be ignored. (Ditto if the program is
c running under IKP but the single output is available.)
c
c
c wclparam ("Windowing command line parameter"): This string is appended
c to '-mc' and the result looked for on the command line to determine
c how to transpose the data. A single integer should follow,
c corresponding to the line-header keyword MCTrSp (MultiComponent
c Trace Spacing). If the user does not specify anything, you'll get the
c default: whatever is specified in dmctrsp below, or if that is
c negative the same as the corresponding input (if one exists), or
c if there isn't one then a single-component-gather file (mctrsp=2).
c If the user specifies 0, you'll get "Leon-style" multicomponent I/O.
c (Several scalar files.)
c 
c
c ncomp is the number of components wanted.
c
c
c complist is an array of dimension (2,ncomp) specifying for each
c component:
c 1:  Source comp #
c 2:  Rec comp #
c For SCALAR output, set the source component number of the single
c component to -1; the IKP socket number to use for this scalar
c output inside IKP will be in "ikpsock" (see below).
c
c
c ucinput gives the "corresponding input". Whether or not to do
c pass through, etc, depends on the setting for the corresponding
c input.
c
c If there is a corresponding input several parameters will be read
c from there instead of from the uspoutput command line. (The ones
c the define the data sample rate and the data dimensions: ud1,
c un2, un3. These will be RETURNED as well, so don't pass zeroes in
c those slots!) The trace-data length un1, the total trace length
c trlength, the number of components ncomp, and the component list
c will all be checked to make sure they match. If they don't you'll
c get an error message.
c
c If there is no corresponding input set ucinput to 0.
c
c
c dmctrsp gives the default MCTrSp to use. (Usually it should be -1.)
c Choices are:
c -1: Use the standard default; 2 if creating a file from scratch, or
c     whatever MCTrSp was for the corresponding input if there was one.
c 0: Use Leon-style multi-file MC output.
c 1: Use "component fast" single-file output.
c 2: Use "single-component gathers" single-file output.
c 3: Use "single-component lines" single-file output.
c
c In any case, whatever is specified on the command line overrides.
c (Unless in IKP, in which case the configuration of output pipes
c can override; if the single-output pipe is open, then you'll get
c single-file output instead of multiple-file output no matter what.)
c
c
c ikpoffs ("IKP socket offset") is a number to add to the standard
c socket numbers (10 * Source Comp No + Receiver Comp No + 5 + ikpoffs)
c for IKP purposes for this file. Usually ikpoffs will be 0;
c the purpose of this parameter is to allow multiple multicomponent
c file output from inside IKP.
c
c
c ikpsock is the ikp socket number to look for a single output on.
c If this is not grounded, then all the output will be looked for
c here instead of in multiple sockets. Usually ikpsock will be 1,
c meaning standard output.
c
c
c seekalot guides uspinput in deciding how to buffer the data.
c
c A value of 0 means "this program will mostly plod through the data
c reading in order (alternated perhaps with passes through the data
c writing in order)". If you're going to mostly plod with only the
c occasional long seek, that would also count as 0. (Basically, if
c you would have left the sisio buffers on yourself, then seekalot
c should be 0.)
c
c uspseek is smart enough not to flush sisio's buffers unnecessarily,
c so don't worry about short forward seeks on reading: if that's
c all you're going to do, then seekalot can also be 0, because
c usprtrace will avoid the buffer flush by reading the intervening
c traces and discarding them for you. (Repeated short forward seeks
c on writing, however, count as bona-fide disruptive seeks. In that
c case seekalot should probably be 1.)
c
c A value of 1 means "this program will jump around within a gather
c a lot, but not often between different gathers". If the program
c is going to furiously alternate reading and writing nearby traces
c (for example, read a trace, write it back out, read the next trace,
c write it back out, etc) that also counts as 1. (This mode will
c turn off the sisio large buffers unless a gather will be contained
c within the span of uspio's in-core transpose buffer. In that case
c it leaves them on.)
c
c If the program will jump at random all over the data, even between
c different gathers and lines, then seekalot should be 2. This will
c always turn off sisio large buffers.
c
c
c seekstat can be 0, 1, or 2.
c
c 0 means "feel free to seek all you want".
c
c 1 means "you are reading or writing to a pipe or socket, so
c seeking backwards is not allowed". (It's OK to seek forwards.
c If you're reading it will read the intervening traces and
c discard them; if you're writing it will write null traces for
c you.)
c
c 2 means "pass-through is enabled so no seeking is allowed at all".
c (If pass through is enabled, just calling uspseek will generate
c an error.)
c
c
c un1, un2, un3, ud1 ("u" for "user") are as per SEPlib notation
c conventions:
c  un1 = number of samples in a trace (NumSmp),
c  ud1 = floating-point sampling rate (SmpInt)
c  un2 = number of traces in a gather (NumTrc)
c  un3 = number of gathers in a line  (NumRec)
c
c If there is a "corresponding input", ud1, un2 and un3 will be
c SET FOR YOU. Any previous value they may have had will be
c ignored.
c
c If you're appending, there's no reason not to set un3 to zero.
c
c
c lheader is the line header. You can either provide a header yourself
c or ask that uspoutput create one for you. In either case, the lheader
c you provide should be allocated big enough to contain a USP line header.
c
c
c lhedlength is the length in bytes of the line header. If 0, then
c you are asking for uspoutput to create a header for you from
c scratch.
c
c
c trlength is the length in full words (floats) of a single trace,
c including the trace header. You'll need to allocate
c trlength * sizeof(float) * the number of components (returned ncomp)
c bytes to store a single MC trace. Note you need to SET trlength.
c It is not returned. Until variable length headers are introduced,
c just use trlength = ITRWRD + un1.
c
c If there is a "corresponding input", trlength must match the trace
c length for the corresponding input.
c
c
c roufile is the "root output file name", which you may find useful
c if you want to construct a filename for auxilliary output. Note
c if there was no '-O' field on the command line uspoutput will be
c forced to invent a name. If the output is to standard output, the
c name will be 'stdout'. If to a single IKP socket it will be of
c the form 'IKP.sN', where N is the socket number. If MC output
c through one or more IKP sockets, the name will be of the form
c 'IKP.oN', where N is the socket offset.
c
c
c icon is the "connection number", the number you'll have to use
c to refer to this connection later.
c
c
c ierror will be incremented once for each error that was found.
c Unless ierror is zero this output cannot be used, and
c attempting to do so will result in an error.
c
c
c******************************************************************
c******************************************************************
c******************************************************************

#include <save_defs.h>
#include <f77/iounit.h>
#include <f77/lhdrsz.h>
#include <f77/sisdef.h>

#ifdef SUNSYSTEM
      implicit   none
#endif
      character  mode*(*), clparam*(*), wclparam*(*)
      integer    ncomp, ncomp2
      integer    complist(2,ncomp)
      integer    ucinput, dmctrsp
      integer    ikpoffs, ikpsock, seekalot, seekstat
      integer    un1, un2, un3
      real       ud1
      integer    lheader (1)
      integer    lhedlength, trlength
      character  roufile*(*)
      integer    icon
      integer    ierror


      character  oufile * 100, mcoufile * 100
      character  tempstring * 100
      character  estr * 100
      character  cmode * 10
      character  namstr * 120
      integer    luou, mcomp
      integer    ierr, istat, inikpok
      integer    icomp, icomp2, ipipeno, mcoulen
      integer    ikp, iseek, ikpsoungr
      integer    ifile, jcomp, mctrsp
      integer    ifmt_MCList, l_MCList, ln_MCList
      integer    comps(MCLLen), luousv
      pointer    (pluousv, luousv(2))
      integer    isrc, irec
      integer    ileft, iright, icount


      integer    ludsk, lutty, sisbufsz
      integer    pipchk
      integer    in_ikp, nblen, usptba, uspsta

      integer    iform, nsi, lbytes, nbytes, lbyout, numcmp
      integer    slength

      character  rname*(*)
      parameter (rname = 'uspoutput()')



c******************************************************************
c
c Load all the global file information, saved in a common block
c
c******************************************************************
#include "uspinfo.h"





      ierror = 0


      write (LERR, *) '***************',
     1'****************************************************************'

      write (LERR, *)
      write (LERR, *) 'Summary of uspoutput() calling parameters:'
      write (LERR, *)

      slength = nblen(mode)
      if (slength .eq. 0) then
          slength = 1
      endif
      write (LERR, *) 'mode=  ', '''', mode(1:slength), ''''

      slength = nblen(clparam)
      if (slength .eq. 0) then
          slength = 1
      endif
      write (LERR, *) 'clparam=  ', '''', clparam(1:slength), ''''

      if (wclparam .ne. 'XXX IGNORE! IGNORE! XXX') then
          slength = nblen(wclparam)
          if (slength .eq. 0) then
              slength = 1
          endif
          write (LERR, *) 'wclparam=  ', '''', wclparam(1:slength), ''''
      endif

      write (LERR, *) 'ncomp=', ncomp
      write (LERR, *) 'ucinput=', ucinput
      write (LERR, *) 'ikpoffs=', ikpoffs
      write (LERR, *) 'ikpsock=', ikpsock
      write (LERR, *) 'seekalot=', seekalot
      write (LERR, *) 'n1=', un1
      write (LERR, *) 'lhedlength=', lhedlength
      write (LERR, *) 'trlength=', trlength

      write (LERR, *)
      write (LERR, *)


c******************************************************************
c
c Check for various fatal bonehead errors
c
c******************************************************************

      if (curcon .lt. 0) then
       write (LERR, *) name(1:nblen(name)), ' ', rname,
     1              ': uspioinit() must be called before uspoutput()!'
#ifdef SUNSYSTEM
       call flush(LERR)
#endif
       write (LER , *) name(1:nblen(name)), ' ', rname,
     1              ': uspioinit() must be called before uspoutput()!'
       write (LER , *) name(1:nblen(name)),
     1              ': FATAL'
       stop
      endif


c
c    Increment the file number counter
c
      curcon = curcon + 1


      if (curcon .gt. maxfile) then
       write (LERR, *) name(1:nblen(name)), ' ', rname,
     1              ': More files opened than promised to uspioinit().'
#ifdef SUNSYSTEM
       call flush(LERR)
#endif
       write (LER , *) name(1:nblen(name)), ' ', rname,
     1              ': More files opened than promised to uspioinit().'
       write (LER , *) name(1:nblen(name)),
     1              ': FATAL'
       stop
      endif

      if (ikpsock .lt. 0) then
       write (LERR, *) name(1:nblen(name)), ' ', rname,
     1 ': Invalid ikp single-output socket number (', ikpsock, ').'
#ifdef SUNSYSTEM
       call flush(LERR)
#endif
       write (LER , *) name(1:nblen(name)), ' ', rname,
     1 ': Invalid ikp single-output socket number (', ikpsock, ').'
       write (LER , *) name(1:nblen(name)), ' ', rname,
     1              ': FATAL'
       stop
      endif

c
c    Make a copy of "mode" into "cmode"
c    (we may need to make it longer or rewrite it, you see)
c    and set the mode.
c

      if (mode .eq. ' ' .or. mode .eq. 'w') then

          cmode = 'w'

          write (LERR, *) name(1:nblen(name)), ' ', rname,
     1    ': Output number ', curcon,
     1    ' is being opened for writing only.'

          canwrite(curcon) = 1

      else if (mode .eq. 'rw' .or. mode .eq. 'r' .or.
     1                                     mode .eq. 'wr') then

          cmode = 'w+'

          write (LERR, *) name(1:nblen(name)), ' ', rname,
     1    ': Output number ', curcon,
     1    ' is being opened for both reading and WRITING.'

          canread(curcon) = 1
          canwrite(curcon) = 1

      else if (mode .eq. 'ra' .or. mode .eq. 'ar' .or.
     1                                     mode .eq. 'a') then

          cmode = 'w+'

          write (LERR, *) name(1:nblen(name)), ' ', rname,
     1    ': Output number ', curcon,
     1    ' is being opened for reading, WRITING, and APPENDING.'

          canread(curcon) = 1
          canwrite(curcon) = 1
          append(curcon) = 1

      else

          write (LERR, *) name(1:nblen(name)), ' ', rname,
     1    ': Unknown read/write mode ''', mode(1:nblen(mode)), '''.'
#ifdef SUNSYSTEM
          call flush(LERR)
#endif
          write (LER , *) name(1:nblen(name)), ' ', rname,
     1    ': Unknown read/write mode ''', mode(1:nblen(mode)), '''.'
          write (LER , *) name(1:nblen(name)), ' ', rname,
     1    ': FATAL'
          stop

      endif


      if (ncomp .gt. maxcomp) then
       write (LERR, *) name(1:nblen(name)), ' ', rname,
     1 ': More components requested (', ncomp,
     1 ') than max allowed by uspioinit() (', maxcomp, ').'
#ifdef SUNSYSTEM
       call flush(LERR)
#endif
       write (LER , *) name(1:nblen(name)), ' ', rname,
     1 ': More components requested (', ncomp,
     1 ') than max allowed by uspioinit() (', maxcomp, ').'
       write (LER , *) name(1:nblen(name)),
     1              ': FATAL'
       stop
      endif

      if (ncomp .le. 0) then
       write (LERR, *) name(1:nblen(name)), ' ', rname,
     1 ': Requested number of components (', ncomp, ') is invalid.'
#ifdef SUNSYSTEM
       call flush(LERR)
#endif
       write (LER , *) name(1:nblen(name)), ' ', rname,
     1 ': Requested number of components (', ncomp, ') is invalid.'
       write (LER , *) name(1:nblen(name)), ' ', rname,
     1              ': FATAL'
       stop
      endif


c******************************************************************
c
c Slightly less fatal bonehead errors and general commentary
c
c******************************************************************

c
c If there is only one component, and the source component number
c is -1, then it's a SCALAR trace.
c
      if (ncomp .eq. 1 .and.
     1    complist(COMPSRC,1) .eq. SCALAR) then

          write (LERR, *) name(1:nblen(name)), ' ', rname,
     1    ': Output number ', curcon,
     1    ' is scalar.'

      else

c
c Otherwise, it's a multicomponent trace and the source and
c receiver component numbers must be between 1 and 3, inclusive.
c
          write (LERR, *) name(1:nblen(name)), ' ', rname,
     1    ': Output number ', curcon,
     1    ' has ', ncomp, ' component(s):'


          do icomp = 1, ncomp

              write (LERR, *) '        ',
     1        'Comp No. ', icomp, 
     1        ',   Src Comp. = ', complist(COMPSRC,icomp),
     1        ',   Rec Comp. = ', complist(COMPREC,icomp), ' .'

              if (complist(COMPSRC,icomp) .lt. 1 .or.
     1            complist(COMPSRC,icomp) .gt. COMPRANGE .or.
     1            complist(COMPREC,icomp) .lt. 1 .or.
     1            complist(COMPREC,icomp) .gt. COMPRANGE) then

                  write (LERR, *) name(1:nblen(name)), ' ', rname,
     1            ': Component numbers must be between 1 and ',
     1            COMPRANGE, '.'
#ifdef SUNSYSTEM
                  call flush(LERR)
#endif

                  write (LER , *) name(1:nblen(name)), ' ', rname,
     1            ': Component numbers must be between 1 and ',
     1            COMPRANGE, '.'

                  ierror = ierror + 1
              endif

c
c Also check that we don't have to return the same component
c multiple times. That would cause unnecessary complications
c when seeking later on.
c
              do jcomp = 1, icomp-1
                  if ((complist(COMPSRC,icomp) .eq.
     1                complist(COMPSRC,jcomp)) .and.
     1                (complist(COMPREC,icomp) .eq.
     1                complist(COMPREC,jcomp))) then

                      write (LERR, *) name(1:nblen(name)), ' ', rname,
     1                ': Component ', icomp,
     1                ' repeats component ', jcomp, '.'
#ifdef SUNSYSTEM
                      call flush(LERR)
#endif

                      write (LER , *) name(1:nblen(name)), ' ', rname,
     1                ': Component ', icomp,
     1                ' repeats component ', jcomp, '.'

                      ierror = ierror + 1
                  endif
              enddo

          enddo

      endif

      write (LERR, *)


      if (ierror .gt. 0) then
          goto 667
      endif


c**********************************************************************
c
c Corresponding input stuff, windowing / pass-through initialization,
c and determining visible data dimensions
c
c**********************************************************************

      if (ncomp .eq. 1 .and.
     1    complist(COMPSRC,1) .eq. SCALAR) then

c
c    Avoid a bogus mismatch because the Receiver number on a scalar
c    output doesn't match the receiver number on a scalar input;
c    just set the receiver number to be the same thing.
c
 
          complist(COMPREC,1) = SCALAR
 
      endif


      if (ucinput .eq. 0) then

          ipass(curcon) = 0

          n1(curcon) = un1
          d1(curcon) = ud1

          vn2(curcon) = un2
          vn3(curcon) = un3
          wn2(curcon) = un2
          wn3(curcon) = un3

          wf2(curcon) = 1
          wf3(curcon) = 1

          trlen(curcon) = trlength

      else if (ucinput .le. 0 .or. ucinput .ge. curcon) then

          write (LERR, *) name(1:nblen(name)), ' ', rname,
     1    ': Corresponding input (', ucinput,
     1    ')', ' is invalid.'
#ifdef SUNSYSTEM
          call flush(LERR)
#endif
          write (LER , *) name(1:nblen(name)), ' ', rname,
     1    ': Corresponding input (', ucinput,
     1    ')', ' is invalid.'

          ierror = ierror + 1
          goto 667

      else
c
c If there is a corresponding input, first check that the input is a
c reasonable one.
c
          write (LERR, *) name(1:nblen(name)), ' ', rname,
     1    ': Output number ', curcon,
     1    ' corresponds to input number ', ucinput, '.'



          if (lustatus(ucinput) .eq. IO_NOK) then

              write (LERR, *) name(1:nblen(name)), ' ', rname,
     1        ': Corresponding input (', ucinput,
     1        ')', ' is not open.'
#ifdef SUNSYSTEM
              call flush(LERR)
#endif
              write (LER , *) name(1:nblen(name)), ' ', rname,
     1        ': Corresponding input (', ucinput,
     1        ')', ' is not open.'

              ierror = ierror + 1
              goto 667

          endif

c
c Now get various parameters from the corresponding input:
c data sample rate, data dimensions. Check that n1 (NumSmp),
c trace length, ncomp and the component list match.
c

          if (un1 .ne. n1(ucinput)) then

              write (LERR, *) name(1:nblen(name)), ' ', rname,
     1        ': NumSmp (', un1,
     1        ')', ' does not match corresponding input ', '(',
     1        n1(ucinput), ').'
#ifdef SUNSYSTEM
              call flush(LERR)
#endif
              write (LER , *) name(1:nblen(name)), ' ', rname,
     1        ': NumSmp (', un1,
     1        ')', ' does not match corresponding input ', '(',
     1        n1(ucinput), ').'

              ierror = ierror + 1

          endif

          if (trlength .ne. trlen(ucinput)) then

              write (LERR, *) name(1:nblen(name)), ' ', rname,
     1        ': Trace length (', trlength,
     1        ')', ' does not match corresponding input ', '(',
     1        trlen(ucinput), ').'
#ifdef SUNSYSTEM
              call flush(LERR)
#endif
              write (LER , *) name(1:nblen(name)), ' ', rname,
     1        ': Trace length (', trlength,
     1        ')', ' does not match corresponding input ', '(',
     1        trlen(ucinput), ').'

              ierror = ierror + 1

          endif

          if (ncomp .ne. nc(ucinput)) then

              write (LERR, *) name(1:nblen(name)), ' ', rname,
     1        ': Number of components (', ncomp,
     1        ')', ' does not match corresponding input ', '(',
     1        nc(ucinput), ').'
#ifdef SUNSYSTEM
              call flush(LERR)
#endif
              write (LER , *) name(1:nblen(name)), ' ', rname,
     1        ': Number of components (', ncomp,
     1        ')', ' does not match corresponding input ', '(',
     1        nc(ucinput), ').'

              ierror = ierror + 1

          endif

          do icomp = 1, ncomp

              if ((complist(COMPSRC,icomp) .ne.
     1 LCOMP(COMPSRC,icomp,ucinput)
     1        ) .or.
     1        (complist(COMPREC,icomp) .ne.
     1 LCOMP(COMPREC,icomp,ucinput)
     1        )) then

                  write (LERR, *) name(1:nblen(name)), ' ', rname,
     1            ': Component ', icomp,
     1            ' does not match corresponding input.'
#ifdef SUNSYSTEM
                  call flush(LERR)
#endif
                  write (LER , *) name(1:nblen(name)), ' ', rname,
     1            ': Component ', icomp,
     1            ' does not match corresponding input.'

                  ierror = ierror + 1

              endif

          enddo


c
c   Trace length(s) and trace sample interval. The latter gets
c   set for you, using the value from the corresponding input.
c
          n1(curcon) = un1
          trlen(curcon) = trlength

          ud1 = d1(ucinput)
          d1(curcon) = ud1

c
c    The user only knows about the windowed part, regardless of
c    whether there is pass through or not!
c
          un2 = wn2(ucinput)
          un3 = wn3(ucinput)

c
c    The windowed parts are always the same size in input
c    and output.
c
          wn2(curcon) = wn2(ucinput)
          wn3(curcon) = wn3(ucinput)


c
c    The pass-through mode depends on the pass-through mode
c    of the corresponding input.
c
          ipass(curcon) = ipass(ucinput)

          if (ipass(curcon) .ne. 0) then
c
c    If we're going to do pass through,
c    save the corresponding input in the uspinfo structure;
c    we'll need to know this to do the pass through.
c    (If ipass was zero, then cinput will remain -1, and this
c    output will never be found as a match.)
c
              cinput(curcon) = ucinput

c
c    Pass through. The actual output data has the same
c    dimensions as the UNWINDOWED input data. The user
c    program will only see the computational window part of this.
c

              vn2(curcon) = vn2(ucinput)
              vn3(curcon) = vn3(ucinput)

              wf2(curcon) = wf2(ucinput)
              wf3(curcon) = wf3(ucinput)

          else
c
c    No pass through. The output data has the dimensions
c    of the WINDOWED input data; what the user program can
c    see is ALL THERE IS!
c
              vn2(curcon) = wn2(ucinput)
              vn3(curcon) = wn3(ucinput)

              wf2(curcon) = 1
              wf3(curcon) = 1

          endif


c
c    Some more goof checks.
c

          if (inmctrsp(ucinput) .lt. 0 .or. inmctrsp(ucinput) .gt. 3)
     1                                                             then
              write (LERR, *) name(1:nblen(name)), ' ', rname,
     1        ': Corresponding input has invalid mctrsp ',
     1        inmctrsp(ucinput), '.'
#ifdef SUNSYSTEM
              call flush(LERR)
#endif

              write (LER , *) name(1:nblen(name)), ' ', rname,
     1        ': Corresponding input has invalid mctrsp ',
     1        inmctrsp(ucinput), '.'

              ierror = ierror + 1

          endif

          if (ipass(curcon) .ne. 0 .and. append(curcon) .ne. 0) then

              write (LERR, *) name(1:nblen(name)), ' ', rname,
     1            ': Cannot both append and pass through.'
#ifdef SUNSYSTEM
              call flush(LERR)
#endif
              write (LER , *) name(1:nblen(name)), ' ', rname,
     1            ': Cannot both append and pass through.'

              ierror = ierror + 1

          endif

          if (ipass(curcon) .ne. 0 .and. canread(ucinput) .eq. 0) then

              write (LERR, *) name(1:nblen(name)), ' ', rname,
     1            ': Cannot pass through from non-input',
     1            ' connection ', ucinput, '.'
#ifdef SUNSYSTEM
              call flush(LERR)
#endif
              write (LER , *) name(1:nblen(name)), ' ', rname,
     1            ': Cannot pass through from non-input',
     1            ' connection ', ucinput, '.'

              ierror = ierror + 1

          endif

      endif

      trhlen(curcon) = trlen(curcon) - n1(curcon)


      if (ierror .eq. 0 .and. (n1(curcon) .lt. 0 .or.
     1    vn2(curcon) .le. 0 .or.
     1    vn3(curcon) .lt. 0 .or.
     1    (append(curcon) .eq. 0 .and. vn3(curcon) .le. 0))
     1                                                     ) then

          write (LERR, *) name(1:nblen(name)), ' ', rname,
     1            ': Invalid dataset dimensions!'
          write (LERR, *) '     NumSmp = ', n1(curcon)
          write (LERR, *) '     NumTrc = ', vn2(curcon)
          write (LERR, *) '     NumRec = ', vn3(curcon)
#ifdef SUNSYSTEM
          call flush(LERR)
#endif
          write (LER , *) name(1:nblen(name)), ' ', rname,
     1            ': Invalid dataset dimensions!'
          write (LER , *) '     NumSmp = ', n1(curcon)
          write (LER , *) '     NumTrc = ', vn2(curcon)
          write (LER , *) '     NumRec = ', vn3(curcon)

          ierror = ierror + 1

      endif


      if (ierror .eq. 0 .and.
     1    (trlen(curcon) .le. 0 .or. trhlen(curcon) .lt. 0 .or.
     1    trhlen(curcon) .gt. trlen(curcon))) then

          write (LERR, *) name(1:nblen(name)), ' ', rname,
     1            ': Invalid trace length!'
          write (LERR, *) '     NumSmp = ', n1(curcon)
          write (LERR, *) '     Words per trace = ', trlen(curcon)
          write (LERR, *) '     Words per trace header = ',
     1                                               trhlen(curcon)
#ifdef SUNSYSTEM
          call flush(LERR)
#endif
          write (LER , *) name(1:nblen(name)), ' ', rname,
     1            ': Invalid trace length!'
          write (LER , *) '     NumSmp = ', n1(curcon)
          write (LER , *) '     Words per trace = ', trlen(curcon)
          write (LER , *) '     Words per trace header = ',
     1                                               trhlen(curcon)

          ierror = ierror + 1

      else

#ifdef DEBUG
          write (LERR, *) '     NumSmp = ', n1(curcon)
          write (LERR, *) '     Words per trace = ', trlen(curcon)
          write (LERR, *) '     Words per trace header = ',
     1                                               trhlen(curcon)
#ifdef SUNSYSTEM
          call flush(LERR)
#endif
#endif

      endif


      write (LERR, *)

      if (ierror .gt. 0) then
          goto 667
      endif


c*******************************************************************
c
c Save any other passed information we'll need later
c
c*******************************************************************


      if (nblen(clparam) .gt. 0) then
          namstr = clparam(1:nblen(clparam))
      else
          namstr = ' '
      endif



      nc(curcon) = ncomp
      icon = curcon


      do icomp = 1, ncomp

          isrc = complist(COMPSRC,icomp)
          irec = complist(COMPREC,icomp)

          LCOMP(COMPSRC,icomp,curcon) =
     1                      isrc
          LCOMP(COMPREC,icomp,curcon) =
     1                      irec
      enddo

c
c    Keep track of whether we can seek on all components
c    of this MC file.
c
      iseek = 1



c********************************************************************
c
c Figure out the MCTrSp and pass-through status for this file.
c
c********************************************************************

c
c    Increment the "number of times we checked the command line"
c    counter. We need a separate counter for this because it's
c    possible the opening of this file will fail later on, and the
c    file won't actually get opened and curcon will get decremented
c    back again... but we still will have irrevocably read the command
c    line nevertheless!
c
      curclcon = curclcon + 1

      if (curclcon .gt. maxfile) then
       write (LERR, *) name(1:nblen(name)), ' ', rname,
     1              ': More files checked than promised to uspioinit().'
#ifdef SUNSYSTEM
       call flush(LERR)
#endif
       write (LER , *) name(1:nblen(name)), ' ', rname,
     1              ': More files checked than promised to uspioinit().'
       write (LER , *) name(1:nblen(name)),
     1              ': FATAL'
       stop
      endif


      whichopen(curclcon) = 2


      if (nblen(wclparam) .gt. 0) then
          winstr(curclcon) = wclparam(1:nblen(wclparam))
      else
          winstr(curclcon) = ' '
      endif

c
c    Check to see if this one has already been read
c    for a previous input file.
c
      do ifile = 1, curclcon - 1
          if (whichopen(ifile) .eq. 2 .and.
     1               winstr(ifile) .eq. winstr(curclcon)) then
c
c    It has; use the numbers we read that time and jump over
c    the part where we check the command line. (Remember
c    the command-line parsing routines only let you read any
c    given field in the command line ONCE!)
c
              mctrsp = clmctrsp(ifile)
              goto 100
          endif
      enddo

c
c    We haven't looked for this one before. Go look for it on
c    the command line. We have to use a temporary variable
c    because FORTRAN doesn't understand zero-length strings.
c

      tempstring = '-mc'//winstr(curclcon)
      call argi4 (tempstring(1:nblen(tempstring)), mctrsp,  -1,  -1)

      if (mctrsp .lt. -1 .or. mctrsp .gt. MC_LINE) then

          write (LERR, *) name(1:nblen(name)), ' ', rname,
     1              ': Invalid -mc mctrsp; ignoring.'
#ifdef SUNSYSTEM
          call flush(LERR)
#endif
          write (LER , *) name(1:nblen(name)), ' ', rname,
     1              ': Invalid -mc mctrsp; ignoring.'

          mctrsp = -1

      endif

 100  continue
c
c    Save what we got in case we need to refer to it again later
c
      clmctrsp(curclcon) = mctrsp


      if (ncomp .eq. 1 .and.
     1    complist(COMPSRC,1) .eq. SCALAR) then
c
c    If it's scalar, then it's scalar, no matter what mctrsp
c    may have been specified anywhere else!
c

          mctrsp = MC_SCALAR

      else if (ncomp .eq. 1 .and.
     1    complist(COMPSRC,1) .ne. SCALAR) then
c
c    If there is only one component, might as well have mctrsp be 1;
c    all the different mctrsp modes become equivalent in this case.
c

          mctrsp = MC_TRACE

      else
c
c    If they specified the output mctrsp on the command line, use that.
c    If they didn't, then use the default mctrsp given in the argument
c    list, or if there isn't one then use the mctrsp of the associated
c    input. (If there isn't an associated input, then use MC_GATHER.)
c

          if (mctrsp .lt. 0) then
              if (dmctrsp .ge. 0) then
                  mctrsp = dmctrsp
              else if (ucinput .gt. 0) then
                  mctrsp = inmctrsp(ucinput)
              else
                  mctrsp = MC_GATHER
              endif
          endif

      endif

c
c    Note! We still aren't QUITE through figuring out mctrsp.
c    If we're inside IKP, then the value depends on how the output
c    spigots are open/grounded. If there is no -O name, then we
c    can't use mctrsp = 0.
c

#ifdef DEBUG
          write (LERR, *) name(1:nblen(name)), ' ', rname,
     1              ': mctrsp = ', mctrsp, '.'
#endif


c********************************************************************
c
c Collect some background information about our current environment.
c
c********************************************************************

c
c    Look for input file name '-Osuffix', where clparam gives the
c    suffix. Spaces mean just look for '-O'. (Fortran doesn't allow
c    null strings, so we have to do a bit of kluging.)
c

      tempstring = '-O'//namstr

#ifdef DEBUG
      write (LERR, *) 'Searching command line for ',
     1    '''', tempstring(1:nblen(tempstring)), '''.'
#endif

      call argstr (tempstring(1:nblen(tempstring)), oufile, ' ', ' ')

#ifdef DEBUG
      if (oufile .eq. ' ') then
          write (LERR, *) 'Did not find it.'
      else
          write (LERR, *) 'Found ', '''',
     1        oufile(1:nblen(oufile)), '''.'
      endif
#endif

c
c    Copy across the file name found. roufile is the root output
c    file name. (If it's stdout update it now. If it's an IKP socket,
c    we'll update it later.)
c
      roufile = oufile
      if (roufile .eq. ' ') then
          roufile = 'stdout'
      endif


c
c    Are we in IKP?
c
      ikp = in_ikp()
      ikpsoungr = 1

      if (ikp .ne. 0) then
          write (LERR, *) name(1:nblen(name)), ' ', rname,
     1                ': running under the control of IKP.'

c
c    In IKP-style multiple-socket output, the corresponding
c    single output (usually will be stdout, but can be any
c    socket number) should be connected to ground. If we've
c    found an output file name already, though, don't bother.
c

          if (oufile .eq. ' ' .and. ikpsock .ge. 0) then
              ikpsoungr = pipchk(ikpsock, istat)
          else
              ikpsoungr = 0
          endif

          if (ikpsoungr .gt. 0) then
              if (ikpsock .eq. STDOUTNO) then
                  write (LERR, *) name(1:nblen(name)), ' ', rname,
     1            ': Single output (stdout) is connected.'
              else
                  write (LERR, *) name(1:nblen(name)), ' ', rname,
     1            ': Single output (socket ', ikpsock, ') is connected.'
              endif
          endif

      endif



c********************************************************************
c
c This next part is a mess of intertwined program flows!
c Here are all the different kinds of output that are supported:
c
c  [1.] IKP style, with one socket per component, with components
c       distinguished by "canonical socket numbering". (Canonical
c       socket numbering puts the Source Component number in the
c       10's place, the Receiver Component number in the 1's place,
c       with an additional 5 for output instead of input.)
c  [2.] IKP, but with the output all out through one socket.
c  [3.] "Leon" style: one file per component, with the components
c       identified by filename suffix.
c  [4.] "Compatible" style: all components in one file,
c       with components distinguished via line header information
c       (at this point we don't yet have to worry about whether
c       the data is ordered component fast or component slow).
c       It's called "compatible" because "scalar" USP programs
c       will be able to do many sorts of reasonable processing
c       on such datasets.
c  [5.] Scalar (1 component, with source component number -1).
c
c   For outputs 2, 4, and 5 there are two sub-cases to consider: output
c   to standard out, or output to elsewhere (a named file or
c   an IKP socket).
c
c********************************************************************


      if (ikp .ne. 0 .and.
     1    oufile .eq. ' ' .and.
     1    ikpsoungr .eq. 0 .and.
     1    (ncomp .ne. 1 .or. complist(COMPSRC, 1) .ne. SCALAR)
     1                                                        ) then
c--------------------------------------------------------------------
c    [1.] IKP-style multiple-socket output case
c
c    IKP mode is set, the associated single output is grounded,
c    and there isn't an output filename specified by -Osomething
c    on the command line.
c--------------------------------------------------------------------
       multiin(curcon) = MI_MULT
c
c    In this case it's Leon-style output, so set mctrsp to 0.
c
       mctrsp = MC_SCALAR


       write (LERR, *) name(1:nblen(name)), ' ', rname,
     1 ': Using IKP-style output with ', ikpoffs, ' socket offset.'
#ifdef SUNSYSTEM
       call flush(LERR)
#endif


c
c     Construct a suitable "root output file name". Put the ikpoffs
c     into the name in case there are multiple IKP inputs/outputs.
c     We have to do a little extra work because Fortran has no
c     built-in way of left justifying and we won't allow a space
c     in the output file name.
c
       roufile = 'IKP.o'
 
       write (tempstring, '(i8)') ikpoffs
       iright = nblen(tempstring)
       ileft = iright
       do icount = 1, iright
           if (tempstring(icount:icount) .ne. ' ') then
               ileft = icount
               goto 111
           endif
       enddo
 111   continue
 
       if (iright .gt. 0) then
           roufile(6:6+(iright-ileft)) = tempstring(ileft:iright)
       endif


c
c    Look for all the requested components.
c

       do icomp = 1, ncomp
c
c    MC convention: If X is the source component number (1-3)
c    and Y is the receiver component number (1-3) then
c    unit number XY (as two digits) is the standard place to
c    read that component from. For more flexibility, however,
c    we also add an offset to the socket number. This is to allow
c    multiple multi-component files to be used, for example.
c
           isrc = complist(COMPSRC,icomp)
           irec = complist(COMPREC,icomp)

           ipipeno = isrc * 10 + irec + ikpoffs + IKP_OUT_OFFSET


c
c    If it's not connected at all don't try to open it.
c    (Note grounded is OK though, unlike for the input case!)
c

           inikpok = pipchk(ipipeno, istat)

           if (inikpok .gt. 0 .or. istat .gt. 0) then
               call sisfdfit (luou, ipipeno)
               estr = '(sisfdfit error)'
           else
               luou = -1
               estr = '(unconnected)'
           endif


           if (luou .lt. 0) then

               write (LER , *) name(1:nblen(name)), ' ', rname,
     1         ': Error opening component ', complist(COMPSRC, icomp),
     1         complist(COMPREC, icomp),
     1         ', socket ', ipipeno, ' ', estr(1:nblen(estr))
               write (LERR, *) name(1:nblen(name)), ' ', rname,
     1         ': Error opening component ', complist(COMPSRC, icomp),
     1         complist(COMPREC, icomp),
     1         ', socket ', ipipeno, ' ', estr(1:nblen(estr))
#ifdef SUNSYSTEM
               call flush(LERR)
#endif

               LUIOMC(icomp,curcon) = -1
       write(FINAME(icomp,curcon),*)
     1         'IKP spigot ', ipipeno, ' ', estr(1:nblen(estr))

               ierror = ierror + 1

           else

               LUIOMC(icomp,curcon) = luou

               if (istat .gt. 0) then

       write(FINAME(icomp,curcon),*)
     1             'IKP spigot ', ipipeno, ' ', '(grounded)'

c
c    Don't worry about whether or not we can seek on /dev/null .
c    It's a pointless question!
c

               else

       write(FINAME(icomp,curcon),*)
     1             'IKP spigot ', ipipeno

c
c    If any of them aren't a disk, then we can't seek
c
                   if (ludsk(luou) .eq. 0) then
                       iseek = 0
                   endif

               endif

           endif

       enddo

      else if (ikp .ne. 0 .and.
     1         oufile .eq. ' ' .and.
     1         ikpsoungr .gt. 0 .and. ikpsock .ne. STDOUTNO) then
c--------------------------------------------------------------------
c    [2.] IKP-style single-socket output case
c
c    IKP mode is set, there isn't an output filename specified
c    by -Osomething on the command line, and the associated
c    single output is NOT grounded, nor is it missing a connector
c    altogether. We should write to the single-output IKP socket.
c    Note if the output is standard out, we will instead handle
c    it in the catch-all case below; no special IKP processing
c    is required for that case.
c--------------------------------------------------------------------
       multiin(curcon) = MI_PIPE

       write (LERR, *) name(1:nblen(name)), ' ', rname,
     1 ': Using IKP-style single-socket output for socket ',
     1 ikpsock, '.'
#ifdef SUNSYSTEM
       call flush(LERR)
#endif

c
c    In this case, mctrsp had better not be 0 (unless we're supposed
c    to be scalar).
c

       if ((ncomp .ne. 1 .or. complist(COMPSRC,1) .ne. SCALAR) .and.
     1     mctrsp .eq. MC_SCALAR) then

           write (LERR, *) name(1:nblen(name)), ' ', rname,
     1         ': Warning, cannot use multiple-file output for ',
     1         'connection ', curcon, ';',
     1         ' single-output IKP spigot is not grounded.'
           write (LERR, *) 
     1     '        Using single-file output instead.'
#ifdef SUNSYSTEM
           call flush(LERR)
#endif


c
c     Construct a suitable "root output file name". Put the ikpsock
c     into the name in case there are multiple IKP inputs/outputs.
c     We have to do a little extra work because Fortran has no
c     built-in way of left justifying and we won't allow a space
c     in the output file name.
c
       roufile = 'IKP.s'
 
       write (tempstring, '(i8)') ikpsock
       iright = nblen(tempstring)
       ileft = iright
       do icount = 1, iright
           if (tempstring(icount:icount) .ne. ' ') then
               ileft = icount
               goto 112
           endif
       enddo
 112   continue
 
       if (iright .gt. 0) then
           roufile(6:6+(iright-ileft)) = tempstring(ileft:iright)
       endif


           if (ncomp .eq. 1 .and. complist(COMPSRC,1) .ne. SCALAR) then
c
c    If there is only one component, might as well have mctrsp be 1;
c    all the different mctrsp modes become equivalent in this case.
c

               mctrsp = MC_TRACE

           else

c
c    Just use the usual default; single-component gathers.
c

               mctrsp = MC_GATHER

           endif

       endif

c
c    If we got here, there must be a non-grounded output file
c    to write to. So no need to check again.
c
       ipipeno = ikpsock
       call sisfdfit (luou, ipipeno)

       if (luou .lt. 0) then

           write (LER , *) name(1:nblen(name)), ' ', rname,
     1     ': Error from sisfdfit opening socket number ', ipipeno, '.'
           write (LERR, *) name(1:nblen(name)), ' ', rname,
     1     ': Error from sisfdfit opening socket number ', ipipeno, '.'
#ifdef SUNSYSTEM
           call flush(LERR)
#endif

           do icomp = 1, ncomp
       LUIOMC(icomp,curcon) = -1
       write(FINAME(icomp,curcon),*)
     1         'IKP spigot ', ipipeno, ' (sisfdfit error)'
           enddo

           ierror = ierror + 1

       else

           do icomp = 1, ncomp
       LUIOMC(icomp,curcon) = luou
       write(FINAME(icomp,curcon),*)
     1         'IKP spigot ', ipipeno
           enddo

c
c    If it's not a disk, then we can't seek
c
           if (ludsk(luou) .eq. 0) then
               iseek = 0
           endif
       endif


      else
c--------------------------------------------------------------------
c
c    NOT the IKP-style case.
c
c--------------------------------------------------------------------

       if (ikp .ne. 0) then
           write (LERR, *)  name(1:nblen(name)), ' ', rname,
     1     ': Falling through to non-IKP-style output.'
       endif


c
c    Can't have Leon-style multiple output files if there is no
c    -O output file base name to construct the multiple file names
c    from!
c

       if ((ncomp .ne. 1 .or. complist(COMPSRC,1) .ne. SCALAR) .and.
     1     mctrsp .eq. MC_SCALAR .and. oufile .eq. ' ') then

           write (LERR, *) name(1:nblen(name)), ' ', rname,
     1         ': Warning, cannot use multiple-file output for ',
     1         'connection ', curcon,
     1         ' without a base output file name.'
           write (LERR, *) 
     1     '        Using single-file output instead.'
#ifdef SUNSYSTEM
           call flush(LERR)
#endif

           if (ncomp .eq. 1 .and. complist(COMPSRC,1) .ne. SCALAR) then
c
c    If there is only one component, might as well have mctrsp be 1;
c    all the different mctrsp modes become equivalent in this case.
c

               mctrsp = MC_TRACE

           else

c
c    Just use the usual default; single-component gathers.
c

               mctrsp = MC_GATHER

           endif

       endif



       if (mctrsp .eq. MC_SCALAR .and.
     1     (ncomp .ne. 1 .or. complist(COMPSRC,1) .ne. SCALAR)) then

c--------------------------------------------------------------------
c     [3.] Leon-style Multiple Output File case
c--------------------------------------------------------------------
           multiin(curcon) = MI_MULT

c
c    This shouldn't be able to happen, but just in case someone
c    later changes the code or there's a bug, we'll trap it.
c
           if (oufile .eq. ' ') then
               oufile = 'unspecified'
           endif

           mcoufile = oufile
           mcoulen = nblen(mcoufile)
           mcoufile(mcoulen+1:mcoulen+1) = '.'

           do icomp = 1, ncomp

               write(mcoufile(mcoulen+2:mcoulen+2),'(i1)')
     1               complist(COMPSRC, icomp)
               write(mcoufile(mcoulen+3:mcoulen+3),'(i1)')
     1               complist(COMPREC, icomp)


               call getln(luou , mcoufile, cmode, -1)

               if (luou .lt. 0) then
                   write (LERR, *)
     1                  name(1:nblen(name)), ' ', rname,
     1                  ': Could not open ',
     1                  mcoufile(1:nblen(mcoufile)), ', ',
     1                  ' Component ', complist(COMPSRC, icomp),
     1                  complist(COMPREC, icomp)
#ifdef SUNSYSTEM
                   call flush(LERR)
#endif
                   write (LER , *)
     1                   name(1:nblen(name)), ' ', rname,
     1                   ': Could not open ',
     1                   mcoufile(1:nblen(mcoufile)), ', ',
     1                   ' Component ', complist(COMPSRC, icomp),
     1                   complist(COMPREC, icomp)

       LUIOMC(icomp,curcon) = -1
       write(FINAME(icomp,curcon),*)
     1                  mcoufile(1:nblen(mcoufile)),
     1                  ' ', '(error)'
                   ierror = ierror + 1

               else

       LUIOMC(icomp,curcon) = luou
       write(FINAME(icomp,curcon),*)
     1                  mcoufile(1:nblen(mcoufile))
c
c    If any of them aren't a disk, then we can't seek
c
                   if (ludsk(luou) .eq. 0) then
                       iseek = 0
                   endif

               endif

           enddo


       else
c----------------------------------------------------------------------
c
c     [4.], [5.]
c     (and [2.] also, when IKP single output is stdout)
c
c     The Single Output File case:
c
c     We've got a single output file to write to. If "oufile"
c     is blank, the output file must be standard out. (The case
c     of a single output IKP socket not to standard out was
c     already handled in another case above.)
c
c----------------------------------------------------------------------

           if (ncomp .ne. 1 .or. complist(COMPSRC, 1) .ne. SCALAR) then
c
c [4.] MC case
c
               if (oufile .ne. ' ') then
                   write (LERR, *) name(1:nblen(name)), ' ', rname,
     1                           ': Found ''',
     2                           oufile(1:nblen(oufile)), ''';',
     3                           ' attempting to open it for MC I/O.'
               else
                   if (ikp .eq. 0) then
                        write (LERR, *) name(1:nblen(name)), ' ', rname,
     1                  ': Attempting to open stdout for MC output.'
                   else
                        write (LERR, *) name(1:nblen(name)), ' ', rname,
     1                  ': Opening stdout for MC output, instead of ',
     1                  'multiple IKP sockets.'
                   endif
               endif
           else
c
c [5.] Scalar case
c
               if (oufile .ne. ' ') then
                   write (LERR, *) name(1:nblen(name)), ' ', rname,
     1                      ': Found ''',
     2                      oufile(1:nblen(oufile)), ''';',
     3                      ' attempting to open it for scalar I/O.'
               else
                    write (LERR, *) name(1:nblen(name)), ' ', rname,
     1              ': Attempting to open stdout for scalar output.'
               endif
           endif

#ifdef SUNSYSTEM
           call flush(LERR)
#endif

c
c [4.] and [5.] together
c

           estr = ' '

c
c    Now that we know it's there, let's try to connect to it.
c
           call getln(luou , oufile, cmode, STDOUTNO)

c
c    Just in case the user goofed and didn't connect standard output
c    to anything... let's check and make sure it's not connected to
c    their terminal!
c
           if (luou .ge. 0) then
               if (lutty(luou) .ne. 0) then

                   write (LERR, *) name(1:nblen(name)), ' ', rname,
     1             ': This output is connected to a terminal!'
#ifdef SUNSYSTEM
                   call flush(LERR)
#endif
                   write (LER , *) name(1:nblen(name)), ' ', rname,
     1       ': You are attempting to write usp data to your terminal!'

                   estr = '(a terminal)'
                   ierror = ierror + 1

               endif
           else
c
c    We returned from getln with an error. (Somebody must have
c    turned off the automatic abort on not being able to open
c    a file! Normally if getln has problems, it will abort the
c    whole program right then and there.)
c
               write (LERR, *) name(1:nblen(name)), ' ', rname,
     1         ': Error from getln().'
#ifdef SUNSYSTEM
               call flush(LERR)
#endif
               write (LER , *) name(1:nblen(name)), ' ', rname,
     1         ': Error from getln().'

               estr = '(getln error)'
               ierror = ierror + 1

           endif

c
c    See whether we can seek or not
c
           if (luou .ge. 0) then
               if (ludsk(luou) .eq. 0) then
                   iseek = 0
               endif
           endif

c
c    For now, set everything up according to the assumption that
c    we won't be able to open the file multiple times. (We will
c    try to "upgrade" to MI_MOPEN later.)
c
           multiin(curcon) = MI_PIPE

           do icomp = 1, ncomp

       LUIOMC(icomp,curcon) = luou

               if (estr .eq. ' ') then
                   if (oufile .ne. ' ') then
       write(FINAME(icomp,curcon),*)
     1                 oufile(1:nblen(oufile))
                   else
       write(FINAME(icomp,curcon),*)
     1                 'stdout'
                   endif
               else
                   if (oufile .ne. ' ') then
       write(FINAME(icomp,curcon),*)
     1                 oufile(1:nblen(oufile)), ' ', estr(1:nblen(estr))
                   else
       write(FINAME(icomp,curcon),*)
     1                 'stdout', ' ', estr(1:nblen(estr))
                   endif
               endif

           enddo


       endif
      endif



c********************************************************************
c********************************************************************
c
c    WHEW! We're done tracking down all the files!!!!!
c
c    Now that we've got them all, do some double-checking on them.
c
c********************************************************************
c********************************************************************


c
c    Call uspmic to see if we're trying to write more than one output
c    file to the same logical unit... that can't possibly work.
c    It will also check to see if we're actually able to read and
c    write as required for our current file. If we can't, it will
c    update the values of "canread" and "canwrite" accordingly.
c    (It can also increment ierror, depending on a switch in the code,
c    in which case it's a fatal error.)
c

      call uspmic(ierror)

c
c    If we're appending, we'll have to be able to seek back to
c    rewrite the line header at the end. So appending requires
c    being able to seek.
c

      if (ierror .eq. 0 .and.
     1                iseek .eq. 0 .and. append(curcon) .ne. 0) then

          ierror = ierror + 1

          write (LERR, *) name(1:nblen(name)), ' ', rname,
     1    ': Sorry, must be able to seek to append to a file.'
#ifdef SUNSYSTEM
          call flush(LERR)
#endif
          write (LER , *) name(1:nblen(name)), ' ', rname,
     1    ': Sorry, must be able to seek to append to a file.'

      endif


c********************************************************************
c
c Just give up and return if we had any serious errors.
c
c********************************************************************

      write (LERR,*)

      if (ierror .gt. 0) then
          write (LERR, *) name(1:nblen(name)), ' ', rname,
     1    ': ', ierror,
     1    ' serious error(s) occurred opening one or more files.'
          write (LERR, *) name(1:nblen(name)), ' ', rname,
     1    ': Giving up on opening output number ', curcon, '.'

          goto 666
      endif


      luseek(curcon) = iseek

      if (iseek .eq. 0) then
           write (LERR, *) name(1:nblen(name)), ' ', rname,
     1                     ': No seeking backwards on output ', curcon
      else
           write (LERR, *) name(1:nblen(name)), ' ', rname,
     1                     ': We can seek on output ', curcon
      endif

c
c    Let the user know whether they can seek or not.
c
 
      if (ipass(curcon) .ne. 0) then
          seekstat = 2
      else
          if (luseek(curcon) .eq. 0) then
              seekstat = 1
          else
              seekstat = 0
          endif
      endif


      write (LERR, *)


c**********************************************************************
c
c See if we need to turn the large SISIO buffers off for efficiency.
c
c For now, if seekalot is either 1 or 2 then turn off large buffers.
c NOTE! We may want to change this if the logic in sisio for combining
c seeks and large buffers is improved. ( ??????  !!!!!! )
c Note there is one other place further down (when we multiply open
c a single file) where we also check the value of seekalot and decide
c whether to turn off buffering or not.
c
c**********************************************************************

      if (seekalot .ne. 0) then

          write (LERR, *) name(1:nblen(name)), ' ', rname,
     1        ': Turning large buffers off for output ', curcon

          if (multiin(curcon) .eq. MI_PIPE .and.
     1                                       nc(curcon) .gt. 0) then
              mcomp = 1
          else
              mcomp = nc(curcon)
          endif

          do icomp = 1, mcomp

              if (
     1 LUIOMC(icomp,curcon)
     1                             .ge. 0) then

                  call sislgbuf(
     1 LUIOMC(icomp,curcon)
     1                                  , 'off')
              endif
          enddo

      endif


#ifdef DEBUG
c
c    Write out the names of all the files we opened.
c

      if (multiin(curcon) .eq. MI_PIPE .and.
     1                                   nc(curcon) .gt. 0) then
          mcomp = 1
      else
          mcomp = nc(curcon)
      endif

      do icomp = 1, mcomp

          write (LERR, *) '    ', 'comp ', icomp, ' = ',
     1 FINAME(icomp,curcon)
     1 (1:nblen(FINAME(icomp,curcon))),
     1          ' = logical unit ',
     1 LUIOMC(icomp,curcon)
#ifdef SUNSYSTEM
          call flush(LERR)
#endif

      enddo

#endif



c*******************************************************************
c
c    Tell the user what mctrsp we ended up with.
c
c*******************************************************************

      if (mctrsp .eq. MC_SCALAR) then

          if (ncomp .eq. 1 .and.
     1        complist(COMPSRC,1) .eq. SCALAR) then

#ifdef DEBUG
c
c    They've already been told this.
c
              write (LERR, *) name(1:nblen(name)), ' ', rname,
     1           ': Output ', curcon, ' is scalar.'
#endif

          else

              write (LERR, *) name(1:nblen(name)), ' ', rname,
     1           ': Output ', curcon, ' consists of ', ncomp,
     1           ' single-component scalar files.'

          endif

      else if (mctrsp .eq. MC_TRACE) then

          write (LERR, *) name(1:nblen(name)), ' ', rname,
     1           ': Output ', curcon, ' is component-axis fast.'

      else if (mctrsp .eq. MC_GATHER) then

          write (LERR, *) name(1:nblen(name)), ' ', rname,
     1           ': Single-component gathers in output ', curcon, '.'

      else if (mctrsp .eq. MC_LINE) then

          write (LERR, *) name(1:nblen(name)), ' ', rname,
     1           ': Single-component lines in output ', curcon, '.'

      else

          write (LERR, *) name(1:nblen(name)), ' ', rname,
     1    ': Invalid mctrsp = ', mctrsp,
     1    ', THIS SHOULD NOT BE ABLE TO HAPPEN.'
#ifdef SUNSYSTEM
          call flush(LERR)
#endif
          write (LER , *) name(1:nblen(name)), ' ', rname,
     1    ': Invalid mctrsp = ', mctrsp,
     1    ', THIS SHOULD NOT BE ABLE TO HAPPEN.'
          stop

      endif

#ifdef SUNSYSTEM
      call flush(LERR)
#endif



c*******************************************************************
c
c    Figure out the n2, n3 dimensions of the output files.
c
c*******************************************************************

      if (mctrsp .eq. MC_SCALAR) then

c
c    In this case each component is in a separate file, and
c    each file has vn2*vn3 traces in it.
c

          n2(curcon) = vn2(curcon)
          n3(curcon) = vn3(curcon)

          lunc(curcon) = 1
          lun1(curcon) = 1

      else if (mctrsp .eq. MC_TRACE) then

          n2(curcon) = vn2(curcon) * ncomp
          n3(curcon) = vn3(curcon)

          lunc(curcon) = ncomp
          lun1(curcon) = 1

      else if (mctrsp .eq. MC_GATHER) then

          n2(curcon) = vn2(curcon)
          n3(curcon) = vn3(curcon) * ncomp

          lunc(curcon) = ncomp
          lun1(curcon) = n2(curcon)

      else if (mctrsp .eq. MC_LINE) then

          n2(curcon) = vn2(curcon)
          n3(curcon) = vn3(curcon) * ncomp

          lunc(curcon) = ncomp
          lun1(curcon) = n2(curcon) * n3(curcon) / ncomp

      else

          write (LERR, *) name(1:nblen(name)), ' ', rname,
     1    ': Invalid mctrsp = ', mctrsp,
     1    ', THIS SHOULD NOT BE ABLE TO HAPPEN.'
#ifdef SUNSYSTEM
          call flush(LERR)
#endif
          write (LER , *) name(1:nblen(name)), ' ', rname,
     1    ': Invalid mctrsp = ', mctrsp,
     1    ', THIS SHOULD NOT BE ABLE TO HAPPEN.'
          stop

      endif


      write (LERR, *)  name(1:nblen(name)), ' ', rname,
     1      ': USP dimensions for output ', curcon, ':'
 
      write(LERR,*) 'NumSmp=', n1(curcon),
     1      '        SmpInt=', d1(curcon)
      write(LERR,*) 'NumTrc=', n2(curcon)
      write(LERR,*) 'NumRec=', n3(curcon)


      write (LERR, *)
      write (LERR, *) '   For output ', curcon,
     1 ':   ns = ', wf2(curcon),
     1 ',  ne = ', wn2(curcon) + wf2(curcon) - 1,
     1 ',  rs = ', wf3(curcon),
     1 ',  re = ', wn3(curcon) + wf3(curcon) - 1

      write (LERR, *)
      write (LERR, *) '   Output dataset ''Cube'' dimensions:'
      write (LERR, *) '         NumSmp = ', n1(curcon)
      if (ncomp .ne. 1 .or.
     1    complist(COMPSRC,1) .ne. SCALAR) then
          write (LERR, *) '         NumCmp = ', ncomp
      endif
      write (LERR, *) '         NumTrc = ', vn2(curcon)
      write (LERR, *) '         NumRec = ', vn3(curcon)

c
c    Flag all the cases where the input and output dimensions are
c    different:
c
c    C) We're windowing over the NumTrc or NumRec dimensions
c

      if  (
     1      wn2(curcon) .ne. vn2(curcon) .or.
     1      wn3(curcon) .ne. vn3(curcon) ) then

          write (LERR, *)
          write (LERR, *) '   Visible dataset ''Cube'' dimensions:'
          write (LERR, *) '         NumSmp = ', n1(curcon)
          if (ncomp .ne. 1 .or.
     1        complist(COMPSRC,1) .ne. SCALAR) then
              write (LERR, *) '         NumCmp = ', ncomp
          endif
          write (LERR, *) '         NumTrc = ', wn2(curcon)
          write (LERR, *) '         NumRec = ', wn3(curcon)

      endif

      write (LERR, *) 



c*******************************************************************
c
c    set L-U-C-N and L-U-O-R-D-E-R
c
c*******************************************************************

      if (multiin(curcon) .eq. MI_PIPE) then
c
c    In this case all the components are in one file.
c

c
c    L-U-C-N tells us what component number in the file goes with each
c    of our components. Since we're creating the file from scratch,
c    the order is the same and L-U-C-N just counts up.
c

c
c    L-U-ORDER tells us what order we should handle the components in.
c    In this case they're already in the right order, so just count
c    up.
c

          do icomp = 1, nc(curcon)

              LUCN(icomp,curcon) = icomp
              LUORDER(icomp,curcon) = icomp

          enddo

      else

          do icomp = 1, nc(curcon)
c
c    In this case each component is in a separate file, so L-U-C-N
c    is always 1.
c

              LUCN(icomp,curcon) = 1
              LUORDER(icomp,curcon) = icomp

          enddo

      endif



c*******************************************************************
c
c    Figure out how we're going to transpose
c
c Here are the possibilities, in order of preference:
c
c    lustatus    multiin
c ------------------------------------------------------------------
c 1) IO_OK       (either): It doesn't need to be transposed at all!
c 3) IO_MEMBUF,  MI_PIPE:  We can transpose it in a memory buffer
c 4) IO_SKIP,    MI_MOPEN: We can open the same file several times
c 5) IO_SEEK,    MI_PIPE:  We can transpose it by seeking on disk
c 6) IO_NOK             :  Uh oh. The user is going to be disappointed
c
c***********************************************************************


c
c (1)
c Check for an important special case: the traces happen to be
c in exactly the order we want to write them.
c
c Note this branch will ALWAYS catch the Leon-style multiple-file
c output case!
c
      if (ierror .eq. 0 .and.
     1    lun1(curcon) .eq. 1) then

          lustatus(curcon) = IO_OK

      endif

c
c (3)
c Try setting up a transpose buffer in memory.
c Make sure the transpose area required is of reasonable size, and the
c user isn't planning on jumping around between gathers much.
c (If seeking isn't possible, then try it this way even if they are
c going to jump around a lot, as it's their only remaining possibility!)
c

      if (ierror .eq. 0 .and.
     1    lustatus(curcon) .eq. IO_NOK .and.
     1    lun1(curcon)*lunc(curcon)*
     1    (n1(curcon)+trhlen(curcon))*SZSMPD
     1    .le. MAXTBUFSZ
     1    .and.
     1    (seekalot .le. 1 .or. luseek(curcon) .eq. 0)
     1                                            ) then


          if (usptba(curcon,
     1         lun1(curcon)*lunc(curcon)*
     1         (n1(curcon)+trhlen(curcon))*SZSMPD,
     1         maxfile, ierror) .ne. 0) then

              write(LERR,*)
     1        name(1:nblen(name)), ' ', rname,
     1        ': Unable to allocate ',
     1        lun1(curcon)*lunc(curcon)*
     1        (n1(curcon)+trhlen(curcon))*SZSMPD,
     1        ' bytes for transpose buffer.'
#ifdef SUNSYSTEM
              call flush(LERR)
#endif
          else
              write(LERR,*)
     1        name(1:nblen(name)), ' ', rname,
     1        ': allocating ',
     1        lun1(curcon)*lunc(curcon)*
     1        (n1(curcon)+trhlen(curcon))*SZSMPD,
     1        ' bytes,'
              write(LERR,*) '               ',
     1        lun1(curcon)*lunc(curcon), ' traces ',
     1        'for transpose buffer ',
     1        curcon, '.'

              lustatus(curcon) = IO_MEMBUF

c
c If seekalot is "1", then we should turn the buffers back on again.
c Seeking within a gather will be handled within the transpose
c buffer, so won't matter to sisio, so the buffers can be left on.
c
              if (seekalot .eq. 1) then

                  write (LERR, *)
     1                name(1:nblen(name)), ' ', rname,
     1  ': Turning large buffers on again for output ', curcon

                  call sislgbuf(luou, 'on')

              endif

          endif

      endif


c
c (4)
c If we can open the same file multiple times, try doing that.
c There are no outstanding errors
c We haven't found a better way of doing it yet
c We need to write more than one component
c We have a file name for it
c We can seek on it
c The component axis is not fast
c The user isn't planning on seeking around between gathers a lot
c

c
c ????? Unfortunately, SISIO screws this case up, so we can't use it for
c writing at this time.
c
      if (.false. .and. ierror .eq. 0 .and.
     1   lustatus(curcon) .eq. IO_NOK .and.
     1   ncomp .gt. 1  .and.
     1   oufile .ne. ' ' .and.
     1   luseek(curcon) .ne. 0 .and.
     1   lun1(curcon) .ne. 1 .and.
     1   seekalot .le. 1) then

          call galloc (pluousv, ncomp * SZSMPD, ierr, 1)

c
c The file is already open once; use that one for the first component.
c

          icomp = 1
          luousv(icomp) = luou

          do icomp = 2, ncomp

              call getln(luousv(icomp) , oufile, cmode, -1)

              if (luousv(icomp) .lt. 0) then

                  write (LERR, *)
     1            name(1:nblen(name)), ' ', rname,
     1            ': Unable to reopen ''',
     1            oufile(1:nblen(oufile)), '''.'
#ifdef SUNSYSTEM
                  call flush(LERR)
#endif
                  write (LER , *)
     1            name(1:nblen(name)), ' ', rname,
     1            ': Unable to reopen ''',
     1            oufile(1:nblen(oufile)), '''.'

c
c    Be neat and close the ones we just opened, since it turns out
c    we can't use them after all.
c
                  do icomp2 = 2, icomp - 1
                      call lbclos (luousv(icomp2))
                  enddo

                  goto 220

              else

c
c We managed to open it again! Set the buffering on this new stream,
c same as we already did for the first.
c
                  if (seekalot .ne. 0) then
                      call sislgbuf(luousv(icomp), 'off')
                  endif

              endif
          enddo

c
c Yup, we managed to open the file multiple times.
c
          do icomp = 2, ncomp
       LUIOMC(icomp,curcon) =
     1                                luousv(icomp)
          enddo

          multiin(curcon) = MI_MOPEN
          lustatus(curcon) = IO_SKIP


c*******************************************************************
c
c NOTE! NOTE! NOTE! NOTE!   ???????????????  !!!!!!!!!!!!!!!!!
c
c For future efficiency, at this point in the code we should
c explicitly SET the buffer size for EACH of the multiple outputs
c in sisio (including the first) to correspond to the size:
c
c       lun1 * (trlen * SZSMPD + sizeof(a greenword))
c
c *UNLESS*, of course, the buffers were already smaller than that;
c in that case leave them alone.
c
c That will be the "natural buffer size" for the dataset. Any larger
c buffers than that would be wasted (smaller ones would be OK, though).
c Unfortunately as of this time there is no provision in sisio for
c setting the buffer size from user code.
c
c*******************************************************************


220       continue

          call gfree (pluousv)

      endif


c
c (5)
c If we can seek, try doing that as a last resort.
c Probably this means that "seekalot" is 2, meaning the user is
c planning on seeking _between different gathers_ a lot. If that's
c the case, it's pointless to try any fancy buffering; we should
c just seek to each individual component trace as we need it.
c
      if (ierror .eq. 0 .and.
     1    lustatus(curcon) .eq. IO_NOK .and.
     1    luseek(curcon) .ne. 0) then

          lustatus(curcon) = IO_SEEK

c
c Turn off buffers in SISIO. They behave very ungracefully
c when there are lots of seeks, repeatedly flushing buffers again
c and again. (Buffers are even flushed when the seek is a no-op!)
c

          write (LERR, *)
     1        name(1:nblen(name)), ' ', rname,
     1  ': Ensuring large buffers are off for output ', curcon

          call sislgbuf(luou, 'off')

      endif


c
c (6)
c Uh oh! We've run out of options!
c
      if (ierror .eq. 0 .and.
     1                lustatus(curcon) .eq. IO_NOK) then

          write(LERR,*) name(1:nblen(name)), ' ', rname,
     1    ': Unable to find a way to transpose output ',
     1    curcon, '.'
#ifdef SUNSYSTEM
          call flush(LERR)
#endif
          write(LER ,*) name(1:nblen(name)), ' ', rname,
     1    ': Unable to find a way to transpose output ',
     1    curcon, '.'

          ierror = ierror + 1

      endif


      if (ierror .gt. 0) then
          write (LERR, *) name(1:nblen(name)), ' ', rname,
     1    ': ', ierror,
     1    ' serious error(s) occurred opening one or more files.'
          write (LERR, *) name(1:nblen(name)), ' ', rname,
     1    ': Giving up on opening output number ', curcon, '.'

          goto 666
      endif



c*******************************************************************
c
c    Update or construct an outgoing line header and write it out
c    to each open file as necessary.
c
c    First, fill in the required parameters in the line header.
c
c*******************************************************************

c
c Data dimensions
c
      call savew( lheader, 'NumSmp', n1(curcon) , LINEHEADER)
      call savew( lheader, 'NumTrc', n2(curcon)  , LINEHEADER)
      call savew( lheader, 'NumRec', n3(curcon)  , LINEHEADER)

c
c Converting the sample interval into a USP-style SmpInt key
c requires a little more work.
c

      if (d1(curcon) .le. 0.) then
         write(LER ,*) name(1:nblen(name)),
     1         ': warning, sample interval ', d1(curcon),
     1         ' is not positive.'
         write(LERR,*) name(1:nblen(name)),
     1         ': warning, sample interval ', d1(curcon),
     1         ' is not positive.'
      endif
 

c
c    First try to convert it as milliseconds
c

      nsi = nint (d1(curcon) * 1000.)

      if (nsi .gt. 32 .or. (real(nsi) / 1000.) .ne. d1(curcon)) then
c
c    OK, try it as microseconds then
c

          nsi = nint (d1(curcon) * 1000000.)

          if (nsi .le. 32) then

c
c    Oops, milliseconds will have to do.
c

              nsi = nint (d1(curcon) * 1000.)

              write(LER ,*) name(1:nblen(name)),
     1             ': sample interval ', d1(curcon),
     1             ' is not a whole number of milliseconds;',
     1             ' using ', (real(nsi) / 1000.), ' instead.'
              write(LERR,*) name(1:nblen(name)),
     1             ': sample interval ', d1(curcon),
     1             ' is not a whole number of milliseconds;',
     1             ' using ', (real(nsi) / 1000.), ' instead.'

          else

              if ((real(nsi) / 1000000.) .ne. d1(curcon)) then
                  write(LER ,*) name(1:nblen(name)),
     1                 ': sample interval ', d1(curcon),
     1                 ' is not a whole number of microseconds;',
     1                 ' using ', (real(nsi) / 1000000.), ' instead.'
                  write(LERR,*) name(1:nblen(name)),
     1                 ': sample interval ', d1(curcon),
     1                 ' is not a whole number of microseconds;',
     1                 ' using ', (real(nsi) / 1000000.), ' instead.'
              endif

          endif

      endif


      call savew( lheader, 'SmpInt',  nsi  , LINEHEADER)


c
c    Multi-Component information
c

      if (ncomp .eq. 1 .and. complist(COMPSRC,1) .eq. SCALAR) then
c
c    If it's scalar, then "we don't know anything about this newfangled
c    MC stuff", and we should simply leave the MC-related line header
c    entries unmolested.
c

      else if (multiin(curcon) .eq. MI_MULT) then

c
c    One component per file ("Leon style"), with each component being
c    in a scalar file.
c
c    In that case, NumCmp and MCTrSp are zero.
c

          numcmp = 0
          call savew( lheader, 'NumCmp',  numcmp  , LINEHEADER)
          call savew( lheader, 'MCTrSp',  mctrsp  , LINEHEADER)

      else
c
c    One file (possibly opened multiple times)
c
          numcmp = ncomp
          call savew( lheader, 'NumCmp',  numcmp  , LINEHEADER)
          call savew( lheader, 'MCTrSp',  mctrsp  , LINEHEADER)

c
c    Write component list
c

          if (ncomp .gt. MCLLen) then
c
c    The maximum number of components that can be stored in a USP
c    line header.
c
              ncomp2 = MCLLen

              write (LERR, *) name(1:nblen(name)), ' ', rname,
     1        ': Too many components (', ncomp, ');',
     1        ' truncating to ', ncomp2, '.'
 
#ifdef SUNSYSTEM
              call flush(LERR)
#endif
              write (LER , *) name(1:nblen(name)), ' ', rname,
     1        ': Too many components (', ncomp, ');',
     1        ' truncating to ', ncomp2, '.'

          else

              ncomp2 = ncomp

          endif


          do icomp = 1, ncomp2

              isrc = complist(COMPSRC,icomp)
              irec = complist(COMPREC,icomp)

              comps(icomp) = COMPBASE * isrc + irec

          enddo

          call savelu('MCList', ifmt_MCList, l_MCList,
     1             ln_MCList, LINEHEADER)
 
          call savew2(lheader, ifmt_MCList, l_MCList,
     1               ncomp2, comps, LINEHEADER)

      endif


c
c If we're creating this from scratch, save the current
c command line stuff in the historical line header.
c

      if (lhedlength .eq. 0) then

c
c Create a header from scratch
c
          iform = USUAL_USP_FORMAT
          call savew( lheader, 'Format', iform , LINEHEADER)

          nbytes = 2 * SZHFWD
          call savew( lheader, 'HlhEnt',  0   , LINEHEADER)
          call savew( lheader, 'HlhByt', nbytes , LINEHEADER)


          write (LERR, *)
     1'---------------------------------------------------------------',
     1'--------------'
          write (LERR, *) 'Line header for output #', curcon
          write (LERR, *)

          lbytes = HSTOFF

c
c Update the historical part of the line header.
c

          call hlhprt (lheader, lbytes, name, nblen(name), LERR)

c
c Inject the current command line into the historical line header.
c
          call savhlh( lheader, lbytes, lbyout )
          lhedlength = lbyout

          write (LERR, *)
     1'---------------------------------------------------------------',
     1'--------------'

      endif

      write (LERR,*)
      write (LERR, *) name(1:nblen(name)), ' ', rname,
     1   ': Line header length for connection ', curcon,
     1   ' is ', lhedlength, ' bytes.'

c
c Finally, write the output line header OUT!
c

      if ((multiin(curcon) .eq. MI_MULT) .or.
     1    (multiin(curcon) .eq. MI_MOPEN) .or.
     1    (ncomp .eq. 1 .and. complist(COMPSRC,1) .eq. SCALAR)) then

c
c Sisio would freak out if we tried to seek into the middle of a
c dataset without having written a line header first, so go ahead
c and write a line header for each component even in the "MI_MOPEN"
c case.
c
          do icomp = 1, ncomp

              call wrtape (
     1 LUIOMC(icomp,curcon),
     1                              lheader, lhedlength )

          enddo

      else

c
c Write the single MC-dataset lineheader
c

          call wrtape ( luou, lheader, lhedlength )

      endif



c*******************************************************************
c
c    Final initializations
c
c*******************************************************************

      write (LERR, *) name(1:nblen(name)), ' ', rname,
     1   ': Trace header length for connection ', curcon,
     1   ' is ', trhlen(curcon), ' full words.'
      write (LERR, *) name(1:nblen(name)), ' ', rname,
     1   ': Total trace length for connection ', curcon,
     1   ' is ', trlen(curcon), ' full words.'

c
c    This file is empty, so set the "amount that currently exists"
c    to zero.
c
      lumax(curcon) = 0
      luumax(curcon) = 0

c
c    If there is a trace transpose buffer, then THIS variable keeps
c    track of how long the file actually is on disk (versus how long
c    it appears to be to the user, who can only access the file through
c    the transpose I/O buffer)
c
      trbendpoint(curcon) = lumax(curcon)

c
c    This variable keeps track of the "expected" size of the file
c    at close time. If we aren't appending, it's an error to go
c    beyond this.
c
      luulim(curcon) = wn2(curcon) * wn3(curcon)


c
c    Make sure we have sufficient temporary space to store
c    the longest trace we can encounter
c
      if (maxtrlen .lt. ncomp * trlen(curcon) * SZSMPD) then
          maxtrlen = ncomp * trlen(curcon) * SZSMPD

          if (uspsta(maxtrlen, pscrtrace, pnulltrace) .ne. 0) then

              write(LERR,*) name(1:nblen(name)), ' ', rname,
     1        ': Unable to allocate ', maxtrlen,
     1        ' bytes for scratch trace buffer.'
#ifdef SUNSYSTEM
              call flush(LERR)
#endif
              write(LER ,*) name(1:nblen(name)), ' ', rname,
     1        ': Unable to allocate ', maxtrlen,
     1        ' bytes for scratch trace buffer.'

              ierror = ierror + 1
              goto 666

          endif

      endif


      lubufsz(curcon) = sisbufsz(
     1 LUIOMC(1,curcon))

      write(LERR,*)

      if (lubufsz(curcon) .gt. 0) then
          write(LERR,*) name(1:nblen(name)), ' ', rname,
     1    ': SIS IO buffers of ', lubufsz(curcon), ' bytes.'
      else
          write(LERR,*) name(1:nblen(name)), ' ', rname,
     1    ': SIS IO is unbuffered.'
      endif


c*******************************************************************
c
c    We're done!
c
c*******************************************************************

 666  continue

c
c    If there has been an error, make sure they can't try to use
c    this connection.
c
      if (ierror .ne. 0) then
          lustatus(curcon) = IO_NOK
      endif

c
c Summarize the results for their edification.
c
      write (LERR,*)
      call uspinsum(curcon)


 667  continue

      write (LERR, *) 

      if (ierror .eq. 0) then

          write (LERR, *) name(1:nblen(name)), ' ', rname,
     1    ': Connection ', curcon, ' is now open and ready for use.'
          write (LERR, *)
     1'++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++'

      else

c
c    Close all open files associated with this botched connection
c
          write (LERR, *) name(1:nblen(name)), ' ', rname,
     1    ': Closing down partially opened connection.'

          call uspclall (curcon)

c
c    Reset all the uspinfo structures to their initial clean state,
c    to remove all traces this embarrassing output ever happened.
c
          call uspcleanslate (curcon)
          curcon = curcon - 1
          icon = -1

          write (LERR, *)
     1'----------------------------------------------------------------'

      endif

      write (LERR, *) 
#ifdef SUNSYSTEM
      call flush(LERR)
#endif

      return
      end
