/***********************************************************************
 *                copyright 2001, Amoco Production Company             *
 *                            All Rights Reserved                      *
 *                    an affiliate of BP America Inc.                  *
 ***********************************************************************/

/*	This subroutine contains C callable functions to correspond
	to the Fortran header access routines saver,savew,saver2, and
	savew2. Because of the promotion to double of floating point
	values in C, the parameter size of an argument to be stored 
	was moved to the end of the parameter list for these C funtions.

	The callable functions and their formats are:
	
	  get_hw_val(buffer,keystr,hdr_type,value);
	  put_hw_val(buffer,keystr,hdr_type,value);
	  get_indexed_hw_val(buffer,format,index,length,hdr_type,value);
	  put_indexed_hw_val(buffer,format,index,length,hdr_type,value);

	where
	  buffer -  the address of the header to/from which values are 
		    to be stored/retrieved.

	  keystr -  mnemonic for selecting header value to be stored 
		    or retrieved.

	  hdr_type - flags contained in ~usp/include/save_defs.h for
			declaring the type of header we're dealing with.

	  format -  also defined in ~usp/include/save_defs.h: this 
		    can be retrieved for a particular mnemonic using
		    a call to the defined routine C_SAVELU().

	  index  -  index based on type. For example, if the format defined
		    a short and the index value is 2, the actual offset into
		    the header would be bytes 5 and 6, assuming 2-byte shorts.

	  length -  Number of elements of a particular type associated with
		    the given mnemonic. For example, "CrwNam" in the USP line
		    header contains six characters.

	  value  -  User-supplied value for items to be stored or address
		    for values to be retrieved. If a multi-element value
		    is being referenced, the value will always be an 
		    address.

*/
#include <stdio.h>
#include <localsys.h>
#include <save.h>
#include <save_defs.h>
#include <stdarg.h>
#include <sys/types.h>

void put_hw_int();
void put_hw_float();
void put_hw_short();
void put_hw_char();
void savewfr();

void get_hw_int();
void get_hw_float();
void get_hw_short();
void get_hw_char();
void saverfr();

int put_indexed_hw_val( caddr_t itr, int format, int index, int length, int line_trace, ...);
int get_indexed_hw_val( caddr_t itr, int format, int index, int length, int line_trace, ... );

int get_hw_val(caddr_t itr, char *str, int lintrc, ...)
{
	va_list args;
	int rc;
	int ival;
	double val;
	int format,index,length;

	va_start(args,lintrc);

	val = (double) va_arg(args, double);
	C_SAVELU(str,&format,&index,&length,lintrc);
	rc = get_indexed_hw_val(itr,format,index,length,lintrc, val);

	va_end(args);
	return(rc);
}
int put_hw_val(caddr_t itr, char *str, int lintrc, ...)
{
	va_list args;
	int rc;
	double val;
	int format,index,length;

	va_start(args,lintrc);
	val = (double) va_arg(args,double);

	C_SAVELU(str,&format,&index,&length,lintrc);
	rc = put_indexed_hw_val(itr,format,index,length,lintrc, val);

	va_end(args);
	return(rc);
}

int get_indexed_hw_val( caddr_t itr, int format, int index, int length, int line_trace, ... )
{
/*	int rc; */
	caddr_t val;
	struct hdr *hd;
	va_list args;

	va_start(args, line_trace);

	if(line_trace == LINEHEADER)
		hd = lhdr;
	else if(line_trace == TRACEHEADER)
		hd = thdr;
	else if(line_trace == VANLHEADER)
		hd = vnlhdr;
	else if(line_trace == VSPNHEADER)
		hd = vspn_hdr;
	else if(line_trace == SEGYLINEHEADER)
		hd = segy_lhdr;
	else if(line_trace == SEGYTRACEHEADER)
		hd = segy_thdr;
	else
		{
		fprintf(stderr,"Error save: invalid header flag %d\n",
				line_trace);
		return((int) 0);
		}

	if (index < 0) {
		fprintf(stderr,"Error save: negative index value %d\n",index);
		return((int) 0);
		}

	if (length < 1) {
		fprintf(stderr,"Error save: non-positive length value %d\n",
			length);
		return((int) 0);
		}

	switch (format) {
	  case SAVE_CHAR_DEF:
		val = va_arg(args,char *);
		get_hw_char(itr,val,index,length);
		break;
	  case SAVE_SHORT_DEF:
		val = (caddr_t) va_arg(args,short *);
		get_hw_short(itr,val,index,length);
		break;
	  case SAVE_LONG_DEF:
		val = (caddr_t) va_arg(args,int *);
		get_hw_int(itr,val,index,length);
		break;
	  case SAVE_FLOAT_DEF:
		val = (caddr_t) va_arg(args,double *);
		get_hw_float(itr,val,index,length);
		break;
	  case SAVE_FKFLT_DEF:
		val = (caddr_t) va_arg(args,double *);
		saverfr(itr,val,index,length);
		break;
	  }
	return((int) 1);
}

/*
int put_indexed_hw_val(caddr_t itr, int format, int index, int length, int lintrc, ...)
{
	va_list args;
	int rc;
	int ival;

	va_start(args, lintrc);

	rc = put_hw_val_by_index(itr,format,index,length,lintrc,args);

	va_end(args);
	return(rc);
}
*/

int put_indexed_hw_val( caddr_t itr, int format, int index, int length, int line_trace, ...)
{
/*	int rc; */
	short sval;
	int ival;
	float fval;
	struct hdr *hd;
	caddr_t val;
	va_list args;

	va_start(args, line_trace);

	if(line_trace == LINEHEADER)
		hd = lhdr;
	else if(line_trace == TRACEHEADER)
		hd = thdr;
	else if(line_trace == VANLHEADER)
		hd = vnlhdr;
	else if(line_trace == VSPNHEADER)
		hd = vspn_hdr;
	else if(line_trace == SEGYLINEHEADER)
		hd = segy_lhdr;
	else if(line_trace == SEGYTRACEHEADER)
		hd = segy_thdr;
	else
		{
		fprintf(stderr,"Error save: invalid header flag %d\n",
				line_trace);
		return((int) 0);
		}

	if (index < 0) {
		fprintf(stderr,"Error save: negative index value %d\n",index);
		return((int) 0);
		}

	if (length < 1) {
		fprintf(stderr,"Error save: non-positive length value %d\n",
			length);
		return((int) 0);
		}

	switch (format) {
	  case SAVE_CHAR_DEF:
		val = va_arg(args,char *);
		put_hw_char(itr,val,index,length);
		break;
	  case SAVE_SHORT_DEF:
		if (length == 1) {
#if defined( SGISYSTEM ) || defined ( LINUXSYSTEM )
		  sval = (short) va_arg(args,int); /* system bug? */
#else
		  sval = (short) va_arg(args,short);
#endif
		  val = (caddr_t) &sval;
		  }
		else {
		  val = (caddr_t) va_arg(args,short *);
		  }
		put_hw_short(itr,val,index,length);
		break;
	  case SAVE_LONG_DEF:
		if (length == 1) {
		  ival = (int) va_arg(args,int);
		  val = (caddr_t) &ival;
		  }
		else
		  val = (caddr_t) va_arg(args,int *);
		put_hw_int(itr,val,index,length);
		break;
	  case SAVE_FLOAT_DEF:
		if (length == 1) {
		  /* fval = (float) va_arg(args,float); */
		  fval = (float) va_arg(args,double);
		  val = (caddr_t) &fval;
		  }
		else
		  val = (caddr_t) va_arg(args,float *);
		put_hw_float(itr,val,index,length);
		break;
	  case SAVE_FKFLT_DEF:
		if (length == 1) {
		  /* fval = (float) va_arg(args,float); */
		  fval = (float) va_arg(args,double);
		  val = (caddr_t) &fval;
		  }
		else
		  val = (caddr_t) va_arg(args,float *);
		savewfr(itr,val,index,length);
		break;
	  }
	
	return((int) 1);
}
void get_hw_int(itr,val,pos,length) /* read int from position pos return as val */
int	*itr;
int	*val,
	pos,
	length;
{
	memcpy(val,itr+pos,length*sizeof(int));
}

void get_hw_float(itr,val,pos,length)
/* save real in line header itr[pos] = val */
float	*itr;
float	*val;
int	pos,
	length;
{
	memcpy(val,itr+pos,length*sizeof(float));
}

void get_hw_short(itr,val,pos,length)
/* val = itr[pos] */
short	*itr;
short	*val;
int	pos,
	length;
{
	memcpy(val,itr+pos,length*sizeof(short));
}

void get_hw_char(itr,val,pos,length)
char	*itr;
char	*val;
int	pos,
	length;
{
	strncpy(val, (char *)(itr+pos), length);
}

void saverfr(itr,val,pos,length)		/* fake reals */
char	*itr;
float	*val;
int	pos,
	length;
{
	int i;
	for (i=0;i<length;i++) {
	  C_GETFP2(itr,SAVE_FAKE_REAL_DEF,pos+i,length,(val+i),NULL);
	  }
}

void put_hw_int(itr,val,pos,length)
/* save integer in line header itr[pos] = val */
int	*itr;
int	*val,
	pos,
	length;
{
	memcpy(itr+pos,val,length*sizeof(int));
}

void put_hw_float(itr,val,pos,length)
/* save real in line header itr[pos] = val */
float	*itr;
float	*val;
int	pos,
	length;
{
	memcpy(itr+pos,val,length*sizeof(float));
}

void put_hw_short(itr,val,pos,length)
/* save short integer in line header itr[pos] = (short)val */
short	*itr;
short	*val;
int 	pos,
	length;
{
	memcpy(itr+pos,val,length*sizeof(short));
}

void put_hw_char(itr,val,pos,length)
char	*itr;
char	*val;
int	pos;
int	length;
{
#ifndef CRAYSYSTEM
	strncpy((char *)(itr+pos), val, length);
#else
	strncpy((char *)(itr+pos), (char *)*(int *)val, length);
#endif
}
void savewfr(itr,val,pos,length)
/* save real in line header itr[pos] = val */
short 	*itr;
float	*val;
int	pos,
	length;
{
	int i;
/*
 * - added this 11/22/00 to do the right thing on the Linux box - j.m.wade
 */
/*
#if ( BYTE_ORDER == LITTLE_ENDIAN )
	HTOCS(val,1);
	HTOCS(((char *)val+2),1);
	HTOCL(val,1);
#endif
*/
	for (i=0;i<length;i++) {
/* 	  itr[pos+i] = *(val+i); */
	  C_PUTFP2(itr,SAVE_FAKE_REAL_DEF,pos+i,length,*(val+i),NULL);
	  }
}
