/***********************************************************************
 *                copyright 2001, Amoco Production Company             *
 *                            All Rights Reserved                      *
 *                    an affiliate of BP America Inc.                  *
 ***********************************************************************/
/*
        File    :       whence.c
        Author  :       Dean Kopesky
        Company :       Minnesota Supercomputer Center, Inc.
        Date    :       October 1990
        Changed :       November 1990
                        Added check_name().  -dmk

			March 1993
			Modified check_name to see if a name can be
			looked up via gethostent(). If so, pass the 
			truncated name back to the main. If not, keep
			trying. Also fixed a bug whereby the hostname
			did not have a null character at the end of 
			it because it had been truncated in the utmp
			table. I now malloc a new spot for it if that
			has happened. Also limit string compares to the
			length of the host or name being checked for.
			To simplify things in the future, I also added
			a debug option flagged on the command line with
			"-D".			- joe m. wade

			December 1993
			Modified check_name, adding the routine is_numeric
			to see if an address matched by gethostbyname consists
			solely of numbers and the character '.'. If it does,
			do not pass it out. Due to a Solaris bug, any numeric
			piece of an address gets matched.
						- joe m. wade


        whence prints the host name of the machine from which you logged in.
        This is useful for setting the DISPLAY variable for X.

        Usage details can be found in print_usage() and print_help().
*/


#include        <stdio.h>
#include        <unistd.h>
#include        <sys/types.h>
#include        <ctype.h>
#ifdef UTMPX
#include        <utmpx.h>
#endif
#include        <utmp.h>
#include        <netdb.h>
#include 	<string.h>
static  struct
{
        int     minus_x;
        char *  defname;
        char *  prefix;
        char *  suffix;
	short   debug;
}
        opts;

void    command_line();
char *  from_where();
char *  check_name();
int     is_local();
void    print_usage();
void    print_help();

#ifdef  CRAY
#define ut_name ut_user
extern  struct  utmp *  getutent();
#else
#ifdef  UTMPX
struct  utmpx *  getutxent();
#else
struct  utmp *  getutent();
#endif
#endif

extern  void                    exit();
extern  struct  hostent *       gethostbyname();
extern  int                     gethostname();
extern  char *                  ttyname();


int     main( argc, argv )

        int     argc;
        char *  argv[];

{
        char *  hostname;

        command_line( argc, argv );

        hostname = from_where();

        hostname = check_name( hostname );
        if ( hostname == NULL || strlen( hostname ) == 0 )
                hostname = opts.defname;

        /* set host to "unix" if -x and remote = local */
        if ( opts.minus_x && is_local( hostname ) ) hostname = "unix";

        (void) printf( "%s%s%s\n", opts.prefix, hostname, opts.suffix );

        return 0;
}


char *  from_where()

{
#ifdef  UTMPX
        struct  utmpx *  utmpp;
#else
        struct  utmp *  utmpp;
#endif
        char *          myname;
        char *          mytty;
	char *		new_ut_host;
	FILE *		testfile;

        myname = getlogin( );
        if ( myname == NULL ) return NULL;

        if ( ( mytty = ttyname( fileno( stdin ) ) ) == NULL &&
                        ( mytty = ttyname( fileno( stdout ) ) ) == NULL &&
                        ( mytty = ttyname( fileno( stderr ) ) ) == NULL )
                return NULL;
        mytty += 5;     /* bump past "/dev/" */

	if (opts.debug !=0 ) {
	  fprintf(stderr,"%s logged in on %s; searching utmp table:\n\n",
		myname,mytty);
	  fprintf(stderr,"%*s\t%*s\t%*s\t%s\n",
		sizeof(utmpp->ut_line), "terminal",
		sizeof(utmpp->ut_name), "user",
		sizeof(utmpp->ut_host), "host",
		"time");
	  }

#ifdef  UTMPX
        while ( ( utmpp = getutxent() ) != NULL )
#else
        while ( ( utmpp = getutent() ) != NULL )
#endif
        {
/*
                if ( strcmp( myname, utmpp->ut_name ) == 0 &&
                                strcmp( mytty, utmpp->ut_line ) == 0 )
*/
	if ((opts.debug != 0) && (strlen(utmpp->ut_name) != 0))
	  fprintf(stderr,"%*.*s\t%*.*s\t%*.*s\t%d\n",
		sizeof(utmpp->ut_line), sizeof(utmpp->ut_line), utmpp->ut_line,
		sizeof(utmpp->ut_name), sizeof(utmpp->ut_name), utmpp->ut_name,
		sizeof(utmpp->ut_host), sizeof(utmpp->ut_host), utmpp->ut_host,
#ifdef UTMPX
		utmpp->ut_tv
#else
		utmpp->ut_time
#endif
		);

                if ( strncmp( myname, utmpp->ut_name, strlen(myname) ) == 0 &&
                     strncmp( mytty, utmpp->ut_line, strlen(mytty) ) == 0 ) {

			if (sizeof(utmpp->ut_host) <= strlen(utmpp->ut_host)) {
			 new_ut_host =
				(char *) malloc(sizeof(utmpp->ut_host) + 1);
			 strncpy(new_ut_host, utmpp->ut_host,
				sizeof(utmpp->ut_host));
			 new_ut_host[sizeof(utmpp->ut_host)] = '\0';
			 return new_ut_host;
			 }
			else
                         return utmpp->ut_host;
			}
        }

        return NULL;
}


char *  check_name( hostname )

        char *  hostname;

{
        char *  s;
	char *	short_name;
	char temp[10];
	struct hostent * host_entry;
	int is_numeric();
	int i=0;

        if ( hostname == NULL ) return NULL;

        /* clobber ":0.0" */
        if ( strncmp( hostname + strlen( hostname ) - 4, ":0.0", 4 ) == 0 )
                hostname[ strlen( hostname ) - 4 ] = '\0';

        /* look for garbage */
        s = hostname;
        while ( *s != '\0' )
        {
                if ( !isalnum( *s ) && *s != '.' && *s != '-' ) {
                        return NULL;
			}
		else if ( *s == '.' ) {
			short_name = (char *)
				malloc (((s-hostname) + 1) * sizeof(char));
			strncpy(short_name,hostname,(s-hostname));
			short_name[(s-hostname)] = '\0';
	 		if ((is_numeric(short_name) == 0) &&
			 ((host_entry = gethostbyname(short_name)) != NULL )) {
				if (opts.debug != 0) {
fprintf(stderr,"gethostbyname match:\n");
fprintf(stderr,"h_name = %s, aliases = ",host_entry->h_name);
while ( host_entry->h_aliases[i] != NULL ) {
  fprintf(stderr,"%s",host_entry->h_aliases[i++]);
  }
fprintf(stderr,"\n");	
				  fprintf(stderr,
				    "truncated hostname to %s\n",short_name);
				  }
				  return short_name;
				}
			}
                s += 1;
        }

	return hostname;
	
}


int     is_local( hostname )

        char *  hostname;

{
        char    localhost[ 32 ];
        char ** alias;
        struct  hostent *       hentp;

        (void) gethostname( localhost, 32 );
        hentp = gethostbyname( localhost );

        if ( strcmp( hostname, hentp->h_name ) == 0 ) return 1;

        alias = hentp->h_aliases;
        while ( *alias != NULL )
        {
                if ( strcmp( hostname, *alias ) == 0 ) return 1;
                alias += 1;
        }

        return 0;
}


void    command_line( argc, argv )

        int     argc;
        char *  argv[];

{
        int     err = 0;
        extern  int     optind;
        extern  char    *optarg;

        static  char *  empty     = "";
        static  char *  x_defname = "unix";
        static  char *  x_prefix  = "";
        static  char *  x_suffix  = ":0.0";

        opts.minus_x = 0;
        opts.defname = empty;
        opts.prefix  = empty;
        opts.suffix  = empty;
        opts.debug   = 0;

        while ( 1 ) switch ( getopt( argc, argv, "HDxd:p:s:" ) )
        {
        case 'H':
                print_help();
                exit( 0 );

        case 'x':
                opts.minus_x = 1;
                opts.defname = x_defname;
                opts.prefix  = x_prefix;
                opts.suffix  = x_suffix;
                break;

        case 'D':
                opts.debug = 1;
                break;

        case 'd':
                opts.defname = optarg;
                break;

        case 'p':
                opts.prefix = optarg;
                break;

        case 's':
                opts.suffix = optarg;
                break;

        case '?':
                err++;
                break;

        case EOF:
                if ( argc - optind != 0 ) err++;
                if ( err )
                {
                        print_usage();
                        exit( 1 );
                }
                return;
        }
}

void    print_usage()

{
        (void) fprintf( stderr, "\
usage:  whence  -H (for help)\n\
        whence  [ -x ] [ -d default ] [ -p prefix ] [ -s suffix ]\n\
" );
}


void    print_help()

{
        (void) fprintf( stderr, "\n" );
        print_usage();
        (void) fprintf( stderr, "\
\n\
        -H              Print this help listing.\n\
        -D              Turn on debugging output.\n\
        -x              Equivalent to \"-d unix -p '' -s :0.0\".\n\
        -d default      Default name if no host name can be determined.\n\
        -p prefix       Prefix to prepend to the host name.\n\
        -s suffix       Suffix to append to the host name.\n\
\n\
whence prints the host name of the machine from which you logged in.  This\n\
is useful for setting the DISPLAY variable for the X Window System (the -x\n\
option is particularly convenient for this).\n\
\n\
" );
}


#if !defined( CRAY ) && !defined( SOLARIS )
struct  utmp *  getutent()

{
        static  struct  utmp    utent;
        static  int             isopen = 0;
        static  FILE *          up;

        if ( ! isopen )
        {
                up = fopen( "/etc/utmp", "r" );
                if ( up == NULL ) return NULL;
                isopen = 1;
        }

        if ( fread( (char *) &utent, sizeof(utent), 1, up ) != 1 )
                return NULL;

/*
	if ((opts.debug != 0) && (strlen(utent.ut_name) != 0))
	  fprintf(stderr,"%*.*s\t%*.*s\t%*.*s\t%d\n",
		sizeof(utent.ut_line), sizeof(utent.ut_line), utent.ut_line,
		sizeof(utent.ut_name), sizeof(utent.ut_name), utent.ut_name,
		sizeof(utent.ut_host), sizeof(utent.ut_host), utent.ut_host,
		utent.ut_time);
*/
        return &utent;
}
#endif

int is_numeric(text)
char *text;
{
	int i;
	for (i = 0; i < strlen(text); i++) 
	  if (!isdigit(text[i]) && text[i] != '.') return ( 0 );
	return ( 1 );
}
