/*
 *	Copyright (c) 1990 by Columbia University.
 */

#include <stdio.h>
#include <errno.h>
#include <time.h>
#include <sys/types.h>
#include <sys/stat.h>

#include "ikp_defines.h"
#include "ikp_externs.h"
#include "globalE.h"

/*
   Due to the large amount of file descriptor shuffling which
   occurs just prior to a process being forked, it is necessary
   to 'lock' all the connector descriptors of a local process and
   any other local processes to which it is connected during
   this period. This is to ensure that a connector descriptor
   is not inadvertently used as a temporary descriptor in pst_openpipe()
   or friends and then close()d after the real connector descriptor has
   been dup2()ed immediately after the fork().

   This is done by maintaining a list of all such connectors (cntlocks).
   When a new connector is added to the list, its corresponding file
   descriptor is locked by dup2()ing /dev/null into it, thus
   preventing it from being returned by any open()-related call.
   A descriptor can be multiply locked by making additional entries
   in the list (one for each local process which has a connector that
   references it).

   As each process is forked, its lock structures are deleted. When the
   last lock structure referencing a descriptor is removed, the descriptor
   is unlocked by close()ing it.
*/

boolean cntl_locked(int fd)
{
  Cntlock *cl;
  
  for (cl= cntlocks; cl != (Cntlock *) NULL; cl= cl->cntl_next) {
    if (fd == cl->cntl_cnt->cnt_desc)
      return(TRUE);
  }
  return(FALSE);
}

boolean cntl_open(int fd)
{
  struct stat stbuf;
  
  if (fstat(fd, &stbuf) == 0)
    return(TRUE);
  else
    return(FALSE);
}

int cntl_add(Connector *c)
{
  Cntlock *cl;
  int fd;
  boolean found;
  struct stat stbuf;
  
  fd= c->cnt_desc;
  
  /* connector already locked? */
  for (cl= cntlocks; cl != (Cntlock *) NULL; cl= cl->cntl_next) {
    if (cl->cntl_cnt == c)
      break;
  }
  if (cl != (Cntlock *) NULL)
    return(IKP_SUCCESS);
  
  /* descriptor already present elsewhere in list? */
  found= FALSE;
  for (cl= cntlocks; cl != (Cntlock *) NULL; cl= cl->cntl_next) {
    if (fd == cl->cntl_cnt->cnt_desc) {
      found= TRUE;
      break;
    }
  }
  
  /* descriptor not in lock list */
  if (!found) {
    
    /* descriptor not in use */
    if (fstat(fd, &stbuf) < 0) {
      if (errno != EBADF)
	return(IKP_FSTAT);
      
      /* lock descriptor */
      if (dup2(dev_null, fd) < 0)
	return(IKP_DUP2);
      if (verbose >= 5)
	fprintf(stderr, "P lock %d\n", fd);
    }
    
    /* descriptor already in use */
    else {
      c->cnt_flags&= ~CNT_LOCKED;
      return(IKP_SUCCESS);
    }
  }
  
  /* insert new lock structure into lock list */
  if ((cl= (Cntlock *) calloc(1, sizeof(Cntlock))) == (Cntlock *) NULL)
    ikp_err("Memory allocation error.", FATAL);
  cl->cntl_cnt= c;
  cl->cntl_next= cntlocks;
  cntlocks= cl;
  
  c->cnt_flags|= CNT_LOCKED;
  return(IKP_SUCCESS);
}

int cntl_delete(Connector *c)
{
  int fd;
  Cntlock *cl, *dl;
  
  fd= c->cnt_desc;
  
  for (cl= cntlocks; cl != (Cntlock *) NULL; cl= cl->cntl_next) {
    if (cl->cntl_cnt == c)
      break;
  }
  if (cl == (Cntlock *) NULL)
    return(IKP_FAILURE);
  
  /* remove from list */
  if (cl == cntlocks)
    cntlocks= cl->cntl_next;
  else {
    for (dl= cntlocks; dl->cntl_next != cl; dl= dl->cntl_next);
    dl->cntl_next= cl->cntl_next;
  }
  free((char *) cl);
  
  /* last lock on this fd? */
  for (cl= cntlocks; cl != (Cntlock *) NULL; cl= cl->cntl_next) {
    if (cl->cntl_cnt->cnt_desc == fd)
      break;
  }
  if (cl == (Cntlock *) NULL) {
    if (close(fd) < 0)
      return(IKP_CLOSE);
    if (verbose >= 5)
      fprintf(stderr, "P ulck %d\n", fd);
  }
  c->cnt_flags&= ~CNT_LOCKED;
  return(IKP_SUCCESS);
}

int cntl_lockprc(Process *p)
{
  Process *q;
  Connector *c, *d;
  int ret_val;
  
  for (c= p->prc_input; c != (Connector *) NULL; c= c->cnt_next) {
    if (p->prc_flags & PRC_REMOTE)
      c->cnt_flags&= ~CNT_LOCKED;
    else {
      if ((ret_val= cntl_add(c)) != IKP_SUCCESS) {
	ikp_err("Cannot lock connector descriptor.", NONFATAL);
	return(ret_val);
      }
    }
    
    if (c->cnt_conn == (Connector *) NULL)
      continue;
    q= (Process *) c->cnt_conn->cnt_process;
    if ((q->prc_flags & PRC_REMOTE) || (q->prc_flags & PRC_FORKED))
      continue;
    
    for (d= q->prc_input; d != (Connector *) NULL; d= d->cnt_next) {
      if ((ret_val= cntl_add(d)) != IKP_SUCCESS) {
	ikp_err("Cannot lock connector descriptor.", NONFATAL);
	return(ret_val);
      }
    }
    for (d= q->prc_output; d != (Connector *) NULL; d= d->cnt_next) {
      if ((ret_val= cntl_add(d)) != IKP_SUCCESS) {
	ikp_err("Cannot lock connector descriptor.", NONFATAL);
	return(ret_val);
      }
    }
    for (d= q->prc_diag; d != (Connector *) NULL; d= d->cnt_next) {
      if ((ret_val= cntl_add(d)) != IKP_SUCCESS) {
	ikp_err("Cannot lock connector descriptor.", NONFATAL);
	return(ret_val);
      }
    }
  }
  for (c= p->prc_output; c != (Connector *) NULL; c= c->cnt_next) {
    if (p->prc_flags & PRC_REMOTE)
      c->cnt_flags&= ~CNT_LOCKED;
    else {
      if ((ret_val= cntl_add(c)) != IKP_SUCCESS) {
	ikp_err("Cannot lock connector descriptor.", NONFATAL);
	return(ret_val);
      }
    }
    
    if (c->cnt_conn == (Connector *) NULL)
      continue;
    q= (Process *) c->cnt_conn->cnt_process;
    if ((q->prc_flags & PRC_REMOTE) || (q->prc_flags & PRC_FORKED))
      continue;
    
    for (d= q->prc_input; d != (Connector *) NULL; d= d->cnt_next) {
      if ((ret_val= cntl_add(d)) != IKP_SUCCESS) {
	ikp_err("Cannot lock connector descriptor.", NONFATAL);
	return(ret_val);
      }
    }
    for (d= q->prc_output; d != (Connector *) NULL; d= d->cnt_next) {
      if ((ret_val= cntl_add(d)) != IKP_SUCCESS) {
	ikp_err("Cannot lock connector descriptor.", NONFATAL);
	return(ret_val);
      }
    }
    for (d= q->prc_diag; d != (Connector *) NULL; d= d->cnt_next) {
      if ((ret_val= cntl_add(d)) != IKP_SUCCESS) {
	ikp_err("Cannot lock connector descriptor.", NONFATAL);
	return(ret_val);
      }
    }
  }
  for (c= p->prc_diag; c != (Connector *) NULL; c= c->cnt_next) {
    if (p->prc_flags & PRC_REMOTE)
      c->cnt_flags&= ~CNT_LOCKED;
    else {
      if ((ret_val= cntl_add(c)) != IKP_SUCCESS) {
	ikp_err("Cannot lock connector descriptor.", NONFATAL);
	return(ret_val);
      }
    }
    
    if (c->cnt_conn == (Connector *) NULL)
      continue;
    q= (Process *) c->cnt_conn->cnt_process;
    if ((q->prc_flags & PRC_REMOTE) || (q->prc_flags & PRC_FORKED))
      continue;
    
    for (d= q->prc_input; d != (Connector *) NULL; d= d->cnt_next) {
      if ((ret_val= cntl_add(d)) != IKP_SUCCESS) {
	ikp_err("Cannot lock connector descriptor.", NONFATAL);
	return(ret_val);
      }
    }
    for (d= q->prc_output; d != (Connector *) NULL; d= d->cnt_next) {
      if ((ret_val= cntl_add(d)) != IKP_SUCCESS) {
	ikp_err("Cannot lock connector descriptor.", NONFATAL);
	return(ret_val);
      }
    }
    for (d= q->prc_diag; d != (Connector *) NULL; d= d->cnt_next) {
      if ((ret_val= cntl_add(d)) != IKP_SUCCESS) {
	ikp_err("Cannot lock connector descriptor.", NONFATAL);
	return(ret_val);
      }
    }
  }
  
  return(IKP_SUCCESS);
}

int cntl_unlockprc(Process *p)
{
  int ret_val;
  Connector *c;
  
  if (p->prc_flags & PRC_REMOTE)
    return(IKP_SUCCESS);
  
  for (c= p->prc_input; c != (Connector *) NULL; c= c->cnt_next) {
    if (!(c->cnt_flags & CNT_LOCKED))
      continue;
    if ((ret_val= cntl_delete(c)) != IKP_SUCCESS) {
      ikp_err("Cannot unlock connector descriptor.", NONFATAL);
      return(ret_val);
    }
  }
  for (c= p->prc_output; c != (Connector *) NULL; c= c->cnt_next) {
    if (!(c->cnt_flags & CNT_LOCKED))
      continue;
    if ((ret_val= cntl_delete(c)) != IKP_SUCCESS) {
      ikp_err("Cannot unlock connector descriptor.", NONFATAL);
      return(ret_val);
    }
  }
  for (c= p->prc_diag; c != (Connector *) NULL; c= c->cnt_next) {
    if (!(c->cnt_flags & CNT_LOCKED))
      continue;
    if ((ret_val= cntl_delete(c)) != IKP_SUCCESS) {
      ikp_err("Cannot unlock connector descriptor.", NONFATAL);
      return(ret_val);
    }
  }
  return(IKP_SUCCESS);
}

int cntl_unlockall()
{
  int ret_val;
  
  ret_val= IKP_SUCCESS;
  while (cntlocks != (Cntlock *) NULL) {
    if (cntl_delete(cntlocks->cntl_cnt) != IKP_SUCCESS) {
      ikp_err("Cannot unlock connector descriptor.", NONFATAL);
      ret_val= IKP_FAILURE;
    }
  }
  return(ret_val);
}

void cntl_querylock(Process *p,int *net,int *tmp)
{
  Connector *c, *d;
  Process *q;
  int i, max_fds;
  fd_set tmp_fds, net_fds;
  boolean cntl_open();
  
  *net= *tmp= 0;
  FD_ZERO(&tmp_fds);
  FD_ZERO(&net_fds);
  
  for (c= p->prc_input; c != (Connector *) NULL; c= c->cnt_next) {
    if (!(p->prc_flags & PRC_REMOTE)) {
      if (!cntl_open(c->cnt_desc))
	FD_SET(c->cnt_desc, &tmp_fds);
    }
    if (c->cnt_conn == (Connector *) NULL)
      continue;
    q= (Process *) c->cnt_conn->cnt_process;
    if ((q->prc_flags & PRC_REMOTE) || (q->prc_flags & PRC_FORKED))
      continue;
    
    for (d= q->prc_input; d != (Connector *) NULL; d= d->cnt_next) {
      if (!cntl_open(d->cnt_desc)) {
	FD_SET(d->cnt_desc, &tmp_fds);
	FD_SET(d->cnt_desc, &net_fds);
      }
    }
    for (d= q->prc_output; d != (Connector *) NULL; d= d->cnt_next) {
      if (!cntl_open(d->cnt_desc)) {
	FD_SET(d->cnt_desc, &tmp_fds);
	FD_SET(d->cnt_desc, &net_fds);
      }
    }
    for (d= q->prc_diag; d != (Connector *) NULL; d= d->cnt_next) {
      if (!cntl_open(d->cnt_desc)) {
	FD_SET(d->cnt_desc, &tmp_fds);
	FD_SET(d->cnt_desc, &net_fds);
      }
    }
  }
  for (c= p->prc_output; c != (Connector *) NULL; c= c->cnt_next) {
    if (!(p->prc_flags & PRC_REMOTE)) {
      if (!cntl_open(c->cnt_desc))
	FD_SET(c->cnt_desc, &tmp_fds);
    }
    if (c->cnt_conn == (Connector *) NULL)
      continue;
    q= (Process *) c->cnt_conn->cnt_process;
    if ((q->prc_flags & PRC_REMOTE) || (q->prc_flags & PRC_FORKED))
      continue;
    
    for (d= q->prc_input; d != (Connector *) NULL; d= d->cnt_next) {
      if (!cntl_open(d->cnt_desc)) {
	FD_SET(d->cnt_desc, &tmp_fds);
	FD_SET(d->cnt_desc, &net_fds);
      }
    }
    for (d= q->prc_output; d != (Connector *) NULL; d= d->cnt_next) {
      if (!cntl_open(d->cnt_desc)) {
	FD_SET(d->cnt_desc, &tmp_fds);
	FD_SET(d->cnt_desc, &net_fds);
      }
    }
    for (d= q->prc_diag; d != (Connector *) NULL; d= d->cnt_next) {
      if (!cntl_open(d->cnt_desc)) {
	FD_SET(d->cnt_desc, &tmp_fds);
	FD_SET(d->cnt_desc, &net_fds);
      }
    }
  }
  for (c= p->prc_diag; c != (Connector *) NULL; c= c->cnt_next) {
    if (!(p->prc_flags & PRC_REMOTE)) {
      if (!cntl_open(c->cnt_desc))
	FD_SET(c->cnt_desc, &tmp_fds);
    }
    if (c->cnt_conn == (Connector *) NULL)
      continue;
    q= (Process *) c->cnt_conn->cnt_process;
    if ((q->prc_flags & PRC_REMOTE) || (q->prc_flags & PRC_FORKED))
      continue;
    
    for (d= q->prc_input; d != (Connector *) NULL; d= d->cnt_next) {
      if (!cntl_open(d->cnt_desc)) {
	FD_SET(d->cnt_desc, &tmp_fds);
	FD_SET(d->cnt_desc, &net_fds);
      }
    }
    for (d= q->prc_output; d != (Connector *) NULL; d= d->cnt_next) {
      if (!cntl_open(d->cnt_desc)) {
	FD_SET(d->cnt_desc, &tmp_fds);
	FD_SET(d->cnt_desc, &net_fds);
      }
    }
    for (d= q->prc_diag; d != (Connector *) NULL; d= d->cnt_next) {
      if (!cntl_open(d->cnt_desc)) {
	FD_SET(d->cnt_desc, &tmp_fds);
	FD_SET(d->cnt_desc, &net_fds);
      }
    }
  }
  
  max_fds= getdtablesize();
  
  for (i= 0; i < max_fds; i++) {
    if (FD_ISSET(i, &tmp_fds))
      (*tmp)++;
    if (FD_ISSET(i, &net_fds))
      (*net)++;
  }
  
  return;
}
