/*
 *	Copyright (c) 1990 by Columbia University.
 */

#include <stdio.h>
#include <sys/file.h>

#include "ikp_defines.h"
#include "ikp_externs.h"
#include "globalE.h"

extern void prc_erase(Process *);
extern void prc_draw(Process *);
extern void seg_drawpath(Connector *);

void cnt_drawgrnd(Process *,Connector *);
void cnt_drawgrnd_erase(Process *,Connector *);
void cnt_drawfile();
void cnt_drawnull();
void cnt_drawfile_grnd(Process *,Connector *,int );

void cnt_link_okCB(Widget,XtPointer,XtPointer);

Connector *c1_lnk,*c2_lnk;

Connector *cnt_create(Process *p,
		      int iodir,int class,int fd)
{
  Connector *new, *c;
  
  if ((fd < 0) || (fd > getdtablesize()-1)) {
    ikp_err("Connector descriptor out of bounds.", NONFATAL);
    return((Connector *) NULL);
  }
  
  if ((new= (Connector *) calloc(1, sizeof(Connector))) == (Connector *) NULL)
    ikp_err("Memory allocation error.", FATAL);
  
  if (p == (Process *) NULL) {
    ikp_err("Connector not bound to process.", NONFATAL);
    free(new);
    return((Connector *) NULL);
  }
  new->cnt_process= (caddr_t) p;
  switch (iodir) {
  case CNT_INPUT:
    for (c= p->prc_input; c != (Connector *) NULL; c= c->cnt_next) {
      if (fd == c->cnt_desc) {
	ikp_err("Duplicate input connector descriptor.", NONFATAL);
	free(new);
	return((Connector *) NULL);
      }
    }
    for (c= p->prc_output; c != (Connector *) NULL; c= c->cnt_next) {
      if (fd == c->cnt_desc) {
	ikp_err("Duplicate output connector descriptor.", NONFATAL);
	free(new);
	return((Connector *) NULL);
      }
    }
    for (c= p->prc_diag; c != (Connector *) NULL; c= c->cnt_next) {
      if (fd == c->cnt_desc) {
	ikp_err("Duplicate diagnostic connector descriptor.", NONFATAL);
	free(new);
	return((Connector *) NULL);
      }
    }
    new->cnt_next= p->prc_input;
    p->prc_input= new;
    break;
  case CNT_OUTPUT:
    for (c= p->prc_input; c != (Connector *) NULL; c= c->cnt_next) {
      if (fd == c->cnt_desc) {
	ikp_err("Duplicate input connector descriptor.", NONFATAL);
	free(new);
	return((Connector *) NULL);
      }
    }
    for (c= p->prc_output; c != (Connector *) NULL; c= c->cnt_next) {
      if (fd == c->cnt_desc) {
	ikp_err("Duplicate output connector descriptor.", NONFATAL);
	free(new);
	return((Connector *) NULL);
      }
    }
    for (c= p->prc_diag; c != (Connector *) NULL; c= c->cnt_next) {
      if (fd == c->cnt_desc) {
	ikp_err("Duplicate diagnostic connector descriptor.", NONFATAL);
	free(new);
	return((Connector *) NULL);
      }
    }
    new->cnt_next= p->prc_output;
    p->prc_output= new;
    break;
  case CNT_DIAG:
    for (c= p->prc_input; c != (Connector *) NULL; c= c->cnt_next) {
      if (fd == c->cnt_desc) {
	ikp_err("Duplicate input connector descriptor.", NONFATAL);
	free(new);
	return((Connector *) NULL);
      }
    }
    for (c= p->prc_output; c != (Connector *) NULL; c= c->cnt_next) {
      if (fd == c->cnt_desc) {
	ikp_err("Duplicate output connector descriptor.", NONFATAL);
	free(new);
	return((Connector *) NULL);
      }
    }
    for (c= p->prc_diag; c != (Connector *) NULL; c= c->cnt_next) {
      if (fd == c->cnt_desc) {
	ikp_err("Duplicate diagnostic connector descriptor.", NONFATAL);
	free(new);
				return((Connector *) NULL);
      }
    }
    new->cnt_next= p->prc_diag;
    p->prc_diag= new;
    break;
  default:
    ikp_err("Invalid connector type.", NONFATAL);
    free(new);
    return((Connector *) NULL);
  }
  new->cnt_iodir= iodir;
  new->cnt_class= class;
  new->cnt_desc= fd;
  new->cnt_pipe= (Pipeslot *) NULL;
  new->cnt_filename= (char *) NULL;
  new->cnt_seglist= (Segment *) NULL;
  new->cnt_conn= (Connector *) NULL;
  new->cnt_flags= 0;
  return(new);
}

cnt_link(Connector *c1,Connector *c2)
{
  int root_x,root_y;
  
  if ((c1 == (Connector *) NULL) || (c2 == (Connector *) NULL)) {
    ikp_err("Null connector.", NONFATAL);
    return(IKP_FAILURE);
  }
  if (c1->cnt_flags & CNT_FILE) {
    ikp_err("Connector already connected.", NONFATAL);
    return(IKP_FAILURE);
  }
  if ((c1->cnt_conn != (Connector *) NULL) || (c2->cnt_conn != (Connector *) NULL)) {
    ikp_err("Connector already connected.", NONFATAL);
    return(IKP_FAILURE);
  }
  switch (c1->cnt_iodir) {
  case CNT_INPUT:
    switch (c2->cnt_iodir) {
    case CNT_INPUT:
      ikp_err("Connector flow mismatch.", NONFATAL);
      return(IKP_FAILURE);
    case CNT_OUTPUT:
    case CNT_DIAG:
      break;
    }
    break;
  case CNT_OUTPUT:
    switch (c2->cnt_iodir) {
    case CNT_INPUT:
      break;
    case CNT_OUTPUT:
    case CNT_DIAG:
      ikp_err("Connector flow mismatch.", NONFATAL);
      return(IKP_FAILURE);
    }
    break;
  case CNT_DIAG:
    switch (c2->cnt_iodir) {
    case CNT_INPUT:
      break;
    case CNT_OUTPUT:
    case CNT_DIAG:
      ikp_err("Connector flow mismatch.", NONFATAL);
      return(IKP_FAILURE);
    }
    break;
  }

  c1_lnk=c1;
  c2_lnk=c2;

  if ((c1->cnt_class != CNT_ALL) && (c2->cnt_class != CNT_ALL) &&
      (c1->cnt_class != c2->cnt_class)) {
    locate_cursor(&root_x,&root_y);
    XBell(display,100);
    XtUnmanageChild(XmMessageBoxGetChild(confirm,XmDIALOG_HELP_BUTTON));
    XtVaSetValues(confirm,
		  XmNokLabelString,XMstr(" Yes "),
		  XmNcancelLabelString,XMstr("No"),
		  XmNdefaultButtonType,XmDIALOG_OK_BUTTON,
		  XmNdefaultPosition,False,
		  XmNdialogTitle,XMstr("Link Connector"),
		  XmNmessageString,
		  XMstr("Warning -- connector data class mismatch.\n Do you want to still want this link?"),
		  XmNokCallback,CBl(cnt_link_okCB,0),
		  XmNx,root_x,
		  XmNy,root_y,
		  NULL);
    
    XtManageChild(confirm);
  } else {
    cnt_link_okCB(NULL,0,NULL);
  }
}

void cnt_link_okCB(Widget w,
		   XtPointer client_data,
		   XtPointer call_data)
{
  Process *p1,*p2,*p3;
  Net *n1, *n2;

  c1_lnk->cnt_conn= c2_lnk;
  c2_lnk->cnt_conn= c1_lnk;
  
  /* merge Nets if necessary */
  p1= (Process *) c1_lnk->cnt_process;
  p2= (Process *) c2_lnk->cnt_process;
  n1= (Net *) p1->prc_net;
  n2= (Net *) p2->prc_net;
  if (n1 != n2) {
    for (p3= p1; p3->prc_next != (Process *) NULL; p3= p3->prc_next);
    p3->prc_next= n2->net_prclist;
    n2->net_prclist= (Process *) NULL;
    for (p3= p3->prc_next; p3 != (Process *) NULL; p3= p3->prc_next)
      p3->prc_net= (caddr_t) n1;
		net_unlink(n2);
    net_free(n2);
  }
  curr_net= n1;
  
  seg_bldpath(c1_lnk, c2_lnk);
/* ???????????????
  seg_drawpath(c1_lnk, c2_lnk);
*/
  seg_drawpath(c1_lnk);
  seg_drawpath(c2_lnk);
  return;
}

cnt_ground(Connector *c)
{
  Process *p;
  Connector *d;
  int val, mode, newfd;
  char *filenm, *descptr;
  
  p= (Process *) c->cnt_process;
  curr_net= (Net *) p->prc_net;
  if (!(c->cnt_flags & CNT_FILE)) {
    c->cnt_flags|= CNT_FILE;
    cnt_drawgrnd(p, c);
  }
  
  if (net_op == CNT_GRNDNULL) {
    cnt_drawgrnd(p, c);
    if (c->cnt_filename != (char *) NULL)
      free(c->cnt_filename);
    if ((c->cnt_filename= calloc((unsigned) (strlen("/dev/null")+1), sizeof(char))) == (char *) NULL)
      ikp_err("Memory allocation error.", FATAL);
    (void) strcpy(c->cnt_filename, "/dev/null");
    cnt_drawgrnd(p, c);
    return(IKP_SUCCESS);
  }
  
  cnt_adjgrnd(c);

  return(IKP_SUCCESS);
}

cnt_unlink(Connector *c)
{
  Process *p;
  Segment *s, *t;
  void seg_erasepath();
  
  if (c == (Connector *) NULL)
    return(IKP_FAILURE);
  if (!(c->cnt_flags & CNT_FILE) && (c->cnt_conn == (Connector *) NULL))
    return(IKP_FAILURE);
  
  p= (Process *) c->cnt_process;
  if ((c->cnt_flags & CNT_FILE) && (c->cnt_flags & CNT_GRNDDRAWN))
    cnt_drawgrnd_erase(p, c);
  else
    seg_erasepath(c);
  for (s= c->cnt_seglist; s != (Segment *) NULL; s= t) {
    t= s->seg_next;
    free((char *) s);
  }
  if (c->cnt_conn != (Connector *) NULL) {
    c->cnt_conn->cnt_conn= (Connector *) NULL;
    c->cnt_conn->cnt_seglist= (Segment *) NULL;
  }
  c->cnt_conn= (Connector *) NULL;
  c->cnt_seglist= (Segment *) NULL;
  if (c->cnt_filename != (char *) NULL) {
    free(c->cnt_filename);
    c->cnt_filename= (char *) NULL;
  }
  c->cnt_flags&= ~CNT_FILE;
  return(IKP_SUCCESS);
}

void cnt_draw(Process *p,
	      Connector *c)
{
  int x, y, g;
  void cnt_drawarrow(), cnt_drawdesc();
  
  x= p->prc_x+c->cnt_x;
  y= p->prc_y+c->cnt_y;
  g= CNT_WIDTH/2;

  switch (c->cnt_orient) {
  case UP:
    draw_fill_rectangle(net_cgc, x-g, y-CNT_LIP, (2*g)+1, CNT_LIP+3);
    draw_line(net_gc, x-g, y-CNT_LIP, x-g, y+1);
    draw_line(net_gc, x+g, y-CNT_LIP, x+g, y+1);
    
    switch (c->cnt_iodir) {
    case CNT_INPUT:
      cnt_drawarrow(x, y-17, DOWN, c->cnt_iodir);
      break;
    case CNT_OUTPUT:
    case CNT_DIAG:
      cnt_drawarrow(x, y-CNT_LIP-2, UP, c->cnt_iodir);
      break;
    default:
      break;
		}
    break;
  case DOWN:
    draw_fill_rectangle(net_cgc, x-g, y-1, (2*g)+1, CNT_LIP+2);
    draw_line(net_gc, x-g, y-1, x-g, y+CNT_LIP);
    draw_line(net_gc, x+g, y-1, x+g, y+CNT_LIP);

    switch (c->cnt_iodir) {
    case CNT_INPUT:
      cnt_drawarrow(x, y+17, UP, c->cnt_iodir);
      break;
    case CNT_OUTPUT:
    case CNT_DIAG:
      cnt_drawarrow(x, y+CNT_LIP+2, DOWN, c->cnt_iodir);
      break;
    default:
      break;
    }
    break;
  case LEFT:
    draw_fill_rectangle(net_cgc, x-CNT_LIP, y-g, CNT_LIP+3, (2*g)+1);
    draw_line(net_gc, x-CNT_LIP, y-g, x+1, y-g);
    draw_line(net_gc, x-CNT_LIP, y+g, x+1, y+g);
    
    switch (c->cnt_iodir) {
    case CNT_INPUT:
      cnt_drawarrow(x-17, y, RIGHT, c->cnt_iodir);
      break;
    case CNT_OUTPUT:
    case CNT_DIAG:
      cnt_drawarrow(x-CNT_LIP-2, y, LEFT, c->cnt_iodir);
      break;
    default:
      break;
    }
    break;
  case RIGHT:
    draw_fill_rectangle(net_cgc, x-1, y-g, CNT_LIP+2, (2*g)+1);
    draw_line(net_gc, x-1, y-g, x+CNT_LIP, y-g);
    draw_line(net_gc, x-1, y+g, x+CNT_LIP, y+g);
    
    switch (c->cnt_iodir) {
    case CNT_INPUT:
      cnt_drawarrow(x+17, y, LEFT, c->cnt_iodir);
      break;
    case CNT_OUTPUT:
    case CNT_DIAG:
      cnt_drawarrow(x+CNT_LIP+2, y, RIGHT, c->cnt_iodir);
      break;
    default:
      break;
    }
    break;
  default:
    break;
  }
  
  switch (p->prc_type) {
  case PRC_CUSTOM:
  case PRC_LIBRARY:
    cnt_drawdesc(p, c);
    break;
  default:
    break;
  }

  if (c->cnt_flags & CNT_FILE)
    cnt_drawgrnd(p, c);
  else
    seg_drawpath(c);
  
  return;
}

void cnt_drawdesc(Process *p,
		  Connector *c)
{
  int x, y, tw, asc;
  char label[20];
  
  x= p->prc_x+c->cnt_x;
  y= p->prc_y+c->cnt_y;
  asc= small_font->max_bounds.ascent;

  sprintf(label, "%0d", c->cnt_desc);
  tw= XTextWidth(reg_font, label, strlen(label));
  
  x=(float)x*net_scale;
  y=(float)y*net_scale;

  switch (c->cnt_orient) {
  case UP:
    XDrawString(display,net_cpm,net_sfxgc, 
		x-(tw/2)+1, y+small_fontht+1, label, strlen(label));
    break;
  case DOWN:
    XDrawString(display,net_cpm,net_sfxgc, 
		x-(tw/2)+1, y-3, label, strlen(label));
    break;
  case LEFT:
    XDrawString(display,net_cpm,net_sfxgc, 
		x+(small_charwd/2), y+(asc/2), label, strlen(label));
    break;
  case RIGHT:
    XDrawString(display,net_cpm,net_sfxgc, 
		x-tw-(small_charwd/2), y+(asc/2), label, strlen(label));
    break;
	default:
    break;
  }
  return;
}

void cnt_drawgrnd(Process *p,Connector *c)
{
  
  if (c->cnt_filename != (char *) NULL)
    if (strcmp(c->cnt_filename, "/dev/null"))
      cnt_drawfile(p, c , 0);
    else
      cnt_drawnull(p, c,0);
  
  c->cnt_flags^= CNT_GRNDDRAWN;
  return;
}


void cnt_drawgrnd_erase(Process *p,Connector *c)
{
  
  if (c->cnt_filename != (char *) NULL)
    if (strcmp(c->cnt_filename, "/dev/null"))
      cnt_drawfile(p, c,1);
    else
      cnt_drawnull(p, c,1);
  
  c->cnt_flags^= CNT_GRNDDRAWN;
  return;
}

void cnt_drawfile(Process *p,Connector *c,int erase_flag)
{
  int x, y;

  x= p->prc_x+c->cnt_x;
  y= p->prc_y+c->cnt_y;

  switch (c->cnt_orient) {
  case UP:
    if (erase_flag==0) {
      draw_line(net_gc, x-5, y-CNT_LIP-15, x+5, y-CNT_LIP-15);
      draw_line(net_gc, x-5, y-CNT_LIP-19, x+5, y-CNT_LIP-19);
      draw_line(net_gc, x-5, y-CNT_LIP-18, x-5, y-CNT_LIP-16);
      draw_line(net_gc, x+5, y-CNT_LIP-18, x+5, y-CNT_LIP-16);
      
      draw_line(net_xgc, x-5, y-CNT_LIP-15, x+5, y-CNT_LIP-15);
      draw_line(net_xgc, x-5, y-CNT_LIP-19, x+5, y-CNT_LIP-19);
      draw_line(net_xgc, x-5, y-CNT_LIP-18, x-5, y-CNT_LIP-16);
      draw_line(net_xgc, x+5, y-CNT_LIP-18, x+5, y-CNT_LIP-16);
    }
    
    draw_line(net_xgc, x-5, y-CNT_LIP-15, x+5, y-CNT_LIP-15);
    draw_line(net_xgc, x-5, y-CNT_LIP-19, x+5, y-CNT_LIP-19);
    draw_line(net_xgc, x-5, y-CNT_LIP-18, x-5, y-CNT_LIP-16);
    draw_line(net_xgc, x+5, y-CNT_LIP-18, x+5, y-CNT_LIP-16);
    break;
  case DOWN:
    if (erase_flag==0) {
      draw_line(net_gc, x-5, y+CNT_LIP+15, x+5, y+CNT_LIP+15);
      draw_line(net_gc, x-5, y+CNT_LIP+19, x+5, y+CNT_LIP+19);
      draw_line(net_gc, x-5, y+CNT_LIP+16, x-5, y+CNT_LIP+18);
      draw_line(net_gc, x+5, y+CNT_LIP+16, x+5, y+CNT_LIP+18);
      
      draw_line(net_xgc, x-5, y+CNT_LIP+15, x+5, y+CNT_LIP+15);
      draw_line(net_xgc, x-5, y+CNT_LIP+19, x+5, y+CNT_LIP+19);
      draw_line(net_xgc, x-5, y+CNT_LIP+16, x-5, y+CNT_LIP+18);
      draw_line(net_xgc, x+5, y+CNT_LIP+16, x+5, y+CNT_LIP+18);
    }
    draw_line(net_xgc, x-5, y+CNT_LIP+15, x+5, y+CNT_LIP+15);
    draw_line(net_xgc, x-5, y+CNT_LIP+19, x+5, y+CNT_LIP+19);
    draw_line(net_xgc, x-5, y+CNT_LIP+16, x-5, y+CNT_LIP+18);
    draw_line(net_xgc, x+5, y+CNT_LIP+16, x+5, y+CNT_LIP+18);
    break;
  case LEFT:
    if (erase_flag==0) {
      draw_line(net_gc, x-CNT_LIP-15, y-5, x-CNT_LIP-15, y+5);
      draw_line(net_gc, x-CNT_LIP-19, y-5, x-CNT_LIP-19, y+5);
      draw_line(net_gc, x-CNT_LIP-18, y-5, x-CNT_LIP-16, y-5);
      draw_line(net_gc, x-CNT_LIP-18, y+5, x-CNT_LIP-16, y+5);
      
      draw_line(net_xgc, x-CNT_LIP-15, y-5, x-CNT_LIP-15, y+5);
      draw_line(net_xgc, x-CNT_LIP-19, y-5, x-CNT_LIP-19, y+5);
      draw_line(net_xgc, x-CNT_LIP-18, y-5, x-CNT_LIP-16, y-5);
      draw_line(net_xgc, x-CNT_LIP-18, y+5, x-CNT_LIP-16, y+5);
    }
    draw_line(net_xgc, x-CNT_LIP-15, y-5, x-CNT_LIP-15, y+5);
    draw_line(net_xgc, x-CNT_LIP-19, y-5, x-CNT_LIP-19, y+5);
    draw_line(net_xgc, x-CNT_LIP-18, y-5, x-CNT_LIP-16, y-5);
    draw_line(net_xgc, x-CNT_LIP-18, y+5, x-CNT_LIP-16, y+5);
    break;
  case RIGHT:
    if (erase_flag==0) {
      draw_line(net_gc, x+CNT_LIP+15, y-5, x+CNT_LIP+15, y+5);
      draw_line(net_gc, x+CNT_LIP+19, y-5, x+CNT_LIP+19, y+5);
      draw_line(net_gc, x+CNT_LIP+16, y-5, x+CNT_LIP+18, y-5);
      draw_line(net_gc, x+CNT_LIP+16, y+5, x+CNT_LIP+18, y+5);
      
      draw_line(net_xgc, x+CNT_LIP+15, y-5, x+CNT_LIP+15, y+5);
      draw_line(net_xgc, x+CNT_LIP+19, y-5, x+CNT_LIP+19, y+5);
      draw_line(net_xgc, x+CNT_LIP+16, y-5, x+CNT_LIP+18, y-5);
      draw_line(net_xgc, x+CNT_LIP+16, y+5, x+CNT_LIP+18, y+5);
    }
    draw_line(net_xgc, x+CNT_LIP+15, y-5, x+CNT_LIP+15, y+5);
    draw_line(net_xgc, x+CNT_LIP+19, y-5, x+CNT_LIP+19, y+5);
    draw_line(net_xgc, x+CNT_LIP+16, y-5, x+CNT_LIP+18, y-5);
    draw_line(net_xgc, x+CNT_LIP+16, y+5, x+CNT_LIP+18, y+5);
    break;
  default:
    break;
  }

  return;
}


void cnt_drawnull(Process *p,Connector *c,int erase_flag)
{
  int x, y;
  
  x= p->prc_x+c->cnt_x;
  y= p->prc_y+c->cnt_y;
  
  switch (c->cnt_orient) {
  case UP:
    if (erase_flag==0) {
      draw_line(net_gc, x-5, y-CNT_LIP-15, x+5, y-CNT_LIP-15);
      draw_line(net_gc, x-3, y-CNT_LIP-17, x+3, y-CNT_LIP-17);
      draw_line(net_gc, x-1, y-CNT_LIP-19, x+1, y-CNT_LIP-19);
      
      draw_line(net_xgc, x-5, y-CNT_LIP-15, x+5, y-CNT_LIP-15);
      draw_line(net_xgc, x-3, y-CNT_LIP-17, x+3, y-CNT_LIP-17);
      draw_line(net_xgc, x-1, y-CNT_LIP-19, x+1, y-CNT_LIP-19);
    }
    draw_line(net_xgc, x-5, y-CNT_LIP-15, x+5, y-CNT_LIP-15);
    draw_line(net_xgc, x-3, y-CNT_LIP-17, x+3, y-CNT_LIP-17);
    draw_line(net_xgc, x-1, y-CNT_LIP-19, x+1, y-CNT_LIP-19);
    break;
  case DOWN:
    if (erase_flag==0) {
      draw_line(net_gc, x-5, y+CNT_LIP+15, x+5, y+CNT_LIP+15);
      draw_line(net_gc, x-3, y+CNT_LIP+17, x+3, y+CNT_LIP+17);
      draw_line(net_gc, x-1, y+CNT_LIP+19, x+1, y+CNT_LIP+19);
      
      draw_line(net_xgc, x-5, y+CNT_LIP+15, x+5, y+CNT_LIP+15);
      draw_line(net_xgc, x-3, y+CNT_LIP+17, x+3, y+CNT_LIP+17);
      draw_line(net_xgc, x-1, y+CNT_LIP+19, x+1, y+CNT_LIP+19);
    }
    draw_line(net_xgc, x-5, y+CNT_LIP+15, x+5, y+CNT_LIP+15);
    draw_line(net_xgc, x-3, y+CNT_LIP+17, x+3, y+CNT_LIP+17);
    draw_line(net_xgc, x-1, y+CNT_LIP+19, x+1, y+CNT_LIP+19);
    break;
  case LEFT:
    if (erase_flag==0) {
      draw_line(net_gc, x-CNT_LIP-15, y-5, x-CNT_LIP-15, y+5);
      draw_line(net_gc, x-CNT_LIP-17, y-3, x-CNT_LIP-17, y+3);
      draw_line(net_gc, x-CNT_LIP-19, y-1, x-CNT_LIP-19, y+1);
      
      draw_line(net_xgc, x-CNT_LIP-15, y-5, x-CNT_LIP-15, y+5);
      draw_line(net_xgc, x-CNT_LIP-17, y-3, x-CNT_LIP-17, y+3);
      draw_line(net_xgc, x-CNT_LIP-19, y-1, x-CNT_LIP-19, y+1);
    }
    draw_line(net_xgc, x-CNT_LIP-15, y-5, x-CNT_LIP-15, y+5);
    draw_line(net_xgc, x-CNT_LIP-17, y-3, x-CNT_LIP-17, y+3);
    draw_line(net_xgc, x-CNT_LIP-19, y-1, x-CNT_LIP-19, y+1);
    break;
  case RIGHT:
    if (erase_flag==0) {
      draw_line(net_gc, x+CNT_LIP+15, y-5, x+CNT_LIP+15, y+5);
      draw_line(net_gc, x+CNT_LIP+17, y-3, x+CNT_LIP+17, y+3);
      draw_line(net_gc, x+CNT_LIP+19, y-1, x+CNT_LIP+19, y+1);
      
      draw_line(net_xgc, x+CNT_LIP+15, y-5, x+CNT_LIP+15, y+5);
      draw_line(net_xgc, x+CNT_LIP+17, y-3, x+CNT_LIP+17, y+3);
      draw_line(net_xgc, x+CNT_LIP+19, y-1, x+CNT_LIP+19, y+1);
    }
    draw_line(net_xgc, x+CNT_LIP+15, y-5, x+CNT_LIP+15, y+5);
    draw_line(net_xgc, x+CNT_LIP+17, y-3, x+CNT_LIP+17, y+3);
    draw_line(net_xgc, x+CNT_LIP+19, y-1, x+CNT_LIP+19, y+1);
    break;
  default:
    break;
  }

  return;
}

void cnt_drawarrow(int x,int y,int orient,int type)
{
  switch (orient) {
  case UP:
    draw_fill_rectangle(net_cgc, x-2, y-12, 5, 13);
    if (type == CNT_DIAG) {
      draw_line(net_gc, x, y-12, x, y-10);
      draw_line(net_gc, x, y-7, x, y-5);
      draw_line(net_gc, x, y-2, x, y);
    } else
      draw_line(net_gc, x, y-12, x, y);
    draw_line(net_gc, x-1, y-11, x+1, y-11);
    draw_line(net_gc, x-2, y-10, x+2, y-10);
    break;
  case DOWN:
    draw_fill_rectangle(net_cgc, x-2, y, 5, 13);
    if (type == CNT_DIAG) {
      draw_line(net_gc, x, y, x, y+2);
      draw_line(net_gc, x, y+5, x, y+7);
      draw_line(net_gc, x, y+10, x, y+12);
    } else
      draw_line(net_gc, x, y, x, y+12);
    draw_line(net_gc, x-1, y+11, x+1, y+11);
    draw_line(net_gc, x-2, y+10, x+2, y+10);
    break;
  case LEFT:
    draw_fill_rectangle(net_cgc, x-12, y-2, 13, 5);
    if (type == CNT_DIAG) {
      draw_line(net_gc, x-12, y, x-10, y);
      draw_line(net_gc, x-7, y, x-5, y);
      draw_line(net_gc, x-2, y, x, y);
    } else
      draw_line(net_gc, x-12, y, x, y);
    draw_line(net_gc, x-11, y-1, x-11, y+1);
    draw_line(net_gc, x-10, y-2, x-10, y+2);
    break;
  case RIGHT:
    draw_fill_rectangle(net_cgc, x, y-2, 13, 5);
    if (type == CNT_DIAG) {
      draw_line(net_gc, x, y, x+2, y);
      draw_line(net_gc, x+5, y, x+7, y);
      draw_line(net_gc, x+10, y, x+12, y);
    } else
      draw_line(net_gc, x, y, x+12, y);
    draw_line(net_gc, x+11, y-1, x+11, y+1);
    draw_line(net_gc, x+10, y-2, x+10, y+2);
    break;
  default:
    break;
  }
  return;
}

void cnt_erase(Process *p,
	       Connector *c)
{
  int x, y, g;
  void cnt_drawdesc();
  void seg_erasepath();
  
  if (p == (Process *) NULL)
    return;
  
  x= p->prc_x+c->cnt_x;
  y= p->prc_y+c->cnt_y;
  g= CNT_WIDTH/2;
  
  if (c->cnt_flags & CNT_FILE)
/*    cnt_drawgrnd(p, c); */
    cnt_drawgrnd_erase(p, c);
  else
    seg_erasepath(c);
  
  switch (c->cnt_orient) {
  case UP:
    draw_fill_rectangle(net_cgc, x-g, y-CNT_LIP, (2*g)+2, CNT_LIP+2);
    draw_fill_rectangle(net_cgc, x-2, y-CNT_LIP-14, 7, 14);
    break;
  case DOWN:
    draw_fill_rectangle(net_cgc, x-g, y-1, (2*g)+2, CNT_LIP+4);
    draw_fill_rectangle(net_cgc, x-2, y+CNT_LIP+2, 7, 14);
    break;
  case LEFT:
    draw_fill_rectangle(net_cgc, x-CNT_LIP, y-g, CNT_LIP+2, (2*g)+2);
    draw_fill_rectangle(net_cgc, x-CNT_LIP-14, y-2, 14, 7);
    break;
  case RIGHT:
    draw_fill_rectangle(net_cgc, x-1, y-g, CNT_LIP+3, (2*g)+3);
    draw_fill_rectangle(net_cgc, x+CNT_LIP+2, y-2, 14, 7);
    break;
  default:
    break;
  }
  
  switch (p->prc_type) {
  case PRC_CUSTOM:
  case PRC_LIBRARY:
    cnt_drawdesc(p, c);
    break;
  default:
    break;
  }
  
  return;
}

/*
   Locate a Connector in virtual canvas coordinate space.
   */
Connector *cnt_locate(int x,int y)
{
  Net *n;
  Process *p;
  Connector *c;
  boolean found;

  x=(float)x/net_scale;
  y=(float)y/net_scale;
  
  found= FALSE;
  c= (Connector *) NULL;
  for (n= nets; n != (Net *) NULL; n= n->net_next) {
    for (p= n->net_prclist; p != (Process *) NULL; p= p->prc_next) {
      for (c= p->prc_input; c != (Connector *) NULL; c= c->cnt_next) {
	if ((p->prc_x+c->cnt_x-(CNT_WIDTH/2) <= x) &&
	    (p->prc_x+c->cnt_x+(CNT_WIDTH/2) >= x) &&
	    (p->prc_y+c->cnt_y-(CNT_WIDTH/2) <= y) &&
	    (p->prc_y+c->cnt_y+(CNT_WIDTH/2) >= y)) {
	  found= TRUE;
	  break;
	}
      }
      if (found)
	break;
      for (c= p->prc_output; c != (Connector *) NULL; c= c->cnt_next) {
	if ((p->prc_x+c->cnt_x-(CNT_WIDTH/2) <= x) &&
	    (p->prc_x+c->cnt_x+(CNT_WIDTH/2) >= x) &&
	    (p->prc_y+c->cnt_y-(CNT_WIDTH/2) <= y) &&
	    (p->prc_y+c->cnt_y+(CNT_WIDTH/2) >= y)) {
	  found= TRUE;
	  break;
	}
      }
      if (found)
	break;
      for (c= p->prc_diag; c != (Connector *) NULL; c= c->cnt_next) {
	if ((p->prc_x+c->cnt_x-(CNT_WIDTH/2) <= x) &&
	    (p->prc_x+c->cnt_x+(CNT_WIDTH/2) >= x) &&
	    (p->prc_y+c->cnt_y-(CNT_WIDTH/2) <= y) &&
	    (p->prc_y+c->cnt_y+(CNT_WIDTH/2) >= y)) {
	  found= TRUE;
	  break;
	}
      }
      if (found)
	break;
    }
    if (found)
      break;
  }
  return(c);
}

cnt_adjust(Process *p,Connector *c,int x,int y)
{
  int ax, ay, w, h, dist, orient;
  Connector *d;
  
  if (p == (Process *) NULL)
    return;
  if (c == (Connector *) NULL)
    return;
  
  w= p->prc_w;
  h= p->prc_h;
  
  /* find nearest point (ax, ay) on the Process border */
  
  /* (x, y) to left of or within CNT_WIDTH of left border */
  if (x <= CNT_WIDTH) {
    
    /* too close to upper left corner */
    if (y <= CNT_WIDTH)
      return;
    
    /* too close to lower left corner */
    else if (y >= h-CNT_WIDTH-1)
      return;
    
    else
      orient= LEFT;
  }
  
  /* (x, y) to right of or within CNT_WIDTH of right border */
  else if (x >= w-CNT_WIDTH-1) {
    
    /* too close to upper right corner */
    if (y <= CNT_WIDTH)
      return;
    
    /* too close to lower right corner */
    else if (y >= h-CNT_WIDTH-1)
      return;
    
    else
      orient= RIGHT;
  }
  
  /* (x, y) above top border */
  else if (y <= CNT_WIDTH)
    orient= UP;
  
  /* (x, y) below bottom border */
  else if (y >= h-CNT_WIDTH-1)
    orient= DOWN;
  
  /* (x, y) inside process border */
  else {
    
    /* assume (x, y,) closest to left border */
    dist= x;
    orient= LEFT;
    
    /* closer to right border? */
    if (w-x-1 < dist) {
      dist= w-x-1;
      orient= RIGHT;
    }
    
    /* closer to top border? */
    if (y < dist) {
      dist= y;
      orient= UP;
    }
    
    /* closer to bottom border? */
    if (h-y-1 < dist) {
      dist= h-y-1;
      orient= DOWN;
    }
  }
  
  switch (orient) {
  case LEFT:
    ax= 0;
    ay= y;
    break;
  case RIGHT:
    ax= w-1;
    ay= y;
    break;
  case UP:
    ax= x;
    ay= 0;
    break;
  case DOWN:
    ax= x;
    ay= h-1;
    break;
  default:
    break;
  }
  
  /* make sure that new point (ax, ay) doesn't
     conflict with an existing connector */
  for (d= p->prc_input; d != (Connector *) NULL; d= d->cnt_next) {
    if (d == c)
      continue;
    if (d->cnt_orient != orient)
      continue;
    switch (orient) {
    case UP:
    case DOWN:
      if ((ax >= d->cnt_x-CNT_WIDTH-2) &&
	  (ax <= d->cnt_x+CNT_WIDTH+2))
	return;
      break;
    case LEFT:
    case RIGHT:
      if ((ay >= d->cnt_y-CNT_WIDTH-2) &&
	  (ay <= d->cnt_y+CNT_WIDTH+2))
	return;
      break;
    default:
      break;
    }
  }
  for (d= p->prc_output; d != (Connector *) NULL; d= d->cnt_next) {
    if (d == c)
      continue;
    if (d->cnt_orient != orient)
      continue;
    switch (orient) {
    case UP:
    case DOWN:
      if ((ax >= d->cnt_x-CNT_WIDTH-2) &&
	  (ax <= d->cnt_x+CNT_WIDTH+2))
	return;
      break;
    case LEFT:
		case RIGHT:
      if ((ay >= d->cnt_y-CNT_WIDTH-2) &&
	  (ay <= d->cnt_y+CNT_WIDTH+2))
	return;
      break;
    default:
      break;
    }
  }
  for (d= p->prc_diag; d != (Connector *) NULL; d= d->cnt_next) {
    if (d == c)
      continue;
    if (d->cnt_orient != orient)
      continue;
    switch (orient) {
    case UP:
    case DOWN:
      if ((ax >= d->cnt_x-CNT_WIDTH-2) &&
	  (ax <= d->cnt_x+CNT_WIDTH+2))
	return;
      break;
    case LEFT:
    case RIGHT:
      if ((ay >= d->cnt_y-CNT_WIDTH-2) &&
	  (ay <= d->cnt_y+CNT_WIDTH+2))
	return;
      break;
    default:
      break;
    }
  }
  
  c->cnt_x= ax;
  c->cnt_y= ay;
  c->cnt_orient= orient;
  return;
}

void cnt_highlight(Connector *c,int erase_flag)
{
  Process *p;
  int x, y, g;

  p= (Process *) c->cnt_process;
  x= p->prc_x+c->cnt_x;
  y= p->prc_y+c->cnt_y;
  g= CNT_WIDTH/2;
  
  switch (c->cnt_orient) {
  case UP:
    if (erase_flag==0) {
      draw_line(net_gc, x-g-5, y-CNT_LIP-4, x-g+1, y-CNT_LIP-4);
      draw_point(net_gc, x-g, y-CNT_LIP-5);
      draw_point(net_gc, x-g, y-CNT_LIP-3);
      draw_line(net_gc, x+g-1, y-CNT_LIP-4, x+g+5, y-CNT_LIP-4);
      draw_point(net_gc, x+g, y-CNT_LIP-5);
      draw_point(net_gc, x+g, y-CNT_LIP-3);
      
      draw_line(net_xgc, x-g-5, y-CNT_LIP-4, x-g+1, y-CNT_LIP-4);
      draw_point(net_xgc, x-g, y-CNT_LIP-5);
      draw_point(net_xgc, x-g, y-CNT_LIP-3);
      draw_line(net_xgc, x+g-1, y-CNT_LIP-4, x+g+5, y-CNT_LIP-4);
      draw_point(net_xgc, x+g, y-CNT_LIP-5);
      draw_point(net_xgc, x+g, y-CNT_LIP-3);
    }
    draw_line(net_xgc, x-g-5, y-CNT_LIP-4, x-g+1, y-CNT_LIP-4);
    draw_point(net_xgc, x-g, y-CNT_LIP-5);
    draw_point(net_xgc, x-g, y-CNT_LIP-3);
    draw_line(net_xgc, x+g-1, y-CNT_LIP-4, x+g+5, y-CNT_LIP-4);
    draw_point(net_xgc, x+g, y-CNT_LIP-5);
    draw_point(net_xgc, x+g, y-CNT_LIP-3);
    break;
  case DOWN:
    if (erase_flag==0) {
      draw_line(net_gc, x-g-5, y+CNT_LIP+4, x-g+1, y+CNT_LIP+4);
      draw_point(net_gc, x-g, y+CNT_LIP+3);
      draw_point(net_gc, x-g, y+CNT_LIP+5);
      draw_line(net_gc, x+g-1, y+CNT_LIP+4, x+g+5, y+CNT_LIP+4);
      draw_point(net_gc, x+g, y+CNT_LIP+3);
      draw_point(net_gc, x+g, y+CNT_LIP+5);
      
      draw_line(net_xgc, x-g-5, y+CNT_LIP+4, x-g+1, y+CNT_LIP+4);
      draw_point(net_xgc, x-g, y+CNT_LIP+3);
      draw_point(net_xgc, x-g, y+CNT_LIP+5);
      draw_line(net_xgc, x+g-1, y+CNT_LIP+4, x+g+5, y+CNT_LIP+4);
      draw_point(net_xgc, x+g, y+CNT_LIP+3);
      draw_point(net_xgc, x+g, y+CNT_LIP+5);
    }
    draw_line(net_xgc, x-g-5, y+CNT_LIP+4, x-g+1, y+CNT_LIP+4);
    draw_point(net_xgc, x-g, y+CNT_LIP+3);
    draw_point(net_xgc, x-g, y+CNT_LIP+5);
    draw_line(net_xgc, x+g-1, y+CNT_LIP+4, x+g+5, y+CNT_LIP+4);
    draw_point(net_xgc, x+g, y+CNT_LIP+3);
    draw_point(net_xgc, x+g, y+CNT_LIP+5);
    break;
  case LEFT:
    if (erase_flag==0) {
      draw_line(net_gc, x-CNT_LIP-4, y-g-5, x-CNT_LIP-4, y-g+1);
      draw_point(net_gc, x-CNT_LIP-5, y-g);
      draw_point(net_gc, x-CNT_LIP-3, y-g);
      draw_line(net_gc, x-CNT_LIP-4, y+g-1, x-CNT_LIP-4, y+g+5);
      draw_point(net_gc, x-CNT_LIP-5, y+g);
      draw_point(net_gc, x-CNT_LIP-3, y+g);
      
      draw_line(net_xgc, x-CNT_LIP-4, y-g-5, x-CNT_LIP-4, y-g+1);
      draw_point(net_xgc, x-CNT_LIP-5, y-g);
      draw_point(net_xgc, x-CNT_LIP-3, y-g);
      draw_line(net_xgc, x-CNT_LIP-4, y+g-1, x-CNT_LIP-4, y+g+5);
      draw_point(net_xgc, x-CNT_LIP-5, y+g);
      draw_point(net_xgc, x-CNT_LIP-3, y+g);
    }
    draw_line(net_xgc, x-CNT_LIP-4, y-g-5, x-CNT_LIP-4, y-g+1);
    draw_point(net_xgc, x-CNT_LIP-5, y-g);
    draw_point(net_xgc, x-CNT_LIP-3, y-g);
    draw_line(net_xgc, x-CNT_LIP-4, y+g-1, x-CNT_LIP-4, y+g+5);
    draw_point(net_xgc, x-CNT_LIP-5, y+g);
    draw_point(net_xgc, x-CNT_LIP-3, y+g);
    break;
  case RIGHT:
    if (erase_flag==0) {
      draw_line(net_gc, x+CNT_LIP+4, y-g-5, x+CNT_LIP+4, y-g+1);
      draw_point(net_gc, x+CNT_LIP+3, y-g);
      draw_point(net_gc, x+CNT_LIP+5, y-g);
      draw_line(net_gc, x+CNT_LIP+4, y+g-1, x+CNT_LIP+4, y+g+5);
      draw_point(net_gc, x+CNT_LIP+3, y+g);
      draw_point(net_gc, x+CNT_LIP+5, y+g);
      
      draw_line(net_xgc, x+CNT_LIP+4, y-g-5, x+CNT_LIP+4, y-g+1);
      draw_point(net_xgc, x+CNT_LIP+3, y-g);
      draw_point(net_xgc, x+CNT_LIP+5, y-g);
      draw_line(net_xgc, x+CNT_LIP+4, y+g-1, x+CNT_LIP+4, y+g+5);
      draw_point(net_xgc, x+CNT_LIP+3, y+g);
      draw_point(net_xgc, x+CNT_LIP+5, y+g);
    }
    draw_line(net_xgc, x+CNT_LIP+4, y-g-5, x+CNT_LIP+4, y-g+1);
    draw_point(net_xgc, x+CNT_LIP+3, y-g);
    draw_point(net_xgc, x+CNT_LIP+5, y-g);
    draw_line(net_xgc, x+CNT_LIP+4, y+g-1, x+CNT_LIP+4, y+g+5);
    draw_point(net_xgc, x+CNT_LIP+3, y+g);
    draw_point(net_xgc, x+CNT_LIP+5, y+g);
    break;
  }
}

