/*
 *	Copyright (c) 1990 by Columbia University.
 */

#include <stdio.h>
#include <sys/types.h>
#include <signal.h>
#include <errno.h>

#include "ikp_defines.h"
#include "ikp_externs.h"
#include "globalE.h"

extern void prc_draw(Process *p);
extern void prc_erase(Process *p);
extern void prc_closechldfds(Process *);
extern void ikp_childerr(char *,int);
extern void cntl_querylock(Process *,int *,int *);


/*
   Select the next Process in the Net to be forked.
   The selected Process will have the smallest net
   increase in open file descriptors among all those
   processes which do not require the opening of more
   file descriptors than are currently available.
   If no as of yet unforked process can be successfully
   forked because of a lack of free descriptors, a pointer
   to one will be returned anyway, along with the number
   of descriptors required. It is the caller's responsibility
   to ensure that enough descriptors are available.
   The return of a NULL pointer means that all processes
   have already been forked.
*/
Process *net_selforkprc(Net *nt,int *fds)
{
  Process *pc, *qc, *sel, *badsel;
  Connector *ct;
  int free_fds, curr_netchange, curr_tmpopen;
  int sel_netchange, sel_tmpopen, badsel_tmpopen;
  
  if (nt == (Net *) NULL)
    return((Process *) NULL);
  if (nt->net_prclist == (Process *) NULL)
    return((Process *) NULL);
  
  free_fds= pst_freefds(nt);
  sel= badsel= (Process *) NULL;
  sel_tmpopen= 0;
  for (pc= nt->net_prclist; pc != (Process *) NULL; pc= pc->prc_next) {
    
    /* pc already forked? */
    if (pc->prc_flags & PRC_FORKED)
      continue;
    
    /* determine net change in open file descriptors and
       the number of descriptors which must be temporarily
       opened before the process is forked */
    cntl_querylock(pc, &curr_netchange, &curr_tmpopen);
    for (ct= pc->prc_input; ct != (Connector *) NULL; ct= ct->cnt_next) {
      if (pc->prc_flags & PRC_REMOTE) {
	if (!(ct->cnt_flags & CNT_FILE)) {
	  qc= (Process *) ct->cnt_conn->cnt_process;
	  if (!(qc->prc_flags & PRC_REMOTE)) {
	    if (ct->cnt_pipe == (Pipeslot *) NULL) {
	      curr_netchange++;
	      curr_tmpopen++;
	    }
	  }
	}
      }
      else {
	if (ct->cnt_flags & CNT_FILE)
	  curr_tmpopen++;
	else if (ct->cnt_pipe == (Pipeslot *) NULL) {
	  qc= (Process *) ct->cnt_conn->cnt_process;
	  if (qc->prc_flags & PRC_REMOTE)
	    curr_tmpopen++;
	  else {
	    curr_netchange++;
	    curr_tmpopen+= 2;
	  }
	}
	else
	  curr_netchange--;
      }
    }
    for (ct= pc->prc_output; ct != (Connector *) NULL; ct= ct->cnt_next) {
      if (pc->prc_flags & PRC_REMOTE) {
	if (!(ct->cnt_flags & CNT_FILE)) {
	  qc= (Process *) ct->cnt_conn->cnt_process;
	  if (!(qc->prc_flags & PRC_REMOTE)) {
	    if (ct->cnt_pipe == (Pipeslot *) NULL) {
	      curr_netchange++;
	      curr_tmpopen++;
	    }
	  }
	}
      }
      else {
	if (ct->cnt_flags & CNT_FILE)
	  curr_tmpopen++;
	else if (ct->cnt_pipe == (Pipeslot *) NULL) {
	  qc= (Process *) ct->cnt_conn->cnt_process;
	  if (qc->prc_flags & PRC_REMOTE)
	    curr_tmpopen++;
	  else {
	    curr_netchange++;
	    curr_tmpopen+= 2;
	  }
	}
	else
	  curr_netchange--;
      }
    }
    for (ct= pc->prc_diag; ct != (Connector *) NULL; ct= ct->cnt_next) {
      
      /* diagnostic connectors don't have to be connected */
      if (!(ct->cnt_flags & CNT_FILE) && (ct->cnt_conn == (Connector *) NULL))
	continue;
      
      if (pc->prc_flags & PRC_REMOTE) {
	if (!(ct->cnt_flags & CNT_FILE)) {
	  qc= (Process *) ct->cnt_conn->cnt_process;
	  if (!(qc->prc_flags & PRC_REMOTE)) {
	    if (ct->cnt_pipe == (Pipeslot *) NULL) {
	      curr_netchange++;
	      curr_tmpopen++;
	    }
	  }
	}
      }
      else {
	if (ct->cnt_flags & CNT_FILE)
	  curr_tmpopen++;
	else if (ct->cnt_pipe == (Pipeslot *) NULL) {
	  qc= (Process *) ct->cnt_conn->cnt_process;
	  if (qc->prc_flags & PRC_REMOTE)
	    curr_tmpopen++;
	  else {
	    curr_netchange++;
	    curr_tmpopen+= 2;
	  }
	}
	else
	  curr_netchange--;
      }
    }
    
    /* if no as yet unforked process has been encountered yet,
       remember this one regardless of descriptor availability */
    if (badsel == (Process *) NULL) {
      badsel= pc;
      badsel_tmpopen= curr_tmpopen;
    }
    
    /* enough fds available and no process selected yet */
    if ((sel == (Process *) NULL) && (curr_tmpopen < free_fds)) {
      sel= pc;
      sel_netchange= curr_netchange;
      sel_tmpopen= curr_tmpopen;
    }
    
    /* enough fds available and smaller net increase in used fds */
    else if ((curr_netchange < sel_netchange) && (curr_tmpopen < free_fds)) {
      sel= pc;
      sel_netchange= curr_netchange;
      sel_tmpopen= curr_tmpopen;
    }
    
    /* enough fds available, same net increase in used fds --
       if the process would temporarily consume more fds,
       fork it now to get it out of the way while we still can */
    else if ((curr_netchange == sel_netchange) && (curr_tmpopen < free_fds) && (curr_tmpopen > sel_tmpopen)) {
      sel= pc;
      sel_netchange= curr_netchange;
      sel_tmpopen= curr_tmpopen;
    }
  }
  if ((sel == (Process *) NULL) && (badsel != (Process *) NULL)) {
    sel= badsel;
    sel_tmpopen= badsel_tmpopen;
  }
  *fds= sel_tmpopen;
  return(sel);
}

/*
   Returns TRUE if all non-diagnostic Connectors of all Processes
   in nt are connected to other Processes or non-null files in nt.
   Diagnostic Connectors bound to files must have non-null filenames.
*/
boolean net_connected(Net *nt)
{
  Process *p, *q;
  Connector *ct;
  
  if (nt == (Net *) NULL) {
    ikp_err("Null net.", NONFATAL);
    return(FALSE);
  }
  if (nt->net_prclist == (Process *) NULL) {
    ikp_err("Empty net.", NONFATAL);
    return(FALSE);
  }
  
  for (p= nt->net_prclist; p != (Process *) NULL; p= p->prc_next) {
    for (ct= p->prc_input; ct != (Connector *) NULL; ct= ct->cnt_next) {
      if (ct->cnt_flags & CNT_FILE) {
	if ((ct->cnt_filename == (char *) NULL)||
	    (strlen(ct->cnt_filename) == 0)) {
	  ct->cnt_flags^= CNT_FILE;
	  if ((ct->cnt_filename= calloc((unsigned) (strlen("/dev/null")+1), sizeof(char))) == (char *) NULL)
	    ikp_err("Memory allocation error.", FATAL);
	  strcpy(ct->cnt_filename, "/dev/null"); 
	}
      } else {
	if (ct->cnt_conn == (Connector *) NULL) {
	  ct->cnt_flags^= CNT_FILE;
	  if ((ct->cnt_filename= calloc((unsigned) (strlen("/dev/null")+1), sizeof(char))) == (char *) NULL)
	    ikp_err("Memory allocation error.", FATAL);
	  strcpy(ct->cnt_filename, "/dev/null"); 
	} else {
	  q= (Process *) ct->cnt_conn->cnt_process;
	  if (q == (Process *) NULL)
	    return(FALSE);
	  if ((Net *) q->prc_net != nt)
	    return(FALSE);
	}
      }
    }
    for (ct= p->prc_output; ct != (Connector *) NULL; ct= ct->cnt_next) {
      if (ct->cnt_flags & CNT_FILE) {
	if ((ct->cnt_filename == (char *) NULL)||
	    (strlen(ct->cnt_filename) == 0)) {
	  ct->cnt_flags^= CNT_FILE;
	  if ((ct->cnt_filename= calloc((unsigned) (strlen("/dev/null")+1), sizeof(char))) == (char *) NULL)
	    ikp_err("Memory allocation error.", FATAL);
	  strcpy(ct->cnt_filename, "/dev/null"); 
	}
      } else {
	if (ct->cnt_conn == (Connector *) NULL) {
	  ct->cnt_flags^= CNT_FILE;
	  if ((ct->cnt_filename= calloc((unsigned) (strlen("/dev/null")+1), sizeof(char))) == (char *) NULL)
	    ikp_err("Memory allocation error.", FATAL);
	  strcpy(ct->cnt_filename, "/dev/null"); 
	} else {
	  q= (Process *) ct->cnt_conn->cnt_process;
	  if (q == (Process *) NULL)
	    return(FALSE);
	  if ((Net *) q->prc_net != nt)
	    return(FALSE);
	}
      }
    }
    for (ct= p->prc_diag; ct != (Connector *) NULL; ct= ct->cnt_next) {
      if (ct->cnt_flags & CNT_FILE) {
	if ((ct->cnt_filename == (char *) NULL)||
	    (strlen(ct->cnt_filename) == 0)) {
	  ct->cnt_flags^= CNT_FILE;
	  if ((ct->cnt_filename= calloc((unsigned) (strlen("/dev/null")+1), sizeof(char))) == (char *) NULL)
	    ikp_err("Memory allocation error.", FATAL);
	  strcpy(ct->cnt_filename, "/dev/null"); 
	}
      }
    }
  }
  return(TRUE);
}

int net_fork(Net *nt)
{
  Process *p;
  Connector *ct;
  int req_fds, ret_val,check_flag;
  char *wkdir=NULL;
  Process *net_selforkprc();
  
  if (nt == (Net *) NULL) {
    ikp_err("Null net.", NONFATAL);
    return(IKP_NULLNET);
  }
  if (nt->net_prclist == (Process *) NULL) {
    ikp_err("Empty net.", NONFATAL);
    return(IKP_EMPTYNET);
  }
  if (!net_connected(nt)) {
    ikp_err("Net not connected.", NONFATAL);
    return(IKP_UNCONNECTED);
  }

  /* build process arg lists, determine if process is local or remote */
  for (p= nt->net_prclist, rem_procs= 0; p != (Process *) NULL; p= p->prc_next) {
    if (prc_bldargv(p) != IKP_SUCCESS)
      return(IKP_BADPRCARGV);
    p->prc_flags&= ~PRC_REMOTE;
    if (p->prc_host != (char *) NULL) {
      if (strlen(p->prc_host) > 0) {
	if (strcmp(hostname, p->prc_host)) {
	  p->prc_flags|= PRC_REMOTE;
	  p->prc_flags&= ~PRC_SOCKOPEN;
	  rem_procs++;
	}
      }
    }
  }
  
  /* allocate pipeslots */
  if (nt->net_pst != (Pipeslot *) NULL)
    free((char *) nt->net_pst);
  nt->net_npipes= getdtablesize()-RESERVED_FDS-rem_procs;
  nt->net_pst= pst_allocblk(nt->net_npipes);
  
  /* open control sockets and set up environments for all remote processes */
  check_flag=0;
  for (p= nt->net_prclist; p != (Process *) NULL; p= p->prc_next)
    if (p->prc_flags & PRC_REMOTE)
      if (strcmp(p->prc_host,"*")==0) 
	check_flag=1;

  if (check_flag)
    check_load();
  for (p= nt->net_prclist; p != (Process *) NULL; p= p->prc_next) {
    if (p->prc_flags & PRC_REMOTE) {
      if (rp_setup(p) != IKP_SUCCESS) {
	net_abortfork(nt);
	return(IKP_FAILURE);
      }
    }
  }
  
  /* initialize process pid and state flags */
  for (p= nt->net_prclist; p != (Process *) NULL; p= p->prc_next) {
    p->prc_pid= 0;
    p->prc_flags&= ~PRC_FORKED;
    p->prc_flags&= ~PRC_RUNNING;
    p->prc_flags&= ~PRC_STOPPED;
    p->prc_flags&= ~PRC_EXITED;
    p->prc_flags&= ~PRC_IKPSTOPPED;
    p->prc_flags&= ~PRC_IKPKILLED;
  }
  run_net= nt;
  poll_count= 0;
  
  while ((p= net_selforkprc(nt, &req_fds)) != (Process *) NULL) {
    if (verbose >= 5)
      fprintf(stderr, "%ld = %s\n", (long) p, p->prc_comm);
    
    if (pst_freefds(nt) < req_fds) {
      ikp_err("Required file descriptors unavailable.", NONFATAL);
      net_abortfork(nt);
      return(IKP_NOFDS);
    }
    
    if ((ret_val= cntl_lockprc(p)) != IKP_SUCCESS) {
      net_abortfork(nt);
      return(ret_val);
    }
    
    for (ct= p->prc_input; ct != (Connector *) NULL; ct= ct->cnt_next) {
      if (ct->cnt_pipe == (Pipeslot *) NULL) {
	if (pst_openpipe(nt, ct) != IKP_SUCCESS) {
	  net_abortfork(nt);
	  return(IKP_PIPE);
	}
      }
    }
    for (ct= p->prc_output; ct != (Connector *) NULL; ct= ct->cnt_next) {
      if (ct->cnt_pipe == (Pipeslot *) NULL) {
	if (pst_openpipe(nt, ct) != IKP_SUCCESS) {
	  net_abortfork(nt);
	  return(IKP_PIPE);
	}
      }
    }
    for (ct= p->prc_diag; ct != (Connector *) NULL; ct= ct->cnt_next) {
      
      /* diagnostic connectors don't have to be connected */
      if (!(ct->cnt_flags & CNT_FILE) && (ct->cnt_conn == (Connector *) NULL))
	continue;
      
      if (ct->cnt_pipe == (Pipeslot *) NULL) {
	if (pst_openpipe(nt, ct) != IKP_SUCCESS) {
	  net_abortfork(nt);
	  return(IKP_PIPE);
	}
      }
    }
    
    /* start remote process */
    if (p->prc_flags & PRC_REMOTE) {
      if (rp_exec(p) != IKP_SUCCESS) {
	net_abortfork(nt);
	return(IKP_FAILURE);
      }
      
      /* the presence of a pipe structure in the three loops
	 which follow imply connection with another process
	 (never a file);  if connection is to a local process,
	 there is only one descriptor opened in this pipe (the
	 socket), and it is always closed after the local
	 process is forked (not the remote process!); if the
	 connection is to a remote process, there are never
	 any open descriptors in the slot */
      for (ct= p->prc_input; ct != (Connector *) NULL; ct= ct->cnt_next) {
	
	if (ct->cnt_pipe != (Pipeslot *) NULL) {
	  ct->cnt_pipe->pst_flags&= ~PST_RDACTIVE;
	  if (!(ct->cnt_pipe->pst_flags & PST_WRACTIVE)) {
	    ct->cnt_pipe= (Pipeslot *) NULL;
	    ct->cnt_conn->cnt_pipe= (Pipeslot *) NULL;
	  }
	}
      }
      for (ct= p->prc_output; ct != (Connector *) NULL; ct= ct->cnt_next) {
	
	if (ct->cnt_pipe != (Pipeslot *) NULL) {
	  ct->cnt_pipe->pst_flags&= ~PST_WRACTIVE;
	  if (!(ct->cnt_pipe->pst_flags & PST_RDACTIVE)) {
	    ct->cnt_pipe= (Pipeslot *) NULL;
	    ct->cnt_conn->cnt_pipe= (Pipeslot *) NULL;
	  }
	}
      }
      for (ct= p->prc_diag; ct != (Connector *) NULL; ct= ct->cnt_next) {
	
	if (ct->cnt_pipe != (Pipeslot *) NULL) {
	  ct->cnt_pipe->pst_flags&= ~PST_WRACTIVE;
	  if (!(ct->cnt_pipe->pst_flags & PST_RDACTIVE)) {
	    ct->cnt_pipe= (Pipeslot *) NULL;
	    ct->cnt_conn->cnt_pipe= (Pipeslot *) NULL;
	  }
	}
      }
    } else {    /* fork local process */
      if (p->prc_type == PRC_LIBRARY)
	wkdir= p->prc_module->mdl_dir;
      else
	wkdir= p->prc_dir;
	
      switch (p->prc_pid= fork()) {
      case -1:
	ikp_err("Fork error.", NONFATAL);
	net_abortfork(nt);
	return(IKP_FORK);
      case 0:

	/* change to specified working directory */
	if ((wkdir != (char *) NULL) && (strlen(wkdir) != 0)) {
	  if (chdir(wkdir) == -1) {
	    (void) sprintf(ikp_errbuf, "Cannot change directory to \"%s\".", wkdir);
	    ikp_childerr(ikp_errbuf, IKP_CHDIR);
	  }
	}
	
	/* unconnected diagnostic connectors must be
	   dup2()ed from parent stderr -- this must be
	   done right away in case stderr is being
	   used by the child for any other connector */
	for (ct= p->prc_diag; ct != (Connector *) NULL; ct= ct->cnt_next) {
	  if (!(ct->cnt_flags & CNT_FILE) && (ct->cnt_conn == (Connector *) NULL)) {
	    if (ct->cnt_desc == 2)
	      continue;
	    
	    if (verbose >= 5)
	      fprintf(stderr, "C ddup %ld: 2 -> %d\n", (long) p, ct->cnt_desc);
	    if (dup2(2, ct->cnt_desc) == -1) {
	      (void) sprintf(ikp_errbuf, "Child diagnostic dup2(2, %d) error %d.", ct->cnt_desc, errno);
	      ikp_childerr(ikp_errbuf, IKP_DUP2);
	    }
	  }
	}
	
	for (ct= p->prc_input; ct != (Connector *) NULL; ct= ct->cnt_next) {
	  if (verbose >= 5)
	    fprintf(stderr, "C idup %ld: %d -> %d\n", (long) p, ct->cnt_pipe->pst_fd[0], ct->cnt_desc);
	  if (dup2(ct->cnt_pipe->pst_fd[0], ct->cnt_desc) == -1) {
	    (void) sprintf(ikp_errbuf, "Child input dup2(%d, %d) error %d.", ct->cnt_pipe->pst_fd[0], ct->cnt_desc, errno);
	    ikp_childerr(ikp_errbuf, IKP_DUP2);
	  }
	  if (verbose >= 5)
	    fprintf(stderr, "C icls %ld: %d\n", (long) p, ct->cnt_pipe->pst_fd[0]);
	  if (close(ct->cnt_pipe->pst_fd[0]) == -1)
	    ikp_childerr("Child input close error.", IKP_CLOSE);
	}
	for (ct= p->prc_output; ct != (Connector *) NULL; ct= ct->cnt_next) {
	  if (verbose >= 5)
	    fprintf(stderr, "C odup %ld: %d -> %d\n", (long) p, ct->cnt_pipe->pst_fd[1], ct->cnt_desc);
	  if (dup2(ct->cnt_pipe->pst_fd[1], ct->cnt_desc) == -1) {
	    (void) sprintf(ikp_errbuf, "Child output dup2(%d, %d) error %d.", ct->cnt_pipe->pst_fd[1], ct->cnt_desc, errno);
	    ikp_childerr(ikp_errbuf, IKP_DUP2);
	  }
	  if (verbose >= 5)
	    fprintf(stderr, "C ocls %d: %d\n", (long) p, ct->cnt_pipe->pst_fd[1]);
	  if (close(ct->cnt_pipe->pst_fd[1]) == -1)
	    ikp_childerr("Child output close error.", IKP_CLOSE);
	}
	for (ct= p->prc_diag; ct != (Connector *) NULL; ct= ct->cnt_next) {
	  
	  /* unconnected diagnostic connectors
	     have already been taken care of */
	  if (!(ct->cnt_flags & CNT_FILE) && (ct->cnt_conn == (Connector *) NULL))
	    continue;
	  
	  if (verbose >= 5)
	    fprintf(stderr, "C ddup %ld: %d -> %d\n", (long) p, ct->cnt_pipe->pst_fd[1], ct->cnt_desc);
	  if (dup2(ct->cnt_pipe->pst_fd[1], ct->cnt_desc) == -1) {
	    (void) sprintf(ikp_errbuf, "Child diagnostic dup2(%d, %d) error %d.", ct->cnt_pipe->pst_fd[1], ct->cnt_desc, errno);
	    ikp_childerr(ikp_errbuf, IKP_DUP2);
	  }
	  if (verbose >= 5)
	    fprintf(stderr, "C dcls %d: %d\n", (long) p, ct->cnt_pipe->pst_fd[1]);
	  if (close(ct->cnt_pipe->pst_fd[1]) == -1)
	    ikp_childerr("Child diagnostic close error.", IKP_CLOSE);
	}
	
	prc_closechldfds(p);
	execvp(p->prc_argv[0], p->prc_argv);
	(void) sprintf(ikp_errbuf, "Exec error on process '%s'.", p->prc_argv[0]);
	ikp_childerr(ikp_errbuf, IKP_EXEC);
	break;
      }
      
      /* parent code continues here */
      for (ct= p->prc_input; ct != (Connector *) NULL; ct= ct->cnt_next) {
	if (close(ct->cnt_pipe->pst_fd[0]) == -1) {
	  ikp_err("Close error.", NONFATAL);
	  net_abortfork(nt);
	  return(IKP_CLOSE);
	}
	if (verbose >= 5)
	  fprintf(stderr, "P icls %d: %d\n", (long) p, ct->cnt_pipe->pst_fd[0]);
	ct->cnt_pipe->pst_flags&= ~PST_RDACTIVE;
	if (ct->cnt_pipe->pst_flags & PST_SOCKET)
	  ct->cnt_pipe->pst_flags&= ~PST_SOCKACTIVE;
	if (!(ct->cnt_pipe->pst_flags & PST_WRACTIVE)) {
	  ct->cnt_pipe= (Pipeslot *) NULL;
	  if (ct->cnt_conn != (Connector *) NULL)
	    ct->cnt_conn->cnt_pipe= (Pipeslot *) NULL;
	}
      }
      for (ct= p->prc_output; ct != (Connector *) NULL; ct= ct->cnt_next) {
	if (close(ct->cnt_pipe->pst_fd[1]) == -1) {
	  ikp_err("Close error.", NONFATAL);
	  net_abortfork(nt);
	  return(IKP_CLOSE);
	}
	if (verbose >= 5)
	  fprintf(stderr, "P ocls %d: %d\n", (long) p, ct->cnt_pipe->pst_fd[1]);
	ct->cnt_pipe->pst_flags&= ~PST_WRACTIVE;
	if (ct->cnt_pipe->pst_flags & PST_SOCKET)
	  ct->cnt_pipe->pst_flags&= ~PST_SOCKACTIVE;
	if (!(ct->cnt_pipe->pst_flags & PST_RDACTIVE)) {
	  ct->cnt_pipe= (Pipeslot *) NULL;
	  if (ct->cnt_conn != (Connector *) NULL)
	    ct->cnt_conn->cnt_pipe= (Pipeslot *) NULL;
	}
      }
      for (ct= p->prc_diag; ct != (Connector *) NULL; ct= ct->cnt_next) {
	if (!(ct->cnt_flags & CNT_FILE) && (ct->cnt_conn == (Connector *) NULL))
	  continue;
	
	if (close(ct->cnt_pipe->pst_fd[1]) == -1) {
	  ikp_err("Close error.", NONFATAL);
	  net_abortfork(nt);
	  return(IKP_CLOSE);
	}
	if (verbose >= 5)
	  fprintf(stderr, "P dcls %d: %d\n", (long) p, ct->cnt_pipe->pst_fd[1]);
	ct->cnt_pipe->pst_flags&= ~PST_WRACTIVE;
	if (ct->cnt_pipe->pst_flags & PST_SOCKET)
	  ct->cnt_pipe->pst_flags&= ~PST_SOCKACTIVE;
	if (!(ct->cnt_pipe->pst_flags & PST_RDACTIVE)) {
	  ct->cnt_pipe= (Pipeslot *) NULL;
	  if (ct->cnt_conn != (Connector *) NULL)
	    ct->cnt_conn->cnt_pipe= (Pipeslot *) NULL;
	}
      }
      if (verbose >= 5)
	sleep(2);
    }
    if ((ret_val= cntl_unlockprc(p)) != IKP_SUCCESS) {
      net_abortfork(nt);
      return(ret_val);
    }
    p->prc_flags|= PRC_FORKED;
    p->prc_flags|= PRC_RUNNING;
    if (usage_mode == INTERACTIVE)
      prc_showactive(p);
  }
  
  return(IKP_SUCCESS);
}

/* 
  A SIGINT signal has been received (ie. a Ctrl-C of a batch run ) and
  we want to exit the net gracefully so the daemon processes don't keep
  going waiting on the next timeout.
					- joe m. wade 3/29/96
*/
void sigint_hdlr()
{
  if (run_net != (Net *) NULL)
    net_abortfork(run_net);
  exit(1);
}

/*
   Abort a forked net by killing all running
   child and remote processes and closing all open
   connector and remote control socket descriptors.
*/
int net_abortfork(Net *nt)
{
  Process *p;
  Connector *c;
  char childnm[40];
  
  if (run_net == (Net *) NULL)
    return(IKP_FAILURE);
  if (nt != run_net)
    return(IKP_FAILURE);
  
  (void) cntl_unlockall();
  for (p= nt->net_prclist; p != (Process *) NULL; p= p->prc_next) {
    if (usage_mode == INTERACTIVE) {
      prc_erase(p);
      prc_draw(p);
    }
    if ((p->prc_flags & PRC_RUNNING) || (p->prc_flags & PRC_STOPPED)) {
      
      if (p->prc_flags & PRC_REMOTE) {
	(void) rp_closedown(p);
	p->prc_flags&= ~PRC_RUNNING;
	p->prc_flags&= ~PRC_STOPPED;
	p->prc_flags|= PRC_IKPKILLED;
      }
      
      else {
	
#ifdef WAIT4
	qid= wait4(p->prc_pid, &status, WNOHANG | WUNTRACED, (struct rusage *) NULL);
	
	/* process still running or stopped */
	if ((qid == 0) ||
	    ((qid == p->prc_pid) && WIFSTOPPED(status))) {
	  if (kill(p->prc_pid, SIGKILL) < 0) {
	    prc_bldlabel(p, childnm, 39);
	    (void) sprintf(ikp_errbuf, "Process '%s' (PID %d) cannot be killed.", childnm, p->prc_pid);
	    ikp_err(ikp_errbuf, NONFATAL);
	  }
	  
	  /* the process may actually still be
	     running or stopped, but from our point
	     of view it's out of both sight and mind,
	     so mark it killed */
	  wtp_add(p->prc_pid);
	  p->prc_flags&= ~PRC_RUNNING;
	  p->prc_flags&= ~PRC_STOPPED;
	  p->prc_flags|= PRC_IKPKILLED;
	}
	
	/* process exited */
	else {
	  p->prc_flags&= ~PRC_RUNNING;
	  p->prc_flags&= ~PRC_STOPPED;
	  p->prc_flags|= PRC_EXITED;
	}
#else
	/* process may have exited already (we haven't
	   wait()ed for it yet), so don't worry about
	   return value from kill() */
	(void) kill(p->prc_pid, SIGKILL);
	
	wtp_add(p->prc_pid);
	p->prc_flags&= ~PRC_RUNNING;
	p->prc_flags&= ~PRC_STOPPED;
	p->prc_flags|= PRC_IKPKILLED;
#endif
      }
    }
    for (c= p->prc_input; c != (Connector *) NULL; c= c->cnt_next) {
      if (c->cnt_pipe == (Pipeslot *) NULL)
	continue;
      if (c->cnt_pipe->pst_flags & PST_RDACTIVE) {
	if (close(c->cnt_pipe->pst_fd[0]) < 0)
	  ikp_err("Close error.", NONFATAL);
	c->cnt_pipe->pst_flags&= ~PST_RDACTIVE;
      }
      if (c->cnt_pipe->pst_flags & PST_WRACTIVE) {
	if (close(c->cnt_pipe->pst_fd[1]) < 0)
	  ikp_err("Close error.", NONFATAL);
	c->cnt_pipe->pst_flags&= ~PST_WRACTIVE;
      }
      c->cnt_pipe= (Pipeslot *) NULL;
    }
    for (c= p->prc_output; c != (Connector *) NULL; c= c->cnt_next) {
      if (c->cnt_pipe == (Pipeslot *) NULL)
	continue;
      if (c->cnt_pipe->pst_flags & PST_RDACTIVE) {
	if (close(c->cnt_pipe->pst_fd[0]) < 0)
	  ikp_err("Close error.", NONFATAL);
	c->cnt_pipe->pst_flags&= ~PST_RDACTIVE;
      }
      if (c->cnt_pipe->pst_flags & PST_WRACTIVE) {
	if (close(c->cnt_pipe->pst_fd[1]) < 0)
	  ikp_err("Close error.", NONFATAL);
	c->cnt_pipe->pst_flags&= ~PST_WRACTIVE;
      }
      c->cnt_pipe= (Pipeslot *) NULL;
    }
    for (c= p->prc_diag; c != (Connector *) NULL; c= c->cnt_next) {
      if (c->cnt_pipe == (Pipeslot *) NULL)
	continue;
      if (c->cnt_pipe->pst_flags & PST_RDACTIVE) {
	if (close(c->cnt_pipe->pst_fd[0]) < 0)
	  ikp_err("Close error.", NONFATAL);
	c->cnt_pipe->pst_flags&= ~PST_RDACTIVE;
      }
      if (c->cnt_pipe->pst_flags & PST_WRACTIVE) {
	if (close(c->cnt_pipe->pst_fd[1]) < 0)
	  ikp_err("Close error.", NONFATAL);
	c->cnt_pipe->pst_flags&= ~PST_WRACTIVE;
      }
      c->cnt_pipe= (Pipeslot *) NULL;
    }
  }
  
  run_net= (Net *) NULL;
  if (usage_mode == BATCH)
    batch_status= IKP_FAILURE;
  
  return(IKP_SUCCESS);
}
