/*
 *	Copyright (c) 1990 by Columbia University.
 */

#include <stdio.h>
#include <ctype.h>

#include "ikp_defines.h"
#include "ikp_externs.h"
#include "globalE.h"

extern Module *mdl_create(Library *,Process *,char *);
extern void ikp_ioerr(char *,int);
extern void net_draw(Net *);

void net_mktags(Net *);

int net_read(int x,int y,char *filename)
{
  int nread, retval;
  boolean error;
  FILE *fp;
  
  if (filename == (char *) NULL) {
    ikp_err("Null filename.", NONFATAL);
    return(IKP_FAILURE);
  }
  if ((fp= fopen(filename, "r")) == (FILE *) NULL) {
    ikp_err("File open error.", NONFATAL);
    return(IKP_FAILURE);
  }
  
  nread= 0;
  error= FALSE;
  ikp_linenum= 0;
  while (ikp_matchkey("net", fp) == IKP_SUCCESS) {
    if ((retval= net_readnet(x, y, fp)) != IKP_SUCCESS) {
      error= TRUE;
      break;
    }
    else
      nread++;
  }
  if (!error && !nread) {
    ikp_err("File contains no net configuration.", NONFATAL);
    fclose(fp);
    return(IKP_FAILURE);
  }
  
  fclose(fp);
  return(retval);
}

int net_write(Net *nt,char *filename,int flag_location)
{
  int x, y;
  Process *p;
  Connector *c;
  char cntfilenm[MAXFILENMSZ+3];
  FILE *fp;
  
  if (nt == (Net *) NULL) {
    ikp_err("Null net.", NONFATAL);
    return(IKP_FAILURE);
  }
  if (filename == (char *) NULL) {
    ikp_err("Null filename.", NONFATAL);
    return(IKP_FAILURE);
  }
  if (flag_location) {
    if ((fp= fopen(filename, "a")) == (FILE *) NULL) {
      ikp_err("File open error.", NONFATAL);
      return(IKP_FAILURE);
    }
  } else {
    if ((fp= fopen(filename, "w")) == (FILE *) NULL) {
      ikp_err("File open error.", NONFATAL);
      return(IKP_FAILURE);
    }
  }
  
  net_mktags(nt);
  
  /* compute origin of smallest bounding
     rectangle containing all process icons */
  if (nt->net_prclist != (Process *) NULL) {
    x= nt->net_prclist->prc_x;
    y= nt->net_prclist->prc_y;
  }
  for (p= nt->net_prclist; p != (Process *) NULL; p= p->prc_next) {
    if (x > p->prc_x)
      x= p->prc_x;
    if (y > p->prc_y)
      y= p->prc_y;
  }
  
  fprintf(fp, "net {\n\n");
  if (nt->name!=NULL)
    fprintf(fp,"\tnet_name\t%s\n",nt->name);
  for (p= nt->net_prclist; p != (Process *) NULL; p= p->prc_next) {
    char prctype[20];  

    if (prc_bldargv(p) != IKP_SUCCESS)
      return(IKP_BADPRCARGV);
    
    switch (p->prc_type) {
    case PRC_SOURCE:
      (void) sprintf(prctype, "source");
      break;
    case PRC_SINK:
      (void) sprintf(prctype, "sink");
      break; 
    case PRC_SCRIPT:
      (void) sprintf(prctype, "script");
      break;
   case PRC_FILTER:
      (void) sprintf(prctype, "filter");
      break;
    case PRC_CUSTOM:
      (void) sprintf(prctype, "custom");
      break;
    case PRC_LIBRARY:
      (void) sprintf(prctype, "library");
      break;
    default:
      break;
    }
    
    fprintf(fp, "\tprocess %s {\n", prctype);
    
    if (p->prc_host != (char *) NULL)
      fprintf(fp, "\t\thost		\"%s\"\n", p->prc_host);
    if (p->prc_user != (char *) NULL)
      fprintf(fp, "\t\tuser		\"%s\"\n", p->prc_user);
    if (p->prc_comm != (char *) NULL)
      fprintf(fp, "\t\tcommand		\"%s\"\n", p->prc_comm);
    if (p->prc_type==PRC_LIBRARY) {
      if (p->prc_module->mdl_path != (char *) NULL)
	fprintf(fp, "\t\tpathprefix	\"%s\"\n", p->prc_module->mdl_path);
      if (p->prc_module->mdl_label != (char *) NULL)
	fprintf(fp, "\t\tmodule		\"%s\"\n", p->prc_module->mdl_label);

/* reorder this to change effect - 6/28/95 ************ j.m.wade

    }
    if (p->prc_dir != (char *) NULL) {
      fprintf(fp, "\t\tdirectory	\"%s\"\n", p->prc_dir);
    } else {
      if (p->prc_type==PRC_LIBRARY)
	if (p->prc_module->mdl_dir != (char *) NULL)
	  fprintf(fp, "\t\tdirectory	\"%s\"\n", p->prc_module->mdl_dir);
    }
*/
      if (p->prc_module->mdl_dir != (char *) NULL)
	fprintf(fp, "\t\tdirectory	\"%s\"\n", p->prc_module->mdl_dir);
    } else {
      if (p->prc_dir != (char *) NULL)
        fprintf(fp, "\t\tdirectory	\"%s\"\n", p->prc_dir);
    }
    if (p->prc_label != (char *) NULL)
      fprintf(fp, "\t\tlabel		\"%s\"\n", p->prc_label);
    if (p->prc_imagefile != (char *) NULL)
      fprintf(fp, "\t\timagefile	\"%s\"\n", p->prc_imagefile);
    
    if (flag_location)
      fprintf(fp, "\t\tparams		%d %d %d %d %d\n", p->prc_flags, p->prc_x, p->prc_y, p->prc_w, p->prc_h);
    else
      fprintf(fp, "\t\tparams		%d %d %d %d %d\n", p->prc_flags, p->prc_x-x, p->prc_y-y, p->prc_w, p->prc_h);
								     
    for (c= p->prc_input; c != (Connector *) NULL; c= c->cnt_next) {
      if (c->cnt_filename == (char *) NULL)
	(void) sprintf(cntfilenm, "\"\"");
      else if (strlen(c->cnt_filename) == 0)
	(void) sprintf(cntfilenm, "\"\"");
      else
	(void) sprintf(cntfilenm, "\"%s\"", c->cnt_filename);
      fprintf(fp, "\t\tinput		%d %d %d %d %d %d %d %d %s\n", c->cnt_class, c->cnt_desc, c->cnt_orient, c->cnt_flags, c->cnt_x, c->cnt_y, c->cnt_tag, c->cnt_conntag, cntfilenm);
    }
    for (c= p->prc_output; c != (Connector *) NULL; c= c->cnt_next) {
      if (c->cnt_filename == (char *) NULL)
	(void) sprintf(cntfilenm, "\"\"");
      else if (strlen(c->cnt_filename) == 0)
	(void) sprintf(cntfilenm, "\"\"");
      else
	(void) sprintf(cntfilenm, "\"%s\"", c->cnt_filename);
      fprintf(fp, "\t\toutput		%d %d %d %d %d %d %d %d %s\n", c->cnt_class, c->cnt_desc, c->cnt_orient, c->cnt_flags, c->cnt_x, c->cnt_y, c->cnt_tag, c->cnt_conntag, cntfilenm);
    }
    for (c= p->prc_diag; c != (Connector *) NULL; c= c->cnt_next) {
      if (c->cnt_filename == (char *) NULL)
	(void) sprintf(cntfilenm, "\"\"");
      else if (strlen(c->cnt_filename) == 0)
	(void) sprintf(cntfilenm, "\"\"");
      else
	(void) sprintf(cntfilenm, "\"%s\"", c->cnt_filename);
      fprintf(fp, "\t\tdiagnostic	%d %d %d %d %d %d %d %d %s\n", c->cnt_class, c->cnt_desc, c->cnt_orient, c->cnt_flags, c->cnt_x, c->cnt_y, c->cnt_tag, c->cnt_conntag, cntfilenm);
    }
    fprintf(fp, "\t}\n\n");
  }
  fprintf(fp, "}\n\n");

  fclose(fp);
  return(IKP_SUCCESS);
}

int net_readnet(int x,int y,FILE *fp)
{
  Net *nt;
  Process *p;
  boolean complete;
  char *linestart, token[MAXTOKENSZ+1];
  
  if ((nt= net_create()) == (Net *) NULL)
    return(IKP_FAILURE);
  
  complete= FALSE;
  while (ikp_getline(&linestart, fp) == IKP_SUCCESS) {
    if (ikp_gettoken(linestart, 0, token, MAXTOKENSZ) != IKP_SUCCESS)
      continue;
    
    if (!strcmp(token, "process")) {
      if (net_readprc(x, y, nt, fp) != IKP_SUCCESS) {
	for (p= nt->net_prclist; p != (Process *) NULL; p= p->prc_next) {
	  prc_unlink(p);
	  prc_free(p, PRC_RETAINNET);
	}
	net_free(nt);
	return(IKP_FAILURE);
      }
    }
    else if (!strcmp(token, "}")) {
      complete= TRUE;
      break;
    }
    else if (!strncmp(token, "#", 1))
      continue;
    else if (!strcmp(token, "net_name")) {
      strcpy(buf2,"");
      sscanf(linestart,"%s %s",buf1,buf2);
      nt->name=strdup(buf2);
      continue;
    }
    else {
      sprintf(ikp_errbuf, "Unrecognized token '%s'", token);
      ikp_ioerr(ikp_errbuf, NONFATAL);
      net_free(nt);
      return(IKP_FAILURE);
    }
  }
  
  if (!complete) {
    ikp_err("Incomplete net configuration file.", NONFATAL);
    for (p= nt->net_prclist; p != (Process *) NULL; p= p->prc_next) {
      prc_unlink(p);
      prc_free(p, PRC_RETAINNET);
    }
    net_free(nt);
    return(IKP_FAILURE);
  }
  
  /* connect all connectors, build segpaths */
  if (net_rdtags(nt) != IKP_SUCCESS) {
    for (p= nt->net_prclist; p != (Process *) NULL; p= p->prc_next) {
      prc_unlink(p);
      prc_free(p, PRC_RETAINNET);
    }
    net_free(nt);
    return(IKP_FAILURE);
  }
  
  /* link into net list */
  nt->net_next= nets;
  nets= nt;
  
  if (usage_mode == INTERACTIVE)
    net_draw(nt);
  curr_net= nt;
  
  return(IKP_SUCCESS);
}

int net_readprc(int x,int y,Net *nt,FILE *fp)
{
  char *linestart, token[MAXTOKENSZ+1];
  char prctype[MAXTOKENSZ+1],module_name[MAXTOKENSZ+1];
  char path_name[MAXTOKENSZ+1];
  Process *p, *q;
  Connector *c;
  Mdlitem *mi;
  int type, class, desc, orient, flags;
  int cx, cy, tag, conntag;
  boolean complete, params_found;
  
  if (ikp_gettoken(ikp_iobuf, 1, prctype, MAXTOKENSZ) == IKP_FAILURE) {
    ikp_ioerr("Malformed process declaration line", NONFATAL);
    return(IKP_FAILURE);
  }
  
  if (!strcmp(prctype, "source"))
    type= PRC_SOURCE;
  else if (!strcmp(prctype, "sink"))
    type= PRC_SINK;
  else if (!strcmp(prctype, "script"))
    type= PRC_SCRIPT;
  else if (!strcmp(prctype, "filter"))
    type= PRC_FILTER;
  else if (!strcmp(prctype, "custom"))
    type= PRC_CUSTOM;
  else if (!strcmp(prctype, "library"))
    type= PRC_LIBRARY;
  else {
    ikp_ioerr("Unrecognized process type", NONFATAL);
    return(IKP_FAILURE);
  }
  
  if ((p= (Process *) calloc(1, sizeof(Process))) == (Process *) NULL)
    ikp_err("Memory allocation error.", FATAL);
  p->prc_host= (char *) NULL;
  p->prc_user= (char *) NULL;
  p->prc_comm= (char *) NULL;
  p->prc_argv= (char **) NULL;
  p->prc_dir= (char *) NULL;
  p->prc_type= type;
  p->prc_flags= 0;
  p->prc_input= p->prc_output= p->prc_diag= (Connector *) NULL;
  p->prc_image= (XImage *) NULL;
  p->prc_imagefile= (char *) NULL;
  p->prc_module= (Module *) NULL;
  p->prc_next= (Process *) NULL;
  
  complete= params_found= FALSE;
  while (ikp_getline(&linestart, fp) == IKP_SUCCESS) {
    if (ikp_gettoken(linestart, 0, token, MAXTOKENSZ) != IKP_SUCCESS)
      continue;
    
    if (!strcmp(token, "host")) {
      if (ikp_gettoken(linestart, 1, token, MAXTOKENSZ) != IKP_SUCCESS) {
	ikp_ioerr("Malformed process host line", NONFATAL);
	prc_free(p, PRC_RETAINNET);
	return(IKP_FAILURE);
      }
      if (p->prc_host != (char *) NULL) {
	free(p->prc_host);
	p->prc_host= (char *) NULL;
      }
      if (strlen(token) > 0) {
	if ((p->prc_host= calloc((unsigned) (strlen(token)+1), sizeof(char))) == (char *) NULL)
	  ikp_err("Memory allocation error.", FATAL);
	(void) strcpy(p->prc_host, token);
      }
    }
    
    else if (!strcmp(token, "user")) {
      if (ikp_gettoken(linestart, 1, token, MAXTOKENSZ) != IKP_SUCCESS) {
	ikp_ioerr("Malformed process user line", NONFATAL);
	prc_free(p, PRC_RETAINNET);
	return(IKP_FAILURE);
      }
      if (p->prc_user != (char *) NULL) {
	free(p->prc_user);
	p->prc_user= (char *) NULL;
      }
      if (strlen(token) > 0) {
	if ((p->prc_user= calloc((unsigned) (strlen(token)+1), sizeof(char))) == (char *) NULL)
	  ikp_err("Memory allocation error.", FATAL);
	(void) strcpy(p->prc_user, token);
      }
    }
    
    else if (!strcmp(token, "command")) {
      if (ikp_gettoken(linestart, 1, token, MAXTOKENSZ) != IKP_SUCCESS) {
	ikp_ioerr("Malformed process command line", NONFATAL);
	prc_free(p, PRC_RETAINNET);
	return(IKP_FAILURE);
      }
      if (p->prc_comm != (char *) NULL) {
	free(p->prc_comm);
	p->prc_comm= (char *) NULL;
      }
      if (strlen(token) > 0) {
	if ((p->prc_comm= calloc((unsigned) (strlen(token)+1), sizeof(char))) == (char *) NULL)
	  ikp_err("Memory allocation error.", FATAL);
	(void) strcpy(p->prc_comm, token);
      }
    }
    
    else if (!strcmp(token, "directory")) {
      if (ikp_gettoken(linestart, 1, token, MAXTOKENSZ) != IKP_SUCCESS) {
	ikp_ioerr("Malformed process directory line", NONFATAL);
	prc_free(p, PRC_RETAINNET);
	return(IKP_FAILURE);
      }
      if (p->prc_dir != (char *) NULL) {
	free(p->prc_dir);
	p->prc_dir= (char *) NULL;
      }
      if (strlen(token) > 0) {
	if ((p->prc_dir= calloc((unsigned) (strlen(token)+1), sizeof(char))) == (char *) NULL)
	  ikp_err("Memory allocation error.", FATAL);
	(void) strcpy(p->prc_dir, token);
      }
    }
    
    else if (!strcmp(token, "label")) {
      if (ikp_gettoken(linestart, 1, token, MAXTOKENSZ) != IKP_SUCCESS) {
	ikp_ioerr("Malformed process label line", NONFATAL);
	prc_free(p, PRC_RETAINNET);
	return(IKP_FAILURE);
      }
      if (p->prc_label != (char *) NULL) {
	free(p->prc_label);
	p->prc_label= (char *) NULL;
      }
      if (strlen(token) > 0) {
	if ((p->prc_label= calloc((unsigned) (strlen(token)+1), sizeof(char))) == (char *) NULL)
	  ikp_err("Memory allocation error.", FATAL);
	(void) strcpy(p->prc_label, token);
      }
    }
    
    else if (!strcmp(token, "imagefile")) {
      if (ikp_gettoken(linestart, 1, token, MAXTOKENSZ) != IKP_SUCCESS) {
	ikp_ioerr("Malformed process imagefile line", NONFATAL);
	prc_free(p, PRC_RETAINNET);
	return(IKP_FAILURE);
      }
      if (p->prc_imagefile != (char *) NULL) {
	free(p->prc_imagefile);
	p->prc_imagefile= (char *) NULL;
      }
      if (strlen(token) > 0) {
	if ((p->prc_imagefile= calloc((unsigned) (strlen(token)+1), sizeof(char))) == (char *) NULL)
	  ikp_err("Memory allocation error.", FATAL);
	(void) strcpy(p->prc_imagefile, token);
      }
    }
    
    else if (!strcmp(token, "params")) {
      
      if (sscanf(linestart, "%*s %d %d %d %d %d %d\n",
		 &(p->prc_flags), &(p->prc_x), &(p->prc_y),
		 &(p->prc_w), &(p->prc_h)) != 5) {
	ikp_ioerr("Net configuration file format error", NONFATAL);
	prc_free(p, PRC_RETAINNET);
	return(IKP_FAILURE);
      }
      p->prc_x+= x;
      p->prc_y+= y;
      params_found= TRUE;
    }
    
    else if (!strcmp(token, "input")) {
      if (sscanf(linestart, "%*s %d %d %d %d %d %d %d %d", &class, &desc, &orient, &flags, &cx, &cy, &tag, &conntag) != 8) {
	ikp_ioerr("Net configuration file format error", NONFATAL);
	prc_free(p, PRC_RETAINNET);
	return(IKP_FAILURE);
      }
      if ((c= cnt_create(p, CNT_INPUT, class, desc)) == (Connector *) NULL) {
	prc_free(p, PRC_RETAINNET);
	return(IKP_FAILURE);
      }
      c->cnt_orient= orient;
      c->cnt_x= cx;
      c->cnt_y= cy;
      c->cnt_tag= tag;
      c->cnt_conntag= conntag;
      c->cnt_flags= flags & ~CNT_GRNDDRAWN;
      if (ikp_gettoken(linestart, 9, token, MAXTOKENSZ) != IKP_SUCCESS) {
	ikp_ioerr("Malformed input line", NONFATAL);
	prc_free(p, PRC_RETAINNET);
	return(IKP_FAILURE);
      }
      if (strlen(token) == 0)
	c->cnt_filename= (char *) NULL;
      else {
	if ((c->cnt_filename= calloc((unsigned) (strlen(token)+1), sizeof(char))) == (char *) NULL)
	  ikp_err("Memory allocation error.", FATAL);
	(void) strcpy(c->cnt_filename, token);
      }
    }
    
    else if (!strcmp(token, "output")) {
      if (sscanf(linestart, "%*s %d %d %d %d %d %d %d %d", &class, &desc, &orient, &flags, &cx, &cy, &tag, &conntag) != 8) {
	ikp_ioerr("Net configuration file format error", NONFATAL);
	prc_free(p, PRC_RETAINNET);
	return(IKP_FAILURE);
      }
      if ((c= cnt_create(p, CNT_OUTPUT, class, desc)) == (Connector *) NULL) {
	prc_free(p, PRC_RETAINNET);
	return(IKP_FAILURE);
      }
      c->cnt_orient= orient;
      c->cnt_x= cx;
      c->cnt_y= cy;
      c->cnt_tag= tag;
      c->cnt_conntag= conntag;
      c->cnt_flags= flags & ~CNT_GRNDDRAWN;
      if (ikp_gettoken(linestart, 9, token, MAXTOKENSZ) != IKP_SUCCESS) {
	ikp_ioerr("Malformed output line", NONFATAL);
	prc_free(p, PRC_RETAINNET);
	return(IKP_FAILURE);
      }
      if (strlen(token) == 0)
	c->cnt_filename= (char *) NULL;
      else {
	if ((c->cnt_filename= calloc((unsigned) (strlen(token)+1), sizeof(char))) == (char *) NULL)
	  ikp_err("Memory allocation error.", FATAL);
	(void) strcpy(c->cnt_filename, token);
      }
    }
    
    else if (!strcmp(token, "diagnostic")) {
      if (sscanf(linestart, "%*s %d %d %d %d %d %d %d %d", &class, &desc, &orient, &flags, &cx, &cy, &tag, &conntag) != 8) {
	ikp_ioerr("Net configuration file format error", NONFATAL);
	prc_free(p, PRC_RETAINNET);
	return(IKP_FAILURE);
      }
      if ((c= cnt_create(p, CNT_DIAG, class, desc)) == (Connector *) NULL) {
	prc_free(p, PRC_RETAINNET);
	return(IKP_FAILURE);
      }
      c->cnt_orient= orient;
      c->cnt_x= cx;
      c->cnt_y= cy;
      c->cnt_tag= tag;
      c->cnt_conntag= conntag;
      c->cnt_flags= flags & ~CNT_GRNDDRAWN;
      if (ikp_gettoken(linestart, 9, token, MAXTOKENSZ) != IKP_SUCCESS) {
	ikp_ioerr("Malformed diagnostic line", NONFATAL);
	prc_free(p, PRC_RETAINNET);
	return(IKP_FAILURE);
      }
      if (strlen(token) == 0)
	c->cnt_filename= (char *) NULL;
      else {
	if ((c->cnt_filename= calloc((unsigned) (strlen(token)+1), sizeof(char))) == (char *) NULL)
	  ikp_err("Memory allocation error.", FATAL);
	(void) strcpy(c->cnt_filename, token);
      }
    }
    else if (!strcmp(token, "module")) {
      if (ikp_gettoken(linestart, 1, token, MAXTOKENSZ) != IKP_SUCCESS) {
	ikp_ioerr("Malformed process command line", NONFATAL);
	prc_free(p, PRC_RETAINNET);
	return(IKP_FAILURE);
      }
      strcpy(module_name,token);
    }
    else if (!strcmp(token, "pathprefix")) {
      if (ikp_gettoken(linestart, 1, token, MAXTOKENSZ) != IKP_SUCCESS) {
	ikp_ioerr("Malformed process command line", NONFATAL);
	prc_free(p, PRC_RETAINNET);
	return(IKP_FAILURE);
      }
      strcpy(path_name,token);
    }
    else if (!strcmp(token, "}")) {
      if (p->prc_type==PRC_LIBRARY) {
	sscanf(module_name,"%s ",buf1);
	if (strrchr(buf1,'/')!=NULL) {
	  strcpy(buf2,strrchr(buf1,'/')+1);
	  strcpy(buf1,buf2);
	}
	p->prc_module=mdl_create(NULL,p,buf1);
	if (p->prc_module!=NULL) {
	  p->prc_module->mdl_path=strdup(path_name);
	  if (p->prc_module->mdl_dir != (char *) NULL) {
	    free(p->prc_module->mdl_dir);
	    p->prc_module->mdl_dir=NULL;
	  }
	  if (p->prc_dir!=NULL)
	    p->prc_module->mdl_dir=strdup(p->prc_dir);
	}
      }
      complete= TRUE;
      break;
    }
    
    else if (!strncmp(token, "#", 1))
      continue;
    
    else {
      sprintf(ikp_errbuf, "Unrecognized token '%s'", token);
      ikp_ioerr(ikp_errbuf, NONFATAL);
      prc_free(p, PRC_RETAINNET);
      return(IKP_FAILURE);
    }
  }
  
  if (!complete || !params_found) {
    ikp_err("Incomplete net configuration file.", NONFATAL);
    prc_free(p, PRC_RETAINNET);
    return(IKP_FAILURE);
  }
  
  p->prc_net= (caddr_t) nt;
  p->prc_next= (Process *) NULL;
  
  if (nt->net_prclist == (Process *) NULL)
    nt->net_prclist= p;
  else {
    for (q= nt->net_prclist; q->prc_next != (Process *) NULL; q= q->prc_next);
    q->prc_next= p;
  }
  
  return(IKP_SUCCESS);
}

/*
   Tag all connectors with a unique integer id --
   this is done so that all connectors can be properly
   reconnected when this file is read at a later time.
*/
void net_mktags(Net *nt)
{
  int tag;
  Process *p;
  Connector *c;
  
  tag= 1;
  for (p= nt->net_prclist; p != (Process *) NULL; p= p->prc_next) {
    for (c= p->prc_input; c != (Connector *) NULL; c= c->cnt_next)
      c->cnt_tag= tag++;
    for (c= p->prc_output; c != (Connector *) NULL; c= c->cnt_next)
      c->cnt_tag= tag++;
    for (c= p->prc_diag; c != (Connector *) NULL; c= c->cnt_next)
      c->cnt_tag= tag++;
  }
  for (p= nt->net_prclist; p != (Process *) NULL; p= p->prc_next) {
    for (c= p->prc_input; c != (Connector *) NULL; c= c->cnt_next) {
      if (c->cnt_conn == (Connector *) NULL)
	c->cnt_conntag= 0;
      else
	c->cnt_conntag= c->cnt_conn->cnt_tag;
    }
    for (c= p->prc_output; c != (Connector *) NULL; c= c->cnt_next) {
      if (c->cnt_conn == (Connector *) NULL)
	c->cnt_conntag= 0;
      else
	c->cnt_conntag= c->cnt_conn->cnt_tag;
    }
    for (c= p->prc_diag; c != (Connector *) NULL; c= c->cnt_next) {
      if (c->cnt_conn == (Connector *) NULL)
	c->cnt_conntag= 0;
      else
	c->cnt_conntag= c->cnt_conn->cnt_tag;
    }
  }
  return;
}

/*
   Use tag information written out by net_mktags()
   to reconnect all connectors properly.
*/
int net_rdtags(Net *nt)
{
  Process *p, *q;
  Connector *c, *d;
  boolean found;
  
  for (p= nt->net_prclist; p != (Process *) NULL; p= p->prc_next) {
    
    for (c= p->prc_input; c != (Connector *) NULL; c= c->cnt_next) {
      
      /* never connected or already connected? */
      if (c->cnt_conntag == 0)
	continue;
      
      found= FALSE;
      for (q= nt->net_prclist; q != (Process *) NULL; q= q->prc_next) {
	for (d= q->prc_output; d != (Connector *) NULL; d= d->cnt_next) {
	  if (d->cnt_tag == c->cnt_conntag) {
	    found= TRUE;
	    break;
	  }
	}
	if (found)
	  break;
	for (d= q->prc_diag; d != (Connector *) NULL; d= d->cnt_next) {
	  if (d->cnt_tag == c->cnt_conntag) {
	    found= TRUE;
	    break;
	  }
	}
	if (found)
	  break;
      }
      if (!found) {
	ikp_err("Net configuration file connector tag match error.", NONFATAL);
	return(IKP_FAILURE);
      }
      c->cnt_conn= d;
      d->cnt_conn= c;
      
      /* make sure we don't make the connection twice */
      c->cnt_conntag= d->cnt_conntag= 0;
      
      seg_bldpath(c, d);
    }
    
    for (c= p->prc_output; c != (Connector *) NULL; c= c->cnt_next) {
      
      /* never connected or already connected? */
      if (c->cnt_conntag == 0)
	continue;
      
      found= FALSE;
      for (q= nt->net_prclist; q != (Process *) NULL; q= q->prc_next) {
	for (d= q->prc_input; d != (Connector *) NULL; d= d->cnt_next) {
	  if (d->cnt_tag == c->cnt_conntag) {
	    found= TRUE;
	    break;
	  }
	}
	if (found)
	  break;
      }
      if (!found) {
	ikp_err("Net configuration file connector tag match error.", NONFATAL);
	return(IKP_FAILURE);
      }
      c->cnt_conn= d;
      d->cnt_conn= c;
      
      /* make sure we don't make the connection twice */
      c->cnt_conntag= d->cnt_conntag= 0;
      
      seg_bldpath(c, d);
    }
    
    for (c= p->prc_diag; c != (Connector *) NULL; c= c->cnt_next) {
      
      /* never connected or already connected? */
      if (c->cnt_conntag == 0)
	continue;
      
      found= FALSE;
      for (q= nt->net_prclist; q != (Process *) NULL; q= q->prc_next) {
	for (d= q->prc_input; d != (Connector *) NULL; d= d->cnt_next) {
	  if (d->cnt_tag == c->cnt_conntag) {
	    found= TRUE;
	    break;
	  }
	}
	if (found)
	  break;
      }
      if (!found) {
	ikp_err("Net configuration file connector tag match error.", NONFATAL);
	return(IKP_FAILURE);
      }
      c->cnt_conn= d;
      d->cnt_conn= c;
      
      /* make sure we don't make the connection twice */
      c->cnt_conntag= d->cnt_conntag= 0;
      
      seg_bldpath(c, d);
    }
    
  }
  
  return(IKP_SUCCESS);
}
