C***********************************************************************
c                 copyright 2001, Amoco Production Company             *
c                             All Rights Reserved                      *
c                     an affiliate of BP America Inc.                  *
C***********************************************************************
c Program rmo
c
c Changes:
c
c January 21, 2003: Fixed a bug in the output of the -afile gamma volume
c                   that required CrwNam to be RMOFOR in order for the 
c                   file to be recognized as gamma values on the way
c                   back through rmo after conditioning of the volume.
c                   Reported by Walter Reitveldt.
c Garossino
c
c August 30, 1999: Changed routine to read DstSgn instead of DstUsg.  This is
c                  the standard [if there is such a thing] location for offset
c                  in the USP world.  Users were always wondering why this code
c                  did nothing....when in fact it was doing lots, just on zero
c                  offset all the time.....
c Garossino
c
c March 9, 2000: Added option to do gamma scans using sqrt[z^2 + (g^2-1)*(x/2)^2]
c                with suitable limits.  The code can now be run in 2 passes:
c                1. compute alpha (for -parab) or gamma (for -gamma)
c                2. apply previously computed alphas or gammas
c                Two passes allows user smoothing which is especially useful for
c                3D.  For the gamma option the user specifies a shift limit at
c                maximum offset and max depth:  
c                zdel = sqrt[z^2 + (g^2-1)*(xmax/2)^2] - z0
c                and this is used to check the cmd line shift limits -loshift
c                and -hishift.  If either of the cmd line parameters result in
c                the max shifts exceeding the limit then it is recalculated to
c                be within this limit (at max offset and max depth).  This limit
c                is used throughout to prevent any shallower shift from exceeding
c                this value keeping the shallower data from undergoing extreme
c                moveout.
c March 9, 2000: removed the bandwidth dependent interpolation/oversampling
c                routines DYNAMIC proprietary to Paradigm and replaced with sinc
c                interpolation and quadratic interpolation. The Paradigm method
c                is somewhat superior and this rmo is available as a service
c                from them.
c Gutowski
c

        program rmo
        implicit none

C-DESCRIPTION:
C
C       RMO performs residual moveout analysis and compensation.
C       Input data is assumed to be CDP ordered and NMO corrected.
C       Residual moveout errors are assumed to be of the form:
C               DeltaT = alpha * offset**2 * (1 - (offset/xref)**2)
C
C       Trial stacks with a range of alpha and xref values are performed.
C       The stack with the highest absolute amplitude is used to select
C       a particular alpha and xref for each CDP at each time sample.
C       These peak alpha,xref values are then used to dynamically
C       stretch/compress the data for output.
C
C
C       RMO
C        |
C        |--> HELP - Terse on-line help
C        |
C        |--> CMDLN  - read and check user command line parameters
C        |      |
C        |      |--> VERBAL - Echo info
C        |
C        |--> INIT - Initialize computational arrays
C        |      |
C        |      |--> oversampt - Create optimum sinc interpolation coefficients
C        |
C        |--> PROC
C               |
C               |--> ANALYZE - Performs trial moveout and stack on a CDP
C               |       |
C               |       |--> oversampt  - supersample input traces
C               |
C               |--> P_ANALYZE - Performs parabolic trial moveout and stack
C               |       |
C               |       |--> oversampt  - supersample input traces
C               |
C               |--> APPLY - Selects best moveout and applies dynamic shifts
C                            [4th order moveout]
C                       |
C                       |--> vqint  - dynamic (time-variant) stretch/compress
C               |--> P_APPLY - Selects best moveout and applies dynamic shifts
C                            [parabolic or gamma moveout]
C                       |
C                       |--> vqint  - dynamic (time-variant) stretch/compress
C
C
C
c
C-INCLUDE:
c         get machine dependent parameters 
        integer jerr      ! Stupid includes aren't compatible with implicit none
        character name*4
c  ! Defines: LINHED
#include <f77/sisdef.h>
c  ! Defines: SZLNHD, SZSMPD, SZTRHD, ITHWP1, etc.
#include <f77/lhdrsz.h>
#include <save_defs.h> 
#include <rmocb.fin>
c  ! Defines: LER, LERR, LUIN, LUOUT, LUPPRT, etc.
#include <f77/iounit.h>

C-VARIABLES:
C ----> Local variables
        integer qtr(SZLNHD)
        integer argis
        integer error
        integer tot_err
        integer n_bytes
        integer tot_bytes
        integer abort
        integer max_n_shifts   ! Maximium number of shifts

        real max_shift    ! Maximum allowed shift at max_offset
        real t_short      ! Max allowed shift at offsets less than xref
        real stable       ! Stability weight, a user parameter
        real lo_shift     ! Parabolic minimum shift
        real hi_shift     ! Parabolic maximum shift


        real vfi
        pointer (ixr_vfi, vfi(10))
        real map
        pointer (ixr_map, map(10))
        real alpha
        pointer (ixr_alpha, alpha(10))
        real weights
        pointer (ixr_weights, weights(10))
        real xrefs
        pointer (ixr_xref, xrefs(10))
        integer thdr
        pointer (ixl_thdr, thdr(10))
        real cdp
        pointer (ixr_cdp, cdp(10))
        real sum
        pointer (ixr_sum, sum(10))
        real work1
        pointer (ixr_work1, work1(10))
        real work2
        pointer (ixr_work2, work2(10))
        real work3
        pointer (ixr_work3, work3(10))
        real work4
        pointer (ixr_work4, work4(10))
        real work5
        pointer (ixr_work5, work5(10))
        real work6
        pointer (ixr_work6, work6(10))

C-DATA:
        data name/"RMO"/
        data abort/0/
        data pipe/3/

C-REVISED:   8-Nov-1996 by Steve Markley  - Initial version
c       Framework copied from prgm_record.F
c       Algorithm modified from DISCO RMO module.
C-REVISED: March 1997        by Steve Markley
C       Added parabolic option. Added auxiliary QC outputs. Added feature
C       to stabilize results for low fold taper regions at beginning and
C       end of the line.


C **********************************************************************
C 3456789012345678901234567890123456789012345678901234567890123456789012
C        1         2         3         4         5         6         7

c       opens and date stamps printout file
c    ! Uses jerr, name(1:2), LERR, and lupprt
#include <f77/open.h>

C **********************************************************************
C ***** Parameter Processing *******************************************
C **********************************************************************


C ***** Quick on-line parameter help ***********************************

        if (argis ('-?') .gt. 0 .or. 
     *      argis ('-h') .gt. 0 .or. 
     *      argis ('-help') .gt. 0 ) then
            call help()
            stop
        endif


C ***** Process user parameters and open input and output datasets *****

        call cmdln (max_shift, lo_shift, hi_shift, t_short, stable,
     :              qtr)


C **********************************************************************
C ***** Dynamic memory allocation (yes rmo is a memory hog) ************
C **********************************************************************

        stride = nint (bandw * super_rate)       ! Determine super sampling rate
        super_length = (length - 1) * stride + 1

        if (.not. parabolic .AND. .not. gamma) then
            max_n_shifts = nint (max_shift * bandw) + 2
        elseif (parabolic) then
            max_n_shifts = nint (lo_shift * bandw) + 
     *                     nint (hi_shift * bandw) + 1
        elseif (gamma) then
            max_n_shifts = nscans
        endif

C ----> Sample interpolation tables (Vector Function Interpolation)
        n_bytes = SZSMPD * n_coeff * stride
        call galloc (ixr_vfi, n_bytes, error, abort)
        tot_err = error
        tot_bytes = n_bytes

C ----> Stretch index table for trace super sample
        n_bytes = SZSMPD * super_length
        call galloc (ixr_map, n_bytes, error, abort)
        tot_err = tot_err + error
        tot_bytes = tot_bytes + n_bytes

C ----> Reference offsets for analysis
        if (.not. parabolic .AND. .not. gamma) then
            n_bytes = SZSMPD * max_n_shifts * n_xref
            call galloc (ixr_xref, n_bytes, error, abort)
            tot_err = tot_err + error
            tot_bytes = tot_bytes + n_bytes
        endif

C ----> Curvature parameter values for anlysis
        n_bytes = SZSMPD * max_n_shifts * n_xref
        call galloc (ixr_alpha, n_bytes, error, abort)
        tot_err = tot_err + error
        tot_bytes = tot_bytes + n_bytes

C ----> Stability weights for analysis
        n_bytes = SZSMPD * max_n_shifts * n_xref
        call galloc (ixr_weights, n_bytes, error, abort)
        tot_err = tot_err + error
        tot_bytes = tot_bytes + n_bytes

C ----> Trace header storage
        n_bytes = SZSMPD * maxntr * thdrlen * (cdp_delay + 1)
        call galloc (ixl_thdr, n_bytes, error, abort)
        tot_err = tot_err + error
        tot_bytes = tot_bytes + n_bytes

C ----> Raw CDP storage
        n_bytes = SZSMPD * maxntr * length * (cdp_delay + 1)
        call galloc (ixr_cdp, n_bytes, error, abort)
        tot_err = tot_err + error
        tot_bytes = tot_bytes + n_bytes


C ***** Initialize computational arrays (including calc n_anl) *********

        call rmo_init (map, xrefs, alpha, weights, lo_shift,
     *                 hi_shift, max_shift, t_short, stable, vfi,
     *                 qtr)

C ----> Spatially smoothed stacked amplitudes
        n_bytes = SZSMPD * n_anl * n_smash * length
        call galloc (ixr_sum, n_bytes, error, abort)
        tot_err = tot_err + error
        tot_bytes = tot_bytes + n_bytes

C ----> Working array
        n_bytes = SZSMPD * n_anl * length
        call galloc (ixr_work1, n_bytes, error, abort)
        tot_err = tot_err + error
        tot_bytes = tot_bytes + n_bytes

C ----> Working array
        n_bytes = SZSMPD * (super_length + n_coeff)
        call galloc (ixr_work2, n_bytes, error, abort)
        tot_err = tot_err + error
        tot_bytes = tot_bytes + n_bytes

C ----> Working array
        n_bytes = SZSMPD * (super_length + n_coeff)
        call galloc (ixr_work3, n_bytes, error, abort)
        tot_err = tot_err + error
        tot_bytes = tot_bytes + n_bytes

C ----> Working array
        n_bytes = SZSMPD * length
        call galloc (ixr_work4, n_bytes, error, abort)
        tot_err = tot_err + error
        tot_bytes = tot_bytes + n_bytes

C ----> Working array
        n_bytes = SZSMPD * (length + n_coeff)
        call galloc (ixr_work5, n_bytes, error, abort)
        tot_err = tot_err + error
        tot_bytes = tot_bytes + n_bytes

C ----> Working array
        n_bytes = SZSMPD * max (length, maxntr)
        call galloc (ixr_work6, n_bytes, error, abort)
        tot_err = tot_err + error
        tot_bytes = tot_bytes + n_bytes

        if (tot_err .ne. 0) then
            write (LERR, 2000) tot_bytes
            write (LER, 2000) tot_bytes
 2000       format (/' RMO: Unable to allocate workspace:', i10, 
     *                                                      '  bytes')
            go to 999
        else
            write (LERR, 2100) tot_bytes
            write (LER, 2100) tot_bytes
 2100       format (/' RMO: Allocating workspace:', i10, ' bytes')
        endif


C **********************************************************************
C ***** Process the dataset ********************************************
C **********************************************************************

        call proc (thdr, cdp, Sum, vfi, map, xrefs, alpha, weights, 
     *                     Work1, Work2, Work3, Work4, Work5, Work6)


C **********************************************************************
C ***** Close **********************************************************
C **********************************************************************

c       close data files 

        call lbclos (luin)
        call lbclos (luout)
        if (afile) call lbclos (lu_alpha)
        if (xfile) call lbclos (lu_xref)
        if (QC) call lbclos (luqc)
        write (LERR, *) 'RMO: Normal Termination'
        write (LER, *) 'RMO: Normal Termination'
        stop

C ----> On error, jump here
 999    continue

        call lbclos (luin)
        call lbclos (luout)
        if (afile) call lbclos (lu_alpha)
        if (xfile) call lbclos (lu_xref)
        write (LERR, *) 'RMO: ABNORMAL Termination'
        write (LER, *) 'RMO: ABNORMAL Termination'
        stop
        end









C***********************************************************************
C                                                                      *
c                 copyright 2001, Amoco Production Company             *
c                             All Rights Reserved                      *
c                     an affiliate of BP America Inc.                  *
C***********************************************************************

        subroutine help
        implicit none

C-FUNCTION:
c        provide terse online help [detailed help goes in man page]
C
C-INCLUDE:
c    ! Defines: LER, LERR, LUIN, LUOUT, LUPPRT, etc.
#include <f77/iounit.h>

C-ARGUMENTS:
C-VARIABLES:
C-DATA:
C-REVISED:   8-Nov-1996 by Steve Markley  - Initial version
C-REVISED: February 1997        by Steve Markley
C       Added parabolic option. Added auxiliary QC outputs.


C **********************************************************************
        write (LER, 1000)
        write (LER, 2000)
        write (LER, 3000)
        write (LER, 4000)
        write (LER, 5000)
        write (LER, 5250)
        write (LER, 5260)
        write (LER, 5500)
        write (LER, 5510)
        write (LER, 6000)
        write (LER, 7000)
        write (LER, 8000)

C **********************************************************************
 1000   format (/
     *        '===================================================='//
     *        ' Command Line Arguments for PRGM: rmo'//
     *        ' For a more detailed description of these parameters'/
     *        ' see the online man page using uman, xman or xuspman'/)

 2000   format (
     *        'Input........................................... (def)'//
     *        '-N[]       -- Input data file name             (stdin)'/
     *        '-O[]       -- Output data file name           (stdout)'/
     *        '-maxoff[]  -- Maximum input offset              (none)')

 3000   format (
     *        '-deltat[]  -- Max far offset shift (ms)          (200)'/
     *        '-wind[]    -- Time smoother (ms)          (40 samples)'/
     *        '-minxref[] -- Minimum reference offset     (.5*maxoff)'/
     *        '-maxxref[] -- Maximum reference offset     (.8*maxoff)')
 
 4000   format (
     *        '-nxref[]   -- Number of reference offsets          (8)'/
     *        '-ncdp[]    -- Number of CDPs to smooth results     (5)'/
     *        '-tshort[]  -- Max shift on short traces (ms)  (4 samp)'/
     *        '-bandw[]   -- data bandwidth fraction             (.5)')
 
 5000   format (
     *        '-stabl[]   -- Stability weight                    (.9)'/
     *        '-qcfile[]  -- QC output semblances              (none)'/
     *        '-afile[]   -- QC output of alpha or gamma vals  (none)'/
     *        '-xfile[]   -- QC output file of ref offsets     (none)')
 
 5250   format (
     *        '----------------------------------------------------'/
     *        '-parab     -- Parabolic residual moveout option'/
     *        '-maxoff[]  -- Maximum input offset              (none)'/
     *        '-loshift[] -- Parabolic: Minimum shift          (-200)')
 
 5260   format (
     *        '----------------------------------------------------'/
     *        '-gamma     -- gamma residual moveout option'/
     *        '-maxoff[]  -- Maximum input offset              (none)'/
     *        '-loshift[] -- Gamma: Minimum shift               (0.9)')

 5500   format (
     *        '-hishift[] -- Parabolic: Maximum shift           (200)'/
     *        '-limit[]   -- limit absolute max shift % trc length(5)'/
     *        '-A         -- separately compute/apply rmo            '/
     *        '(i.e. separate runs: one to generate alpha data set   '/
     *        '      and one to read cdps and alpha data and apply.  '/
     *        '      The second run cmd line must have maxoff -A and '/
     *        '      -afile[] to read in the alpha/gamma field       '/
     *        '----------------------------------------------------'/
     *        '-V         -- Verbose printout'/
     *        '-foff[]    -- Verbose: first offset        (maxoff/10)')

 5510   format (
     *        '-hishift[] -- Gamma: Maximum shift               (1.1)'/
     *        '-A         -- separately compute/apply rmo            '/
     *        '(i.e. separate runs: one to generate alpha data set   '/
     *        '      and one to read cdps and alpha data and apply.  '/
     *        '      The second run cmd line must have maxoff -A and '/
     *        '      -afile[] to read in the alpha/gamma field       '/
     *        '----------------------------------------------------'/
     *        '-V         -- Verbose printout'/
     *        '-foff[]    -- Verbose: first offset        (maxoff/10)')

 
 6000   format (
     *        '-loff[]    -- Verbose: last offset            (maxoff)'/
     *        '-noff[]    -- Verbose: number of offsets          (10)'/
     *        '-tinc[]    -- Verbose: time increment (ms)      (wind)'/
     *        '-mincdp[]  -- Verbose: first record to print       (1)')

 7000   format (
     *        '-cdpinc[]  -- Verbose: print record increment    (100)'/
     *        'or the increment to output semblance QC gathers       '/
     *        '----------------------------------------------------'/)
 
 8000   format (/'Usage:'/
     *        '     rmo -N[] -O[] -maxoff[] - deltat[] -wind[] -V'/
     *        '            -maxxref[] -minxref[] -nxref[] -ncdp[]'/
     *        '            -tshort[] -bandw[] -stabl[] -afile[]'/
     *        '            -xfile[] -parab -loshift[] -hishift[] -A'/
     *        '            -qcfile[] -gamma -loshift[] -hishift[]  '/
     *        '            -foff[] -loff[] -noff[] -tinc[]'/
     *        '====================================================')
C **********************************************************************

        return
        end





C***********************************************************************
C                                                                      *
c                 copyright 2001, Amoco Production Company             *
c                             All Rights Reserved                      *
c                     an affiliate of BP America Inc.                  *
C***********************************************************************

        subroutine cmdln (max_shift, lo_shift, hi_shift, t_short,stable,
     :                    qtr)
        implicit none

C-FUNCTION:
C       CMDLN reads and checks user's command line arguments
C
C-INCLUDE:
#include <f77/lhdrsz.h>
#include <f77/sisdef.h>
#include <save_defs.h> 
#include <rmocb.fin>
#include <f77/iounit.h>

C-ARGUMENTS:
        real max_shift    ! Maximum allowed shift at max_offset (Output)
        real lo_shift     ! Maximum parabolic overcorrection in samples (O/P)
        real hi_shift     ! Maximum parabolic undercorrection in samples (O/P)
        real t_short      ! Max allowed shift at offsets less than xref (Output)
        real stable       ! Stability weight, a user parameter (Output)

C-VARIABLES:
        integer n_err
        integer argis
        integer lbytes
        integer lbyout
        integer dt        ! Sample interval from line header
        integer iform
        integer nqcrec
        integer itr(SZLNHD)       ! Needed for call getln
        integer status
	integer in_ikp
	integer idz, nsi
	integer str(SZLNHD)
	integer qtr(SZLNHD)

        real wind         ! Time smoother window length (milliseconds)
        real max_time     ! Extended trace length in milliseconds
        real delta_t

        character name*4
        character ident * 6
        character ntap*255
        character otap*255
        character atap*255
        character xtap*255
        character qctap*255

	logical   IKP

C-DATA:
        data name/"RMO"/
        data IKP/.false./

C-REVISED:   8-Nov-1996 by Steve Markley  - Initial version
C       Modified from DISCO RMO module.
C-REVISED: February 1997        by Steve Markley
C       Added parabolic option. Added auxiliary QC outputs.
C-REVISED: November 1998        by Steve Markley
C       Fixed a botched error message for the upper limit on the WIND parameter.

C **********************************************************************
C 3456789012345678901234567890123456789012345678901234567890123456789012
C        1         2         3         4         5         6         7


        forward = .false.
C **********************************************************************
C ***** Scan user parameters *******************************************
C **********************************************************************

        call argstr ('-afile', atap, ' ', ' ')   ! QC file name for curve parm

        call argstr ('-qcfile', qctap, ' ', ' ')   ! QC file name for semblances

        call argr4 ('-bandw', bandw, .5, .5)     ! Temporal resolution factor

        call argi4 ('-cdpinc', cdp_inc, 100, 100)     
                                 ! Verbose printout record increment

        call argr4 ('-deltat', max_shift, 200., 200.)     
                                 ! Maximum shift to analyze

        call argr4 ('-foff', first_off, -1., -1.)
                                 ! Verbose printout, first offset

        call argr4 ('-hishift', hi_shift, 200., 200.)
                                 ! Parabolic or gamma shift maximum

        call argr4 ('-loff', last_off, -1., -1.)
                                 ! Verbose printout, last offset

        call argr4 ('-loshift', lo_shift, -200., -200.)
                                 ! Parabolic or gamma shift minimum

        call argr4 ('-maxoff', max_offset, 0., 0.)     
                                 ! Maximum source to receiver offset

        call argr4 ('-maxxref', max_xref, -1., -1.)
                                 ! Maximum reference offset for analysis

        call argi4 ('-mincdp', start_cdp, 1, 1)     
                                 ! Verbose printout starting record

        call argr4 ('-minxref', min_xref, -1., -1.)
                                 ! Minimum reference offset for analysis

        call argi4 ('-ncdp', n_smash, 5, 5)      ! Spatial smoother

        call argi4 ('-noff', n_off, 10, 10)     
                                 ! Number of offset tables for verbose print out

        call argi4 ('-nxref', n_xref, 8, 8)     
                                 ! Number of analysis reference offsets

        call argi4 ('-nscans', nscans, 31, 31) ! max number of gamma scans    

        call argstr ('-N', ntap, ' ', ' ')       ! Input file name

        call argstr ('-O', otap, ' ', ' ')       ! Output file name

        parabolic = (argis ('-parab') .gt. 0)    ! Parabolic residula moveout

        gamma     = (argis ('-gamma') .gt. 0)    ! Gamma scans (instead of parabs)

        call argr4 ('-stabl', stable, .9, .9)    ! Stability weight

        call argr4 ('-tinc', delta_t, -1., -1.)     
                                 ! Verbose printout time increment

        call argr4 ('-tshort', t_short, -1., -1.)
                                 ! Maxshift for inside the reference offset

        verbose = (argis ('-V') .gt. 0)          ! Print option

        a_in    = (argis ('-A') .gt. 0)        ! output alpha instead of nmo cdps

        call argr4 ('-wind', wind, -1., -1.)    ! Time smoother length

        call argr4 ('-limit', limit, 5.0, 5.0)  ! gamma shift limit (% trc length)

        call argstr ('-xfile', xtap, ' ', ' ')   ! QC file name for offset parm


      if (a_in) then
         if (.not. gamma) parabolic = .true.
         if (parabolic)
     1   write(LERR,*)'Will use parabolic option'
         if (gamma)
     1   write(LERR,*)'Will use gamma option'
         if (gamma) then
            if (lo_shift .eq. -200.) lo_shift = .5
            if (hi_shift .eq. +200.) hi_shift = 1.5
            if (lo_shift .ge. hi_shift) then
              write(LERR,*)'FATAL ERROR in rmo gamma option:'
              write(LERR,*)'Must have loshift < hishift (usually'
              write(LERR,*)'loshift < 1 and hishift > 1)'
              write(LER ,*)'FATAL ERROR in rmo gamma option:'
              write(LER ,*)'Must have loshift < hishift (usually'
              write(LER ,*)'loshift < 1 and hishift > 1)'
              call ccexit (666)
            endif
         endif
         xfile = .false.
         xtap = ' '
	 n_smash = 1
      endif

      max_offset2 = .25 * max_offset * max_offset

C **********************************************************************
C ***** check for extraneous arguments and abort if found **************
C ***** (catch all manner of user typo's) ******************************
C **********************************************************************

        call xtrarg (name, ler, .FALSE., .FALSE.)
        call xtrarg (name, lerr, .FALSE., .TRUE.)


C **********************************************************************
C ***** Process Input dataset ******************************************
C **********************************************************************


C ***** Open Input dataset *********************************************

        call getln (luin , ntap, 'r', 0)

c       read input line header and save certain parameters
        call rtape (luin, itr, lbytes)
        if (lbytes .eq. 0) then
            write (LER, *) ntap
            write (LERR, *) ntap
 3000       format (' RMO: no line header on input file '/ 
     *                                                 255a / 'FATAL')
            stop
        endif


C ***** Read line header values ****************************************

        call saver (itr, 'NumSmp', length, LINHED)
        call saver (itr, 'SmpInt', dt, LINHED)
        call saver (itr, 'NumTrc', maxntr, LINHED)
        call saver (itr, 'NumRec', n_cdp, LINHED)
        call saver (itr, 'Format', iform, LINHED)
        call saver (itr, 'CrwNam', ident, LINHED)
        call saver (itr, 'TmSlIn', idz 	, LINHED)
        call saver (itr, 'UnitSc', unitsc , LINHED)
        if (gamma .AND. idz .eq. 0) then
           write(LERR,*)'FATAL ERROR in rmo gamma option:'
           write(LERR,*)'zero depth interval detected in line header'
           write(LERR,*)'word Dz1000.  Use utop to fix this:'
           write(LERR,*)'utop -Nindata -Oindata -h0Dz1000=idz,'
           write(LERR,*)'where idz is the depth interval x 1000'
           write(LER ,*)'FATAL ERROR in rmo gamma option:'
           write(LER ,*)'zero depth interval detected in line header'
           write(LER ,*)'word Dz1000.  Use utop to fix this:'
           write(LER ,*)'utop -Nindata -Oindata -h0Dz1000=idz,'
           write(LER ,*)'where idz is the depth interval x 1000'
           stop
        endif
        nsi = dt
        if (unitsc .eq. 0.0) unitsc = .001
        dz = idz / 1000.
        dzdz = dz * dz
        if (gamma) write(LERR,*)'DZ = ',dz

        if (n_smash .gt. n_cdp) then
           write(LERR,*)'WARNING: number of cdps to smooth= ',n_smash
           write(LERR,*)'> than the number of input recs= ',n_cdp
           write(LERR,*)'Will set to 1'
           n_smash = 1
        endif

	if(in_ikp() .gt. 0) IKP = .true.

C ----> Check for stacked data
        if (maxntr .le. 1) then
            write (LERR, 400)
            write (LER, 400)
 400        format (' RMO: Input appears to be stacked data')
            stop
        endif


C ***** Set up pointers to input trace headers *************************

c ----> Trace number
        call savelu ('TrcNum', ifmt_TrcNum, l_TrcNum, ln_TrcNum, 
     *                                 TRACEHEADER)

c ----> Record number
        call savelu ('RecNum', ifmt_RecNum, l_RecNum, ln_RecNum, 
     *                                 TRACEHEADER)

c ----> Common depth point  index
        call savelu ('DphInd', ifmt_DphInd, l_DphInd, ln_DphInd, 
     *                                 TRACEHEADER)

c ----> Line  index
        call savelu ('LinInd', ifmt_LinInd, l_LinInd, ln_LinInd, 
     *                                 TRACEHEADER)

c ----> Static correction
        call savelu ('StaCor', ifmt_StaCor, l_StaCor, ln_StaCor, 
     *                                 TRACEHEADER)

c ----> Signed source to receiver distance
        call savelu ('DstSgn', ifmt_DstSgn, l_DstSgn, ln_DstSgn, 
     *                                 TRACEHEADER)


C **********************************************************************
C ***** Set up output dataset ******************************************
C **********************************************************************

c ----> Update historical line header and save
        call hlhprt (itr, lbytes, name, 3, LERR)
        call savhlh (itr, lbytes, lbyout)
	call vmov (itr, 1, str, 1, SZLNHD)
	call vmov (itr, 1, qtr, 1, SZLNHD)



C **********************************************************************
C ***** Initialize variables *******************************************
C **********************************************************************

C ----> Trace header length
        thdrlen = ITRWRD

C ----> Sample interval
	dt_ms = nsi    ! millisecs or microsecs
	dt = unitsc * nsi  ! secs
        if (gamma) dt_ms = dz

C ----> Trace length in milliseconds (for local use)
        max_time = real (length) * dt_ms


C **********************************************************************
C ***** Parameter checking *********************************************
C **********************************************************************

        n_err = 0

C ----> Temporal resolution factor
        if (bandw .gt. 1.) then
            write (LER, 1000)
            write (LERR, 1000)
 1000       format (' RMO: The BANDW parameter must be less than 1.0')
            n_err = n_err + 1
            bandw = 1.

        elseif (bandw .lt. .2) then
            write (LER, 1010)
            write (LERR, 1010)
 1010       format (' RMO: The BANDW parameter must be greater than .2')
            n_err = n_err + 1
            bandw = .2

        endif

C ----> Maximum parabolic shift
        if (parabolic) then
            if (hi_shift .lt. 0.) then
                write (LER, 1020)
                write (LERR, 1020)
 1020           format (' RMO: The HISHIFT parameter must be ',
     *                                                'greater than 0.')
                n_err = n_err + 1
c - for some reason this confused the Linux Absoft compiler 
c               hi_shift = max_time / -20.
                hi_shift = -(max_time / 20.)

            elseif (hi_shift .gt. max_time/20.) then
                write (LER, 1030)
                write (LERR, 1030)
 1030           format (' RMO: The HISHIFT parameter must be less than',
     *                               ' the trace length / 20.')
                n_err = n_err + 1
                hi_shift = 0.

            endif
            hi_shift = hi_shift / dt_ms

        elseif (gamma) then

            hi_shift = (abs(hi_shift))
            if (hi_shift .lt. 1.0) then
               write(LER ,1071)
               write(LERR,1071)
 1071          format (' RMO gamma opt: HISHIFT parameter must be gt',
     *                 ' than 1.0')
               n_err = n_err + 1
               hi_shift = 1.5
            endif

        endif

C ----> Minimum parabolic shift
        if (parabolic) then
            if (lo_shift .lt. -max_time/20.) then
                write (LER, 1050)
                write (LERR, 1050)
 1050           format (' RMO: The LOSHIFT parameter must be greater ',
     *                               'than the trace length / 20.')
                n_err = n_err + 1
c - for some reason this confused the Linux Absoft compiler 
c               lo_shift = max_time / -20.)
                lo_shift = -(max_time / 20.)

            elseif (lo_shift .gt. 0.) then
                write (LER, 1060)
                write (LERR, 1060)
 1060           format (' RMO: The LOSHIFT parameter must be less ',
     *                                                        'than 0.')
                n_err = n_err + 1
                lo_shift = -16.

            endif
            lo_shift = abs (lo_shift) / dt_ms
            nscans = nint (lo_shift * bandw) +
     *                     nint (hi_shift * bandw) + 1

        elseif (gamma) then

            lo_shift = (abs(lo_shift))
            if (lo_shift .gt. 1.0) then
               write(LER ,1070)
               write(LERR,1070)
 1070          format (' RMO gamma opt: LOSHIFT parameter must be less',
     *                 ' than 1.0')
               n_err = n_err + 1
               lo_shift = .5
            endif

            if ((hi_shift-lo_shift) .lt. .2) then
              write(LERR,*)'WARNING from rmo gamma option:'
              write(LERR,*)'loshift and hishift are very close. Are'
              write(LERR,*)'you sure you want to scan this tight?'
              write(LER ,*)'WARNING from rmo gamma option:'
              write(LER ,*)'loshift and hishift are very close. Are'
              write(LER ,*)'you sure you want to scan this tight?'
            endif

        endif


C ----> Maximum shift to analyze
        if (.not. parabolic .AND. .not. gamma) then
            if (max_shift .lt. 4. * dt_ms) then
                write (LER, 1100)
                write (LERR, 1100)
 1100           format (' RMO: The DELTAT parameter must be greater ',
     *                                                 'than 4 samples')
                n_err = n_err + 1
                max_shift = 4. * dt_ms

            elseif (max_shift .gt. max_time / 20.) then
                write (LER, 1110) 
                write (LERR, 1110)
 1110           format (' RMO: The DELTAT parameter must be less than',
     *                                            ' trace length / 20')
                n_err = n_err + 1
                max_shift = max_time / 20.

            endif

        endif
        max_shift = max_shift / dt_ms               ! Convert to samples

C ----> Maximum source to receiver offset
        if (max_offset .le. 0.) then
            write (LER, 1400)
            write (LERR, 1400)
 1400       format (' RMO: The MAXOFF parameter must be specified and',
     *                                   ' must be greater than zero.')
            n_err = n_err + 1
        endif

C ----> Minimum reference offset for analysis
        if ((min_xref .eq. -1.) .or. parabolic .or. gamma) then
            min_xref = max_offset / 2.

        elseif (min_xref .lt. max_offset / 5.) then
            write (LER, 1600)
            write (LERR, 1600)
 1600       format (' RMO: The MINXREF parameter must be greater than',
     *                        ' 20% of the maximum offset, MAXOFF')
            n_err = n_err + 1
            min_xref = max_offset / 5.

        elseif (min_xref .gt. .8 * max_offset) then
            write (LER, 1610) 
            write (LERR, 1610)
 1610       format (' RMO: The MINXREF parameter must be less than ',
     *                            '80% of the maximum offset, MAXOFF.')
            n_err = n_err + 1
            min_xref = .8 * max_offset

        endif

C ----> Maximum reference offset for analysis
        if ((max_xref .eq. -1.) .or. parabolic .or. gamma) then
            max_xref = .8 * max_offset

        elseif (max_xref .lt. min_xref) then
            write (LER, 1500)
            write (LERR, 1500)
 1500       format (' RMO: The MAXXREF parameter must be greater than',
     *                        ' the minimum reference offset, MINXREF')
            n_err = n_err + 1
            max_xref = min_xref

        elseif (max_xref .gt. .9 * max_offset) then
            write (LER, 1510) 
            write (LERR, 1510)
 1510       format (' RMO: The MAXXREF parameter must be less than ',
     *                                ' the maximum offset, MAXOFF.')
            n_err = n_err + 1
            max_xref = .9 * max_offset

        endif

C ----> Spatial smoother
        if (n_smash .lt. 1) then
            write (LER, 1700)
            write (LERR, 1700)
 1700       format (' RMO: The NCDP parameter must be greater than 1.')
            n_err = n_err + 1
            n_smash = 1

        endif
        if (mod (n_smash, 2) .eq. 0) n_smash = n_smash + 1
        cdp_delay = (n_smash - 1) / 2

C ----> Number of analysis reference offsets
        if (.not. parabolic .AND. .not. gamma) then
            if (n_xref .lt. 0) then
                write (LER, 1900)
                write (LERR, 1900)
 1900           format (' RMO: The NXREF parameter must be greater ',
     *                                                        'than 0.')
                n_err = n_err + 1
                n_xref = 1

            elseif (n_xref .gt. 101) then
                write (LER, 1910) 
                write (LERR, 1910)
 1910           format (' RMO: The NXREF parameter must be less ',
     *                                                      'than 101.')
                n_err = n_err + 1
                n_xref = 101

            endif
        else
            n_xref = 1

        endif

C ----> Stability weight
        if (stable .lt. .5) then
            write (LER, 2000)
            write (LERR, 2000)
 2000       format (' RMO: The STABL parameter must be greater than .5')
            n_err = n_err + 1
            stable = .5

        elseif (stable .gt. 1.) then
            write (LER, 2010) 
            write (LERR, 2010)
 2010       format (' RMO: The STABL parameter must be less than 1.')
            n_err = n_err + 1
            stable = 1.

        endif

C ----> Maxshift for inside the reference offset
        if ((t_short .eq. -1.) .or. parabolic .or. gamma) then
            t_short = 4. * dt_ms
            if (gamma) t_short = 4. * dz

        elseif (t_short .lt. dt_ms) then
            write (LER, 2200)
            write (LERR, 2200)
 2200       format (' RMO: The TSHORT parameter must be greater than',
     *                                                    ' one sample')
            n_err = n_err + 1
            t_short = dt_ms
            if (gamma) t_short = dz

        elseif (t_short .gt. (max_shift * dt_ms / 6.)) then
            write (LER, 2210) 
            write (LERR, 2210)
 2210       format (' RMO: The TSHORT parameter must be less than ',
     *                                                   ' DELTAT / 6')
            n_err = n_err + 1
            t_short = max_shift * dt_ms / 6.
            if (gamma) t_short = max_shift * dz / 6.

        endif
        if (gamma) then
           t_short = t_short / dz
        else
           t_short = t_short / dt_ms
        endif

C ----> Time smoother length
        if (wind .eq. -1.) then
            wind = 40. * dt_ms

        elseif (wind .lt. (10. * dt_ms)) then
            write (LER, 2300)
            write (LERR, 2300)
 2300       format (' RMO: The WIND parameter must be greater than',
     *                                                   ' ten samples')
            n_err = n_err + 1
            wind = 10. * dt_ms

        elseif (wind .gt. (max_time / 4.)) then
            write (LER, 2310) 
            write (LERR, 2310)
 2310       format (' RMO: The WIND parameter must be less than ',
     *                                              ' trace length / 4')
            n_err = n_err + 1
            wind = max_time / 4.

        endif
        half_wlen = nint (wind / (2. * dt_ms))
        w_len = 2 * half_wlen

C ----> Verbose printout, first offset
        if (first_off .eq. -1.) then
            first_off = max_offset / 10.

        elseif (first_off .lt. 0.) then
            write (LER, 1200)
            write (LERR, 1200)
 1200       format (' RMO: The FOFF parameter must be greater than 0.')
            n_err = n_err + 1
            first_off = 0.

        elseif (first_off .gt. max_offset) then
            write (LER, 1210) 
            write (LERR, 1210)
 1210       format (' RMO: The FOFF parameter must be less than ',
     *                                   ' the maximum offset, MAXOFF.')
            n_err = n_err + 1
            first_off = max_offset

        endif

C ----> Verbose printout, last offset
        if (last_off .eq. -1.) then
            last_off = max_offset

        elseif (last_off .lt. first_off) then
            write (LER, 1300)
            write (LERR, 1300)
 1300       format (' RMO: The LOFF parameter must be greater than',
     *                        ' the first offset, FOFF')
            n_err = n_err + 1
            last_off = 0.

        elseif (last_off .gt. max_offset) then
            write (LER, 1310) 
            write (LERR, 1310)
 1310       format (' RMO: The LOFF parameter must be less than ',
     *                                ' the maximum offset, MAXOFF.')
            n_err = n_err + 1
            last_off = max_offset

        endif

C ----> Number of offset tables for verbose print out
        if (n_off .lt. 1) then
            write (LER, 1800)
            write (LERR, 1800)
 1800       format (' RMO: The NOFF parameter must be greater than 0.')
            n_err = n_err + 1
            n_off = 1

        elseif (n_off .gt. 10) then
            write (LER, 1810) 
            write (LERR, 1810)
 1810       format (' RMO: The NOFF parameter must be less than 10.')
            n_err = n_err + 1
            n_off = 10

        endif

C ----> Verbose printout: CDP increment
        if (cdp_inc .lt. 1) then
            write (LER, 2400)
            write (LERR, 2400)
 2400       format(' RMO: The CDPINC parameter must be greater than 1.')
            n_err = n_err + 1
            cdp_inc = 1

        endif

C ----> Verbose printout time increment
        if (delta_t .eq. -1.) then     ! Default
            t_inc = w_len

        elseif (delta_t .lt. dt_ms) then
            write (LER, 2100)
            write (LERR, 2100)
 2100       format (' RMO: The TINC parameter must be greater than',
     *                                                    ' one sample')
            n_err = n_err + 1
            t_inc = 1

        else
            t_inc = nint (delta_t / dt_ms)

        endif

        call savew (itr, 'MxTrOf', ifix(max_offset), LINHED)

C ***** Auxiliary QC outputs *******************************************

c ----> Open QC semblance output file (parab or gamma)
c ----> Open QC semblance output file (parab or gamma)
c       but don't write out LH yet because in rmo_init we might change
c       the number of scans for gamma option if loshift or hishift
c       is restrictive enough
        if (qctap .ne. ' ' .AND. (parabolic .or. gamma)) then
            call getln (luqc, qctap, 'w', -1)
            if (luqc .gt. 1) then
               nqcrec = n_cdp / cdp_inc
               if (nqcrec .eq. 0) nqcrec = 1
               call savew (qtr, 'NumTrc', nscans, LINHED)
               call savew (qtr, 'NumRec', nqcrec, LINHED)
c              call wrtape (luqc, qtr, lbyout)
               QC = .true.
            else
               write(LERR,*)'WARNING in rmo:'
               write(LERR,*)'Could not open QC data set ',qctap
               QC = .false.
            endif
        endif


c ----> Open alpha (curvature) parameter file
	if (a_in) then

c ----> For alpha disk file
           if (atap .ne. ' ') then

              afile = .true.
              call getln (lu_alpha, atap, 'r', -1) ! try to read a LH
              write(LERR,*)'alpha/gamma option:'
              write(LERR,*)'alpha/gamma file name= ',atap
              write(LERR,*)'alpha/gamma file unit= ',lu_alpha
              call rtape (lu_alpha, itr, lbytes)
              if (lbytes .eq. 0) then  ! if no LH then forward alpha
                 write(LERR,*)'WARNING in rmo:  alpha/gamma option:'
                 write(LERR,*)'No LH detected; assuming forward run'
                 write(LER ,*)'WARNING in rmo:  alpha/gamma option:'
                 write(LER ,*)'No LH detected; assuming forward run'
                 call lbclos (lu_alpha)
                 call getln (luout, atap, 'w', -1)
                 call savew (itr, 'CrwNam', 'RMOFOR', LINHED)
                 call savew (itr, 'NumTrc', 1, LINHED)
                 call wrtape (luout, itr, lbyout)
                 lu_alpha = -1
                 forward = .false.
              else
                 call saver (itr, 'CrwNam', ident, LINHED)
                 write(LERR,*)'alpha/gamma option:'
                 write(LERR,*)'Detecting for status LH word'
                 if (ident .eq. 'RMOFOR') then
                    write(LERR,*)'word detected: we in application mode'
                    forward = .true.
		    call getln (luout, otap, 'w', 1)
		    call wrtape (luout, str, lbyout)
                 else
                    write(LERR,*)'FATAL ERROR in rmo: alpha/gamma optn:'
                    write(LERR,*)'Doesnt appear to be alpha/gamma data'
                    write(LER ,*)'FATAL ERROR in rmo: alpha/gamma optn:'
                    write(LER ,*)'Doesnt appear to be alpha/gamma data'
                    call ccexit (666)
                 endif
              endif
                 
c ----> For alpha ikp socket
           elseif (atap .eq. ' ' .AND. .not.IKP) then

              write(LERR,*)'alpha/gamma option:'
              write(LERR,*)'No alpha/gamma file name detected --'
              write(LERR,*)'Assume alpha/gamma generation mode'
              call getln (luout, otap, 'w', 1)
              call savew (itr, 'CrwNam', 'RMOFOR', LINHED)
              call savew (itr, 'NumTrc', 1, LINHED)
              call wrtape (luout, itr, lbyout)
	      forward = .false.

c ----> For alpha ikp socket
           elseif (atap .eq. ' ' .AND. IKP) then

              write(LERR,*)'alpha/gamma option:'
              write(LERR,*)'Running in ikp'
              call pipchk (pipe, status)
              call sisfdfit (lu_alpha, pipe)
              call rtape (lu_alpha, itr, lbytes)
              if (lbytes .eq. 0) then  ! if no LH then forward alpha
                 write(LERR,*)'WARNING in rmo:  alpha/gamma option/ikp:'
                 write(LERR,*)'No LH detected; assuming forward run'
                 write(LER ,*)'WARNING in rmo:  alpha/gamma option:'
                 write(LER ,*)'No LH detected; assuming forward run'
                 call getln (luout, otap, 'w', 1)
                 call savew (itr, 'CrwNam', 'RMOFOR', LINHED)
                 call savew (itr, 'NumTrc', 1, LINHED)
                 call wrtape (luout, itr, lbyout)
              else
                 call saver (itr, 'CrwNam', ident, LINHED)
                 if (ident .eq. 'RMOFOR') then
                    write(LERR,*)'alpha/gamma option/ikp:'
                    write(LERR,*)'Read alphas/gammas and apply to cdps'
                    forward = .true.
                    call getln (luout, otap, 'w', 1)
                    call wrtape (luout, str, lbyout)

                 else
                    write(LERR,*)'FATAL ERROR in rmo: alpha/gamma optn:'
                    write(LERR,*)'Doesnt appear to be alpha/gamma data'
                    write(LER ,*)'FATAL ERROR in rmo: alpha option:'
                    write(LER ,*)'Doesnt appear to be alpha/gamma data'
                    call ccexit (666)
                 endif
              endif

           endif

	else

c Default rmo logic
c ----> Open output file
           call getln (luout, otap, 'w', 1)
           call wrtape (luout, itr, lbyout)   ! Save o/p history and line header

           call savew (itr, 'NumTrc', 1, LINHED)       ! Set QC output fold to 1
           afile = .false.
           if (atap .ne. ' ') then
               afile = .true.
               call getln (lu_alpha, atap, 'w', 1)

c add line header entry so that dataset is recognized as gammas on the way back
               call savew (itr, 'CrwNam', 'RMOFOR', LINHED)

               call wrtape (lu_alpha, itr, lbyout)   ! Save history and line hdr
           endif

           forward = .true.

	endif

c ----> Open reference offset parameter file
        xfile = .false.
        if (xtap .ne. ' ') then
            xfile = .true.
            call getln (lu_xref, xtap, 'w', 1)
            call wrtape (lu_xref, itr, lbyout)   ! Save history and line hdr
        endif

C **********************************************************************
C ***** Printout *******************************************************
C **********************************************************************

c       verbose output of all pertinent information before processing begins
        call verbal (ntap, otap, atap, xtap, iform, max_shift, 
     *               t_short, stable, lo_shift, hi_shift, qctap)

        iplim = nint (.01 * limit * length)
        limit = iplim

C **********************************************************************
C **********************************************************************
C **********************************************************************

        if (n_err .gt. 0) stop        ! Abort on error


        return
        end








C***********************************************************************
C                                                                      *
c                 copyright 2001, Amoco Production Company             *
c                             All Rights Reserved                      *
c                     an affiliate of BP America Inc.                  *
C***********************************************************************

        subroutine rmo_init (map, xrefs, alpha, weights, lo_shift,
     *                        hi_shift, max_shift, t_short, stable, vfi,
     *                        qtr)
        implicit none

C-FUNCTION:
C       INIT precomputes some analysis arrays
C
C-INCLUDE:
#include <f77/lhdrsz.h>
#include <f77/sisdef.h>
#include <save_defs.h> 
#include <rmocb.fin>
c    ! Defines: LER, LERR, LUIN, LUOUT, LUPPRT, etc.
#include <f77/iounit.h>

C-ARGUMENTS:
        real map(1:*)     ! Super sample time index array (Output)
        real xrefs(1:*)   ! Reference offsets for analysis (Output)
        real alpha(1:*)   ! Curvature parameters for analysis (Output)
        real weights(1:*) ! Stability weights for analysis (Output)
        real lo_shift     ! Minimum allowed parabolic shift at max_offset (I/P)
        real hi_shift     ! Maximum allowed parabolic shift at max_offset (I/P)
        real max_shift    ! Maximum allowed shift at max_offset (Input)
        real t_short      ! Max allowed shift at offsets less than xref (Input)
        real stable       ! Stability weight, a user parameter (Input)
        real vfi(1:*)     ! Sample interpolation coefficients (Output)

        integer  qtr(SZLNHD)

C-DESCRIPTION:
C       This subroutine handles a bunch of totally unrelated initializations.
C
C-VARIABLES:
        integer i_samp    ! Ith sample
        integer i_anl     ! Ith analysis
        integer i_xref    ! Ith reference offset
        integer i_shift   ! Ith time shift
        integer n_alpha   ! Number of alpha values/analyses for this xref
        integer lo_n_shifts  ! Number of parabolic overcorrections to search
        integer hi_n_shifts  ! Number of parabolic undercorrections to search
        integer icount
        integer lbyout

        real max_alpha    ! Maximum alpha coefficient for a reference offset
        real alpha1       ! Max moveout coefficient based on max_shift
        real alpha2       ! Max moveout coefficient based on t_short
        real xrefxref     ! Reference offset squared
        real fraction     ! Interpolation fraction
        real ratio
        real shift_limit  ! Maximum shift at the extreme offset
        real shift        ! Shift (in samples) at max offset
        real lo_shift0, hi_shift0  ! save shift cmd line shift limits
        real xi, xx, xy, xii, xs
        real shiftneg, shiftpos, shiftmax, z0, zdel, zmax, gam, gam2

        real work1(1:n_coeff)   ! Work array
        real work2(1:n_coeff)   ! Work array
        real work3(1:n_coeff)   ! Work array

C-DATA:
C-REVISED:   8-Nov-1996 by Steve Markley  - Initial version
C       Modified from DISCO RMO module.
C-REVISED: February 1997        by Steve Markley
C       Added parabolic option. Added auxiliary QC outputs.

C **********************************************************************
C 3456789012345678901234567890123456789012345678901234567890123456789012
C        1         2         3         4         5         6         7


C **********************************************************************
C ***** Initialize sample interpolation coefficient tables *************
C **********************************************************************

c---
c  build sinc filter only: work arrays are dummies
c  after this call coeffs are stored in vfi(-n_coeff/2+1:n_coeff/2,stride)
c  iflag set to 0 in oversampt, ready for application mode
c---
        iflag = 1
        call oversampt (work1, length, work2, super_length, stride,
     1                  vfi, n_coeff, iflag)


C **********************************************************************
C ***** Analysis super sample stretch map ******************************
C **********************************************************************

        do i_samp = 0, super_length - 1
            map(1 + i_samp) = 1. + real(i_samp) / real(stride)

        enddo


C **********************************************************************
C ***** Normal 4th order residual moveout tables ***********************
C **********************************************************************

        if (.not. parabolic .AND. .not. gamma) then

C --------> Analysis alpha and xref tables
            i_anl = 1
            do i_xref = 0, n_xref - 2
                xrefxref = min_xref + real(i_xref) * 
     *                       (max_xref - min_xref) / real (n_xref - 1)

                xrefxref = xrefxref * xrefxref
                ratio = max_offset**2 * (1. - max_offset**2 / xrefxref)
                alpha1 = -max_shift / ratio
                alpha2 = 4. * t_short / xrefxref
                max_alpha = min (alpha1, alpha2)
                shift_limit = max_alpha * ratio
                n_alpha = nint (abs (shift_limit * bandw))

                do i_shift = 1, n_alpha
                    xrefs(i_anl) = xrefxref
                    fraction = real (i_shift) / real (n_alpha)
                    alpha(i_anl) = max_alpha * fraction
                    weights(i_anl) = 1. + (stable - 1.) * fraction
                    i_anl = i_anl + 1

                enddo
            enddo

C --------> Last reference offset (additional analysis at alpha=0)
            xrefxref = max_xref * max_xref
            ratio = max_offset**2 * (1. - max_offset**2 / xrefxref)
            alpha1 = -max_shift / ratio
            alpha2 = 4. * t_short / xrefxref
            max_alpha = min (alpha1, alpha2)
            shift_limit = max_alpha * ratio
            n_alpha = nint (abs (shift_limit * bandw)) + 1

            do i_shift = 1, n_alpha
                xrefs(i_anl) = xrefxref
                fraction = real (i_shift - 1) / real (n_alpha - 1)
                alpha(i_anl) = max_alpha * fraction
                weights(i_anl) = 1. + (stable - 1.) * fraction
                i_anl = i_anl + 1

            enddo

            n_anl = i_anl - 1

 
C **********************************************************************
C ***** Parabolic residual moveout tables ******************************
C **********************************************************************
 
        elseif (parabolic) then
            lo_n_shifts = nint (lo_shift * bandw)
            hi_n_shifts = nint (hi_shift * bandw)
            n_anl = lo_n_shifts + hi_n_shifts + 1
 
            shift_limit = max (lo_shift, hi_shift)
 
            do i_anl = 1, n_anl
                shift = real (i_anl - lo_n_shifts - 1) / bandw
                alpha(i_anl) = shift / max_offset**2
                weights(i_anl) =
     *                    1. + (stable - 1.) * abs (shift) / shift_limit
            enddo
c     write(0,*)(weights(i_anl),i_anl=1,n_anl)

c   currently we don't change the number of scans for the parab option but
c   set the QC (if any) LH here in order to have code symmetric to the
c   gamma option below
            if (QC) then
               call savew (qtr, 'NumTrc', nscans, LINHED)
               call wrtape (luqc, qtr, lbyout)
            endif


C **********************************************************************
C ***** Gamma residual moveout tables ******************************
C **********************************************************************

        elseif (gamma) then

            write(LERR,*)'Input gamma loshift= ',lo_shift
            write(LERR,*)'Input gamma hishift= ',hi_shift
c     write(0,*)'loshift= ',lo_shift
c     write(0,*)'hishift= ',hi_shift
            lo_shift = lo_shift * lo_shift
            hi_shift = hi_shift * hi_shift
            lo_shift0 = lo_shift
            hi_shift0 = hi_shift
            shiftneg = abs(lo_shift - 1.)
            shiftpos = abs(hi_shift - 1.)
            shiftmax = amax1 (shiftneg, shiftpos)
c     write(0,*)'shift= ',shiftneg,shiftpos,shiftmax
            z0 = float( 100 * int(float(length)/100.) )
            zmax = sqrt(z0*z0 + shiftmax * max_offset2/dzdz)
c     write(0,*)'limit,length,z0,zmax= ',limit,length,z0,zmax
            zdel = zmax - z0
            if (abs(zdel) .gt. limit) then
                gam2 = ((limit+z0)**2 - z0*z0)/(max_offset2/dzdz) + 1.
                gam  = sqrt (gam2)
                if (zmax .ge. z0) then
                   write(LERR,*)'resetting hishift ',gam
                   hi_shift = gam2
                else
                   write(LERR,*)'resetting loshift ',gam
                   lo_shift = gam2
                endif
            endif
c     write(0,*)'loshift= ',lo_shift
c     write(0,*)'hishift= ',hi_shift
            dgamma = (hi_shift - lo_shift) / float(nscans-1)
            n_anl = nscans

            shift_limit = max (1.-lo_shift, hi_shift-1.)

            icount = 0
            do i_anl = 1, n_anl
                shift = (lo_shift + float(i_anl-1) * dgamma)
                if (shift .ge. lo_shift0 .AND. shift .le. hi_shift0) 
     :          then
c     write(0,*)icount,lo_shift0,shift,hi_shift0
                    icount = icount + 1
                    shift = shift - 1.0
                    alpha(icount) = shift
                    weights(icount) = 1. + (stable - 1.) * abs (shift) 
     *                               / shift_limit
                endif
            enddo

            if (icount .lt. nscans/2 .OR. icount .lt. 11) then
               write(LERR,*)'FATAL ERROR in rmo gamma option (rmoinit):'
               write(LERR,*)'Your range of -loshift[] and/or -hishift[]'
               write(LERR,*)'is too narrow and will result in only'
               write(LERR,*)nscans,' scans. Expand the range and rerun'
               write(LER ,*)'FATAL ERROR in rmo gamma option (rmoinit):'
               write(LER ,*)'Your range of -loshift[] and/or -hishift[]'
               write(LER ,*)'is too narrow and will result in only'
               write(LER ,*)nscans,' scans. Expand the range and rerun'
               call ccexit (666)
            endif

            n_anl  = icount
            nscans = icount
c     write(0,*)(weights(i_anl),i_anl=1,n_anl)
c  we may have changed the number of scans so this affects the QC output
c  (if any)
            if (QC) then
               call savew (qtr, 'NumTrc', nscans, LINHED)
               call wrtape (luqc, qtr, lbyout)
            endif


        endif


C **********************************************************************
C ***** Optional parameter print out ***********************************
C **********************************************************************

            if (.not. parabolic .AND. .not. gamma) then
                write (LERR, 8050) min_xref, max_xref, max_shift,
     *                                                           t_short
 8050           format (//' RMO analysis parameters:'
     *              //' Values of alpha, xref, and stability weights'
     *              /'         minimum reference offset:    ', f12.3,
     *              /'         maximum reference offset:    ', f12.3,
     *              /'         maximum shift (samples):     ', f12.4,
     *              /'         short offset shift (samples):', f12.4,
     *              //'   Ith anlysis        alpha*e6        xref    ',
     *                '        weight    shift at max offset (ms)')

                do i_anl = 1, n_anl
                    write (LERR, 8000) i_anl, 1.E6 * alpha(i_anl),
     *                   sqrt (xrefs(i_anl)), weights(i_anl), 
     *                   alpha(i_anl) * max_offset**2 *
     *                   dt_ms * (1. - max_offset**2 / xrefs(i_anl))

 8000               format (i10, f20.8, f13.2, f18.6, f12.3)
                enddo

            elseif (parabolic) then         ! Else parabolic analysis ...
                write (LERR, 8150) -1. * lo_shift, hi_shift
 8150           format (//' RMO analysis parameters:'
     *              //' Values of alpha and stability weights'
     *              /'         Parabolic analysis requested ',
     *              /'         minimum shift (samples):     ', f12.4,
     *              /'         maximum shift (samples):     ', f12.4,
     *              //'   Ith analysis       alpha*e6   ',
     *                '        weight      shift at max offset (ms)')
                do i_anl = 1, n_anl
                    write (LERR, 8100) i_anl,
     *                  1.E6 * alpha(i_anl),
     *                  weights(i_anl),
     *                  alpha(i_anl) * max_offset**2
 8100               format (i10, f20.8, f18.6, f16.3)
                enddo
 
            elseif (gamma) then         ! Else gamma analysis ...
                write (LERR, 8151) sqrt(lo_shift), sqrt(hi_shift),
     *                             iplim, dgamma
 8151           format (//' RMO analysis parameters:'
     *              //' Values of gamma and stability weights'
     *              /'         Gamma analysis requested ',
     *              /'         minimum shift :     ', f12.4,
     *              /'         maximum shift :     ', f12.4,
     *              /'         shift limit   :     ', i5,
     *              /'         delta gamma   :     ', f12.4)
                write (LERR, 8152)
 8152           format (//'   Ith analysis       alpha      ',
     *                '        weight      shift at max offset (samps)')
              do  i_samp = 1, length, 100
                write(LERR,*)' '
                write(LERR,*)'Shifts at time (samps) = ',(i_samp-1)
                xi  = (i_samp-1)
                xii = (i_samp-1)*(i_samp-1)
                do i_anl = 1, n_anl
                   xs = (alpha(i_anl) * max_offset2)/dzdz
                   xx = xii + xs
                   if (xx .ge. 0.0) then
                    xy = sqrt(xx)
                    write (LERR, 8101) i_anl,
     *                  alpha(i_anl),
     *                  weights(i_anl),
     *                  xy,
     *                  xy-xi
 8101               format (i10, f20.8, f18.6, f16.3, f16.3)
                   endif
                enddo
              enddo
              write(LERR,*)' '
              write(LERR,*)' '

            endif
 

        return
        end








C***********************************************************************
C                                                                      *
c                 copyright 2001, Amoco Production Company             *
c                             All Rights Reserved                      *
c                     an affiliate of BP America Inc.                  *
C***********************************************************************

        subroutine verbal (ntap, otap, atap, xtap, iform, max_shift, 
     *                     t_short, stable, lo_shift, hi_shift,
     *                     qctap)
        implicit none

C-FUNCTION:
C       VERBAL writes some informative data to the output listing
C
C-INCLUDE:
#include <rmocb.fin>
c    ! Defines: LER, LERR, LUIN, LUOUT, LUPPRT, etc.
#include <f77/iounit.h>

C-ARGUMENTS:
        character ntap*(*)  ! Input file name (Input)
        character otap*(*)  ! Output file name (Input)
        character atap*(*)  ! QC output file name for alpha values (Input)
        character xtap*(*)  ! QC output file name for reference offsets (Input)
        character qctap*(*) ! QC output file name for semblances

        integer iform     ! Sample format (Input)

        real max_shift    ! Maximum allowed shift at max_offset (Input)
        real t_short      ! Max allowed shift at offsets less than xref (Input)
        real stable       ! Stability weight, a user parameter (Input)
        real lo_shift     ! Maximum parabolic overcorrection in samples (O/P)
        real hi_shift     ! Maximum parabolic undercorrection in samples (O/P)
        real dg           ! local dgamma

C-VARIABLES:
C-DATA:
C-REVISED:   8-Nov-1996 by Steve Markley  - Initial version
C       Modified from DISCO RMO module.
C-REVISED: February 1997        by Steve Markley
C       Added parabolic option. Added auxiliary QC outputs.


C***********************************************************************

        write (LERR,*)' '
        write (LERR,*)' RMO:'
        write (LERR,*)' Input Line Header Parameters'
        write (LERR,*)' '
        write (LERR,*) ' input data set name   =  ', ntap
        write (LERR,*) ' samples per trace     =  ', length
        write (LERR,*) ' traces per record     =  ', maxntr
        write (LERR,*) ' number of records     =  ', n_cdp
        write (LERR,*) ' data format           =  ', iform
        write (LERR,*) ' sample interval (ms)  =  ', dt_ms
        write (LERR,*)' '
        write (LERR,*)' '
        write (LERR,*)' Command Line Parameters '
        write (LERR,*)' '
        write (LERR,*)' Output data set name             = ', otap
        write (LERR,*)' Maximum input offset             = ', max_offset
        write (LERR,*)' Max far offset shift (ms)        = ', max_shift 
     *                                                           * dt_ms
        write (LERR,*)' Time smoother (ms)               = ', 
     *                                              real (w_len) * dt_ms
        if (.not. parabolic .AND. .not. gamma) then
            write (LERR,*)' Minimum reference offset         = ',
     *                                                          min_xref
            write (LERR,*)' Maximum reference offset         = ',
     *                                                          max_xref
            write (LERR,*)' Number of reference offsets      = ', n_xref
            write (LERR,*)' Number of CDPs to smooth results = ',n_smash
            write (LERR,*)' Max shift on short traces (ms)   = ',t_short 
     *                                                           * dt_ms
        elseif (parabolic) then
            write (LERR,*)' Parabolic Analysis requested'
            write (LERR,*)' Most negative shift at max offset = ',
     *                                                 -lo_shift * dt_ms
            write (LERR,*)' Most positive shift at max offset = ',
     *                                                  hi_shift * dt_ms
            write (LERR,*)' Number of CDPs to smooth results = ',n_smash
        elseif (gamma) then
            dg = (hi_shift - lo_shift) / float(nscans-1)
            write (LERR,*)' Gamma Analysis requested'
            write (LERR,*)' Smallest gamma = ',lo_shift
            write (LERR,*)' Geatest gamma  = ',hi_shift
            write (LERR,*)' Delta gamma    = ',dg
            write (LERR,*)' Number scans   = ',nscans
            write (LERR,*)' Number of CDPs to smooth results = ',n_smash
            write (LERR,*)' Max sample shift = ',nint(.01*limit*length)
        endif
        write (LERR,*)' Data bandwidth fraction          = ', bandw
        write (LERR,*)' Stability weight                 = ', stable
        if (verbose) then
            write (LERR, *)' Verbose printout requested'
            write (LERR, *)'    first offset            = ', first_off
            write (LERR, *)'    last offset             = ', last_off
            write (LERR, *)'    number of offsets       = ', n_off
            write (LERR, *)'    time increment (ms)     = ', 
     *                                                     t_inc * dt_ms
            write (LERR, *)'    first CDP               = ', start_cdp
            write (LERR, *)'    CDP increment           = ', cdp_inc

        endif
        if (afile) write (LERR,*)
     *                     ' QC auxiliary output for alphas   = ', atap
        if (QC) write (LERR,*)
     *                     ' QC semblance output = ',qctap
        if (xfile) write (LERR,*)
     *                     ' QC auxiliary output for offsets  = ', xtap
        write (LERR,*)' '
        write (LERR,*)'========================================== '
        write (LERR,*)' '

 
        return
        end








C***********************************************************************
C                                                                      *
c                 copyright 2001, Amoco Production Company             *
c                             All Rights Reserved                      *
c                     an affiliate of BP America Inc.                  *
C***********************************************************************

        subroutine proc (thdr, cdp, Sum, vfi, map, xrefs, alpha, 
     *                weights, Work1, Work2, Work3, Work4, Work5, Work6)
        implicit none

C-FUNCTION:
C       PROC analyzes and processes records in the input dataset.
C
C-INCLUDE:
#include <f77/lhdrsz.h>
#include <f77/sisdef.h>
#include <save_defs.h>
#include <rmocb.fin>
#include <f77/iounit.h>

C-ARGUMENTS:
        integer thdr(1:thdrlen, 1:maxntr, 1:*)
        real cdp(1:length, 1:maxntr, 1:*)
        real Sum(1:length, 1:n_anl, 1:n_smash)
        real vfi(1:*)        ! Sample interpolation coefficients (Input)
        real map(1:*)        ! Super sample time index array (Input)
        real xrefs(1:*)      ! Reference offsets for analysis (Input)
        real alpha(1:*)      ! Curvature parameters for analysis (Input)
        real weights(1:*)    ! Stability weights for analysis (Input)
        real Work1(1:*)
        real Work2(1:*)
        real Work3(1:*)
        real Work4(1:*)
        real Work5(1:*)
        real Work6(1:*)

C-VARIABLES:
        integer i_tr         ! Ith trace
        integer ip_cdp       ! Address in cylydrical buffer of current input cdp
        integer op_cdp       ! Address in cylydrical buffer of output cdp
        integer kx_cdp       ! CDP index in Sum buffer
        integer nbytes
        integer obytes
        integer itr(SZLNHD)
        integer atr(SZLNHD)
        integer qtr(SZLNHD)


C-DATA:
C-REVISED:   8-Nov-1996 by Steve Markley  - Initial version
C       Modified from DISCO RMO module.
C-REVISED: March 1997        by Steve Markley
C       Added parabolic option. Added auxiliary QC outputs. Added feature
C       to stabilize results for low fold taper regions at beginning and
C       end of the line.

c-------------
c   if a_in = T (2 step process) then
c      forward = F  (step 1)
c      forward = T  (step 2)
c-------------

        obytes = SZTRHD + length * SZSMPD


        irec = 1
C **********************************************************************
C ***** Short line *****************************************************
C **********************************************************************

c---prg
c  June 4, 2001
c  Kenny G found a bug that cause the short line logic to be executed
c  if you input a single record data set. This erroneously resulted from
c  a bogus special case comparision with cdp_delay+1
c---prg

        if (n_cdp .le. cdp_delay+1 .AND. n_cdp .gt. 1) then

            DO i_cdp = 1, n_cdp

                irec = irec + 1
C ------------> Read all the traces in an input record
                DO i_tr = 1, maxntr

                    nbytes = 0
                    call rtape (luin, itr, nbytes)

                    if (nbytes .eq. 0) then     ! if end of data encountered 
                                                !      (nbytes=0) then bail out
                        write (LERR, 1000) i_cdp, i_tr
 1000                   format (/ ' RMO: Premature EOF on input at:'
     *                               '  rec= ', i8 ,'  trace= ', i5 //
     *                               ' RMO: ABNORMAL Termination')
                        write (LER, *) ' RMO: ABNORMAL Termination'
                        call lbclos (luin)
                        call lbclos (luout)
                        stop
                    endif

                    call vmov (itr(ITHWP1), 1, cdp(1, i_tr, i_cdp), 1, 
     *                                                          length)
                    call vmov (itr, 1, thdr(1, i_tr, i_cdp), 1,thdrlen)

                ENDDO

C ------------> Analyze the current input record
                if (a_in .AND. .not.forward) then
                call rmo_analyze (cdp(1, 1, i_cdp), thdr(1, 1, i_cdp), 
     *                            Sum, map, vfi, xrefs, alpha, weights, 
     *                            Work1, Work2, work3, work6)
                elseif (.not. a_in) then
                call rmo_analyze (cdp(1, 1, i_cdp), thdr(1, 1, i_cdp), 
     *                            Sum, map, vfi, xrefs, alpha, weights, 
     *                            Work1, Work2, work3, work6)
                endif

            ENDDO

            DO i_cdp = 1, n_cdp

C ------------> Select corrections and apply to a CDP
                if (.not. parabolic .AND. .not. gamma) then
                    call rmo_apply (cdp(1, 1, i_cdp), thdr(1, 1, i_cdp), 
     *                              sum(1, 1, i_cdp + 1), vfi, xrefs, 
     *                              alpha, Work1, Work2, Work3, Work4, 
     *                              work5, work6, qtr)

                else

                    if (a_in .AND. forward) then
                       call rtape (lu_alpha, atr, nbytes)
                       if (nbytes .eq. 0) then
                         write(LERR,*)'FATAL ERROR in rmo: alpha opt'
                         write(LERR,*)'Hit end of data. Check length'
                         write(LERR,*)'of alpha data set, # recs, etc'
                         call ccexit (666)
                       endif
                       call vmov (atr(ITHWP1), 1, Work2, 1, length)
                       if (gamma) call toalpha (Work2, length)
                    endif

                    call rmo_p_apply (cdp(1, 1, i_cdp), 
     *                              thdr(1, 1, i_cdp), 
     *                              sum(1, 1, i_cdp + 1), vfi, alpha, 
     *                              Work1, Work2, Work4, work5, qtr)
                endif

                if (a_in) then

C ------------> Write optional auxilliary outputs
                   if (.not. forward) then
                      if (gamma) call togamma (work2, length)
                      call vmov (work2, 1, itr(ITHWP1), 1, length)
                      call vmov (thdr(1, 1, i_cdp), 1, itr, 1, thdrlen)
                      call wrtape (luout, itr, obytes)
                   else
C ------------> Output all the traces in the record
                      do i_tr = 1, maxntr
                          call vmov (cdp(1, i_tr,i_cdp),1,itr(ITHWP1),1, 
     *                                                           length)
                          call vmov (thdr(1,i_tr,i_cdp),1,itr,1,thdrlen)
                          call wrtape (luout, itr, obytes)
                      enddo
                   endif

                else

C Default rmo logic
C ------------> Write optional auxilliary outputs
                  if (afile) then             ! Alpha (Curvature) parameters
                      if (gamma) call togamma (work2, length)
                      call vmov (work2, 1, itr(ITHWP1), 1, length)
                      call vmov (thdr(1, 1, i_cdp), 1, itr, 1, thdrlen)
                      call wrtape (lu_alpha, itr, obytes)
                  endif
                  if (xfile) then             ! Reference offsets
                      call vmov (work3, 1, itr(ITHWP1), 1, length)
                      call vmov (thdr(1, 1, i_cdp), 1, itr, 1, thdrlen)
                      call wrtape (lu_xref, itr, obytes)
                  endif

C ------------> Output all the traces in the record
                  do i_tr = 1, maxntr
                      call vmov (cdp(1, i_tr, i_cdp), 1, itr(ITHWP1), 1, 
     *                                                           length)
                      call vmov (thdr(1, i_tr, i_cdp),1,itr,1,thdrlen)
                      call wrtape (luout, itr, obytes)
 
                  enddo

                endif
            ENDDO

            return

        endif


C **********************************************************************
C ***** Roll-on (Analysis only) ****************************************
C **********************************************************************

        DO i_cdp = 1, cdp_delay

            irec = irec + 1
C --------> Read all the traces in an input record
            DO i_tr = 1, maxntr

                nbytes = 0
                call rtape (luin, itr, nbytes)

                if (nbytes .eq. 0) then     ! if end of data encountered 
                                            !      (nbytes=0) then bail out
                    write (LERR, 1000) i_cdp, i_tr
                    write (LER, *) ' RMO: ABNORMAL Termination'
                    call lbclos (luin)
                    call lbclos (luout)
                    stop
                endif
                if (i_tr .eq. 1) then
                   call saver2(itr,ifmt_LinInd,l_LinInd,ln_LinInd,
     1             linind, TRACEHEADER)
                   call saver2(itr,ifmt_DphInd,l_DphInd,ln_DphInd,
     1             dphind, TRACEHEADER)
                endif

                call vmov (itr(ITHWP1), 1, cdp(1, i_tr, i_cdp), 1, 
     *                                                           length)
                call vmov (itr, 1, thdr(1, i_tr, i_cdp), 1, thdrlen)

            ENDDO

            if (mod(irec,cdp_inc) .eq. 0) then
            write(LERR,*)'Read cdp RecNum, LI, DI: ',irec,linind,dphind
            write(LER ,*)'Read cdp RecNum, LI, DI: ',irec,linind,dphind
            endif

C --------> Analyze the current input record
            if (a_in .AND. .not.forward) then
            call rmo_analyze (cdp(1, 1, i_cdp), thdr(1, 1, i_cdp), 
     *                        Sum, map, vfi, xrefs, alpha, weights, 
     *                        Work1, Work2, work3, work6)
            else
            call rmo_analyze (cdp(1, 1, i_cdp), thdr(1, 1, i_cdp), 
     *                        Sum, map, vfi, xrefs, alpha, weights, 
     *                        Work1, Work2, work3, work6)
            endif

        ENDDO


C **********************************************************************
C ***** Roll-through (Analysis and apply) ******************************
C **********************************************************************


        DO i_cdp = cdp_delay + 1, n_cdp
            ip_cdp = mod (i_cdp - 1, cdp_delay + 1) + 1

            irec = irec + 1
C --------> Read all the traces in an input record
            DO i_tr = 1, maxntr

                nbytes = 0
                call rtape (luin, itr, nbytes)

                if (nbytes .eq. 0) then     ! if end of data encountered 
                                            !      (nbytes=0) then bail out
                    write (LERR, 1000) i_cdp, i_tr
                    write (LER, *) ' RMO: ABNORMAL Termination'
                    call lbclos (luin)
                    call lbclos (luout)
                    stop
                endif
                if (i_tr .eq. 1) then
                   call saver2(itr,ifmt_LinInd,l_LinInd,ln_LinInd,
     1             linind, TRACEHEADER)
                   call saver2(itr,ifmt_DphInd,l_DphInd,ln_DphInd,
     1             dphind, TRACEHEADER)
                endif

                call vmov (itr(ITHWP1), 1, cdp(1, i_tr, ip_cdp), 1, 
     *                                                           length)
                call vmov (itr, 1, thdr(1, i_tr, ip_cdp), 1, thdrlen)

            enddo

            if (mod(irec,cdp_inc) .eq. 0) then
            write(LERR,*)'Read cdp RecNum, LI, DI: ',irec,linind,dphind
            write(LER ,*)'Read cdp RecNum, LI, DI: ',irec,linind,dphind
            endif

C --------> Analyze the current input record
            if (a_in .AND. .not.forward) then
            call rmo_analyze (cdp(1, 1, ip_cdp), thdr(1, 1, ip_cdp), 
     *                        Sum, map, vfi, xrefs, alpha, weights, 
     *                        Work1, Work2, work3, work6)
            elseif (.not. a_in) then
            call rmo_analyze (cdp(1, 1, ip_cdp), thdr(1, 1, ip_cdp), 
     *                        Sum, map, vfi, xrefs, alpha, weights, 
     *                        Work1, Work2, work3, work6)
            endif

C --------> Select corrections and apply to a deflayed CDP
            op_cdp = mod (i_cdp, cdp_delay + 1) + 1
            kx_cdp = mod (i_cdp - cdp_delay, n_smash) + 1

            if (.not. parabolic .AND. .not. gamma) then
                call rmo_apply (cdp(1, 1, op_cdp), thdr(1, 1, op_cdp), 
     *                         sum(1, 1, kx_cdp), vfi, xrefs, alpha, 
     *                         Work1, Work2, Work3, Work4, work5, work6,
     *                         qtr)

            else

                if (a_in .AND. forward) then
                   call rtape (lu_alpha, atr, nbytes)
                   if (nbytes .eq. 0) then
                     write(LERR,*)'FATAL ERROR in rmo: alpha opt'
                     write(LERR,*)'Hit end of data. Check length'
                     write(LERR,*)'of alpha data set, # recs, etc'
                     call ccexit (666)
                   endif
                   call saver2(atr,ifmt_LinInd,l_LinInd,ln_LinInd,
     1             linind, TRACEHEADER)
                   call saver2(atr,ifmt_DphInd,l_DphInd,ln_DphInd,
     1             dphind, TRACEHEADER)
                   write(LERR,*)'Read alpha RecNum, LI, DI: ',
     1             irec,linind,dphind
                   call vmov (atr(ITHWP1), 1, Work2, 1, length)
                   if (gamma) call toalpha (Work2, length)
                endif

                call rmo_p_apply (cdp(1, 1, op_cdp), thdr(1, 1, op_cdp), 
     *                         sum(1, 1, kx_cdp), vfi, alpha, Work1, 
     *                         Work2, Work4, work5, qtr)

            endif

            if (a_in) then

C ------------> Write optional auxilliary outputs
               if (.not. forward) then
                  call vmov (thdr(1, 1, op_cdp), 1, itr, 1, thdrlen)
                  call saver2(itr,ifmt_LinInd,l_LinInd,ln_LinInd,
     1            linind, TRACEHEADER)
                  call saver2(itr,ifmt_DphInd,l_DphInd,ln_DphInd,
     1            dphind, TRACEHEADER)
                  write(LERR,*)'Write alpha RecNum, LI, DI: ',
     1            irec,linind,dphind
                  if (gamma) call togamma (work2, length)
                  call vmov (work2, 1, itr(ITHWP1), 1, length)
                  call wrtape (luout, itr, obytes)
               else
C ------------> Output all the traces in the record
                  call vmov (thdr(1,1,op_cdp),1,itr,1,thdrlen)
                  call saver2(itr,ifmt_LinInd,l_LinInd,ln_LinInd,
     1            linind, TRACEHEADER)
                  call saver2(itr,ifmt_DphInd,l_DphInd,ln_DphInd,
     1            dphind, TRACEHEADER)
                  write(LERR,*)'Write cdp RecNum, LI, DI: ',
     1            irec,linind,dphind
                  call vmov (work2, 1, itr(ITHWP1), 1, length)
                  do i_tr = 1, maxntr
                      call vmov (cdp(1, i_tr,op_cdp),1,itr(ITHWP1),1,
     *                                                       length)
                      call vmov (thdr(1,i_tr,op_cdp),1,itr,1,thdrlen)
                      call wrtape (luout, itr, obytes)
                  enddo
               endif

            else

C Default rmo logic
C ------------> Write optional auxilliary outputs
              if (afile) then             ! Alpha (Curvature) parameters
                  if (gamma) call togamma (work2, length)
                  call vmov (work2, 1, itr(ITHWP1), 1, length)
                  call vmov (thdr(1, 1, op_cdp), 1, itr, 1, thdrlen)
                  call wrtape (lu_alpha, itr, obytes)
              endif
              if (xfile) then             ! Reference offsets
                  call vmov (work3, 1, itr(ITHWP1), 1, length)
                  call vmov (thdr(1, 1, op_cdp), 1, itr, 1, thdrlen)
                  call wrtape (lu_xref, itr, obytes)
              endif
C --------> Output all the traces in the delayed record
              DO i_tr = 1, maxntr
                  call vmov (cdp(1, i_tr, op_cdp), 1, itr(ITHWP1), 1, 
     *                                                           length)
                  call vmov (thdr(1, i_tr, op_cdp), 1, itr, 1, thdrlen)
                  call wrtape (luout, itr, obytes)
              enddo


            endif

        ENDDO


C **********************************************************************
C ***** Roll-off (Apply Only) ******************************************
C **********************************************************************

        DO i_cdp = n_cdp + 1, n_cdp + cdp_delay

            irec = irec + 1
C --------> Select corrections and apply to a deflayed CDP
            op_cdp = mod (i_cdp, cdp_delay + 1) + 1
            kx_cdp = mod (i_cdp - cdp_delay, n_smash) + 1

            if (.not. parabolic .AND. .not. gamma) then
                call rmo_apply (cdp(1, 1, op_cdp), thdr(1, 1, op_cdp), 
     *                         sum(1, 1, kx_cdp), vfi, xrefs, alpha, 
     *                         Work1, Work2, Work3, Work4, work5, work6,
     *                         qtr)

            else

                if (a_in .AND. forward) then
                   call rtape (lu_alpha, atr, nbytes)
                   if (nbytes .eq. 0) then
                     write(LERR,*)'FATAL ERROR in rmo: alpha opt'
                     write(LERR,*)'Hit end of data. Check length'
                     write(LERR,*)'of alpha data set, # recs, etc'
                     call ccexit (666)
                   endif
                   call vmov (atr(ITHWP1), 1, Work2, 1, length)
                   if (gamma) call toalpha (Work2, length)
                endif
                call saver2(atr,ifmt_LinInd,l_LinInd,ln_LinInd,
     1          linind, TRACEHEADER)
                call saver2(atr,ifmt_DphInd,l_DphInd,ln_DphInd,
     1          dphind, TRACEHEADER)

                if (mod(irec,cdp_inc) .eq. 0) then
                write(LERR,*)'Read alpha RecNum, LI, DI: ',
     1          irec,linind,dphind
                write(LER ,*)'Read alpha RecNum, LI, DI: ',
     1          irec,linind,dphind
                endif

                call rmo_p_apply (cdp(1, 1, op_cdp), thdr(1, 1, op_cdp), 
     *                         sum(1, 1, kx_cdp), vfi, alpha, 
     *                         Work1, Work2, Work4, work5, qtr)
            endif

            if (a_in) then

C ------------> Write optional alpha output
               if (.not. forward) then
                  call vmov (thdr(1,1,op_cdp),1,itr,1,thdrlen)
                  call saver2(itr,ifmt_LinInd,l_LinInd,ln_LinInd,
     1            linind, TRACEHEADER)
                  call saver2(itr,ifmt_DphInd,l_DphInd,ln_DphInd,
     1            dphind, TRACEHEADER)
                  write(LERR,*)'Write alpha cdp RecNum, LI, DI: ',
     1            irec,linind,dphind
                  if (gamma) call togamma (work2, length)
                  call vmov (work2, 1, itr(ITHWP1), 1, length)
                  call vmov (thdr(1, 1, op_cdp), 1, itr, 1, thdrlen)
                  call wrtape (luout, itr, obytes)
               else
                  call vmov (thdr(1,1,op_cdp),1,itr,1,thdrlen)
                  call saver2(itr,ifmt_LinInd,l_LinInd,ln_LinInd,
     1            linind, TRACEHEADER)
                  call saver2(itr,ifmt_DphInd,l_DphInd,ln_DphInd,
     1            dphind, TRACEHEADER)
                  write(LERR,*)'Write cdp RecNum, LI, DI: ',
     1            irec,linind,dphind
C ------------> Output all the traces in the record
                  do i_tr = 1, maxntr
                      call vmov (cdp(1, i_tr,op_cdp),1,itr(ITHWP1),1,
     *                                                       length)
                      call vmov (thdr(1,i_tr,op_cdp),1,itr,1,thdrlen)
                      call wrtape (luout, itr, obytes)
                  enddo
               endif

            else

C Default rmo logic
C ------------> Write optional auxilliary outputs
              if (afile) then             ! Alpha (Curvature) parameters
                  if (gamma) call togamma (work2, length)
                  call vmov (work2, 1, itr(ITHWP1), 1, length)
                  call vmov (thdr(1, 1, op_cdp), 1, itr, 1, thdrlen)
                  call wrtape (lu_alpha, itr, obytes)
              endif
              if (xfile) then             ! Reference offsets
                  call vmov (work3, 1, itr(ITHWP1), 1, length)
                  call vmov (thdr(1, 1, op_cdp), 1, itr, 1, thdrlen)
                  call wrtape (lu_xref, itr, obytes)
              endif

C ------------> Output all the traces in the record
              do i_tr = 1, maxntr
                  call vmov (cdp(1, i_tr, op_cdp), 1, itr(ITHWP1), 1,
     *                                                       length)
                  call vmov (thdr(1, i_tr, op_cdp),1,itr,1,thdrlen)
                  call wrtape (luout, itr, obytes)

              enddo

            endif


        ENDDO


        return
        end


C***********************************************************************
C                                                                      *
c                 copyright 2001, Amoco Production Company             *
c                             All Rights Reserved                      *
c                     an affiliate of BP America Inc.                  *
C***********************************************************************

        subroutine rmo_analyze (cdp, thdr, Sum, map, vfi, xrefs,
     *                      alpha, weights, stack, stretch, work3, sort)
        implicit none

C+ RMO_ANALYZE
C-FUNCTION: Perform trial moveouts and stack for residual moveout analysis
C
C-INCLUDE:
#include <rmocb.fin>
#include <f77/iounit.h>

C-ARGUMENTS:
        real cdp(1:length, 1:*)        ! Input CDP gather (INPUT)
        integer thdr(1:thdrlen, 1:*)   ! Input trace headers (INPUT)
        real sum(1:length, 1:n_anl, 1:n_smash)
                             ! Summed, smashed stacked abs amplitudes (MODIFIED)
        real map(1:*)        ! Super sample time index array (Input)
        real vfi(1:*)        ! Sample interpolation coefficients (Input)
        real xrefs(1:*)      ! Reference offsets for analysis (Input)
        real alpha(1:*)      ! Curvature parameters for analysis (Input)
        real weights(1:*)    ! Stability weights for analysis (Output)
        real stack(1:length, 1:n_anl)  ! Work for stacked traces (MODIFIED)
        real stretch(1:*)    ! Stretched input trace work bugger (Modified)
        real work3(1:*)      ! Work array (Modified)
        real sort(1:*)       ! Work array (Modified)

C-DESCRIPTION:
C       Note that rmo_analyze is somewhat sloppy about sample interpolation.
C       Basically, this subroutine uses a good interpolator to get to
C       about a 16-times super sampled intermediate trace.  For the
C       acual static shift, the nearest sample is used from the 
C       super-sampled (i.e. "stretched") intermediate result.  This
C       algorithm is acceptable only for the analysis.  A more precise
C       scheme is used for the actual output samples.  This cheap, sloppy
C       algorithm is used here, because the precise algorithm chewed up
C       80% of the CPU time!
C       
C-VARIABLES:
        integer i_tr       ! Ith trace 
        integer i_smash    ! Ith CDP in the spatial smoothing window
        integer ix_cdp     ! Index of the first trace in a CDP
        integer i_anl      ! Ith analysis
        integer ip_samp    ! Source sample index from stretched trace
        integer op_samp    ! Destination sample index
        integer first_op   ! First shifted sample to stack
        integer last_op    ! Last shifted sample to stack
        integer live_count ! Count of non-zero traces in the CDP
        integer i,j,k,n    ! Sort variables
        integer DstSgn     ! Signed source to receiver distance
        integer StaCor     ! Static Correction/dead trace flag
        integer cdp_val    ! CDP number of the current gather
        integer ip0
        integer ipd

        real x           ! Source to receiver offset
        real xx          ! Offset squared
        real shift       ! Static time shift in samples
        real tr_length   ! Trace length in samples
        real rstep       ! Stretch rate
        real this_weight ! Stability weight for the current analysis
        real dx          ! Distance (source to receiver) increment
        real ideal_dx    ! Ideal source to recever increment
        real slop        ! Sloppiness estimate for the current CDP gather
        real xp0

C-SAVE:
C-DATA:
C-REVISED:      25-Sept-1996 by Steve Markley - Initial version
C-REVISED: March 1997        by Steve Markley
C       Added parabolic option. 
C       to stabilize results for low fold taper regions at beginning and
C       end of the line.
C++
C **********************************************************************
C 3456789012345678901234567890123456789012345678901234567890123456789012
C        1         2         3         4         5         6         7


C **********************************************************************
C ***** Clear out previous CDP from the sum buffer *********************
C **********************************************************************

        ix_cdp = mod (i_cdp + cdp_delay, n_smash) + 1
        call vfill (0., sum(1, 1, ix_cdp), 1, length*n_anl)


C **********************************************************************
C ***** Compute stacked amplitude for each analysis ********************
C **********************************************************************

        call vfill (0., stack, 1, n_anl*length)

        tr_length = real (length)
        rstep = real (stride)

 
C ***** Normal quartic residual moveout ********************************
 
        live_count = 0
        if (.not. parabolic .AND. .not. gamma) then
            do i_tr = 1, maxntr                     ! For each input trace
                call saver2 (thdr(1, i_tr), ifmt_StaCor, l_StaCor, 
     *                 ln_StaCor, StaCor, TRACEHEADER)    ! Get the trace static

                if (StaCor .ne. 30000) then
                    live_count = live_count + 1

                    call saver2 (thdr(1, i_tr), ifmt_DstSgn, l_DstSgn, 
     *                     ln_DstSgn, DstSgn, TRACEHEADER)
                    x = abs( real (DstSgn) )
                    xx = x * x
                    sort(live_count) = x
C                          ! Save input trace offset in work buffer for sorting

C ----------------> Super sample the trace
                    call oversampt (cdp(1, i_tr), length, stretch,
     1                              super_length, stride, vfi, n_coeff,
     2                              iflag)

C ----------------> Shift and sum the trace for each anlysis
                    do i_anl = 1, n_anl
                        shift = alpha(i_anl) * xx * 
     *                                          (1. - xx / xrefs(i_anl))

                        first_op = max (1, 
     *                             1 + length - int (tr_length + shift))
                        last_op = min (length, int (tr_length - shift))
                        ip_samp = 1 + stride * (first_op - 1) +
     *                                              nint (rstep * shift)

C --------------------> Stack amplitudes from samples nearest to exact time
                        do op_samp = first_op, last_op
                            stack(op_samp, i_anl) = 
     *                          stack(op_samp, i_anl) + stretch(ip_samp)

                            ip_samp = ip_samp + stride

                        enddo
                    enddo
                endif
            enddo
 

C ***** Parabolic residual moveout *************************************
 
        elseif (parabolic) then
            do i_tr = 1, maxntr                     ! For each input trace
                call saver2 (thdr(1, i_tr), ifmt_StaCor, l_StaCor, 
     *                 ln_StaCor, StaCor, TRACEHEADER)    ! Get the trace static

                if (StaCor .ne. 30000) then
                    live_count = live_count + 1

                    call saver2 (thdr(1, i_tr), ifmt_DstSgn, l_DstSgn, 
     *                     ln_DstSgn, DstSgn, TRACEHEADER)
                    x = abs ( real (DstSgn) )
                    xx = x * x
                    sort(live_count) = x
C                          ! Save input trace offset in work buffer for sorting

C ----------------> Super sample the trace
                    call oversampt (cdp(1, i_tr), length, stretch,
     1                              super_length, stride, vfi, n_coeff,
     2                              iflag)

C ----------------> Shift and sum the trace for each anlysis
                    do i_anl = 1, n_anl
                        shift = alpha(i_anl) * xx

                        first_op = max (1, 
     *                             1 + length - int (tr_length + shift))
                        last_op = min (length, int (tr_length - shift))
                        ip_samp = 1 + stride * (first_op - 1) +
     *                                              nint (rstep * shift)

C --------------------> Stack amplitudes from samples nearest to exact time
                        do op_samp = first_op, last_op
                            stack(op_samp, i_anl) = 
     *                          stack(op_samp, i_anl) + stretch(ip_samp)

                            ip_samp = ip_samp + stride

                        enddo
                    enddo
                endif
            enddo
        elseif (gamma) then

            do i_tr = 1, maxntr                     ! For each input trace
                call saver2 (thdr(1, i_tr), ifmt_StaCor, l_StaCor,
     *                 ln_StaCor, StaCor, TRACEHEADER)    ! Get the trace static

                if (StaCor .ne. 30000) then
                    live_count = live_count + 1

                    call saver2 (thdr(1, i_tr), ifmt_DstSgn, l_DstSgn,
     *                     ln_DstSgn, DstSgn, TRACEHEADER)
                    x = .5 * abs ( real (DstSgn) )
                    xx = x * x / dzdz
                    sort(live_count) = x
C                          ! Save input trace offset in work buffer for sorting

C ----------------> Super sample the trace
                    call oversampt (cdp(1, i_tr), length, stretch,
     1                              super_length, stride, vfi, n_coeff,
     2                              iflag)

C ----------------> Shift and sum the trace for each anlysis
                    do i_anl = 1, n_anl
                        shift = alpha(i_anl) * xx

C --------------------> Stack amplitudes from samples nearest to exact time
                        do op_samp = 1, length
                            xp0 = float(op_samp*op_samp) + shift
                            if (xp0 .gt. 0.0) then
                              xp0 = sqrt(xp0)
                              ip0 = nint ( xp0 )
                              ipd = iabs(ip0 - op_samp)
                              if (ip0 .ge. 1 .AND. ip0 .le. length
     *                            .AND. ipd .le. iplim) then
                                  ip_samp = nint (rstep * xp0)
                                  stack(op_samp, i_anl) =
     *                            stack(op_samp, i_anl) + 
     *                            stretch(ip_samp)
                              endif
                            endif
                        enddo
                    enddo
                endif
            enddo

        endif
 
        if (live_count .le. 1) return      ! Do nothing to single fold CDPs


C **********************************************************************
C ***** Evaluate the quality of this CDP *******************************
C **********************************************************************
 
C       There is nothing particularly scientific about this algorithm
C       One need not try to read too much into the value of "slop"
C       that results.  It is simply a crude way of guessing at how
C       good is the input CDP.  A "good" input cdp contains evenly
C       spaced offsets (for aliasing considerations) that span a
C       wide range.  Both near traces and far traces are needed for
C       a good stable analysis.  Near traces, however, seem especially
C       important.
 
C ***** Sort offsets into increasing order (Heap sort) *****************
 
C ----> Start sort
        k = live_count/2 + 1
        n = live_count
1000    continue
            if (k .gt. 1) then
                k = k - 1
                x = sort(k)
            else
                x = sort(n)
                sort(n) = sort(1)
                n = n -1
                if (n .eq. 1) then
                    sort(1) = x
                    goto 2000
                endif
            endif
            i = k
            j = k + k
            do while (j .le. n)
                if (j .lt. n) then
                    if (sort(j) .lt. sort(j+1)) j = j+1
                endif
                if (x .lt. sort(j)) then
                    sort(i) = sort(j)
                    i = j
                    j = j + j
                else
                    j = n + 1
                endif
            enddo
            sort(i) = x

        goto 1000       
2000    continue
 
 
C ***** Evaluate the quality of each offset increment ******************
 
C ----> Determine best possible offset increment
        ideal_dx =  (sort(live_count) - sort(1)) / real (live_count - 1)
        if (ideal_dx .le. 0.) return       ! Discard if one live offset
 
C ----> Loop through all the offsets, penalized traces that are close together
        slop = real (live_count)    
c                            ! Here: slop is approx equal to num unique offsets
        do i_tr = 1, live_count - 1
            dx = sort(i_tr + 1) - sort(i_tr)
 
            if (dx .lt. ideal_dx) then
                slop = slop - ((ideal_dx - dx) / ideal_dx)
 
            endif
        enddo
        if (slop .le. 0.) return       ! Discard bad CDP
        slop = real (live_count) / slop
C                    At this point: slop is ratio of traces to unique offsets
        slop = .5 * (slop - 1.)**2
C                    Penalize very bad distributions but not distributions that
C                    are just a little off (like land data not shot between
C                    groups)
 
C ----> Near trace's contribution  (a long offset near trace is _very_ bad)
        slop = slop + (sort(live_count) / 
     *                                  (sort(live_count) - sort(1)))**2
 
c       Note: "slop" has a value greater than 1.  If the near trace has
C       an offset of zero and each offset is exactly ideal_dx, then
C       slop is 1.  As the offsets become more irregular and the near
C       trace distance grows, slop grows rapidly.
 
C ----> Optionally, print the quality of each input CDP
        call saver2 (thdr(1, 1), ifmt_DphInd, l_DphInd, ln_DphInd, 
     *                                             cdp_val, TRACEHEADER)
        if (verbose) write (LERR, 4000) cdp_val, 1. / slop
 4000   format (' RMO: estimated quality of the offset ',
     *                'distribution for CDP number:', i6, ' is ', f10.5)
 
 
C **********************************************************************
C ***** Sum the absolute value of the stack into each CDP's ************
C ***** analysis buffer ************************************************
C **********************************************************************

        do i_anl = 1, n_anl
            call vabs (stack(1, i_anl), 1, stack(1, i_anl), 1, length)

            this_weight = weights(i_anl) ** slop
 
C                    make the weighting towards zero shift stronger as
C                    the quality of the cdp offset distribution decreases
 
            do i_smash = 1, n_smash
                call vsma (stack(1, i_anl), 1, this_weight,
     *                         sum(1, i_anl, i_smash), 1,
     *                         sum(1, i_anl, i_smash), 1, length)
 
            enddo
        enddo

        return
        end









C***********************************************************************
C                                                                      *
c                 copyright 2001, Amoco Production Company             *
c                             All Rights Reserved                      *
c                     an affiliate of BP America Inc.                  *
C***********************************************************************

        subroutine rmo_apply (cdp, thdr, sum, vfi, xrefs, alpha, 
     *                      smooth, v_alpha, v_xref, time, work5, work6,
     *                      qtr)
        implicit none

C+ RMO_APPLY
C-FUNCTION: Stretch/compress a CDP gather of traces
C-CALLING SEQUENCE:
C       call rmo_apply (cdp, thdr, sum, vfi, xrefs, alpha, 
C    *               smooth, v_alpha, v_xref, time, work5, work6)
C
C-INCLUDE:
#include <rmocb.fin>
#include <f77/iounit.h>
#include <f77/lhdrsz.h>

C-ARGUMENTS:
        real cdp(1:length, 1:*)      ! CDP gather of traces (MODIFIED)
        integer thdr(1:thdrlen, 1:*) ! Trace headers for the CDP gather (INPUT)
        integer qtr (*)
        real sum(1:length, 1:n_anl)  ! Stacked amplitude (MODIFIED)
        real vfi(1:*)                ! Sample interpolation coefficients (I/P)
        real xrefs(1:*)              ! Reference offsets for analysis (I/P)
        real alpha(1:*)              ! Curvature parameters for analysis (I/P)
        real smooth(1:length, 1:n_anl)   ! Smoothed stacked amplitude (MODIFIED)
        real v_alpha(1:length)       ! Alpha as a function of time (Output)
        real v_xref(1:length)        ! Xref as a function of time (Output)
        real time(1:length)          ! Time reference trace (MODIFIED)
        real work5(1:*)              ! Work buffer for traces (MODIFIED)
        real work6(1:*)              ! Work buffer for traces (MODIFIED)

C-DESCRIPTION:
C       Given the raw stacked amplitudes in sum(1:length,1:n_anl),
C       rmo_apply edits, smooths, and peak picks to determine the
C       optimum residual quartic moveout.  The moveout is then applied
C       to the input CDP gather, cdp(1:length,1:maxntr), based upon
C       the offsets defined in the trace headers, thdr(1:thdrlen,1:maxntr).
C       The stretched traces replace the input traces in cdp(1:*,1:*).
C
C-VARIABLES:
        integer i_tr       ! Ith trace 
        integer i_samp     ! Ith sample of a trace
        integer i_anl      ! Ith moveout analysis
        integer i_off      ! Ith offset in the verbose print option
        integer cdp_val    ! CDP number of the current gather
        integer recnum     ! Record count
        integer DstSgn     ! Signed trace distance
        integer StaCor     ! Static correction (dead trace flag)
        integer idist      ! distance
	integer obytes     ! output # bytes/trc

        real xxref         ! Reference offset squared
        real x             ! Source to receiver offset
        real xx            ! Source to receiver offset squared
        real this_alpha    ! Alpha parameter for the current time sample
        real peak_amplitude
        real amplitude
        real t_0
        real fraction
        real delta_t(10)        ! Time shifts for print out
        real amp1, amp2, amp3   ! Alpha/xref values for median edit

C-SAVE:
C-DATA:
C-REVISED:      25-Sept-1996 by Steve Markley - Initial version
C-REVISED: February 1997        by Steve Markley
C       Altered to match DISCO code.
C++
C **********************************************************************
C 3456789012345678901234567890123456789012345678901234567890123456789012
C        1         2         3         4         5         6         7

c       write (*, 7999) i_cdp, thdr(l_RecNum, 1)
c7999   format (//' i_cdp: ', i3, '     ix_cdp: ',i3, '    cdp: ',i8)
c       write (*,8000) (sum(i_samp, 20, ix_cdp), i_samp=1,length)
c8000   format (/' spatially smashed absolute amp stack for i_anl 20',
c    *            9999(/ 5e14.6))

        obytes = SZTRHD + length * SZSMPD

C **********************************************************************
C ***** Temporally smooth **********************************************
C **********************************************************************

        do i_anl = 1, n_anl

C --------> Temporal integration of stacked energy
            sum(1, i_anl) = sum(1, i_anl) * sum(1, i_anl)
            do i_samp = 2, length
                sum(i_samp, i_anl) = sum(i_samp - 1, i_anl) +
     *                        sum(i_samp, i_anl) * sum(i_samp, i_anl)
            enddo

c --------> Difference window edges
            smooth (1 + half_wlen, i_anl) = sum(1 + w_len, i_anl)
            do i_samp = 2 + half_wlen, length - half_wlen
                smooth(i_samp, i_anl) = sum(i_samp + half_wlen, i_anl) -
     *                                sum(i_samp - half_wlen - 1, i_anl)
            enddo
        enddo

        call saver2 (thdr(1, 1), ifmt_DphInd, l_DphInd, 
     *               ln_DphInd, cdp_val, TRACEHEADER)
        if (QC .AND. (mod(irec,cdp_inc) .eq. 0) ) then
           call vmov (thdr(1,1), 1, qtr, 1, thdrlen)
           do  i_anl = 1, n_anl
               idist = alpha(i_anl) / unitsc
               call savew2 (qtr, ifmt_DstSgn, l_DstSgn,
     *                      ln_DstSgn, idist, TRACEHEADER)
               call savew2 (qtr, ifmt_TrcNum, l_TrcNum,
     *                      ln_TrcNum, i_anl, TRACEHEADER)

               call vmov (smooth(1,i_anl), 1, qtr(ITHWP1), 1, length)
               call wrtape (luqc, qtr, obytes)
           enddo
         endif
           
           


C **********************************************************************
C ***** Search for peak amplitudes *************************************
C **********************************************************************

        do i_samp = 1 + half_wlen, length - half_wlen
            peak_amplitude = 0.
            v_alpha(i_samp) = 0.
            v_xref(i_samp) = max_xref * max_xref

            do i_anl = 1, n_anl
                amplitude = smooth (i_samp, i_anl)

                if (amplitude .gt. peak_amplitude) then
                    peak_amplitude = amplitude
                    v_alpha(i_samp) = alpha(i_anl)
                    v_xref(i_samp) = xrefs(i_anl)

                endif
            enddo
        enddo


C **********************************************************************
C ***** Smooth the xref/alpha values ***********************************
C **********************************************************************


C ***** Preliminary edit by 3 sample median ****************************

C ----> Edit alpha values
        do i_samp = 2 + half_wlen, length - half_wlen - 1
            amp1 = v_alpha(i_samp - 1)
            amp2 = v_alpha(i_samp)
            amp3 = v_alpha(i_samp + 1)

            if (amp2 .gt. amp1) then
                if (amp3 .gt. amp2) then
                    work5(i_samp) = amp2
                elseif (amp3 .gt. amp1) then
                    work5(i_samp) = amp3
                else
                    work5(i_samp) = amp1
                endif
            else
                if (amp3 .gt. amp1) then
                    work5(i_samp) = amp1
                elseif (amp3 .gt. amp2) then
                    work5(i_samp) = amp3
                else
                    work5(i_samp) = amp2
                endif
            endif
        enddo
        work5(1 + half_wlen) = work5(2 + half_wlen)
        work5(length - half_wlen) = work5(length - half_wlen - 1)

C ----> Edit xref values
        do i_samp = 2 + half_wlen, length - half_wlen - 1
            amp1 = v_xref(i_samp - 1)
            amp2 = v_xref(i_samp)
            amp3 = v_xref(i_samp + 1)

            if (amp2 .gt. amp1) then
                if (amp3 .gt. amp2) then
                    work6(i_samp) = amp2
                elseif (amp3 .gt. amp1) then
                    work6(i_samp) = amp3
                else
                    work6(i_samp) = amp1
                endif
            else
                if (amp3 .gt. amp1) then
                    work6(i_samp) = amp1
                elseif (amp3 .gt. amp2) then
                    work6(i_samp) = amp3
                else
                    work6(i_samp) = amp2
                endif
            endif
        enddo
        work6(1 + half_wlen) = work6(2 + half_wlen)
        work6(length - half_wlen) = work6(length - half_wlen - 1)


C ***** Running smoother of user-specified length **********************

c ----> Time integration
        do i_samp = 2 + half_wlen, length - half_wlen
            work5(i_samp) = work5(i_samp - 1) + work5(i_samp)
            work6(i_samp) = work6(i_samp - 1) + work6(i_samp)
        enddo

C ----> Difference gate edges and normalize
        v_alpha(1 + w_len) = work5(1 + w_len + half_wlen) / 
     *                                                  real (w_len + 1)
        v_xref(1 + w_len) = work6(1 + w_len + half_wlen) / 
     *                                                  real (w_len + 1)
        do i_samp = 2 + w_len, length - w_len
            v_alpha(i_samp) = (work5(i_samp + half_wlen) - 
     *                         work5(i_samp - half_wlen - 1)) /
     *                                                  real (w_len + 1)
            v_xref(i_samp) = (work6(i_samp + half_wlen) - 
     *                        work6(i_samp - half_wlen - 1)) / 
     *                                                  real (w_len + 1)

        enddo

C ----> Extrapolate top and bottom
        do i_samp = 1, w_len
            fraction = real (i_samp - 1) / real (w_len)
            v_alpha(i_samp) = v_alpha(1 + w_len) * fraction
            v_xref(i_samp) = max_xref**2 + 
     *                      (v_xref(1 + w_len) - max_xref**2) * fraction
        enddo
        do i_samp = length - w_len + 1, length
            fraction = real (length - i_samp) / real (w_len)
            v_alpha(i_samp) = v_alpha(length - w_len) * fraction
            v_xref(i_samp) = max_xref**2 + 
     *                 (v_xref(length - w_len) - max_xref**2) * fraction
        enddo


C **********************************************************************
C ***** Create shift vector and stretch trace **************************
C **********************************************************************

        do i_tr = 1, maxntr
            call saver2 (thdr(1, i_tr), ifmt_StaCor, l_StaCor, 
     *              ln_StaCor, StaCor, TRACEHEADER)       ! Get the trace static

            if (StaCor .ne. 30000) then
                call saver2 (thdr(1, i_tr), ifmt_DstSgn, l_DstSgn, 
     *                                   ln_DstSgn, DstSgn, TRACEHEADER)
                x = abs ( real (DstSgn) )
                xx = x * x

                do i_samp = 1, length
                    t_0 = real (i_samp)
                    time(i_samp) = t_0 + v_alpha(i_samp) * 
     *                                   xx * (1. - xx / v_xref(i_samp))
                enddo

c---
c  use quadratic interpolation to remap input data onto new time grid
c  Note: quad interpolator needs time indexing to start at 0.0
c---
                call vclr (work5, 1, length+2)
                do  i_samp = 1, length
                    time (i_samp) = time (i_samp) - 1.
                enddo
                call vqint (cdp(1, i_tr), length, time, 1, 
     1                      work5, 1, length)
                call vmov (work5, 1, cdp(1, i_tr), 1, length)

            endif
        enddo


C **********************************************************************
C ***** Optional verbose print out *************************************
C **********************************************************************

        if (verbose) then
            call saver2 (thdr(1, 1), ifmt_RecNum, l_RecNum, 
     *              ln_RecNum, recnum, TRACEHEADER)

            call saver2 (thdr(1, 1), ifmt_DphInd, l_DphInd, 
     *              ln_DphInd, cdp_val, TRACEHEADER)

            if (cdp_val .ge. start_cdp) then
                if (mod ((cdp_val - start_cdp), cdp_inc) .eq. 0) then

                    write (LERR, 4000) cdp_val, recnum
 4000               format (//' RMO analysis results for CDP number:',
     *                       i6, '      Record number:', 
     *                       i6, /'  time (ms)  xref    alpha*E6   ',
     *                          '   shifts (ms):')

                    do i_samp = 0, length - 1, t_inc
                        this_alpha = v_alpha(1 + i_samp)
                        xxref = v_xref(1 + i_samp)
                    
                        do i_off = 1, n_off
                            x = first_off + (last_off - first_off) * 
     *                           real (i_off - 1) / real (n_off - 1)
                            xx = x * x

                            delta_t(i_off) = dt_ms * this_alpha * xx * 
     *                                                 (xx / xxref - 1.)

                        enddo

                        write (LERR, 4010) dt_ms * real (i_samp), 
     *                       sqrt (xxref), 1.E6 * this_alpha, 
     *                       (delta_t(i_off), i_off = 1, n_off)

 4010                   format (f8.1, f10.1, f12.7, 10f8.2)

                    enddo
                    write (LERR, 4020)
 4020               format (' ')
                endif
            endif
        endif

        return
        end







C***********************************************************************
C                                                                      *
c                 copyright 2001, Amoco Production Company             *
c                             All Rights Reserved                      *
c                     an affiliate of BP America Inc.                  *
C***********************************************************************

        subroutine rmo_p_apply (cdp, thdr, sum, vfi, alpha, 
     *                      smooth, v_alpha, time, work5, qtr)
        implicit none

C+ RMO_APPLY
C-FUNCTION: Stretch/compress a CDP gather of traces
C-CALLING SEQUENCE:
C       call rmo_apply (cdp, thdr, sum, vfi, alpha, 
C    *                  smooth, v_alpha, time, work5)
C
C-INCLUDE:
#include <rmocb.fin>
#include <f77/iounit.h>
#include <f77/lhdrsz.h>

C-ARGUMENTS:
        real cdp(1:length, 1:*)      ! CDP gather of traces (MODIFIED)
        integer thdr(1:thdrlen, 1:*) ! Trace headers for the CDP gather (INPUT)
	integer qtr (*)
        real sum(1:length, 1:n_anl)  ! Stacked amplitude (MODIFIED)
        real vfi(1:*)                ! Sample interpolation coefficients (I/P)
        real alpha(1:*)              ! Curvature parameters for analysis (I/P)
        real smooth(1:length, 1:n_anl)   ! Smoothed stacked amplitude (MODIFIED)
        real v_alpha(1:length)       ! Alpha as a function of time (Output)
        real time(1:length)          ! Time reference trace (MODIFIED)
        real work5(1:*)              ! Work buffer for traces (MODIFIED)

C-DESCRIPTION:
C       Given the raw stacked amplitudes in sum(1:length,1:n_anl),
C       rmo_apply edits, smooths, and peak picks to determine the
C       optimum residual quartic moveout.  The moveout is then applied
C       to the input CDP gather, cdp(1:length,1:maxntr), based upon
C       the offsets defined in the trace headers, thdr(1:thdrlen,1:maxntr).
C       The stretched traces replace the input traces in cdp(1:*,1:*).
C
C-VARIABLES:
        integer i_tr       ! Ith trace 
        integer i_samp     ! Ith sample of a trace
        integer i_anl      ! Ith moveout analysis
        integer i_off      ! Ith offset in the verbose print option
        integer cdp_val    ! CDP number of the current gather
        integer recnum     ! Record count
        integer DstSgn     ! Signed trace distance
        integer StaCor     ! Static correction (dead trace flag)
	integer idist      ! distance
	integer obytes     ! output # bytes/trc
	integer ipd
	integer peak_loc

        real x             ! Source to receiver offset
        real xx            ! Source to receiver offset squared
        real this_alpha    ! Alpha parameter for the current time sample
        real peak_amplitude
        real amplitude
        real t_0
        real t_xx
        real t_x
        real timx
        real fraction
        real delta_t(10)        ! Time shifts for print out
        real amp1, amp2, amp3   ! Alpha/xref values for median edit
        real c1, c2, c3, px, py
        real slice(1000)

C-SAVE:
C-DATA:
C-REVISED:      25-Sept-1996 by Steve Markley - Initial version
C-REVISED: February 1997        by Steve Markley
C       Coppied and modified from rmo_apply.
C++
C **********************************************************************
C 3456789012345678901234567890123456789012345678901234567890123456789012
C        1         2         3         4         5         6         7

c       write (*, 7999) i_cdp, thdr(l_RecNum, 1)
c7999   format (//' i_cdp: ', i3, '     ix_cdp: ',i3, '    cdp: ',i8)
c       write (*,8000) (sum(i_samp, 20, ix_cdp), i_samp=1,length)
c8000   format (/' spatially smashed absolute amp stack for i_anl 20',
c    *            9999(/ 5e14.6))

      obytes = SZTRHD + length * SZSMPD

      if (a_in .AND. forward) go to 3999

C **********************************************************************
C ***** Temporally smooth **********************************************
C **********************************************************************


        do i_anl = 1, n_anl

            do  i_samp = 1, length
                smooth(i_samp, i_anl) = 0.0
            enddo

C --------> Temporal integration of stacked energy
            sum(1, i_anl) = sum(1, i_anl) * sum(1, i_anl)
            do i_samp = 2, length
                sum(i_samp, i_anl) = sum(i_samp - 1, i_anl) +
     *                        sum(i_samp, i_anl) * sum(i_samp, i_anl)
            enddo

c --------> Difference window edges
            smooth (1 + half_wlen, i_anl) = sum(1 + w_len, i_anl)
            do i_samp = 2 + half_wlen, length - half_wlen
                smooth(i_samp, i_anl) = sum(i_samp + half_wlen, i_anl) -
     *                                sum(i_samp - half_wlen - 1, i_anl)
            enddo
        enddo

        call saver2 (thdr(1, 1), ifmt_DphInd, l_DphInd, 
     *               ln_DphInd, cdp_val, TRACEHEADER)

        if (QC .AND. (mod(irec,cdp_inc) .eq. 0) ) then
           call vmov (thdr(1,1), 1, qtr, 1, thdrlen)
           do  i_anl = 1, n_anl
               idist = alpha(i_anl) / unitsc
               call savew2 (qtr, ifmt_DstSgn, l_DstSgn,
     *                      ln_DstSgn, idist, TRACEHEADER)
               call savew2 (qtr, ifmt_TrcNum, l_TrcNum,
     *                      ln_TrcNum, i_anl, TRACEHEADER)

               call vmov (smooth(1,i_anl), 1, qtr(ITHWP1), 1, length)
               call wrtape (luqc, qtr, obytes)
           enddo
         endif


C **********************************************************************
C ***** Search for peak amplitudes *************************************
C **********************************************************************

        do i_samp = 1 + half_wlen, length - half_wlen
            peak_amplitude = 0.
            peak_loc = 0
            v_alpha(i_samp) = 0.
            do i_anl = 1, n_anl
               slice (i_anl) = smooth (i_samp, i_anl)
            enddo

            do i_anl = 1, n_anl
c               amplitude = smooth (i_samp, i_anl)
                amplitude = slice (i_anl)

                if (amplitude .gt. peak_amplitude) then
                    peak_amplitude = amplitude
                    peak_loc = i_anl
c                   v_alpha(i_samp) = alpha(i_anl)

                endif
            enddo
            if (peak_loc .gt. 1 .AND. peak_loc .lt. n_anl) then
                c1 = slice (peak_loc-1)
                c2 = slice (peak_loc  )
                c3 = slice (peak_loc+1)
                call parab (c1, c2, c3, px, py)
                v_alpha(i_samp) = alpha(peak_loc) + px * dgamma
            else
                v_alpha(i_samp) = alpha(peak_loc)
            endif
        enddo


C **********************************************************************
C ***** Smooth the alpha values ****************************************
C **********************************************************************


C ***** Preliminary edit by 3 sample median ****************************

        do i_samp = 2 + half_wlen, length - half_wlen - 1
            amp1 = v_alpha(i_samp - 1)
            amp2 = v_alpha(i_samp)
            amp3 = v_alpha(i_samp + 1)

            if (amp2 .gt. amp1) then
                if (amp3 .gt. amp2) then
                    work5(i_samp) = amp2
                elseif (amp3 .gt. amp1) then
                    work5(i_samp) = amp3
                else
                    work5(i_samp) = amp1
                endif
            else
                if (amp3 .gt. amp1) then
                    work5(i_samp) = amp1
                elseif (amp3 .gt. amp2) then
                    work5(i_samp) = amp3
                else
                    work5(i_samp) = amp2
                endif
            endif
        enddo
        work5(1 + half_wlen) = work5(2 + half_wlen)
        work5(length - half_wlen) = work5(length - half_wlen - 1)


C ***** Running smoother of user-specified length **********************

c ----> Time integration
        do i_samp = 2 + half_wlen, length - half_wlen
            work5(i_samp) = work5(i_samp - 1) + work5(i_samp)
        enddo

C ----> Difference gate edges and normalize
        v_alpha(1 + w_len) = work5(1 + w_len + half_wlen) / 
     *                                                  real (w_len + 1)
        do i_samp = 2 + w_len, length - w_len
            v_alpha(i_samp) = (work5(i_samp + half_wlen) - 
     *                         work5(i_samp - half_wlen - 1)) /
     *                                                  real (w_len + 1)

        enddo

C ----> Extrapolate top and bottom
        do i_samp = 1, w_len
            fraction = real (i_samp - 1) / real (w_len)
            v_alpha(i_samp) = v_alpha(1 + w_len) * fraction
        enddo
        do i_samp = length - w_len + 1, length
            fraction = real (length - i_samp) / real (w_len)
            v_alpha(i_samp) = v_alpha(length - w_len) * fraction
        enddo

        if (a_in .AND. .not.forward) return

3999    continue


C **********************************************************************
C ***** Create shift vector and stretch trace **************************
C **********************************************************************

        do i_tr = 1, maxntr
            call saver2 (thdr(1, i_tr), ifmt_StaCor, l_StaCor, 
     *              ln_StaCor, StaCor, TRACEHEADER)       ! Get the trace static

            if (StaCor .ne. 30000) then
                call saver2 (thdr(1, i_tr), ifmt_DstSgn, l_DstSgn, 
     *                                   ln_DstSgn, DstSgn, TRACEHEADER)
                x = abs ( real (DstSgn) )
                xx = x * x

                if (parabolic) then
                   do i_samp = 1, length
                       t_0 = real (i_samp)
                       time(i_samp) = t_0 + v_alpha(i_samp) * xx
                   enddo
                else
                   xx = .25 * xx
                   do i_samp = 1, length
                       t_0 = real (i_samp)
                       t_x = v_alpha(i_samp) * xx / dzdz
                       t_xx = t_0 * t_0 + t_x
                       if (t_xx .le. 0.) then
                          time(i_samp) = t_0
                       else
                          timx = sqrt (t_xx)
                          ipd = nint ( abs(timx-t_0) )
                          if (ipd .le. iplim) then
                             time(i_samp) = timx
                          else
                             time(i_samp) = t_0
                          endif
                       endif
                   enddo
                endif

c---
c  use quadratic interpolation to remap input data onto new time grid
c  Note: quad interpolator needs time indexing to start at 0.0
c---
                call vclr (work5, 1, length+2)
                do  i_samp = 1, length
                    time (i_samp) = time (i_samp) - 1.
                enddo
                call vqint (cdp(1, i_tr), length, time, 1, 
     1                      work5, 1, length)
                call vmov (work5, 1, cdp(1, i_tr), 1, length)

            endif
        enddo


C **********************************************************************
C ***** Optional verbose print out *************************************
C **********************************************************************

        if (verbose) then
            call saver2 (thdr(1, 1), ifmt_RecNum, l_RecNum, 
     *              ln_RecNum, recnum, TRACEHEADER)

            call saver2 (thdr(1, 1), ifmt_DphInd, l_DphInd, 
     *              ln_DphInd, cdp_val, TRACEHEADER)

            if (cdp_val .ge. start_cdp) then
                if (mod ((cdp_val - start_cdp), cdp_inc) .eq. 0) then

                    write (LERR, 4000) cdp_val, recnum
 4000               format (//' RMO analysis results for CDP number:',
     *                    i6, '      Record number:', 
     *                    i6, /'  time (ms)  alpha*E6     shifts (ms):')

                    do i_samp = 0, length - 1, t_inc
                        this_alpha = v_alpha(1 + i_samp)
                    
                        do i_off = 1, n_off
                            x = first_off + (last_off - first_off) * 
     *                           real (i_off - 1) / real (n_off - 1)
                            xx = x * x

                            if (parabolic) then
                               delta_t(i_off) = dt_ms * this_alpha * xx
                            else
                               delta_t(i_off) = sqrt (i_samp*i_samp +
     *                                          this_alpha * xx / dzdz )
                            endif

                        enddo

                        if (parabolic) then
                           write (LERR, 4010) dt_ms * real (i_samp), 
     *                          1.E6 * this_alpha, 
     *                          (delta_t(i_off), i_off = 1, n_off)
                        else
                           write (LERR, 4010) dt_ms * real (i_samp), 
     *                          this_alpha, 
     *                          (delta_t(i_off), i_off = 1, n_off)
                        endif

 4010                   format (f8.1, f12.7, 10f8.2)

                    enddo
                    write (LERR, 4020)
 4020               format (' ')
                endif
            endif
        endif

        return
        end







c---
c  change from gamma to alpha
c  alpha = gamma^2 - 1
c---
      subroutine toalpha (x, n)
      real    x(*)
      integer n

      do  i = 1, n
c         x (i) = (x(i) * x(i) - 1) / 4
          x (i) = (x(i) * x(i) - 1)
      enddo
      return
      end

c---
c  change from alpha to gamma
c  gamma = sqrt (alpha+1)
c---
      subroutine togamma (x, n)
      real    x(*)
      integer n

      do  i = 1, n
c         x (i) = sqrt (4*x(i) + 1)
          xi = x(i) + 1
          if (xi .ge. 0.0) then
             x (i) = sqrt (xi)
          else
             x (i) = 1.0
          endif
      enddo
      return
      end
