C***********************************************************************
C                 copyright 2001, 2002 Amoco Production Company             *
C                             All Rights Reserved                      *
C                     an affiliate of BP America Inc.                  *
C***********************************************************************
c**********************************************************************c
c
c sis_xy reads seismic trace data from an input file,
c performs minimum entropy deconvolution and
c writes the results to an output file
c
c
c**********************************************************************c
c 
c Changes:
c
c Feb 21, 2002: corrected bug in reading card input for reverse option
c   Abma
c
c Feb 19, 2002: put in dynamic memory stuff for really long traces
c               added -SI -UnitSc options to deal with X axis units
c               when USP in and ASCII out, the -SI option results in
c               output X axis being multiplied by UnitSc from lineheader
c               when ASCII in and USP out the -SI flag causes routine to
c               assume that input X axis is in seconds [ meters or feet
c               or whatever] and divides by UnitSc_in when calculating
c               global parameters to go into output line header.  
c               added -TmMsFS flag to cause the program to believe the
c               line header entry of the same name and set the time of 
c               the output sample one.
c               I also changed the routine to honour command line entries
c               for -TmMsFS, -s[], -e[] when picking up a dataset previously
c               created by sis_xy -R where the DgTrk1,2 entries are set to S,X
c               Previously this would result in a complete override of command line
c               parameters.
c               I also revamped the verbal and help subroutines to make things
c               a little clearer for the user as far as parameter presentation is
c               concerned
c               The routine is currently internally consistent, in that it will correctly
c               read ASCII input that it has created and vice-versa.  On INTEL platforms
c               if the user cats the dataset into this routine the byte order
c               is still native instead of USP which causes an i/o error in rtape.
c Garossino             
c
c May 1, 2001:  added UnitSc capability to ensure correct time listing
c               in ascii output.  Also fixed up dynamic memory allocation.
c Garossino
c
c     declare variables
c

      implicit none

#include <f77/iounit.h>
#include <f77/lhdrsz.h>
#include <f77/sisdef.h>
#include <save_defs.h>

c declare standard USP variables
 
      integer jerr, ist, iend, nsamp, nsi, ntrc, nrec, iform
      integer luin , lbytes, nbytes, luout, luo, lbyout, obytes
      integer irs, ire, ns, ne, JJ, KK
      integer argis

      real UnitSc, si

      character ntap * 255, otap * 255, name*6

      logical verbos, flat

c variables used in dynamic memory allocation

      integer itr, abort, errcd, errcd1, items

      real tri, work, X, Y

      pointer     (mem_itr, itr(200))
      pointer     (mem_tri, tri(2))
      pointer     (mem_Y, Y(2))
      pointer     (mem_X, X(2))
      pointer     (mem_work, work(2))

c declare program dependent static variables

      integer RecNum, TrcNum, ncomp, rdel, tdel
      integer npts ( SZLNHD )
      integer length, lenth, luval

      real dels, nullval, TmMsFS

      character dg1*1, dg2*1, card*80

      logical stream, w_col, blnk, mbs, rev, SI_units, believe

c declare variables uncovered by implicit none

      integer idec, nsampi, ntrci, nreci, ierr, j, i, nc, nfunc, maxpts
      integer minpts, minx, maxx, intx, nbyt, idels
      integer ifmt_TrcNum, l_TrcNum, ln_TrcNum
      integer ifmt_RecNum, l_RecNum, ln_RecNum
      integer nout, ntr, lc, iflag, istrc, ic, ns0, nel, itt

      real wvel, xi, yi, t, UnitSc_in

c-----
c initialize variables
c-----

      data lbytes / 0 /
      data nbytes / 0 /
      data name /'SIS_XY'/
      data abort / 0 /
      data luin /1/
      
c-----
c     give program help if requested
c-----

      if ( argis ( '-?' ) .gt. 0 .or. 
     :     argis( '-h' ) .gt. 0 .or. 
     :     argis( '-help' ) .gt. 0) then
         call help()
         stop
      endif
 
c-----
c     open printout file
c-----

#include <f77/open.h>

c-----
c parse command line
c-----
 
      call gcmdln ( ntap, ist, iend, ns, ne, irs, ire, blnk, verbos,
     :     dels, ncomp, idec, rdel, tdel, mbs, rev, otap, nullval,
     :     stream, nsampi, ntrci, nreci, w_col, wvel, SI_units, 
     :     UnitSc_in, believe, flat, si )

      IF (rev) THEN

c-----
c  read a file of X-Y values and generate USP format traces
c-----

         if (ntap(1:1) .eq. ' ') then
            write(LERR,*)' '
            write(LERR,*)'Opening input stream for X-Y data. The'
            write(LERR,*)'number of samples per trace must be the'
            write(LERR,*)'same'
            write(LERR,*)' '
         else
            luval = 29
            open(unit=luval, file = ntap, status='old', iostat=ierr)
 
            if(ierr .ne. 0) then
               write(LERR,*)'SIS_XY: Could not open X-Y input file'
               write(LERR,*)'       Check existence/permissions'
               write(LERR,*)'FATAL'
               write(LER ,*)'SIS_XY: Could not open X-Y input file'
               write(LER ,*)'       Check existence/permissions'
               write(LER ,*)'FATAL'
               stop
            endif
         endif

c-----
c     get logical unit numbers for input and output
c-----

         call getln ( luout , otap, 'w', 1 )


         if ( .not. stream ) then

c-----
c read through input ASCII file and determine global parameters
c-----

            rewind luval

            j = 1
            i = 0

            DO  while (1 .eq. 1)
c-----
c determine number of traces entered and length in samples of each
c bail out to statment 50 on EOF
c-----
               read (luval, '(a80)', end=50) card
               nc = lenth (card)
               i = i + 1

               if (nc .le. 1) then
                  npts (j) = i-1
                  j = j + 1
                  i = 0
               endif
            ENDDO

c-----
c if we get here then we had a premature end of data on the 
c in put file.  Warn the user and kill the program
c-----

            write(LERR,*)' '
            length = lenth(ntap)
            if (length .gt. 0) then
               write(LERR,*)'Did not hit end of file on input - ',
     1              'check file ',ntap(1:length)
            else
               write(LERR,*)'Did not hit end of file on input - ',
     1              'check file ',ntap
            endif

            write(LER ,*)' '
            write(LER ,*)'SIS_XY:'
            if (length .gt. 0) then
               write(LER,*)'Did not hit end of file on input - ',
     1              'check file ',ntap(1:length)
            else
               write(LER,*)'Did not hit end of file on input - ',
     1              'check file ',ntap
            endif
            write(LER,*)'FATAL'
            write(LER,*)' '

            stop
            
 50         continue

            if (i .eq. 0) then
               j = j - 1
            else
               npts (j) = i
            endif

            nfunc = j

            rewind luval

c-----
c policeman: determine if all functions truly have the same number
c            of samples.  This is required in order to build USP 
c            traces where each trace requires the same number of 
c            samples
c-----

            call imax (nfunc, npts, maxpts) 
            call imax (nfunc, npts, minpts)

            if ( minpts .ne. maxpts ) then

               write(LERR,*)'sis_xy fatal error:'
               write(LERR,*)'Unequal length X-Y functions'
               write(LERR,*)'Found ',nfunc,' functions:'

               do  j = 1, nfunc
                  write(LERR,*)'function ',j,' has ',npts(j),' pts'
               enddo

               write(LERR,*)'Must be all equal length'

               write(LER,*)' '
               write(LER,*)'SIS_XY:'
               write(LER,*)' Unequal length X-Y functions'
               write(LER,*)' Found ',nfunc,' functions:'

               do  j = 1, nfunc
                  write(LER,*)' function ',j,' has ',npts(j),' pts'
               enddo

               write(LER,*)' Must be all equal length'
               write(LER,*)'FATAL'
               write(LER,*)' '

               stop

            endif
            
            write(LERR,*)' '
            write(LERR,*)'Number of functions read  = ',nfunc
            write(LERR,*)'Number of points/function = ',maxpts
            write(LERR,*)' '

c-----
c generic allocation of X,Y array buffer memory
c-----

            call galloc (mem_X, maxpts * SZSMPD, errcd, abort)
            call galloc (mem_Y, maxpts * SZSMPD, errcd1, abort)
            if ( errcd .ne. 0 .and.
     :           errcd1 .ne. 0 ) then
               write(LERR,*)' '
               write(LERR,*)'Unable to allocate workspace:'
               write(LERR,*) 2 * maxpts * SZSMPD,'  bytes'
               write(LER ,*)' '
               write(LER ,*)'SIS_XY: '
               write(LER ,*)' Unable to allocate workspace:'
               write(LER ,*)  2 * maxpts * SZSMPD,'  bytes'
               write(LER ,*)'FATAL'
               goto 999
            else
               write(LERR,*)' '
               write(LERR,*)'Allocating workspace:'
               write(LERR,*) 2 * maxpts * SZSMPD,'  bytes'
            endif

c initialize memory

            call move ( 0, X, 0, maxpts * SZSMPD )
            call move ( 0, Y, 0, maxpts * SZSMPD )
         
c-----
c determine the minimum, maximum and delta variables for the units axis
c need to buffer first trace in memory to do this
c-----

c-----
c load first trace to memory
c-----
            call LoadFirstTrace ( luval, X, Y, maxpts, nullval, minx, 
     :           maxx, intx, UnitSc_in, SI_units )

c-----
c default interval to unity if not obvious from the initial input
c if s.i. override on cmd line then deal with that
c PRG:10-20-2003
c     changed intx calculation so that intx = whatever you put on cmd line
c     (ms or us as in man page)
c----

            if (si .eq. 0.0) then
               if (intx .eq. 0) intx = 1
            else
c              intx = nint (si/UnitSc_in)
               intx = nint (si)
            endif

            if ( nreci .eq. 1 .AND. ntrci .eq. 1) then
               ntrci = nfunc
            else
               nfunc = nreci * ntrci
            endif

         else

c-----
c input is on stdin, since we cannot stream data without losing it we must
c stream and buffer the first trace.  In this case the user must have supplied
c the number of samples per trace on the command line so we can load up the 
c initial array based on that assumtion.
c-----

            luval = LIN
            maxpts = nsampi
            nfunc = ntrci * nreci

c-----
c allocate memory for X,Y arrays based on information from command line
c-----

            errcd = 0
            call galloc (mem_X, maxpts * SZSMPD, errcd1, abort)
            errcd = errcd + errcd1
            call galloc (mem_Y, maxpts * SZSMPD, errcd1, abort)
            errcd = errcd + errcd1
            if ( errcd .ne. 0 ) then
               write(LERR,*)' '
               write(LERR,*)'Unable to allocate workspace:'
               write(LERR,*) 2 * maxpts * SZSMPD,'  bytes'
               write(LER ,*)' '
               write(LER ,*)'SIS_XY: '
               write(LER ,*)' Unable to allocate workspace:'
               write(LER ,*)  2 * maxpts * SZSMPD,'  bytes'
               write(LER ,*)'FATAL'
               goto 999
            else
               write(LERR,*)' '
               write(LERR,*)'Allocating workspace:'
               write(LERR,*) 2 * maxpts * SZSMPD,'  bytes'
            endif

            call move ( 0, X, 0, maxpts * SZSMPD )
            call move ( 0, Y, 0, maxpts * SZSMPD )

c-----
c load first trace to memory so as not to back up on the pipe
c-----
         
            call LoadFirstTrace ( luval, X, Y, maxpts, nullval, minx, 
     :           maxx, intx, UnitSc_in, SI_units)

            
         endif

c-----
c     We now have enough global information to build a line header for the output
c     so here is the initial dynamic allocation of ITR to allow for line header 
c     construction
c-----

         if (flat) then
            nsamp = nsampi
         else
            nsamp = maxpts
         endif

         errcd = 0
         call galloc ( mem_itr, SZLNHD * SZSMPD, errcd1, abort )
         errcd = errcd + errcd1
         call galloc ( mem_tri, nsamp  * SZSMPD, errcd1, abort )
         errcd = errcd + errcd1
         if (errcd .ne. 0) then
            write(LERR,*) 'ERROR: Unable to allocate workspace '
            write(LERR,*) '       ',SZLNHD * SZSMPD,' bytes requested '
            write(LERR,*) '       ',nsamp * SZSMPD,' bytes requested '
            write(LERR,*) 'FATAL'
            write(LER,*) ' '
            write(LER,*) 'SIS_XY:'
            write(LER,*) ' Unable to allocate workspace '
            write(LER,*) '  ',SZLNHD * SZSMPD,' bytes requested '
            write(LER ,*) '       ',nsamp * SZSMPD,' bytes requested '
            write(LER,*) 'FATAL'
            write(LER,*) ' '
            stop
         else
            write(LERR,*)'Allocating workspace:'
            write(LERR,*) SZLNHD * SZSMPD,' bytes requested '
            write(LERR,*) nsamp * SZSMPD,' bytes requested '
         endif

c initialize memory

         call vclr ( itr, 1, SZLNHD )
         call vclr ( tri, 1, nsamp )
         
         call savelu('TrcNum',ifmt_TrcNum,l_TrcNum,ln_TrcNum,
     :        TRACEHEADER)
         call savelu('RecNum',ifmt_RecNum,l_RecNum,ln_RecNum,
     :        TRACEHEADER)

c build outgoing line header 

         call savew (itr, 'NumRec', nreci , LINHED)
         call savew (itr, 'Format',  3    , LINHED)
         call savew (itr, 'NumTrc', ntrci , LINHED)
         call savew (itr, 'SmpInt', intx  , LINHED)
         call savew (itr, 'NumSmp', nsamp , LINHED)
         call savew (itr, 'MxShDp', maxx  , LINHED)
         call savew (itr, 'MnShDp', minx  , LINHED)
         call savew (itr, 'IntInc', intx  , LINHED)
         call savew (itr, 'UnitSc', UnitSc_in, LINHED)
         dg1 = 'S'
         dg2 = 'X'
         call savew (itr, 'DgTrk1', dg1   , LINHED)
         call savew (itr, 'DgTrk2', dg2   , LINHED)
         
         obytes = SZTRHD + SZSMPD*nsamp
         lbytes = HSTOFF
         nbyt = 2 * SZHFWD
         call savew ( itr, 'HlhEnt',  0   , LINHED)
         call savew ( itr, 'HlhByt', nbyt , LINHED)

c write output line header

         call savhlh( itr, lbytes, lbyout )
         CALL WRTAPE ( LUOUT, ITR, LBYOUT )

      ELSE

c-----
c  read USP format trace data and output X-Y values
c-----


c-----
c     open input dataset
c-----

         call getln( luin , ntap,'r', 0 )

c-----
C     initial dynamic allocation of ITR to allow for line header 
c     interaction
c-----

      errcd = 0
      call galloc ( mem_itr, SZLNHD * SZSMPD, errcd, abort )
      if (errcd .ne. 0) then
         write(LERR,*) 'ERROR: Unable to allocate workspace '
         write(LERR,*) '       ',SZLNHD * SZSMPD,' bytes requested '
         write(LERR,*) 'FATAL'
         write(LER,*) ' '
         write(LER,*) 'SIS_XY:'
         write(LER,*) ' Unable to allocate workspace '
         write(LER,*) '  ',SZLNHD * SZSMPD,' bytes requested '
         write(LER,*) 'FATAL'
         write(LER,*) ' '
         stop
      else
         write(LERR,*)'Allocating workspace for line header:'
         write(LERR,*) SZLNHD * SZSMPD,' bytes requested '
      endif

      call vclr ( itr, 1, SZLNHD )

c-----
c     read line header of input
c     save certain parameters
c-----

         call rtape ( luin, itr, lbytes)
         if(lbytes .eq. 0) then
            write(LER,*)'SIS_XY: no header read from unit ',luin
            write(LER,*)'FATAL'
            stop
         endif

         if (otap(1:1) .eq. ' ') then
            luo = LOT
         else
            call alloclun (luo)
            open (unit=luo, file=otap, status='unknown', iostat=ierr)

            if(ierr .ne. 0) then
               write(LERR,*)'SIS_XY: Could not open X-Y output file'
               write(LERR,*)'       Check existence/permissions'
               write(LERR,*)'FATAL'
               write(LER ,*)'SIS_XY: Could not open X-Y output file'
               write(LER ,*)'       Check existence/permissions'
               write(LER ,*)'FATAL'
               stop
            endif
         endif
         

c------
c     build pointers to required trace header mnemonics 
c     [see prgm.F template for details]
c------

         call savelu('TrcNum',ifmt_TrcNum,l_TrcNum,ln_TrcNum,
     :        TRACEHEADER)
         call savelu('RecNum',ifmt_RecNum,l_RecNum,ln_RecNum,
     :        TRACEHEADER)

c get pertinent global variables from incoming line header

         call saver (itr, 'NumSmp', nsamp, LINHED)
         call saver (itr, 'SmpInt', nsi  , LINHED)
         call saver (itr, 'NumTrc', ntrc , LINHED)
         call saver (itr, 'NumRec', nrec , LINHED)
         call saver (itr, 'Format', iform, LINHED)
         call saver (itr, 'MxShDp', maxx , LINHED)
         call saver (itr, 'MnShDp', minx , LINHED)
         call saver (itr, 'IntInc', intx , LINHED)
         call saver (itr, 'DgTrk1', dg1  , LINHED)
         call saver (itr, 'DgTrk2', dg2  , LINHED)
         call saver (itr, 'UnitSc', UnitSc, LINHED)
         call saver (itr, 'TmMsFS', TmMsFS, LINHED)
         
         if ( UnitSc .eq. 0.0 ) UnitSc = 0.001
         
c we want dels to be in seconds or meters as a default unless the
c user has flagged the command line entry [-UOD] for units of the
c data.

         if (dels .eq. 0.0) then
            dels = float(nsi)
         endif

         call saver(itr, 'TmSlIn', idels, LINHED)
         if (mbs) dels = float(idels) / 1000.
         
         nsi = dels
         si  = dels
         
         if (dg1 .eq. 'S' .AND. dg2 .eq. 'X') then
            nsi = intx
            ns0 = minx
            if ( believe ) ns0 = nint(TmMsFS)
            ist  = nint (float(ist)/si)
            iend = nint (float(iend)/si)
            if ( ist .le. 1 ) ist = 1
            if ( iend .eq. 0 ) iend = nsamp
            nout = nsamp
         else
            ns0  = 0
            if ( believe ) ns0 = nint(TmMsFS)
            ist  = nint (float(ist)/si)
            iend = nint (float(iend)/si)
            if (ist  .le. 1) ist = 1
            if (iend .lt. 1) iend = nsamp
            if (iend .gt. nsamp) iend = nsamp
            nout = iend - ist + 1
         endif
c-----
c     ensure that command line values are compatible with data set
c-----
         call cmdchk(ns,ne,irs,ire,ntrc,nrec)
         
         ntr = ne - ns + 1
         
         if ( ncomp .gt. ntr ) then
            write(LERR,*)'WARNING:'
            write(LERR,*)'Number of columns to output = ',ncomp,' is'
            write(LERR,*)'greater then number of traces selected from'
            write(LERR,*)'each record= ',ntr,' Will set # cols= ',ntr
            ncomp = ntr
         endif
         
c-----
c
c end of IF (rev) THEN logic
c
c-----

      ENDIF

c-----
c     verbose output of all pertinent information before
c     processing begins
c-----

      call verbal ( ntap, otap, ist, iend, ns, ne, irs, ire, blnk, 
     :     verbos, dels, ncomp, idec, rdel, tdel, mbs, rev, nullval, 
     :     stream, w_col, wvel, SI_units, nsamp, nsi, ntrc, nrec, 
     :     iform, UnitSc, UnitSc_in, TmMsFs, believe, flat, si )

      if (ncomp .gt. 1) then

         items = ntr * nout * SZSMPD
      
         call galloc (mem_work, items, errcd, abort)
 
         if (errcd .ne. 0 ) then
            write(LERR,*)' '
            write(LERR,*)'Unable to allocate workspace:'
            write(LERR,*) items,'  bytes'
            write(LER ,*)' '
            write(LER ,*)'SIS_XY: '
            write(LER ,*)' Unable to allocate workspace:'
            write(LER ,*) items,'  bytes'
            write(LER ,*)'FATAL'
            go to 999
         else
            write(LERR,*)' '
            write(LERR,*)'Allocating workspace:'
            write(LERR,*) items,'  bytes'
         endif

c initialize memory

         call move (0, work, 0, items )
         
      endif

c dynamic reallocation of itr to allow for trace data rather than 
c line header

      errcd = 0
      call grealloc ( mem_itr, ITRWRD * SZSMPD + nsamp * SZSMPD, errcd, 
     :     abort )

      if ( errcd .ne. 0 ) then
         write(LERR,*) 'ERROR: Unable to allocate workspace '
         write(LERR,*) ITRWRD + nsamp * SZSMPD,' bytes requested '
         write(LERR,*) '       FATAL'
         write(LER,*) 'SIS_XY: '
         write(LER,*) ' Unable to allocate workspace '
         write(LER,*) ITRWRD + nsamp * SZSMPD,' bytes requested '
         write(LER,*) 'FATAL'
         goto 999
      else
         write(LERR,*)'Allocating workspace:'
         write(LERR,*) ITRWRD + nsamp * SZSMPD,' bytes requested '
      endif

c initialize memory

      call move (0, itr, 0, (ITRWRD + nsamp) * SZSMPD )

c-----
c     BEGIN PROCESSING
c-----

      IF (rev ) THEN

c-----
c reading X,Y ASCII data, generating USP data
c-----


c-----
c first trace is already in memory so simply dump it with assigned trace
c headers for record and trace number
c-----

         if (flat) then
            call vclr (tri, 1, nsamp)
            do  i = 1, maxpts
                itt = ifix(X(i)/si) + 1
                tri(itt) = Y(i)
            enddo
         else
            call vmov (Y, 1, tri, 1, maxpts)
         endif
         call vmov (tri, 1, itr(ITHWP1), 1, nsamp)
         call savew2(itr,ifmt_TrcNum,l_TrcNum, ln_TrcNum,1 ,TRACEHEADER)
         call savew2(itr,ifmt_RecNum,l_RecNum, ln_RecNum,1, TRACEHEADER)
         call wrtape (luout, itr, obytes)

c-----
c continue to read ASCII data and write USP traces until out of input
c-----

         DO  j = 2, nfunc

             i = 0
             call vclr (Y, 1, maxpts)

             do  while (1.eq.1)

                 read (luval, '(a80)', end=65) card

                 lc = lenth(card)
                 go to 66
 
65               continue
                 lc = 0
 
66               continue

                 if ( lc .gt. 1 ) then
                    call fsscnf ( card, '%f %f', Xi, Yi )
                    i = i + 1
                    if (abs(nullval-Yi) .lt. 1.e-30) then
                       Y(i) = 0.
                    else
                       Y(i) = Yi
                    endif

                 elseif ( lc .le. 1 .and. i .gt. 0 ) then

                    if (flat) then
                       call vclr (tri, 1, nsamp)
                       do  i = 1, maxpts
           
                           itt = ifix(X(i)/si) + 1
                           tri(itt) = Y(i)
                       enddo
                    else
                       call vmov (Y, 1, tri, 1, maxpts)
                    endif

                    call vmov (tri, 1, itr(ITHWP1), 1, nsamp)
                    call savew2(itr,ifmt_TrcNum,l_TrcNum, ln_TrcNum,
     1                   j, TRACEHEADER)
                    call wrtape (luout, itr, obytes)

                    go to 70

                 endif
             enddo

70           continue

         ENDDO

      ELSE

c-----
c reading USP data, writing ASCII data
c-----

c-----
c     skip unwanted records
c-----

         call recskp(1,irs-1,luin,ntrc,itr)

c-----
c allocate tri[] memory
c-----

         errcd = 0
         call galloc ( mem_tri, nsamp * SZSMPD, errcd, abort )

         if ( errcd .ne. 0 ) then
            write(LERR,*) 'ERROR: Unable to allocate workspace '
            write(LERR,*) nsamp * SZSMPD,' bytes requested '
            write(LERR,*) '       FATAL'
            write(LER,*) 'SIS_XY: '
            write(LER,*) ' Unable to allocate workspace '
            write(LER,*) nsamp * SZSMPD,' bytes requested '
            write(LER,*) 'FATAL'
            goto 999
         else
            write(LERR,*)'Allocating workspace:'
            write(LERR,*) nsamp * SZSMPD,' bytes requested '
         endif

c initialize memory

         call move (0, tri, 0, nsamp * SZSMPD )
c-----
c     process desired trace records
c-----
         iflag = 0

         DO 1000 JJ = irs, ire, rdel
 
c-----
c  skip to start trace
c-----

            call trcskp(JJ,1,ns-1,luin,ntrc,itr)
 
            DO 1001 KK = ns, ne, tdel

               nbytes = 0
               call rtape( luin, itr, nbytes)
               if(nbytes .eq. 0) then
                  write(LERR,*)'Premature EOF on input:'
                  write(LERR,*)'  rec= ',JJ,'  trace= ',KK
                  go to 999
               endif

               call vmov (itr(ITHWP1), 1, tri, 1, nsamp)
               
               call saver2(itr,ifmt_RecNum,l_RecNum, ln_RecNum,
     1              RecNum , TRACEHEADER)
               call saver2(itr,ifmt_TrcNum,l_TrcNum, ln_TrcNum,
     1              TrcNum , TRACEHEADER)

               IF (ncomp .eq. 1) THEN

c-----
c  calculate time,amplitude data and output according to user
c  defined choice of format
c-----

                  if (w_col) then

c-----
c do not output any samples from the top of the trace whose amplitude
c is less than wvel.  This is the -w[] option on the command line
c-----

                     do i = 1, iend
                        if ( (tri(i) - wvel ) .gt. 1.e-10) then
                           ist = i - 1
                           go to 201
                        endif
                     enddo
 201                 continue

                     do i = ist, iend, idec
                        t = ns0 + dels * float(i-1)
                        write(luo,101) t,tri(i)
                     enddo

                  else

c-----
c output all data for trace
c-----

                     do i = ist, iend, idec

c-----
c unless flagged for units of the data [-UOD] on command line 
c we want to output the time/depth values in units of seconds or
c meters. 
c-----

                        if ( SI_units ) then
                           t = ( ns0 + dels * float(i-1) ) * UnitSc
                        else
                           t = ns0 + dels * float(i-1)
                        endif

                        write(luo,101) t,tri(i)
                     enddo

                  endif

 101              format(f15.5,5x,e15.8)

c-----
c  output null line for compatibility with xgraph plotting, without null line
c  the default situation would be for xgraph to overprint all traces in one
c  plot.  The blank allows for each trace to get its own label and breaks the
c  line draw between traces.
c-----

                  if ( .not. blnk )
     1                 write(luo,102)
 102              format()

               ELSE

c-----
c  dealing with multi-component data.  Fill up array of output trc values
c-----

                  istrc = (KK-1) * nout
                  call vmov (tri(ist), 1, work(istrc+1), 1, nout)

               ENDIF

               if(verbos)write(LERR,*)'ri ',RecNum,' trace ',TrcNum

c-----
c  skip to next trace
c-----

               if (tdel .gt. 1 .and. (KK+tdel) .le. ntrc) then
                  call trcskp(JJ,KK+1,KK+tdel-1,luin,ntrc,itr)
               endif
               ic = KK

 1001       continue

            IF ( ncomp .gt. 1 ) THEN

c-----
c  create time ampl axis using format for multi-component dataset
c-----

               nel = nout * ntr
               do  105  i = 1, nout

c-----
c unless flagged for units of the data [-UOD] on command line 
c we want to output the time/depth values in units of seconds or
c meters. 
c-----

                  if ( SI_units ) then
                     t =  ( ns0 + dels * float(i-1) ) * UnitSc
                  else
                     t = ns0 + dels * float(i-1)
                  endif

                  write(luo,*) t,
     1                 (work(i+(j-1)*nout), j = 1, ncomp)
 105           continue

            ENDIF
 
c-----
c  skip to end of record
c-----

            if (tdel .gt. 1 .and. (ic+1) .lt. ntrc) then
               call trcskp(JJ,ic+1,ntrc,luin,ntrc,itr)
            endif

c-----
c  skip to next record
c-----

            if (rdel .gt. 1)
     1           call recskp(JJ+1,JJ+rdel-1,luin,ntrc,itr)
 
 1000    CONTINUE

      ENDIF

c-----
c     close data files - normal termination
c-----

      if (rev) then
         call lbclos ( luout )
         close (luval)
      else
         call lbclos ( luin )
         close (luo)
      endif
      write(LERR,*)'processed',nrec,' record(s)',
     :     ' with ',ntrc, ' traces'
      write(LER,*)'sis_xy: Normal Termination'
      stop

 999  continue

c-----
c     close data files - abnormal termination
c-----

      if (rev) then
         call lbclos ( luout )
         close (luval)
      else
         call lbclos ( luin )
         close (luo)
      endif

      write(LERR,*)'end of sis_xy, processed',nrec,' record(s)',
     :     ' with ',ntrc, ' traces'

      write(LER,*)'sis_xy: Abnormal Termination'

      stop
      end
