C***********************************************************************
C                 copyright 2001, Amoco Production Company             *
C                             All Rights Reserved                      *
C                     an affiliate of BP America Inc.                  *
C***********************************************************************
C  ROUTINE:       VTTD                                                 *
C  ROUTINE TYPE:  MAIN                                                 *
C  PURPOSE:  CONVERT A TIME SECTION TO A DEPTH SECTION                 *
C                            -OR-                                      *
C          - CONVERT A DEPTH SECTION TO A TIME SECTION                 *
C  AUTHOR:   Mary Ann Thornton                  ORIGIN DATE: 90/06/11  *
C  ROUTINES CALLED:                                                    *
C      CMDLIN          - GET COMMAND LINE ARGUMENTS                    *
C      LBOPEN          - OPEN TAPE                                     *
C      OPENPR          - OPEN PRINT FILE                               *
C      GAMOCO          - PRINT TORCH AND OVAL                          *
C      ICOPEN  INTEGER - OPEN CARD FILE                                *
C      RTAPE           - READ TAPE                                     *
C      MOVE            - MOVE CHARACTER STRING                         *
C      XMATRX          - BUILD VELOCITY MATRIX                         *
C      HLHPRT          - PRINT & UPDATE HISTORICAL LINEHEADER          *
C      WRTAPE          - WRITE TAPE                                    *
C      CCUINT          - CUBIC INTERPOLATOR                            *
C      LBCLOS          - CLOSE TAPE                                    *
C  FILES:                                                              *
C      LCRD   ( INPUT  SEQUENTIAL ) - INPUT CARD FILE                  *
C      lprt   ( OUTPUT SEQUENTIAL ) - PRINT FILE                       *
C      LLIST  ( OUTPUT SEQUENTIAL ) - PARENT PID LIST                  *
C      LMXC   ( INPUT  SEQUENTIAL ) - MODEL FILE                       *
C  STOP CODES:                                                         *
C      200      ( 1) - PRINT FILE ERRORS, and OUT OF BOUNDS ARRAYS     *
C      50       ( 5) - CARD FILE ERRORS                                *
C      75       ( 5) - TAPEIO ERRORS                                   *
C      100      ( 3) - OTHER ERRORS                                    *
C      500      ( 1) - ERROR IN VELOCITY MATRIX                        *
C      =BLANK=  ( 5) - NO ERRORS                                       *
C       +------------------------------------------------------+       *
C       |             OTHER DOCUMENTATION DETAILS              |       *
C       +------------------------------------------------------+       *
C  ERROR HANDLING:  ???                                                *
c  GENERAL DESCRIPTION:  Original program read mxc file and built a    *
C               Matrix, this version replaced mxc with velocity tape   *
C  REVISED BY:  MARY ANN THORNTON             REVISION DATE: 89/02/20  *
C            -  ADD MAXIMUM TIME AND DEPTH PARAMETERS AND MAKE KSAMP   *
C            -  CORRESPOND                                             *
C  REVISED BY:  MARY ANN THORNTON             REVISION DATE: 89/02/28  *
C            -  MAKE THE INPUT & OUTPUT TABLES THE SAME (LONGEST)      *
C  REVISED BY:  MARY ANN THORNTON             REVISION DATE: 89/03/03  *
C            -  CORRECT INPUT & OUTPUT TABLES ALL WRONG BEFORE         *
C  REVISED BY:  MARY ANN THORNTON             REVISION DATE: 89/03/23  *
C            -  CHECK FOR INVALID DTMS VALUE IN HEADER WORD 13         *
C  REVISED BY:  MARY ANN THORNTON             REVISION DATE: 89/03/23  *
C            -  Moved code to Sun for Maintenance from the Sun         *
C            -  Corrected error found in reading vel.tape lineheader   *
C            -  Changed CALL MOVES to CALL VCLR AND CALL VMOV          *
C  REVISED BY:  CURTIS KRUSE                  REVISION DATE: 91/??/??  *
C               Added 2 checks to make sure not getting samples below  *
C               end of tape                                            *
C  REVISED BY:  Mary Ann Thornton             REVISION DATE: 91/??/??  *
C               Added 2 checks from Curtis's change above to the code  *
C  REVISED BY:  Mary Ann Thornton  V3.0       REVISION DATE: 91/??/??  *
C               removed DX parameter (not necessary) replaced the cubic*
C               interpolator with the linear interpolator.             *
C  REVISED BY:  Mary Ann Thornton  V3.1       REVISION DATE: 92/03/26  *
C               Call openpr with full program name for OS 6.1          *
C  REVISED BY:  Mary Ann Thornton  V3.2       REVISION DATE: 92/05/12  *
C               Rename topen to cmdlin for new sun compiler            *
C  REVISED BY:  Gary Murphy        V3.3       REVISION DATE: 92/06/12  *
C               Added verbos print statements for negative velocities  *
C  REVISED BY:  Mary Ann Thornton  V3.4       REVISION DATE: 92/07/30  *
C               Allowed the program to run when the velocity tape does
C               not match the seismic data precisely. The last velocity
C               trace or the last velocity sample will be used when the
C               velocity tape is not as big in size as the seismic data
C               Corrected error in trace numbers when doing the depth 
C               to time conversion.  Cleaned up the code and tried to
C               make the error messages more meaningful to a user.
C  REVISED BY:  Mary Ann Thornton  V3.5       REVISION DATE: 92/08/25  
C               Added statements to rewind the velocity tape in between
C               input records, so vttd would work on multi-record input
C  REVISED BY:  Mary Ann Thornton  V3.6       REVISION DATE: 92/11/02  
C               Copied vttd from ~mbs to ~usp so vttdusp could be
C               eliminated - recompiled to see if everything works as
C               a usp group member - code will be in mbs and usp for now
C  REVISED BY:  Mary Ann Thornton  V3.7       REVISION DATE: 92/11/09  
C               Added a change to the way the table is calculated for
C               the time to depth conversion to make it more accurate
C               The values will be the average slowness over the 
C               boundaries at the base of the layers.
C               Also increased the maximum sample limit and resampled
C               data on input to 1ms data when doing time to depth
C               conversion. 
C  REVISED BY:  Mary Ann Thornton  V3.8       REVISION DATE: 93/02/25  
C               Remove the call to lbopen after the call to rwd.  The
C               tape does not need to be reopened.  Each time lbopen was
C               called a new lu number was assigned, and after 100
C               calls, the program bombs with too many files open.
C  REVISED BY:  Dan Whitmore       V3.9       REVISION DATE: 93/03/12 
C               Added option to allow input velocity tape to be in time.
C  REVISED BY:  Gary Murphy        V4.0       REVISION DATE: 93/03/12
c               added 3d option to allow multirecord input without
c               rewinding the velocity tape.  added lbclos call after
c               lbopen in velocity tape handling.
c               added velocity tape input option (vt).
C  REVISED BY:  Mary Ann Thornton  V4.1       REVISION DATE: 93/03/23  
C               removed a call to vmov and replaced with a do loop
C               for moving data in at the end of traces when it is a	
C               velocity tape. Also increased line header size.
C  REVISED BY:  Mary Ann Thornton  V4.2       REVISION DATE: 93/04/02  
C               Changed comments that get printed to describe how to use
C               the program and added check for dt on command line.
C               dt must be entered on command line.
C  REVISED BY:  Mary Ann Thornton  V4.3       REVISION DATE: 93/05/24  
C               Corrected an error in 3d option.  the 'endif' was in the
C               wrong place when checking to see if running in 3d mode
C               When in 3d mode, a call to rtape at the end of each record    
C               loop read a velocity trace off the next record, when this
C               call to rtape was really made for reading the line 
C               header after rewinding the velocity tape before
C               going to the next data record.
C  REVISED BY:  Mary Ann Thornton  V4.4       REVISION DATE: 93/06/10  
C               increased the mxsam to 12000 to allow more space for the
C               work arrays when requesting a large amount of time on
C               output, and changed the error message to be more 
C               informative to a user.
C  REVISED BY:  Gary Murphy        V4.5       REVISION DATE: 93/07/13  
C               Put in crude fix to make sure output velocity tape     
C               never contains zeros.                                  
C  REVISED BY:  Mary Ann Thornton  V4.6       REVISION DATE: 93/07/21  
C               Remove fix of version 4.5.                             
C               Change code when doing depth to time conversion on a
C               seismic dataset to extend the trace with zeroes rather
C               than extend the trace with the last data sample BEFORE
C               the conversion is done.
C               Also, added many more comments to try to make code more
C               understandable, and added checks where there was a 
C               chance for the arrays to go out of bounds.
C               Changed cmdlin to make the comments better, which
C               requires a change be made to vttd.pat to match comments
C               Changed calculation of nzvt:(added +1.0 to calculation)
C  REVISED BY:  Mary Ann Thornton  V4.7       REVISION DATE: 93/08/26  
C               Changed to issue a warning message when the velocity   
C               dataset is shorter or longer than the dataset to be    
C               converted. Also made a change to allow 1 trace records 
C               to be handled correctly whether or not the 3d option is
C               selected.                                              
C  REVISED BY:  Mary Ann Thornton  V4.8       REVISION DATE: 93/10/25  
C               Changed to include the command line arguments in L.header
C  REVISED BY:  Mary Ann Thornton  V4.9       REVISION DATE: 94/01/04  
C               Added dx1000 to the output line header and dz1000 when
C               output is in depth.                                    
C               A new command line argument was added for dx.  If it is
C               missing, pgm will look for dx1000 in input velocity field,
C               If that is missing, too, pgm will look for dx1000 in the
C               input dataset to be converted.  If that is blank, then
C               proceed with nothing put in dx1000 location.  Also if 3d
C               option is requested, ILClIn was placed in line header.
C               The format statements were revised for easier and more
C               informative reading. The dx and dz values among the inputs
C               and the output will be compared, and the user notified
C               of mismatches, but program will continue execution.
C  REVISED BY:  Mary Ann Thornton  V5.0       REVISION DATE: 94/03/22  
C               Switch the calls ccuint and fliint for seismic and velocity
C               input.
C  REVISED BY:  Mary Ann Thornton  V5.1       REVISION DATE: 94/03/24  
C               Remove the limit on traces in; there is no need for it.    
C  REVISED BY:  Mary Ann Thornton  V5.2       REVISION DATE: 94/11/17  
C               Add a -lhdt parameter to force the program to use the  
C               sample rate from the input dataset for -dt
C  REVISED BY:  Paul Garossino     V5.3       REVISION DATE: 96/03/05 
C               Allow velocity tape to be a pipe (IKP ability)
C               He established a symbolic link vttdusp  vttd
C  REVISED BY:  Mary Ann Thornton  V5.4       REVISION DATE: 96/04/22  
C               If 3d, and velocity field has more traces than seismic,
C               read past the last traces in the velocity field record
C               before beginning the next record.
C  REVISED BY:  Gary Murphy        V5.5       REVISION DATE: 97/01/13 
C               Bumped limit for number of samples.
C  REVISED BY:  Jerry Ehlers       V6.0       REVISION DATE: 99/07/28
C               Use sinc interpolation & total rewrite
C  REVISED BY:  Jerry Ehlers       V6.1       REVISION DATE: 00/02/16
C               Add -4d and -qc options
C  REVISED BY:  Jerry Ehlers       V6.1       REVISION DATE: 00/05/19
C               Make running sums double precision
C  Ported to USP by: Paul Garossino  V6.1     Revision Date: Sept 1, 2003
c                    All future support will be from 
c                    USP crew as support for old MBS code is failing.
c                    Created USP style error codes, fixed UnitSc/SmpInt
c                    dependencies, created updated man page and pattern
c                    file.  Got rid of I star 2 variables
c  REVISED BY: Paul G.A. Garossino   V6.1     REVISION DATE: Sept 8,2003
c              fixed hard assignment to word 106,107 assuming I*2 to
c              savew2[] calls.  Former was entering crap in the datastream.
c
c  REVISED BY: Paul G.A. Garossino   V6.1     REVISION DATE: Sept 22,2003
c              if velocity dataset and dataset of interest have coincident
c              dead traces simply put out a dead trace, do not crash as 
c              is the current behaviour.
c
c  REVISED BY: Paul G.A. Garossino   V6.1     REVISION DATE: Sept 30,2003
c              depricated -lhdt.  must now have both -dt and -dz defined
c              on the command line
c
c
c  REVISED BY: Paul G.A. Garossino   V6.1     REVISION DATE: Oct 21,2003
c              depricated the check and resample of the velocity dataset
c              should dz2 be found different than dz.  In this new version
c              -dt and -dz are given on the command line and used.  What
c              is in the dataset DZ1000 slot is looked at, and complaints
c              are registered as warnings if they differ from the command
c              line but the command line sample rates are enforced.
c
C***********************************************************************

      implicit none

#include <localsys.h>
#include <f77/iounit.h>
#include <f77/lhdrsz.h>
#include <f77/sisdef.h>
#include <save_defs.h> 
      
      integer mxsam, lhead, lcrd, llist, lmxc, mxxxx, nmax, init

      parameter (mxsam=32000,lhead=szlnhd)
      parameter (lcrd=25,llist=27,lmxc=28)
      parameter (mxxxx=mxsam+itrwrd,nmax=8)
      parameter (init=1)

      integer ihead(lhead),ivhd(lhead)
      integer itr(szlnhd)
      integer irx(lntrhd),ivx(lntrhd)

      integer pipe

      real w(mxsam,-nmax:nmax),iz(mxsam),zz(4*mxsam),tmp(mxsam)
      real tablz(mxsam),tabl1(mxsam),tabl2(mxsam)
      real rxx(mxxxx),data(mxsam)
      real vel(mxxxx),veld(mxsam)

      real UnitSc, UnitSc_V
      real*8    sum
c     
      logical verbos,threed,fourd,vtin

      character*1 parr(66)
      character*2 pname
      character*4 version
      character*4 ppname
      character*128 ntap,otap,nvel

c     variables required for header interaction

      integer ifmt_RecNum, l_RecNum, ln_RecNum
      integer ifmt_TrcNum, l_TrcNum, ln_TrcNum
      integer ifmt_DphInd, l_DphInd, ln_DphInd
      integer ifmt_LinInd, l_LinInd, ln_LinInd
      integer ifmt_StaCor, l_StaCor, ln_StaCor, StaCor, v_StaCor
      

c variables picked off by implicit none

      integer icode, lpprt, ierr, iflag, itype, ipipi, ipipo
      integer nsinc, jerr, nlin, lu1, lu2, lu3, jeof
      integer ksamp, ltr, isi, nrec, iform, idx, idz
      integer len, jeof2, nqc, ksampv, isiv, nrecv, iformv
      integer ntr, mrec, ntrv, idtms, itmax, ksin, nzout, nout, ntout
      integer lbyout, newksin, k, ltrv, ncount, mr, l, idi, ili, jeofv
      integer idiv, iliv, nzvt, ind, ltabl2, i, lword, kk, jbytes, mmm

      integer dzo_count

      real  dzo, unitsco

      real dx, dx1, dx2, dz, dz1, dz2, dtms, zmax, tmax, dt, dtin, tm
      real dold, dnew, sold, res, snew 

      logical done
      
c     
      equivalence (ppname,pname)
      equivalence (ihead(1),itr(1))
      equivalence (rxx(1),irx(1)),(rxx(ithwp1),data(1))
      equivalence (vel(1),ivx(1)),(vel(ithwp1),veld(1))

      
      data version/' 6.1'/
      data ppname/'VTTD'/
      data parr/
     1     ' ',' ',' ',' ',' ',' ',' ',' ',' ',' ',
     2     'V','E','L','O','C','I','T','Y',' ','T',
     3     'A','P','E',' ','T','I','M','E','/','D',
     4     'E','P','T','H',' ','C','O','N','V','E',
     5     'R','S','I','O','N',' ',' ',' ',' ',' ',
     6     ' ',' ',' ',' ',' ',' ',' ',' ',' ',' ',
     7     ' ',' ',' ',' ',' ',' '/
      data pipe/3/
      data icode/0/
      data lpprt/0/
      data ierr/0/

c parse command line

      call cmdlin( ntap, otap, nvel, iflag, dx, dz, dtms, zmax, tmax,
     :     itype, ipipi, ipipo, ler, threed, fourd, vtin, nsinc, 
     :     nqc, verbos )

cccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc
c     open printfile
cccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc

      call openpr(llist,lerr,ppname,jerr)
      if (jerr.ne.0) then
         write(ler,*) 'unable to open print files.'
         stop 200
      endif
      nlin=1

c print the torch and oval...nice touch, eh.
      call gamoco(parr,nlin,lerr)

cccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc

      if (ipipi.eq.0) then
         call lbopen(lu1,ntap,'r')
      else
         lu1=0
      endif

      if (ipipo.eq.0) then
         call lbopen(lu2,otap,'w')
      else
         lu2=1 
      endif
c     
c     open velocity tape
c

      if (nvel.ne.' ') then
         call lbopen(lu3,nvel,'r')
      elseif (threed .or. fourd) then
         call sisfdfit (lu3,pipe)
         write(LERR,*)'vttd assumed to be running inside IKP'
      else
         write(lerr,*)' '
         write(lerr,*)' VTTD: You cannot pipe the velocity data'
         write(lerr,*)'       into this routine unless using the'
         write(lerr,*)'       -3d or -4d option inside XIKP.'
         write(lerr,*)'       No backing up on pipes allowed.'
         write(lerr,*)' FATAL'
         write(LER,*)' '
         write(LER,*)' VTTD: You cannot pipe the velocity data'
         write(LER,*)'       into this routine unless using the'
         write(LER,*)'       -3d or -4d option inside XIKP. '
         write(LER,*)'       No backing upon pipes allowed.'
         write(LER,*)' FATAL'
         stop
      endif

      write(lerr,8001) ntap,otap,nvel
 8001 format(' INPUT DATASET = ',/,A128,/,' OUTPUT DATASET = '/,A128/,
     *     ' VELOCITY DATASET ='/,A128)

cccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc
c     check for command line arguments
ccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc

      if ((iflag.eq.0).and.(dz.eq.0.0)
     *     .and.(zmax.le.0.0).and.(tmax.le.0.0)) then
         write(lerr,*)'  all parameters have been defaulted'
         write(lerr,*)'  JOB TERMINATED...................'
         write(LER,*)' '
         write(LER,*)' VTTD: You cannot default all parameters'
         write(LER,*)' FATAL'
         stop 50
      elseif (dz.le.0.0) then
         write(lerr,*)' dz must be > 0.0 - JOB TERMINATED '
         write(LER,*)' '
         write(LER,*)' VTTD: -dz[] must be > 0.0 '
         write(LER,*)' FATAL'
         stop 100
      elseif (dtms.le.0.0) then
         write(lerr,*)' dt must be > 0.0 - JOB TERMINATED '
         write(LER,*)' '
         write(LER,*)' VTTD: -dt[] must be > 0.0 '
         write(LER,*)' FATAL'
         stop 100
      endif

      if (nsinc.gt.nmax) then
         write(lerr,*)' nsn must be <= 8 - JOB TERMINATED '
         write(LER,*)' '
         write(LER,*)' VTTD: -nsn[] must be <= 8 '
         write(LER,*)' FATAL'
         stop 110
      endif

      if (fourd) then
         if (threed) then
            write(lerr,*)
     *         ' Cannot have both -3d and -4d options - JOB TERMINATED '
            write(LER,*)' '
            write(LER,*)' VTTD: both -3d and -4d flags present'
            write(LER,*)' FATAL'
            stop 110
         endif

         write(ler,*) "running in 4d mode"
         write (lerr,*) "running in 4d mode"
      elseif (threed) then
         write(ler,*) "running in 3d mode"
         write (lerr,*) "running in 3d mode"
      endif

      iflag=max(1,min(2,iflag))
      itype=max(1,min(2,itype))

c set up pointers to header mnemonic RecNum, TrcNum

      call savelu ( 'RecNum', ifmt_RecNum, l_RecNum, ln_RecNum, 
     :     TRACEHEADER )
      call savelu ( 'TrcNum', ifmt_TrcNum, l_TrcNum, ln_TrcNum, 
     :     TRACEHEADER )
      call savelu ( 'RecNum', ifmt_RecNum, l_RecNum, ln_RecNum, 
     :     TRACEHEADER )
      call savelu ( 'DphInd', ifmt_DphInd, l_DphInd, ln_DphInd, 
     :     TRACEHEADER )
      call savelu ( 'LinInd', ifmt_LinInd, l_LinInd, ln_LinInd, 
     :     TRACEHEADER )
      call savelu ( 'StaCor', ifmt_StaCor, l_StaCor, ln_StaCor, 
     :     TRACEHEADER )

ccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc
c     read the lineheader of the tape to be converted
ccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc

      jeof = 0
      call rtape(lu1,ihead,jeof)
      if (jeof.eq.0) goto 1000

      call saver(ihead,'NumSmp',KSAMP,LINHED)
      call saver(ihead,'NumTrc',ltr,LINHED)
      call saver(ihead,'SmpInt',ISI,LINHED)
      call saver(ihead,'NumRec',NREC,LINHED)
      call saver(ihead,'Format',IFORM,LINHED)
      call saver(ihead,'UnitSc',UnitSc,LINHED)

c read MBS heritage dx and dz header locations from input

      call saver(ihead,'Dx1000',idx,LINHED)
      call saver(ihead,'Dz1000',idz,LINHED)

c     dx1 = trace spacing of input tape if available
c     dz1 = depth spacing of input tape if available

      dx1 = idx/1000.0
      dz1 = idz/1000.0

      if ( UnitSc .eq. 0.0 ) then
         if ( iflag .eq. 1 ) then
c time data coming in with undefined unit scalar, set to ms
c and warn user
            write(LERR,*)'WARNING: '
            write(LERR,*)' sample unit scaler in LH = ',UnitSc
            write(LERR,*)' will set to .001 (millisec default)'
            UnitSc = 0.001
         else
c depth data coming in with no unit scalar, set to feet [meters]
c and warn user
            write(LERR,*)'WARNING: '
            write(LERR,*)' sample unit scaler in LH = ',UnitSc
            write(LERR,*)' will set to 1.0 [feet or meter default]'
            UnitSc = 1.0
         endif
      endif

c echo HLH to printout file
      len=4
      call hlhprt(ihead,jeof,ppname,len,lerr)

      if (ksamp.gt.mxsam) then
         write(lerr,*)' ERROR*****max # samples allowed is ',mxsam
         write(LER,*)' '
         write(LER,*)' VTTD: you have asked for process '
         write(LER,*)        ksamp,' samples. Maximum allowed is ',mxsam
         write(LER,*)' FATAL'
         stop 100
      endif

ccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc
c     read the velocity tape lineheader
ccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc

      jeof2 = 0
      call rtape(lu3,ivhd,jeof2)
      if (jeof2.eq.0) goto 1020

      call saver(ivhd,'NumSmp',KSAMPV,LINHED)
      call saver(ivhd,'NumTrc',ltrV,LINHED)
      call saver(ivhd,'SmpInt',ISIV,LINHED)
      call saver(ivhd,'NumRec',NRECV,LINHED)
      call saver(ivhd,'Format',IFORMV,LINHED)
      call saver(ivhd,'Dx1000',idx,LINHED)
      call saver(ivhd,'Dz1000',idz,LINHED)
      call saver(ivhd,'UnitSc',UnitSc_V,LINHED)

c     dx2 = trace spacing of the velocity field to use if available
c     dz2 = depth spacing of the velocity field to use if available

      dx2 = idx/1000.0
      dz2 = idz/1000.0

      if ( UnitSc_V .eq. 0.0 ) then
         if ( itype .eq. 2 ) then
c velocity time data coming in with undefined unit scalar, set to ms
c and warn user
            write(LERR,*)'WARNING: '
            write(LERR,*)' vel sample unit scaler in LH = ',UnitSc_V
            write(LERR,*)' will set to .001 (millisec default)'
            UnitSc_V = 0.001
         elseif ( itype .eq. 1 ) then
c depth data coming in with no unit scalar, set to feet [meters]
c and warn user
            write(LERR,*)'WARNING: '
            write(LERR,*)' vel sample unit scaler in LH = ',UnitSc_V
            write(LERR,*)' will set to 1.0 [feet or meter default]'
            UnitSc_V = 1.0
         endif
      endif

      if ( dz2 .le. 0.0 ) dz2 = float(isiv) * UnitSc_V

      if (threed.and.nrecv.le.1) then

         write(lerr,*)'******NOTICE******'
         write(lerr,*)' You have asked for the -3D option, but have',
     *                ' only one record on your velocity field'
         write(LER,*)'VTTD:'
         write(ler,*)' You have asked for the -3D option, but have'
         write(ler,*)' only one record on your velocity field'
         write(LER,*)'WARNING'

      elseif (fourd.and.nrecv*ltrv.ne.1.and.nrecv*ltrv.ne.nrec) then

         write(lerr,*)'******NOTICE******'
         write(lerr,*)' You have asked for the -4D option, but have'
         write(lerr,*)' more than 1 velocity trace but not equal to'
         write(lerr,*)' nrec of seismic data'
         write(LER,*)'VTTD:'
         write(ler,*)' You have asked for the -4D option, but have'
         write(ler,*)' more than 1 velocity trace but not equal to'
         write(ler,*)' nrec of seismic data'
         write(LER,*)'WARNING'

      endif

ccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc
c     check the parameters
c     if 1-trace records are coming in, let ntr=nrec and output only 1
c     record with ntr traces
c
c
c it seems since we in USP are now maintaining this routine that should
c 1-trace records be coming in, 1-trace records should go out.  If multi
c trace records are coming in [i.e. MBS standard] then multi-trace records
c should go out........what could it hurt .... pgag################
ccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc
c     dataset to convert:

      ntr =ltr
      mrec=nrec
      if (ltr.eq.1.and..not.fourd) then
         ntr =nrec
         mrec=1
         call savew(ihead,'NumTrc',ntr,linhed)
         call savew(ihead,'NumRec',mrec,linhed)
      endif

c     check dx1000 and dz1000 and put in line header (and ILClIn if 3d)

      if (dx1.ne.dx2) then
         write(lerr,*)'trace spacing on the input tapes do not match'
         write(LER,*)'VTTD:'
         write(ler,*)' trace spacing on the input tapes do not match'
         write(LER,*)'WARNING'
      endif

      if (dx.le.0.0) dx = dx2
      if (dx.le.0.0) dx = dx1
      if (dx.gt.0.0) then
         idx = dx * 1000.0
         call savew(ihead,'Dx1000',idx,linhed)
         if (threed) then
            call savew(ihead,'ILClIn',dx,linhed)
         endif

         if (dx.ne.dx1) then
            write(lerr,*)'******NOTICE******'
            write(lerr,*)'The trace spacing of the output does not',
     *           ' match the trace spacing of the input tape.'
            write(LER,*)'VTTD:'
            write(ler,*)' The trace spacing of the output does not'
            write(ler,*)' match the trace spacing of the input dataset.'
            write(LER,*)'WARNING'
         endif

         if (dx.ne.dx2) then
            write(lerr,*)'******NOTICE******'
            write(lerr,*)'The trace spacing of the output does not',
     *           ' match the trace spacing of the velocity field.'
           write(LER,*)'VTTD:'
           write(ler,*)' The trace spacing of the output does not'
           write(ler,*)' match the trace spacing of the velocity field.'
           write(LER,*)'WARNING'
         endif
      endif

      if (iflag.eq.1) then

         idz = dz * 1000.0
         call savew(ihead,'Dz1000',idz,LINHED) 

c here is an algorithm to scale UnitSc and SmpInt appropriately for output
c This code demands that you enter both -dz and -dt on the command line and
c reads them both as real variables.  Assuming someone has put in a floating
c point entry behind the decimal place we will need to back that out

         dzo = dz
         dzo_count = 0
         done = .false.
         unitsco = 1.0

         do while ( .not. done .and. dzo_count .le. 4 )
            if ( amod ( dzo,1.0 ) .gt. 0 ) then
               dzo = dzo * 10.
               unitsco = unitsco / 10.
               dzo_count = dzo_count + 1
            else
               done = .true.
            endif
         enddo

         idz = nint ( dzo )
         call savew ( ihead,'SmpInt', idz, LINHED )
         call savew ( ihead,'UnitSc', unitsco, LINHED )

      endif

C     dataset containing velocity field:

      if (ltrv.gt.1) then
         ntrv=ltrv
      else
         ntrv=max(1,nrecv)
      endif

      if (fourd) then
         ntrv=1
         if (ltrv*nrecv.eq.1.and.nrec.gt.1) then

            write(LER,*)'VTTD:'
            write(ler ,*)' The dataset containing the velocity field'
            write(ler ,*)' has only one which will be used for the'
            write(ler ,*)' entire conversion.'
            write(LER,*)'WARNING'
            write(lerr,*)' ******WARNING******'
            write(lerr,*)' The dataset containing the velocity field'
            write(lerr,*)' has only one which will be used for the'
            write(lerr,*)' entire conversion.'

         endif
      elseif (ntrv.lt.ntr.and.ntrv.eq.1) then

         write(LER,*)'VTTD:'
         write(ler ,*)' The dataset containing the velocity field'
         write(ler ,*)' has only one which will be used for the'
         write(ler ,*)' entire conversion.'
         write(LER,*)'WARNING'
         write(lerr,*)' ******WARNING******'
         write(lerr,*)' The dataset containing the velocity field'
         write(lerr,*)' has only one which will be used for the'
         write(lerr,*)' entire conversion.'

      elseif (ntrv.lt.ntr) then

         write(LER,*)'VTTD:'
         write(ler ,*)' The dataset containing the velocity field'
         write(ler ,*)' contains less traces than the dataset to be'
         write(ler ,*)' converted.  The last trace will be used '
         write(ler ,*)' multiple times.  For more accurate results, use' 
         write(ler ,*)' vtrsiz to resize the velocity field dataset.'
         write(LER,*)'WARNING'

         write(lerr,*)' ******WARNING******'
         write(lerr,*)' The dataset containing the velocity field'
         write(lerr,*)' contains less traces than the dataset to be'
         write(lerr,*)' converted.  The last trace will be used '
         write(lerr,*)' multiple times.  For more accurate results, use'
         write(lerr,*)' vtrsiz to resize the velocity field dataset.'

      elseif (ntrv.gt.ntr) then

         write(LER,*)'VTTD:'
         write(ler ,*)' ******NOTICE******'
         write(ler ,*)' The dataset containing the velocity field '
         write(ler ,*)' contains more traces than the dataset to be'
         write(ler ,*)' converted.  Only the first ',ltr,' traces ' 
         write(ler ,*)' will be used.' 
         write(LER,*)'WARNING'

         write(lerr,*)' ******NOTICE******'
         write(lerr,*)' The dataset containing the velocity field '
         write(lerr,*)' contains more traces than the dataset to be'
         write(lerr,*)' converted.  Only the first ',ltr,' traces ' 
         write(lerr,*)' will be used.' 
      endif

      if (ksampv.gt.mxsam) then
         write(lerr,*)' ERROR*****max # samples allowed is ',mxsam
         write(LER,*)' VTTD: you have asked for process '
         write(LER,*)        ksampv,' samples. Maximum allowed is '
     :        ,mxsam
         write(LER,*)' FATAL'
         stop 100
      endif
      
      IF (iflag.eq.1) THEN

ccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc
c     
c     time to depth conversion  (iflag=1)
c     
ccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc

         if (zmax.le.0.0) then
            write(lerr,*)' MAXIMUM DEPTH MUST BE ENTERED'
            write(LER,*)'VTTD:' 
            write(LER,*)' maximum depth must be entered'
            write(LER,*)' fix -z[] entry on command line and try again'
            write(LER,*)'FATAL'
            stop 100
         endif

c if we made it this far then we have a delta time for the input dataset
c either from the command line or from the input dataset

         idtms=dtms
         dt=dtms/1000.0
         dtin = dt

c if user asks to use more time in than we have, we will use
c only as much as we have. Hmmm does not look like the MBS 
c crowd paid attention to sample one being from time zero    

         if (tmax.le.0.0) tmax = dtms * ksamp
         tm = dtms * ksamp
         tmax = min(tm,tmax)
         itmax = tmax
         ksin = itmax/idtms
         

c Remember, there was a check done at the start of the program
c to guard against dz being <= 0.0 at this point.

         nzout = zmax/dz + 1
         nout = nzout
         idz = dz

c load output line header entries for number of samples, sample
c interval and unit scalar.  Since it is depth out, unit scalar
c is set to feet or meters.

         call savew(ihead,'NumSmp',nzout,linhed)
         call savew(ihead,'SmpInt',idz,linhed)
         call savew(ihead,'UnitSc',1.0,linhed)

         if (verbos) then
            write(lerr,*) ' tm,dtms,ksamp=',tm,dtms,ksamp
            write(lerr,*) ' itmax,tmax,idtms,ksin=', itmax,tmax,
     *           idtms,ksin
            write(lerr,*) ' dt= ',dt
         endif

         if (vtin) then

c data being converted is velocity data

            write(lerr,8011) idtms,dx1,itmax,ksin,dx,dz,zmax,nzout
         else
            write(lerr,8111) idtms,dx1,itmax,ksin,dx,dz,zmax,nzout
         endif

         if (itype.ge.2) then

c input velocity data is time data

            write(lerr,8121) dx2,nrecv,ltrv,ksampv

            if (isiv.ne.idtms) then
               write(lerr,*)'WARNING--the delta-t of the velocity ',
     *              'dataset (',isiv,') does not match that of the ',
     *              'input dataset (',idtms,')'
               write(LER,*)'VTTD:' 
               write(LER,*)' delta-t of the velocity dataset (',isiv
               write(LER,*)') does not match that of the input'
               write(LER,*)' (',idtms,')'
               write(LER,*)'WARNING'
            endif
         else

c input velocity data is depth data

            write(lerr,8131) dx2,dz2,nrecv,ltrv,ksampv

            if (dz.ne.dz2) then
               write(lerr,*)'******NOTICE******'
               write(lerr,*)'The depth spacing of the output does',
     *              ' not match that of the velocity field.'
               write(LER,*)'VTTD:' 
               write(ler,*)' The depth spacing of the output does'
               write(ler,*)' not match that of the velocity field.'
               write(LER,*)'WARNING'
            endif
         endif  

c format statements

 8011    format(
     *     ' THE INPUT DATASET CONTAINS VELOCITY DATA TO BE CONVERTED',
     *     ' FROM TIME TO DEPTH'/,
     *     '   Input spacing in milliseconds          ', 10X,'=', I15 ,/
     *     '   Input trace spacing in feet/meters     ', 10X,'=',F15.5,/
     *     '   Maximum time of input to use           ', 10X,'=', I15 ,/
     *     '   Number of samples on input to use      ', 10X,'=', I15 ,/
     *     ' THE OUTPUT DATASET CONTAINS VELOCITY DATA IN DEPTH',/
     *     '     Output trace spacing in feet/meters  ', 10x,'=',F15.5,/
     *     '     Output depth spacing in feet/meters  ', 10x,'=',F15.5,/
     *     '     Maximum depth on output              ', 10x,'=',F15.5,/
     *     '     Number of samples on output          ', 10x,'=', I15)
 8111    format(
     *     ' THE INPUT DATASET CONTAINS SEISMIC DATA TO BE CONVERTED',
     *     ' FROM TIME TO DEPTH',/,
     *     '   Input spacing in milliseconds          ', 10X,'=', I15 ,/
     *     '   Input trace spacing in feet/meters     ', 10X,'=',F15.5,/
     *     '   Maximum time of input to use           ', 10X,'=', I15 ,/
     *     '   Number of samples on input to use      ', 10X,'=', I15 ,/
     *     ' THE OUTPUT DATASET CONTAINS SEISMIC DATA IN DEPTH',/
     *     '     Output trace spacing in feet/meters  ', 10x,'=',F15.5,/
     *     '     Output depth spacing in feet/meters  ', 10x,'=',F15.5,/
     *     '     Maximum depth on output              ', 10x,'=',F15.5,/
     *     '     Number of samples on output          ', 10x,'=', I15)
 8121    format(
     *     ' THE INPUT VELOCITY FIELD IS IN TIME',/
     *     '       Trace spacing in feet/meters       ', 10x,'=',F15.5,/
     *     '       Number of records                  ', 10x,'=',I15,/  
     *     '       Number of traces                   ', 10x,'=',I15,/  
     *     '       Number of samples                  ', 10x,'=',I15)  
 8131    format(
     *     ' THE INPUT VELOCITY FIELD IS IN DEPTH',/
     *     '       Trace spacing in feet/meters       ', 10x,'=',F15.5,/
     *     '       Depth spacing in feet/meters       ', 10x,'=',F15.5,/
     *     '       Number of records                  ', 10x,'=',I15,/  
     *     '       Number of traces                   ', 10x,'=',I15,/  
     *     '       Number of samples                  ', 10x,'=',I15)  

      ELSE

ccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc
c     
c     depth to time conversion  (iflag=2)
c     
ccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc

        if (tmax.le.0.0) then

c we need to know the max time to convert for as most depth datasets 
c will convert to massive time registries we make it manditory for 
c the user to think about this and enter something sensible on the 
c command line

            write(lerr,*)' MAXIMUM TIME MUST BE ENTERED'
            write(LER,*)'VTTD:' 
            write(LER,*)' maximum time must be entered'
            write(LER,*)' fix -t[] entry on command line and try again'
            write(LER,*)'FATAL'
            stop
         endif

         itmax = tmax

         dt = dtms / 1000.0
         dtin = dt
         idtms = dtms

c dtms has been calculated in milliseconds so hardwire UnitSc to be 0.001
c and write out both sample interval and unit scalar.

         unitsco = 0.001
         call savew(ihead,'SmpInt',idtms,linhed)
         call savew(ihead,'UnitSc',unitsco,linhed)

         if (zmax.le.0.0) then
            zmax=dz*(ksamp-1)
         endif

         ksin = 1.0+zmax/dz
c
c        if user asks to use more samples in than we have, 
c        limit to the number of samples we have.
c
         if (ksin.gt.ksamp) ksin=ksamp
         if (idtms.gt.32) then
            ntout = tmax/dtms+1
         else
            ntout = itmax/idtms+1
         endif
         nout = ntout

         call savew(ihead,'NumSmp',ntout,linhed)

         if (verbos) write(lerr,*)' DT',dt
c
c        check for dz mismatches between velocity(in depth) & input
c
         if (itype.le.1) then
            if (dz1.ne.dz2) then

               write(lerr,*)'******NOTICE******'
               write(lerr,*)'The depth spacing of the input does',
     *              ' not match that of the velocity field.'
               write(LER,*)'VTTD:' 
               write(ler,*)' The depth spacing of the input does'
               write(ler,*)' not match that of the velocity field.'
               write(LER,*)'WARNING' 
            endif
         endif

c        check for dz mismatches between inputs and output

         if (dz.ne.dz1.and.dz.ne.dz2) then
            write(lerr,*)'******NOTICE******'
            write(lerr,*)'The depth spacing of the output does',
     *           ' not match that of the input.'
            write(LER,*)'VTTD:' 
            write(ler,*)' The depth spacing of the output does'
            write(ler,*)' not match that of the input.'
            write(LER,*)'WARNING' 
         endif

         if (vtin) then
            write(lerr,8012) dz,dx1,zmax,ksin,dx,idtms,itmax,ntout
         else
            write(lerr,8112) dz,dx1,zmax,ksin,dx,idtms,itmax,ntout
         endif

         write(lerr,8036) mrec,ntr

         if (itype.ge.2) then
            write(lerr,8121) dx2,nrecv,ltrv,ksampv
         else
            write(lerr,8131) dx2,dz2,nrecv,ltrv,ksampv
         endif 

c format statements

 8012    format(
     *     ' THE INPUT DATASET CONTAINS VELOCITY DATA TO BE CONVERTED',
     *     ' FROM DEPTH TO TIME',/
     *     '   Input depth spacing in feet/meters     ', 10X,'=',F15.5,/
     *     '   Input trace spacing in feet/meters     ', 10X,'=',F15.5,/
     *     '   Maximum depth of input to use          ', 10X,'=',F15.5,/
     *     '   Number of samples on input to use      ', 10X,'=',I15  ,/
     *     ' THE OUTPUT DATASET CONTAINS VELOCITY DATA IN TIME',/
     *     '     Output trace spacing in feet/meters  ', 10x,'=',F15.5,/
     *     '     Output time spacing in milliseconds  ', 10x,'=',I15  ,/
     *     '     Maximum time on output               ', 10x,'=',I15  ,/
     *     '     Number of samples on output          ', 10x,'=',I15)
 8112    format(
     *     ' THE INPUT DATASET CONTAINS SEISMIC DATA TO BE CONVERTED',
     *     ' FROM DEPTH TO TIME',/
     *     '   Input depth spacing in feet/meters     ', 10x,'=',F15.5,/
     *     '   Input trace spacing in feet/meters     ', 10X,'=',F15.5,/
     *     '   Maximum depth of input to use          ', 10x,'=',F15.5,/
     *     '   Number of samples on input to use      ', 10x,'=',I15  ,/
     *     ' THE OUTPUT DATASET CONTAINS SEISMIC DATA IN TIME',/
     *     '     Output trace spacing in feet/meters  ', 10x,'=',F15.5,/
     *     '     Output time spacing in milliseconds  ', 10x,'=',I15  ,/
     *     '     Maximum time on output               ', 10x,'=',I15  ,/
     *     '     Number of samples on output          ', 10x,'=',I15)
 8036    format(
     *     '     Number of records on output          ', 10x,'=', I15,/,
     *     '     Number of traces per record on output', 10x,'=', I15)
      endif
c
c     put command line into the historical part of line header
c
      call savhlh(itr,jeof,lbyout)

c write output line header

      call wrtape(lu2,ihead,lbyout)
c      
ccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc
c
c     start processing dataset                                       
c      
c     build input and output time tables for resampling input data
c      
ccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc
c      
c     time to depth conversion (iflag=1)
c     depth to time conversion (iflag=2)
c     determine amount of oversampling desired
c     build tables for interpolation
c      
      if (iflag.eq.1) then
         dold=dt
         if (vtin) then
            newksin=ksin
         else
            dt=min(0.001,dt/2.0)
            newksin=tmax/(1000.0*dt)
         endif
         dnew=dt
         do k=1,mxsam
            tabl2(k) = (k-1)*dz
            tablz(k) = (k-1)*dz
         enddo
      else
         dold=dz
         if (vtin) then
            newksin=ksin
         else
            dz=dz/2.0
            newksin=1.0+zmax/dz
         endif
         dnew=dz
         do k=1,mxsam
            tabl2(k) = (k-1)*dt
            tablz(k) = (k-1)*dz
         enddo
      endif
     
c     are the arrays long enough to handle this?  (mxsam)
     
      if (newksin.gt.mxsam) then
         write(lerr,*) 'Check Point 1: newksin',newksin,' >',mxsam
         if (iflag.eq.1) then
            write(lerr,*) '   tmax,dt,dz=',tmax,dt,dz
         else
            write(lerr,*) '   zmax,dt,dz=',zmax,dt,dz
         endif
         goto 2000
      endif

c.......................................................................
c     loop over input records
c.......................................................................

      ncount=0

      do mr=1,mrec

         if (verbos) write(lerr,*) '  PROCESSING RECORD ',mr

c......................................................................
c        loop over input traces
c.......................................................................

         do l=1,ntr

c......................................................................
c           read next input trace.
c.......................................................................

            jeof=0
            call rtape(lu1,rxx,jeof)
            if (jeof.eq.0) goto 1500
            call saver2(irx, ifmt_DphInd, l_DphInd, ln_DphInd, idi, 
     :           TRACEHEADER)
            call saver2(irx, ifmt_LinInd, l_LinInd, ln_LinInd, ili, 
     :           TRACEHEADER)
            call saver2(irx, ifmt_StaCor, l_StaCor, ln_StaCor, StaCor, 
     :           TRACEHEADER)

c.......................................................................
c           read next velocity trace
c
c           if 4d: turn off reading anymore if only 1 trace (ntrv=0)
c.......................................................................

            if (l.le.ntrv.and.mr.le.nrecv) then
               if (fourd.and.ltr*nrecv.eq.1) ntrv=0

               jeofv = 0
               call rtape(lu3,vel,jeofv)
               if (jeofv.eq.0) goto 1520
               call saver2(irx, ifmt_DphInd, l_DphInd, ln_DphInd, idiv, 
     :              TRACEHEADER)
               call saver2(irx, ifmt_LinInd, l_LinInd, ln_LinInd, iliv, 
     :              TRACEHEADER)
               call saver2(irx, ifmt_StaCor, l_StaCor, ln_StaCor, 
     :              v_StaCor, TRACEHEADER)

c if velocity trace is flagged dead, check to make sure incoming data 
c trace is also dead.  If not then out you go, if so then output a dead
c trace and keep going.  Of course if this is a fourd pass then all bets
c are off and out you go.....pgag

               if ( v_StaCor .eq. 30000 ) then
                  if ( StaCor .ne. 30000 .or. fourd ) then

c you have a live trace with no live velocity data to support it....goodbye.
c or you have a dead velocity trace in a fourd app .... goodbye again.

                     write(LERR,*)' you have a dead velocity'
                     write(LERR,*)' trace at sequential record ',mr
                     write(LERR,*)' sequential trace ',l
                     write(LERR,*)'FATAL'
                     write(LER,*)' '
                     write(LER,*)'VTTD:'
                     write(LER,*)' you have a dead velocity'
                     write(LER,*)' trace at sequential record ',mr
                     write(LER,*)' sequential trace ',l
                     write(LER,*)'FATAL'
                     write(LER,*)' '
                     stop
                  else
                     goto 10
                  endif
               endif


               if (itype.ge.2) then
c     
c                 convert input velocity from time to depth
c     
                  sum=0.0
                  do k=1,ksampv
                     tabl1(k)=sum
                     sum=sum+veld(k)*dtin/2.0
                  enddo 
                  nzvt = tabl1(ksampv)/dz + 1.0
                  if (nzvt.gt.mxsam) then
                     write(lerr,*) 'Check Point 2: nzvt',nzvt,' >',mxsam
                     write(lerr,*) '   dtin,dz=',dtin,dz
                     goto 2000
                  endif
c     
                  call fliint(tabl1,veld,ksampv,tablz,tmp,nzvt,
     *                 iz,zz,init)
                  call vmov(tmp,1,veld,1,nzvt)
c     
c                 extended the velocity
c     
                  do k=nzvt+1,mxsam
                     veld(k) = veld(nzvt)
                  enddo
               else
c     
c                 interpolate velocity to oversample delta, 
c                 extended last sample

c                 since we now enforce that dt and dz must 
c                 both be on the command line and are to be honoured
c                 then we make the assumption that the user has
c                 attached a velocity dataset of appropriate sampling
c                 if not then the output will be wrong but many 
c                 warnings have been given about this already.
c  


c                  if (dz2.ne.dz) then
c
c                     do k=1,ksampv
c                        tabl1(k)=(k-1)*dz2
c                     enddo
c                     nzvt = tabl1(ksampv)/dz + 1.0
c                     if (nzvt.gt.mxsam) then
c                        write(lerr,*) 'Check Point 3: nzvt',nzvt,' >',
c     *                       mxsam
c                        write(lerr,*) '   dz,dz2=',dz,dz2
c                        goto 2000
c                     endif
c     
c                     call fliint(tabl1,veld,ksampv,tablz,tmp,nzvt,
c     *                    iz,zz,init)
c                     call vmov(tmp,1,veld,1,nzvt)
c                  else
                     nzvt=ksampv
c                  endif
                  do k=nzvt+1,mxsam
                     veld(k) = veld(nzvt)
                  enddo
               endif
c     
c              build input table (tabl1) for converting input
c              by calculating dz's in terms of dt's
c              or by calculating dt's in terms of dz's
c     
               k=1
               sum=0.0
               tabl1(k)=sum
               sold=1.0/veld(1)
               if (iflag.eq.1) then
                  do while(tabl1(k).lt.zmax.and.k.lt.mxsam)
                     k=k+1
                     res=(tabl1(k-1)+0.5*dt/sold)/dz+1.0
                     ind=res
                     res=res-ind
                     if (ind.ge.mxsam) then
                        write(lerr,*) 'Check Point 4: ind',ind,' >',
     *                       mxsam
                        write(lerr,*) '   dt,dz=',dt,dz
                        goto 2000
                     endif
c     
c                    check divisor for zero before doing the divide
c     
                     if (veld(ind).le.0.0.or.veld(ind+1).le.0.0) then
                        write(LER,*)'VTTD:'
                        write(ler,*)' Unflagged dead trace? '
                        write(ler,*)' velocity <= zero on trace ',l
                        write(ler,*)' sample',ind,' record ',mr
                        write(ler,*)'FATAL'
                        write(lerr,*) 'velocity <= zero on trace ',l,
     *                       ' sample',ind,' record ',mr
                        write(lerr,*) 'Job terminated'
                        icode = 200
                        goto 5000
                     endif
                     snew=(1-res)/veld(ind)+res/veld(ind+1)
                     sum=sum+0.5*dt*(1.0/sold+1.0/snew)/2.0
                     sold=snew
                     tabl1(k)=sum
                  enddo
                  len=k
                  ltabl2=1.0+tabl1(len)/dz
               else
                  do while(tabl1(k).lt.tmax/1000.0.and.k.lt.mxsam)
                     k=k+1
c     
c                    check divisor for 0 value before dividing
c     
                     if (veld(k).le.0.0) then
                        write(LER,*)'VTTD:'
                        write(ler,*)' velocity < zero on trace ',l
                        write(ler,*)' sample',k,' record ',mr
                        write(ler,*)'FATAL'
                        write(lerr,*) 'velocity < zero on trace ',l,
     *                       ' sample ',k,' record ',mr
                        write(lerr,*) 'job terminated'
                        icode = 200
                        goto 5000
                     endif
                     snew=1.0/veld(k)
                     sum=sum+2.0*dz*(sold+snew)/2.0
                     sold=snew
                     tabl1(k)=sum
                  enddo
                  len = k
                  ltabl2 = 1.0+tabl1(len)/dt
               endif

               if (ltabl2.gt.mxsam) then
                  write(lerr,*) 'Check Point 5: ltabl2',ltabl2,' >',
     *                 mxsam
                  write(lerr,*) '   dt,dz=',dt,dz
                  goto 2000
               endif
            endif
c.......................................................................
c  
c           Process Data
c
c           1) oversample input trace with sinc function if seismic
c           2) extend trace to "len" samples
c           3) resample with liner for non-seismic & cubic for seismic
c     
c           each output trace is different in length depending upon 
c           the velocities at that x location.
c.......................................................................
            if (nqc.gt.0) then
               if (mod(ncount,nqc).eq.0) then
                  write(lerr,8200) l,ili,idi,iliv,idiv
 8200             format(' QC: trc=',I5,'   Seismic:',2I8,
     *                '   Velocity', 2I8)
               endif
               ncount=ncount+1
            endif
c
            if (vtin) then
               call vmov(data,1,tmp,1,ksin)
               do i=ksin+1,len
                  tmp(i) = tmp(ksin)
               enddo
               call fliint(tabl1,tmp,len,tabl2,data,ltabl2,iz,zz,init)
            else
               call oversamp(data,ksin,dold,tmp,newksin,dnew,nsinc,
     *              iz,zz,w,.false.)
               do i=newksin+1,len
                  tmp(i) = 0.0
               enddo
               call ccuint(tabl1,tmp,len,tabl2,data,ltabl2,iz,zz,init)
            endif
c.......................................................................
c   
c           Write Output
c
c           If trace is shorter than nzout, extend it to nzout.  
c           If converted trace is seismic data, zero extend it
c           else extend the last velocity value
c.......................................................................
            if (ltabl2.lt.nout) then
               lword = nout-ltabl2
               if (vtin) then
                  do kk = 1,lword
                     data(ltabl2+kk) = data(ltabl2)
                  enddo
               else
                  call vclr(data(ltabl2+1),1,lword)
               endif
            endif

 10         continue

            call savew2(irx, ifmt_RecNum, l_RecNum, ln_RecNum, mr, 
     :           TRACEHEADER)
            call savew2(irx, ifmt_TrcNum, l_TrcNum, ln_TrcNum, l, 
     :           TRACEHEADER)
            call savew2(irx, ifmt_StaCor, l_StaCor, ln_StaCor, StaCor, 
     :           TRACEHEADER)

c            irx(106) = mr
c            irx(107) = l
            jbytes = nout*szsmpd + sztrhd
            call wrtape(lu2,rxx,jbytes)
            if (jbytes.eq.0) goto 1600
         enddo
c.......................................................................
c  
c        Reposition Velocity iIput
c
c     
c        if 2d: rewind velocity input if more than one seismic record
c        if 3d: position velocity to beginning of next record
c        if 4d: simply continue on
c.......................................................................
         if (threed) then
            do mmm=ntr+1,ntrv
               jeofv= 0
               call rtape(lu3,vel,jeofv)
               if (jeofv.eq.0) goto 200
            enddo
 200        continue
         elseif (.not.fourd) then
            call rwd(lu3)
            jeof2 = 0
            call rtape(lu3,ivhd,jeof2)
            if (jeof2.eq.0) goto 1020
         endif
      enddo
c.......................................................................
c     finished
c.......................................................................
      write(ler,*) 'vttd: Normal Termination'
      write(ler,*) 'processed: ',mrec, 'records'

      icode = 0
      goto 5000

cccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc
c     error handling                               
c     line header errors                             
cccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc

 1000 write(lerr,1010)
 1010 format(2x,'ERROR reading line header from input data')
      write(LER,*)' '
      write(LER,*)' VTTD: ERROR reading line header from input data'
      write(LER,*)' FATAL'
      call lbclos(lu1)
      call lbclos(lu2)
      stop 75
c
 1020 write(lerr,1030) 
 1030 format(2x,'ERROR reading line header from velocity field')
      write(LER,*)' '
      write(LER,*)' VTTD: ERROR reading line header from velocity field'
      write(LER,*)' FATAL'
      call lbclos(lu1)
      call lbclos(lu2)
      call lbclos(lu3)
      stop 75
cccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc
c     tapeio errors
cccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc
 1500 write(lerr,1510)mr,l
 1510 format(' ERROR reading record',i5,' trace',i5,' input data')
      icode = 75
      goto 5000
c
 1520 write(lerr,1530)mr,l
 1530 format(' ERROR reading record',i5,' trace',i5,' velocity field')
      icode = 75
      goto 5000
c
 1600 write(lerr,1610)mr,l
 1610 format(' ERROR writing output record',i5,' trace',i5)
      icode = 75
      goto 5000
cccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc
c     data longer than array limits of program
cccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc
 2000 write(ler,*)' Internal arrays are not large enough to accomodate'
      write(ler,*)' the amount of time you have asked for on output.'
      write(ler,*)' You may ask for a shorter amount of time [-t] '
      write(ler,*)' and try again.'
      write(ler,*)' *** Job terminated. ***'
      write(lerr,*)' Internal arrays are not large enough to accomodate'
      write(lerr,*)' the amount of time you have asked for on output'
      write(lerr,*)' You may ask for a shorter amount of time [-t] '
      write(lerr,*)' and try again.'
      write(lerr,*)' *** Job terminated. ***'
      icode = 200
cccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc
c     end of job                            
cccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc
 5000 call lbclos(lu1)
      call lbclos(lu2)
      call lbclos(lu3)
      if (icode.eq.200) stop 200
      if (icode.eq.75)  stop 75
      end
cccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc
c
c     subroutine cmdlin                        
c      
cccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc
      subroutine cmdlin(ntap,otap,nvel,iflag,dx,dz,dtms,zmax,tmax,itype,
     *     ipipi,ipipo,ler,threed,fourd,vtin,nsinc,nqc,verbos)
      integer argis,nqc
      logical verbos,threed,fourd,vtin
      character*128 ntap,otap,nvel
c     set defaults to no pipes
      ipipi=0
      ipipo=0
      verbos=.false.
      threed=.false.
      vtin  =.false.
      if (  argis ('-h') .gt. 0  .or. 
     :      argis ( '-?') .gt. 0 .or. 
     :      argis ( '-help') .gt. 0 ) then
         write(ler,*)'      COMMAND LINE ARGUMENTS:'
         write(ler,*)' TIME TO DEPTH CONVERSION OF SEISMIC DATA'
         write(ler,*)' OR VELOCITY DATA USING A VELOCITY FIELD'
         write(ler,*)' '
         write(ler,*)' INPUT '
         write(ler,*)'-N[]   ..INPUT DATASET NAME'
         write(ler,*)'-O[]   ..OUTPUT DATASET NAME'
         write(ler,*)'-VT[]  ..VELOCITY FIELD NAME'
         write(ler,*)'-type[]..VELOCITY FIELD TYPE FLAG'
         write(ler,*)'         (=1: v(z,x), = 2: v(t,x))'
         write(ler,*)'-flag[]..CONVERSION TYPE FLAG       '
         write(ler,*)'         (=1: Time to Depth =2: Depth to Time)'
         write(ler,*)'-vt    ..FLAG INDICATES THE DATA TO BE'
         write(ler,*)'         CONVERTED IS VELOCITY DATA NOT SEISMIC'
         write(ler,*)'          This flag must be present when velocity'
         write(ler,*)'          data is being converted rather than'
         write(ler,*)'          seismic data'
         write(ler,*)'-3d    ..FLAG INDICATES THE DATA IS 3d'
         write(ler,*)'          This flag must be present when a 3d'
         write(ler,*)'          dataset is being converted, and its'
         write(ler,*)'          corresponding velocity dataset is'
         write(ler,*)'          also 3d (multi-record). If in XIKP '
         write(ler,*)'          this can be a pipe.  If outside it '
         write(ler,*)'          can be a named pipe.'
         write(ler,*)'-4d    ..FLAG INDICATES THE DATA IS 4d'
         write(ler,*)'          This flag must be present when a 4d'
         write(ler,*)'          dataset of gathers is being converted,'
         write(ler,*)'          and its corresponding velocity dataset'
         write(ler,*)'          is either a single Vz trace or a Vxyz '
         write(ler,*)'          with ntrc*nrec = nrec of the gathers'
         write(ler,*)'          If in XIKP this can be a pipe.  If '
         write(ler,*)'          outside it can be a named pipe.'
         write(ler,*)'-dt[]  ..SAMPLE RATE(ms) OF THE TIME DATASETS'
         write(ler,*)'          The coordinates of the datasets '
         write(ler,*)'              should match.'
         write(ler,*)'          NO DEFAULT.'
         write(ler,*)'-t[]   ..MAXIMUM TIME OF TIME DATASETS'
         write(ler,*)'          Depth to Time Conversion : NO DEFAULT'
         write(ler,*)'          Time to Depth Conversion default :'
         write(ler,*)'             Maximum time may be calculated using'
         write(ler,*)'             the -dt from the command line'
         write(ler,*)'             and the number of samples from'
         write(ler,*)'             the input line header'
         write(ler,*)'-dx[]  ..DELTA X (Trc spacing) OF OUTPUT DATASET'
         write(ler,*)'          This value*1000 will be placed'
         write(ler,*)'          in the output line header. DEFAULT is'
         write(ler,*)'          the Dx1000 value from input velocity'
         write(ler,*)'          field line header or from the input'
         write(ler,*)'          dataset to be converted.'
         write(ler,*)'-dz[]  ..DELTA Z OF THE DEPTH DATASETS'
         write(ler,*)'           The coordinates of the datasets '
         write(ler,*)'               should match.'
         write(ler,*)'           NO DEFAULT'
         write(ler,*)'-z[]   ..MAXIMUM DEPTH OF DEPTH DATASETS'
         write(ler,*)'           Time to Depth Conversion : NO DEFAULT'
         write(ler,*)'           Depth to Time Conversion default :'
         write(ler,*)'             Maximum depth may be calculated'
         write(ler,*)'             using the -dz parameter from the'
         write(ler,*)'             command line and the number of'
         write(ler,*)'             samples from the input line header'
         write(ler,*)'-nsn[] ..SIZE OF SINC FUNCTION'
         write(ler,*)'           Seismic data oversampling sinc '
         write(ler,*)'           function length (-nsn:nsn) : default 8'
         write(ler,*)'-qc[]  ..FREQUENCY OF QC DUMPS'
         write(ler,*)'           LIs and DIs will be dumped for the '
         write(ler,*)'           Seismic and velocity trace every '
         write(ler,*)'           nqcth seismic trace (default 1000 '
         write(ler,*)'           if only -qc specified) '
         write(ler,*)'-V    .. VERBOSE PRINTOUT'
         write(ler,*)'USAGE:'
         write(ler,*)'vttd -N[] -O[] -VT[] -[vt] -[3d] -[4d] -type[] '
         write(ler,*)'    -flag[]  -dt[] -t[] -dz[] -z[] -nsn[] -v '
         stop
      endif
      call argstr('-N',NTAP,' ',' ')
      call argstr('-O',OTAP,' ',' ')
      call argstr('-VT',NVEL,' ',' ')
      call argI4 ('-flag',iflag,0,0)
      call argI4 ('-type',itype,0,0)
      call argI4 ('-nsn',nsinc,8,8)
      call argI4 ('-qc',nqc,1000,0)
      call argR4 ('-dx',dx,0.0,0.0)
      call argR4 ('-dz',dz,0.0,0.0)
      call argR4 ('-dt',dtms,0.0,0.0)
      call argR4 ('-z',zmax,0.0,0.0)
      call argR4 ('-t',tmax,0.0,0.0)
      verbos = (argis('-V').gt.0)
      vtin   = (argis('-vt').gt.0)
      threed = (argis('-3d').gt.0)
      fourd  = (argis('-4d').gt.0)
c     make the ntap a pipe
      if (ntap.eq.' ') ipipi=1
c     make the otap a pipe
      if (otap.eq.' ') ipipo=1

c check for extraneous arguments and abort if found to
c catch all manner of user typo's

      call xtrarg ( name, ler, .FALSE., .FALSE. )
      call xtrarg ( name, lerr, .FALSE., .TRUE. )


      return
      end
