/***********************************************************************
 *                copyright 2001, Amoco Production Company             *
 *                            All Rights Reserved                      *
 *                    an affiliate of BP America Inc.                  *
 ***********************************************************************/
#define SUN
/* #define CRAY */


#include <stdio.h>
#include <stdarg.h>
#ifdef SUN
#include <sys/file.h>   /* R_OK */
#endif
#ifdef CRAY
#include <sys/unistd.h>   /* R_OK */
#endif
#include <sys/types.h>
#include <time.h>
#include <string.h>
#ifdef USE_SYS_MALLOC
#include <sys/malloc.h>
#else
#include <malloc.h>
#endif
#include <stdlib.h>	/* atof */
#include <unistd.h>	/* R_OK */
#include <ctype.h>  /* isascii, islower */
#include "std.h"
#include "agm_io.h"
#include "agm.h"
#ifdef SUN
#include "agm_datatypes.h"
#include "agm_collection_types.h"
#endif
#ifdef CRAY
#include "agm_da_types.h"
#include "agm_co_types.h"
#endif
#include "agm_tables.h"


#define FLATFILE		1
#define DATABASE		2

#define MAX_MESSAGE_LEN		256
#define MAX_INGRES_ROW_LEN	2000
#define LINEAGE_STRING_LEN	(MAX_INGRES_ROW_LEN - 8 - 4 - 4)
#define MAX_LINE_LEN		LINEAGE_STRING_LEN

#define DATA_SET		-1

typedef struct historytree {
   unsigned int		page;
   String		history;
   struct historytree	*leftchild;
   struct historytree	*rightchild;
} Historytree, *Historytreeptr;

typedef struct versionlist {
   Version		version;
   Historytreeptr	historytree;
   struct versionlist	*next;
} Versionlist, *Versionlistptr;

typedef struct modellist {
   String		modelname;
   Versionlistptr	versionlist;
   struct modellist	*next;
} Modellist, *Modellistptr;


extern int	agm( unsigned command, ... );
extern int	agm_db_read( String );
extern int	agm_db_write( String );

int		agm_io( unsigned command, ... );
int		agm_io_( unsigned *command, ... );
String		agm_io_generate_dsn( void );
void		agm_io_delete_model( String modelname );

static String concat( String s, ... );
static String fstring_to_cstring( va_list args );
static void cstring_to_fstring( va_list args, String string );
static int agm_flatfile_read( String filename );
static int load_attrib_files( void );
static int get_2d_grid_info( String dsn, float *xorigin, float *zorigin, float *xintsize, float *zintsize, unsigned *numintvx, unsigned *numintvz, String *attrib_type, unsigned *attrib_index );
static int tabletype( String tablename );
static Boolean nonexistant_model( String modelname );
static Boolean nonexistant_version( String modelname, Version version );
static String build_history_string( String modelname, Version version, unsigned page, String history );
static Historytreeptr tree_insert( Historytreeptr node, unsigned page, String history );
static void read_tree( String *string, Historytreeptr node );
static int agm_flatfile_write( String filename );
static int idcompare( const void *i, const void *j );
static int unload_attrib_file( String dsn );
static char *allocstr( String s );
static char *uppercase( String s );
static char *lowercase( String s );
static int rtend( String s );
static int getline( FILE *fp, String s );
static int ungetline( String s );
static char *tokenize( String string, String front_delimeter_set, String back_delimeter_set );
static char *todays_date( void );
static char *current_time( void );
static char *extract_path( String dsn );
static Boolean write_access( String dir );

static char	message[MAX_MESSAGE_LEN+1] = {'\0'};
static String	filename = NULL;
static String	override_attrib_dsn = NULL;
static Boolean	fresh_read = TRUE;
static Modellistptr   modellist = NULL;


/*******************/
/*** C Interface ***/
/*******************/

int agm_io( unsigned command, ... )
{
   static String         dbname = NULL;
   static unsigned int   location = NULL;

   va_list        args;
   unsigned int   action;
   int            rc = 0;
   String         *outstring;


   va_start( args, command );

   /* first argument must be a command code */
   switch( command ) {
   case PutFileName:
      if( filename )
         free( filename );
      filename = strdup( va_arg( args, String ) );
      location = FLATFILE;
      break;
   case PutDBName:
      if( dbname )
         free( dbname );
      dbname = strdup( va_arg( args, String ) );
      location = DATABASE;
      break;
   case GetFileName:
      if( filename )  {
         outstring = va_arg( args, String * );
         *outstring = filename;
      } else {
         (void)sprintf( message, "Filename has not been PUT yet." );
         rc = -1;
      }
      break;
   case GetDBName:
      if( dbname )  {
         outstring = va_arg( args, String * );
         *outstring = dbname;
      } else {
         (void)sprintf( message, "Database name has not been PUT yet." );
         rc = -2;
      }
      break;
   case GetIOMessage:
      outstring = va_arg( args, String * );
      *outstring = message;
      break;
   case Action:
      switch( (action = va_arg( args, unsigned int )) )  {
      case Read:
         switch( location )  {
         case FLATFILE:
            fresh_read = TRUE;
            rc = agm_flatfile_read( filename );
            break;
         case DATABASE:
            fresh_read = TRUE;
            rc = agm_db_read( dbname );
            break;
         }
         break;
      case Write:
         switch( location )  {
         case FLATFILE:
            rc = agm_flatfile_write( filename );
            break;
         case DATABASE:
            rc = agm_db_write( dbname );
            break;
         }
         break;
      default:
         fprintf( stderr, "agm_io: invalid action code (%d)\n", action );
         rc = -3;
      }
      break;
   default:
      fprintf( stderr, "agm_io: invalid command code (%d)\n", command );
      rc = -4;
   }

   va_end( args );

   return( rc );
}


/*************************/
/*** FORTRAN Interface ***/
/*************************/

#ifdef SUN
int agmio_ ( int *command, ... )
#endif
#ifdef CRAY
int AGMIO( int *command, ... )
#endif
{
   static String	dbname = NULL;
   static unsigned int	location = NULL;

   va_list        args;
   int            *action;
   int            rc = 0;


   va_start( args, command );

   /* first argument must be a command code */
   switch( *command )  {  /* note: call by ref. */
   case PutFileName:
      if( filename )
         free( filename );
      filename = fstring_to_cstring( args );
      location = FLATFILE;
      break;
   case PutDBName:
      if( dbname )
         free( dbname );
      dbname = fstring_to_cstring( args );
      location = DATABASE;
      break;
   case GetFileName:
      if( filename )
         cstring_to_fstring( args, filename );
      else {
         (void)sprintf( message, "Filename has not been PUT yet." );
         rc = -1;
      }
      break;
   case GetDBName:
      if( dbname )
         cstring_to_fstring( args, dbname );
      else {
         (void)sprintf( message, "Database name has not been PUT yet." );
         rc = -2;
      }
      break;
   case GetIOMessage:
      cstring_to_fstring( args, message );
      break;
   case Action:
      switch( *(action = va_arg( args, int * )) )  {  /* note: call by ref. */
      case Read:
         switch( location )  {
         case FLATFILE:
            fresh_read = TRUE;
            rc = agm_flatfile_read( filename );
            break;
         case DATABASE:
            fresh_read = TRUE;
            rc = agm_db_read( dbname );
            break;
         }
         break;
      case Write:
         switch( location )  {
         case FLATFILE:
            rc = agm_flatfile_write( filename );
            break;
         case DATABASE:
            rc = agm_db_write( dbname );
            break;
         }
         break;
      default:
         fprintf( stderr, "agm_io: invalid action code (%d)\n", action );
         rc = -3;
      }
      break;
   default:
      fprintf( stderr, "agm_io: invalid command code (%d)\n", command );
      rc = -4;
   }

   va_end( args );

   return( rc );
}


#ifdef CRAY
#include <fortran.h>
#endif


static String fstring_to_cstring( va_list args )
{
   String               fstring, string;
   unsigned int         namelen;


#ifdef SUN
   fstring = va_arg( args, String );
   namelen = va_arg( args, unsigned int );
#endif
#ifdef CRAY
   _fcd   fcd;
   fcd = va_arg( args, _fcd );
   fstring = _fcdtocp( fcd );
   namelen = _fcdlen( fcd );
#endif
   string = strncpy( (String)malloc( namelen + 1 ), fstring, namelen );
   string[namelen] = '\0';
   string[rtend( string )] = '\0';

   return( string );
}


static void cstring_to_fstring( va_list args, String string )
{
   String               outstring, cptr;
   unsigned int         namelen, cnamelen;
 

#ifdef SUN
   outstring = va_arg( args, String );
   namelen = va_arg( args, unsigned int );
#endif
#ifdef CRAY
   _fcd   fcd;
   fcd = va_arg( args, _fcd );
   outstring = _fcdtocp( fcd );
   namelen = _fcdlen( fcd );
#endif
   cnamelen = strlen( string );
   (void)strncpy( outstring, string, MIN( namelen, cnamelen ) );
   for( cptr = outstring+cnamelen;  cptr < outstring+namelen;  cptr++ )
      *cptr = ' ';  /* FORTRAN expects blank padding */
}
 
 
/******************************************************************************/


static int agm_flatfile_read( String filename )
{
   FILE		*fp;
   char		line[MAX_LINE_LEN+1], tablename[TABLE_NAME_LEN+1], firstchar;
   Boolean	done = FALSE, leave = FALSE, have_comments = FALSE;
   Boolean	have_MDATBPOOL = FALSE, have_MDATBABRV = FALSE;
   Boolean	have_MDATBCOMP = FALSE, have_MDATBLOC = FALSE;
   String	messageptr, comments;
   String	modelname, userid, ownername, expirydate, description;
   String	projectionsystem, abbreviation, history, fullhistory;
   String	creationdate, creationtime, lastupdate, lastuptime;
   String	tmp_abbrev = NULL;
   Units	units;
   Unit		unit;
   Factors	factors;
   Factor	factor;
   Extents	extents;
   Location	location;
   Version	version, lastversion = (Version)NULL, versionparent;
   Type		collectiontype;
   Id		collectionid, segmentid, attributeid, componentid, locationid;
   Id		pointid;
   Id           parentid;
   float	zoffset, azimuth, latitude, longitude, component;
   int		i, page, len;
   int		rc = 0;


   if( (rc = access( filename, R_OK )) != 0 )  {
      (void)sprintf( message,
                      "File '%s' either does not exist or is not accessible.",
                      filename );
      return( rc );
   }

   if( (fp = fopen( filename, "r" )) == NULL )  {
      (void)sprintf( message,
                     "Unable to open file '%s'; is nonexistant or unacessible.",
                      filename );
      return( -1 );
   }

   while( (len = getline( fp, line )) != EOF )  {
      sscanf( line, "%c", &firstchar );

      if( firstchar == '!' )
         if( !have_comments )  {
            comments = (String)malloc( strlen( &line[1] ) + 1 );
            (void)strcpy( comments, &line[1] );
            while( (len = getline( fp, line )) != EOF )  {
               sscanf( line, "%c", &firstchar );
               if( firstchar == '!' )  {
                  comments = (String)realloc( comments, strlen( comments ) + 1
                                                     + strlen( &line[1] ) + 1 );
                  (void)strcat( comments, "\n" );
                  (void)strcat( comments, &line[1] );
               } else
                  break;
            }

            have_comments = TRUE;
            (void)agm( PutComments, comments );
            free( comments );
         }

      if( firstchar != '$' )
         continue;

      sscanf( &line[1], "%*[^:]%*[:] %s", tablename );

      switch( tabletype( tablename ) )  {
      case MDMODMISC:
         while( (len = getline( fp, line )) != EOF )  {
            sscanf( line, "%c", &firstchar );

            if( firstchar == '$' )  {
               (void)ungetline( line );
               break;
            }

            if( firstchar != '"'  &&  firstchar != '\'' )
               continue;
            else  {
               /* Note: to tokenize the line, strtok would normally be used.
                *       This approach was attempted, but proved unsatisfactory.
                *       Since the token delimeters may change from token to
                *       token and a token's text is never zero length (i.e., "")
                *       as defined by strtok, then in the case of two adjacent,
                *       but different delimeters, the starting delimeter is not
                *       recognized.  In my opinion, this gives strtok less
                *       utility.  Zero length text tokens should be allowed.
                *       Thus, tokenize is used instead.
                */
               /* Note: tokenize replaces characters following token with \0 */
               if( !(modelname = tokenize( line, "\"' ", "\"' " )) )
                  break;
               (void)tokenize( (String)NULL, ", ", "\"'" );  /* up to quote */
               if( !(userid = tokenize( (String)NULL, "\"'", "\"'" )) )
                  break;  /* note: may be blank */
               (void)tokenize( (String)NULL, ", ", "\"'" );  /* up to quote */
               if( !(ownername = tokenize( (String)NULL, "\"',", "\"'" )) )
                  break;  /* note: may be blank */
               (void)tokenize( (String)NULL, ", ", "\"'" );  /* up to quote */
               if( !(expirydate = tokenize( (String)NULL, "\"',", "\"'" )) )
                  break;  /* note: may be blank */
               (void)tokenize( (String)NULL, ", ", "\"'" );  /* up to quote */
               if( !(description = tokenize( (String)NULL, "\"',", "\"'" )) )
                  break;  /* note: may be blank */

/* flakey      if( (modelname = strtok( &line[1], "\"\', " )) == NULL )
                  break;
               (void)strtok( (char*)NULL, "\"\'" );
               userid = strdup( strtok( (char*)NULL, "\"\'" ) );
               (void)strtok( (char*)NULL, "\"\'" );
               ownername = strdup( strtok( (char*)NULL, "\"\'" ) );
               (void)strtok( (char*)NULL, "\"\'" );
               expirydate = strdup( strtok( (char*)NULL, "\"\'" ) );
               (void)strtok( (char*)NULL, "\"\'" );
               description = strdup( strtok( (char*)NULL, "\"\'" ) );
*/

/* printf( "modelname(%s) userid(%s) ownername(%s) expirydate(%s) modeldescription(%s)\n", modelname, userid, ownername, expirydate, description );
fflush( stdout ); */

               if( nonexistant_model( modelname ) )  {
                  if( (rc = agm( CreateModel, modelname )) != 0 )  {
                     (void)agm( GetMessage, &messageptr );
                     (void)strcpy( message, messageptr );
                     (void)fprintf( stderr, "%s\n", message );
                     leave = TRUE;
                     done = TRUE;
                     break;
                  }
                  lastversion = (Version)NULL;
               } else  /* model already in model list */
                  if( fresh_read )  { /* attempt to load model on top of self */
                     rc = -2;
                     (void)strcpy( message,
                               "Attempted load of an already existing model." );
                     (void)fprintf( stderr, "%s\n", message );
                     leave = TRUE;
                     done = TRUE;
                     break;
                  }

               fresh_read = FALSE;

               if( (rc = agm( PutModelMisc, userid, ownername, expirydate,
                              description )) != 0 )  {
                  (void)agm( GetMessage, &messageptr );
                  (void)strcpy( message, messageptr );
                  (void)fprintf( stderr, "%s\n", message );
                  leave = TRUE;
                  done = TRUE;
                  break;
               }

               break;  /* one mdmodmisc row per model */
            }
         }

         if( leave )
            break;
         if( len == EOF )
            done = TRUE;

         break;
      case MDMODWRLD:
         while( (len = getline( fp, line )) != EOF )  {
            sscanf( line, "%c", &firstchar );

            if( firstchar == '$' )  {
               (void)ungetline( line );
               break;
            }

            if( firstchar != '"'  &&  firstchar != '\'' )
               continue;
            else  {
               /* Note: tokenize replaces characters following token with \0 */
               if( !(modelname = tokenize( line, "\"' ", "\"' " )) )
                  break;
               (void)tokenize( (String)NULL, ", ", "\"'" );  /* up to quote */
               if( !(units.x = tokenize( (String)NULL, "\"'", "\"'" )) )
                  break;  /* note: may be blank */
               (void)tokenize( (String)NULL, ", ", "\"'" );  /* up to quote */
               if( !(units.y = tokenize( (String)NULL, "\"'", "\"'" )) )
                  break;  /* note: may be blank */
               (void)tokenize( (String)NULL, ", ", "\"'" );  /* up to quote */
               if( !(units.z = tokenize( (String)NULL, "\"'", "\"'" )) )
                  break;  /* note: may be blank */
               factors.x = atof( tokenize( (String)NULL, ", ", "," ) );
               factors.y = atof( tokenize( (String)NULL, "", "," ) );
               factors.z = atof( tokenize( (String)NULL, "", "," ) );
               extents.min.x = atof( tokenize( (String)NULL, "", "," ) );
               extents.max.x = atof( tokenize( (String)NULL, "", "," ) );
               extents.min.y = atof( tokenize( (String)NULL, "", "," ) );
               extents.max.y = atof( tokenize( (String)NULL, "", "," ) );
               extents.min.z = atof( tokenize( (String)NULL, "", "," ) );
               extents.max.z = atof( tokenize( (String)NULL, "", "," ) );

               if( nonexistant_model( modelname ) )  {
                  if( (rc = agm( CreateModel, modelname )) != 0 )  {
                     (void)agm( GetMessage, &messageptr );
                     (void)strcpy( message, messageptr );
                     (void)fprintf( stderr, "%s\n", message );
                     leave = TRUE;
                     done = TRUE;
                     break;
                  }
                  lastversion = (Version)NULL;
               } else  /* model already in model list */
                  if( fresh_read )  { /* attempt to load model on top of self */
                     rc = -2;
                     (void)strcpy( message,
                               "Attempted load of an already existing model." );
                     (void)fprintf( stderr, "%s\n", message );
                     leave = TRUE;
                     done = TRUE;
                     break;
                  }

               fresh_read = FALSE;

               if( (rc = agm( PutModelWorld, &units, &factors, &extents ))
                   != 0 )  {
                  (void)agm( GetMessage, &messageptr );
                  (void)strcpy( message, messageptr );
                  (void)fprintf( stderr, "%s\n", message );
                  leave = TRUE;
                  done = TRUE;
                  break;
               }

               break;  /* one mdmodwrld row per model */
            }
         }

         if( leave )
            break;
         if( len == EOF )
            done = TRUE;

         break;
      case MDMODLOC:
         while( (len = getline( fp, line )) != EOF )  {
            sscanf( line, "%c", &firstchar );

            if( firstchar == '$' )  {
               (void)ungetline( line );
               break;
            }

            if( firstchar != '"'  &&  firstchar != '\'' )
               continue;
            else  {
               /* Note: tokenize replaces characters following token with \0 */
               if( !(modelname = tokenize( line, "\"' ", "\"' " )) )
                  break;
               location.x = atof( tokenize( (String)NULL, ", ", "," ) );
               location.y = atof( tokenize( (String)NULL, "", "," ) );
               location.z = atof( tokenize( (String)NULL, "", "," ) );
               zoffset = atof( tokenize( (String)NULL, "", "," ) );
               azimuth = atof( tokenize( (String)NULL, "", "," ) );
               latitude = atof( tokenize( (String)NULL, "", "," ) );
               longitude = atof( tokenize( (String)NULL, "", "," ) );
               (void)tokenize( (String)NULL, " ", "\"'" );  /* up to quote */
               if( !(projectionsystem = tokenize( (String)NULL, "\"'", "\"'" )))
                  break;  /* note: may be blank */

               if( nonexistant_model( modelname ) )  {
                  if( (rc = agm( CreateModel, modelname )) != 0 )  {
                     (void)agm( GetMessage, &messageptr );
                     (void)strcpy( message, messageptr );
                     (void)fprintf( stderr, "%s\n", message );
                     leave = TRUE;
                     done = TRUE;
                     break;
                  }
                  lastversion = (Version)NULL;
               } else  /* model already in model list */
                  if( fresh_read )  { /* attempt to load model on top of self */
                     rc = -2;
                     (void)strcpy( message,
                               "Attempted load of an already existing model." );
                     (void)fprintf( stderr, "%s\n", message );
                     leave = TRUE;
                     done = TRUE;
                     break;
                  }

               fresh_read = FALSE;

               if( (rc = agm( PutModelLoc, &location, zoffset, azimuth,
                              latitude, longitude, projectionsystem ))
                   != 0 )  {
                  (void)agm( GetMessage, &messageptr );
                  (void)strcpy( message, messageptr );
                  (void)fprintf( stderr, "%s\n", message );
                  leave = TRUE;
                  done = TRUE;
                  break;
               }

               break;  /* one mdmodloc row per model */
            }
         }

         if( leave )
            break;
         if( len == EOF )
            done = TRUE;

         break;
      case MDVERMISC:
         while( (len = getline( fp, line )) != EOF )  {
            sscanf( line, "%c", &firstchar );

            if( firstchar == '$' )  {
               (void)ungetline( line );
               break;
            }

            if( firstchar != '"'  &&  firstchar != '\'' )
               continue;
            else  {
               /* Note: tokenize replaces characters following token with \0 */
               if( !(modelname = tokenize( line, "\"' ", "\"' " )) )
                  break;
               version = atoi( tokenize( (String)NULL, ", ", "," ) );
               versionparent = atoi( tokenize( (String)NULL, "", "," ) );
               (void)tokenize( (String)NULL, " ", "\"'" );  /* up to quote */
               if( !(creationdate = tokenize( (String)NULL, "\"'", "\"'" )) )
                  break;  /* note: may be blank */
               (void)tokenize( (String)NULL, ", ", "\"'" );  /* up to quote */
               if( !(creationtime = tokenize( (String)NULL, "\"'", "\"'" )) )
                  break;  /* note: may be blank */
               (void)tokenize( (String)NULL, ", ", "\"'" );  /* up to quote */
               if( !(lastupdate = tokenize( (String)NULL, "\"'", "\"'" )) )
                  break;  /* note: may be blank */
               (void)tokenize( (String)NULL, ", ", "\"'" );  /* up to quote */
               if( !(lastuptime = tokenize( (String)NULL, "\"'", "\"'" )) )
                  break;  /* note: may be blank */
               (void)tokenize( (String)NULL, ", ", "\"'" );  /* up to quote */
               if( !(description = tokenize( (String)NULL, "\"'", "\"'" )) )
                  break;  /* note: may be blank */

               if( nonexistant_model( modelname ) )  {
                  if( (rc = agm( CreateModel, modelname )) != 0 )  {
                     (void)agm( GetMessage, &messageptr );
                     (void)strcpy( message, messageptr );
                     (void)fprintf( stderr, "%s\n", message );
                     leave = TRUE;
                     break;
                  }
                  lastversion = (Version)NULL;
               } else  /* model already in model list */
                  if( fresh_read )  { /* attempt to load model on top of self */
                     rc = -2;
                     (void)strcpy( message,
                               "Attempted load of an already existing model." );
                     (void)fprintf( stderr, "%s\n", message );
                     leave = TRUE;
                     done = TRUE;
                     break;
                  }

               fresh_read = FALSE;

               if( nonexistant_version( modelname, version ) )
                  (void)agm( CreateVersion, version );
               else if( version != lastversion )
                  (void)agm( ActivateVersion, version );
               lastversion = version;

               if( (rc = agm( PutVersionMisc, versionparent, creationdate,
                              creationtime, lastupdate, lastuptime,
                              description )) != 0 )  {
                  (void)agm( GetMessage, &messageptr );
                  (void)strcpy( message, messageptr );
                  (void)fprintf( stderr, "%s\n", message );
                  (void)fprintf( stderr, "%s\n", messageptr );
                  leave = TRUE;
                  break;
               }
            }
         }

         if( leave )
            break;
         if( len == EOF )
            done = TRUE;

         break;
      case MDLINEAGE:
         while( (len = getline( fp, line )) != EOF )  {
            sscanf( line, "%c", &firstchar );

            if( firstchar == '$' )  {
               (void)ungetline( line );
               break;
            }

            if( firstchar != '"'  &&  firstchar != '\'' )
               continue;
            else  {
               /* Note: tokenize replaces characters following token with \0 */
               if( !(modelname = tokenize( line, "\"' ", "\"' " )) )
                  break;
               version = atoi( tokenize( (String)NULL, ", ", "," ) );
               page = atoi( tokenize( (String)NULL, "", "," ) );
               (void)tokenize( (String)NULL, " ", "\"'" );  /* up to quote */
               if( !(history = tokenize( (String)NULL, "\"'", "\"'" )) )
                  break;  /* note: may be blank */

               if( nonexistant_model( modelname ) )  {
                  if( (rc = agm( CreateModel, modelname )) != 0 )  {
                     (void)agm( GetMessage, &messageptr );
                     (void)strcpy( message, messageptr );
                     (void)fprintf( stderr, "%s\n", message );
                     leave = TRUE;
                     break;
                  }
                  lastversion = (Version)NULL;
               } else  /* model already in model list */
                  if( fresh_read )  { /* attempt to load model on top of self */
                     rc = -2;
                     (void)strcpy( message,
                               "Attempted load of an already existing model." );
                     (void)fprintf( stderr, "%s\n", message );
                     leave = TRUE;
                     done = TRUE;
                     break;
                  }

               fresh_read = FALSE;

               if( nonexistant_version( modelname, version ) )
                  (void)agm( CreateVersion, version );
               else if( version != lastversion )
                  (void)agm( ActivateVersion, version );
               lastversion = version;

               fullhistory = build_history_string( modelname, version, page,
                                                   history );

               if( (rc = agm( PutVersionLineage, fullhistory )) != 0 )  {
                  (void)agm( GetMessage, &messageptr );
                  (void)strcpy( message, messageptr );
                  (void)fprintf( stderr, "%s\n", message );
                  free( fullhistory );
                  leave = TRUE;
                  break;
               }

               free( fullhistory );
            }
         }

         if( leave )
            break;
         if( len == EOF )
            done = TRUE;

         break;
      case MDCOLMISC:
         while( (len = getline( fp, line )) != EOF )  {
            sscanf( line, "%c", &firstchar );

            if( firstchar == '$' )  {
               (void)ungetline( line );
               break;
            }

            if( firstchar != '"'  &&  firstchar != '\'' )
               continue;
            else  {
               /* Note: tokenize replaces characters following token with \0 */
               if( !(modelname = tokenize( line, "\"' ", "\"' " )) )
                  break;
               version = atoi( tokenize( (String)NULL, ", ", "," ) );
               collectionid = atoi( tokenize( (String)NULL, "", "," ) );
               parentid = atoi( tokenize( (String)NULL, "", "," ) );
               collectiontype = atoi( tokenize( (String)NULL, "", "," ) );
               (void)tokenize( (String)NULL, " ", "\"'" );  /* up to quote */
               description = tokenize( (String)NULL, "\"'", "\"'" );
               if( description == NULL )
                  description = strdup( " " );
               if( strlen( description ) == 0 )
                  description = strdup( " " );

               if( nonexistant_model( modelname ) )  {
                  if( (rc = agm( CreateModel, modelname )) != 0 )  {
                     (void)agm( GetMessage, &messageptr );
                     (void)strcpy( message, messageptr );
                     (void)fprintf( stderr, "%s\n", message );
                     leave = TRUE;
                     break;
                  }
                  lastversion = (Version)NULL;
               } else  /* model already in model list */
                  if( fresh_read )  { /* attempt to load model on top of self */
                     rc = -2;
                     (void)strcpy( message,
                               "Attempted load of an already existing model." );
                     (void)fprintf( stderr, "%s\n", message );
                     leave = TRUE;
                     done = TRUE;
                     break;
                  }

               fresh_read = FALSE;

               if( nonexistant_version( modelname, version ) )
                  (void)agm( CreateVersion, version );
               else if( version != lastversion )
                  (void)agm( ActivateVersion, version );
               lastversion = version;

               if( (rc = agm( PutIdCollectionMisc, collectionid, parentid,
                              collectiontype, description ))
                   != 0 )  {
                  (void)agm( GetMessage, &messageptr );
                  (void)strcpy( message, messageptr );
                  (void)fprintf( stderr, "%s\n", message );
                  leave = TRUE;
                  break;
               }
            }
         }

         if( leave )
            break;
         if( len == EOF )
            done = TRUE;

         break;
      case MDCOLSEGS:
         while( (len = getline( fp, line )) != EOF )  {
            sscanf( line, "%c", &firstchar );

            if( firstchar == '$' )  {
               (void)ungetline( line );
               break;
            }

            if( firstchar != '"'  &&  firstchar != '\'' )
               continue;
            else  {
               /* Note: tokenize replaces characters following token with \0 */
               if( !(modelname = tokenize( line, "\"' ", "\"' " )) )
                  break;
               version = atoi( tokenize( (String)NULL, ", ", "," ) );
               collectionid = atoi( tokenize( (String)NULL, "", "," ) );
               segmentid = atoi( tokenize( (String)NULL, "", "," ) );

               if( nonexistant_model( modelname ) )  {
                  if( (rc = agm( CreateModel, modelname )) != 0 )  {
                     (void)agm( GetMessage, &messageptr );
                     (void)strcpy( message, messageptr );
                     (void)fprintf( stderr, "%s\n", message );
                     leave = TRUE;
                     break;
                  }
                  lastversion = (Version)NULL;
               } else  /* model already in model list */
                  if( fresh_read )  { /* attempt to load model on top of self */
                     rc = -2;
                     (void)strcpy( message,
                               "Attempted load of an already existing model." );
                     (void)fprintf( stderr, "%s\n", message );
                     leave = TRUE;
                     done = TRUE;
                     break;
                  }

               fresh_read = FALSE;

               if( nonexistant_version( modelname, version ) )
                  (void)agm( CreateVersion, version );
               else if( version != lastversion )
                  (void)agm( ActivateVersion, version );
               lastversion = version;

               if( (rc = agm( PutIdCollectionSegmentLink, collectionid,
                              segmentid )) != 0 )  {
                  (void)agm( GetMessage, &messageptr );
                  (void)strcpy( message, messageptr );
                  (void)fprintf( stderr, "%s\n", message );
                  leave = TRUE;
                  break;
               }
            }
         }

         if( leave )
            break;
         if( len == EOF )
            done = TRUE;

         break;
      case MDCOLATBS:
         while( (len = getline( fp, line )) != EOF )  {
            sscanf( line, "%c", &firstchar );

            if( firstchar == '$' )  {
               (void)ungetline( line );
               break;
            }

            if( firstchar != '"'  &&  firstchar != '\'' )
               continue;
            else  {
               /* Note: tokenize replaces characters following token with \0 */
               if( !(modelname = tokenize( line, "\"' ", "\"' " )) )
                  break;
               version = atoi( tokenize( (String)NULL, ", ", "," ) );
               collectionid = atoi( tokenize( (String)NULL, "", "," ) );
               attributeid = atoi( tokenize( (String)NULL, "", "," ) );

               if( nonexistant_model( modelname ) )  {
                  if( (rc = agm( CreateModel, modelname )) != 0 )  {
                     (void)agm( GetMessage, &messageptr );
                     (void)strcpy( message, messageptr );
                     (void)fprintf( stderr, "%s\n", message );
                     leave = TRUE;
                     break;
                  }
                  lastversion = (Version)NULL;
               } else  /* model already in model list */
                  if( fresh_read )  { /* attempt to load model on top of self */
                     rc = -2;
                     (void)strcpy( message,
                               "Attempted load of an already existing model." );
                     (void)fprintf( stderr, "%s\n", message );
                     leave = TRUE;
                     done = TRUE;
                     break;
                  }

               fresh_read = FALSE;

               if( nonexistant_version( modelname, version ) )
                  (void)agm( CreateVersion, version );
               else if( version != lastversion )
                  (void)agm( ActivateVersion, version );
               lastversion = version;

               if( (rc = agm( PutIdCollectionAttributeLink, collectionid,
                              attributeid )) != 0 )  {
                  (void)agm( GetMessage, &messageptr );
                  (void)strcpy( message, messageptr );
                  (void)fprintf( stderr, "%s\n", message );
                  leave = TRUE;
                  break;
               }
            }
         }

         if( leave )
            break;
         if( len == EOF )
            done = TRUE;

         break;
      case MDATBMISC:
         while( (len = getline( fp, line )) != EOF )  {
            sscanf( line, "%c", &firstchar );

            if( firstchar == '$' )  {
               (void)ungetline( line );
               break;
            }

            if( firstchar != '"'  &&  firstchar != '\'' )
               continue;
            else  {
               /* Note: tokenize replaces characters following token with \0 */
               if( !(modelname = tokenize( line, "\"' ", "\"' " )) )
                  break;
               (void)tokenize( (String)NULL, ", ", "\"'" );  /* up to quote */
               if( !(abbreviation = tokenize( (String)NULL, "\"'", "\"'" )) )
                  break;  /* note: may be blank */
               (void)tokenize( (String)NULL, ", ", "\"'" );  /* up to quote */
               if( !(unit = tokenize( (String)NULL, "\"'", "\"'" )) )
                  break;  /* note: may be blank */
               factor = atof( tokenize( (String)NULL, ", ", "," ) );
               (void)tokenize( (String)NULL, " ", "\"'" );  /* up to quote */
               if( !(description = tokenize( (String)NULL, "\"'", "\"'" )) )
                  break;  /* note: may be blank */

               if( nonexistant_model( modelname ) )  {
                  if( (rc = agm( CreateModel, modelname )) != 0 )  {
                     (void)agm( GetMessage, &messageptr );
                     (void)strcpy( message, messageptr );
                     (void)fprintf( stderr, "%s\n", message );
                     leave = TRUE;
                     break;
                  }
                  lastversion = (Version)NULL;
               } else  /* model already in model list */
                  if( fresh_read )  { /* attempt to load model on top of self */
                     rc = -2;
                     (void)strcpy( message,
                               "Attempted load of an already existing model." );
                     (void)fprintf( stderr, "%s\n", message );
                     leave = TRUE;
                     done = TRUE;
                     break;
                  }

               fresh_read = FALSE;

               if( (rc = agm( PutAttributeMisc, abbreviation, unit, factor,
                              description )) != 0 )  {
                  (void)agm( GetMessage, &messageptr );
                  (void)strcpy( message, messageptr );
                  (void)fprintf( stderr, "%s\n", message );
                  leave = TRUE;
                  break;
               }
            }
         }

         if( leave )
            break;
         if( len == EOF )
            done = TRUE;

         break;
      case MDATBPOOL:  /* NOTE: obsolete (version 0.9) */
         have_MDATBPOOL = TRUE;

         while( (len = getline( fp, line )) != EOF )  {
            sscanf( line, "%c", &firstchar );

            if( firstchar == '$' )  {
               (void)ungetline( line );
               break;
            }

            if( firstchar != '"'  &&  firstchar != '\'' )
               continue;
            else  {
               /* Note: tokenize replaces characters following token with \0 */
               if( !(modelname = tokenize( line, "\"' ", "\"' " )) )
                  break;
               attributeid = atoi( tokenize( (String)NULL, ", ", "," ) );
               (void)tokenize( (String)NULL, " ", "\"'" );  /* up to quote */
               if( !(abbreviation = tokenize( (String)NULL, "\"'", "\"'" )) )
                  break;  /* note: may be blank */
               component = atof( tokenize( (String)NULL, ", ", "," ) );
               location.x = atof( tokenize( (String)NULL, "", "," ) );
               location.y = atof( tokenize( (String)NULL, "", "," ) );
               location.z = atof( tokenize( (String)NULL, "", "," ) );

               if( nonexistant_model( modelname ) )  {
                  if( (rc = agm( CreateModel, modelname )) != 0 )  {
                     (void)agm( GetMessage, &messageptr );
                     (void)strcpy( message, messageptr );
                     (void)fprintf( stderr, "%s\n", message );
                     leave = TRUE;
                     break;
                  }
                  lastversion = (Version)NULL;
               } else  /* model already in model list */
                  if( fresh_read )  { /* attempt to load model on top of self */
                     rc = -2;
                     (void)strcpy( message,
                               "Attempted load of an already existing model." );
                     (void)fprintf( stderr, "%s\n", message );
                     leave = TRUE;
                     done = TRUE;
                     break;
                  }

               fresh_read = FALSE;

/* old         if( (rc = agm( PutAttributePool, attributeid, abbreviation,
                              component, &location )) != 0 )  {
*/
               if( !have_MDATBABRV )
                  if( (rc = agm( PutIdAttributeAbbreviation, attributeid,
                                 abbreviation )) != 0 )  {
                     (void)agm( GetMessage, &messageptr );
                     (void)strcpy( message, messageptr );
                     (void)fprintf( stderr, "%s\n", message );
                     leave = TRUE;
                     break;
                  }

               if( !have_MDATBCOMP )  {
                  tmp_abbrev = uppercase( strdup( abbreviation ) );
                  if( strcmp( tmp_abbrev, "RGB" ) == MATCH )  {
                     /* RGB triplet is stored in location field */
                     for( i = 0;  i < 3;  i++ )
                        if((rc = agm( PutIdAttributeComponent, attributeid, i+1,
                                       (float)*(&location.x+i) )) != 0 )  {
                           (void)agm( GetMessage, &messageptr );
                           (void)strcpy( message, messageptr );
                           (void)fprintf( stderr, "%s\n", message );
                           leave = TRUE;
                           break;
                        }
                  } else if( (rc = agm( PutIdAttributeComponent, attributeid, 1,
                                        component )) != 0 )  {
                     (void)agm( GetMessage, &messageptr );
                     (void)strcpy( message, messageptr );
                     (void)fprintf( stderr, "%s\n", message );
                     leave = TRUE;
                     break;
                  }
               } else
                  tmp_abbrev = NULL;

               if( !have_MDATBLOC )
                  if( tmp_abbrev )  {
                     if( strcmp( tmp_abbrev, "RGB" ) != MATCH )
                        if( (rc = agm( PutIdAttributeLocation, attributeid, 1,
                                       &location )) != 0 )  {
                           (void)agm( GetMessage, &messageptr );
                           (void)strcpy( message, messageptr );
                           (void)fprintf( stderr, "%s\n", message );
                           leave = TRUE;
                           break;
                        }
                  } else if( (rc = agm( PutIdAttributeLocation, attributeid, 1,
                                        &location )) != 0 )  {
                     (void)agm( GetMessage, &messageptr );
                     (void)strcpy( message, messageptr );
                     (void)fprintf( stderr, "%s\n", message );
                     leave = TRUE;
                     break;
                  }

               if( tmp_abbrev )  {
                  free( (char *)tmp_abbrev );
                  tmp_abbrev = NULL;
               }
            }
         }

         if( leave )
            break;
         if( len == EOF )
            done = TRUE;

         break;
      case MDATBABRV:
         have_MDATBABRV = TRUE;

         while( (len = getline( fp, line )) != EOF )  {
            sscanf( line, "%c", &firstchar );

            if( firstchar == '$' )  {
               (void)ungetline( line );
               break;
            }

            if( firstchar != '"'  &&  firstchar != '\'' )
               continue;
            else  {
               /* Note: tokenize replaces characters following token with \0 */
               if( !(modelname = tokenize( line, "\"' ", "\"' " )) )
                  break;
               attributeid = atoi( tokenize( (String)NULL, ", ", "," ) );
               (void)tokenize( (String)NULL, " ", "\"'" );  /* up to quote */
               if( !(abbreviation = tokenize( (String)NULL, "\"'", "\"'" )) )
                  break;  /* note: may be blank */

               if( nonexistant_model( modelname ) )  {
                  if( (rc = agm( CreateModel, modelname )) != 0 )  {
                     (void)agm( GetMessage, &messageptr );
                     (void)strcpy( message, messageptr );
                     (void)fprintf( stderr, "%s\n", message );
                     leave = TRUE;
                     break;
                  }
                  lastversion = (Version)NULL;
               } else  /* model already in model list */
                  if( fresh_read )  { /* attempt to load model on top of self */
                     rc = -2;
                     (void)strcpy( message,
                               "Attempted load of an already existing model." );
                     (void)fprintf( stderr, "%s\n", message );
                     leave = TRUE;
                     done = TRUE;
                     break;
                  }

               fresh_read = FALSE;

               if( !have_MDATBPOOL )
                  if( (rc = agm( PutIdAttributeAbbreviation, attributeid,
                                 abbreviation )) != 0 )  {
                     (void)agm( GetMessage, &messageptr );
                     (void)strcpy( message, messageptr );
                     (void)fprintf( stderr, "%s\n", message );
                     leave = TRUE;
                     break;
                  }
            }
         }

         if( leave )
            break;
         if( len == EOF )
            done = TRUE;

         break;
      case MDATBCOMP:
         have_MDATBCOMP = TRUE;

         while( (len = getline( fp, line )) != EOF )  {
            sscanf( line, "%c", &firstchar );

            if( firstchar == '$' )  {
               (void)ungetline( line );
               break;
            }

            if( firstchar != '"'  &&  firstchar != '\'' )
               continue;
            else  {
               /* Note: tokenize replaces characters following token with \0 */
               if( !(modelname = tokenize( line, "\"' ", "\"' " )) )
                  break;
               attributeid = atoi( tokenize( (String)NULL, ", ", "," ) );
               componentid = atoi( tokenize( (String)NULL, "", "," ) );
               component = atof( tokenize( (String)NULL, "", "," ) );

               if( nonexistant_model( modelname ) )  {
                  if( (rc = agm( CreateModel, modelname )) != 0 )  {
                     (void)agm( GetMessage, &messageptr );
                     (void)strcpy( message, messageptr );
                     (void)fprintf( stderr, "%s\n", message );
                     leave = TRUE;
                     break;
                  }
                  lastversion = (Version)NULL;
               } else  /* model already in model list */
                  if( fresh_read )  { /* attempt to load model on top of self */
                     rc = -2;
                     (void)strcpy( message,
                               "Attempted load of an already existing model." );
                     (void)fprintf( stderr, "%s\n", message );
                     leave = TRUE;
                     done = TRUE;
                     break;
                  }

               fresh_read = FALSE;

/*             if( !have_MDATBPOOL ) can't do here since compatibility prob */
               if( (rc = agm( PutIdAttributeComponent, attributeid, componentid,
                              component )) != 0 )  {
                  if( !have_MDATBPOOL ) { /* we know will cause error message */
                     (void)agm( GetMessage, &messageptr );
                     (void)strcpy( message, messageptr );
                     (void)fprintf( stderr, "%s\n", message );
                     leave = TRUE;
                     break;
                  }
               }
            }
         }

         if( leave )
            break;
         if( len == EOF )
            done = TRUE;

         break;
      case MDATBLOC:
         have_MDATBLOC = TRUE;

         while( (len = getline( fp, line )) != EOF )  {
            sscanf( line, "%c", &firstchar );

            if( firstchar == '$' )  {
               (void)ungetline( line );
               break;
            }

            if( firstchar != '"'  &&  firstchar != '\'' )
               continue;
            else  {
               /* Note: tokenize replaces characters following token with \0 */
               if( !(modelname = tokenize( line, "\"' ", "\"' " )) )
                  break;
               attributeid = atoi( tokenize( (String)NULL, ", ", "," ) );
               locationid = atoi( tokenize( (String)NULL, "", "," ) );
               location.x = atof( tokenize( (String)NULL, "", "," ) );
               location.y = atof( tokenize( (String)NULL, "", "," ) );
               location.z = atof( tokenize( (String)NULL, "", "," ) );

               if( nonexistant_model( modelname ) )  {
                  if( (rc = agm( CreateModel, modelname )) != 0 )  {
                     (void)agm( GetMessage, &messageptr );
                     (void)strcpy( message, messageptr );
                     (void)fprintf( stderr, "%s\n", message );
                     leave = TRUE;
                     break;
                  }
                  lastversion = (Version)NULL;
               } else  /* model already in model list */
                  if( fresh_read )  { /* attempt to load model on top of self */
                     rc = -2;
                     (void)strcpy( message,
                               "Attempted load of an already existing model." );
                     (void)fprintf( stderr, "%s\n", message );
                     leave = TRUE;
                     done = TRUE;
                     break;
                  }

               fresh_read = FALSE;

/*             if( !have_MDATBPOOL ) can't do here since compatibility prob */
               if( (rc = agm( PutIdAttributeLocation, attributeid, locationid,
                              &location )) != 0 )  {
                  if( !have_MDATBPOOL ) { /* we know will cause error message */
                     (void)agm( GetMessage, &messageptr );
                     (void)strcpy( message, messageptr );
                     (void)fprintf( stderr, "%s\n", message );
                     leave = TRUE;
                     break;
                  }
               }
            }
         }

         if( leave )
            break;
         if( len == EOF )
            done = TRUE;

         break;
      case MDSEGPOOL:
         while( (len = getline( fp, line )) != EOF )  {
            sscanf( line, "%c", &firstchar );

            if( firstchar == '$' )  {
               (void)ungetline( line );
               break;
            }

            if( firstchar != '"'  &&  firstchar != '\'' )
               continue;
            else  {
               /* Note: tokenize replaces characters following token with \0 */
               if( !(modelname = tokenize( line, "\"' ", "\"' " )) )
                  break;
               segmentid = atoi( tokenize( (String)NULL, ", ", "," ) );
               pointid = atoi( tokenize( (String)NULL, "", "," ) );
               location.x = atof( tokenize( (String)NULL, "", "," ) );
               location.y = atof( tokenize( (String)NULL, "", "," ) );
               location.z = atof( tokenize( (String)NULL, "", "," ) );

               if( nonexistant_model( modelname ) )  {
                  if( (rc = agm( CreateModel, modelname )) != 0 )  {
                     (void)agm( GetMessage, &messageptr );
                     (void)strcpy( message, messageptr );
                     (void)fprintf( stderr, "%s\n", message );
                     leave = TRUE;
                     break;
                  }
                  lastversion = (Version)NULL;
               } else  /* model already in model list */
                  if( fresh_read )  { /* attempt to load model on top of self */
                     rc = -2;
                     (void)strcpy( message,
                               "Attempted load of an already existing model." );
                     (void)fprintf( stderr, "%s\n", message );
                     leave = TRUE;
                     done = TRUE;
                     break;
                  }

               fresh_read = FALSE;

               if( (rc = agm( PutSegmentPoint, segmentid, pointid, &location ))
                   != 0 )  {
                  (void)agm( GetMessage, &messageptr );
                  (void)strcpy( message, messageptr );
                  (void)fprintf( stderr, "%s\n", message );
                  leave = TRUE;
                  break;
               }
            }
         }

         if( leave )
            break;
         if( len == EOF )
            done = TRUE;

         break;
      default:
         (void)sprintf( message,
                         "Encountered bad table name (%s) in file '%s'.",
                         tablename, filename );
         (void)fprintf( stderr, "%s\n", message );
         continue;
      }

      if( done )
         break;
   }

   fclose( fp );


   if( rc == 0 )
      rc = load_attrib_files();

   /* reset collection types to ALL_TYPES (from DATA_SET) */
   if( (rc = agm( RestrictCollectionType, ALL_TYPES )) != 0 )  {
      (void)agm( GetMessage, &messageptr );
      (void)strcpy( message, messageptr );
      fprintf( stderr, "%s\n", message );
   }

   return( rc );
}


static int load_attrib_files( void )
{
   FILE			*fp;
/* Location		location; */
   String		dsn, messageptr, attrib_type;
/* Id			parent_id;*/ /*, attrib_id; */
   float		xorigin, zorigin, xintsize, zintsize, *attrib_array;
   unsigned int		numintvx, numintvz, attrib_index;
   unsigned int         nversions, ithversion, ncollections, ithcollect;
   unsigned int         nelements;
/* unsigned int         i, j, k; */
   int			nbytes, rc = 0;


/* location.y = 0.0; */

   (void)agm( GetNumberVersions, &nversions );
   for( ithversion = 1;  ithversion <= nversions;  ithversion++ )  {
      if( agm( ActivateVersion, ithversion ) != 0 )  {
         (void)agm( GetMessage, &messageptr );
         (void)strcpy( message, messageptr );
         fprintf( stderr, "%s\n", message );
         continue;
      }

      if( (rc = agm( RestrictCollectionType, DATA_SET )) != 0 )  {
         (void)agm( GetMessage, &messageptr );
         (void)strcpy( message, messageptr );
         fprintf( stderr, "%s\n", message );
         continue;
      }
      if( (rc = agm( GetNumberCollections, &ncollections )) != 0 )  {
         (void)agm( GetMessage, &messageptr );
         (void)strcpy( message, messageptr );
         fprintf( stderr, "%s\n", message );
         continue;
      }

      /* cycle thru data sets (== collections) */
      for( ithcollect = 1;  ithcollect <= ncollections;  ithcollect++ )  {
         if( ithcollect > 1 )
            if( (rc = agm( RestrictCollectionType, DATA_SET )) != 0 )  {
               (void)agm( GetMessage, &messageptr );
               (void)strcpy( message, messageptr );
               fprintf( stderr, "%s\n", message );
               continue;
            }
         if( (rc = agm( ActivateIndexedCollection, ithcollect )) != 0 )  {
            (void)agm( GetMessage, &messageptr );
            (void)strcpy( message, messageptr );
            (void)fprintf( stderr, "%s\n", message );
            continue;
         }
         if( (rc = agm( GetCollectionDescription, &dsn )) != 0 )  {
            (void)agm( GetMessage, &messageptr );
            (void)strcpy( message, messageptr );
            (void)fprintf( stderr, "%s\n", message );
            continue;
         }
/*       if( (rc = agm( GetCollectionParent, &parent_id )) != 0 )  {
            (void)agm( GetMessage, &messageptr );
            (void)strcpy( message, messageptr );
            (void)fprintf( stderr, "%s\n", message );
            continue;
         }
*/

         if( (rc = access( dsn, R_OK )) != 0 )  {
            (void)sprintf( message,
                        "File '%s' either does not exist or is not accessible.",
                           dsn );
            (void)fprintf( stderr, "%s\n", message );
            continue;
         }

         if( (fp = fopen( dsn, "r" )) == 0 )  {
            (void)sprintf( message,
                     "Unable to open file '%s'; is nonexistant or unacessible.",
                           dsn );
            (void)fprintf( stderr, "%s\n", message );
            continue;
         }

         if( get_2d_grid_info( dsn, &xorigin, &zorigin, &xintsize, &zintsize,
                               &numintvx, &numintvz,
                               &attrib_type, &attrib_index ) != 0 )  {
            fclose( fp );
            continue;
         }

         nelements = (numintvx + 1) * (numintvz + 1);
         nbytes = nelements * sizeof( float );

         if( (attrib_array = (float *)malloc( (unsigned int)nbytes ))
             == NULL )  {
            (void)sprintf( message,
                           "Unable to allocate %d bytes for file '%s'.",
                           nbytes, dsn );
            (void)fprintf( stderr, "%s\n", message );
            fclose( fp );
            continue;
         }

         if( fread( (char *)attrib_array, nbytes, 1, fp ) != 1 )  {
            (void)sprintf( message, "Zero items read from file '%s'.", dsn );
            (void)fprintf( stderr, "%s\n", message );
            free( (char *)attrib_array );
            fclose( fp );
            continue;
         }

         fclose( fp );


         if( (rc = agm( ActivateCollectionIndexedAttribute, attrib_index ))
             != 0 )  {
            (void)agm( GetMessage, &messageptr );
            (void)strcpy( message, messageptr );
            fprintf( stderr, "%s\n", message );
            free( (char *)attrib_array );
            continue;
         }

         rc = agm( PutAttributeSet, dsn, nelements, attrib_array );


/*       for( k = i = 0;  i < numintvx+1;  i++ )  {
            location.x = xorigin + i * xintsize;

            for( j = 0;  j < numintvz+1;  j++, k++ )  {
               location.z = zorigin + j * zintsize;

               rc = agm( CreateAttribute, NULL );
                 if( rc != 0 ) break;
               rc = agm( PutAttributeComponent, attrib_array[k] );
                 if( rc != 0 ) break;
               rc = agm( PutAttributeAbbreviation, attrib_type );
                 if( rc != 0 ) break;
               rc = agm( PutAttributeLocations, 1, &location );
                 if( rc != 0 ) break;
               rc = agm( GetAttributeId, &attrib_id );
                 if( rc != 0 ) break;
               rc = agm( PutCollectionAttributeAssociation,
                         parent_id, attrib_id );
                 if( rc != 0 ) break;
            }

            if( rc != 0 ) break;
         }

         free( (char *)attrib_array );

         if( rc != 0 )  {
            (void)agm( GetMessage, &messageptr );
            (void)strcpy( message, messageptr );
            fprintf( stderr, "%s\n", message );
            continue;
         }
*/
      }
   }

   return( rc );
}


static int get_2d_grid_info( String dsn, float *xorigin, float *zorigin, float *xintsize, float *zintsize, unsigned *numintvx, unsigned *numintvz, String *attrib_type, unsigned *attrib_index )
{
   String		messageptr, *abbrevs;
   Id			parent_id;
   Type			type;
   float		*components;
   unsigned int         nabbrevs, ncomps, i;
   int			rc = 0;


   *xorigin = *zorigin = *xintsize = *zintsize = 0.0;
   *numintvx = *numintvz = 0;
   *attrib_type = NULL;
   *attrib_index = 0;

   if( (rc = agm( GetCollectionParent, &parent_id )) != 0 )  {
      (void)agm( GetMessage, &messageptr );
      (void)strcpy( message, messageptr );
      (void)fprintf( stderr, "%s\n", message );
      return( rc );
   }
   if( (rc = agm( ActivateCollection, parent_id )) != 0 )  {
      (void)agm( GetMessage, &messageptr );
      (void)strcpy( message, messageptr );
      (void)fprintf( stderr, "%s\n", message );
      return( rc );
   }
   if( (rc = agm( GetCollectionType, &type )) != 0 )  {
      (void)agm( GetMessage, &messageptr );
      (void)strcpy( message, messageptr );
      (void)fprintf( stderr, "%s\n", message );
      return( rc );
   }
   if( type != Regular2DGrid )  {
      (void)sprintf( message,
                     "File '%s' must be a 2-D regular grid.", dsn );
      (void)fprintf( stderr, "%s\n", message );
      return( 1 );
   }

   if( (rc = agm( GetCollectionAttributeAbbreviations, &nabbrevs, &abbrevs ))
       != 0 )  {
      (void)agm( GetMessage, &messageptr );
      (void)strcpy( message, messageptr );
      (void)fprintf( stderr, "%s\n", message );
      return( rc );
   }

   for( i = 0;  i < nabbrevs;  i++ )  {
      if( (rc = agm( ActivateCollectionIndexedAttribute, i+1 )) != 0 )  {
         (void)agm( GetMessage, &messageptr );
         (void)strcpy( message, messageptr );
         (void)fprintf( stderr, "%s\n", message );
         break;
      }

      if( (rc = agm( GetAttributeComponents, &ncomps, &components )) != 0 ) {
         (void)agm( GetMessage, &messageptr );
         (void)strcpy( message, messageptr );
         (void)fprintf( stderr, "%s\n", message );
         break;
      }

      if( strcmp( abbrevs[i], "XORIGIN" ) == MATCH )
         *xorigin = components[0];
      else if( strcmp( abbrevs[i], "ZORIGIN" ) == MATCH )
         *zorigin = components[0];
      else if( strcmp( abbrevs[i], "XINTSIZE" ) == MATCH )
         *xintsize = components[0];
      else if( strcmp( abbrevs[i], "ZINTSIZE" ) == MATCH )
         *zintsize = components[0];
      else if( strcmp( abbrevs[i], "XNUMINTV" ) == MATCH )
         *numintvx = (unsigned int)components[0];
      else if( strcmp( abbrevs[i], "ZNUMINTV" ) == MATCH )
         *numintvz = (unsigned int)components[0];
      else  {  /* this is the file's attribute type (e.g., VEP or DE) */
         *attrib_type = abbrevs[i];
         *attrib_index = i+1;
      }
   }

   if( rc != 0 )
      return( rc );

   if( *xintsize == 0.0  ||  *zintsize == 0.0  ||
       *numintvx == 0    ||  *numintvz == 0   )  {
      (void)sprintf( message,
                     "Interval information is missing for file '%s'.", dsn );
      (void)fprintf( stderr, "%s\n", message );
      return( 2 );
   }

   return 0;
}


static int tabletype( String tablename )
{
   static char   *tablenames[] = {
                     "mdmodmisc",
                     "mdmodwrld",
                     "mdmodloc",
                     "mdlineage",
                     "mdvermisc",
                     "mdcolmisc",
                     "mdcolsegs",
                     "mdcolatbs",
                     "mdatbmisc",
                     "mdatbpool",
                     "mdatbabrv",
                     "mdatbcomp",
                     "mdatbloc",
                     "mdsegpool"
                 };
   static unsigned int    ntabletypes = sizeof( tablenames ) / sizeof( char* );
   unsigned int           i;


   for( i = 0;  i < ntabletypes;  i++ )
      if( strcmp( lowercase( tablename ), tablenames[i] ) == MATCH )
         return( i+1 );

   return( 0 );
}


static Boolean nonexistant_model( String modelname )
{
   Modellistptr   mptr;


   for( mptr = modellist;  mptr;  mptr = mptr->next )  {
      if( strcmp( modelname, mptr->modelname ) == MATCH )
         return( FALSE );

      if( mptr->next == NULL )
         break;
   }

   if( mptr == NULL )
      mptr = modellist = new_node( Modellist );
   else  {
      mptr->next = new_node( Modellist );
      mptr = mptr->next;
   }

   mptr->modelname = strdup( modelname );
   mptr->versionlist = NULL;
   mptr->next = NULL;

   return( TRUE );
}


static Boolean nonexistant_version( String modelname, Version version )
{
   Modellistptr     mptr;
   Versionlistptr   vptr;


   for( mptr = modellist;  mptr;  mptr = mptr->next )
      if( strcmp( modelname, mptr->modelname ) == MATCH )  {
         for( vptr = mptr->versionlist;  vptr;  vptr = vptr->next )  {
            if( version == vptr->version )
               return( FALSE );

            if( vptr->next == NULL )
               break;
         }

         if( vptr == NULL )
            vptr = mptr->versionlist = new_node( Versionlist );
         else  {
            vptr->next = new_node( Versionlist );
            vptr = vptr->next;
         }

         vptr->version = version;
         vptr->historytree = NULL;
         vptr->next = NULL;

         return( TRUE );
      }

   /* should never get here */
   (void)fprintf( stderr,
                 "agm_io(nonexistant_version): model (%s) not found in list.\n",
                   modelname );
   exit( -1 );
   return 0; /* make the compiler happy */
}


static String build_history_string( String modelname, Version version, unsigned page, String history )
{
   Modellistptr     mptr;
   Versionlistptr   vptr;
   String           historystring;


   for( mptr = modellist;  mptr;  mptr = mptr->next )
      if( strcmp( modelname, mptr->modelname ) == MATCH )
         for( vptr = mptr->versionlist;  vptr;  vptr = vptr->next )
            if( version == vptr->version )  {
               vptr->historytree = tree_insert( vptr->historytree, page,
                                                history );
               historystring = strdup( "" );  /* malloc for realloc */
               read_tree( &historystring, vptr->historytree );

               return( historystring );
            }

   /* should never get here */
   (void)fprintf( stderr,
"agm_io(build_history_string): model (%s) or version (%d) not found in list.\n",
                   modelname, version );
   exit( -1 );
   return 0; /* make compiler happy */
}


static Historytreeptr tree_insert( Historytreeptr node, unsigned page, String history )
{
   if( node == NULL )  {
      node = new_node( Historytree );
      node->page = page;
      node->history = strdup( history );
      node->leftchild = node->rightchild = NULL;
   } else if( page < node->page )
      node->leftchild = tree_insert( node->leftchild, page, history );
   else if( page > node->page )
      node->rightchild = tree_insert( node->rightchild, page, history );
   else  {
      (void)sprintf( message,
                      "agm_io(tree_insert): duplicate history page (%d) found.",
                      page );
      (void)fprintf( stderr, "%s\n", message );
   }

   return( node );
}


static void read_tree( String *string, Historytreeptr node )
   /* must be ptr to string since realloc may chng */
{
   unsigned int   oldlen;


   if( node != NULL )  {
      read_tree( string, node->leftchild );

      /* infix order to get in ascending order */
      oldlen = strlen( *string );
      *string = (String)realloc( *string, oldlen + strlen( node->history ) +1 );
      (void)strcpy( string[oldlen], node->history );

      read_tree( string, node->rightchild );
   }
}


static int agm_flatfile_write( String filename )
{
   FILE         *fp;
   String	messageptr;
   String	modelname, comments, userid, ownername, expirydate;
   String	description, projectionsystem, history;
   String	creationdate, creationtime, lastupdate, lastuptime;
   String	abbreviation, tmp_abbrev;
   Units        units;
   Unit         unit;
   Factors      factors;
   Extents      extents;
   Location     location;
   Tupleptr     segment, ptr;
   Version      version, versionparent;
   Type         collectiontype;
   Id		collectionid, segmentid, attributeid, componentid, locationid;
   Id		parentid;
   float        zoffset, azimuth, latitude, longitude, factor, component;
   unsigned int i, j, n, offset, len;
   unsigned int nmodels, model, nversions, nthversion, nsegments, nsegmentpts;
   unsigned int page, pt;
   int          rc;
   /* *** 7-16-90 START *** */
   /* Interum code to renumber ids starting at 1 to conform with the old gmff
      library used by MSTS.  NOTE: code is flagged by 7-16-90 date. */
   Id		*idlist, coll_subtend, seg_subtend, atb_subtend;
   /* *** 7-16-90 END *** */


   if( (rc = access( filename, R_OK )) == 0 )  {
      (void)sprintf( message, "File '%s' already exists.", filename );
      return( -1 );
   }

   if( (fp = fopen( filename, "w" )) == NULL )  {
      (void)sprintf( message, "Unable to create file '%s'.", filename );
      return( -2 );
   }

   (void)agm( GetNumberModels, &nmodels );

   for( model = 1;  model <= nmodels;  model++ )  {
      if( (rc = agm( GetNthModelName, model, &modelname )) != 0 )  {
         (void)agm( GetMessage, &messageptr );
         (void)strcpy( message, messageptr );
         (void)fprintf( stderr, "%s\n", message );
         continue;
      }

      if( (rc = agm( GetComments, &comments )) == 0 )
         if( comments )  {
            (void)strtok( comments, "\n" );
            while( comments )  {
               (void)fprintf( fp, "! %s\n", comments );
               comments = strtok( (String)NULL, "\n" );
            }
            (void)fprintf( fp, "\n" );
            fflush( fp );
         }

      /* MDMODMISC */
      (void)agm( GetUserId, &userid );
      if( userid == NULL )
         userid = " ";
      if( strlen( userid ) == 0 )
         userid = " ";
      (void)agm( GetOwnerName, &ownername );
      if( ownername == NULL )
         ownername = " ";
      if( strlen( ownername ) == 0 )
         ownername = " ";
      (void)agm( GetExpiryDate, &expirydate );
      if( expirydate == NULL )
         expirydate = " ";
      if( strlen( expirydate ) == 0 )
         expirydate = " ";
      (void)agm( GetModelDescription, &description );
      if( description == NULL )
         description = " ";
      if( strlen( description ) == 0 )
         description = " ";
      fprintf( fp, "$table : mdmodmisc\n" );
      fprintf( fp, "! ModelName  Userid  Owner Name  Expiry Date" );
      fprintf( fp, "   Model Description\n" );
      fprintf( fp, "'%s', '%s', '%s', '%s', '%s'\n",
                   modelname, userid, ownername, expirydate, description );
      fflush( fp );

      /* MDMODWRLD */
      (void)agm( GetUnits, &units );
      if( units.x == NULL )
         units.x = " ";
      if( strlen( units.x ) == 0 )
         units.x = " ";
      if( units.y == NULL )
         units.y = " ";
      if( strlen( units.y ) == 0 )
         units.y = " ";
      if( units.z == NULL )
         units.z = " ";
      if( strlen( units.z ) == 0 )
         units.z = " ";
      (void)agm( GetFactors, &factors );
      (void)agm( GetExtents, &extents );
      fprintf( fp, "\n$table : mdmodwrld\n" );
      fprintf( fp, "! ModelName  X,Y,Z units  X,Y,Z factors" );
      fprintf( fp, "   Xmin, Xmax, Ymin, Ymax, Zmin, Zmax\n" );
      fprintf( fp, "'%s', '%s', '%s', '%s', %f,%f,%f, %f,%f,%f,%f,%f,%f\n",
                   modelname, units.x, units.y, units.z,
                   factors.x, factors.y, factors.z,
                   extents.min.x, extents.max.x, extents.min.y, extents.max.y,
                   extents.min.z, extents.max.z );
      fflush( fp );

      /* MDMODLOC */
      (void)agm( GetLocation, &location );
      (void)agm( GetZoffset, &zoffset );
      (void)agm( GetAzimuth, &azimuth );
      (void)agm( GetLatitude, &latitude );
      (void)agm( GetLongitude, &longitude );
      (void)agm( GetProjectionSystem, &projectionsystem );
      if( projectionsystem == NULL )
         projectionsystem = "  000000";
      if( strlen( projectionsystem ) == 0 )
         projectionsystem = "  000000";
      fprintf( fp, "\n$table : mdmodloc\n" );
      fprintf( fp, "! ModelName  X,Y,Z location  Z-offset  Azimuth" );
      fprintf( fp, "  Latitude  Longitude  Projection System\n" );
      fprintf( fp, "'%s', %f,%f,%f, %f,%f, %f,%f, '%s'\n",
                   modelname, location.x, location.y, location.z, zoffset,
                   azimuth, latitude, longitude, projectionsystem );
      fflush( fp );


      (void)agm( GetNumberVersions, &nversions );
      for( nthversion = 1;  nthversion <= nversions;  nthversion++ )  {
         if( (rc = agm( GetNthVersionNumber, nthversion, &version ))
             != 0 )  {
            (void)agm( GetMessage, &messageptr );
            (void)strcpy( message, messageptr );
            (void)fprintf( stderr, "%s\n", message );
            continue;
         }

         /* MDLINEAGE */
         if( (rc = agm( GetVersionHistory, &history )) == 0 )  {
            if( history == NULL )
               history = " ";
            if( strlen( history ) == 0 )
               history = " ";
            fprintf( fp, "\n$table : mdlineage\n" );
            fprintf( fp, "! ModelName  Version  Page  History\n" );
            len = strlen( history );  page = 1;
            for( offset = 0;  offset < len;  offset += LINEAGE_STRING_LEN )
               fprintf( fp, "'%s', %d, %d, '%s'\n",
                            modelname, version, page++, &history[offset] );
            fflush( fp );
         }

         /* MDVERMISC */
         (void)agm( GetVersionParent, &versionparent );
         if( (rc = agm( GetVersionCreationDate, &creationdate )) != 0 )
            creationdate = todays_date();
         if( creationdate == NULL )
            creationdate = todays_date();
         if( strlen( creationdate ) == 0 )
            creationdate = todays_date();
         if( (rc = agm( GetVersionCreationTime, &creationtime )) != 0 )
            creationtime = current_time();
         if( creationtime == NULL )
            creationtime = current_time();
         if( strlen( creationtime ) == 0 )
            creationtime = current_time();
         if( (rc = agm( GetVersionLastUpDate, &lastupdate )) != 0 )
            lastupdate = creationdate;
         if( lastupdate == NULL )
            lastupdate = creationdate;
         if( strlen( lastupdate ) == 0 )
            lastupdate = creationdate;
         if( (rc = agm( GetVersionLastUpTime, &lastuptime )) != 0 )
            lastuptime = creationtime;
         if( lastuptime == NULL )
            lastuptime = creationtime;
         if( strlen( lastuptime ) == 0 )
            lastuptime = creationtime;
         if( (rc = agm( GetVersionDescription, &description )) != 0 )
            description = " ";
         if( description == NULL )
            description = " ";
         if( strlen( description ) == 0 )
            description = " ";
         fprintf( fp, "\n$table : mdvermisc\n" );
         fprintf( fp, "! ModelName  Version  Parent  Creation Date/Time" );
         fprintf( fp, "  Last Update/time  Description\n" );
         fprintf( fp, "'%s', %d, %d, '%s', '%s', '%s', '%s', '%s'\n",
                      modelname, version, versionparent, creationdate,
                      creationtime, lastupdate, lastuptime, description );
         fflush( fp );

         /* MDCOLMISC */
         (void)agm( RestrictCollectionType, ALL_TYPES );
         if( (rc = agm( GetNumberCollections, &n )) != 0 )  {
            (void)agm( GetMessage, &messageptr );
            (void)strcpy( message, messageptr );
            (void)fprintf( stderr, "%s\n", message );
            n = 0;
         }
         if( n > 0 )  {
   /* *** 7-16-90 START *** */
   idlist = (Id *)malloc( n * sizeof( Id ) );
   (void)agm( GetNthCollectionMisc, 1, &idlist[0], &parentid, &collectiontype,
              &description );
   for( i = 1;  i < n;  i++ )
      (void)agm( GetNextCollectionMisc, &idlist[i], &parentid, &collectiontype,
                 &description );
/* printf( "id's before sort:\n" );
for( i = 0;  i < n;  i++ ) printf( "%d ", idlist[i] );
printf( "\n" );
*/
   qsort( (char *)idlist, n, sizeof( Id ), idcompare );
/* printf( "id's after sort:\n" );
for( i = 0;  i < n;  i++ ) printf( "%d ", idlist[i] );
printf( "\n" );
*/
   coll_subtend = idlist[0] - 1;  /* subtract from all collection ids */
/* printf( "coll_subtend=%d\n", coll_subtend ); */
   free( (char *)idlist );
   /* *** 7-16-90 END *** */
            fprintf( fp, "\n$table : mdcolmisc\n" );
            fprintf( fp, "! ModelName  Version  CollectionID  Parent  Type" );
            fprintf( fp, "  Description\n" );
            if( (rc = agm( GetNthCollectionMisc, 1,
                           &collectionid, &parentid, &collectiontype,
                           &description )) == 0 )  {
               if( collectiontype == DATA_SET )  {
                  unload_attrib_file( description );
                  (void)agm( ActivateCollection, collectionid );
               }
               if( description == NULL )
                  description = " ";
               if( strlen( description ) == 0 )
                  description = " ";
   if( parentid != 0 )
      parentid -= coll_subtend;		/* *** 7-16-90 *** */
   collectionid -= coll_subtend;	/* *** 7-16-90 *** */
               (void)fprintf( fp, "'%s', %d, %d, %d, %d, '%s'\n",
                                   modelname, version, collectionid, parentid, 
                                   collectiontype, description );
            } else {
               (void)agm( GetMessage, &messageptr );
               (void)strcpy( message, messageptr );
               (void)fprintf( stderr, "%s\n", message );
            }
            for( i = 1;  i < n;  i++ )  {
               if( (rc = agm( GetNextCollectionMisc,
                              &collectionid, &parentid, &collectiontype,
                              &description )) != 0 )  {	/* faster */
                  (void)agm( GetMessage, &messageptr );
                  (void)strcpy( message, messageptr );
                  (void)fprintf( stderr, "%s\n", message );
                  continue;
               }
               if( collectiontype == DATA_SET )  {
                  unload_attrib_file( description );
                  (void)agm( ActivateCollection, collectionid );
               }
               if( description == NULL )
                  description = " ";
               if( strlen( description ) == 0 )
                  description = " ";
   if( parentid != 0 )
      parentid -= coll_subtend;		/* *** 7-16-90 *** */
   collectionid -= coll_subtend;	/* *** 7-16-90 *** */
               (void)fprintf( fp, "'%s', %d, %d, %d, %d, '%s'\n",
                                   modelname, version, collectionid, parentid, 
                                   collectiontype, description );
            }
            fflush( fp );
         }

         /* MDCOLSEGS */
         if( (rc = agm( GetNumberCollectionSegmentLinks, &n )) != 0 )  {
            (void)agm( GetMessage, &messageptr );
            (void)strcpy( message, messageptr );
            (void)fprintf( stderr, "%s\n", message );
            n = 0;
         }
         if( n > 0 )  {
   /* *** 7-16-90 START *** */
   idlist = (Id *)malloc( n * sizeof( Id ) );
   (void)agm( GetNthCollectionSegmentLink, 1, &collectionid, &idlist[0] );
   for( i = 1;  i < n;  i++ )
      (void)agm( GetNextCollectionSegmentLink, &collectionid, &idlist[i] );
   qsort( (char *)idlist, n, sizeof( Id ), idcompare );
   seg_subtend = idlist[0] - 1;  /* subtract from all segment ids */
   free( (char *)idlist );
   /* *** 7-16-90 END *** */
            fprintf( fp, "\n$table : mdcolsegs\n" );
            fprintf( fp, "! ModelName  Version  CollectionID  SegmentID\n" );
            if( (rc = agm( GetNthCollectionSegmentLink, 1,
                           &collectionid, &segmentid )) == 0 )
{	/* *** 7-16-90 *** */
   collectionid -= coll_subtend;	/* *** 7-16-90 *** */
   segmentid -= seg_subtend;		/* *** 7-16-90 *** */
               fprintf( fp, "'%s', %d, %d, %d\n",
                            modelname, version, collectionid, segmentid );
}	/* *** 7-16-90 *** */
            for( i = 1;  i < n;  i++ )  {
               if( (rc = agm( GetNextCollectionSegmentLink,
                              &collectionid, &segmentid )) != 0 ) /* faster*/
                  continue;
   collectionid -= coll_subtend;	/* *** 7-16-90 *** */
   segmentid -= seg_subtend;		/* *** 7-16-90 *** */
               fprintf( fp, "'%s', %d, %d, %d\n",
                            modelname, version, collectionid, segmentid );
            }
            fflush( fp );
         }

         /* MDCOLATBS */
         if( (rc = agm( GetNumberCollectionAttributeLinks, &n )) != 0 )  {
            (void)agm( GetMessage, &messageptr );
            (void)strcpy( message, messageptr );
            (void)fprintf( stderr, "%s\n", message );
            n = 0;
         }
         if( n > 0 )  {
   /* *** 7-16-90 START *** */
   idlist = (Id *)malloc( n * sizeof( Id ) );
   for( i = 1;  i <= n;  i++ )
      (void)agm( GetNthCollectionAttributeLink, i,
                 &collectionid, &idlist[i-1] );
   qsort( (char *)idlist, n, sizeof( Id ), idcompare );
   atb_subtend = idlist[0] - 1;  /* subtract from all attribute ids */
   free( (char *)idlist );
   /* *** 7-16-90 END *** */
            fprintf( fp, "\n$table : mdcolatbs\n" );
            fprintf( fp, "! ModelName  Version  CollectionID  AttributeID\n" );
            for( i = 1;  i <= n;  i++ )  {
               if( (rc = agm( GetNthCollectionAttributeLink, i,
                              &collectionid, &attributeid )) != 0 )
                  continue;
   collectionid -= coll_subtend;	/* *** 7-16-90 *** */
   attributeid -= atb_subtend;		/* *** 7-16-90 *** */
               fprintf( fp, "'%s', %d, %d, %d\n",
                            modelname, version, collectionid, attributeid );
            }
            fflush( fp );
         }
      }  /* END version LOOP */

      /* MDATBMISC */
      (void)agm( GetNumberAttributeDescriptions, &n );
      if( n > 0 )  {
         fprintf( fp, "\n$table : mdatbmisc\n" );
         fprintf( fp, "! ModelName  Abbrev  Unit  Factor  Description\n" );
         for( i = 1;  i <= n;  i++ )  {
            if( (rc = agm( GetNthAttributeDescription, i,
                           &abbreviation, &unit, &factor, &description ))
                != 0 )
               continue;
            if( abbreviation == NULL )
               abbreviation = " ";
            if( strlen( abbreviation ) == 0 )
               abbreviation = " ";
            if( unit == NULL )
               unit = " ";
            if( strlen( unit ) == 0 )
               unit = " ";
            if( description == NULL )
               description = " ";
            if( strlen( description ) == 0 )
               description = " ";
            fprintf( fp, "'%s', '%s', '%s', %f, '%s'\n",
                         modelname, abbreviation, unit, factor, description );
         }
         fflush( fp );
      }

      /* MDATBABRV */
      (void)agm( GetNumberAttributeAbbreviations, &n );
      if( n > 0 )  {
         fprintf( fp, "\n$table : mdatbabrv\n" );
         fprintf( fp, "! ModelName  AttributeId  Abbrev\n" );
         for( i = 1;  i <= n;  i++ )  {
            if( (rc = agm( GetNthAttributeAbbreviation, i,
                           &attributeid, &abbreviation )) != 0 )
               continue;
   attributeid -= atb_subtend;		/* *** 7-16-90 *** */
            if( abbreviation == NULL )
               abbreviation = " ";
            if( strlen( abbreviation ) == 0 )
               abbreviation = " ";
            fprintf( fp, "'%s', %d, '%s'\n",
                         modelname, attributeid, abbreviation );
         }
         fflush( fp );
      }

      /* MDATBCOMP */
      (void)agm( GetNumberAttributeComponents, &n );
      if( n > 0 )  {
         fprintf( fp, "\n$table : mdatbcomp\n" );
         fprintf( fp, "! ModelName  AttributeId  ComponentId  Value\n" );
         for( i = 1;  i <= n;  i++ )  {
            if( (rc = agm( GetNthAttributeComponent, i,
                           &attributeid, &componentid, &component )) != 0 )
               continue;
   attributeid -= atb_subtend;		/* *** 7-16-90 *** */
            fprintf( fp, "'%s', %d, %d, %f\n", 
                         modelname, attributeid, componentid, component );
         }
         fflush( fp );
      }

      /* MDATBLOC */
      (void)agm( GetNumberAttributeLocations, &n );
      if( n > 0 )  {
         fprintf( fp, "\n$table : mdatbloc\n" );
         fprintf( fp, "! ModelName  AttributeId  LocationID  X  Y  Z\n" );
         for( i = 1;  i <= n;  i++ )  {
            if( (rc = agm( GetNthAttributeLocation, i,
                           &attributeid, &locationid, &location )) != 0 )
               continue;
   attributeid -= atb_subtend;		/* *** 7-16-90 *** */
            fprintf( fp, "'%s', %d, %d, %f, %f, %f\n", 
                         modelname, attributeid, locationid,
                         location.x, location.y, location.z );
         }
         fflush( fp );
      }

      /* MDATBPOOL */
      /* NOTE: MDATBPOOL is output for the sake of the old gmff lib users. */
      /*       Only output 1st component & location for multi-valued attribs. */
      (void)agm( GetNumberAttributeAbbreviations, &n );
      if( n > 0 )  {
         fprintf( fp, "\n$table : mdatbpool\n" );
         fprintf( fp, "! ModelName  AttributeId  Abbrev  Value  X  Y  Z\n" );
         for( i = 1;  i <= n;  i++ )  {
            if( (rc = agm( GetNthAttributeAbbreviation, i,
                           &attributeid, &abbreviation )) != 0 )
               continue;
            if( abbreviation == NULL )
               abbreviation = " ";
            if( strlen( abbreviation ) == 0 )
               abbreviation = " ";
            tmp_abbrev = uppercase( strdup( abbreviation ) );
            if( strcmp( tmp_abbrev, "RGB" ) == MATCH )  {
               /* RGB triplet gets stored in location field */
               for( j = 0;  j < 3;  j++ )
                  if( (rc = agm( GetIdAttributeNthComponent, attributeid, j+1,
                                 &componentid, &location.x+j )) != 0 )
                     *(&location.x+j) = 0.0;
               component = 0.0;
            } else {
               if( (rc = agm( GetIdAttributeNthComponent, attributeid, 1,
                              &componentid, &component )) != 0 )
                  component = 0.0;
               if( (rc = agm( GetIdAttributeNthLocation, attributeid, 1,
                              &locationid, &location )) != 0 )
                  location.x = location.y = location.z = 0.0;
            }
            free( (char *)tmp_abbrev );
   attributeid -= atb_subtend;		/* *** 7-16-90 *** */
            fprintf( fp, "'%s', %d, '%s', %f, %f, %f, %f\n",
                         modelname, attributeid, abbreviation, component, 
                         location.x, location.y, location.z );
         }
         fflush( fp );
      }

      /* MDSEGPOOL */
      (void)agm( GetNumberSegments, &nsegments );
      if( nsegments > 0 )  {
         fprintf( fp, "\n$table : mdsegpool\n" );
         fprintf( fp, "! ModelName  SegmentID  PointID  X Y Z\n" );
         if( (rc = agm( GetNthSegment, 1,
                        &segmentid, &segment, &nsegmentpts )) == 0 )  {
   segmentid -= seg_subtend;		/* *** 7-16-90 *** */
            for( pt = 1, ptr = segment;  pt <= nsegmentpts;  pt++, ptr++ )
               fprintf( fp, "'%s', %d, %d, %f,%f,%f\n",
                        modelname, segmentid, pt, ptr->x, ptr->y, ptr->z );
            for( i = 1;  i < nsegments;  i++ )  {
               if( (rc = agm( GetNextSegment,
                              &segmentid, &segment, &nsegmentpts )) != 0 )
                  continue;
   segmentid -= seg_subtend;		/* *** 7-16-90 *** */
               for( pt = 1, ptr = segment;  pt <= nsegmentpts;  pt++, ptr++ )
                  fprintf( fp, "'%s', %d, %d, %f,%f,%f\n",
                           modelname, segmentid, pt, ptr->x, ptr->y, ptr->z );
            }
         }
      }
   }  /* END for model LOOP */

   fclose( fp );

   return( rc );
}
/* *** 7-16-90 START *** */
/* static Id */		/* Id's are unsigned int's...thus, can't do this way */
static int idcompare( const void *i, const void *j )
/* Id *i, *j; */
{
   return( *(int*)i - *(int*)j );
}
/* *** 7-16-90 END *** */


static int unload_attrib_file( String dsn )
{
   FILE			*fp;
   String		path, messageptr, attrib_type;
   float		xorigin, zorigin, xintsize, zintsize, *attrib_array;
   unsigned int		numintvx, numintvz, attrib_index, nelements;
   unsigned int		suffix;
   int			rc;


   if( override_attrib_dsn != NULL )
      dsn = override_attrib_dsn;

   /* check if have write access to path area */
   if( !write_access( (path = extract_path( dsn )) ) )  {
      (void)sprintf( message, "Do not have write access to directory '%s'.",
                     path );
      (void)fprintf( stderr, "%s\n", message );
      return( 1 );
   }
   free( (char *)path );
      
   FOREVER  {
      if( (rc = access( dsn, R_OK )) == 0 )  {
         /* file already exists; increment suffix */
         (void)sscanf( &dsn[strlen( dsn )-6], "%u", &suffix );
         (void)sprintf( &dsn[strlen( dsn )-6], "%06u", ++suffix );
      } else
         break;
   }

   if( (fp = fopen( dsn, "w" )) == NULL )  {
      (void)sprintf( message, "Unable to create file '%s'.", dsn );
      (void)fprintf( stderr, "%s\n", message );
      return( 2 );
   }

   if( strncmp( dsn, "/tmp/", 5 ) == MATCH )  {
      (void)sprintf( message,
      "Note: wrote AGM file '%s' to /tmp/ area.  File will need to be moved!\n",
                     dsn );
      (void)fprintf( stderr, "%s\n", message );
   }

   if( (rc = get_2d_grid_info( dsn, &xorigin, &zorigin, &xintsize, &zintsize,
                               &numintvx, &numintvz,
                               &attrib_type, &attrib_index )) != 0 )  {
      fclose( fp );
      return( rc );
   }

   if( (rc = agm( ActivateCollectionIndexedAttribute, attrib_index ))
       != 0 )  {
      (void)agm( GetMessage, &messageptr );
      (void)strcpy( message, messageptr );
      fprintf( stderr, "%s\n", message );
      fclose( fp );
      return( rc );
   }

   if( (rc = agm( GetAttributeSet, &nelements, &attrib_array )) != 0 )  {
      (void)agm( GetMessage, &messageptr );
      (void)strcpy( message, messageptr );
      fprintf( stderr, "%s\n", message );
      fclose( fp );
      return( rc );
   }

   if( nelements != (numintvx + 1) * (numintvz + 1) )  {
      (void)sprintf( message,
                     "Mismatch in number of elements for attribute file '%s'.",
                     dsn );
      (void)fprintf( stderr, "%s\n", message );
      fclose( fp );
      return( 3 );
   }

   if( fwrite( (char *)attrib_array, (int)(nelements*sizeof( float )), 1, fp )
       != 1 )  {
      (void)sprintf( message, "Zero items written to attribute file '%s'.",
                     dsn );
      (void)fprintf( stderr, "%s\n", message );
      fclose( fp );
      return( 4 );
   }

   return 0;
}


String agm_io_generate_dsn( void )
{
   static String	dsn, template = "agm_atrb000001";

   String		path;
   unsigned int		suffix;


   if( !filename )
      filename = "./";

   /* check if have write access to path area */
   if( !write_access( (path = extract_path( filename )) ) )
      path = "/tmp/";

   dsn = concat( path, template, NULL );

   FOREVER  {
      if( access( dsn, R_OK ) == 0 )  {
         /* file already exists; increment suffix */
         (void)sscanf( &dsn[strlen( dsn )-6], "%u", &suffix );
         (void)sprintf( &dsn[strlen( dsn )-6], "%06u", ++suffix );
      } else
         break;
   }

   free( (char *)path );
      
   return( dsn );
}


void agm_io_delete_model( String modelname )
{
   Modellistptr		mptr, prev_mptr;
   Versionlistptr	vptr, next_vptr;


   for( prev_mptr = NULL, mptr = modellist;  mptr;  mptr = mptr->next )  {
      if( strcmp( modelname, mptr->modelname ) == MATCH )  {
         if( prev_mptr == NULL )
            modellist = mptr->next;
         else
            prev_mptr->next = mptr->next;

         free( (char *)mptr->modelname );

         for( vptr = mptr->versionlist;  vptr;  vptr = next_vptr )  {
            next_vptr = vptr->next;

            if( vptr->historytree )
               free( (char *)vptr->historytree );

            free( (char *)vptr );
         }

         free( (char *)mptr );

         break;
      }

      prev_mptr = mptr;
   }
}


static char *allocstr( String s )
{
   char *p;


   if( s == NULL ) return NULL;

   p = malloc( (unsigned) ( strlen( s ) + 1 ) );
   if( p == NULL ) {
      fprintf( stderr, "allocstr: malloc failure.\n" );
      return p;
   }

   (void)strcpy( p, s );
   return p;
}


static char *uppercase( String s )
{
   char *c = s;


   for( ;  *c;  c++ )
      if( isascii( *c ) )
         if( islower( *c ) )
            *c = toupper( *c );

   return s;
}


static char *lowercase( String s )
{
   char *c = s;


   for( ;  *c;  c++ )
      if( isascii( *c ) )
         if( isupper( *c ) )
            *c = tolower( *c );

   return s;
}


static int rtend( String s )  /* return the length of the blank delimited string. */
{
   char *p = s + strlen( s ) - 1;


   while( *p == ' ' )
      --p;

   return( p - s + 1 );
}


static Boolean buffered = FALSE;
static char    *buffer = NULL;


static int getline( FILE *fp, String s )
{
   int  rc, nbytes;


   if( buffered )  {
      buffered = FALSE;

      (void)strcpy( s, buffer );
      (void)free( buffer );

      return( strlen( s ) );
   }

   nbytes = 0;
   rc = fscanf( fp, "%[^\n]%*[\n]%n", s, &nbytes );

   if( rc == EOF  ||  rc == 0 )
      return( EOF );
   else
      return( nbytes );
}


static int ungetline( String s )
{
   if( buffered )  {
      fprintf( stderr,
               "ungetline: requested unget without intermittant get.\n" );
      (void)free( buffer );
   }

   buffer = (char *)malloc( strlen( s ) + 1 );
   (void)strcpy( buffer, s );

   buffered = TRUE;

   return( 0 );
}

 
static char *tokenize( String string, String front_delimeter_set, String back_delimeter_set )
{
   static char           *buffer;
   char                  *token;
   unsigned int          start, end;


   if( string != NULL )
      buffer = string;

   if( (int)strlen( front_delimeter_set ) > 0 )
      start = strspn( buffer, front_delimeter_set );
   else
      start = 0;
   token = buffer += start;

   if( (end = strcspn( buffer, back_delimeter_set )) == 0 )
      return( NULL );
   buffer += end;

   *buffer++ = '\0';

   return( token );
}


static char *todays_date( void )  /* return today's date in 'mm/dd/yy' form */
{
   static char  today[11+1];
   time_t       timeval;
   char         *date;


   time( &timeval );
   date = ctime( &timeval );

   (void)strncpy( today, date+4, 7 );   today[7] = '\0';
   (void)strncat( &today[7], date+20, 4 );

   return( today );
}


static char *current_time( void )  /* return current time as "xx:xx:xx" */
{
   static char  currenttime[8+1];
   time_t       timeval;
   char         *date;


   time( &timeval );   date = ctime( &timeval );

   strncpy( currenttime, date+11, 8 );   currenttime[8] = '\0';

   return( currenttime );
}


static char *extract_path( String dsn )
{
   int		i;
   char         *path; 
 
 
   for( i = strlen( dsn )-1;  i >=0;  i-- )
      if( dsn[i] == '/' )
         break;
 
   if( i > 0 )  {
      path = strdup( dsn );
      path[i+1] = '\0';
      return( path ); 
   } else
      return( strdup( "./" ) );  /* make deallocatable */
}


static Boolean write_access( String dir )
{
   FILE		*fp;
   char		*fn;


   fp = fopen( (fn = tempnam( dir, NULL )), "w" );
   free( (char *)fn );

   if( fp == NULL )
      return FALSE;
   else  {
      fclose( fp );
      (void)unlink( fn );
      return TRUE;
   }
}


/*
	Notes: strdup obtains the space for the new string using malloc(3C).
		realloc'ing this space is dicey since realloc may be linked
		with malloc(3X).
		(The malloc(3X)  routines  are  space-efficient,  but have
		slower performance.
		The standard, fully  SCD-compliant  malloc(3C)  routines
		are a trade-off between performance and space-efficiency.)
*/


static String concat( String s, ... )
{
	va_list		args;
	String		compoundString = NULL;
	unsigned int	newlen;

	va_start( args, s );
	while( s )
	{
		if( compoundString )
			(void)strcat(
				compoundString = realloc( compoundString,
					(newlen+=(int)strlen(s)) + 1 ),
				s );
		else
#if defined(sun) && defined(__SVR4)
		{
			/* strlen/strdup combo unsafe on SOLARIS */
			compoundString = (String)malloc( (newlen=(int)strlen(s))+1 );
			(void)strcpy( compoundString, s );
		}
#else
			newlen = (int)strlen( compoundString = strdup( s ) );
#endif

		s = va_arg( args, String );
	}

	va_end( args );

	return compoundString;
}
