/***********************************************************************
 *                copyright 2001, Amoco Production Company             *
 *                            All Rights Reserved                      *
 *                    an affiliate of BP America Inc.                  *
 ***********************************************************************/
/*
	Author: Bob Mars
	Purpose: linked list object class hierarchy
*/

#ifndef _linked_list
#define _linked_list

#include <sys/types.h>		/* caddr_t */
#include <memory.h>		/* memcpy */
#ifdef USE_SYS_MALLOC
#include <sys/malloc.h>
#else
#include <malloc.h>
#endif
#include "funcproto.h"


typedef struct _LinkedListChain {
	caddr_t				element;
	int				size;
	struct _LinkedListChain	*next;
} LinkedListChain, *LinkedListChainPtr;

/* -------------------------------- PUBLIC ---------------------------------- */

#if NeedFunctionPrototypes
typedef struct _LinkedList *MakeKnown;

typedef struct _LinkedList {
	LinkedListChainPtr	front;
	LinkedListChainPtr	ptr;
	void			(*reset)( struct _LinkedList* );
	void			(*insert)( struct _LinkedList*, caddr_t e, int l );
	void			(*append)( struct _LinkedList*, caddr_t e, int l );
	caddr_t			(*element)( struct _LinkedList* );
	char			*(*string_element)( struct _LinkedList* );
	int			(*element_size)( struct _LinkedList* );
	caddr_t			(*prev_element)( struct _LinkedList* );
	caddr_t			(*next_element)( struct _LinkedList* );
	int			(*end)( struct _LinkedList* );
	int			(*delete)( struct _LinkedList* );
	int			(*nelements)( struct _LinkedList* );
	caddr_t			*(*to_list)( struct _LinkedList* );
	caddr_t			*(*to_string_list)( struct _LinkedList* );
	void			(*clean_up)( struct _LinkedList* );
} LinkedList, *LinkedListPtr;
/* #define LLIST_CLASS \
	LinkedListChainPtr	front; \
	LinkedListChainPtr	ptr; \
	void			(*reset)( struct _LinkedList* ); \
	void			(*insert)( struct _LinkedList*, caddr_t e, int l ); \
	void			(*append)( struct _LinkedList*, caddr_t e, int l ); \
	caddr_t			(*element)( struct _LinkedList* ); \
	char			*(*string_element)( struct _LinkedList* ); \
	int			(*element_size)( struct _LinkedList* ); \
	caddr_t			(*prev_element)( struct _LinkedList* ); \
	caddr_t			(*next_element)( struct _LinkedList* ); \
	int			(*end)( struct _LinkedList* ); \
	int			(*delete)( struct _LinkedList* ); \
	int			(*nelements)( struct _LinkedList* ); \
	caddr_t			*(*to_list)( struct _LinkedList* ); \
	caddr_t			*(*to_string_list)( struct _LinkedList* ); \
	void			(*clean_up)( struct _LinkedList* );

typedef struct _LinkedList {
	LLIST_CLASS;
} LinkedList, *LinkedListPtr;
*/
#else
typedef struct _LinkedList {
/* private */
	LinkedListChainPtr	front;		/* front of list */
	LinkedListChainPtr	ptr;		/* current element */
/* public */
	void			(*reset)();		/* to front of list */
	void			(*insert)();		/* in front of */
	void			(*append)();		/* append after */
	caddr_t			(*element)();		/* return current */
	char			*(*string_element)();	/* return " + '\0' */
	int			(*element_size)();	/* size of current */
	caddr_t			(*prev_element)();	/* pt to and return it*/
	caddr_t			(*next_element)();	/* pt to and return it*/
	int			(*end)();		/* end of list? */
	int			(*delete)();		/* current element */
	int			(*nelements)();		/* # elements in list */
	caddr_t			*(*to_list)();		/* contiguous list */
	caddr_t			*(*to_string_list)();	/*     "  string list */
	void			(*clean_up)();		/* free space */
} LinkedList, *LinkedListPtr;
#endif


/* -------------------------------- PRIVATE --------------------------------- */

static LinkedListChainPtr
linked_list_prev( self )
	LinkedListPtr	self;
{
	LinkedListChainPtr	ptr = self->front, prev = NULL;


	if( self->ptr == NULL ) {
		for( ;  ptr;  prev = ptr, ptr = ptr->next )
			if( ptr->next == NULL ) {
				self->ptr = ptr;
				return ptr;
			}
	} else
		for( ;  ptr;  prev = ptr, ptr = ptr->next )
			if( ptr == self->ptr )
				return prev;

	return NULL;
}


/* ------------------- PSEUDO PUBLIC (through indirection) ------------------ */

static void
linked_list_reset( self )
	LinkedListPtr	self;
{
	self->ptr = self->front;
}


static void
linked_list_insert( self, element, element_size )
	LinkedListPtr	self;
	caddr_t		element;
	int		element_size;
{
	LinkedListChainPtr	next, prev;


	if( !self->front ) {
		self->ptr = self->front =
			(LinkedListChainPtr)malloc( sizeof( LinkedListChain ) );
		next = NULL;
	} else {
		next = self->ptr;

		if( (prev = linked_list_prev( self )) )
			prev->next = self->ptr =
			(LinkedListChainPtr)malloc( sizeof( LinkedListChain ) );
		else
			self->front = self->ptr =
			(LinkedListChainPtr)malloc( sizeof( LinkedListChain ) );
	}

	self->ptr->element = (caddr_t)malloc( element_size );
	(void)memcpy( (char *)self->ptr->element, (char *)element,
			element_size );
	self->ptr->size = element_size;
	self->ptr->next = next;
}


static void
linked_list_append( self, element, element_size )
	LinkedListPtr	self;
	caddr_t		element;
	int		element_size;
{
	for( self->ptr = self->front;  self->ptr;  self->ptr = self->ptr->next )
		if( self->ptr->next == NULL )
			break;

	if( !self->ptr )
		self->ptr = self->front =
			(LinkedListChainPtr)malloc( sizeof( LinkedListChain ) );
	else {
		self->ptr->next =
			(LinkedListChainPtr)malloc( sizeof( LinkedListChain ) );
		self->ptr = self->ptr->next;
	}

	self->ptr->element = (caddr_t)malloc( element_size );
	(void)memcpy( (char *)self->ptr->element, (char *)element,
			element_size );
	self->ptr->size = element_size;
	self->ptr->next = NULL;
}


static char
*linked_list_string_element( self )
	LinkedListPtr	self;
{
	char		*tmp;

	if( self->ptr )  {
		tmp = (char *)malloc( self->ptr->size + 1 );
		(void)memcpy( tmp, (char *)self->ptr->element, self->ptr->size);
		tmp[self->ptr->size] = '\0';
		return tmp;
	} else
		return NULL;
}


static caddr_t
linked_list_element( self )
	LinkedListPtr	self;
{
	if( self->ptr )
		return self->ptr->element;
	else
		return NULL;
}


static int
linked_list_element_size( self )
	LinkedListPtr	self;
{
	if( self->ptr )
		return self->ptr->size;
	else
		return 0;
}


static caddr_t
linked_list_prev_element( self )
	LinkedListPtr	self;
{
	return (linked_list_prev( self ))->element;
}


static caddr_t
linked_list_next_element( self )
	LinkedListPtr	self;
{
	if( self->ptr ) {
		self->ptr = self->ptr->next;

		if( self->ptr )			/* note: is next already */
			return self->ptr->element;
		else
			return NULL;
	} else
		return NULL;
}


static int
linked_list_end( self )		/* return 1(True) if end of list */
	LinkedListPtr	self;
{
	if( self->ptr == NULL )
		return 1;
	else
		return 0;
}


static int
linked_list_delete( self )
	LinkedListPtr	self;
{
	LinkedListChainPtr	prev;


	if( self->ptr == NULL )
		return 1;

	free( (char *)self->ptr->element );

	if( (prev = linked_list_prev( self )) ) {
		prev->next = self->ptr->next;
		free( (char *)self->ptr );
		self->ptr = prev->next;
	} else {
		prev = self->front = self->front->next;
		free( (char *)self->ptr );
		self->ptr = prev;
	}

	return 0;
}


static int
linked_list_nelements( self )
	LinkedListPtr	self;
{
	LinkedListChainPtr	ptr = self->front;
	int			count = 0;


	for( ;  ptr;  ptr = ptr->next )
		++count;

	return count;
}


static caddr_t
*linked_list_to_list( self )
	LinkedListPtr	self;
{
	LinkedListChainPtr	ptr;
	int			count = 0;
	caddr_t			*generic_list, *g;


	for( ptr = self->front;  ptr;  ptr = ptr->next )
		++count;

	if( count == 0 )
		return (caddr_t *)0;

	generic_list = (caddr_t *)malloc( (count+1) * sizeof( caddr_t ) );

	for( ptr = self->front, g = generic_list;  ptr;  ptr = ptr->next, g++ ){
		*g = (caddr_t)malloc( ptr->size );
		(void)memcpy( (char *)*g, (char *)ptr->element, ptr->size );
	}

	*g = NULL;

	return generic_list;
}


static char
**linked_list_to_string_list( self )
	LinkedListPtr	self;
{
	LinkedListChainPtr	ptr;
	int			count = 0;
	char			**string_list, **s;


	for( ptr = self->front;  ptr;  ptr = ptr->next )
		++count;

	if( count == 0 )
		return (char **)0;

	string_list = (char **)malloc( (count+1) * sizeof( char * ) );

	for( ptr = self->front, s = string_list;  ptr;  ptr = ptr->next, s++ ){
		*s = (char *)malloc( ptr->size + 1 );
		(void)memcpy( (char *)*s, (char *)ptr->element, ptr->size );
		(*s)[ptr->size] = '\0';
	}

	*s = NULL;

	return string_list;
}


static void
linked_list_clean_up( self )
	LinkedListPtr	self;
{
	LinkedListChainPtr	ptr = self->front, next;


	for( ;  ptr;  ptr = next )  {
		next = ptr->next;
		free( (char *)ptr->element );
		free( (char *)ptr );
	}

	self->front = NULL;
	self->ptr = NULL;
}


static void
linked_list_init( linked_list )
	LinkedListPtr	linked_list;
{
	linked_list->front = NULL;
	linked_list->ptr = NULL;
	linked_list->reset = linked_list_reset;
	linked_list->insert = linked_list_insert;
	linked_list->append = linked_list_append;
	linked_list->element = linked_list_element;
	linked_list->string_element = linked_list_string_element;
	linked_list->element_size = linked_list_element_size;
	linked_list->prev_element = linked_list_prev_element;
	linked_list->next_element = linked_list_next_element;
	linked_list->end = linked_list_end;
	linked_list->delete = linked_list_delete;
	linked_list->nelements = linked_list_nelements;
	linked_list->to_list = linked_list_to_list;
	linked_list->to_string_list = linked_list_to_string_list;
	linked_list->clean_up = linked_list_clean_up;
}

#endif /* _linked_list */
