/***********************************************************************
 *                copyright 2001, Amoco Production Company             *
 *                            All Rights Reserved                      *
 *                    an affiliate of BP America Inc.                  *
 ***********************************************************************/
/* 
	Program: xsd
	Purpose: X-windows seismic display program.
*/

#include <signal.h>
#ifdef USE_SYS_MALLOC
#include <sys/malloc.h>
#else
#include <malloc.h>
#endif
#ifdef SunOS
#if defined(sun) && !defined(__SVR4)
#include <memory.h>	/* memcpy */
#endif
#endif
#include <time.h>
#include <string.h>
#include "display.h"		/* Header file contain global variables. */
#include "xsd.bit"		/* Bitmap file for xsd'x icon. */
#include "arg.h"
#include "prototyping.h"
#include <ut_defs.h>
#include <cu_defs.h>

#define VERSION  8.36		/* Version number. */
#define MAX_FAILURES		5
#define STAY_IN_MEMORY		19000	/* SunOS swaps out after 20 sec. */
#define TIME_OUT		300000	/* 5 min. data server timeout check */
/* #define NO_TIME_OUT */
#define NO_SWAP_OUT

extern char *header;
extern int file_rc;
extern Boolean server_flag;
extern Boolean disco_flag;
extern XtIntervalId time_out;

String	ServerName = NULL;
String	Homedir = NULL;

static void internal_error(int);
#ifndef NO_SWAP_OUT
static void do_nothing(XtPointer,XtIntervalId*);
#endif

static Widget process;

static String app_defs[] = {
	"*fontList:		9x15",
	"*doubleClickInterval:	1000",
/*	"*background:		white", */
	NULL };


int main(int argc, char **argv) {
  Boolean	  flag = False, debug = False;
  long	  	  id = 1L;	/* RJM: handle 64-bit architectures */
  XVisualInfo	  info;
  String	  tmp;
  static Colormap colormap = (Colormap)NULL;
  int		  C_ARGIS();

  flag=(Boolean)C_ARGIS("-?", &argc,argv) || (Boolean)C_ARGIS("-h",&argc,argv);
  if (flag) {
    fprintf(stderr,"Usage: xsd[.new] [-3d] [-r remote_hostname] [-l loginname] \\\n");
    fprintf(stderr,"       [-f/F filename] [-v/V verbose] [-debug] [-d/D server debug] [-h/?]\n");
    exit(-1);
  }

  debug = (Boolean)C_ARGIS("-debug", &argc, argv);

  /* change argv[0] since is consumed by X for Title Bar */
  /* rm .exe for mainmenu title */
  if ((tmp = strstr(argv[0], ".exe"))) *tmp = '\0';
  if ((tmp = strrchr(argv[0], '/'))) (void)strcpy(argv[0], tmp+1);

  if ((Homedir=getenv("HOME")) == NULL) {
    fprintf(stderr, "Env. variable HOME not defined.\n");
    fprintf(stderr, "e.g.  setenv HOME /home/zuserid\n");
  }

  ServerName = getenv("SERVER_NAME");
  if (ServerName == NULL || 0 == (int)strlen(ServerName)) {
    fprintf(stderr, "Env. variable SERVER_NAME not defined.\n");
    fprintf(stderr, "e.g.  setenv SERVER_NAME ddsServer3\n");
    exit(-1);
  }

  /* ------------------------------------------------------------------ */
  /*      CAN'T SET VISUAL WITH THIS APPROACH SINCE NEED display/app_context 1st
	  top_level = XtVaAppInitialize(&app_context, "xsd", NULL, 0,
	  &argc, argv, app_defs,
	  NULL);
	  display=XtDisplay(top_level);
	  screen=XDefaultScreen(display);

	  if (XMatchVisualInfo(display,screen,8,PseudoColor,&info))
	  visual = info.visual;
	  else
	  {
	  (void)fprintf(stderr,
	  "xsd must be run on an 8-bit pseudo color device.\n");
	  exit(1);
	  }

	  */
  /* ------------------------------------------------------------------ */
  XtToolkitInitialize();
  app_context = XtCreateApplicationContext();
  XtAppSetFallbackResources(app_context, app_defs);

  /* XtOpenDisplay consumes argv! */
  savedArgc = argc;
  savedArgv = (char**)XtMalloc(argc * sizeof(char*));
  (void)memcpy(savedArgv, argv, argc * sizeof(char*));

  if ((display = XtOpenDisplay(app_context, (String)NULL,
			       ApplicationName, ApplicationClass,
			       (XrmOptionDescRec *)NULL, (Cardinal)0,
			       &argc, argv)) == NULL) {
    XtAppError(app_context, "xsd: cannot open display.");
    exit(1);
  }

  screen=XDefaultScreen(display);

  if (XMatchVisualInfo(display,screen,8,PseudoColor,&info)) {
    visual = info.visual;
    screenDepth = info.depth;
  } else {
    (void)fprintf(stderr, "xsd must be run on an 8-bit pseudo color device.\n");
    exit(1);
  }

  if (visual == DefaultVisual(display, screen)) {
    colormap = DefaultColormap(display, screen);
  } else {
    colormap = XCreateColormap(display, 
			       RootWindow(display, screen), visual, AllocNone);
  }

  {
    Arg	args[10];
    int	nargs = 0;
    XtSetArg(args[nargs], XmNargc, savedArgc); ++nargs;
    XtSetArg(args[nargs], XmNargv, savedArgv); ++nargs;
    XtSetArg(args[nargs], XmNvisual, visual); ++nargs;
    XtSetArg(args[nargs], XmNdepth, screenDepth); ++nargs;
    XtSetArg(args[nargs], XmNcolormap, colormap); ++nargs;
    if ((top_level = XtAppCreateShell(ApplicationName, ApplicationClass,	
				      applicationShellWidgetClass, display,
				      args, nargs)) == NULL) {
      XtAppError(app_context, "xsd: cannot create AppShell.");
      exit(1);
    }
  }

  /* ------------------------------------------------------------------ */

  quit_frame(top_level);

  /* DWN replace DEBUG with debug command line arg
     #ifndef DEBUG
     XSetErrorHandler(Xerrorhandler);
     #endif
     */

#ifdef DEBUG
  XSynchronize(display,True); /* DWN can you do this from command line? */
#endif

  if (debug==False) XSetErrorHandler(Xerrorhandler);

  (void)strcpy(hostname, tmp = C_ARGSTR("-r", NULL, "", "", &argc, argv));
  free(tmp);
  (void)strcpy(username, tmp = C_ARGSTR("-l", NULL, "", "", &argc, argv));
  free(tmp);
  (void)strcpy(xdisplay[id].filename, tmp = C_ARGSTR("-f", NULL, "", "", &argc, argv));
  free(tmp);
  if (strlen(xdisplay[id].filename) == 0) {
    (void)strcpy(xdisplay[id].filename, tmp = C_ARGSTR("-F", NULL, "", "", &argc, argv));
    free(tmp);
  }

  verbose = (Boolean)C_ARGIS("-v", &argc, argv);
  if (!verbose) verbose = (Boolean)C_ARGIS("-V", &argc, argv);
  if (verbose) {
    fprintf(stderr,"\n");
    fprintf(stderr,"Version %1.2f\n",VERSION);
  }

  server_debug = (Boolean)C_ARGIS("-d", &argc, argv);
  if (!server_debug) server_debug = (Boolean)C_ARGIS("-D", &argc, argv);

  do_view3d = (Boolean)C_ARGIS("-3d", &argc, argv);
  if (verbose==True) {
    if (do_view3d==True) {
      fprintf(stderr,"\n");
      fprintf(stderr,"3D viewer will be included\n");
    } else {
      fprintf(stderr,"\n");
      fprintf(stderr,"3D viewer will be excluded\n");
    }
  }

  xsd_pixmap=XCreateBitmapFromData(display,
				   RootWindowOfScreen(XtScreen(top_level)),
				   (char *)xsd_bits,xsd_width,xsd_height); 
  XtVaSetValues(top_level, XtNiconPixmap, xsd_pixmap,
		NULL);

  main_form=XmCreateFormVa(top_level,"form", NULL);

  white=WhitePixel(display,screen);	/* RJM: bad assumption! */
  black=BlackPixel(display,screen);

  xdisplay[0].mode=1;	/* Indicate general mode. */

  app_menus(main_form,0L,False);

  load_color_global(0L);

  init_defaults();                 

#ifndef NO_SWAP_OUT
  time_out=XtAppAddTimeOut(app_context,STAY_IN_MEMORY,do_nothing,0);  
  /* keeps xsd from swapping out on disks. */
#endif

  XtRealizeWidget(top_level);
				/* Check if filename was on the command
				   line. If so load the line header. */
  if (strlen(xdisplay[id].filename)!=0) load_command();

				/* Signal handlers for crashs. Learned this from
				   Bob Mars ... pretty neat.  */
  /* DWN replace DEBUG with debug command line arg
     #ifndef DEBUG
     signal(SIGSEGV, internal_error);
     signal(SIGBUS, internal_error);
     signal(SIGFPE, internal_error);
     signal(SIGILL, internal_error);
     #endif
     */

  if (debug==False) {
    signal(SIGSEGV, internal_error);
    signal(SIGBUS, internal_error);
    signal(SIGFPE, internal_error);
    signal(SIGILL, internal_error);
  } else {
    fprintf(stderr,"Internal exception handling is disabled.\n");
  }
  XtAppMainLoop(app_context);

  (void)fprintf(stderr, "xsd abandoned XtAppMainLoop!!!\n");

  exit(2);
}


/* 
  This function load line header for a seismic 
  file (which was on the command line. 
*/
void load_command(void)
{
  char machine_type[1];
  long id;	/* RJM: handle 64-bit architectures */

  id=next_image();
  show_watchCB(top_level,(XtPointer)top_level,NULL);
#ifdef CRAY
  sprintf(machine_type,"1");
#else
  sprintf(machine_type,"0");
#endif

#ifndef COMPRESS
  amoco_start_server(hostname,username,ServerName,machine_type);
#else
  amoco_start_server(hostname,username,"cxsd_server2",machine_type);
#endif

  amoco_color_range((int)(start_color&0xffffffff),(int)(end_color&0xffffffff));

#ifndef NO_TIME_OUT
  time_out=XtAppAddTimeOut(app_context,TIME_OUT,check_server,0);
#endif

  server_flag=True;

  if (amoco_check_server(buf)==-1) {
#ifndef NO_TIME_OUT
    XtRemoveTimeOut(time_out);  /* abort error dialog pop-up */
#endif
    hide_watchCB(top_level,(XtPointer)top_level,NULL);
    error_message(buf);
    error_message("Server error (0). Try restarting server.");
    xdisplay[id].init=0;
    return;
  }

  amoco_detect_format();

  file_rc=amoco_read_line_header(&header,xdisplay[id].filename);
  if (file_rc < 0) {
    hide_watchCB(top_level,(XtPointer)top_level,NULL);
    reportDSerror(file_rc);
    if (header) free(header);
    return;
  }

  C_SAVER((int*)header,"NumRec",&nrec[id],0);
  C_SAVER((int*)header,"NumTrc",&ntrace[id],0);
  C_SAVER((int*)header,"NumSmp",&nsamp[id],0);
  srec[id]=1; erec[id]=1; irec[id]=1;
  strace[id]=1; etrace[id]=ntrace[id]; itrace[id]=1;
  ssamp[id]=1; esamp[id]=nsamp[id]; isamp[id]=1;

  data(main_form,id,True);

  hide_watchCB(top_level,(XtPointer)top_level,NULL);

  if (header) free(header);
}


void messages(void)
{
  Widget form;
  Widget button;
  FILE *in,*out;
  int length,message_number,current_message;
  char buf_text[4000];
  char file[1024];
  char *error, *s;

  form=XmCreateFormDialogVa(top_level,"form",
			    XmNmarginHeight,10,
			    XmNmarginWidth,10,
			    XmNdeleteResponse,XmDO_NOTHING,
			    XmNdialogTitle,XMstr("Message Window"),
			    XtNmanageChild,False,
			    NULL);
  XtAddCallback(form,XmNhelpCallback,helpCB,"help.help");

				/* Check message number in ".xsd_define". */
  strcpy(buf,Homedir);
  strcat(buf,"/.xsd_define");
  in=fopen(buf,"r");

  if (in==NULL) {
    error_message("Error reading ~/.xsd_define file.");
    return;
  }

  error=fgets(buf,1024,in);
  if (error!=NULL) error=fgets(buf,1024,in);

  if (error==NULL) {
    message_number=0;
  } else sscanf(buf,"%d",&message_number);
  fclose(in);

 				/* The enviroment variable XSD_DIR
				   contains the path to the help files. */

  /* DWN 10/19/97 change XSD_DIR to XSD_HELP_DIR */

  if ((s=getenv("XSD_HELP_DIR")) == NULL) {
    error_message("XSD_HELP_DIR not defined.");
    return;
  }

				/* Get path of the help files. */
  strcpy(file,s);
  /* DWN 10/19/97 to deal with geodevized path to help files
     strcat(file,"help/");
     */
  strcat(file,"update.help");

				/* Open help file. */
  in=fopen(file,"r");
  if (in==NULL) {
    error_message("Error opening help file.");
    fprintf(stderr, "Error opening help file '%s'\n", file);
    return;
  }

				/* Read help. Maximum of 4000 bytes. */
  length=fread(buf_text,sizeof(char),4000,in);
  buf_text[length]='\0';

  fclose(in);

  sscanf(buf_text,"Message #%d",&current_message);

  if (current_message<=message_number) return;

				/* Check message number in ".xsd_define". */
  strcpy(buf,Homedir);
  strcat(buf,"/.xsd_define");
  out=fopen(buf,"w");

  fprintf(out,"Don't erase this file.\n%d\n",current_message);

  fclose(out);
  
  button=XmCreatePushButtonVa(form,"Cancel",
			      XmNbottomAttachment,XmATTACH_FORM,
			      XmNleftAttachment,XmATTACH_FORM,
			      NULL);
  XtAddCallback(button,XmNactivateCallback,destroyCB,form);
  
  (void)XmCreateScrolledTextVa(form,"text",
			       XmNbottomAttachment,XmATTACH_WIDGET,
			       XmNbottomWidget,button,
			       XmNtopAttachment,XmATTACH_FORM,
			       XmNleftAttachment,XmATTACH_FORM,
			       XmNrightAttachment,XmATTACH_FORM,
			       XmNvalue,buf_text,
			       XmNrows,15,
			       XmNcolumns,80,
			       XmNwordWrap,True,
			       XmNeditable,False,
			       XmNeditMode,XmMULTI_LINE_EDIT,
			       XmNscrollHorizontal,False,
			       NULL);

  XtManageChild(form);
}


/*
  This function destroys old widgets.
*/
void destroyCB(Widget w, XtPointer client_data, XtPointer call_data)
{
				/* Make sure widge exists. */
  if (client_data!=NULL) XtDestroyWidget((Widget)client_data);
}


/*
  This function display the
  version number.
*/
void versionCB(Widget w, XtPointer client_data, XtPointer call_data)
{
  Widget error;

  sprintf(buf,"Version %1.2f",VERSION);
  
  error=XmCreateInformationDialogVa(top_level,"version",
				    XmNmessageString,XMstr(buf),
				    XmNdialogTitle,XMstr("On Version"),
				    NULL);
  XtUnmanageChild(XmMessageBoxGetChild(error,XmDIALOG_HELP_BUTTON));
  XtUnmanageChild(XmMessageBoxGetChild(error,XmDIALOG_CANCEL_BUTTON));

  XtAddCallback(error,XmNokCallback,destroyCB,error);
  XtAddCallback(error,XmNcancelCallback,destroyCB,error);
}


/*
  This function display the
  processing message.
*/
void start_processing(Widget w)
{
  process=XmCreateInformationDialogVa(top_level,"version",
				      XmNmessageString,
				      XMstr("Processing. Please wait"),
				      XmNdialogTitle,XMstr("Processing"),
				      NULL);
  XtAddCallback(process,XmNokCallback,destroyCB,process);
  XtAddCallback(process,XmNcancelCallback,destroyCB,process);

  XmUpdateDisplay(process);
}


/*
  This function display the
  processing message.
*/
void stop_processing(void)
{
  destroyCB(process,(XtPointer)process,NULL);
}


/*
  This function stops the data
  server before exiting.
*/
void exitCB(Widget w, XtPointer client_data, XtPointer call_data)
{
  if (server_flag) amoco_end_server();
  
  if (NULL != savedArgv) XtFree((char*)savedArgv);

  XCloseDisplay(display);
  
  fprintf(stderr, "xsd terminated.\n");
  exit(0);
}


/*
  This function prompts the user
  to verify that he/she wants to
  exit the program.
*/
void exit_verifyCB(Widget w, XtPointer client_data, XtPointer call_data)
{
  int root_x,root_y;
  Widget dialog;

  XBell(display,100);
  locate_cursor(&root_x,&root_y);

  dialog=XmCreateQuestionDialogVa(w,"Exit Verification",
				  XmNdefaultPosition,False,
				  XmNx,root_x,
				  XmNy,root_y,
				  XmNmessageString,
				  XMstr("Do you want to exit?"),
				  XmNdialogTitle,XMstr("Exit Verification"),
				  XmNokLabelString,XMstr(" Yes "),
				  XmNcancelLabelString,XMstr("No"),
				  NULL);
  XtUnmanageChild(XmMessageBoxGetChild(dialog,XmDIALOG_HELP_BUTTON));

  XtAddCallback(dialog,XmNokCallback,destroyCB,dialog);
  XtAddCallback(dialog,XmNokCallback,exitCB,0);	/* RJM: put after above destroyCB */
  XtAddCallback(dialog,XmNcancelCallback,destroyCB,dialog);
}


/*
  Function that does nothing
  but kepts xsd from swapping
  to disk.
*/
#ifndef NO_SWAP_OUT
static void do_nothing(XtPointer client_data,XtIntervalId *id)
{
  time_out=XtAppAddTimeOut(app_context,STAY_IN_MEMORY,do_nothing,0);
}
#endif


static char *signal_name(int sig)
{
  static char buffer[30];

  switch(sig) {
  case SIGSEGV:	
    strcpy(buffer, "segmentation_fault");
    break;
  case SIGBUS:	
    strcpy(buffer, "bus_error");
    break;
  case SIGFPE:	
    strcpy(buffer, "floating_point_error");
    break;
  case SIGILL:	
    strcpy(buffer, "illegal_address");
    break;
  default:
    (void)sprintf(buffer, "signal %d", sig);
  }

  return buffer;
}


static void internal_error(int sig)	/* got this idea from Bob Mars ... nice */
{
  static int	error_count=0;

  (void)fprintf(stderr, "internal_error: signal %d\n", signal);

  /* disable signal traps */
#if defined(sgi)
  /*
    Signals raised by the instruction stream, SIGILL, SIGEMT, SIGBUS, SIGSEGV
    will cause infinite loops if their handler returns, or the action is set
    to SIG_IGN.
    */
#else
  signal(SIGSEGV, SIG_IGN);
  signal(SIGBUS, SIG_IGN);
  signal(SIGFPE, SIG_IGN);
  signal(SIGILL, SIG_IGN);
#endif

  if (error_count >= MAX_FAILURES) exitCB((Widget)NULL,(XtPointer)NULL,(XtPointer)NULL);

  error_count++;
  
  sprintf(buf,"xsd has had an internal (%s) error.\nYou must exit and restart the program.\n\nYou can try performing one or more of the following\nbefore exiting xsd.\n\n      Save out pick segments.\n      Save out images\n      Save the xsd state.",signal_name(sig));
  error_message(buf);

  XmUpdateDisplay(top_level);

#if !defined(sgi)
  signal(SIGSEGV, internal_error); /* reinstantiate */
  signal(SIGBUS, internal_error);
  signal(SIGFPE, internal_error);
  signal(SIGILL, internal_error);
#endif

  XtAppMainLoop(app_context);
}


Xerrorhandler(Display *display, XErrorEvent *err)
{
  XGetErrorText(display,err->error_code,buf,1024);
  fprintf(stderr,"Error code %s\n",buf);

  return 0;
}


void quit_frame(Widget w)
{
  Atom delete_atom;

  delete_atom=XmInternAtom(XtDisplay(w),"WM_DELETE_WINDOW",False);
  if (delete_atom!=(Atom)None) {
    XmAddWMProtocols(w,&delete_atom, 1);
    XmAddWMProtocolCallback(w,delete_atom,exit_verifyCB,0);
  } else fprintf(stderr,"Couldn't intern atom\n");
}
