!...|....1....|....2....|....3....|....4....|....5....|....6....|....7....|....8
!*******************************************************************************
!
! Create a function that smooths the velocity field to remove high wave number
! features in the map from which the Vz is hung. Honor the upper surface
! but gradually smooth the contours as the distance from that surface
! increases.
!
! Basically the user chooses some depth, usr_z, in the Vz function. I make the
! point at that depth follow a smooth version of the surface the Vz is hung on.
! This causes the Vz to be stretched and squeezed as the distance between the
! hanging surface and the smooth surface changes. Vz values below usr_z simply
! shift up and down following the structure on the smoothed surface.
!
! At maximum smoothing, the smoothed surface is a plane parallel to a the
! least-squares plane through the hanging surface. Minimum smoothing creates
! a surface at depth that looks like the upper surface but has an N-point
! smoother applied.  N starts at NSMOOTH and grows by NSMOOTH for every
! irsm z values that fit into the users chosen depth. The user chooses
! how to weight these end members with argument usr_sm.
!
! This routine returns the surface at depth that the sample at depth usr_z of
! the Vz will follow.
!
!*******************************************************************************
      subroutine make_smoothing_fxn (Map_in,Map_out, nx,ny, dx,dy, x0,
     :  y0, dz, usr_sm,usr_z,irsm, lcoef,slope_x,slope_y,zbias, verbose)

      implicit none

      integer  :: nx,ny
      real     :: x0,y0, dx,dy, dz
      real     :: Map_in(nx,ny), Map_out(nx,ny)
      real     :: usr_sm,usr_z
      real     :: slope_x,slope_y,zbias
      integer  :: irsm
      logical  :: lcoef,verbose

      integer  :: ix,iy
      real     :: z_here,y_here,x_here

      integer  :: ixsm, iysm, ixradius, iyradius, idiv
      integer  :: ix1,ix2,iix, iy1,iy2,iiy
      real     :: sm_dist
      real     :: z_plane, z_smooth, z_ave
      double precision  :: sum

      integer,parameter :: NSMOOTH=5

!-------------------------------------------------------------------------------
! Done with declarations
!-------------------------------------------------------------------------------
!
! get coefficients for a least-squares plane unless user has supplied them
!
      if (.not.lcoef) then
        call get_plane_coeff ( x0,dx,nx, y0,dy,ny, Map_in, 
     :  slope_x,slope_y,zbias )
      endif

!
! Get a smooth version of Map_in. More smoothing as distance from Map_in grows
!
      sm_dist = NSMOOTH*(max(dx,dy)) * (usr_z/dz/irsm)
      ixsm = int(sm_dist/dx) + 1
      iysm = int(sm_dist/dy) + 1

      ! make sure the smoother widths are odd numbers
      if (mod(float(ixsm),2.0) == 0.0) ixsm=ixsm+1
      if (mod(float(iysm),2.0) == 0.0) iysm=iysm+1

      ! this is the # of input points before/after the output point
      ixradius = (ixsm-1) / 2
      iyradius = (iysm-1) / 2

      ! loop over input volume
      do iy = 1, ny
        iy1 = max(1,iy-iyradius)
        iy2 = min(ny,iy+iyradius)
        do ix = 1, nx
          ix1 = max(1,ix-ixradius)
          ix2 = min(nx,ix+ixradius)

          sum  = 0.0
          idiv = ((iy2-iy1)+1)*((ix2-ix1)+1)

          ! summation of neighboring points
          do iiy = iy1,iy2
            do iix = ix1,ix2
              sum = sum + Map_in(iix,iiy)
            enddo
          enddo

          Map_out(ix,iy) = sum / float(idiv)

        enddo
      enddo

!
! Now get get Map_out, the weighted average of the least-squares plane and
! the smoothed surface, shifted by usr_z.
!
      do iy = 1,ny
        y_here = y0+dy*(iy-1)
        do ix = 1,nx
          x_here = x0+dx*(ix-1)
          z_plane = x_here*slope_x + y_here*slope_y + zbias
          z_smooth = Map_out(ix,iy)
          z_ave = usr_sm*z_plane + (1.0-usr_sm)*z_smooth
          Map_out(ix,iy) = z_ave + usr_z
        enddo
      enddo

      return
      end subroutine make_smoothing_fxn
