c...|....1....|....2....|....3....|....4....|....5....|....6....|....7..
C***********************************************************************
C Copyright 2002, Allied Geophysics, Inc. All Rights Reserved          *
C***********************************************************************
C Portions of this code and/or subroutines  used by this code are      *
C protected by the following copyright(s):                             *
C***********************************************************************
C                 copyright 2001, Amoco Production Company             *
C                             All Rights Reserved                      *
C                     an affiliate of BP America Inc.                  *
C***********************************************************************
C Use of this program and source code is subject to the terms and      *
C conditions of the FreeUSP License agreement.                         *
C***********************************************************************
c -----------------  Main Routine -----------------------
c vsm3d - smooth a 3D [velocity] grid
c
c  Program Description:
c
c     vsm3d smooths a 3D grid with 1 or more iterations of a boxcar
c     averager. The user controls the size of the smoothing window on
c     each of the 3 axes, the number of iterations, and whether the
c     input is inverted prior to smoothing. vsm3d recognizes an embedded
c     mask in the data and ignores these points when constructing sums
c     for averaging.
c
c  Program Changes:
c
c    Aug 25, 2002 - fixed bug in cmdln call to enable smoothing on axis 1
c                   removed utility routine findarg - now part of usp
c    Jan 21, 2002 - added implicit none and required declarations also fixed
c                   bug in call to help subroutine where (name) was not
c                   passed but was expected in subroutine. This caused
c                   -h or -help to abort  --- PGAG
c    Jan 14, 2002 - major cleanup, made default behavior ignore mask,
c                   improved online help, expanded routine verbal() to
c                   be more inclusive, created man page and pattern file
c    Oct 30, 2001 - adopted emask as name for embedded mask
c    Dec 15, 1994 - original version
c
c get machine dependent parameters 

      implicit none

#include <f77/iounit.h>
#include <f77/lhdrsz.h>
#include <f77/sisdef.h>
#include <save_defs.h> 

c dimension standard USP variables 
      integer     itr(3*SZLNHD)
     
      integer     n1,n2,n3, obytes, JERR
      integer     luin,luout, lbytes,nbytes,lbyout
      integer     argis

      character   ntap*255, otap*255, name*5

      logical     verbos

c Program Specific - dynamic memory variables

      integer TraceSize, VolumeSize, HeaderSize, MemSize, abort
      integer errcd1, errcd2, errcd3, errcd4
      integer Trace, Headers

      real    Volume1, Volume2

      pointer (ptr_Trace,   Trace(2))
      pointer (ptr_Volume1, Volume1(2))
      pointer (ptr_Volume2, Volume2(2))
      pointer (ptr_Headers, Headers(2))

c Program Specific - static memory variables

      integer n1sm, n2sm, n3sm, niter, npt_smth
      integer hdr_index, tr_index, JJ, KK, iteration
      real    emask
      logical linvert, lmask

c Initialize variables
      data abort/0/
      data name/'VSM3D'/

c give command line help if requested

      if ( argis ( '-?' ) .gt. 0 .or.
     :     argis ( '-h' ) .gt. 0 .or. 
     :     argis ( '-help' ) .gt. 0 ) then
         call help(name)
         stop
      endif

c open printout file
#include <f77/open.h>

c get command line input parameters
      call cmdln (ntap, otap, n1sm, n2sm, n3sm, emask, linvert, lmask, 
     :     niter, name, verbos)

c open input and output files
      call getln (luin , ntap,'r', 0)
      call getln (luout, otap,'w', 1)

c read input line header and save certain parameters
      call rtape (luin,itr,lbytes)
      if(lbytes.eq.0)then
         write(LER,*)name,': no line header on input file',ntap
         write(LER,*)'FATAL'
         stop
      endif

      call saver(itr, 'NumSmp', n1, LINHED)
      call saver(itr, 'NumTrc', n2, LINHED)
      call saver(itr, 'NumRec', n3, LINHED)

c print HLH to printout file 
      call hlhprt (itr, lbytes, name, 4, LERR)

c save out hlh and line header
      call savhlh (itr, lbytes, lbyout)
      call wrtape (luout, itr, lbyout)

c verbose output of all pertinent information before processing begins
      call verbal (ntap,otap, n1,n2,n3, n1sm,n2sm,n3sm,
     :             emask, linvert,lmask,verbos)

c dynamic memory allocation:  
      TraceSize  = (n1 + ITRWRD) * SZSMPD
      VolumeSize = n3 * n2 * n1 * SZSMPD
      HeaderSize = n3 * n2 * ITRWRD * SZSMPD
      MemSize = (TraceSize+2*VolumeSize+HeaderSize)

      call galloc (ptr_Trace,   TraceSize , errcd1, abort)
      call galloc (ptr_Volume1, VolumeSize, errcd2, abort)
      call galloc (ptr_Volume2, VolumeSize, errcd3, abort)
      call galloc (ptr_Headers, HeaderSize, errcd4, abort)
    
      if (errcd1.ne.0 .or. errcd2.ne.0 .or.
     :    errcd3.ne.0 .or. errcd4.ne.0) then
        write(LERR,*)' '
        write(LERR,*)name,
     :    ': Unable to allocate workspace: ',MemSize,' bytes'
        write(LERR,*)' '
        write(LER,*)' '
        write(LER,*)name,
     :    ': Unable to allocate workspace: ',MemSize,
     :    ' bytes.  ** FATAL **'
        write(LER,*)' '
        call lbclos (luin)
        call lbclos (luout)
        stop
      else
        write(LERR,*)' '
        write(LERR,*)name,': Allocating workspace: ',MemSize,' bytes'
        write(LERR,*)' '
      endif

c initialize memory
      call vclr (Trace,   1, n1 )
      call vclr (Volume1, 1, n1*n2*n3)
      call vclr (Volume2, 1, n1*n2*n3)
      call vclr (Headers, 1, n2*n3*ITRWRD)

c set the number output bytes
      obytes = TraceSize

c BEGIN PROCESSING 

c  Load Volume into Memory
      write(LER,*)' '
      write(LER,*)name,': Loading input Volume.'

      tr_index = 1 - n1
      hdr_index = 1 - ITRWRD
      DO KK = 1, n2*n3
        nbytes = 0
        call rtape (luin, Trace, nbytes)
        if (nbytes .eq. 0) then
          write(LERR,*)name,': Premature EOF on input at:',
     :                      '  rec= ',JJ,'  trace= ',KK,' ** FATAL **'
          call lbclos (luin)
          call lbclos (luout)
          stop
        endif
        tr_index = tr_index + n1
        hdr_index = hdr_index + ITRWRD
        call vmov (Trace(ITHWP1), 1, Volume1(tr_index), 1, n1)
        call vmov (Trace, 1, Headers(hdr_index), 1, ITRWRD)
      ENDDO

c Invert velocity to slowness if requested
      if (linvert) then
        write(LER,*)name,': Inverting values prior to smoothing.'
        call vrecip(Volume1,1,Volume1,1,n1*n2*n3)
        emask = 1./emask
      endif

c Do the smoothing
      npt_smth = min(n1sm,n1)*min(n2sm,n2)*min(n3sm,n3)
      write(LER,*)name,': 3D Smoothing underway using ',npt_smth,
     :                 ' points in ',niter,' iterations'

      do iteration = 1,niter
        if (lmask) then
          call Smooth3d(Volume1,Volume2,n1sm,n2sm,n3sm, n1,n2,n3,emask)
        else
          call SmoothNM(Volume1,Volume2,n1sm,n2sm,n3sm, n1,n2,n3)
        endif
        call vmov (Volume2, 1, Volume1, 1, n1*n2*n3)
      enddo

c Invert back to velocity as needed
      if (linvert) then
        write(LER,*)name,': Inverting values after smoothing.'
        call vrecip(Volume1,1,Volume1,1,n1*n2*n3)
      endif

c Write output data
      write(LER,*)name,': Writing smoothed volume.'

      tr_index = 1 - n1
      hdr_index = 1 - ITRWRD
      DO KK = 1, n2*n3
        tr_index = tr_index + n1
        hdr_index = hdr_index + ITRWRD
        call vmov (Volume1(tr_index), 1, Trace(ITHWP1), 1, n1)
        call vmov (Headers(hdr_index), 1, Trace(1), 1, ITRWRD)
        call wrtape (luout, Trace, obytes)
      ENDDO


c close data files 
      call lbclos (luin)
      call lbclos (luout)
      write(LERR,*)name,': Normal Termination'
      write(LER,*)name,': Normal Termination'
      stop

      end


c -----------------  Subroutine -----------------------
c provide terse online help [detailed help goes in man page]
      subroutine help(name)

#include <f77/iounit.h>
      character name*(*)

      write(LER,*)' '
      write(LER,*)'===================================================='
      write(LER,*)' '
      write(LER,*)' Command Line Arguments for ',name
      write(LER,*)'            Smooth a 3D volume'
      write(LER,*)' '
      write(LER,*)'Input......................................(default)'
      write(LER,*)' '
      write(LER,*)'-N[]      -- input data set                 (stdin:)'
      write(LER,*)'-O[]      -- output data set               (stdout:)'
      write(LER,*)' '
      write(LER,*)'-1sm[1sm] -- Axis 1 smoother length (grid pts)  ( 7)'
      write(LER,*)'-2sm[2sm] -- Axis 2 smoother length (grid pts)  (21)'
      write(LER,*)'-3sm[3sm] -- Axis 3 smoother length (grid pts)  (21)'
      write(LER,*)' '
      write(LER,*)'-I        -- Invert input before smoothing (.FALSE.)'
      write(LER,*)' '
      write(LER,*)'-ns[iter] -- Number of smoothing iterations     ( 1)'
      write(LER,*)' '
      write(LER,*)'-emask[emask] -- Value of embedded mask       (none)'
      write(LER,*)'             using -emask[] on the command'
      write(LER,*)'             line activates the embedded mask'
      write(LER,*)' '
      write(LER,*)'-V        -- verbos printout               (.FALSE.)'
      write(LER,*)' '
      write(LER,*)'Usage:'
      write(LER,*)'       vsm3d -N[] -O[] -1sm[] -2sm[] -3sm[] '
      write(LER,*)'             -ns[] -emask[] -[I V]'
      write(LER,*)' '
      write(LER,*)'===================================================='
      
      return
      end


c -----------------  Subroutine -----------------------
c pick up command line arguments 
      subroutine cmdln ( ntap, otap, n1sm, n2sm, n3sm, emask,
     :                   linvert, lmask, niter, name, verbos )

      implicit none

#include <f77/iounit.h>

      integer    argis, findarg, n1sm,n2sm,n3sm, niter
      real       emask
      character  ntap*(*), otap*(*), name*(*)
      logical    linvert, lmask, verbos

      call argstr ('-N', ntap, ' ', ' ') 
      call argstr ('-O', otap, ' ', ' ') 

      call argi4 ('-1sm', n1sm,  7,  7)
      call argi4 ('-2sm', n2sm, 21, 21)
      call argi4 ('-3sm', n3sm, 21, 21)
      call argi4 ('-ns' , niter, 1,  1)

      lmask = .false.
      if ( findarg('-emask') .gt. 0) then
         lmask = .true.
         call argr4 ('-emask',emask, -1.0e+37, -1.0e+37)
      endif

      verbos = .false.
      if (argis('-V') .gt. 0) verbos = .true.

      linvert = .false.
      if (argis('-I') .gt. 0) linvert = .true.

c check for extraneous arguments and abort if found to
c catch all manner of user typo's

      call xtrarg (name, ler, .FALSE., .FALSE.)
      call xtrarg (name, lerr, .FALSE., .TRUE.)

      if (n1sm .eq. 0) n1sm = 1
      if (n2sm .eq. 0) n2sm = 1
      if (n3sm .eq. 0) n3sm = 1
           
      return
      end


c -----------------  Subroutine -----------------------
c verbal printout of pertinent program particulars
      subroutine verbal (ntap,otap, n1,n2,n3, n1sm,n2sm,n3sm,
     :     emask, linvert,lmask,verbos)
      
      implicit none

#include <f77/iounit.h>

      integer    n1,n2,n3, n1sm,n2sm,n3sm
      real       emask
      character  ntap*(*), otap*(*)
      logical    verbos,linvert,lmask
      integer    nblen

      write(LERR,*)' '
      write(LERR,*)' Input Line Header Parameters'
      write(LERR,*)' '
      write(LERR,*)'   input dataset name    =  ', ntap(1:nblen(ntap))
      write(LERR,*)'   # samples on axis 1   =  ', n1
      write(LERR,*)'   # samples on axis 2   =  ', n2
      write(LERR,*)'   # samples on axis 3   =  ', n3
      write(LERR,*)' '
      write(LERR,*)' '
      write(LERR,*)' Command Line Parameters '
      write(LERR,*)' '
      write(LERR,*)'   output dataset name   =  ', otap(1:nblen(otap))
      write(LERR,*)' '
      write(LERR,*)'   Smoother lengths'
      write(LERR,*)'          axis 1   =  ',n1sm
      write(LERR,*)'          axis 2   =  ',n2sm
      write(LERR,*)'          axis 3   =  ',n3sm
      write(LERR,*)' '
      write(LERR,*)'   Invert prior to smoothing - ',linvert
      if (lmask) then
        write(LERR,*)' '
        write(LERR,*)'   The embedded mask will be honored'
        write(LERR,*)'     mask value  = ',emask
      endif
      if (verbos) then
        write(LERR,*)' '
        write(LERR,*)'   Verbose printout requested'
      endif
      write(LERR,*)' '
      write(LERR,*)'========================================== '
      write(LERR,*)' '

      return
      end
