c ----- ----- ----- -----  Routine fbpicker ----- ----- ----- ----- ---|
c
c     copyright 2004, Amoco Production Company 
c              All rights reserved
c        an affiliate of BP America Inc.
c
c -----------------  ------------ ----------------------- -------------|


c     Program Changes:
c
c      May 10, 2004: Added -sm[] parameter to do median smoothing of 
c                    first break picks.  To accomplish this I had to
c                    switch this process from trace consistent to 
c                    record consistent.
c      Garossino
c
c      May 6, 2004: added -AR -arwin[] -ms[] -ms[] -icode 
c                   -rscode in support of auto regressive algorithm
c                   sensing technique addition to the code.
c      Garossino
c
c      May 4, 2004: - added -freq[] -fwin[] -fthresh[] capability and 
c                   modified   -win1[], -win2[] to work with this 
c                   running window spec decomposition version.
c      Garossino
c
c      Apr, 2004: - added -t0[] -vel[] -twin[] -hws[] capability to aid in
c                 tracking a first break when the average arrival time is
c                 roughly known.  The vel is hyperbolic assuming a flat
c                 velocity interface
c      Garossino

c      - original written: P.G.A. Garossino, March 25, 2004

c     Program Description:

c      - routine to locate and load to a user defined trace header
c        the first break time for any given trace.  I looked in the 
c        old Amoco Blue Books and found an original routine written
c        by Tucay Yasser back in the dark ages.  His algorithm went like:
c
c      compare the std devs of amplitudes in two sliding windows 
c      separated by a gap. when the ratio trips a threshold 
c      stop - this is the start of the FB zone
c
c      based on the time above put a window over the data such that 
c      the last sample corresponds to the first sample in the FB zone. 
c      now start adding a sample at a time comparing the std devs until 
c      a treshold is tripped. that's the FB
c
c      I found that algorithm to be heavily dependant on a constant S/N
c      level in the data.  As the S/N changed, so did the ratio.  I found
c      it very difficult to pick a threshold that worked all the time.
c
c      I modified the algorithm to do the ratio calculation based on two
c      windows separated by a single sample.  This calculation is continued
c      down to a max time specified by the user.  The default is to use
c      the whole trace.  The position of the minimum ratio is then used
c      as the first break index.  This approach seems to be pretty 
c      robust.  At least on the data I have pushed through the picker
c      so far.  This does include data with quite a variation in S/N due
c      to swell noise etc.  If it breaks down the road we can react
c      at that time and tweak the algorithm as needed
c
c      Of course as soon as another variety of data was encountered it
c      broke.  Subsea data, where the receiver is on the seafloor, and
c      the water depths are quite rugose, blew this algorithm out of the 
c      water.  Picks would be made on the near offsets correctly, but then
c      jump to the multiple at the further offsets.  I installed the 
c      spectral decomposition [SD] option where the user can pick a frequency
c      usually the dominant frequency of the first breaks, and set a threshold
c      to identify the onset of data.  I also added the auto regressive option
c      where the first AR coefficient from a sliding window analysis provides
c      a series that can also be thresholded to provide a pick.  I also 
c      added median smoothing spatially of the raw picks and optionally
c      write out both the raw and smoothed picks to the output header.
c
c      For a complete description see the man page.
c

      implicit none

c get machine dependent parameters 

#include <f77/iounit.h>
#include <f77/lhdrsz.h>
#include <f77/sisdef.h>
#include <save_defs.h> 

c dimension standard USP variables 

      integer     nsamp, nsi, ntrc, ntrco, nrec, nreco, iform
      integer     luin , luout, lbytes, nbytes, lbyout, obytes
      integer     ist, iend, irs, ire, ns, ne, argis, jerr
      integer     JJ, KK

      real        UnitSc, dt

      character   ntap*512, otap*512, name*8

      logical     verbos

c Program Specific _ dynamic memory variables

      integer TraceSize, Tsize, TableSize, alloc_size, SumSize, worksize
      integer errcd, errcd1, errcd2, errcd3, errcd4, errcd5
      integer errcd6, errcd7, errcd8, abort
      integer itr, Headers, HeaderSize, RecordSize

c med smoothing arrays

      real tri, ctable, t, tri_buffer, sum, Record
      real v, vc, s, sc, a, r_RawWrd, r_SmoWrd, work

c med smoothing arrays

      integer key, ley1, ley2

      real xx, tmp1, tmp2

      pointer ( mem_itr, itr(2) )
      pointer ( mem_tri, tri(2) )
      pointer ( mem_ctable, ctable(2) )
      pointer ( mem_t, t(2) )
      pointer ( mem_tri_buffer, tri_buffer(2) )
      pointer ( mem_sum, sum(2) )
      pointer ( mem_v, v(2) )
      pointer ( mem_vc, vc(2) )
      pointer ( mem_s, s(2) )
      pointer ( mem_sc, sc(2) )
      pointer ( mem_a, a(2) )
      pointer ( mem_Headers, Headers(2) )
      pointer ( mem_Record, Record(2) )
      pointer ( mem_r_RawWrd, r_RawWrd(2) )
      pointer ( mem_r_SmoWrd, r_SmoWrd(2) )
      pointer ( mem_work, work(2) )
      pointer ( mem_xx, xx(2) )
      pointer ( mem_tmp1, tmp1(2) )
      pointer ( mem_tmp2, tmp2(2) )
      pointer ( mem_key, key(2) )
      pointer ( mem_ley1, ley1(2) )
      pointer ( mem_ley2, ley2(2) )

c Program Specific _ static memory variables

      integer ifmt_RecNum,l_RecNum,ln_RecNum, RecNum
      integer ifmt_TrcNum,l_TrcNum,ln_TrcNum, TrcNum
      integer ifmt_DstSgn,l_DstSgn,ln_DstSgn, DstSgn
      integer ifmt_TzoWrd,l_TzoWrd,ln_TzoWrd, TzoWrd
      integer ifmt_RawWrd,l_RawWrd,ln_RawWrd, RawWrd
      integer ifmt_SmoWrd,l_SmoWrd,ln_SmoWrd, SmoWrd
      integer ifmt_StaCor,l_StaCor,ln_StaCor, StaCor
      integer length, lenth
      integer win1, win2
      integer total_trace_time, total_window_time, max_iterations
      integer maxtime, tzero, twin 
      integer fwin, iwind, ihalf, nomega, itw, nskip
      integer Mstart, Mlast, mdim, icode, akcode, rscode, ocode, ARwin
      integer mix, tr_index, hdr_index, index

      real r_TzoWrd, velocity, freq, pie, radeg
      real fthresh_percent, ARthresh_percent
      real m, ssq, p, dot

      character c_RawWrd*6, c_TzoWrd*6, c_SmoWrd*6

      logical opbias, gaus, trbias, normal_energy, normal_live
      logical phase, AR

c Initialize variables

      data abort/0/
      data name/"FBPICKER"/
      data radeg/57.29578/
      data phase/.false./
      data opbias/.false./
      data trbias/.false./
      data normal_energy/.false./
      data normal_live/.false./
      data gaus/.true./
      data nomega/1/
      data nskip/1/
      data itw/100/
      data akcode/0/

      pie = 4.0 * atan(1.0)

c give command line help if requested

      if ( argis ( '-?' ) .gt. 0 .or. 
     :     argis ( '-h' ) .gt. 0 .or. 
     :     argis ( '-help' ) .gt. 0 ) then
         call help()
         stop
      endif

c open printout file

#include <f77/open.h>

c get command line input parameters

      call cmdln ( ntap, otap, ns, ne, irs, ire, win1, win2, 
     :     maxtime, c_RawWrd, c_TzoWrd, tzero, velocity, twin, name, 
     :     freq, fwin, fthresh_percent, AR, Mstart, Mlast, icode, 
     :     rscode, ARwin, ARthresh_percent, mix, c_SmoWrd, 
     :     verbos )

c allocate memory for line header array itr[]

      alloc_size = SZLNHD
      errcd = 0
      call galloc(mem_itr,alloc_size * SZSMPD,errcd,abort)
      if (errcd .ne. 0) then
	write(LERR,*) 'ERROR: Unable to allocate workspace '
	write(LERR,*) '       ',alloc_size * SZSMPD,' bytes requested '
	write(LERR,*) '       FATAL'
	write(LER,*) 'FBPICKER: Unable to allocate workspace '
	write(LER,*) '     ',alloc_size * SZSMPD,' bytes requested '
	write(LER,*) 'FATAL'
	stop
      else
        write(LERR,*)'Allocating workspace:'
	write(LERR,*) alloc_size * SZSMPD,' bytes requested '
      endif

      call vclr ( itr, 1, alloc_size )

c open input and output files

      call getln(luin , ntap,'r', 0)
      call getln(luout, otap,'w', 1)

c  read input line header and save certain parameters

      call rtape(luin,itr,lbytes)
      if(lbytes.eq.0)then
         length = lenth(ntap)
         write(LERR,*)'FBPICKER: no line header on input dataset',
     :        ntap(1:length)
         write(LER,*)' '
         write(LER,*)'FBPICKER: '
         write(LER,*)' no line header on input dataset',ntap(1:length)
         write(LER,*)'FATAL'
         write(LER,*)' '
         stop
      endif

      call saver(itr, 'NumSmp', nsamp, LINHED)
      call saver(itr, 'SmpInt', nsi  , LINHED)
      call saver(itr, 'NumTrc', ntrc , LINHED)
      call saver(itr, 'NumRec', nrec , LINHED)
      call saver(itr, 'Format', iform, LINHED)
      call saver(itr, 'UnitSc', UnitSc, LINHED)

c define pointers to header words required by your routine

      call savelu('RecNum',ifmt_RecNum,l_RecNum,ln_RecNum,TRACEHEADER)
      call savelu('TrcNum',ifmt_TrcNum,l_TrcNum,ln_TrcNum,TRACEHEADER)
      call savelu('DstSgn',ifmt_DstSgn,l_DstSgn,ln_DstSgn,TRACEHEADER)
      call savelu('StaCor',ifmt_StaCor,l_StaCor,ln_StaCor,TRACEHEADER)
      call savelu(c_RawWrd,ifmt_RawWrd,l_RawWrd,ln_RawWrd,TRACEHEADER)
      call savelu(c_SmoWrd,ifmt_SmoWrd,l_SmoWrd,ln_SmoWrd,TRACEHEADER)
      if ( c_TzoWrd .ne. ' ' )
     :     call savelu(c_TzoWrd,ifmt_TzoWrd,l_TzoWrd,ln_TzoWrd,
     :     TRACEHEADER)

c historical line header and print to printout file 

      call hlhprt (itr, lbytes, name, 8, LERR)

c POLICEMAN: check header for units scaling.  Using UnitSc, remember
c that UnitSc default is milliseconds [i.e. 0.001] and UnitSc
c is a floating point variable.  A UnitSc entry of 1.0 would
c mean units are in seconds.  A UnitSc entry of 0 indicates that
c the unit was not defined.  In this case milliseconds are 
c assumed and loaded to the header for further processing.

      if ( UnitSc .eq. 0.0 ) then
         write(LERR,*)'********************************************'
         write(LERR,*)'WARNING: sample unit scaler in LH = ',UnitSc
         write(LERR,*)'         will set to .001 (millisec default)'
         write(LERR,*)'********************************************'
         UnitSc = 0.001
         call savew ( itr, 'UnitSc', UnitSc, LINHED)
      endif

c compute delta T in seconds

      dt = real (nsi) * UnitSc

c check user supplied boundary conditions and set defaults

      if ( irs .eq. 0 ) irs = 1
      if ( ire .eq. 0 .or. ire .gt. nrec ) ire = nrec

      if ( ns .eq. 0 ) ns = 1
      if ( ne .eq. 0 .or. ne .gt. ntrc ) ne = ntrc

c this logic assumes parameterization in units of the dataset

      ist = 1
      iend = nsamp

      nreco = ire - irs + 1
      ntrco = ne - ns + 1

      if ( freq .gt. 0. ) then
         
c frequency rwspec window length [converted to samples] = iwind

         iwind = int ( float(fwin) / float(nsi) )
         ihalf = iwind / 2

      endif

c modify line header to reflect actual record configuration output
c NOTE: in this case the sample limits ist and iend are used to 
c       limit processing only.   All data samples are actually passed.

      call savew(itr, 'NumRec', nreco, LINHED)
      call savew(itr, 'NumTrc', ntrco  , LINHED)

c number output bytes

      obytes = SZTRHD + SZSMPD * nsamp 

c save out hlh and line header

      call savhlh  ( itr, lbytes, lbyout )
      call wrtape ( luout, itr, lbyout )

c determine if requested detection window sizes are compatible with the 
c data

      total_trace_time = (nsamp -1) * nsi
      
      total_window_time = nint (float(win1)/float(nsi)) + 
     :    nint (float(win2)/float(nsi)) + nsi

      if ( total_window_time .ge. total_trace_time ) then
         write(LERR,*)' '
         write(LERR,*)' Your -win1[]  and -win2[] entries span more '
         write(LERR,*)' data than you have available.  You will need '
         write(LERR,*)' one or the other or both and try again.'
         write(LERR,*)'FATAL '
         write(LERR,*)' '
         write(LER,*)' '
         write(LER,*)'FBPICKER: '
         write(LER,*)' Your -win1[]  and -win2[] entries span more '
         write(LER,*)' data than you have available.  You will need '
         write(LER,*)' one or the other or both and try again.'
         write(LER,*)'FATAL '
         write(LER,*)' '
         goto 999

      endif

c determine the maximum number of iterations possible for the sliding
c window action on the first break detection.

      max_iterations = float(total_trace_time - total_window_time)
     :     / float(nsi)

      if ( maxtime .lt. 0 ) then
         maxtime = nint(float(nsamp-1)  )
      else
         maxtime = nint ( float ( maxtime ) / float(nsi) ) + 1
      endif

      if ( maxtime .gt. max_iterations ) maxtime = max_iterations

c verbose output of all pertinent information before processing begins

      call verbal( ntap, otap, nsamp, nsi, ntrc, nrec, iform, ist, 
     :     iend, irs, ire, ns, ne, win1, win2, maxtime, tzero,
     :     velocity, twin, c_RawWrd, c_TzoWrd, freq, fwin, 
     :     fthresh_percent, AR, ARwin, Mstart, Mlast, icode, 
     :     rscode, ARthresh_percent, mix, c_SmoWrd, verbos )

c reallocate buffer memory to be big enough to include the trace header 
c and trace time series

      alloc_size = SZTRHD + SZSMPD * nsamp
      errcd = 0
      call grealloc(mem_itr,alloc_size,errcd,abort)
      if (errcd .ne. 0) then
	write(LERR,*) 'ERROR: Unable to allocate trace workspace '
	write(LERR,*) '       ',alloc_size,' bytes requested '
	write(LERR,*) '       FATAL'
	write(LER,*) 'FBPICKER: Unable to allocate trace workspace '
	write(LER,*) '       ',alloc_size,' bytes requested '
	write(LER,*) 'FATAL'
	stop
      else
        write(LERR,*)'Allocating workspace:'
	write(LERR,*) alloc_size,' bytes requested '
      endif

c dynamic memory allocation for trace and workspace. 

      TraceSize = nsamp 
      Tsize = iwind + 2
      TableSize = 2 * Tsize * nomega
      SumSize = nsamp * nomega
      HeaderSize = ntrco * ITRWRD
      RecordSize = ntrco * nsamp

      call galloc ( mem_t, Tsize * SZSMPD, errcd, abort )
      call galloc ( mem_tri, TraceSize * SZSMPD, errcd1, abort)
      call galloc ( mem_ctable, TableSize * SZSMPD, errcd2, abort )
      call galloc ( mem_tri_buffer, Tsize * SZSMPD, errcd3, abort )
      call galloc ( mem_sum, SumSize * SZSMPD, errcd4, abort )
      call galloc ( mem_Headers, HeaderSize * SZSMPD, errcd5, abort )
      call galloc ( mem_Record, RecordSize * SZSMPD, errcd6, abort )
    
      if ( errcd .ne. 0 .or. 
     :     errcd1 .ne. 0 .or. 
     :     errcd2 .ne. 0 .or. 
     :     errcd3 .ne. 0 .or. 
     :     errcd4 .ne. 0 .or. 
     :     errcd5 .ne. 0 .or. 
     :     errcd6 .ne. 0 )then
         write(LERR,*)' '
         write(LERR,*)'Unable to allocate workspace:'
         write(LERR,*) HeaderSize * SZSMPD, '  bytes'
         write(LERR,*) RecordSize * SZSMPD, '  bytes'
         write(LERR,*) TraceSize * SZSMPD, '  bytes'
         write(LERR,*) Tsize * 2 * SZSMPD, '  bytes'
         write(LERR,*) TableSize * SZSMPD, '  bytes'
         write(LERR,*) SumSize * SZSMPD, '  bytes'
         write(LERR,*)' '
         write(LER,*)' '
         write(LER,*)'FBPICKER: '
         write(LER,*)'Unable to allocate workspace:'
         write(LER,*) HeaderSize * SZSMPD, '  bytes'
         write(LER,*) RecordSize * SZSMPD, '  bytes'
         write(LER,*) TraceSize * SZSMPD, '  bytes'
         write(LER,*) Tsize * 2 * SZSMPD, '  bytes'
         write(LER,*) TableSize * SZSMPD, '  bytes'
         write(LER,*) SumSize * SZSMPD, '  bytes'
         write(LER,*)' '
         go to 999
      else
         write(LERR,*)' '
         write(LERR,*)'Allocating workspace:'
         write(LERR,*) HeaderSize * SZSMPD, '  bytes'
         write(LERR,*) RecordSize * SZSMPD, '  bytes'
         write(LERR,*) TraceSize * SZSMPD, '  bytes'
         write(LERR,*) Tsize * 2 * SZSMPD, '  bytes'
         write(LERR,*) TableSize * SZSMPD, '  bytes'
         write(LERR,*) SumSize * SZSMPD, '  bytes'
         write(LERR,*)' '
      endif

c initialize memory
         
      call vclr ( itr, 1, ITRWRD + nsamp )
      call vclr ( tri, 1, TraceSize )
      call vclr ( t, 1, TSize )
      call vclr ( tri_buffer, 1, TSize )
      call vclr ( ctable, 1, TableSize )
      call vclr ( Headers, 1, HeaderSize )
      call vclr ( Record, 1, RecordSize )

      if (AR) then

         if ( mlast .lt. 0 ) mlast = nsamp / 4
         mdim = ( Mlast * (Mlast + 1) ) / 2

c allocate AR arrays

         call galloc ( mem_v, mdim * SZSMPD, errcd, abort )
         call galloc ( mem_vc, mdim * SZSMPD, errcd1, abort )
         call galloc ( mem_s, mlast * SZSMPD, errcd2, abort )
         call galloc ( mem_sc, mlast * SZSMPD, errcd3, abort )
         call galloc ( mem_a, SZLNHD * SZSMPD, errcd4, abort )

         if ( errcd .ne. 0 .or. 
     :        errcd1 .ne. 0 .or. 
     :        errcd2 .ne. 0 .or. 
     :        errcd3 .ne. 0 .or. 
     :        errcd4 .ne. 0 )then
            write(LERR,*)' '
            write(LERR,*)'Unable to allocate workspace:'
            write(LERR,*) 2 * mdim * SZSMPD, '  bytes'
            write(LERR,*) 2 * mlast * SZSMPD, '  bytes'
            write(LERR,*) SZLNHD * SZSMPD, '  bytes'
            write(LERR,*)' '
            write(LER,*)' '
            write(LER,*)'FBPICKER: '
            write(LER,*)'Unable to allocate workspace:'
            write(LER,*) 2 * mdim * SZSMPD, '  bytes'
            write(LER,*) 2 * mlast * SZSMPD, '  bytes'
            write(LER,*) SZLNHD * SZSMPD, '  bytes'
            write(LER,*)' '
            go to 999
         else
            write(LERR,*)' '
            write(LERR,*)'Allocating workspace:'
            write(LERR,*) 2 * mdim * SZSMPD, '  bytes'
            write(LERR,*) 2 * mlast * SZSMPD, '  bytes'
            write(LERR,*) SZLNHD * SZSMPD, '  bytes'
            write(LERR,*)' '
         endif

c initialize memory
         
         call vclr ( v, 1, mdim )
         call vclr ( vc, 1, mdim )
         call vclr ( s, 1, mlast )
         call vclr ( sc, 1, mlast )
         call vclr ( a, 1, SZLNHD )

      endif

c allocate memory required for storing picks and  smoothing

      worksize = 2 * mix + ntrco + 1

      call galloc ( mem_r_RawWrd, worksize * SZSMPD, errcd, abort )

      if ( errcd .ne. 0 ) then
         write(LERR,*)' '
         write(LERR,*)'Unable to allocate workspace:'
         write(LERR,*) worksize * SZSMPD, '  bytes'
         write(LER,*)'FBPICKER: '
         write(LER,*)'Unable to allocate workspace:'
         write(LER,*) worksize * SZSMPD, '  bytes'
         write(LER,*)' '
         go to 999
      else
         write(LERR,*)' '
         write(LERR,*)'Allocating workspace:'
         write(LERR,*) worksize * SZSMPD, '  bytes'
         write(LERR,*)' '
      endif

      call vclr ( r_RawWrd, 1, worksize )

      if ( mix .gt. 0 ) then

         if ( mix .gt. ntrco ) then

            write(LERR,*)' '
            write(LERR,*)' You have asked for a ',mix,' trace'
            write(LERR,*)' median smoother but only have ',ntrco
            write(LERR,*)' traces available in any record to smooth'
            write(LERR,*)' Decrease -sm[] and try again.'
            write(LERR,*)'FATAL'
            write(LERR,*)' '
            write(LER,*)' '
            write(LER,*)'FBPICKER: '
            write(LER,*)' You have asked for a ',mix,' trace'
            write(LER,*)' median smoother but only have ',ntrco
            write(LER,*)' traces available in any record to smooth'
            write(LER,*)' Decrease -sm[] and try again.'
            write(LER,*)'FATAL'
            write(LER,*)' '
            goto 999
            
         endif

         call galloc ( mem_r_SmoWrd, worksize * SZSMPD, errcd1, abort )
         call galloc ( mem_work, worksize * SZSMPD, errcd2, abort )
         call galloc ( mem_xx, worksize * SZSMPD, errcd3, abort )
         call galloc ( mem_tmp1, worksize * SZSMPD, errcd4, abort )
         call galloc ( mem_tmp2, worksize * SZSMPD, errcd5, abort )
         call galloc ( mem_key, worksize * SZSMPD, errcd6, abort )
         call galloc ( mem_ley1, worksize * SZSMPD, errcd7, abort )
         call galloc ( mem_ley2, worksize * SZSMPD, errcd8, abort )

         if ( errcd1 .ne. 0 .or. 
     :        errcd2 .ne. 0 .or. 
     :        errcd3 .ne. 0 .or. 
     :        errcd4 .ne. 0 .or. 
     :        errcd5 .ne. 0 .or. 
     :        errcd6 .ne. 0 .or. 
     :        errcd7 .ne. 0 .or. 
     :        errcd8 .ne. 0 )then
            write(LERR,*)' '
            write(LERR,*)'Unable to allocate workspace:'
            write(LERR,*) 8*worksize * SZSMPD, '  bytes'
            write(LER,*)'FBPICKER: '
            write(LER,*)'Unable to allocate workspace:'
            write(LER,*) 8*worksize * SZSMPD, '  bytes'
            write(LER,*)' '
            go to 999
         else
            write(LERR,*)' '
            write(LERR,*)'Allocating workspace:'
            write(LERR,*) 8*worksize * SZSMPD, '  bytes'
            write(LERR,*)' '
         endif

c initialize memory
 
         call vclr ( r_SmoWrd, 1, worksize )
         call vclr ( work, 1, worksize )
         call vclr ( xx, 1, worksize )
         call vclr ( tmp1, 1, worksize )
         call vclr ( tmp2, 1, worksize )
         call vclr ( key, 1, worksize )
         call vclr ( ley1, 1, worksize )
         call vclr ( ley2, 1, worksize )

      endif

c PRECALCULATE Frequency decomposition TABLES if required

      if ( freq .gt. 0. ) then

         call Tables ( ctable, iwind, nomega, opbias, gaus, t,  
     :        itw, ihalf, pie, freq, dt )

      endif

c Read Record and Headers into memory

c skip unwanted input records

      call recskp ( 1, irs-1, luin, ntrc, itr )

      DO JJ = irs, ire
 
c skip to start trace

         call trcskp ( JJ, 1, ns-1, luin, ntrc, itr )

         tr_index = 1 - nsamp
         hdr_index = 1 - ITRWRD

         DO KK = ns, ne

            nbytes = 0
            call rtape( luin, itr, nbytes)

c if end of data encountered (nbytes=0) then bail out

            if(nbytes .eq. 0) then
               write(LERR,*)'Premature EOF on input at:'
               write(LERR,*)'  rec= ',JJ,'  trace= ',KK
               go to 999
            endif

c set array load points for this trace 

            tr_index = tr_index + nsamp
            hdr_index = hdr_index + ITRWRD


c load trace to array Record[]

            call vmov ( itr(ITHWP1), 1, Record(tr_index), 1, nsamp )

c load trace header to array Headers[]
            
            call vmov ( itr, 1, Headers(hdr_index), 1, ITRWRD )

         ENDDO

c BEGIN first break processing on this record
 
         tr_index = 1 - nsamp
         hdr_index = 1 - ITRWRD

         DO KK = ns, ne

c debug - leave this here as it comes in handy when examining datasets
c         that break any/all of the algorithms
c
c            if ( KK .eq. 45 .or. 
c     :           KK .eq. 318 .or. 
c     :           KK .eq. 450 .or. 
c     :           KK .eq. 670 .or. 
c     :           KK .eq. 1200 .or. 
c     :           KK .eq. 1700 .or. 
c     :           KK .eq. ntrc-50 ) then
c               write(LER,*)' made it...debug ', KK
c            endif
c debug

            tr_index = tr_index + nsamp
            hdr_index = hdr_index + ITRWRD
            index = KK - ns + 1

c get required trace header information

            call saver2( Headers(hdr_index), ifmt_RecNum, l_RecNum, 
     :           ln_RecNum, RecNum, TRACEHEADER )

            call saver2( Headers(hdr_index), ifmt_TrcNum, l_TrcNum, 
     :           ln_TrcNum, TrcNum, TRACEHEADER) 

            call saver2( Headers(hdr_index), ifmt_DstSgn, l_DstSgn, 
     :           ln_DstSgn, DstSgn, TRACEHEADER )

            call saver2( Headers(hdr_index), ifmt_StaCor, l_StaCor, 
     :           ln_StaCor, StaCor, TRACEHEADER )

            if ( c_TzoWrd .ne. ' ' ) then

c read user defined t(0) time from trace header if requested

               if ( ifmt_TzoWrd .eq. SAVE_FLOAT_DEF ) then

                  call saver2(Headers(hdr_index), ifmt_TzoWrd, l_TzoWrd, 
     :                 ln_TzoWrd, r_TzoWrd, TRACEHEADER )
                  TzoWrd = nint(r_TzoWrd)

               else

                  call saver2(Headers(hdr_index), ifmt_TzoWrd, l_TzoWrd, 
     :                 ln_TzoWrd, TzoWrd, TRACEHEADER )
                  r_TzoWrd = float(TzoWrd)

               endif

            endif

c we watch for dead traces, flagged or not.

            call saver2( Headers(hdr_index), ifmt_StaCor, l_StaCor, 
     :           ln_StaCor, StaCor, TRACEHEADER )

            call vmov ( Record(tr_index), 1, tri, 1, nsamp )
            call dotpr (tri, 1, tri, 1, dot, nsamp)

c process only live traces

            if ( StaCor .ne. 30000 .and. dot .gt. 1.e-32 ) then

               if ( AR ) then

c use AR coefficient threshold technique
c mstart = min num of coefficients [2]
c mlast = max num of coefficients [nsamp / 4]
c icode, forward = 1, backward = -1, both = 0
c akcode, mean sub from trace = 1, otherwise = 0
c rscode, fast = 0, accurate = 1
c mdim, dimension of V and VC [mdim >= (mlast*(mlast+1))/2
c ocode, optimal found = 0, finish but not optimal = 1, ill posed = 2

                  call fbanal_AR ( tri, nsamp, win1, win2, ARwin, 
     :                 ARthresh_percent, maxtime, r_RawWrd(index), 
     :                 r_TzoWrd, c_TzoWrd, max_iterations,  
     :                 nsi, tzero, velocity, DstSgn, UnitSc,
     :                 Mstart, Mlast, icode, akcode, rscode, mdim,
     :                 v, vc, s, sc, a, m, ssq, p, ocode )

               elseif ( freq .le. 0 ) then

c use Standard Deviation ratio technique

                  call fbanal ( tri, nsamp, win1, win2, maxtime, 
     :                 r_RawWrd(index), r_TzoWrd, c_TzoWrd, 
     :                 max_iterations,  
     :                 nsi, tzero, velocity, DstSgn, twin, UnitSc )

               else

c use running window spectral decomposition coefficient threshold technique

                  call Decompose ( tri, nsamp, tri_buffer, Tsize, ist, 
     :                 iend, nskip, trbias, nomega, iwind, ihalf, 
     :                 ctable, t, sum, radeg, phase, normal_live, 
     :                 normal_energy )

                  call fbanal_freq ( sum, nsamp, win2, maxtime, 
     :                 r_RawWrd(index), r_TzoWrd, c_TzoWrd, 
     :                 max_iterations,  
     :                 nsi, tzero, velocity, DstSgn, UnitSc,
     :                 fthresh_percent )

               endif

c do free format output of raw first break time to c_RawWrd

               if ( ifmt_RawWrd .eq. SAVE_FLOAT_DEF .or.
     :              ifmt_RawWrd .eq. SAVE_FKFLT_DEF ) then

                  call savew2(Headers(hdr_index), ifmt_RawWrd, l_RawWrd, 
     :                 ln_RawWrd, r_RawWrd(index), TRACEHEADER )

               else

                  RawWrd = nint( r_RawWrd(index) )
                  call savew2(Headers(hdr_index), ifmt_RawWrd, l_RawWrd, 
     :                 ln_RawWrd, RawWrd, TRACEHEADER )

               endif

            else

c we have a dead trace, put a pick at max time and install dead trace flag

               r_RawWrd(index) = float((nsamp-1)*nsi)

               if ( ifmt_RawWrd .eq. SAVE_FLOAT_DEF .or.
     :              ifmt_RawWrd .eq. SAVE_FKFLT_DEF ) then
                  
                  call savew2(Headers(hdr_index), ifmt_RawWrd, l_RawWrd, 
     :                 ln_RawWrd, r_RawWrd(index), TRACEHEADER )

               else

                  RawWrd = (nsamp-1)*nsi
                  call savew2(Headers(hdr_index), ifmt_RawWrd, l_RawWrd, 
     :                 ln_RawWrd, RawWrd, TRACEHEADER )

               endif

               StaCor = 30000
               call savew2( Headers(hdr_index), ifmt_StaCor, l_StaCor, 
     :              ln_StaCor, StaCor, TRACEHEADER )

            endif
 
         ENDDO

c smooth picks if requested and load to smoothed header mnemonic

         if ( mix .gt. 0 ) call SmoothPicks( JJ, KK, r_RawWrd, 
     :        r_SmoWrd, work, worksize, ntrco, mix,
     :        xx, tmp1, tmp2, key, ley1, ley2 )

c reset array load points for this trace 

         tr_index = 1 - nsamp
         hdr_index = 1 - ITRWRD

c write output data

         DO KK = ns, ne

            tr_index = tr_index + nsamp
            hdr_index = hdr_index + ITRWRD
            index = KK - ns + 1

            call vmov ( Record(tr_index), 1, itr(ITHWP1), 1, nsamp )

c do free format output of smoothed first break time to c_SmoWrd if 
c it was requested 

            if ( mix .gt. 0 ) then

               if ( ifmt_SmoWrd .eq. SAVE_FLOAT_DEF .or.
     :              ifmt_SmoWrd .eq. SAVE_FKFLT_DEF ) then

                  call savew2(Headers(hdr_index), ifmt_SmoWrd, l_SmoWrd, 
     :                 ln_SmoWrd, r_SmoWrd(index), TRACEHEADER )

               else

                  SmoWrd = nint( r_SmoWrd(index) )
                  call savew2(Headers(hdr_index), ifmt_SmoWrd, l_SmoWrd, 
     :                 ln_SmoWrd, SmoWrd, TRACEHEADER )
                  
               endif

            endif

            call vmov ( Record(tr_index), 1, itr(ITHWP1), 1, nsamp )
            call vmov ( Headers(hdr_index), 1, itr(1), 1, ITRWRD )

c write output trace

            call wrtape (luout, itr, obytes)
 
         ENDDO
 
c skip to end of record

         call trcskp ( JJ, ne+1, ntrc, luin, ntrc, itr )

      ENDDO

c close data files 

      call lbclos ( luin )
      call lbclos ( luout )
      write(LERR,*)'fbpicker: Normal Termination'
      write(LER,*)'fbpicker: Normal Termination'
      stop

 999  continue

      call lbclos ( luin )
      call lbclos ( luout )
      write(LERR,*)'fbpicker: ABNORMAL Termination'
      write(LER,*)'fbpicker: ABNORMAL Termination'
      stop
      end

c -----------------  Subroutine -----------------------

      subroutine help()

c provide terse online help [detailed help goes in man page]

#include <f77/iounit.h>

      write(LER,*)' '
      write(LER,*)'===================================================='
      write(LER,*)' '
      write(LER,*)' Command Line Arguments for fbpicker: USP template'
      write(LER,*)' '
      write(LER,*)' For a more detailed description of these parameters'
      write(LER,*)' see the online man page using uman or see the USP'
      write(LER,*)' intranet site '
      write(LER,*)' '
      write(LER,*)'Input......................................... (def)'
      write(LER,*)' '
      write(LER,*)'-N[]   -- input data set                     (stdin)'
      write(LER,*)'-O[]   -- output data set                   (stdout)'
      write(LER,*)'-ns[]  -- start trace number                     (1)'
      write(LER,*)'-ne[]  -- end trace number              (last trace)'
      write(LER,*)'-rs[]  -- start record                           (1)'
      write(LER,*)'-re[]  -- end record                   (last record)'
      write(LER,*)'-t0[]  -- zero offset start time [units of data] (0)'
      write(LER,*)'-vel[] -- hyperbolic start time velocity  (not used)'
      write(LER,*)'-tmax[] -- max detect time [units of data]   (nsamp)'
      write(LER,*)'-hws[] -- t0 mnemonic                     (not used)'
      write(LER,*)'-hw[]  -- fb time mnemonic                  (Horz08)'
      write(LER,*)'-sm[] -- length of pick smoothing window  (not used)'
      write(LER,*)'-hwsm[] -- smoothed fb time mnemonic        (Horz07)'
      write(LER,*)'-V     -- verbos printout' 
      write(LER,*)' '
      write(LER,*)' Option 1: Ratio of Windowed Standard Deviations'
      write(LER,*)' '
      write(LER,*)'-win1[] -- win1 length                          (20)'
      write(LER,*)'-win2[] -- win2 length                          (20)'
      write(LER,*)'-twin[] -- detection window length              (20)'
      write(LER,*)' '
      write(LER,*)' Option 2: Threshold on RW Spectral Decomposition'
      write(LER,*)' '
      write(LER,*)'-freq[] -- analysis frequency             (not used)'
      write(LER,*)'-fwin[] -- decomp window length                 (80)'
      write(LER,*)'-fthresh[] -- threshold multiplier            (30.0)'
      write(LER,*)'-win2[] -- detection window length              (20)'
      write(LER,*)' '
      write(LER,*)' Option 3: Threshold on RW AR coefficients'
      write(LER,*)' '
      write(LER,*)'-AR  -- if present do AR technique        (not used)'
      write(LER,*)'-arwin[] -- length of AR design window          (80)'
      write(LER,*)'-win1[] -- median extract window length         (20)'
      write(LER,*)'-win2[] -- first break detection window size    (20)'
      write(LER,*)'-arthresh[] -- AR threshold percent            (3.0)'
      write(LER,*)'-ms[] -- min number of coefs for AR              (2)'
      write(LER,*)'-ml[] -- max number of coefs for AR        (nsamp/4)'
      write(LER,*)'-icode[] -- forward 1, backward -1, both 0       (0)'
      write(LER,*)'-rscode[] -- fast 0, accurate 1                  (1)'
      write(LER,*)' '
      write(LER,*)'Usage:'
      write(LER,*)'       fbpicker -N[] -O[] -ns[] -ne[] -rs[] -re[] '
      write(LER,*)'                -win1[] -win2[] -hw[] -hws[] '
      write(LER,*)'                -t0[] -twin[] -tmax[] -vel[] '
      write(LER,*)'                -freq[] -fwin[] -fthresh[]'
      write(LER,*)'                -AR -ms[] -ml[] -icode[] '
      write(LER,*)'                -rscode[] -arwin[] -sm[] -hwsm[] -V'
      write(LER,*)'===================================================='
      
      return
      end

c -----------------  Subroutine -----------------------

c pick up command line arguments 

      subroutine cmdln ( ntap, otap, ns, ne, irs, ire, win1, win2, 
     :     maxtime, c_RawWrd, c_TzoWrd, tzero, velocity, twin, name, 
     :     freq, fwin, fthresh_percent, AR, Mstart, Mlast, icode, 
     :     rscode, ARwin, ARthresh_percent, mix, c_SmoWrd, 
     :     verbos )

      implicit none

#include <f77/iounit.h>

      integer    ns, ne, irs, ire, argis
      integer    win1, win2, mix
      integer    maxtime, tzero, twin, fwin
      integer    Mstart, Mlast, icode, rscode, ARwin

      real velocity, freq, fthresh_percent, ARthresh_percent

      character  ntap*(*), otap*(*), name*(*), c_RawWrd*6, c_TzoWrd*6
      character  c_SmoWrd*6

      logical AR, verbos

           call argr4 ( '-arthresh', ARthresh_percent, 3., 3. )
           call argi4 ( '-arwin', ARwin, 50, 50 )
           AR = (argis('-AR') .gt. 0)

           call argr4 ( '-fthresh', fthresh_percent, 30., 30. )
           call argr4 ( '-freq', freq, -1.0, -1.0 )
           call argi4 ( '-fwin', fwin, 80, 80 )

           call argstr ( '-hwsm', c_SmoWrd, 'Horz07', 'Horz07' ) 
           call argstr ( '-hws', c_TzoWrd, ' ', ' ' ) 
           call argstr ( '-hw', c_RawWrd, 'Horz08', 'Horz08' ) 

           call argi4 ( '-icode', icode, 0, 0 )

           call argi4 ( '-ms', Mstart, 2, 2 )
           call argi4 ( '-ml', Mlast, -1, -1 )


           call argi4 ( '-ne', ne, 0, 0 )
           call argi4 ( '-ns', ns, 0, 0 )
           call argstr ( '-N', ntap, ' ', ' ' ) 

           call argstr ( '-O', otap, ' ', ' ' ) 

           call argi4 ( '-rscode', rscode, 1, 1 )
           call argi4 ( '-re', ire, 0, 0 )
           call argi4 ( '-rs', irs, 0, 0 )

           call argi4 ( '-sm', mix, 0, 0 )

           call argi4 ( '-tmax', maxtime, -99999, -99999 )
           call argi4 ( '-twin', twin, 0, 0 )
           call argi4 ( '-t0', tzero, 0, 0 )

           call argr4 ( '-vel', velocity, -99999, -99999 )
           verbos = (argis('-V') .gt. 0)

           call argi4 ( '-win1', win1, 20, 20 )
           call argi4 ( '-win2', win2, 20, 20 )

c check for extraneous arguments and abort if found to
c catch all manner of user typo's (eat,die)

      call xtrarg ( name, ler, .FALSE., .FALSE. )
      call xtrarg ( name, lerr, .FALSE., .TRUE. )

           
      return
      end

c -----------------  Subroutine -----------------------

c verbal printout of pertinent program particulars


      subroutine verbal ( ntap, otap, nsamp, nsi, ntrc, nrec, iform, 
     :     ist, iend, irs, ire, ns, ne, win1, win2, maxtime, tzero,
     :     velocity, twin, c_RawWrd, c_TzoWrd, freq, fwin,
     :     fthresh_percent, AR, ARwin, Mstart, Mlast, icode,
     :     rscode, ARthresh_percent, mix, c_SmoWrd, verbos )

#include <f77/iounit.h>

      integer    nsamp, ntrc, iform, ist, iend, irs, ire, ns, ne, nsi
      integer    win1, win2, maxtime
      integer    ARwin, Mstart, Mlast, icode, rscode, mix

      real       ARthresh_percent

      character  ntap*(*), otap*(*), c_RawWrd*6, c_SmoWrd*6

      logical    AR, verbos

      write(LERR,*)' '
      write(LERR,*)' Input Line Header Parameters'
      write(LERR,*)' '
      write(LERR,*) ' input data set name   =  ', ntap
      write(LERR,*) ' samples per trace     =  ', nsamp
      write(LERR,*) ' traces per record     =  ', ntrc
      write(LERR,*) ' number of records     =  ', nrec
      write(LERR,*) ' data format           =  ', iform
      write(LERR,*) ' sample interval       =  ', nsi
      write(LERR,*)' '
      write(LERR,*)' '
      write(LERR,*)' Command Line Parameters '
      write(LERR,*)' '
      write(LERR,*) ' output data set name  =  ', otap
      write(LERR,*) ' start record          =  ', irs 
      write(LERR,*) ' end record            =  ', irs 
      write(LERR,*) ' start trace           =  ', ns
      write(LERR,*) ' end trace             =  ', ne
      write(LERR,*) ' processing sample start = ', ist
      write(LERR,*) ' processing sample end   = ', iend
      write(LERR,*) ' top window length     = ', win1
      write(LERR,*) ' bottom window length  = ', win2
      write(LERR,*) ' max sample to check   = ', maxtime
      write(LERR,*) ' detection frequency   = ', freq
      write(LERR,*) ' decomposition window size  = ', freq
      write(LERR,*) ' threshold percentage  = ', fthresh_percent
      write(LERR,*) ' RawWrd         = ', c_RawWrd
      write(LERR,*) ' Auto Regressive Flag = ', AR
      write(LERR,*) ' Minimum num AR coefs = ', Mstart
      write(LERR,*) ' Maximum num AR coefs = ', Mlast
      if ( icode .eq. 0 ) then
         write(LERR,*)' both AR forward - backward prediction used'
      elseif ( icode .eq. -1 ) then
         write(LERR,*)' only AR backward prediction used'
      elseif ( icode .eq. 1 ) then
         write(LERR,*)' only AR forward prediction used'
      endif
      if ( rscode .eq. 0 ) then
         write(LERR,*)'AR fast calculation- residual sum of squares'
      elseif ( icode .eq. 1 ) then
         write(LERR,*)'AR accurate calculation- residual sum of squares'
      endif
      write(LERR,*) ' Auto Regressive threshold percent = '
     :     ,ARthresh_percent
      write(LERR,*)' '
      if ( mix .gt. 0 ) then
         write(LERR,*)mix, ' trace median smoothing of picks requested'
         write(LERR,*)' SmoWrd = ',c_SmoWrd
      endif
      write(LERR,*)' '
      if ( verbos )  write(LERR,*) ' verbose printout requested'
      write(LERR,*)' '
      write(LERR,*)'========================================== '
      write(LERR,*)' '

      return
      end





