C***********************************************************************
C                 copyright 2004, Amoco Production Company             *
C                             All Rights Reserved                      *
C                     an affiliate of BP America Inc.                  *
C***********************************************************************
      subroutine dcmpwt(n,x,sd,w)
c*************************************************************************
c     This subroutine calculates a weight array according to the 
c     probability distribution function.
c
c**** Subroutine inputs.
c
c     n = The length of the array x
c     x = The array with a standard deviation, sd, for which weights are
c         required.
c     sd = The standard deviation of the values of the elements of x.
c
c**** Subroutine ouputs.
c
c     w = The array of weights corresponding to the values of the elements
c         of x.
c
c*************************************************************************

      integer n,i
      real*8 x(n),w(n),sd,t
 
      do i = 1,n

        t = dabs(x(i)) / sd

        if(t.lt.10.0d0) then

          w(i) = dexp(-t * t)

          else

          w(i) = 0.0d0

        endif

      enddo 

      return
      end

      subroutine dcrfrm(m,n,a,alpha)

c**********************************************************************
c     This subroutine forms the crout array from a given m*n array
c
c     See Robinson,"Least Squares Regression Analysis in Terms of
c     Linear Algebra", page 51.
c
c**** Subroutine inputs
c
c     m = The number of rows in the input array
c     n = The number of columns in the input array
c     a = The input array
c
c**** Subroutine outputs
c
c     alpha = m * n crout array.
c
c***********************************************************************

      integer m,n,i,j,j1,mm,k,kp,km,il,lk,ik,l,kl,lj,kj,kk,koff,joff
      real*8 a(m * n),alpha(m * n),sum

c**********************************************************************
c     fill in the first vertical block
c**********************************************************************

      do i = 1,m

        alpha(i) = a(i)

      enddo

c**********************************************************************
c     fill in the first horizontal block
c     j1 = index of the first element in the jth. row 2 <= j <=n
c**********************************************************************

      j1 = m + 1

      do j = 2,n

        alpha(j1) = a(j1) / alpha(1)
        j1 = j1 + m

      enddo

c**********************************************************************
c     fill in the rest of the crout array.
c**********************************************************************

      if(n.lt.m) then

        mm = n

        else

        mm = m

      endif

c**********************************************************************
c     koff = the base element offset of the kth row 2 <= k <= mm.
c**********************************************************************

      koff = m

      do k = 2,mm

        kp = k + 1
        km = k - 1
        kk = koff + k

c**********************************************************************
c     fill in the kth. vertical block
c**********************************************************************

        do i = k,m

          sum = 0.0d0
          il = i

          do l = 1,km

            lk = koff + l
            sum = sum + alpha(il) * alpha(lk)
            il = il + m

          enddo

          ik = koff + i
          alpha(ik) = a(ik) - sum

        enddo

c**********************************************************************
c     fill in the kth. horizontal block
c     joff = the base element offset of the jth row k + 1 <= j <= n.
c**********************************************************************

        joff = koff + m

        do j = kp,n

          sum = 0.0d0
          kl = k

          do l = 1,km

            lj = joff + l
            sum = sum + alpha(kl) * alpha(lj)
            kl = kl + m

          enddo

          kj = joff + k
          alpha(kj) = (a(kj) - sum) / alpha(kk)
          joff = joff + m

        enddo

        koff = koff + m

      enddo

      return
      end
      subroutine dcrout(m,n,a,alpha,nsys,x)

c**********************************************************************
c     This subroutine solves nsys sets of m * m simultaneous equations
c     using the Crout method.
c     The matrix A comprises the m * m matrix of right hand sides
c     and the m * nsys matrix of left hand sides. that is, a is m * n
c     where n = m + nsys. the solution matrix is the m * nsys matrix, X
c
c     See Robinson,"Least Squares Regression Analysis in Terms of
c     Linear Algebra", page 51.
c
c**** Subroutine inputs
c
c     m = The number of unknowns in the simultaneous equations, the
c         number of rows in a.
c     n = The number of columns in a, where n = m + nsys.
c     a = The coefficient matrix, (m * n)
c     nsys = Number of sets of equations to be solved
c
c**** Subroutine outputs
c
c     alpha = The crout array of a.
c     x = m * nsys matrix of solutions to the simultaneous equations
c
c**** Subroutine required
c
c     dcrfrm
c
c***********************************************************************

      integer m,n,nsys,i,k,kp,l,kl,kj,it,mit,mj,kit,lit,itoff,joff,mp1
      integer mm1,m2,koff
      real*8 a(m * n),alpha(m * n),x(m * nsys),sum

c***********************************************************************
c     form the crout array, alpha
c***********************************************************************

      call dcrfrm(m,n,a,alpha)
      mp1 = m + 1
      mm1 = m - 1
      m2 = m * m

c**********************************************************************
c     carry out the back solution of the simultaneous equations.
c     itoff = the base offset of row it where      1 <= it <= nsys
c     joff = the base offset of row j where    m + 1 <= j <= m + nsys
c     koff = the base offset of row k where        m => k >= 1
c**********************************************************************

      mit = m
      itoff = 0
      joff = m2
      mj = joff + m

      do it = 1,nsys

        x(mit) = alpha(mj)
        k = mm1
        kp = m
        koff = m2 - m

        do i = 2,m

          sum = 0.0d0
          kl = koff + k

          do l = kp,m

            lit = itoff + l
            sum = sum + alpha(kl) * x(lit)
            kl = kl + m

          enddo

          kit = itoff + k
          kj = joff + k
          x(kit) = alpha(kj) - sum
          k = k - 1
          kp = kp - 1
          koff = koff - m

        enddo

        itoff = itoff + m
        joff = joff + m
        mit = mit + m
        mj = mj + m

      enddo

      return
      end
      subroutine dmdian(n,x,xmdian)

c***********************************************************************
c     This subroutine determines the median value of an array after
c     sorting into increasing order.
c
c**** Subroutine inputs.
c
c     n = The number of points in the input data array.
c     x = The input array of data values under consideration.
c
c**** Subroutine outputs.
c
c     x = The input array sorted into increasing order.
c     xmdian = The median value of the array x.
c
c**** Subroutines required.
c
c     dslect
c     dminsn
c     dswap
c
c***********************************************************************

      integer n,nind2,index2,indexa
      real*8 x(n),xmdian,x1,x2,absx

      nind2 = n / 2

      if(amod(real(n),2.0).ne.0) then

        call dslect(nind2 + 1,n,x,xmdian)

        else

        call dslect(nind2,n,x,x1)
        call dminsn(nind2,x(nind2 + 1),x2,index2,absx,indexa)
        xmdian = (x1 + x2) / 2.0d0

      endif

      return
      end
      subroutine dminsn(lx,x,xmin,index,axmin,indexa)

c*******************************************************************************
c     This subroutine finds the minimum and the smallest absolute
c     elements in an array with the corresponding index of their
c     locations. The sign of the elements is taken into consideration.
c     see Robinson, "Multichannel Time Series Analysis", page 21
c     This is the double precision version of subroutine "minsn" in
c     Robinson with the addition of the output of the element with the
c     minimum absolute value.
c
c**** Subroutine inputs
c
c     lx = The length of array x
c     x = The array under investigation
c
c**** Subroutine outputs
c
c     xmin = The smallest element in array x
c     index = The fortran subscript of the smallest element, xmin
c     axmin = The element with the smallest absolute value in array x
c     indexa = The fortran subscript of the element with the smallest
c              absolute value, axmin
c
c*******************************************************************************

      integer i,index,indexa,lx
      real*8 x(lx),xmin,axmin

      index = 1
      indexa = 1

      do i = 1,lx

        if(x(index).gt.x(i)) index = i
        if(dabs(x(indexa)).gt.dabs(x(i))) indexa = i

      enddo

      xmin = x(index)
      axmin = x(indexa)
      return
      end
      subroutine dmmove(lx,x,y)

c*******************************************************************************
c     This subroutine copies lx values of array x into array y
c     see Robinson, "Multichannel Time Series Analysis", page 17.
c
c     This subroutine has a check to prevent premature overwriting when
c     the source array overlays the target.
c
c**** Subroutine inputs.
c
c     lx = the number of values to be moved.
c     x = the source array of the values.
c
c**** Subroutine outputs.
c
c     y = the target array that the values are moved into.
c
c*******************************************************************************

      integer lx,init,iend,istep,i
      real*8 x(lx),y(lx)

c*******************************************************************************
c     Determine the start and end of the loop to prevent overlap
c*******************************************************************************

      if (loc(x(1)).ge.loc(y(1))) then

        init = 1
        iend = lx
        istep = 1

        else if (loc(x(1)).lt.loc(y(1))) then

        init = lx
        iend = 1
        istep = -1

        else

        return

      endif

c*******************************************************************************
c     Move the elements.
c*******************************************************************************

      do i = init,iend,istep

        y(i) = x(i)

      enddo

      return
      end
      subroutine dmxadd(a,b,c,m,n)

c***********************************************************************
c     This subroutine sums the two m * n matrices, a and b to yield
c     the m * n matrix c.
c
c**** Subroutine inputs
c
c     a = The first m * n matrix.
c     b = The second m * n matrix.
c     m = The number of rows in matrices a, b, and, c
c     n = The number of columns in matrices a, b, and, c
c
c**** Subroutine outputs.
c
c     c = The m * n matrix that is the sum of matrices a and b.
c
c***********************************************************************

      integer ij,m,n
      real*8 a(m * n),b(m * n),c(m * n)

      do ij = 1,m * n

        c(ij) = a(ij) + b(ij)

      enddo

      return
      end
      subroutine dordr2(r2,n,varlim,ir2opt)

c***********************************************************************
c     During polynomial fitting when the curve begins to be fitted to
c     noise, the goodness of fit (square of the regression coefficient)
c     changes very slowly.  This subroutine determines the order of fit
c     at which this occurs based on searching backwards through the
c     array for the element which deviates from the mean of the
c     subsequent elements by greater than varlim.
c
c**** Subroutine inputs.
c
c     r2 = The array of squares of the regression coefficient.
c     n = The number of regression coefficients, (no. of elements in r2)
c     varlim = The size of the allowed variation from the mean.
c
c**** Subroutine outputs.
c
c     ir2opt = The index of the regression coefficient above which there
c              is little variation (within varlim of the mean of this
c              and the elements with greater indices)
c
c**** Subroutine required
c
c     dxmean
c
c***********************************************************************

      integer n,ir2opt,i
      real*8 r2(n),r2mean,varlim

      r2mean = r2(n)

      do i = n - 1,1,-1

        if(dabs(r2(i) - r2mean).le.varlim) then

          if(i.eq.1) then

            ir2opt = 1
            return

          endif

          call dxmean(r2(i),n - i + 1,r2mean)

          else

          ir2opt = i + 1
          return

        endif

      enddo

      end
      subroutine dscalm(x,lx,s)

c*******************************************************************************
c     This subroutine multiplies each element in an array, x by a scale
c     factor s. The length of array x is lx.
c
c     See Robinson, "Multichannel Time Series Analysis", page 19.
c
c**** Subroutine inputs.
c
c     x = The array of values to be scaled.
c     n = The number of elements in array, x.
c     s = The scaling factor.
c
c**** Subroutine outputs.
c
c     x = The array of values of x after multiplication by s.
c
c***********************************************************************

      integer lx,i
      real*8 x(lx),s

      do i = 1,lx

        x(i) = s * x(i)

      enddo

      return
      end
      subroutine dsdev(x,n,xmean,sdev)

c***********************************************************************
c     This subroutine computes the mean and standard deviation of the
c     elements in an array without round off error
c
c     Hill, I.D. 1979. On calculating a standard deviation.
c     Teaching Statistics, 1: 81-84.
c
c**** Subroutine inputs.
c
c     x = The array of values.
c     n = The number of values in array x.
c
c**** Subroutine outputs.
c
c     xmean = The mean value of the elements in array, x.
c     sdev = The standard deviation of the elements in x.
c
c***********************************************************************

      integer n,i
      real*8 x(n),xmean,sdev,delta,dblei

      xmean = x(1)
      sdev = 0.0d0

      do i = 2,n

        dblei = dble(i)
        delta = (x(i) - xmean) / dblei
        xmean = xmean + delta
        sdev = sdev + dblei * (dblei - 1.0d0) * delta * * 2

      enddo

      sdev = dsqrt(sdev / dble(n - 1))

      return
      end
      subroutine dslect(k,n,arr,kval)

c***********************************************************************
c     This subroutine finds the kth. smallest value of an array.  The
c     input array is arranged such that elements with values less than
c     the kth will be from arr(1:k - 1) and elements larger than the kth
c     will be from arr(k + 1:n) in arbitrary order respectively.
c
c
c**** Subroutine inputs
c
c     k = The size ordered element for which the value is required
c     n = The length of the array, arr
c     arr = The array to be scanned
c
c**** Subroutine outputs
c
c     kval = The value of the kth. element
c     arr = The rearranged but not completely sorted array.
c
c**** Subroutine required
c
c     dswap
c
c***********************************************************************

      integer k,n
      real*8 arr(n)

      integer i,ir,j,l,mid,lp1
      real*8 a,kval

      l = 1
      ir = n

c***********************************************************************
c     Loop until the active partition has only two elements
c***********************************************************************

      do while(ir - l.gt.1)

c***********************************************************************
c     Choose the median of the left,centre, and right elements as the
c     partitioning element, a.
c     Arrange so that arr(l + 1) <= arr(l) <= arr(ir)
c***********************************************************************

        mid = (l + ir) / 2
        lp1 = l + 1
        call dswap(arr(mid),arr(lp1))
        if(arr(lp1).gt.arr(ir)) call dswap(arr(ir),arr(lp1))
        if(arr(l).gt.arr(ir)) call dswap(arr(ir),arr(l))
        if(arr(lp1).gt.arr(l)) call dswap(arr(l),arr(lp1))

c***********************************************************************
c     Initialize pointers for partitioning
c***********************************************************************

        i = lp1
        j = ir
        a = arr(l)

c***********************************************************************
c     begin innermost loop, scan up to find element > a
c***********************************************************************

    2   i = i + 1

        do while(arr(i).lt.a)

          i = i + 1

        enddo

c***********************************************************************
c     scan down to find element < a
c***********************************************************************

       j = j - 1

        do while(arr(j).gt.a)

          j = j - 1

        enddo

c***********************************************************************
c    if pointers do not cross, exchange elements go through inner loop
c     again.
c***********************************************************************

        if(j.ge.i) then

          call dswap(arr(i),arr(j))
          goto 2

        endif

c***********************************************************************
c     Insert partitioning element, keep active the partition containing
c     the kth. element
c***********************************************************************

        arr(l) = arr(j)
        arr(j) = a
        if(j.ge.k) ir = j - 1
        if(j.le.k) l = i

      enddo

c***********************************************************************
c     Active partition contains two elements, swap if necessary then
c     return kval.
c***********************************************************************

      if(ir - l.eq.1) then

        if(arr(ir).lt.arr(l)) call dswap(arr(ir),arr(l))

      endif

      kval = arr(k)
      return
      end
      subroutine dswap(x,y)

c-----
c     This subroutine swaps two values  x and y
c-----

      real*8 x,y,temp

      temp = x
      x = y
      y = temp
      return
      end
      subroutine dxmean(x,n,xmean)

c***********************************************************************
c     This subroutine computes the mean and standard deviation of the
c     elements in an array without round off error
c
c     Hill, I.D. 1979. On calculating a standard deviation.
c     Teaching Statistics, 1: 81-84.
c
c**** Subroutine inputs.
c
c     x = The array of values.
c     n = The number of values in array x.
c
c**** Subroutine outputs.
c
c     xmean = The arithetic mean value of the elements in array, x.
c
c***********************************************************************

      integer n,i
      real*8 x(n),xmean,delta

      xmean = x(1)

      do i = 2,n

        delta = (x(i) - xmean) / dble(i)
        xmean = xmean + delta

      enddo

      return
      end
      subroutine dzero(lx,x)

c*******************************************************************************
c     This subroutine sets all the elements of an array x, of length, lx
c     to zero.
c
c     See Robinson, "Multichannel Time Series Analysis", page 16.
c
c*******************************************************************************

      integer lx,i
      real*8 x(lx)

      do i = 1,lx

        x(i) = 0.0d0

      enddo

      return
      end
      subroutine p3ind(x,y,iord,c)

c***********************************************************************
c     this subroutine calculates the powers of the two independent
c     variables at the point x,y in a three dimensional polynomial.
c     this subroutine is used in the surface fitting program polyfit and
c     the polynomial evaluation program polyint.
c
c*****Subroutine inputs.
c
c     x = The first independent variable value
c     y = The second independent variable value
c     iord = The order of the polynomial
c
c**** Subroutine outputs.
c
c     c = The array of the powers of x and y up to iord.
c
c***********************************************************************

      integer iord,j,jb,k,kb
      real*8 x,y,c((iord + 1) * (iord + 2) / 2)

      c(1) = 1.0d0
      jb = 1

      do j = 1,iord

        do k = 1,j

          jb = jb + 1
          kb = jb - j
          c(jb) = c(kb) * x

        enddo

        jb = jb + 1
        c(jb) = c(kb) * y

      enddo

      return
      end
      subroutine pncoef(ndim,iord,iord2,iord4)

c***********************************************************************
c     Compute the number of coefficients in a polynomial of any given
c     order.
c
c**** Subroutine inputs
c
c     ndim = The number of dimensions
c     iord = The order the polynomial
c
c**** Subroutine inputs
c
c     iord2 = The number of coefficients in the polynomial
c     iord4 = The number of elements in the least squares matrix.
c
c***********************************************************************

      integer ndim,iord,iord2,iord4

      if(ndim.eq.2) then

        iord2 = iord + 1

        else if(ndim.eq.3) then

        iord2 = (iord + 1) * (iord + 2) / 2

        else if(ndim.eq.4) then

        iord2 = (iord + 1) * (iord + 2) * (iord + 3) / 6

      endif

      iord4 = iord2 * iord2
      return
      end
      subroutine pordmx(iopt,ndim,nn,maxin,iord,iord2,iord4,ierr)

c***********************************************************************
c     This subroutine determines the maximum order of polynomial that
c     can be fitted to data points in two, three, or four dimensions.
c
c**** Subroutine inputs
c
c     iopt = Option choice flag
c            iopt < 0  No scaling of the variables is carried out
c            iopt > 0  Scaling of the variables is carried out
c     ndim = The number of dimensions
c     nn = The number of input data points.
c     maxin = The maximum absolute input value.
c
c**** Subroutine outputs
c
c     iord = The maximum possible order of polynomial that can be fitted
c     iord2 = The number of coefficients in the maximum possible order
c             of polynomial that can be fitted
c     iord4 = iord2 * iord2
c     ierr = Error code on exit
c            ierr = 0   A polynomial can be fitted no error
c            ierr = 1   Insufficient data points.
c
c**** Subroutines required
c
c     pncoef
c
c***********************************************************************

      integer iopt,ndim,nn,iord,iord2,iord4,ierr,logmax,depemax
      real*8 maxin
      ierr = 0

c***********************************************************************
c     Find iord for two dimensional data
c***********************************************************************

      if(ndim.eq.2) then

        if(nn.lt.2) then

          goto 1

          else if(nn.ge.15) then

          iord = 14

          else

          iord = nn - 1

        endif

c***********************************************************************
c     Find iord for three dimensional data
c***********************************************************************

        else if(ndim.eq.3) then

        if(nn.lt.3) then

          goto 1

          else if(nn.ge.120) then

          iord = 14

          else

          iord = int((sqrt(9.0 + 8.0 * float(nn - 1)) - 3.0) / 2.0)

        endif

c***********************************************************************
c     Find iord for four dimensional data
c***********************************************************************

        else if(ndim.eq.4) then

        if(nn.lt.4) then

          goto 1

          else if(nn.ge.120) then

          iord = 7

          else if(nn.ge.84.and.nn.lt.120) then

          iord = 6

          else if(nn.ge.56.and.nn.lt.84) then

          iord = 5

          else if(nn.ge.35.and.nn.lt.56) then

          iord = 4

          else if(nn.ge.20.and.nn.lt.35) then

          iord = 3

          else if(nn.ge.10.and.nn.lt.20) then

          iord = 2

          else if(nn.ge.4.and.nn.lt.10) then

          iord = 1

        endif

      endif

c***********************************************************************
c     Check for limits of unscaled data.
c***********************************************************************

      if(iopt.lt.0) then

        logmax = int(dble(depemax(maxin)) / (2.0d0 * dlog10(maxin)))
        iord = min0(logmax,iord)

      endif

c***********************************************************************
c     Compute iord2
c***********************************************************************

      call pncoef(ndim,iord,iord2,iord4)
      return

c***********************************************************************
c     If insufficient points exit with error code = 1
c***********************************************************************

    1 iord = 0
      ierr = 1
      return
      end
      subroutine tsig01(ndfn,ndfd,f,itest)

c***********************************************************************
c     This subroutine carries out the statistical 'f' test at the 0.1%
c     level of significance.
c
c**** Subroutine inputs.
c
c     ndfn = The no. of degrees of freedom of the numerator.
c     ndfd = The no. of degrees of freedom of the denominator.
c     f = The 'f' ratio value to be tested.
c     ftest = The table of values of the test statistic at the desired
c             level of significance according to the CRC. "Handbook of
c             Mathematical Tables".
c
c**** Subroutine outputs.
c
c     itest = The result of the test.
c             itest = 0    The test is not significant
c             itest = 1    The test is significant
c
c***********************************************************************

      real*8 f
      real*4 f4,ftest(646)
      integer ndfn,ndfd,itest,i,j,ij

c***********************************************************************
c     Initialise the table of f-values at the 0.1% level
c***********************************************************************

      data (ftest(i),i = 1,34)/4052.0,98.503,34.116,21.198,16.258,13.745
     :,12.246,11.259,10.561,10.044,9.646,9.3302,9.0738,8.8616,8.6831,8.5
     :310,8.3997,8.2854,8.1850,8.0960,8.0166,7.9454,7.8811,7.8229,7.7698
     :,7.7213,7.6767,7.6356,7.5976,7.5625,7.3141,7.0771,6.8510,6.6349/
      data (ftest(i),i = 35,68)/4999.5,99.0,30.817,18.0,13.274,10.925,9.
     :5466,8.6491,8.0215,7.5594,7.2057,6.9266,6.701,6.5149,6.3589,6.2262
     :,6.1121,6.0129,5.9259,5.8489,5.7804,5.719,5.6637,5.6136,5.568,5.52
     :63,5.4881,5.4529,5.4205,5.3904,5.1785,4.9774,4.7865,4.6052/
      data (ftest(i),i = 69,102)/5403.3,99.166,29.457,16.694,12.06,9.779
     :5,8.4513,7.591,6.9919,6.5523,6.2167,5.9526,5.7394,5.5639,5.417,5.2
     :922,5.185,5.0919,5.0103,4.9382,4.874,4.8166,4.7649,4.7181,4.6755,4
     :.6366,4.6009,4.5681,4.5378,4.5097,4.3126,4.1259,3.9493,3.7816/
      data (ftest(i),i = 103,136)/5624.6,99.249,28.71,15.977,11.392,9.14
     :83,7.8467,7.006,6.4221,5.9943,5.6683,5.4119,5.2053,5.0354,4.8932,4
     :.7726,4.669,4.579,4.5003,4.4307,4.3688,4.3134,4.2635,4.2184,4.1774
     :,4.14,4.1056,4.074,4.0449,4.0179,3.8283,3.6491,3.4796,3.3192/
      data (ftest(i),i = 137,170)/5763.7,99.299,28.237,15.522,10.967,8.7
     :459,7.4604,6.6318,6.0569,5.6363,5.316,5.0643,4.8616,4.695,4.5556,4
     :.4374,4.3359,4.2479,4.1708,4.1027,4.0421,3.988,3.9392,3.8951,3.855
     :,3.8183,3.7848,3.7539,3.7254,3.699,3.5138,3.3389,3.1735,3.0173/
      data (ftest(i),i = 171,204)/5859.0,99.332,27.911,15.207,10.672,8.4
     :661,7.1914,6.3707,5.8018,5.3858,5.0692,4.8206,4.6204,4.4558,4.3185
     :,4.2016,4.1015,4.0146,3.9386,3.8714,3.8117,3.7583,3.7102,3.6667,3.
     :6272,3.5911,3.558,3.5276,3.4995,3.4735,3.291,3.1187,2.9559,2.802/
      data (ftest(i),i = 205,238)/5928.3,99.356,27.672,14.976,10.456,8.2
     :6,6.9928,6.1776,5.6129,5.2001,4.8861,4.6395,4.441,4.2779,4.1415,4.
     :0259,3.9267,3.8406,3.7653,3.6987,3.6396,3.5867,3.539,3.4959,3.4568
     :,3.421,3.3882,3.3581,3.3302,3.3045,3.1238,2.953,2.7918,2.6393/
      data (ftest(i),i = 239,272)/5981.6,99.374,27.489,14.799,10.289,8.1
     :016,6.8401,6.0289,5.4671,5.0567,4.7445,4.4994,4.3021,4.1399,4.0045
     :,3.8896,3.791,3.7054,3.6305,3.5644,3.5056,3.4530,3.4057,3.3629,3.3
     :239,3.2884,3.2558,3.2259,3.1982,3.1726,2.993,2.8233,2.6629,2.5113/
      data (ftest(i),i = 273,306)/6022.5,99.388,27.345,14.659,10.158,7.9
     :761,6.7188,5.9106,5.3511,4.9424,4.6315,4.3875,4.1911,4.0297,3.8948
     :,3.7804,3.6822,3.5971,3.5225,3.4567,3.3981,3.3458,3.2989,3.256,3.2
     :172,3.1818,3.1494,3.1195,3.092,3.0665,2.8876,2.7185,2.5586,2.4073/
      data (ftest(i),i = 307,340)/6055.8,99.399,27.229,14.546,10.051,7.8
     :741,6.6201,5.8143,5.2565,4.8492,4.5393,4.2961,4.1003,3.9394,3.8049
     :,3.6909,3.5931,3.5082,3.4338,3.3682,3.3098,3.2576,3.2106,3.1681,3.
     :1294,3.0941,3.0618,3.0320,3.0045,2.9791,2.8005,2.6318,2.4721,2.320
     :9/
      data (ftest(i),i = 341,374)/6106.3,99.416,27.052,14.374,9.8883,7.7
     :183,6.4691,5.6668,5.1114,4.7059,4.3974,4.1553,3.9603,3.8001,3.6662
     :,3.5527,3.4552,3.3706,3.2965,3.2311,3.1729,3.1209,3.074,3.0316,2.9
     :931,2.9579,2.9256,2.8959,2.8685,2.8431,2.6648,2.4961,2.3363,2.1848
     :/
      data (ftest(i),i = 375,408)/6157.3,99.432,26.872,14.198,9.7222,7.5
     :590,6.3143,5.5151,4.9621,4.5582,4.2509,4.0096,3.8154,3.6557,3.5222
     :,3.4089,3.3117,3.2273,3.1533,3.088,3.0299,2.978,2.9311,2.8887,2.85
     :02,2.815,2.7827,2.753,2.7256,2.7002,2.5216,2.3523,2.1915,2.0385/
      data (ftest(i),i = 409,442)/6208.7,99.449,26.69,14.02,9.5527,7.395
     :8,6.1554,5.3591,4.808,4.4054,4.099,3.8584,3.6646,3.5052,3.3719,3.2
     :588,3.1615,3.0771,3.0031,2.9377,2.8796,2.8274,2.7805,2.738,2.6993,
     :2.6640,2.6316,2.6017,2.5742,2.5487,2.3689,2.1978,2.0346,1.8783/
      data (ftest(i),i = 443,476)/6234.6,99.458,26.598,13.929,9.4665,7.3
     :127,6.0743,5.2793,4.729,4.3269,4.0209,3.7805,3.5868,3.4274,3.294,3
     :.1808,3.0835,2.999,2.9249,2.8594,2.8011,2.7488,2.7017,2.6591,2.620
     :3,2.5848,2.5522,2.5223,2.4946,2.4689,2.288,2.1154,1.95,1.7908/
      data (ftest(i),i = 477,510)/6260.7,99.466,26.505,13.838,9.3793,7.2
     :285,5.9921,5.1981,4.6486,4.2469,3.9411,3.7008,3.507,3.3476,3.2141,
     :3.1007,3.0032,2.9185,2.8422,2.7785,2.72,2.6675,2.6202,2.5773,2.538
     :3,2.5026,2.4699,2.4397,2.4118,2.386,2.2034,2.0285,1.86,1.6964/
      data (ftest(i),i = 511,544)/6286.8,99.474,26.411,13.745,9.2912,7.1
     :432,5.9084,5.1156,4.5667,4.1653,3.8596,3.6192,3.4253,3.2656,3.1319
     :,3.0182,2.9205,2.8354,2.7608,2.6947,2.6359,2.5831,2.5355,2.4923,2.
     :453,2.417,2.384,2.3535,2.3253,2.2992,2.1142,1.936,1.7628,1.5923/
      data (ftest(i),i = 545,578)/6313.0,99.483,26.316,13.652,9.202,7.05
     :68,5.8236,5.0316,4.4831,4.0819,3.7761,3.5355,3.3413,3.1813,3.0471,
     :2.9330,2.8348,2.7493,2.6742,2.6077,2.5484,2.4951,2.4471,2.4035,2.3
     :637,2.3273,2.2938,2.2629,2.2344,2.2079,2.0194,1.8363,1.6557,1.473/
      data (ftest(i),i = 579,612)/6339.4,99.491,26.221,13.558,9.1118,6.9
     :69,5.7372,4.946,4.3978,3.9965,3.6904,3.4494,3.2548,3.0942,2.9595,2
     :.8447,2.7459,2.6597,2.5839,2.5168,2.4568,2.4029,2.3542,2.3099,2.26
     :95,2.2325,2.1984,2.167,2.1378,2.1107,1.9172,1.7263,1.533,1.3246/
      data (ftest(i),i = 613,646)/6366.0,99.501,26.125,13.463,9.0204,6.8
     :801,5.6495,4.8588,4.3105,3.909,3.6025,3.3608,3.1654,3.004,2.8684,2
     :.7528,2.653,2.566,2.4893,2.4212,2.3603,2.3055,2.2559,2.2107,2.1694
     :,2.1315,2.0965,2.0642,2.0342,2.0062,1.8047,1.6006,1.3805,1.0/

c***********************************************************************
c     change the input variable to real*4 to test against the table
c***********************************************************************

      f4 = sngl(f)
      i = 33
      j = 18
      itest = 0
      if(ndfd.lt.120) i = 32
      if(ndfd.lt.60) i = 31
      if(ndfd.lt.40) i = 30
      if(ndfd.le.30) i = ndfd
      if(ndfn.lt.120) j = 17
      if(ndfn.lt.60) j = 16
      if(ndfn.lt.40) j = 15
      if(ndfn.lt.30) j = 14
      if(ndfn.lt.24) j = 13
      if(ndfn.lt.20) j = 12
      if(ndfn.lt.15) j = 11
      if(ndfn.lt.12) j = 10
      if(ndfn.le.10) j = ndfn
      ij = (j - 1) * 34 + i
      if(f4.gt.ftest(ij)) itest = 1
      return
      end

      subroutine d2hsmg(a,iorda,iord,iord2,b)

c*************************************************************************
c     This subroutine fills in the symmetric matrix given by a column by
c     row multiplication of a one dimensional power series using a high
c     speed matrix generator.
c
c     Thompson, G.T., 1992, "The Grand Unified Theory of Least Squares",
c     Computers and Geosciences, Vol. 18, No. 7, pp. 815-822.
c
c**** Subroutine inputs
c
c     a = The vector of powers of one independent variable up to twice the
c         order of the matrix being filled.
c     iorda = The length of the vector, a.
c     iord = The order of the polynomial being tested.
c     iord2 = The number of coeficients in the polynomial being tested.
c
c**** Subroutine outputs
c
c     b = The symmetric matrix of normal equations.
c
c**** Subroutines required.
c
c     dmmove
c
c*************************************************************************

      integer iorda,iord,iord2,i
      real*8 a(iorda),b(iord2 * iord2)

      do i = 1,iord + 1

        call dmmove(iord2,a(i),b((i - 1) * iord2 + 1))

      enddo

      return
      end
      subroutine p2ind(x,iord,c)

c***********************************************************************
c     This subroutine calculates the powers of the independent variable
c     in a polynomial.
c
c**** Subroutine inputs.
c
c     x = The variable value
c     iord = The order of the polynomial
c
c**** Subroutine outputs.
c
c     c = The array of the powers of x up to iord.
c
c***********************************************************************

      integer j,iord
      real*8 x,c(iord + 1)

      c(1) = 1.0d0

      do j = 2,iord + 1

        c(j) = c(j - 1) * x

      enddo

      return
      end
      subroutine dchlin(n,a,ierr)

c***********************************************************************
c     This subroutine computes inverse of a positive definite symmetric
c     matrix by Cholesky decomposition.
c
c     See: Golub, G.H. and Reinsch, C., "Singular Value Decomposition
c     and Least Squares Solutions", Contribution I/1 in "Handbook of
c     Numerical Computation", Vol.2, Linear Algebra, Page 9
c
c**** Subroutine inputs
c
c     n = The number of rows and columns in a.
c     a = The positive definite symmetric matrix to be decomposed
c         stored as an n * (n + 1) matrix so that the last column
c         carries the reciprocals of the diagonals
c
c**** Subroutine outputs
c
c     a = The inverse of the input matrix.
c     ierr = completion code.
c            ierr = 0  Normal completion
c            ierr = 1  The matrix is not positive definite
c
c***********************************************************************

      integer n,ierr
      real*8 a(n * (n + 1))
      real*8 x,y,z
      integer i,j,k,kj1,ki1,jj1,ii1,ij,ji,ik,jk,ij1,ik1,is1,it1
      integer in,jn,km1n,np1,im1,jm1,ioff,joff

      ierr = 0
      np1 = n + 1

c***********************************************************************
c     Form the transpose of the lower triangular matrix of the Cholesky
c     decomposition (i.e. put it into the upper triangle of a)
c***********************************************************************

      in = n

      do i = 1,n

        ii1 = in + i
        jn = in
        im1 = i - 1

        do j = i,n

          ji = in - n + j
          ij1 = jn + i
          x = a(ji)
          ki1 = in + im1
          kj1 = jn + im1

          do k = im1,1,-1

            x = x - a(kj1) * a(ki1)
            ki1 = ki1 - 1
            kj1 = kj1 - 1

          enddo

          if(j.eq.i) then

            if(x.le.0.0d0) then

              ierr = 1
              return

              else

              y = 1.0d0 / dsqrt(x)
              a(ii1) = y

            endif

            else

            a(ij1) = x * y

          endif

          jn = jn + n

        enddo

        in = in + n

      enddo

c***********************************************************************
c     Form the inverse of the transpose of the lower triangular matrix
c     of the Cholesky decomposition (i.e. the inverse of the upper
c     triangle of a)
c***********************************************************************

      ioff = 2

      do i = 1,n

        jn = ioff * n
        joff = i * n

        do j = i + 1,n

          z = 0.0d0
          jm1 = j - 1
          ij1 = jn + i
          jj1 = jn + j
          ik1 = joff + i
          kj1 = jn + jm1

          do k = jm1,i,-1

            z = z - a(kj1) * a(ik1)
            ik1 = ik1 - n
            kj1 = kj1 - 1

          enddo

          a(ij1) = z * a(jj1)
          jn = jn + n
          joff = joff + n

        enddo

        ioff = ioff + 1

      enddo

c***********************************************************************
c     Form the inverse of a in the lower triangle of a.
c***********************************************************************

      do i = 1,n

        do j = i,n

          z = 0.0d0
          km1n = j * n
          ij1 = km1n + i
          ik = ij1
          jk = km1n + j

          do k = j + 1, np1

            z = z + a(jk) * a(ik)
            ik = ik + n
            jk = jk + n

          enddo

          a(ij1) = z

        enddo

      enddo

c***********************************************************************
c     Copy the lower triangle into the upper triangle.
c***********************************************************************

      is1 = np1
      it1 = 1

      do i = 1,n

        call dmmove(n,a(is1),a(it1))
        it1 = is1
        is1 = is1 + n

      enddo

      jn = 0

      do j = 1,n

        ji = jn + j
        ij = ji

        do i = j,n

          a(ij) = a(ji)
          ji = ji + n
          ij = ij + 1

        enddo

        jn = jn + n

      enddo

      return
      end
      subroutine dmmult(a,b,c,l,m,n)

c***********************************************************************
c     This subroutine multiplies two matrices a, and b to yield c.
c
c     Note:    a * b does not equal b * a.
c
c     see Davies, "Statistics and Data Analysis in Geology", page 137.
c
c**** Subroutine inputs.
c
c     a = The first matrix to be multiplied
c     b = The second matrix to be multiplied
c     l = The number of rows in a = The number of rows in c
c     m = The number of columns in b = The number of columns in c
c     n = The number of columns in a = The number of rows in b
c
c**** Subroutine outputs.
c
c     c = The matrix product a * b
c
c***********************************************************************

      real*8 a(l * n),b(n * m),c(l * m)
      integer i,ij,ik,j,k,kj,joff

      do i = 1,l

        joff = 0
        ij = i

        do j = 1,m

          c(ij) = 0.0d0
          ik = i
          kj = joff + 1

          do k = 1,n

            c(ij) = c(ij) + a(ik) * b(kj)
            ik = ik + l
            kj = kj + 1

          enddo

          joff = joff + n
          ij = ij + l

        enddo

      enddo

      return
      end
      subroutine dmxsub(a,b,c,m,n)

c***********************************************************************
c     This subroutine subtracts the two m * n matrices, a and b to yield
c     the m * n matrix c.
c
c**** Subroutine inputs
c
c     a = The first m * n matrix.
c     b = The second m * n matrix.
c     m = The number of rows in matrices a, b, and, c
c     n = The number of columns in matrices a, b, and, c
c
c**** Subroutine outputs.
c
c     c = The m * n matrix that is the difference of matrices a and b.
c
c***********************************************************************

      integer ij,m,n
      real*8 a(m * n),b(m * n),c(m * n)

      do ij = 1,m * n

        c(ij) = a(ij) - b(ij)

      enddo

      return
      end
      subroutine p2eval(x,a,c,iord,iord2,s)

c***********************************************************************
c     Evaluates the estimated value of a two dimensional polynomial with
c     one independent variable.
c
c**** Subroutine inputs.
c
c     x = The value of the independent variable.
c     a = The array of polynomial coefficients.
c     iord = The order of the polynomial.
c     iord2 = The length of the coefficient array, a
c
c**** Subroutine outputs.
c
c     s = The value of the polynomial at the point x,y,z.
c     c = Working space for the values of the power series - length of c
c         is iord2 = iord1 + 1
c
c***********************************************************************

      real*8 x,s
      integer i,iord,iord2
      real*8 a(iord2),c(iord2)

      call p2ind(x,iord,c)
      s = 0.0d0

      do i = iord2,1,-1

        s = s + c(i) * a(i)

      enddo

      return
      end
      subroutine dwsdev(x,w,n,wmean,wsdev)

c***********************************************************************
c     This subroutine computes the weighted mean and standard deviation 
c     of the elements in an array without round off error
c
c     Hill, I.D. 1979. On calculating a standard deviation.
c     Teaching Statistics, 1: 81-84.
c
c**** Subroutine inputs.
c
c     x = The array of values.
c     w = The array of weights
c     n = The number of values in array x.
c
c**** Subroutine outputs.
c
c     wmean = The weighted mean value of the elements in array, x.
c     wsdev = The weighted standard deviation of the elements in x.
c
c***********************************************************************

      integer n,i
      real*8 x(n),w(n),wmean,wsdev
      real*8 delta,div,deptiny,small

      wmean = x(1)
      div = w(1)
      wsdev = 0.0d0
      small = deptiny(small)

      do i = 2,n

        div = div + w(i)

        if(div.gt.small) then

          delta = x(i) - wmean
          wsdev = wsdev + 
     :            w(i) * (div - w(i)) / div * delta * delta
          wmean = wmean + delta  * w(i) / div

        endif

      enddo

      wsdev = dsqrt(wsdev / dble(n - 1))

      return
      end
      subroutine peval2(x,coef,iord,p)

c***********************************************************************
c     Evaluates the estimated value of a two dimensional polynomial with
c     one independent variable.
c
c**** Subroutine inputs.
c
c     x = The value of the independent variable.
c     coef = The array of polynomial coefficients.
c     iord = The order of the polynomial.
c
c**** Subroutine outputs.
c
c     p = The value of the polynomial at the point x.
c
c***********************************************************************

      integer i,iord,iord2
      real*8 x,p
      real*8 coef(iord + 1)

      iord2 = iord + 1
      p = coef(iord2)

      do i = iord,1,-1

        p = p * x + coef(i)

      enddo

      return
      end
      subroutine pxgen (ipoly,x,coef,nsamp,work)

#include <f77/iounit.h>

c routine to contruct gain trace based on polynomial p(x)

      integer ipoly,nsamp,ntrc
      integer j,i

      real   work (nsamp)
      real*8 x(nsamp), coef(*),value


c construct output trace for all samples

c     write(0,*)(sngl(x(i)),i=1,nsamp)
         do i = 1, nsamp

            call peval2(x(i),coef,ipoly,value)
            work (i) = sngl(value)

         enddo

      return
      end

