************************************************************************
*                                                                      *
c                 copyright 2004, Amoco Production Company             *
c                             All Rights Reserved                      *
c                     an affiliate of BP America Inc.                  *
************************************************************************
*  ROUTINE:       MEDIAN                                               *
*  ROUTINE TYPE:  SUBROUTINE                                           *
*  PURPOSE:  Finds the median of a set of data.                        *
*  ENTRY POINTS:                                                       *
*      MEDIAN  (ARR,MINSMP,MAXSMP,MINWIN,MAXWIN)                       *
*  ARGUMENTS:                                                          *
*      ARR     REAL     Update  (DROWS,*) - Amplitude array            *
*      MINSMP  INTEGER  Input             - Declared lower dim of A    *
*      MAXSMP  INTEGER  Input             - Declared upper dim of A    *
*      MINWIN  INTEGER  Input             - Starting position          *
*      MAXWIN  INTEGER  Input             - Ending position            *
*       +------------------------------------------------------+       *
*       |               DEVELOPMENT INFORMATION                |       *
*       +------------------------------------------------------+       *
*  AUTHOR:   Kelly D. Crawford                  ORIGIN DATE: 95/05/15  *
*  LANGUAGE: FORTRAN 77                  DATE LAST COMPILED: 95/05/15  *
*       +------------------------------------------------------+       *
*       |                 EXTERNAL ENVIRONMENT                 |       *
*       +------------------------------------------------------+       *
*  ROUTINES CALLED:                                                    *
*      PART    INTEGER - Partition function                            *
*  INTRINSIC FUNCTIONS CALLED:                                         *
*      REAL    REAL - Converts integer value to a real value           *
*  FILES:                                                              *
*      0  ( OUTPUT SEQUENTIAL ) - Warning message                      *
*  COMMON:           NONE                                              *
*  STOP CODES:       NONE                                              *
*       +------------------------------------------------------+       *
*       |             OTHER DOCUMENTATION DETAILS              |       *
*       +------------------------------------------------------+       *
*  ERROR HANDLING:  None needed.                                       *
*  GENERAL DESCRIPTION:  Uses a partitioning scheme to located the     *
*       median of a set of data (this is the most efficient median     *
*       algorithm known).  Partitions the data until the partition     *
*       element coincides with the middle of the dataset.              *
*  REVISED BY:  ???                           REVISION DATE: ?Y/?M/?D  *
*       +------------------------------------------------------+       *
*       |                 ANALYSIS INFORMATION                 |       *
*       +------------------------------------------------------+       *
*  NONSTANDARD FEATURES:   NONE DETECTED                               *
********************   END OF DOCUMENTATION PACKAGE   ******************
      real function median(arr,minsmp,maxsmp,minwin,maxwin)

      implicit none

c declare variables passed by calling routine

      integer minsmp, maxsmp, minwin, maxwin
      real arr(minsmp:maxsmp)

c declare local variables
 
      integer part
      external part

      integer m, left, right, middle, n

c initialize variables

      left = minwin
      right = maxwin
      n = right - left
      middle = (n / 2) + left

*     If we have 0, 1 or 2 (n < 0, n < 2) values don't do any computation
      if (n .lt. 0) then
         median = 0.0
         return
      else if (n .lt. 2) then
         median = arr(minsmp)
         return
      endif

*     Partition the dataset until the return value from part = middle
 10   m = part(arr, minsmp, maxsmp, left, right)
      if (m .lt. middle) then
         left = m + 1
         goto 10
      else if (m .gt. middle) then
         right = m - 1
         goto 10
      endif

*     Return the middle array value
      median = arr(middle)
      return
      end
************************************************************************
*                                                                      *
c                 copyright 2004, Amoco Production Company             *
c                             All Rights Reserved                      *
c                     an affiliate of BP America Inc.                  *
************************************************************************
*  ROUTINE:       PART                                                 *
*  ROUTINE TYPE:  FUNCTION  INTEGER                                    *
*  PURPOSE:  Median-of-three partitioning routine used in quicksort.   *
*       Partitions an array around a chosen partition element.  Return *
*       value 'part' is an index into the array such that:             *
*          a(minidx..part-1) < a(part) <= a(part+1..maxidx)            *
*  ENTRY POINTS:                                                       *
*      PART  INTEGER  (A,MINIDX,MAXIDX,LEFT,RIGHT)                     *
*  ARGUMENTS:                                                          *
*      A       REAL     Update  (*) - Array of values with indices     *
*                                     between [minidx,maxidx].         *
*      MINIDX  INTEGER  Input       - Minimum declared index of A.     *
*      MAXIDX  INTEGER  Input       - Maximum declared index of A.     *
*      LEFT    INTEGER  Input       - Leftmost position of the array   *
*                                     to partition                     *
*      RIGHT   INTEGER  Input       - Rightmost position of the array  *
*                                     to partition                     *
*       +------------------------------------------------------+       *
*       |               DEVELOPMENT INFORMATION                |       *
*       +------------------------------------------------------+       *
*  AUTHOR:   Kelly D. Crawford                  ORIGIN DATE: 95/05/15  *
*  LANGUAGE: FORTRAN 77                  DATE LAST COMPILED: 95/05/15  *
*       +------------------------------------------------------+       *
*       |                 EXTERNAL ENVIRONMENT                 |       *
*       +------------------------------------------------------+       *
*  ROUTINES CALLED:  NONE                                              *
*  INTRINSIC FUNCTIONS CALLED:  NONE                                   *
*  FILES:            NONE                                              *
*  COMMON:           NONE                                              *
*  STOP CODES:       NONE                                              *
*       +------------------------------------------------------+       *
*       |             OTHER DOCUMENTATION DETAILS              |       *
*       +------------------------------------------------------+       *
*  ERROR HANDLING:  None needed.  However, the caller must make sure   *
*                   that left < right before calling this function!    *
*  GENERAL DESCRIPTION:  From Sedgewick, "Algorithms".  See below.     *
*  REVISED BY:  ???                           REVISION DATE: ?Y/?M/?D  *
*       +------------------------------------------------------+       *
*       |                 ANALYSIS INFORMATION                 |       *
*       +------------------------------------------------------+       *
*  NONSTANDARD FEATURES:   NONE DETECTED                               *
********************   END OF DOCUMENTATION PACKAGE   ******************
************************************************************************
      integer function part(a, minidx, maxidx, left, right)
      integer minidx, maxidx, left, right
      real a(minidx:maxidx)

      integer l, r, m, p, i, j
      real t, v

*     Copy and compute left, right and middle values
      l = left
      r = right
      m = (right-left)/2 + left

* Use the median-of-three partitioning.  Select a median of three elements
* and move it to a(r).  The silly looking if block below, believe it or
* not, is the fastest way to find the median of three elements.  Note that
* at most, there will only be three comparisons.
      if (a(l) .ge. a(m)) then
         if (a(m) .ge. a(r)) then
            p = m
         else if (a(l) .ge. a(r)) then
            p = r
         else
            p = l
         endif
      else if (a(m) .ge. a(r)) then
         if (a(l) .lt. a(r)) then
            p = r
         else
            p = l
         endif
      else
         p = m
      endif
      if (r .ne. p) then
         t = a(r)
         a(r) = a(p)
         a(p) = t
      endif

      v = a(r)
      i = l
      j = r - 1

 10   if (i .le. j) then

 20      if (j .ge. l) then
            if (a(j) .ge. v) then
               j = j - 1
               goto 20
            endif
         endif

 30      if (a(i) .lt. v) then
            i = i + 1
            goto 30
         endif

         if (i .lt. j) then
            t = a(i)
            a(i) = a(j)
            a(j) = t
            i = i + 1
            j = j - 1
         endif

         goto 10
      endif

      if (i .ne. r) then
         t = a(i)
         a(i) = a(r)
         a(r) = t
      endif

*     a(i) is now in place.  Return i.
      part = i

      end
