C***********************************************************************
C                 copyright 2001, Amoco Production Company             *
C                             All Rights Reserved                      *
C                     an affiliate of BP America Inc.                  *
C***********************************************************************
c**********************************************************************c
c stratslab reads seismic trace data from an input file,
C and allows the user to flatten and window about a given horizon
c or geologic formation as defined by the trace headers and command rec
c arguments.
c**********************************************************************c
c______________________________________________________________________
c          Kurt J. Marfurt (Amoco EPTG, Tulsa, OK, USA)
c______________________________________________________________________
c______________________________________________________________________
c     include statements that define the USP environment.
c______________________________________________________________________

      implicit none

#include <f77/iounit.h>
#include <f77/lhdrsz.h>
#include <f77/sisdef.h>
#include <save_defs.h>

c______________________________________________________________________
c     arrays used for the seismic rec header:            
c______________________________________________________________________
      integer   sheader(SZLNHD)
c______________________________________________________________________
c     data arrays:
c______________________________________________________________________
      real        uin(-ITRWRD:SZLNHD+3)
      real        uout(-ITRWRD:SZLNHD)
      real        twgt(0:SZLNHD)
      real        thoriz(20000,2)
     
      real        nhtVal, deadVal, UnitSc, tstart_orig, znull
      real        tdatum, ttaper, tmax, dt, dtmsec
      real        recunit, trcunit, smpunit, recoff, trcoff, smpoff
      real        v2, w2, adiv

      logical     nhtLive
c
      integer ndiv, k2, nfft, nttaper, maxseg, maxpick

      parameter (ndiv=256,adiv=ndiv) 
      real      w(-2:+3,0:ndiv)
c______________________________________________________________________
c     timing arrays:
c______________________________________________________________________
      real        cputim(20),waltim(20)
c
      integer     startrec,endrec,starttrace,endtrace
      real        uw,lw
c______________________________________________________________________
c     file names and control parameters:
c______________________________________________________________________
      character*120  file_in,file_out,file_xsd
      character*6    lhwd,uhwd             

      integer        argis, jerr, luin, luout, luxsd, lbytes, nsamp_in
      integer nsi, nrec, ntrace, ierror, nsamp_out, nbytes_out, lbyout
      integer        ordfft
c_______________________________________________________________
c     miscellaneous statements.
c_______________________________________________________________
      character*9    name
      logical        verbose,query,wrxsd
      logical        uflat,lflat,mflat, impatient

      real pi, twopi, undefined, vtot, wtot

      parameter      (pi=3.1415926,twopi=2.*pi)
      parameter      (undefined=-999999.) 
         
      data           name/'STRATSLAB'/
c_______________________________________________________________
c     initialize timing arrays.
c_______________________________________________________________
      call vclr(cputim,1,20)
      call vclr(waltim,1,20)
      call timstr(vtot,wtot)
      query=((argis( '-?' ) .gt. 0)
     :     .or. (argis( '-h' ) .gt. 0)
     :     .or. (argis( '-help' ) .gt. 0))
      if(query) then
c_______________________________________________________________
c        echo command rec parameters needed to run this program to 
c        the terminal (stderr), then exit gracefully.
c_______________________________________________________________
         call help(ler)
         call exit(0)
      endif
c_______________________________________________________________
c     open printout files
c_______________________________________________________________
#include <f77/open.h>
c_______________________________________________________________
c     read i/o data set names from the command rec.
c_______________________________________________________________
      call argstr('-N',file_in, ' ', ' ')
      call argstr('-X',file_xsd, ' ', ' ')
      call argstr('-O',file_out, ' ', ' ')

      call argr4('-nonHorzTraceValue',nhtVal,0,0)
      call argr4('-DeadTraceValue',deadVal,0,0)
      nhtLive = (argis('nonHorzTraceLive') .gt. 0)

c_______________________________________________________________
c     get logical unit number for input and output.
c_______________________________________________________________
      call getln(luin,file_in,'r',0)
      call getln(luout,file_out,'w',1)
      if(file_xsd .ne. ' ') then
         wrxsd=.true.
         open(luxsd,file=file_xsd,status='unknown')
      else
         wrxsd=.false.
      endif
c_______________________________________________________________
c     read rec header from input dataset.
c_______________________________________________________________
      lbytes=0
      call rtape(luin,sheader,lbytes)
      if(lbytes .eq. 0) then
         write(lerr,*)'STRATSLAB: no header read from unit ',luin
         write(ler,*)'STRATSLAB: no header read from unit ',luin
         call exitfu(4666)
      endif
c______________________________________________________________________
c     print out the  historical rec header.
c______________________________________________________________________
      call hlhprt (sheader,lbytes,name,len(name),lerr)
c_______________________________________________________________________
c     extract relevent values from rec header.
c_______________________________________________________________________
      call saver(sheader,'NumSmp',nsamp_in,LINEHEADER)
      call saver(sheader,'SmpInt',nsi,LINEHEADER)
      call saver(sheader,'NumRec',nrec,LINEHEADER)
      call saver(sheader,'NumTrc',ntrace,LINEHEADER)
      call saver(sheader,'TmMsFS',tstart_orig,LINEHEADER)
      call saver(sheader,'HrzNul',znull,LINEHEADER)
      call saver(sheader,'UnitSc',UnitSc,LINEHEADER)
      if(znull .eq. 0) then
         znull=-30000
      endif
c_______________________________________________________________________
c     calculate the sample increment in msec.
c_______________________________________________________________________
      dt =  float(nsi) * UnitSc
      dtmsec = dt * 1000.0
c_______________________________________________________________
c     read other command rec parameters.             
c_______________________________________________________________
      call argi4('-startrec',startrec,1,1)
      call argi4('-endrec',endrec,nrec,nrec)
      call argi4('-starttrace',starttrace,1,1)
      call argi4('-endtrace',endtrace,ntrace,ntrace)
      call argstr('-uh',uhwd,'Horz01','Horz01')
      call argstr('-lh',lhwd,'      ','      ')
      call argr4('-uw',uw,-48.,-48.)
      call argr4('-lw',lw,+48.,+48.)
      call argr4('-tdatum',tdatum,undefined,undefined)
      call argr4('-ttaper',ttaper,0.,0.)
      call argr4('-tmax',tmax,200.,200.)
      uflat=(argis('-U') .gt. 0)
      lflat=(argis('-L') .gt. 0)
      verbose=(argis('-V') .gt. 0)
      impatient=(argis('-VT') .gt. 0)
c
      ierror=0
      if(.not. uflat .and. .not. lflat) then
         mflat=.true.
      endif
      if(lhwd .eq. ' ') then
         lhwd=uhwd
      endif
      if(tdatum .eq. undefined) then
         tdatum=.5*tmax
      endif
      nsamp_out=nint(tmax/dtmsec)+1
      nbytes_out=(nsamp_out+ITRWRD)*szsmpd
      k2=ordfft(nsamp_in)
      nfft=2**k2
      nttaper=nint(ttaper/dtmsec)
      if(ttaper .gt. tdatum) then
         write(lerr,*) 'taper greater than distance to datum!'
         write(lerr,*) 'ttaper = ',ttaper
         write(lerr,*) 'tdatum = ',tdatum
      endif

      if(wrxsd) then
c_______________________________________________________________________
c        write out the time pick file header in xsd format.
c_______________________________________________________________________
         recunit=1
         trcunit=1
         smpunit=dtmsec
         recoff=0
         trcoff=0
         smpoff=tstart_orig-dtmsec
         maxseg=(endrec-startrec+1)*2
         maxpick=endtrace-starttrace+1
         write(luxsd,200)'Units ',recunit,trcunit,smpunit,
     1                    nrec,ntrace,nsamp_in,
     2                   ' Offset',recoff,trcoff,smpoff,
     3                   ' Count  ',maxseg,maxpick
200      format(a6,f12.6,1x,f12.6,1x,f12.6,1x,i5,1x,i5,1x,i5,
     *          a7,f12.6,1x,f12.6,1x,f12.6,a8,i5,1x,i5)
      endif
c______________________________________________________________________
c     print out miscellaneous parameters.   
c______________________________________________________________________
      write(lerr,*)
      write(lerr,'(a,t40,i5,t50,a)') 'input seismic file name',
     1                         luin,file_in
      write(lerr,'(a,t40,i5,t50,a)') 'output seismic file name',
     1                         luout,file_out
      write(lerr,'(a,t40,a)') 'upper horizon header word',uhwd
      write(lerr,'(a,t40,a)') 'lower horizon header word',lhwd
      write(lerr,'(a,t40,i5)') 'number of input samples',nsamp_in
      write(lerr,'(a,t40,i5)') 'nfft',nfft               
      write(lerr,'(a,t40,i5)') 'number of output samples',nsamp_out
      write(lerr,'(a,t40,i5)') 'number of traces',ntrace
      write(lerr,'(a,t40,i5)') 'number of recs (records)',nrec 
      write(lerr,'(a,t40,f12.6)') 'input data sample interval (msec)', 
     1                         dtmsec
      write(lerr,'(a,t40,f12.6)') 'input trace origin (msec)',
     1                       tstart_orig
      write(lerr,'(a,t40,f12.6)') 'output trace length (msec)',tmax
      write(lerr,'(a,t40,f12.6)') 'output datum, tdatum (msec)',tdatum
      write(lerr,'(a,t40,f12.6)') 'window above upper horizon (msec)',
     1                             uw                 
      write(lerr,'(a,t40,f12.6)') 'window below lower horizon (msec)',
     1                             lw                 
      write(lerr,'(a,t40,f12.6)') 'taper (msec)',ttaper             
      write(lerr,'(a,t40,i5)') 'nbytes_out',nbytes_out           
c
      write(lerr,'(a,t40,f12.6)') 'znull value on trace headers',
     1                             znull
      write(lerr,'(a,t40,l5)') 'flatten on upper horizon?',uflat
      write(lerr,'(a,t40,l5)') 'flatten on lower horizon?',lflat
      write(lerr,'(a,t40,l5)') 'flatten on formation midpoint?',mflat
      write(lerr,'(a,t40,l5)') 'verbose output?',verbose     
      write(lerr,*)' '
c_______________________________________________________________________
c     update the historic rec header. 
c_______________________________________________________________________
      call savhlh(sheader,lbytes,lbyout)
c_______________________________________________________________________
c     update rec header to reflect actual amount of data output
c_______________________________________________________________________
      call savew(sheader,'TmMsFS',0.,LINEHEADER)
      call savew(sheader,'NumSmp',nsamp_out,LINEHEADER)
      call savew(sheader,'NumRec',endrec-startrec+1,LINEHEADER)
      call savew(sheader,'NumTrc',endtrace-starttrace+1,LINEHEADER)
c_____________________________________________________________   
c     write out the rec header.
c_____________________________________________________________   
      call wrtape(luout,sheader,lbyout)
c_____________________________________________________________   
c     calculate interpolation coefficients.
c_____________________________________________________________   
      call getw6(w,ndiv)
c_____________________________________________________________   
c     process the data.
c_____________________________________________________________   
      if(uhwd .eq. lhwd) then
c_____________________________________________________________   
c        upper and lower horizon are identical. flatten the horizon.
c_____________________________________________________________   
         call fhoriz( uin, uout, twgt, w, ndiv, adiv,
     1        luin, luout, lbyout, uhwd,uw,lw,ttaper,
     2            ler,lerr,nsamp_in,nsamp_out,nfft,tdatum,
     3            dtmsec,ITRWRD,verbose,ntrace,nbytes_out,
     4            startrec,endrec,starttrace,endtrace,
     5            thoriz,tstart_orig,wrxsd,luxsd,znull,
     6            nhtLive,nhtVal,deadVal)
      else
c_____________________________________________________________   
c        flatten the formation.
c_____________________________________________________________   
         call fslab(uin,uout,twgt,w,ndiv,adiv,
     1           luin,luout,lbyout,uhwd,lhwd,uw,lw,ttaper,
     2           ler,lerr,nsamp_in,nsamp_out,nfft,tdatum,
     3           dtmsec,ITRWRD,verbose,ntrace,nbytes_out,
     4           startrec,endrec,starttrace,endtrace,
     5           nttaper,uflat,lflat,impatient,
     6           thoriz,tstart_orig,wrxsd,luxsd,znull,
     6            nhtLive,nhtVal,deadVal)
      endif
c_____________________________________________________________   
c     write out timing statistics.
c_____________________________________________________________   
      call timend(cputim(20),vtot,v2,waltim(20),wtot,w2)
      write(lerr,'(A30,2A15,/)') 'routine','cpu time','wall time'
      write(lerr,'(A30,2f15.3)')
     5         'total',cputim(20),waltim(20)
c_____________________________________________________________   
c     close data files
c_____________________________________________________________   
      call lbclos(luin)
      call lbclos(luout)
      if(wrxsd) close(luxsd)

      write(ler,*)'normal completetion. routine STRATSLAB'           
      write(lerr,*)'normal completetion. routine STRATSLAB'           
      close(lerr)
      call exit(0)
c
      end
      subroutine help(ler)

      write(ler,*)
     1'**************************************************************'
      write(ler,*)'window out a STRATigraphic SLAB of seismic data'     
      write(ler,*)'using picks stored in the seismic trace headers'     
      write(ler,*)
     1'**************************************************************'
      write(ler,*)'execute stratslab by typing stratslab and list of'
     1                   //' program parameters.'
      write(ler,*)
     1'note that each parameter is proceeded by -a where "a" is '
      write(ler,*)
     1'a character(s) correlation to some parameter.'
      write(ler,*)
     1'users enter the following parameters, or use the default values'
      write(ler,*)' '
        write(ler,*)
     1' -N [file_in]  (stdin)      : input seismic file name'          
      write(ler,*)
     1' -O [file_out] (stdout)     : output slab file name'                   
      write(ler,*)
     1' -X [file_xsd] (optional)   : echo pick files in xsd format'           
      write(ler,*)
     1' -startrec (1)       : first rec to be processed'
      write(ler,*)
     1' -endrec   (last)    : last rec to be processed'
      write(ler,*)
     1' -starttrace (1)      : first trace to be processed'
      write(ler,*)
     1' -endtrace   (last)   : last trace to be processed'
      write(LER,*)
     1' -uh[uhwd]            : upper horizon header word' 
     2                             //' (default=Horz01)'
      write(LER,*)
     1' -lh[lhwd]            : lower horizon header word' 
     2                             //' (default=uhwd)'
      write(LER,*)
     1' -uw[uw]     (-48 ms) : beginning of window of data to be'
     2       //' extracted, measured relative to the upper horizon' 
      write(LER,*)
     1' -lw[lw]     (+48 ms) : end of window of data to be'
     2       //' extracted, measured relative to the lower horizon' 
      write(LER,*)
     1' -tmax[tmax] (200 ms) : length of output data set in ms'  
     2        //' (needs to be greater than formation thickness'
     3        //' plus tapers)'
      write(LER,*)
     1' -tmax[tmax] (200 ms) : length of output data set in ms'  
      write(LER,*)
     1' -tdatum[tdatum] (tmax/2): output flattened datum in ms'  
      write(LER,*)
     1' -ttaper[ttaper] (0 ms): taper upper and lower ends of the'
     2                     //' data window'
      write(LER,*)
     1' -td[tdatum]          : output datum time (ms or micros)'
        write(ler,*)
     1' -U                   : if present, flatten on upper horizon'
     2       //' (Default - flatten on midpoint)'
        write(ler,*)
     1' -L                   : if present, flatten on lower horizon'
     2       //' (Default - flatten on midpoint)'
        write(ler,*)
     1' -V                   : if present, verbose printout'
        write(ler,*)
     1' -VT                  : if present,  print progress to stderr'
        write(ler,*)
     1'                        during execution'
       write(ler,*)' '
       write(ler,*)'usage:'
       write(ler,*)
       write(ler,*)'stratslab -N[file_in] -O[file_out] -X[file_xsd]'
       write(ler,*)'          -uh[] -lh[] -uw[] -lw[] '
       write(ler,*)'          -tmax[] -tdatum[] -ttaper[]'
       write(ler,*)'          -startrec[] -endrec[] '
       write(ler,*)'          -starttrace[] -endtrace[] '
       write(ler,*)'          -[U,L,VT,V]'
       write(ler,*)' '
       write(ler,*)

      return
      end
