C***********************************************************************
C                 copyright 2004, Amoco Production Company             *
C                             All Rights Reserved                      *
C                     an affiliate of BP America Inc.                  *
C***********************************************************************
C  ROUTINE:       VTTD                                                 *
C  ROUTINE TYPE:  MAIN                                                 *
C  PURPOSE:  CONVERT A TIME SECTION TO A DEPTH SECTION                 *
C                            -OR-                                      *
C          - CONVERT A DEPTH SECTION TO A TIME SECTION                 *
C  AUTHOR:   Mary Ann Thornton                  ORIGIN DATE: 90/06/11  *
C  ROUTINES CALLED:                                                    *
C      CMDLIN          - GET COMMAND LINE ARGUMENTS                    *
C      LBOPEN          - OPEN TAPE                                     *
C      OPENPR          - OPEN PRINT FILE                               *
C      GAMOCO          - PRINT TORCH AND OVAL                          *
C      RTAPE           - READ TAPE                                     *
C      HLHPRT          - PRINT & UPDATE HISTORICAL LINEHEADER          *
C      WRTAPE          - WRITE TAPE                                    *
C      CCUINT          - CUBIC INTERPOLATOR                            *
C      LBCLOS          - CLOSE TAPE                                    *
C      help            - give command line help                        *
C  FILES:                                                              *
C      LLIST  sequential output printout file                          *
C       +------------------------------------------------------+       *
C       |             OTHER DOCUMENTATION DETAILS              |       *
C       +------------------------------------------------------+       *
c  GENERAL DESCRIPTION:  Original program read mxc file and built a    *
C               Matrix, this version replaced mxc with velocity tape   *
C  REVISED BY:  MARY ANN THORNTON             REVISION DATE: 89/02/20  *
C            -  ADD MAXIMUM TIME AND DEPTH PARAMETERS AND MAKE KSAMP   *
C            -  CORRESPOND                                             *
C  REVISED BY:  MARY ANN THORNTON             REVISION DATE: 89/02/28  *
C            -  MAKE THE INPUT & OUTPUT TABLES THE SAME (LONGEST)      *
C  REVISED BY:  MARY ANN THORNTON             REVISION DATE: 89/03/03  *
C            -  CORRECT INPUT & OUTPUT TABLES ALL WRONG BEFORE         *
C  REVISED BY:  MARY ANN THORNTON             REVISION DATE: 89/03/23  *
C            -  CHECK FOR INVALID DTMS VALUE IN HEADER WORD 13         *
C  REVISED BY:  MARY ANN THORNTON             REVISION DATE: 89/03/23  *
C            -  Moved code to Sun for Maintenance from the Sun         *
C            -  Corrected error found in reading vel.tape lineheader   *
C            -  Changed CALL MOVES to CALL VCLR AND CALL VMOV          *
C  REVISED BY:  CURTIS KRUSE                  REVISION DATE: 91/??/??  *
C               Added 2 checks to make sure not getting samples below  *
C               end of tape                                            *
C  REVISED BY:  Mary Ann Thornton             REVISION DATE: 91/??/??  *
C               Added 2 checks from Curtis's change above to the code  *
C  REVISED BY:  Mary Ann Thornton  V3.0       REVISION DATE: 91/??/??  *
C               removed DX parameter (not necessary) replaced the cubic*
C               interpolator with the linear interpolator.             *
C  REVISED BY:  Mary Ann Thornton  V3.1       REVISION DATE: 92/03/26  *
C               Call openpr with full program name for OS 6.1          *
C  REVISED BY:  Mary Ann Thornton  V3.2       REVISION DATE: 92/05/12  *
C               Rename topen to cmdlin for new sun compiler            *
C  REVISED BY:  Gary Murphy        V3.3       REVISION DATE: 92/06/12  *
C               Added verbos print statements for negative velocities  *
C  REVISED BY:  Mary Ann Thornton  V3.4       REVISION DATE: 92/07/30  *
C               Allowed the program to run when the velocity tape does
C               not match the seismic data precisely. The last velocity
C               trace or the last velocity sample will be used when the
C               velocity tape is not as big in size as the seismic data
C               Corrected error in trace numbers when doing the depth 
C               to time conversion.  Cleaned up the code and tried to
C               make the error messages more meaningful to a user.
C  REVISED BY:  Mary Ann Thornton  V3.5       REVISION DATE: 92/08/25  
C               Added statements to rewind the velocity tape in between
C               input records, so vttd would work on multi-record input
C  REVISED BY:  Mary Ann Thornton  V3.6       REVISION DATE: 92/11/02  
C               Copied vttd from ~mbs to ~usp so vttdusp could be
C               eliminated - recompiled to see if everything works as
C               a usp group member - code will be in mbs and usp for now
C  REVISED BY:  Mary Ann Thornton  V3.7       REVISION DATE: 92/11/09  
C               Added a change to the way the table is calculated for
C               the time to depth conversion to make it more accurate
C               The values will be the average slowness over the 
C               boundaries at the base of the layers.
C               Also increased the maximum sample limit and resampled
C               data on input to 1ms data when doing time to depth
C               conversion. 
C  REVISED BY:  Mary Ann Thornton  V3.8       REVISION DATE: 93/02/25  
C               Remove the call to lbopen after the call to rwd.  The
C               tape does not need to be reopened.  Each time lbopen was
C               called a new lu number was assigned, and after 100
C               calls, the program bombs with too many files open.
C  REVISED BY:  Dan Whitmore       V3.9       REVISION DATE: 93/03/12 
C               Added option to allow input velocity tape to be in time.
C  REVISED BY:  Gary Murphy        V4.0       REVISION DATE: 93/03/12
c               added 3d option to allow multirecord input without
c               rewinding the velocity tape.  added lbclos call after
c               lbopen in velocity tape handling.
c               added velocity tape input option (vt).
C  REVISED BY:  Mary Ann Thornton  V4.1       REVISION DATE: 93/03/23  
C               removed a call to vmov and replaced with a do loop
C               for moving data in at the end of traces when it is a	
C               velocity tape. Also increased line header size.
C  REVISED BY:  Mary Ann Thornton  V4.2       REVISION DATE: 93/04/02  
C               Changed comments that get printed to describe how to use
C               the program and added check for dt on command line.
C               dt must be entered on command line.
C  REVISED BY:  Mary Ann Thornton  V4.3       REVISION DATE: 93/05/24  
C               Corrected an error in 3d option.  the 'endif' was in the
C               wrong place when checking to see if running in 3d mode
C               When in 3d mode, a call to rtape at the end of each record    
C               loop read a velocity trace off the next record, when this
C               call to rtape was really made for reading the line 
C               header after rewinding the velocity tape before
C               going to the next data record.
C  REVISED BY:  Mary Ann Thornton  V4.4       REVISION DATE: 93/06/10  
C               increased the mxsam to 12000 to allow more space for the
C               work arrays when requesting a large amount of time on
C               output, and changed the error message to be more 
C               informative to a user.
C  REVISED BY:  Gary Murphy        V4.5       REVISION DATE: 93/07/13  
C               Put in crude fix to make sure output velocity tape     
C               never contains zeros.                                  
C  REVISED BY:  Mary Ann Thornton  V4.6       REVISION DATE: 93/07/21  
C               Remove fix of version 4.5.                             
C               Change code when doing depth to time conversion on a
C               seismic dataset to extend the trace with zeroes rather
C               than extend the trace with the last data sample BEFORE
C               the conversion is done.
C               Also, added many more comments to try to make code more
C               understandable, and added checks where there was a 
C               chance for the arrays to go out of bounds.
C               Changed cmdlin to make the comments better, which
C               requires a change be made to vttd.pat to match comments
C               Changed calculation of nzvt:(added +1.0 to calculation)
C  REVISED BY:  Mary Ann Thornton  V4.7       REVISION DATE: 93/08/26  
C               Changed to issue a warning message when the velocity   
C               dataset is shorter or longer than the dataset to be    
C               converted. Also made a change to allow 1 trace records 
C               to be handled correctly whether or not the 3d option is
C               selected.                                              
C  REVISED BY:  Mary Ann Thornton  V4.8       REVISION DATE: 93/10/25  
C               Changed to include the command line arguments in L.header
C  REVISED BY:  Mary Ann Thornton  V4.9       REVISION DATE: 94/01/04  
C               Added dx1000 to the output line header and dz1000 when
C               output is in depth.                                    
C               A new command line argument was added for dx.  If it is
C               missing, pgm will look for dx1000 in input velocity field,
C               If that is missing, too, pgm will look for dx1000 in the
C               input dataset to be converted.  If that is blank, then
C               proceed with nothing put in dx1000 location.  Also if 3d
C               option is requested, ILClIn was placed in line header.
C               The format statements were revised for easier and more
C               informative reading. The dx and dz values among the inputs
C               and the output will be compared, and the user notified
C               of mismatches, but program will continue execution.
C  REVISED BY:  Mary Ann Thornton  V5.0       REVISION DATE: 94/03/22  
C               Switch the calls ccuint and fliint for seismic and velocity
C               input.
C  REVISED BY:  Mary Ann Thornton  V5.1       REVISION DATE: 94/03/24  
C               Remove the limit on traces in; there is no need for it.    
C  REVISED BY:  Mary Ann Thornton  V5.2       REVISION DATE: 94/11/17  
C               Add a -lhdt parameter to force the program to use the  
C               sample rate from the input dataset for -dt
C  REVISED BY:  Paul Garossino     V5.3       REVISION DATE: 96/03/05 
C               Allow velocity tape to be a pipe (IKP ability)
C               He established a symbolic link vttdusp  vttd
C  REVISED BY:  Mary Ann Thornton  V5.4       REVISION DATE: 96/04/22  
C               If 3d, and velocity field has more traces than seismic,
C               read past the last traces in the velocity field record
C               before beginning the next record.
C  REVISED BY:  Gary Murphy        V5.5       REVISION DATE: 97/01/13 
C               Bumped limit for number of samples.
C  REVISED BY:  Jerry Ehlers       V6.0       REVISION DATE: 99/07/28
C               Use sinc interpolation & total rewrite
C  REVISED BY:  Jerry Ehlers       V6.1       REVISION DATE: 00/02/16
C               Add -4d and -qc options
C  REVISED BY:  Jerry Ehlers       V6.1       REVISION DATE: 00/05/19
C               Make running sums double precision
C  Ported to USP by: Paul Garossino  V6.1     Revision Date: Sept 1, 2003
c                    All future support will be from 
c                    USP crew as support for old MBS code is failing.
c                    Created USP style error codes, fixed UnitSc/SmpInt
c                    dependencies, created updated man page and pattern
c                    file.  Got rid of I star 2 variables
c  REVISED BY: Paul G.A. Garossino   V6.1     REVISION DATE: Sept 8,2003
c              fixed hard assignment to word 106,107 assuming I*2 to
c              savew2[] calls.  Former was entering crap in the datastream.
c
c  REVISED BY: Paul G.A. Garossino   V6.1     REVISION DATE: Sept 22,2003
c              if velocity dataset and dataset of interest have coincident
c              dead traces simply put out a dead trace, do not crash as 
c              is the current behaviour.
c
c  REVISED BY: Paul G.A. Garossino   V6.1     REVISION DATE: Sept 30,2003
c              depricated -lhdt.  must now have both -dt and -dz defined
c              on the command line
c
c
c  REVISED BY: Paul G.A. Garossino   V6.1     REVISION DATE: Oct 21,2003
c              depricated the check and resample of the velocity dataset
c              should dz2 be found different than dz.  In this new version
c              -dt and -dz are given on the command line and used.  What
c              is in the dataset DZ1000 slot is looked at, and complaints
c              are registered as warnings if they differ from the command
c              line but the command line sample rates are enforced.
c
c  REVISED BY: Paul G.A. Garossino  USP version  REVISION DATE: Feb 12,2004
c              depricated all uses and references to all previous MBS style
c              header references.  All dt and dz values are controlled on
c              the command line.  Warnings are still given if data is found
c              to differ.  Unit scalars are defined, used and transmitted to
c              the output data.  Made massive changes to the style and variable
c              names in the code to conform with USP practice.  This makes the
c              code significantly easier to read and maintain into the
c              future.  I set up the variables for dynamic memory allocation
c              but did not have time to actually impliment same at this 
c              time.  I will save that for next time in.  Depricated all
c              reference and usage of -dx[] parameter.  There is no longer
c              any attempt to check the spatial relationships of the 
c              input data and velocity files.  It is assumed that the user
c              has that under control prior to execution.
c
c  REVISED BY: Paul G.A. Garossino  USP version  REVISION DATE: Feb 16,2004
c              fixed a bug in the cmdlin[] call where the name variable
c              was not passed to the subroutine resulting in gibberish 
c              reported to the user when a comand line entry was in error.
c
c  REVISED BY: Paul G.A. Garossino  USP version  REVISION DATE: Feb 18,2004
c              fixed another bug I introduced in the recode.  The 4D if logic
c              was hosed as the final endif was mispositioned.  This is now
c              repaired and the 4D option is again working.  Hopefully this is 
c              the last of the detritus from the move from MBS to USP.  Oh, I
c              also fixed a bug that was in the MBS version for a long time
c              where the indexing for the velocity dataset was taken from the
c              incoming seismic dataset.  I am refering to the LinInd,
c              DphInd and more importantly StaCor!!!
C***********************************************************************

      implicit none

#include <localsys.h>
#include <f77/iounit.h>
#include <f77/lhdrsz.h>
#include <f77/sisdef.h>
#include <save_defs.h> 

      integer mxsam, llist, mxxxx, nmax

      parameter (mxsam=32000)
      parameter (llist=27)
      parameter (mxxxx=mxsam+ITRWRD)
      parameter (nmax=8)

c standard USP variables
      
      integer argis, jerr, luin, luout, iform 
      integer lbyout, lbytes, obytes, nrec, ntrc, nsi, nsamp
      integer JJ, KK

      character*4 name
      character*256 ntap, otap
      character*1 parr(66)

      logical verbos

c variables to be used in future dynamic memory allocation

      integer itr(SZLNHD)
      integer itr_v(SZLNHD)

      real tri(mxsam)
      real tri_v(mxsam)
      real w(mxsam,-nmax:nmax), iz(mxsam), zz(4*mxsam), tmp(mxsam)
      real tablz(mxsam), tabl1(mxsam), tabl2(mxsam)

c local variables


      integer pipe, ntr 
      integer ifmt_RecNum, l_RecNum, ln_RecNum
      integer ifmt_TrcNum, l_TrcNum, ln_TrcNum
      integer ifmt_DphInd, l_DphInd, ln_DphInd
      integer ifmt_LinInd, l_LinInd, ln_LinInd
      integer ifmt_StaCor, l_StaCor, ln_StaCor, StaCor, v_StaCor
      integer lpprt, ierr, iflag, itype, init
      integer nsinc, nlin, luvel, lbytes_v
      integer idz
      integer len, nqc, nsampv, nsiv, nrecv, iformv
      integer mrec, ntrv, idtms, itmax, ksin, nzout, nout, ntout
      integer newksin, k, ntrcv, ncount, idi, ili
      integer idiv, iliv, nzvt, ind, ltabl2, i, lword, ll, mmm
      integer dzo_count

      real UnitSc, UnitSc_V, data_units_in, vel_units_in
      real dzo, unitsco
      real dz, dz2, dtms, zmax, tmax, dt, dtin, tm
      real dold, dnew, sold, res, snew 

      real*8 sum
     
      character*256 vtap

      logical threed, fourd, vtin, done

c initialize variables
      
      data name/'VTTD'/
      data parr/
     1     ' ',' ',' ',' ',' ',' ',' ',' ',' ',' ',
     2     'V','E','L','O','C','I','T','Y',' ','T',
     3     'A','P','E',' ','T','I','M','E','/','D',
     4     'E','P','T','H',' ','C','O','N','V','E',
     5     'R','S','I','O','N',' ',' ',' ',' ',' ',
     6     ' ',' ',' ',' ',' ',' ',' ',' ',' ',' ',
     7     ' ',' ',' ',' ',' ',' '/
      data pipe/3/
      data lpprt/0/
      data ierr/0/
      data init/1/

c check to see if user is requesting command line help, if so then
c echo the help subroutine and quit

      if (  argis ('-h') .gt. 0  .or. 
     :      argis ( '-?') .gt. 0 .or. 
     :      argis ( '-help') .gt. 0 ) then
         call help()
         stop
      endif

c parse command line

      call cmdlin( ntap, otap, vtap, iflag, dz, dtms, zmax, tmax,
     :     itype, threed, fourd, vtin, nsinc, nqc, name, 
     :     LER, LERR, verbos )

c open printfile and print the amoco torch and oval

      jerr = 0
      call openpr(llist,LERR,name,jerr)
      if (jerr.ne.0) then
         write(LER,*) 'VTTD: '
         write(LER,*) ' unable to open printout file.'
         write(LER,*) 'FATAL'
         stop
      endif

      nlin=1
      call gamoco( parr, nlin, LERR )

c open -N[] and -O[] seismic datasets

      call getln (luin, ntap, 'r', 0)
      call getln (luout, otap, 'w', 1)
     
c open velocity dataset


      if ( vtap .ne. ' ' ) then
         call lbopen ( luvel, vtap, 'r')
      elseif (threed .or. fourd) then
         call sisfdfit (luvel,pipe)
         write(LERR,*)'vttd assumed to be running inside IKP'
      else
         write(LERR,*)' '
         write(LERR,*)' VTTD: You cannot pipe the velocity data'
         write(LERR,*)'       into this routine unless using the'
         write(LERR,*)'       -3d or -4d option inside XIKP.'
         write(LERR,*)'       No backing up on pipes allowed.'
         write(LERR,*)' FATAL'
         write(LER,*)' '
         write(LER,*)' VTTD: You cannot pipe the velocity data'
         write(LER,*)'       into this routine unless using the'
         write(LER,*)'       -3d or -4d option inside XIKP. '
         write(LER,*)'       No backing upon pipes allowed.'
         write(LER,*)' FATAL'
         stop
      endif

c  check for command line arguments

      if ( (iflag .eq. 0) .and. (dz .eq. 0.0) .and.
     :     (zmax .le. 0.0) .and. (tmax .le. 0.0) ) then

         write(LERR,*)'  all parameters have been defaulted'
         write(LERR,*)'  JOB TERMINATED...................'
         write(LER,*)' '
         write(LER,*)'VTTD: '
         write(LER,*)' You cannot default all command line parameters'
         write(LER,*)' Please supply required control and try again.'
         write(LER,*)'FATAL'
         goto 999

      elseif ( dz .le. 0.0) then

         write(LERR,*)' -dz[] must be > 0.0 - JOB TERMINATED '
         write(LER,*)' '
         write(LER,*)'VTTD: '
         write(LER,*)' -dz[] must be > 0.0 '
         write(LER,*)'FATAL'
         goto 999

      elseif ( dtms .le. 0.0) then

         write(LERR,*)' dt must be > 0.0 - JOB TERMINATED '
         write(LER,*)' '
         write(LER,*)'VTTD:  '
         write(LER,*)' -dt[] must be > 0.0 '
         write(LER,*)'FATAL'
         goto 999

      endif

      if ( nsinc .gt. nmax ) then

         write(LERR,*)' nsn must be <= 8 - JOB TERMINATED '
         write(LER,*)' '
         write(LER,*)'VTTD: '
         write(LER,*)' -nsn[] must be <= 8 '
         write(LER,*)'FATAL'
         goto 999

      endif

      if ( fourd ) then

         if ( threed ) then

            write(LERR,*)
     *         ' Cannot have both -3d and -4d options - JOB TERMINATED '
            write(LER,*)' '
            write(LER,*)'VTTD: '
            write(LER,*)' both -3d and -4d flags present'
            write(LER,*)'FATAL'
            goto 999

         endif

      endif

      iflag=max(1,min(2,iflag))
      itype=max(1,min(2,itype))

c set up pointers to header mnemonic RecNum, TrcNum

      call savelu ( 'RecNum', ifmt_RecNum, l_RecNum, ln_RecNum, 
     :     TRACEHEADER )
      call savelu ( 'TrcNum', ifmt_TrcNum, l_TrcNum, ln_TrcNum, 
     :     TRACEHEADER )
      call savelu ( 'RecNum', ifmt_RecNum, l_RecNum, ln_RecNum, 
     :     TRACEHEADER )
      call savelu ( 'DphInd', ifmt_DphInd, l_DphInd, ln_DphInd, 
     :     TRACEHEADER )
      call savelu ( 'LinInd', ifmt_LinInd, l_LinInd, ln_LinInd, 
     :     TRACEHEADER )
      call savelu ( 'StaCor', ifmt_StaCor, l_StaCor, ln_StaCor, 
     :     TRACEHEADER )

c read the lineheader of the tape to be converted

      lbytes = 0
      call rtape( luin, itr, lbytes )
      if ( lbytes .eq. 0) then
         write(LERR,*)'VTTD: no line header on input dataset',
     :        ntap
         write(LER,*)' '
         write(LER,*)'VTTD: '
         write(LER,*)' no line header on input dataset',ntap
         write(LER,*)'FATAL'
         write(LER,*)' '
         stop
      endif

c load required line header entries

      call saver(itr,'NumSmp',nsamp,LINHED)
      call saver(itr,'NumTrc',ntrc,LINHED)
      call saver(itr,'SmpInt',nsi,LINHED)
      call saver(itr,'NumRec',nrec,LINHED)
      call saver(itr,'Format',iform,LINHED)
      call saver(itr,'UnitSc',UnitSc,LINHED)

      if ( UnitSc .eq. 0.0 ) then

         if ( iflag .eq. 1 ) then

c time data coming in with undefined unit scalar, set to ms
c and warn user

            write(LERR,*)'WARNING: '
            write(LERR,*)' sample unit scaler in LH = ',UnitSc
            write(LERR,*)' will set to .001 (millisec default)'
            write(LER,*)' '
            write(LER,*)'VTTD: '
            write(LER,*)' sample unit scaler in LH = ',UnitSc
            write(LER,*)' will set to .001 (millisec default)'
            write(LER,*)'WARNING: '
            UnitSc = 0.001

         else

c depth data coming in with no unit scalar, set to feet [meters]
c and warn user

            write(LERR,*)'WARNING: '
            write(LERR,*)' sample unit scaler in LH = ',UnitSc
            write(LERR,*)' will set to 1.0 [feet or meter default]'
            write(LER,*)' '
            write(LER,*)'VTTD: '
            write(LER,*)' sample unit scaler in LH = ',UnitSc
            write(LER,*)' will set to 1.0 [feet or meter default]'
            write(LER,*)'WARNING: '
            UnitSc = 1.0

         endif
      endif

      data_units_in = float(nsi) * UnitSc

c echo HLH to printout file

      call hlhprt(itr,lbytes,name,4,LERR)

      if ( nsamp .gt. mxsam ) then
         write(LERR,*)' ERROR*****max # samples allowed is ',mxsam
         write(LER,*)' '
         write(LER,*)' VTTD: you have asked for process '
         write(LER,*)        nsamp,' samples. Maximum allowed is ',mxsam
         write(LER,*)' FATAL'
         goto 999
      endif

c read the velocity tape lineheader

      lbytes_v = 0
      call rtape ( luvel, itr_v, lbytes_v )
      if ( lbytes_v .eq. 0 ) then
         write(LERR,*) ' error reading line header from velocity field'
         write(LERR,*)' FATAL'
         write(LER,*)' '
         write(LER,*)'VTTD: '
         write(LER,*)' error reading line header from velocity field'
         write(LER,*)'FATAL'
         goto 999
      endif

      call saver(itr_v,'NumSmp',nsampv,LINHED)
      call saver(itr_v,'NumTrc',ntrcv,LINHED)
      call saver(itr_v,'SmpInt',nsiv,LINHED)
      call saver(itr_v,'NumRec',nrecv,LINHED)
      call saver(itr_v,'Format',iformv,LINHED)
      call saver(itr_v,'UnitSc',UnitSc_V,LINHED)

      if ( UnitSc_V .eq. 0.0 ) then

         if ( itype .eq. 2 ) then

c control velocity time data coming in with undefined unit scalar, 
c set to ms and warn user

            write(LERR,*)' vel sample unit scaler in LH = ',UnitSc_V
            write(LERR,*)' will set to .001 (millisec default)'
            write(LERR,*)'WARNING: '
            write(LER,*)' '
            write(LER,*)'VTTD: '
            write(LER,*)' vel sample unit scaler in LH = ',UnitSc_V
            write(LER,*)' will set to .001 (millisec default)'
            write(LER,*)'WARNING: '
            UnitSc_V = 0.001

         elseif ( itype .eq. 1 ) then

c control velocity depth data coming in with no unit scalar, set to 
c feet [meters] and warn user

            write(LERR,*)' vel sample unit scaler in LH = ',UnitSc_V
            write(LERR,*)' will set to 1.0 [feet or meter default]'
            write(LERR,*)'WARNING: '
            write(LER,*)' '
            write(LER,*)'VTTD: '
            write(LER,*)' vel sample unit scaler in LH = ',UnitSc_V
            write(LER,*)' will set to 1.0 [feet or meter default]'
            write(LER,*)'WARNING: '
            UnitSc_V = 1.0

         endif
      endif

      vel_units_in = float(nsiv) * UnitSc_V
      dz2 = vel_units_in 

      if ( threed .and. (( nrecv .ne. nrec ) .or. ( ntrcv .ne. ntrc))) 
     :     then

         write(LERR,*)' You have asked for the -3d option, but have',
     *                ' a mismatch between the number of records and',
     *                ' traces per record in your input and ',
     *                ' reference interval velocity datasets'
         write(LERR,*)'FATAL'
         write(LER,*)' '
         write(LER,*)'VTTD:'
         write(LER,*)' You have asked for the -3d option, but have'
         write(LER,*)' a mismatch between the number of records and'
         write(LER,*)' traces per record in your input and '
         write(LER,*)' reference interval velocity datasets'
         write(LER,*)'FATAL'
         write(LER,*)' '
         goto 990

      elseif  ( fourd .and. 
     :        ( (nrecv * ntrcv) .ne. 1 ) .and. 
     :        ( (nrecv*ntrcv) .ne. nrec ) ) then

         write(LERR,*)' You have asked for the -4D option, and have'
         write(LERR,*)' more than 1 velocity trace.  Your total '
         write(LERR,*)' number of velocity traces however is not '
         write(LERR,*)' equivalent to the number of records in the'
         write(LERR,*)' input data'
         write(LER,*)'VTTD:'
         write(LER,*)' You have asked for the -4D option, and have'
         write(LER,*)' more than 1 velocity trace.  Your total '
         write(LER,*)' number of velocity traces however is not '
         write(LER,*)' equivalent to the number of records in the'
         write(LER,*)' input data'
         write(LER,*)'WARNING'

      endif

ccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc
c     check the parameters
c     if single trace records are coming in, let ntr=nrec and output 
c     only 1 record with ntr traces
c
c
c it seems since we in USP are now maintaining this routine that should
c single trace records be coming in, single trace records should go out.  
c If multi-trace records are coming in [i.e. MBS standard] then multi-trace 
c records should go out........what could it hurt .... pgag
ccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc
c     dataset to convert:

      ntr =ntrc
      mrec=nrec

      if ( ntrc .eq. 1 .and. .not. fourd ) then
         ntr =nrec
         mrec=1
c         call savew(itr,'NumTrc',ntr,linhed)
c         call savew(itr,'NumRec',mrec,linhed)
      endif

      if ( iflag .eq. 1 ) then

c time to depth conversion

c here is an algorithm to scale UnitSc and SmpInt appropriately for output
c This code demands that you enter both -dz and -dt on the command line and
c reads them both as real variables.  Assuming someone has put in a floating
c point entry behind the decimal place we will need to back that out

         dzo = dz
         dzo_count = 0
         done = .false.
         unitsco = 1.0

         do while ( .not. done .and. dzo_count .le. 4 )

            if ( amod ( dzo,1.0 ) .gt. 0 ) then

               dzo = dzo * 10.
               unitsco = unitsco / 10.
               dzo_count = dzo_count + 1

            else

               done = .true.

            endif

         enddo

         idz = nint ( dzo )
         call savew ( itr,'SmpInt', idz, LINHED )
         call savew ( itr,'UnitSc', unitsco, LINHED )

      endif

c     dataset containing velocity field:

      if ( ntrcv .gt. 1) then
         ntrv=ntrcv
      else
         ntrv=max( 1, nrecv )
      endif

      if ( fourd ) then  

         ntrv=1

         if ( (ntrcv*nrecv) .eq. 1 .and. ( nrec .gt. 1 ) ) then

            write(LER,*)'VTTD:'
            write(LER,*)' The dataset containing the velocity field'
            write(LER,*)' has only one trace which will be used for'
            write(LER,*)' the entire conversion.'
            write(LER,*)'WARNING'
            write(LERR,*)' ******WARNING******'
            write(LERR,*)' The dataset containing the velocity field'
            write(LERR,*)' has only one trace which will be used for'
            write(LERR,*)' the entire conversion.'

         endif

      elseif ( ( ntrv .lt. ntr ) .and. ( ntrv .eq. 1 ) ) then

         write(LER,*)'VTTD:'
         write(LER,*)' The dataset containing the velocity field'
         write(LER,*)' has only one trace which will be used for'
         write(LER,*)' the entire conversion.'
         write(LER,*)'WARNING'
         write(LERR,*)' ******WARNING******'
         write(LERR,*)' The dataset containing the velocity field'
         write(LERR,*)' has only one trace which will be used for '
         write(LERR,*)' the entire conversion.'

      elseif ( ntrv .lt .ntr ) then

         write(LER,*)'VTTD:'
         write(LER,*)' The dataset containing the velocity field'
         write(LER,*)' contains less traces than the dataset to be'
         write(LER,*)' converted.  The last trace will be used '
         write(LER,*)' multiple times. You may want to check this'
         write(LER,*)' situation out as it is likely not good.'
         write(LER,*)'WARNING'

         write(LERR,*)' ******WARNING******'
         write(LERR,*)' The dataset containing the velocity field'
         write(LERR,*)' contains less traces than the dataset to be'
         write(LERR,*)' converted.  The last trace will be used '
         write(LERR,*)' multiple times. You may want to check this'
         write(LERR,*)' situation out as it is likely not good.'

      elseif ( ntrv .gt. ntr ) then

         write(LER,*)'VTTD:'
         write(LER,*)' ******NOTICE******'
         write(LER,*)' The dataset containing the velocity field '
         write(LER,*)' contains more traces than the dataset to be'
         write(LER,*)' converted.  Only the first ',ntrc,' traces ' 
         write(LER,*)' will be used.  You may want to check this'
         write(LERR,*)' situation out as it is likely not good.' 
         write(LER,*)'WARNING'

         write(LERR,*)' ******NOTICE******'
         write(LERR,*)' The dataset containing the velocity field '
         write(LERR,*)' contains more traces than the dataset to be'
         write(LERR,*)' converted.  Only the first ',ntrc,' traces ' 
         write(LERR,*)' will be used. You may want to check this'
         write(LERR,*)' situation out as it is likely not good.' 

      endif

      if ( nsampv .gt. mxsam ) then

         write(LERR,*)' ERROR*****max # samples allowed is ',mxsam

         write(LER,*)' '
         write(LER,*)'VTTD: '
         write(LER,*)' you have asked to process ',nsampv
         write(LER,*)' samples. Maximum allowed is ',mxsam
         write(LER,*)' FATAL'
         goto 999
      endif
      
      IF ( iflag .eq. 1 ) THEN

ccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc
c     
c     time to depth conversion  (iflag=1)
c     
ccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc

         if (zmax.le.0.0) then
            write(LERR,*)' MAXIMUM DEPTH MUST BE ENTERED'
            write(LER,*)'VTTD:' 
            write(LER,*)' maximum depth must be entered'
            write(LER,*)' fix -z[] entry on command line and try again'
            write(LER,*)'FATAL'
            goto 999
         endif

c if we made it this far then we have a delta time for the input dataset
c either from the command line or from the input dataset.  It is now 
c supposedly in units of seconds.

         idtms = dtms
         dt = dtms / 1000.0
         dtin = dt

c if user asks to use more time in than we have, we will use
c only as much as we have. Hmmm does not look like the MBS 
c crowd paid attention to sample one being from time zero.  I will
c set sample one to be time zero from here on out...pgag    

         if (tmax .le. 0.0) tmax = dtms * ( nsamp - 1)
         tm = dtms * ( nsamp - 1 )
         tmax = min( tm, tmax )
         itmax = tmax
         ksin = itmax / idtms + 1

c Remember, there was a check done at the start of the program
c to guard against dz being <= 0.0 at this point so that division
c by dz should be kosher here ...pgag

         nzout = zmax / dz + 1
         nout = nzout
         idz = dz

c load output line header entries for number of samples, sample
c interval and unit scalar.  Since it is depth out, unit scalar
c is set to feet or meters.

         call savew(itr,'NumSmp',nzout,linhed)
         call savew(itr,'SmpInt',idz,linhed)
         call savew(itr,'UnitSc',1.0,linhed)

c check for dt mismatches between velocity  & input
c Also watch for mismatches between
c what is specified on the command line and what is specified in the
c data line headers

         if ( itype .eq. 2 ) then

            if ( data_units_in .ne. vel_units_in ) then
               write(LERR,*)' The time spacing of the input does'
               write(LERR,*)' not match that of the velocity field.'
               write(LERR,*)'WARNING' 
               write(LER,*)'  ' 
               write(LER,*)'VTTD:' 
               write(LER,*)' The time spacing of the input does'
               write(LER,*)' not match that of the velocity field.'
               write(LER,*)'WARNING' 
            endif

             if ( data_units_in .ne. dt ) then
               write(LERR,*)' The time spacing of the input does'
               write(LERR,*)' not match -dt[] from the command line'
               write(LERR,*)'WARNING' 
               write(LER,*)'  ' 
               write(LER,*)'VTTD:' 
               write(LER,*)' The time spacing of the input does'
               write(LER,*)' not match -dt[] from the command line'
               write(LER,*)'WARNING' 
            endif

             if ( vel_units_in .ne. dt ) then
               write(LERR,*)' The velocity time spacing does'
               write(LERR,*)' not match -dt[] from the command line'
               write(LERR,*)'WARNING' 
               write(LER,*)'  ' 
               write(LER,*)'VTTD:' 
               write(LER,*)' The velocity time spacing does'
               write(LER,*)' not match -dt[] from the command line'
               write(LER,*)'WARNING' 
            endif

         elseif ( itype .eq. 1 ) then

             if ( vel_units_in .ne. dz ) then
               write(LERR,*)' The velocity depth spacing does'
               write(LERR,*)' not match -dz[] from the command line'
               write(LERR,*)'WARNING' 
               write(LER,*)'  ' 
               write(LER,*)'VTTD:' 
               write(LER,*)' The velocity depth spacing does'
               write(LER,*)' not match -dz[] from the command line'
               write(LER,*)'WARNING' 
            endif

         endif

      ELSE

ccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc
c     
c     depth to time conversion  (iflag=2)
c     
ccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc

        if ( tmax .le. 0.0 ) then

c we need to know the max time to convert for as most depth datasets 
c will convert to massive time registries we make it manditory for 
c the user to think about this and enter something sensible on the 
c command line

            write(LERR,*)' MAXIMUM TIME MUST BE ENTERED'
            write(LER,*)'VTTD:' 
            write(LER,*)' maximum time must be entered'
            write(LER,*)' fix -t[] entry on command line and try again'
            write(LER,*)'FATAL'
            goto 999
         endif

         itmax = tmax
         dt = dtms / 1000.0
         dtin = dt
         idtms = dtms

c dtms has been calculated in milliseconds so hardwire UnitSc to be 0.001
c and write out both sample interval and unit scalar.

         unitsco = 0.001
         call savew(itr,'SmpInt',idtms,linhed)
         call savew(itr,'UnitSc',unitsco,linhed)

         if ( zmax .le. 0.0 ) zmax = dz * ( nsamp - 1 )

c dz cannot be zero at this point so no worries about the division

         ksin = 1.0 + zmax / dz

c        if user asks to use more samples in than we have, 
c        limit to the number of samples we have.

         if ( ksin .gt. nsamp ) ksin = nsamp

         ntout = tmax/dtms+1
         nout = ntout

         call savew(itr,'NumSmp',ntout,linhed)


c check for dz mismatches between velocity (in depth) & input
c if the input is also in depth.  Also watch for mismatches between
c what is specified on the command line and what is specified in the
c data line headers

         if ( itype.le.1 ) then

            if ( data_units_in .ne. vel_units_in ) then

               write(LERR,*)' The depth spacing of the input does'
               write(LERR,*)' not match that of the velocity field.'
               write(LERR,*)'WARNING' 
               write(LER,*)'  ' 
               write(LER,*)'VTTD:' 
               write(LER,*)' The depth spacing of the input does'
               write(LER,*)' not match that of the velocity field.'
               write(LER,*)'WARNING' 
            endif

             if ( data_units_in .ne. dz ) then
               write(LERR,*)' The depth spacing of the input does'
               write(LERR,*)' not match -dz[] from the command line'
               write(LERR,*)'WARNING' 
               write(LER,*)'  ' 
               write(LER,*)'VTTD:' 
               write(LER,*)' The depth spacing of the input does'
               write(LER,*)' not match -dz[] from the command line'
               write(LER,*)'WARNING' 
            endif

             if ( vel_units_in .ne. dz ) then
               write(LERR,*)' The velocity depth spacing does'
               write(LERR,*)' not match -dz[] from the command line'
               write(LERR,*)'WARNING' 
               write(LER,*)'  ' 
               write(LER,*)'VTTD:' 
               write(LER,*)' The velocity depth spacing does'
               write(LER,*)' not match -dz[] from the command line'
               write(LER,*)'WARNING' 
            endif

         elseif ( itype .eq. 2 ) then

             if ( vel_units_in .ne. dt ) then
               write(LERR,*)' The velocity time  spacing does'
               write(LERR,*)' not match -dt[] from the command line'
               write(LERR,*)'WARNING' 
               write(LER,*)'  ' 
               write(LER,*)'VTTD:' 
               write(LER,*)' The velocity time spacing does'
               write(LER,*)' not match -dt[] from the command line'
               write(LER,*)'WARNING' 
            endif

         endif

      endif
c
c     put command line into the historical part of line header
c
      call savhlh( itr, lbytes, lbyout )

c write output line header

      call wrtape( luout, itr, lbyout )

c echo all pertinent program control to printout file

      call verbal ( ntap, otap, vtap, iflag, dz, dtms, zmax, tmax, 
     :     itype, threed, fourd, vtin, nsinc, nqc, nsamp, nsi, ntrc, 
     :     nrec, UnitSc, nsampv, nsiv, ntrcv, nrecv, UnitSC_V, 
     :     mrec, ntr, verbos )

c start processing dataset                                       
      
c build input and output time tables for resampling input data
      
c     time to depth conversion (iflag=1)
c     depth to time conversion (iflag=2)
c     determine amount of oversampling desired
c     build tables for interpolation
      
      if (iflag.eq.1) then

c time to depth conversion

         dold=dt

         if (vtin) then

            newksin=ksin

         else

c going to oversample input but do not let the sample rate drop below
c 1 millisecond

            dt=min(0.001,dt/2.0)
            newksin=tmax/(1000.0*dt)

         endif

         dnew=dt

         do k=1,mxsam
            tabl2(k) = (k-1)*dz
            tablz(k) = (k-1)*dz
         enddo

      else

c depth to time conversion

         dold=dz

         if (vtin) then

            newksin=ksin

         else

            dz=dz/2.0
            newksin=1.0+zmax/dz

         endif

         dnew=dz

c these tables are used in the interpolation routines.  tabl2 is the
c sample index required on output, tablz is the depth sample index

         do k=1,mxsam

            tabl2(k) = (k-1)*dt
            tablz(k) = (k-1)*dz

         enddo

      endif
     
c check if arrays long enough to handle the resample requested

      if ( newksin .gt. mxsam ) then

         write(LERR,*) 'too many samples required for oversampling'
         write(LERR,*) newksin,' >',mxsam
         write(LERR,*) 'FATAL'
         write(LER,*) ' '
         write(LER,*) 'VTTD: '
         write(LER,*) ' too many samples required for oversampling'
         write(LER,*) newksin,' >',mxsam
         write(LER,*) ' contact USP shop for program enhancement'
         write(LER,*) 'FATAL'

         goto 990

      endif

c LOOP OVER INPUT RECORDS

      ncount=0

      DO JJ = 1, mrec

c LOOP OVER INPUT TRACES

         DO KK = 1, ntr

c read next input trace.

            lbytes=0
            call rtape( luin, itr, lbytes)
            if (lbytes.eq.0) then
               write(LERR,*)'Premature EOF on input at:'
               write(LERR,*)'sequential  rec= ',JJ,'  trace= ',KK
               go to 999
            endif

            call vmov ( itr(ITHWP1), 1, tri, 1, nsamp )

            call saver2(itr, ifmt_DphInd, l_DphInd, ln_DphInd, idi, 
     :           TRACEHEADER)
            call saver2(itr, ifmt_LinInd, l_LinInd, ln_LinInd, ili, 
     :           TRACEHEADER)
            call saver2(itr, ifmt_StaCor, l_StaCor, ln_StaCor, StaCor, 
     :           TRACEHEADER)

c read next velocity trace

c if 4d: turn off reading anymore if only 1 trace (ntrv=0)

            if ( KK .le. ntrv .and. JJ .le. nrecv ) then

               if ( fourd .and. ntrc*nrecv .eq. 1 ) ntrv=0

               lbytes_v = 0
               call rtape( luvel, itr_v, lbytes_v )
               if (lbytes_v.eq.0) then
                  write(LERR,*)'Premature EOF on velocity dataset at:'
                  write(LERR,*)'sequential  rec= ',JJ,'  trace= ',KK
                  go to 999
               endif

               call vmov ( itr_v(ITHWP1), 1, tri_v, 1, nsampv )

c            endif

               call saver2(itr_v, ifmt_DphInd, l_DphInd, ln_DphInd, idiv
     :              , TRACEHEADER)
               call saver2(itr_v, ifmt_LinInd, l_LinInd, ln_LinInd, 
     :              iliv, TRACEHEADER)
               call saver2(itr_v, ifmt_StaCor, l_StaCor, ln_StaCor, 
     :              v_StaCor, TRACEHEADER)

c if velocity trace is flagged dead, check to make sure incoming data 
c trace is also dead.  If not then out you go, if so then output a dead
c trace and keep going.  Of course if this is a fourd pass then all bets
c are off and out you go.....pgag

               if ( v_StaCor .eq. 30000 ) then

                  if ( StaCor .ne. 30000 .or. fourd ) then
                  
c you have a live trace with no live velocity data to support it....goodbye.
c or you have a dead velocity trace in a fourd app .... goodbye again.

                     write(LERR,*)' you have a dead velocity'
                     write(LERR,*)' trace at sequential record ',JJ
                     write(LERR,*)' sequential trace ',KK
                     write(LERR,*)'FATAL'
                     write(LER,*)' '
                     write(LER,*)'VTTD:'
                     write(LER,*)' you have a dead velocity'
                     write(LER,*)' trace at sequential record ',JJ
                     write(LER,*)' sequential trace ',KK
                     write(LER,*)'FATAL'
                     write(LER,*)' '
                     goto 999
                  else
                     goto 10
                  endif

               endif

               if (itype.ge.2) then
c     
c input velocity is in time, convert input velocity to depth
c     
                  sum=0.0

                  do k=1,nsampv
                     tabl1(k)=sum
                     sum=sum+tri_v(k)*dtin/2.0
                  enddo 

                  nzvt = tabl1(nsampv)/dz + 1.0
               
                  if (nzvt.gt.mxsam) then

                     write(LERR,*) 'too many samples required for '
                     write(LERR,*) 'oversampling velocity '
                     write(LERR,*)  nzvt,' >',mxsam
                     write(LERR,*) 'FATAL'
                     write(LER,*) ' '
                     write(LER,*) 'VTTD: '
                     write(LER,*) ' too many samples required for '
                     write(LER,*) ' oversampling velocity '
                     write(LER,*)  nzvt,' >',mxsam
                     write(LER,*) ' contact USP shop for upgrade'
                     write(LER,*) 'FATAL'
                     goto 990
                     
                  endif
     
                  call fliint(tabl1,tri_v,nsampv,tablz,tmp,nzvt,
     *                 iz,zz,init)
                  call vmov(tmp,1,tri_v,1,nzvt)
     
c extended the velocity
     
                  do k=nzvt+1,mxsam
                     tri_v(k) = tri_v(nzvt)
                  enddo

               else
     
c interpolate velocity to oversample delta, extended last sample


c OK, this is obscure, dz2 was the delta Z from the velocity DZ1000
c entry that we no longer use.  Now we always double up on the sampling
c of the depth dataset and need to double up on the sampling of  the
c associated velocity curve.  Soooooo since the dz here is really the 
c command line dz / 2  here we go:

                  if ( dz2 .ne. dz ) then
                     
                     do k=1,nsampv
                        tabl1(k)=(k-1)*dz2
                     enddo

                     nzvt = tabl1(nsampv)/dz + 1.0

                     if (nzvt.gt.mxsam) then

                        write(LERR,*) 'too many samples required for'
                        write(LERR,*) 'oversampling velocity '
                        write(LERR,*)  nzvt,' >',mxsam
                        write(LERR,*) 'FATAL'
                        write(LER,*) ' '
                        write(LER,*) 'VTTD: '
                        write(LER,*) ' too many samples required for'
                        write(LER,*) ' oversampling velocity '
                        write(LER,*) ' contact USP shop for upgrade'
                        write(LER,*) 'FATAL'
                        goto 990
                     endif

c use linear interpolation to flush out the doubly sampled
c velocity curve

                     call fliint(tabl1,tri_v,nsampv,tablz,tmp,nzvt,
     *                    iz,zz,init)
                     call vmov(tmp,1,tri_v,1,nzvt)
                     
                  else
                  
                     nzvt = nsampv

                  endif
c flush out to the mxsam with the last interval velocity entry

                  do k=nzvt+1,mxsam
                     tri_v(k) = tri_v(nzvt)
                  enddo

               endif

c build input table (tabl1) for converting input
c by calculating dz's in terms of dt's
c or by calculating dt's in terms of dz's

               k=1
               sum=0.0
               tabl1(k) = sum
               sold = 1.0 / tri_v(1)
               
               if ( iflag .eq. 1 ) then

c doing time to depth conversion so we need to determine
c the z at any sample of the resampled time input for this trace

                  do while( tabl1(k) .lt. zmax .and. k .lt. mxsam )

                     k=k+1
                     res=(tabl1(k-1)+0.5*dt/sold)/dz+1.0
                     ind=res
                     res=res-ind
                  
                     if (ind.ge.mxsam) then

                        write(LERR,*) 'Check Point 4: ind',ind,' >',
     *                       mxsam
                        write(LERR,*) '   dt,dz=',dt,dz
                        goto 990
                        
                     endif
     
c                    check divisor for zero before doing the divide
     
                     if (tri_v(ind).le.0.0.or.tri_v(ind+1).le.0.0) then

                        write(LER,*)'VTTD:'
                        write(LER,*)' Unflagged dead trace? '
                        write(LER,*)' velocity <= zero on trace ',KK
                        write(LER,*)' sample',ind,' record ',JJ
                        write(LER,*)'FATAL'
                        write(LERR,*) 'velocity <= zero on trace ',KK,
     *                       ' sample',ind,' record ',JJ
                        write(LERR,*) 'Job terminated'
                        goto 999

                     endif

                     snew=(1-res)/tri_v(ind)+res/tri_v(ind+1)
                     sum=sum+0.5*dt*(1.0/sold+1.0/snew)/2.0
                     sold=snew
                     tabl1(k)=sum

                  enddo

                  len=k
                  ltabl2=1.0+tabl1(len)/dz

               else

c doing depth to time conversion so we need to determine the time
c for any given sample on the input
                  
                  do while(tabl1(k).lt.tmax/1000.0.and.k.lt.mxsam)

                     k=k+1

c check divisor for 0 value before dividing

                     if (tri_v(k).le.0.0) then
                        write(LER,*)'VTTD:'
                        write(LER,*)' velocity < zero on trace ',KK
                        write(LER,*)' sample',k,' record ',JJ
                        write(LER,*)'FATAL'
                        write(LERR,*) 'velocity < zero on trace ',KK,
     *                       ' sample ',k,' record ',JJ
                        write(LERR,*) 'job terminated'
                        goto 999
                     endif
                     snew=1.0/tri_v(k)
                     sum=sum+2.0*dz*(sold+snew)/2.0
                     sold=snew
                     tabl1(k)=sum
                  enddo
                  len = k
                  ltabl2 = 1.0+tabl1(len)/dt
               endif

               if (ltabl2.gt.mxsam) then

                  write(LERR,*) 'too many samples required for'
                  write(LERR,*) 'oversampling table ', ltabl2,' >',mxsam
                  write(LERR,*) 'This could be caused by zeroes in'
                  write(LERR,*) 'your velocity data.  You might '
                  write(LERR,*) 'check that first.'
                  write(LERR,*) 'FATAL'
                  write(LER,*) ' '
                  write(LER,*) 'VTTD: '
                  write(LER,*) ' too many samples required for'
                  write(LER,*) ' oversampling table ', ltabl2,' >',mxsam
                  write(LER,*) ' contact USP shop for upgrade'
                  write(LER,*) 'This could be caused by zeroes in'
                  write(LER,*) 'your velocity data.  You might '
                  write(LER,*) 'check that first.'
                  write(LER,*) 'FATAL'
                  goto 990
               endif
            endif
c.......................................................................
c  
c           Process Data
c
c           1) oversample input trace with sinc function if seismic
c           2) extend trace to "len" samples
c           3) resample with liner for non-seismic & cubic for seismic
c     
c           each output trace is different in length depending upon 
c           the velocities at that x location.
c.......................................................................

            if (nqc.gt.0) then
               if (mod(ncount,nqc).eq.0) then
                  write(LERR,8200) KK,ili,idi,iliv,idiv
 8200             format(' QC: trc=',I5,'   Seismic:',2I8,
     *                 '   Velocity', 2I8)
               endif
               ncount=ncount+1
            endif
c
            if (vtin) then
               call vmov(tri,1,tmp,1,ksin)
               do i=ksin+1,len
                  tmp(i) = tmp(ksin)
               enddo
               call fliint(tabl1,tmp,len,tabl2,tri,ltabl2,iz,zz,init)
            else
               call oversamp(tri,ksin,dold,tmp,newksin,dnew,nsinc,
     *              iz,zz,w,.false.)
               do i=newksin+1,len
                  tmp(i) = 0.0
               enddo
               call ccuint(tabl1,tmp,len,tabl2,tri,ltabl2,iz,zz,init)
            endif
c.......................................................................
c   
c           Write Output
c
c           If trace is shorter than nzout, extend it to nzout.  
c           If converted trace is seismic data, zero extend it
c           else extend the last velocity value
c.......................................................................
            if (ltabl2.lt.nout) then
               lword = nout-ltabl2
               if (vtin) then
                  do ll = 1,lword
                     tri(ltabl2+ll) = tri(ltabl2)
                  enddo
               else
                  call vclr(tri(ltabl2+1),1,lword)
               endif
            endif

 10         continue

            call savew2(itr, ifmt_RecNum, l_RecNum, ln_RecNum, JJ, 
     :           TRACEHEADER)
            call savew2(itr, ifmt_TrcNum, l_TrcNum, ln_TrcNum, KK, 
     :           TRACEHEADER)
            call savew2(itr, ifmt_StaCor, l_StaCor, ln_StaCor, StaCor, 
     :           TRACEHEADER)
         
            call vmov ( tri(1),1,itr(ITHWP1),1,nout)

            obytes = nout*szsmpd + sztrhd

            call wrtape( luout, itr, obytes )
            if (obytes.eq.0) then
               write(LERR,*)' error writing output file'
               write(LERR,*)'FATAL'
               write(LER,*)' '
               write(LER,*)'VTTD: '
               write(LER,*)' error writing output file'
               write(LER,*)'FATAL'
               goto 999
            endif

c end of trace loop

         ENDDO

c.......................................................................
c  
c        Reposition Velocity Input
c
c     
c        if 2d: rewind velocity input if more than one seismic record
c        if 3d: position velocity to beginning of next record
c        if 4d: simply continue on
c.......................................................................

         if (threed) then

            do mmm=ntr+1,ntrv
               lbytes_v= 0
               call rtape(luvel,itr_v,lbytes_v)

c hmmmm I am not sure what this check does.  If lbytes_v is zero that
c should be bad.  Watch this closely in the debugger....pgag

               if (lbytes_v.eq.0) goto 200
            enddo

 200        continue

         elseif (.not.fourd) then

            call rwd(luvel)

            lbytes_v = 0
            call rtape(luvel,itr_v,lbytes_v)
            if (lbytes_v.eq.0) then
               write(LERR,*)' '
               write(LERR,*)' error reading line header '
               write(LERR,*)' from velocity field after'
               write(LERR,*)' rewind'
               write(LERR,*)'FATAL'
               write(LER,*)' '
               write(LER,*)'VTTD: '
               write(LER,*)' error reading line header '
               write(LER,*)' from velocity field after'
               write(LER,*)' rewind'
               write(LER,*)'FATAL'
               goto 999
            endif

         endif

c end of record loop

      ENDDO

c Normal Termination ... if there ever is such a thing with this code

      call lbclos(luin)
      call lbclos(luout)
      call lbclos(luvel)

      write(LERR,*)' Normal Termination'
      write(LER,*)' vttd: Normal Termination'

      stop

 990  continue

cccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc
c     data longer than array limits of program
cccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc

      write(LER,*)' '
      write(LER,*)'VTTD:'
      write(LER,*)' Internal arrays are not large enough to accomodate'
      write(LER,*)' the amount of time you have asked for on output.'
      write(LER,*)' You may ask for a shorter amount of time [-t] '
      write(LER,*)' and try again.  Another common reason for this'
      write(LER,*)' error is that the UnitSc in your velocity '
      write(LER,*)' dataset is incorrect.'
      write(LER,*)'FATAL'

      write(LERR,*)' Internal arrays are not large enough to accomodate'
      write(LERR,*)' the amount of time you have asked for on output'
      write(LERR,*)' You may ask for a shorter amount of time [-t] '
      write(LERR,*)' and try again.  Another common reason for this'
      write(LERR,*)' error is that the UnitSc in your velocity '
      write(LERR,*)' dataset is incorrect.'
      write(LERR,*)'FATAL'

 999  continue

      call lbclos(luin)
      call lbclos(luout)
      call lbclos(luvel)

      write(LERR,*)' Abnormal Termination'
      write(LER,*)' vttd: Abnormal Termination'

      stop
      end

c subroutines

      subroutine cmdlin ( ntap, otap, vtap, iflag, dz, dtms, zmax, tmax,
     :     itype, threed, fourd, vtin, nsinc, nqc, name, 
     :     ler, lerr, verbos )

      implicit none

c Parse command line
                    
      integer argis, nqc, iflag, itype, nsinc, ler, lerr

      real dz, dtms, zmax, tmax

      logical verbos, threed, fourd, vtin

      character ntap*(*), otap*(*), vtap*(*), name*(*)

c     set defaults to no pipes

      verbos=.false.
      threed=.false.
      vtin  =.false.
      call argstr('-N',ntap,' ',' ')
      call argstr('-O',otap,' ',' ')
      call argstr('-VT',vtap,' ',' ')
      call argI4 ('-flag',iflag,0,0)
      call argI4 ('-type',itype,0,0)
      call argI4 ('-nsn',nsinc,8,8)
      call argI4 ('-qc',nqc,1000,0)
      call argR4 ('-dz',dz,0.0,0.0)
      call argR4 ('-dt',dtms,0.0,0.0)
      call argR4 ('-z',zmax,0.0,0.0)
      call argR4 ('-t',tmax,0.0,0.0)
      verbos = (argis('-V').gt.0)
      vtin   = (argis('-vt').gt.0)
      threed = (argis('-3d').gt.0)
      fourd  = (argis('-4d').gt.0)

c check for extraneous arguments and abort if found to
c catch all manner of user typo's

      call xtrarg ( name, LER, .FALSE., .FALSE. )
      call xtrarg ( name, LERR, .FALSE., .TRUE. )


      return
      end
