C*****  CFFTM   Complex FFT - Mixed-Radix         MATH ADV    REL 3.0
C
C    ** COPYRIGHT 1988 QUANTITATIVE TECHNOLOGY CORPORATION **
C
C  CALL FORMAT
C
C       CALL CFFTM (C,IC,N,IDIR,INIT,ITAB,RTAB,IERR)
C
C       where,
C
C       C       Complex input/output vector.
C
C       IC      Integer input stride for vector C.
C               IC must be greater than 1.
C
C       N       Integer input element count.
C               N must be factorable as integer powers
C               of 2, 3, 5, and/or 7.
C
C       IDIR    Integer input direction flag:
C                  IDIR > 0 - perform forward FFT
C                  IDIR = 0 - no FFT performed
C                  IDIR < 0 - perform inverse FFT
C
C       INIT    Integer input initialization flag:
C                  INIT <> 0 - initialize tables
C                  INIT =  0 - no initialization
C
C       ITAB    Integer input/output vector of length 2 * N + 34
C               Integer initialization tables.
C
C       RTAB    Real input/output vector of length 2 * N + 13
C               Real initialization tables.
C
C       IERR    Integer output completion code:
C                  IERR =  0 - normal completion
C                  IERR >  0 - invalid value of N (see below)
C                  IERR = -1 - invalid value of IC (IC<2)
C                  IERR = -2 - invalid tables (see below)
C
C  DESCRIPTION
C
C       This routine performs either a forward or inverse mixed-radix
C       complex-to-complex FFT on the data stored in vector C, depending
C       on the value of IDIR.
C
C       If IDIR > 0, the routine performs a forward FFT.  The results
C       are overlaid on vector C.  These results are not properly
C       scaled and may be scaled using CFFTSS to multiply by 1/N.
C
C       If IDIR < 0, the routine performs an inverse FFT.  The results
C       are overlaid on vector C and do not need to be scaled.
C
C       The parameters IDIR and INIT operate independently of one another.
C       If INIT is nonzero, then tables corresponding to the specific
C       value of N are generated, stored and returned in the vectors ITAB
C       and RTAB.  If INIT = 0, then ITAB and RTAB are not generated, thus,
C       tables previously generated must be supplied as inputs in ITAB
C       and RTAB.
C
C       If INIT = 0 and the tables were originally generated with
C       different values of IC and IDIR, then the tables are automatically
C       modified.  When the tables are generated, the value of N is stored
C       in the first element of both ITAB and RTAB.  If INIT = 0, then the
C       value of N is checked against the first element of both ITAB and
C       RTAB.  If the values do not match then the tables are declared
C       to be invalid.
C
C       If both IDIR and INIT are nonzero, then the table generation occurs
C       prior to performing the FFT.  If both IDIR and INIT are zero, the
C       only action is the validity checking of the parameter N.
C
C       The length of the vector, N, must be factorable by 2, 3, 5 and/or 7;
C       i.e., N = 2**I * 3**J * 5**K * 7**L where I, J, K, and L are
C       non-negative integers.
C
C       If the value of N is invalid, then the value returned in IERR
C       is the smallest valid length that is greater than N;
C       i.e., IERR is set to the smallest value larger than N that is
C       factorable by 2, 3, 5 and/or 7.
C
C       The tables generated by this routine may not be used as input to
C       the Math Advantage routine RFFTM.
C
C       Refer to Appendix B for machine dependent notes.
C
C
C  REFERENCE
C
C       R.C. Singleton.  June 1969.  An algorithm for computing the
C       mixed-radix fast Fourier transform.  IEEE Trans. on Audio and
C       Electroacoust., Vol. AU-17, pp. 93-103.
C
C       R.C. Singleton.  1979.  Mixed-radix fast Fourier transforms.
C       Programs for Digital Signal Processing, Chapter 1.4.  IEEE Press.
C
C  EXAMPLE
C
C       CALL CFFTM (C, 2, 15, 1, 1, ITAB, RTAB, IERR)
C
C       Input Operands:
C
C       C = ( 0.799425244E+00,  0.448746413E+00)
C           ( 0.881257832E+00,  0.677425504E+00)
C           ( 0.102390274E+00,  0.601069748E+00)
C           ( 0.456132561E+00,  0.280523002E+00)
C           ( 0.441384196E+00,  0.238012567E+00)
C           ( 0.207332477E+00,  0.163377538E+00)
C           ( 0.354108214E-01,  0.952727020E+00)
C           ( 0.720061481E+00,  0.529401958E+00)
C           ( 0.277558297E+00,  0.901505202E-01)
C           ( 0.137023747E+00,  0.378838807E+00)
C           ( 0.960211396E+00,  0.411810011E+00)
C           ( 0.918951392E+00,  0.723790646E+00)
C           ( 0.108292505E+00,  0.655161440E+00)
C           ( 0.895454586E+00,  0.254285604E+00)
C           ( 0.636755645E+00,  0.427239627E+00)
C
C       Output Operands:
C
C       C = ( 0.757764244E+01,  0.683256102E+01)
C           ( 0.920568407E+00,  0.150177884E+01)
C           ( 0.206806839E+00, -0.174792826E+00)
C           ( 0.276668930E+01, -0.558053553E-01)
C           ( 0.482653022E+00,  0.752254426E+00)
C           (-0.139319503E+01, -0.862490118E+00)
C           (-0.440723896E-01, -0.200913405E+01)
C           (-0.227333620E+00, -0.722130477E+00)
C           ( 0.317347407E+00,  0.912802696E+00)
C           ( 0.623837233E+00,  0.486941338E-01)
C           (-0.157559311E+01,  0.217791986E+01)
C           ( 0.156958497E+01, -0.132402384E+01)
C           (-0.108925235E+01,  0.303353220E+00)
C           ( 0.402227521E+00,  0.356001973E+00)
C           ( 0.145346642E+01, -0.100579405E+01)
C
C       ITAB = 15       RTAB =  0.150000000E+02
C               2               0.100000000E+01
C               2               0.866025388E+00
C              14              -0.809017062E+00
C               3               0.587785244E+00
C               5               0.309016973E+00
C               0               0.951056540E+00
C               .               0.623489797E+00
C               .               0.781831503E+00
C               .              -0.222520947E+00
C               0               0.974927902E+00
C               3              -0.900968909E+00
C              25               0.433883607E+00
C              15               0.406736642E+00
C              17               0.743144810E+00
C              11               0.951056540E+00
C              29               0.994521916E+00
C               5               0.866025388E+00
C              19               0.587785184E+00
C               7               0.207911611E+00
C              13              -0.207911789E+00
C              23              -0.587785363E+00
C              21              -0.866025448E+00
C              27              -0.994521916E+00
C               9              -0.951056480E+00
C               0              -0.743144751E+00
C              25              -0.406736493E+00
C              15               0.174845553E-06
C              17               0.913545430E+00
C              11               0.669130623E+00
C              29               0.309016973E+00
C               3              -0.104528509E+00
C              19              -0.500000060E+00
C               5              -0.809017062E+00
C              13              -0.978147626E+00
C              23              -0.978147566E+00
C              21              -0.809016943E+00
C              27              -0.499999911E+00
C               9              -0.104528338E+00
C               7               0.309017122E+00
C               0               0.669130743E+00
C                               0.913545549E+00
C                               0.100000000E+01
C
C       IERR = 0
C
C  HISTORY
C
C     1) Jan 88          R.D. Coleman             Original
C
C---------------------------------------------------------------------
C
      SUBROUTINE CFFTM (C, IC, N, IDIR, INIT, ITAB, RTAB, IERR)
C
C  PARAMETERS:
C
      REAL    C(*), RTAB(*)
      INTEGER IC, N, IDIR, INIT, ITAB(*), IERR
C
C  LOCAL VARIABLES:
C
      INTEGER J
C                      LOOP INDEX
      INTEGER JC
C                      POINTER TO STRIDE IN ITAB
      INTEGER JCON
C                      POINTER TO START OF CONSTANT TABLE IN RTAB
      INTEGER JCOS
C                      POINTER TO START OF COSINE TABLE IN RTAB
      INTEGER JD
C                      POINTER TO DIRECTION FLAG IN RTAB
      INTEGER JDES
C                      POINTER TO START OF DESTINATION TABLE IN ITAB
      INTEGER JF
C                      POINTER TO NUMBER OF FACTORS IN ITAB
      INTEGER JFAC
C                      POINTER TO START OF FACTOR TABLE IN ITAB
      INTEGER JN
C                      POINTER TO ELEMENT COUNT IN ITAB AND RTAB
      INTEGER JP
C                      POINTER TO NUMBER OF PERMUTATIONS IN ITAB
      INTEGER JSIN
C                      POINTER TO START OF SINE TABLE IN RTAB
      INTEGER JSRC
C                      POINTER TO START OF SOURCE TABLE IN ITAB
      INTEGER KC
C                      PREVIOUS STRIDE VALUE
      INTEGER KDIR
C                      INTERNAL DIRECTION FLAG
      INTEGER MCON
C                      SPACE RESERVED FOR CONSTANT TABLE IN RTAB
      INTEGER MFAC
C                      SPACE RESERVED FOR FACTOR TABLE IN ITAB
      INTEGER NF
C                      NUMBER OF FACTORS = ELEMENT COUNT OF FACTOR TABLE
      INTEGER NP
C                      NUMBER OF PERMUTATIONS = ELEMENT COUNT OF SOURCE
C                      AND DESTINATION TABLES
C
      PARAMETER (JN = 1, JD = 2, JC = 2, JF = 3, JP = 4)
      PARAMETER (JCON =  3, JFAC = 5)
      PARAMETER (MCON = 11, MFAC =30)
C
C-----------------------------------------------------------------------
C
      IERR =  0
C
C  INITIALIZE TABLE POINTERS
C
      JSIN = JCON + MCON
      JCOS = JSIN + N
      JSRC = JFAC + MFAC
      JDES = JSRC + N
C
C  SET INTERNAL DIRECTION FLAG
C
      IF      (IDIR .GE. 0) THEN
         KDIR =  1
      ELSE IF (IDIR .LT. 0) THEN
         KDIR = -1
      ENDIF
C
C  SELECT PATH
C
      IF      (INIT .NE. 0) THEN
         GO TO 100
      ELSE IF (IDIR .NE. 0) THEN
         GO TO 200
      ELSE
         CALL QTC050 (N, IERR)
         IF (IERR .EQ. N) IERR = 0
         GO TO 800
      ENDIF
C
C=======================================================================
C                           INITIALIZE TABLES
C=======================================================================
C
  100 CONTINUE
      ITAB(JN) = 0
      RTAB(JN) = 0.0
C
      CALL QTC051 (IC, N, KDIR, ITAB(JFAC), NF, RTAB(JCON),
     &             RTAB(JSIN), RTAB(JCOS), ITAB(JSRC), ITAB(JDES), NP)
C
C  CHECK IF N FACTORED OK
C
      IF (NF .EQ. 0) THEN
         CALL QTC050 (N, IERR)
         GO TO 800
      ENDIF
C
C  CHECK VALIDITY OF STRIDE
C
      IF (IC .LT. 2) THEN
         IERR = -1
         GO TO 800
      ENDIF
C
C  SAVE SOME PARAMETERS IN TABLES
C
      ITAB(JN) = N
      ITAB(JC) = IC
      ITAB(JF) = NF
      ITAB(JP) = NP
      RTAB(JN) = FLOAT( N )
      RTAB(JD) = FLOAT( KDIR )
C
C  WHERE NOW ?
C
      IF (IDIR .NE. 0) THEN
         GO TO 300
      ELSE
         GO TO 800
      ENDIF
C
C=======================================================================
C                        MODIFY EXISTING TABLES
C=======================================================================
C
  200 CONTINUE
C
C  CHECK VALIDITY OF TABLES
C
      IF (N .NE. ITAB(JN) .OR. N .NE. IFIX( RTAB(JN) )) THEN
         IERR = -2
         GO TO 800
      ENDIF
C
      NF = ITAB(JF)
      NP = ITAB(JP)
C
C  IF DIRECTION HAS CHANGED, MODIFY RTAB
C
      IF (KDIR .NE. IFIX( RTAB(JD) )) THEN
         RTAB(JD) = FLOAT( KDIR )
C
         DO 210 J = JSIN, JSIN+N-1
            RTAB(J) = -RTAB(J)
  210    CONTINUE
C
         DO 220 J = JCON, JCON+MCON-1, 2
            RTAB(J) = -RTAB(J)
  220    CONTINUE
      ENDIF
C
C  IF STRIDE HAS CHANGED AND NEW STRIDE IS VALID, MODIFY ITAB
C
      IF (IC .NE. ITAB(JC)) THEN
         IF (IC .LT. 2) THEN
            IERR = -1
            GO TO 800
         ENDIF
C
         KC = ITAB(JC)
         ITAB(JC) = IC
C
         DO 230 J = JSRC, JSRC+NP-1
            ITAB(J) = ((ITAB(J) - 1) / KC) * IC + 1
  230    CONTINUE
C
         DO 240 J = JDES, JDES+NP-1
            ITAB(J) = ((ITAB(J) - 1) / KC) * IC + 1
  240    CONTINUE
      ENDIF
C
C=======================================================================
C                              PERFORM FFT
C=======================================================================
C
  300 CONTINUE
      CALL QTC052 (C(1), C(2), IC, N, KDIR, ITAB(JFAC), NF,
     &             RTAB(JCON), RTAB(JSIN), RTAB(JCOS),
     &             ITAB(JSRC), ITAB(JDES), NP)
C
C=======================================================================
C                               EXIT ROUTINE
C=======================================================================
C
  800 CONTINUE
      RETURN
      END
