C*****  CMES   Complex Envelope Symm Reorder/Factor/Solve  MATH ADV REL 3.0
C
C    ** COPYRIGHT 1988 QUANTITATIVE TECHNOLOGY CORPORATION **
C
C  CALL FORMAT
C
C       CALL CMES (II,JJ,VV,NV,N,ZTOL,IFLG,IWRK,NIW,CWRK,NCW,BX,M,IERR)
C
C       where,
C
C       II      Integer input vector of length NV containing the row
C               indices of the nonzero elements of the matrix A.  Only
C               the upper or lower triangle of the matrix needs to be
C               specified.  Indices may be in any order.  II is accessed
C               only in phases 1 and 2.
C
C       JJ      Integer input vector of length NV containing the column
C               indices of the nonzero elements of the matrix A
C               corresponding to the elements of II.  JJ is accessed
C               only in phases 1 and 2.
C
C       VV      Complex input vector of length NV containing the values
C               of the nonzero elements of the matrix A corresponding to
C               elements of II and JJ.  VV is accessed only in phase 2.
C
C       NV      Integer input element count for II, JJ, and VV.
C
C       N       Integer input order of the matrix A.
C
C       ZTOL    Real input scalar pivot zero tolerance.
C               ZTOL is accessed only in phase 2.
C
C       IFLG    Integer input scalar function flag:
C                  If IFLG mod 8 = 0, no operation is performed
C                  If IFLG mod 8 = 1, Perform phase 1
C                  If IFLG mod 8 = 2, Perform phase 2
C                  If IFLG mod 8 = 3, Perform phases 1 and 2
C                  If IFLG mod 8 = 4, Perform phase 3
C                  If IFLG mod 8 = 5, illegal function flag
C                  If IFLG mod 8 = 6, Perform phases 2 and 3
C                  If IFLG mod 8 = 7, Perform all phases (1, 2, & 3)
C
C       IWRK    Integer scratch/input/output vector of length NIW.
C               A portion of IWRK is output from phase 1 which is then
C               input to phases 2 and 3.  The remainer of IWRK is used
C               for scratch space.
C
C       NIW     Integer input scalar element count for IWRK.
C
C       CWRK    Complex scratch/input/output vector of length NCW.
C               CWRK is output from phase 2 which is then input to phase
C               3.  CWRK is accessed only in phases 2 and 3.
C
C       NCW     Integer input scalar element count for CWRK.
C
C       BX      Complex input/output matrix of dimension N by M.  On input
C               to phase 3, BX contains M right-hand side vectors.  On
C               output from phase 3, BX contains M solution vectors.
C               BX is accessed only in phase 3.
C
C       M       Integer input scalar number of right-hand side vectors.
C
C       IERR    Integer output scalar completion code:
C                  IERR =  0, normal completion
C                  IERR = -1, invalid value of NV, N, M, or IFLG
C                  IERR = -2, invalid index in II or JJ (<1 or >N)
C                  IERR = -3, IWRK is too small
C                  IERR = -4, CWRK is too small
C                  IERR = -5, specified element not in envelope
C                  IERR >  0, matrix is singular (i.e., pivot element <=
C                             ZTOL).  IERR contains the pivot column.
C
C
C  DESCRIPTION
C
C       This routine solves the complex sparse linear system Ax = b
C       where the nonzero elements of the matrix A are specified in list
C       format (II,JJ,VV,NV).  The sparse symmetric envelope method is
C       used.
C
C       The processing is divided into three phases:
C          Phase 1:
C             (a) Reorder the matrix using the reverse Cuthill-McKee
C                 method to minimize the size of the envelope (see
C                 subroutine MORRCM).
C             (b) Generate the arrays that define the envelope structure.
C          Phase 2:
C             (a) Insert the nonzero values into the envelope.
C             (b) Factor the matrix using subroutine CMESFC.
C          Phase 3:
C             (a) Permute the right-hand side vectors to reorder.
C             (b) Solve the system using subroutine CMESSV.
C             (c) Permute the solution vector to obtain the original
C                 ordering.
C
C       The parameter IFLG can be used to select individual phases or
C       pairs of phases.  This is useful when solving many systems with
C       the same nonzero structure and/or solving many systems with the
C       same coefficient matrix.
C
C       Normally, the user need not be concerned with the contents of
C       the work vectors IWRK and CWRK other than to insure that they
C       are passed to the following phases when the phases are used
C       individually or in pairs.
C
C       During phase one, the work vector IWRK requires a minimum of
C       4*N + 2*NE + 3 elements where NE is the number of nonzero
C       elements above (or below) the diagonal of the matrix.  At the
C       completion of phase 1, the first 3*N + 1 elements contain the
C       following arrays that are used by phases 2 and 3:
C          IPERMF - N words (see MORRCM)
C          IPERMR - N words (see MORRCM)
C          ICP    - N+1 words (see CMESFC)
C
C       At the end of phase two, the first NS elements of CWRK contain
C       the factors of A in envelope format (see CMESFC, vector S) that
C       are passed to phase 3.  NS is the size of the envelope.  At the
C       completion of phase 1, IWRK(3*N+1) = NS+1.
C
C
C  REFERENCE
C
C       Alan George and Joseph W-H. Liu.  1981.  Computer Solution of
C       Large Sparse Positive Definite Systems.  Englewood Cliffs, N.J.:
C       Prentice-Hall, Inc.
C
C
C  EXAMPLE
C
C       CALL CMES (II,JJ,VV,NV,N,ZTOL,IFLG,IWRK,NIW,CWRK,NCW,BX,M,IERR)
C
C       Input Operands:
C
C       NV   =  21
C       N    =  10
C       ZTOL =   0.1E-04
C       IFLG =   7
C       NIW  =  65
C       NCW  =  25
C       M    =   1
C
C       II =  1      JJ =  1      VV = (  11.0,  11.0)
C             2            1           (   2.0,  -2.0)
C             5            1           (  -1.0,  -4.0)
C             9            1           (   3.0,   4.0)
C             2            2           ( -12.0,  12.0)
C             3            2           (  -2.0,   3.0)
C             6            2           (  -4.0,   5.0)
C            10            2           (   1.0,   0.0)
C             3            3           (  13.0, -13.0)
C             4            4           ( -14.0, -14.0)
C             7            4           (   5.0,  -5.0)
C             8            4           (   0.0,  -3.0)
C             9            4           (  -3.0,   1.0)
C             5            5           (  15.0,  15.0)
C             8            5           (   4.0,   2.0)
C             6            6           ( -16.0,  16.0)
C             7            7           (  17.0, -17.0)
C             8            8           ( -18.0, -18.0)
C             9            8           (  -5.0,  -1.0)
C             9            9           (  19.0,  19.0)
C            10           10           ( -20.0,  20.0)
C
C       BX = (   2.0,  24.0)
C            (  57.0, -10.0)
C            (   5.0,  -1.0)
C            ( -80.0,  -9.0)
C            ( -45.0,  20.0)
C            ( 105.0, -33.0)
C            (  63.0,-103.0)
C            ( -19.0,  86.0)
C            ( -91.0,  24.0)
C            (-101.0, -61.0)
C
C       Output Operands:
C
C       IERR =   0
C
C       BX = (   2.0,   1.0)
C            (  -1.0,  -1.0)
C            (   0.0,   0.0)
C            (   3.0,  -4.0)
C            (   0.0,   3.0)
C            (  -4.0,  -2.0)
C            (   4.0,   0.0)
C            (  -2.0,  -3.0)
C            (  -3.0,   2.0)
C            (   1.0,   4.0)
C
C  HISTORY
C         1) Jan 88     R.D. Coleman    Original.
C
C         2) Mar 88     R.L. Jacobsen   Corrected the local NS
C                                       variable in phase 3
C-----------------------------------------------------------------------
C
      SUBROUTINE CMES (II, JJ, VV, NV, N, ZTOL, IFLG,
     &                 IWRK, NIW, CWRK, NCW,
     &                 BX, M, IERR)
C
C  PARAMETERS:
C
      INTEGER II(1), JJ(1), NV, N, IFLG, IWRK(1), NIW, NCW, M, IERR
      REAL    VV(1), ZTOL, CWRK(1), BX(1)
C
C  LOCAL VARIABLES:
C
      INTEGER IAD, ICP, IPF, IPR, IWK, JAD, JFLG, MAX, NB, NE, NS
      LOGICAL PHASE1, PHASE2, PHASE3
C
C-----------------------------------------------------------------------
C
      IERR = 0
      JFLG = MOD( IFLG, 8 )
      IF (JFLG .EQ. 0) GO TO 800
      IF (JFLG .EQ. 5) THEN
         IERR = -1
         GO TO 800
      ENDIF
C
      PHASE1 = .FALSE.
      PHASE2 = .FALSE.
      PHASE3 = .FALSE.
C
      IF (JFLG .GE. 4) THEN
         PHASE3 = .TRUE.
         JFLG   = JFLG - 4
      ENDIF
C
      IF (JFLG .GE. 2) THEN
         PHASE2 = .TRUE.
         JFLG   = JFLG - 2
      ENDIF
C
      IF (JFLG .EQ. 1) PHASE1 = .TRUE.
C
      IF (             N  .LE. 0 .OR.
     &    PHASE1 .AND. NV .LE. 0 .OR.
     &    PHASE2 .AND. NV .LE. 0 .OR.
     &    PHASE3 .AND. M  .LE. 0     ) THEN
         IERR = -1
         GO TO 800
      ENDIF
C
      IPF = 1
      IPR = IPF + N
      ICP = IPR + N
      IWK = ICP
C
      IF (PHASE1) GO TO 100
      IF (PHASE2) GO TO 200
      IF (PHASE3) GO TO 300
      GO TO 800
C
C  BUILD STRUCTURE
C
  100 CONTINUE
      IAD = ICP + N + 1
      JAD = IAD + N + 1
      MAX = NIW - JAD + 1
C
      IF (MAX .LE. 0) THEN
         IERR = -3
         GO TO 800
      ENDIF
C
      CALL QTC060 (II, JJ, IWRK(IAD), IWRK(JAD), IWRK(IWK),
     &             NV, N, NE, MAX, IERR)
C
      IF (IERR .NE. 0) THEN
         IERR = - IERR - 1
         GO TO 800
      ENDIF
C
      CALL QTC066 (N, IWRK(IAD), IWRK(JAD), IWRK(IPR), IWRK(IPF),
     &             IWRK(IWK))
C
      CALL QTC076 (N, IWRK(IAD), IWRK(JAD), IWRK(IPR), IWRK(IPF),
     &             IWRK(ICP), NS, NB)
C
      IF (.NOT.PHASE2) GO TO 800
C
C  INSERT DATA AND FACTOR
C
  200 CONTINUE
      NS = IWRK(ICP+N) - 1
      IF (NS .GT. NCW) THEN
         IERR = -4
         GO TO 800
      ENDIF
C
      CALL QTC101 (II, JJ, VV, IWRK(IPF), IWRK(ICP), CWRK, NV, N, IERR)
C
      IF (IERR .NE. 0) THEN
         IERR = 1 - 3 * IERR
         GO TO 800
      ENDIF
C
      CALL CMESFC (CWRK, IWRK(ICP), N, NS, ZTOL, IERR)
C
      IF (IERR .NE. 0) THEN
         IERR = IWRK(IPR-1+IERR)
         GO TO 800
      ENDIF
C
      IF (.NOT.PHASE3) GO TO 800
C
C  SOLVE
C
  300 CONTINUE
      NS = IWRK(ICP+N) - 1
      IF (NS .GT. NCW) THEN
         IERR = -4
         GO TO 800
      ENDIF
C
      CALL QTC107 (BX, N, M, IWRK(IPF), IWRK(IPR))
      CALL CMESSV (CWRK, IWRK(ICP), N, NS, BX, M)
      CALL QTC107 (BX, N, M, IWRK(IPR), IWRK(IPF))
C
  800 CONTINUE
      RETURN
      END
