C******  EUREKA  SYMMETRIC TOEPLITZ MATRIX SOLVER        MTHADV EXT     REL 1.0
C
C   ** COPYRIGHT 1988  QUANTITATIVE TECHNOLOGY CORPORATION **
C
C  CALL FORMAT
C
C       CALL EUREKA (A, B, X, EP, ZTOL, N, IERR)
C
C       where
C
C       A     Real input vector of the diagonal values that specify the
C             Toeplitz matrix. These are typically autocorrelation coefficients.
C
C       B     Real input vector of right side coeffiicents.
C
C       X     Real output solution vector - typically the filter coefficients.
C
C       EP    Real output vector prediction error operator.
C
C       ZTOL  Real input scaler tolerance for singularity tests.
C
C       N     Integer input number of elements in the vectors.
C
C       IERR  Integer output completion code:
C               IERR = 0              normal termination
C               IERR = -N+1           N is less than or equal zero
C               0  <  IERR < (N+1)    matrix has a singular principal minor.
C                                     The value of IERR indicates the problem
C                                     column of the Toeplitz matrix.
C
C  DESCRIPTION:
C
C       A Toeplitz matrix is a special matrix form in which the elements of
C       each diagonal has a constant value.   The linear Toeplitz problem is
C       solved by a bordering method.   This recursive algorithm fails if any
C       of the diagonal, principal minors are zero.  Notice that this does not
C       indicate a singluar matrix and it may be possible to solve the
C       matrix with standard Gaussian Elimination techniques.
C
C  REFERENCES:
C
C       Robinson, E.A. and Treitel, S. 1980, Gelphysical Signal Analysis
C       (Englewood Cliffs, N.J.: Prentice-Hall), p. 163.
C       Press, W.H. et al, 1986, Numerical Recipes (Cambridge University
C       press), p. 47.
C
C  EXAMPLE:
C
C       A linear, symmetric Toeplitz problem is solved in this example for
C       the Toeplitz matrix:
C
C                        1.00  2.00  3.00  4.00  5.00
C                        2.00  1.00  2.00  3.00  4.00
C                        3.00  2.00  1.00  2.00  3.00
C                        4.00  3.00  2.00  1.00  2.00
C                        5.00  4.00  3.00  2.00  1.00
C
C      CALL EUREKA (A, B, X, EP, 1.0e-8, 5, IERR)
C
C      INPUT OPERANDS:
C
C            A =  1.0                 B  =  3.0
C                 2.0                       4.0
C                 3.0                       1.0
C                 4.0                       2.0
C                 5.0                       5.0
C
C      OUTPUT OPERANDS:
C
C            X =   1.166667           EP =  1.00000
C                 -2.000000                -1.20000
C                  2.000000                 0.00000
C                  0.9999998                0.00000
C                 -0.8333332               -0.20000
C
C  SUBPROGRAMS
C
C       INTRINSICS: ABS
C
C  HISTORY:
C       1) Apr 1988          T. G. Mattson               Original
C
C----------------------------------------------------------------------------
       SUBROUTINE EUREKA (A, B, X, EP, ZTOL, N, IERR)
       INTEGER  N,     IERR, L, L1, L2, K, L3, I, J
       REAL     A(*), B(*), X(*), EP(*), ZTOL
       REAL     V, D, Q, HOLD
C
C----------------------------------------------------------------------------
C
       IF (N .LE. 0) THEN
          IERR = -N + 1
          RETURN
       ENDIF
C
C  Initialize recursion
C
       IERR   = 0
       V      = A(1)
       D      = A(2)
       EP(1)  = 1.0
       IF( ABS(V) .LT. ZTOL) THEN
           IERR = 1
           RETURN
       ENDIF
       X(1) = B(1)/V
       Q    = X(1)*D
       IF (N .EQ. 1) THEN
           RETURN
       ENDIF
C
C  Main recursion loop
C
       DO 400 L=2,N
C
C  Find the "prediction error operator" vector, EP
C
          EP(L) = -D/V
          IF (L .GT. 2) THEN
             L1   = (L-2)/2
             L2   = L1+1
             DO 210 J=2,L2
                HOLD  = EP(J)
                K     = L-J+1
                EP(J) = EP(J) + EP(L)*EP(K)
                EP(K) = EP(K) + EP(L)*HOLD
 210         CONTINUE
             IF ( (L1 + L1) .NE. L-2) THEN
                EP(L2+1) = EP(L2+1) + EP(L)*EP(L2+1)
             ENDIF
          ENDIF
C
C   Find the solution vector, X
C
          V     = V+EP(L)*D
          IF( ABS(V) .LT. ZTOL) THEN
              IERR = L
              RETURN
          ENDIF
          X(L)  = (B(L) - Q)/V
          L3    = L-1
          DO 300 J=1,L3
             K    = L-J+1
             X(J) = X(J) + X(L)*EP(K)
 300      CONTINUE
C
C  Exit test
C
          IF (L .EQ. N)THEN
              RETURN
          ENDIF
C
C  Setup for next iteration
C
          D = 0.0
          Q = 0.0
          DO 375 I=1,L
             K = L-I+2
             D = D + EP(I)*A(K)
             Q = Q + X(I)*A(K)
 375      CONTINUE
 400   CONTINUE
C
        RETURN
        END
