C****** NMOCOR   NORMAL MOVEOUT CORRECTION               MTADV EXT    REL 1.0
C
C      ** COPYRIGHT 1988  QUANTITATIVE TECHNOLOGY CORPORATION **
C
C  CALL FORMAT
C
C        CALL NMOCOR (TRC,DT,N,X,V,INV)
C
C        where,
C
C        TRC      Real input/output array of length N.  Contains the source
C                 trace and is overwritten with the result trace.
C
C        DT       Real input scaler, sample interval within a trace.
C
C        N        Integer input number of elements in the trace, TRC.
C
C        X        Real input scaler offset distance.
C
C        V        Real input vector of length N specifying the velocity
C                 as a function of time (depth).
C
C        INV      Logical input scaler.
C                           .TRUE.    for an inverse normal moveout
C                           .FALSE.   for a forward normal moveout
C
C  DESCRIPTION
C
C        This routine will perform a forward or inverse normal moveout
C        on a trace.  The travel time equation as a function of offset, x,
C        is taken to be:
C
C                 (t(x))**2) = (t(0))**2 + (x/v)**2
C
C        The NMO procedure moves an amplitude at t(x) to a new position at
C        t(0).   Linear interpolation is used to find the amplitude at t(x).
C        This routine immediately exits if DT or N are less than or equal
C        to zero.
C
C  REFERENCE
C
C        Yilmaz 0., 1987, Seismic Data Processing, (Tulsa, OK, Society
C        of Exploration Geophysicists), p. 157.
C
C  EXAMPLE
C
C       In the following example, a forward, NMO corection is applied to a
C       trace with a constant NMO velocity of 5000.
C
C        CALL NMOCOR (TRC, 0.002, 10, 50.0, V, .FALSE.)
C
C        Input Operands:
C
C              TRC  =   0.000              V  =  5000
C                       0.500                    5000
C                       1.250                    5000
C                       0.750                    5000
C                      -0.500                    5000
C                      -1.750                    5000
C                      -0.250                    5000
C                       0.500                    5000
C                       1.500                    5000
C                       0.250                    5000
C
C        Output Operands:
C
C              TRC  =  -1.75000000E+00
C                      -1.60147071E+00
C                      -1.17225289E+00
C                      -5.03572464E-01
C                       5.23432493E-02
C                       5.71067810E-01
C                       1.31025028E+00
C                       7.47093201E-01
C                       0.00000000E+00
C                       0.00000000E+00
C
C  SUBPROGRAMS:
C
C        INTRINSICS:  SQRT, ABS
C
C  HISTORY
C        (1) MAY 88           T.G. Mattson           Original
C
C-----------------------------------------------------------------------
C
      SUBROUTINE NMOCOR (TRC, DT, N, X, V, INV)
C
      REAL    TRC(*), DT, X, V(*), DJ, TNMO
      INTEGER N, I, I2, J
      LOGICAL INV
C
C-----------------------------------------------------------------------
C
      IF (N  .LE. 0) GO TO 800
      IF (DT .EQ. 0) GO TO 800
      IF (INV) THEN
C
C  Inverse Normal Moveout Correction.  Issue the "ignore vector dependency"
C  directive for running on CRAY 2.
CDIR$ IVDEP
C
         DO 110 I = N, 1, -1
            TNMO   = SQRT( (I-1)**2 + (X / (V(I) * DT))**2 ) + 1
            J      = TNMO
            DJ     = TNMO - J
            IF ( J .GT. N-1 )THEN
                TRC(J) = 0.0
            ELSE
                TRC(J) = TRC(I) + DJ * (TRC(I+1) - TRC(I))
            ENDIF
  110    CONTINUE
C
         I2 = ABS( X / (V(1) * DT) )
         DO 120 I = 1, I2
            TRC(I) = 0.0
  120    CONTINUE
C
      ELSE
C
C  Forward Normal Moveout Correction.  Issue the "ignore vector dependency"
C  directive for running on CRAY 2.
CDIR$ IVDEP
C
         DO 210 I = 1, N
            TNMO   = SQRT( (I-1)**2 + (X / (V(I) * DT))**2 ) + 1
            J      = TNMO
            DJ     = TNMO - J
            IF ( J .GT. N-1 )THEN
                TRC(I) = 0.0
            ELSE
                TRC(I) = TRC(J) + DJ * (TRC(J+1) - TRC(J))
            ENDIF
  210    CONTINUE
C
      ENDIF
C
 800  RETURN
      END
