C*****  RMTUFC  Real Matrix Tridiagonal Unsymmetric Factor  MATHADV REL 3.0
C
C    ** COPYRIGHT 1984-1985 QUANTITATIVE TECHNOLOGY CORPORATION **
C
C  CALL FORMAT
C
C       CALL RMTUFC (AD, AU, AL, N, ZTOL, IERR)
C
C       where,
C
C       AD      Real input/output vector of length N.
C               On input, AD contains the main diagonal of the
C               tridiagonal matrix to be factored.
C               On output, AD contains the reciprocals of the
C               main diagonal of the factored matrix.
C
C       AU      Real input/output vector of length N-1.
C               On input, AU contains the upper diagonal of the
C               tridiagonal matrix to be factored.
C               On output, AU contains the reciprocals of the
C               upper diagonal of the factored matrix.
C
C       AL      Real input/output vector of length N-1.
C               On input, AL contains the lower diagonal of the
C               tridiagonal matrix to be factored.
C               On output, AL contains the reciprocals of the
C               lower diagonal of the factored matrix.
C
C       N       Integer input number of rows in full matrix.
C
C       ZTOL    Real input scalar, diagonal element zero tolerance.
C
C       IERR    Integer output completion code:
C                   =0 if the routine terminated normally.
C                   >0 if the routine aborted because a diagonal
C                      element was less than or equal to ZTOL.
C                      This implies the matrix is singular.
C                      The value of IERR is the index of the column
C                      where it aborted.
C
C
C  DESCRIPTION
C
C       This routine factors a real tridiagonal unsymmetric matrix A
C       represented by AD, AU and AL into LU form.
C
C       L is lower triangular and U is upper triangular.  The diagonal
C       elements of U are all 1's and are therefore not stored.  The
C       diagonal elements of L are reciprocated.
C
C       This routine does not do pivoting.  Therefore, the matrix A
C       should be numerically stable in the form in which it is input.
C
C
C  REFERENCE
C
C       C. R. Rao.  1962.  A note on a generalized inverse of a
C       matrix with application to problems in mathematical
C       statistics.  J. R. Statis. Soc., Vol. B24,  pp. 152-
C       158.
C
C       M. J. R. Healy.  1968.  Triangular decomposition of a
C       symmetric matrix.  Appl. Statis., Vol. 17,  pp. 195-
C       197.
C
C       G. W. Stewart.  1973.  Introduction to matrix computa-
C       tions.  New York: Academic Press.
C
C
C  EXAMPLE
C
C       CALL RMTUFC (AD, AU, AL, 5, ZTOL, IERR)
C
C       Input Operands:
C
C       AD = -16.000   AU =  8.000   AL =  4.000
C             11.000         4.000        -2.000
C             -8.000        -4.000         5.000
C             17.000         4.000         0.000
C              5.000
C
C       ZTOL = 1.0E-5
C
C       Output Operands:
C
C       AD = -0.0625   AU = -0.5000   AL =  4.000
C             0.0769         0.3077        -2.000
C            -0.1354         0.5417         5.000
C             0.0700         0.2799         0.000
C             0.2000
C
C       IERR = 0
C
C  HISTORY
C         1) Jan 88     L. Shanbeck     Original.
C                       R. Coleman
C
      SUBROUTINE RMTUFC (AD, AU, AL, N, ZTOL, IERR)
C
      REAL    AD(1), AU(1), AL(1), ZTOL
      INTEGER N, IERR, I
C
C   *******************************************************************
C
C     DO VALIDITY CHECKS AND INITIALIZATION
C
      IERR = 0
      IF (N.LE.0) GO TO 999
      I = 1
      IF (ABS( AD(1) ) .LE. ZTOL) GO TO 810
C
      AD(1) = 1.0 / AD(1)
      IF (N .EQ. 1) GO TO 999
C
      AU(1) = AD(1) * AU(1)
      IF (N .EQ. 2) GO TO 120
C
      DO 110 I = 2, N-1
         AD(I) = AD(I) - AL(I-1) * AU(I-1)
         IF (ABS( AD(I) ) .LE. ZTOL) GO TO 810
         AD(I) = 1.0 / AD(I)
         AU(I) = AD(I) * AU(I)
  110 CONTINUE
C
  120 CONTINUE
      I = N
      AD(N) = AD(N) - AL(N-1) * AU(N-1)
      IF (ABS( AD(N) ) .LE. ZTOL) GO TO 810
      AD(N) = 1.0 / AD(N)
      GO TO 999
C
  810 CONTINUE
      IERR = I
  999 RETURN
      END
