c...|....1....|....2....|....3....|....4....|....5....|....6....|....7..
C***********************************************************************
C Copyright 2001, Allied Geophysics, Inc. All Rights Reserved          *
C***********************************************************************
C Portions of this code and/or subroutines  used by this code are      *
C protected by the following copyright(s):                             *
C***********************************************************************
C                 copyright 2001, Amoco Production Company             *
C                             All Rights Reserved                      *
C                     an affiliate of BP America Inc.                  *
C***********************************************************************
c
c hangvz - hang a V(z) function from a map to create a V(xyz) volume
c
c     Program Description:
c
c     Create a 3D velocity volume by hanging a velocity trace on depths
c     desribed by an input map. The output volume is conformable with
c     the XY dimensions of the map. As distance below the map increases,
c     the surface followed by the V(z) can (optionally) be a smoothed
c     version of the map.
c
c     Program Changes:
c      Dec 31, 2001 - minor cleanup of code and man page
c      Oct 29, 2001 - added coefficients for least-squares plane
c                     to command line arguments
c      Oct 21, 2001 - added more user control over smoothing
c                     fixed minor bug in resampling indices
c      Jun  7, 2001 - original version
c

c get machine dependent parameters 

#include <f77/iounit.h>
#include <f77/lhdrsz.h>
#include <f77/sisdef.h>
#include <save_defs.h> 

c dimension standard USP variables 

      integer     itr(3*SZLNHD)
     
      integer     lbytes, nbytes, lbyout, obytes
      integer     luin1,luin2,luin3, luout
      integer     argis

      character   map1file*255, map2file*255, velfile*255
      character   otap*255, name*6

      logical     verbos

c Program Specific _ dynamic memory variables

      integer MapSize, VelSize, InitialMem
      integer errcd1, errcd2, errcd3, errcd4, errcd5, abort

      real    Map1(2), Map2(2), Map3(2), Vel_in(2),Vel_out(2)

      pointer (ptr_Map1, Map1)
      pointer (ptr_Map2, Map2)
      pointer (ptr_Map3, Map3)
      pointer (ptr_Vel_in, Vel_in)
      pointer (ptr_Vel_out, Vel_out)

c Program Specific _ static memory variables

      integer tr_index, ix,iy, nmap
      integer nz,nx,ny, nx1,ny1, nx2,ny2, nz_vel
      real    dz,dx,dy, z0,x0,y0
      real    vtop,vbot, usr_sm,usr_z
      integer irsm

      real    slope_x,slope_y,zbias
      logical lcoef

c Variables for descriptions of a few trace headers

      integer ifmt_RecNum,l_RecNum,ln_RecNum
      integer ifmt_TrcNum,l_TrcNum,ln_TrcNum
      integer ifmt_CDPBCX,l_CDPBCX,ln_CDPBCX
      integer ifmt_CDPBCY,l_CDPBCY,ln_CDPBCY
      integer ifmt_WDepDP,l_WDepDP,ln_WDepDP

c Initialize variables

      data abort/0/
      data name/"HANGVZ"/

c
c give command line help if requested
c
      if ( argis ( '-?' )    .gt. 0 .or. 
     :     argis ( '-h' )    .gt. 0 .or.
     :     argis ( '-help' ) .gt. 0 ) then
         call help(name)
         stop
      endif

c
c open printout file
c
#include <f77/open.h>

c
c get command line input parameters
c
      call cmdln ( map1file, map2file, velfile, otap,
     :             z0,x0,y0, dz,dx,dy, nz, vtop,vbot,
     :             usr_sm,usr_z,irsm, slope_x,slope_y,zbias,lcoef,
     :             name, verbos )

      nmap=1
      if (map2file(1:1) .ne. ' ') nmap = 2

c
c open input and output files
c
      call getln(luout, otap,'w', 1)
      call getln(luin1, map1file,'r', 0)
      if(nmap.eq.2) call getln(luin2, map2file,'r', 2)
      call getln(luin3, velfile,'r', 2)

c
c read the input map line headers and check for consistency
c
      call rtape(luin1,itr,lbytes)
      if(lbytes.eq.0)then
        write(LER,*)name,': no line header on input file',map1file
        write(LER,*)'FATAL'
        stop
      endif

      call saver(itr, 'NumSmp', nx1, LINHED)
      call saver(itr, 'NumTrc', ny1, LINHED)

      if(nmap.eq.2) then
        call rtape(luin2,itr,lbytes)
        if(lbytes.eq.0)then
          write(LER,*)name,': no line header on input file',map2file
          write(LER,*)'FATAL'
          stop
        endif

        call saver(itr, 'NumSmp', nx2, LINHED)
        call saver(itr, 'NumTrc', ny2, LINHED)

        if (nx2.ne.nx1 .or. ny2.ne.ny1) then
          write(LER,*)name,': Maps are not the same size'
          write(LER,*)'FATAL'
          stop
        endif

      endif

c
c also open the velocity trace
c
      call rtape(luin3,itr,lbytes)
      if(lbytes.eq.0)then
        write(LER,*)name,': no line header on input file',velfile
        write(LER,*)'FATAL'
        stop
      endif

      call saver(itr, 'NumSmp', nz_vel, LINHED)
      if (nz.lt.0) nz=nz_vel
      if (usr_z.lt.0) usr_z=dz*(nz_vel-1)

c
c print HLH to printout file (this comes from the velocity trace)
c
      call hlhprt (itr, lbytes, name, 4, LERR)

c
c set internal nx,ny and modify line header to reflect actual record
c configuration output
c
      nx=nx1
      ny=ny1
      call savew(itr, 'NumSmp', nz, LINHED)
      call savew(itr, 'NumTrc', nx, LINHED)
      call savew(itr, 'NumRec', ny, LINHED)
      call savew(itr, 'SmpInt', int(dz), LINHED)
      call savew(itr, 'UnitSc', 1.0, LINHED)
      call savew(itr, 'Dz1000', int(dz*1000), LINHED)
      call savew(itr, 'Dx1000', int(dx*1000), LINHED)
      call savew(itr, 'Dy1000', int(dy*1000), LINHED)

c
c number output bytes in a trace
c
      obytes = SZTRHD + SZSMPD * nz

c
c save out hlh and line header
c
      call savhlh (itr, lbytes, lbyout)
      call wrtape (luout, itr, lbyout)

c
c Get some trace header info so we can use it to stuff headers later
c
      call savelu('RecNum',ifmt_RecNum,l_RecNum,ln_RecNum,TRACEHEADER)
      call savelu('TrcNum',ifmt_TrcNum,l_TrcNum,ln_TrcNum,TRACEHEADER)
      call savelu('CDPBCX',ifmt_CDPBCX,l_CDPBCX,ln_CDPBCX,TRACEHEADER)
      call savelu('CDPBCY',ifmt_CDPBCY,l_CDPBCY,ln_CDPBCY,TRACEHEADER)
      call savelu('WDepDP',ifmt_WDepDP,l_WDepDP,ln_WDepDP,TRACEHEADER)

c
c verbose output of all pertinent information before processing begins
c
      call verbal
     :       ( map1file,map2file,velfile,otap,
     :         nx1,ny1, nx2,ny2, nz_vel,
     :         z0,x0,y0, nz,nx,ny, dz,dx,dy,
     :         vtop,vbot, usr_sm,usr_z,
     :         slope_x,slope_y,zbias,lcoef,
     :         verbos )

c
c dynamic memory allocation:
c
      MapSize = nx*ny
      VelSize = nz_vel
      InitialMem = (3*MapSize+VelSize+nz)*SZSMPD

      call galloc (ptr_Map1,MapSize*SZSMPD,errcd1,abort)
      call galloc (ptr_Map2,MapSize*SZSMPD,errcd2,abort)
      call galloc (ptr_Map3,MapSize*SZSMPD,errcd3,abort)
      call galloc (ptr_Vel_in,VelSize*SZSMPD,errcd4,abort)
      call galloc (ptr_Vel_out,nz*SZSMPD,errcd5,abort)
    
      if (errcd1.ne.0 .or. errcd2.ne.0 .or.
     :    errcd3.ne.0 .or. errcd4.ne.0 .or. errcd5.ne.0) then

        write(LERR,*)' '
        write(LERR,*)
     :    name,': Unable to allocate workspace:',InitialMem,' bytes'
        write(LERR,*)name,': ABNORMAL Termination'
        write(LERR,*)' '

        write(LER,*)' '
        write(LER,*)
     :    name,': Unable to allocate workspace:',InitialMem,' bytes'
        write(LER,*)name,': ABNORMAL Termination'
        write(LER,*)' '

        call lbclos (luin1)
        if(nmap.eq.2) call lbclos (luin2)
        call lbclos (luin3)
        call lbclos (luout)
        stop

      else

        write(LERR,*)' '
        write(LERR,*)name,': Allocating workspace:',InitialMem,' bytes'
        write(LERR,*)' '

      endif

c
c  Load Maps into Memory
c
      tr_index = 1 - nx
      zmax = z0+(nz-1)*dz

      DO iy = 1, ny

        tr_index = tr_index + nx

        nbytes = 0
        call rtape(luin1, itr, nbytes)
        if(nbytes .eq. 0) then
          write(LERR,*)name,': Premature EOF on Map1 at trace ',iy
          call lbclos (luin1)
          if(nmap.eq.2) call lbclos (luin2)
          call lbclos (luin3)
          call lbclos (luout)
          stop
        endif
        call vmov ( itr(ITHWP1), 1, Map1(tr_index), 1, nx )

        if(nmap.eq.2) then
          nbytes = 0
          call rtape(luin2, itr, nbytes)
          if(nbytes .eq. 0) then
            write(LERR,*)name,': Premature EOF on Map2 at trace ',iy
            call lbclos (luin1)
            if(nmap.eq.2) call lbclos (luin2)
            call lbclos (luin3)
            call lbclos (luout)
            stop
          endif
          call vmov ( itr(ITHWP1), 1, Map2(tr_index), 1, nx )
        else
c         Create a map deeper than the greatest output depth
          call vfill ( 2.0*zmax, Map2(tr_index), 1, nx )
        endif

      ENDDO

c
c Now load the velocity trace
c
      nbytes = 0
      call rtape(luin3, itr, nbytes)
      if(nbytes .eq. 0) then
        write(LERR,*)name,': Error reading velocity trace'
        call lbclos (luin1)
        if(nmap.eq.2) call lbclos (luin2)
        call lbclos (luin3)
        call lbclos (luout)
        stop
      endif
      call vmov ( itr(ITHWP1), 1, Vel_in, 1, nz_vel )

      if (vtop.lt.0.0) vtop=Vel_in(1)
      if (vbot.lt.0.0) vbot=Vel_in(nz_vel)

c
c turn velocity into slowness prior to resampling
c
      do iz = 1,nz_vel
        Vel_in(iz) = 1.0/Vel_in(iz)
      enddo
      vtop = 1.0/vtop
      vbot = 1.0/vbot

c
c Create a smoothing function that allows the Vz to honor the surface
c its hung on but gradually gets smoother as the depth below that
c surface increases.
c
      if (usr_sm .gt. 0.0) then
        call make_smoothing_fxn
     :         (Map1,Map3, nx,ny, dx,dy, x0,y0, dz, usr_sm,usr_z,irsm,
     :          lcoef,slope_x,slope_y,zbias, verbos)
      else
        call vsadd ( Map1, 1, usr_z, Map3, 1, nx*ny )
      endif

c
c report coefficients of least-squares plane if warranted
c
      if (usr_sm.gt.0.0 .and. verbos .and. .not.lcoef) then
        write(LERR,*)' '
        write(LERR,*)
     :    ' Coefficients of least-squares plane through upper map'
        write(LERR,*)' '
        write(LERR,*)'    slope_x = ',slope_x
        write(LERR,*)'    slope_y = ',slope_y
        write(LERR,*)'    zbias   = ',zbias
        write(LERR,*)' '
      endif

c
c Loop over the map space filling a trace at a time, fixing a few
c headers and dumping the computed trace.
c
      if(verbos) then
        write(LERR,*)' '
        write(LERR,*)name,': Starting to process ',ny,
     :               ' records with ',nx,' traces.'
        write(LERR,*)' '
      endif

      do iy = 1,ny
        y_here = y0+(iy-1)*dy
        do ix = 1,nx
          x_here = x0+(ix-1)*dx
          ismp = nx*(iy-1)+ix

          call rsamp_vel_trc
     :           ( Vel_in, Map1(ismp), dz, nz_vel, Map3(ismp),
     :             Vel_out,        z0, dz,     nz, Map2(ismp),
     :             vtop,vbot, usr_z )

          do iz = 1,nz
            Vel_out(iz) = 1.0/Vel_out(iz)
          enddo

          call savew2 (itr,ifmt_RecNum,l_RecNum,ln_RecNum,
     :       iy,TRACEHEADER)
          call savew2 (itr,ifmt_TrcNum,l_TrcNum,ln_TrcNum,
     :       ix,TRACEHEADER)
          call savew2 (itr,ifmt_CDPBCX,l_CDPBCX,ln_CDPBCX,
     :       int(x_here),TRACEHEADER)
          call savew2 (itr,ifmt_CDPBCY,l_CDPBCY,ln_CDPBCY,
     :       int(y_here),TRACEHEADER)
          call savew2 (itr,ifmt_WDepDP,l_WDepDP,ln_WDepDP,
     :       int(Map1(ismp)),TRACEHEADER)

          call vmov (Vel_out, 1, itr(ITHWP1), 1, nz)
          call wrtape (luout, itr, obytes)

        enddo

        if(verbos) then
          write(LERR,*)name,':    Done with record ',iy,' of ',ny
        endif

      enddo

c close data files 

      call lbclos (luin1)
      if(nmap.eq.2) call lbclos (luin2)
      call lbclos (luin3)
      call lbclos (luout)
      write(LERR,*)name,': Normal Termination'
      write(LER,*)name,': Normal Termination'
      write(LER,*)' '

      stop
      end

c -----------------  Subroutine -----------------------
c provide terse online help [detailed help goes in man page]

      subroutine help(name)

      character  name*(*)
      integer nblen

#include <f77/iounit.h>

      write(LER,*)' '
      write(LER,*)'===================================================='
      write(LER,*)' '
      write(LER,*)' Command Line Arguments for ',name(1:nblen(name))
      write(LER,*)' '
      write(LER,*)'    hang a V(z) function on a map to create'
      write(LER,*)'    a V(xyz) volume'
      write(LER,*)' '
      write(LER,*)'Input..................................... (default)'
      write(LER,*)' '
      write(LER,*)'-map1[]     -- map surface to hang Vz on     (stdin)'
      write(LER,*)'-map2[]     -- map surface at base of Vz  (optional)'
      write(LER,*)'-vz[]       -- Vz trace to resample/shift     (none)'
      write(LER,*)'-O[]        -- output 3D velocity volume    (stdout)'
      write(LER,*)' '
      write(LER,*)'-z0[z0]     -- Z origin of output              (0.0)'
      write(LER,*)'-x0[x0]     -- X origin of output              (0.0)'
      write(LER,*)'-y0[y0]     -- Y origin of output              (0.0)'
      write(LER,*)' '
      write(LER,*)'-dz[dz]     -- delta Z of output              (10.0)'
      write(LER,*)'-dx[dx]     -- delta X of output              (25.0)'
      write(LER,*)'-dy[dy]     -- delta Y of output              (25.0)'
      write(LER,*)' '
      write(LER,*)'-nz[nz]     -- Number of samples in output (nsmp_vz)'
      write(LER,*)' '
      write(LER,*)'-vtop[vtop] -- Velocity above map1 (first Vz sample)'
      write(LER,*)'-vbot[vbot] -- Velocity below map2  (last Vz sample)'
      write(LER,*)' '
      write(LER,*)'-sm[usr_sm] -- realtive amount of     (no smoothing)'
      write(LER,*)'               smoothing to apply'
      write(LER,*)'               as depth below map1'
      write(LER,*)'               increases: <=0 == no smoothing'
      write(LER,*)'                          1.0 == maximum smoothing'
      write(LER,*)' '
      write(LER,*)'-zsm[usr_z] -- Depth within Vz that is     (zmax_Vz)'
      write(LER,*)'               forced to follow a smooth'
      write(LER,*)'               surface made with -sm'
      write(LER,*)' '
      write(LER,*)'-rsm[irsm]  -- Areal smoother size adjustment   (25)'
      write(LER,*)'               The smoothing fxn applies a 5x5 pt'
      write(LER,*)'               smoother to map1. The XY dimensions'
      write(LER,*)'               of the smoother are increased by 5'
      write(LER,*)'               for each irsm depth points in usr_z'
      write(LER,*)' '
      write(LER,*)'-coef       -- Get coefficients for a      (.false.)'
      write(LER,*)'               least-squares plane through -map1'
      write(LER,*)'               from command line arguments.'
      write(LER,*)' '
      write(LER,*)'-slope_x[slope_x] -- Coefficient for plane     (0.0)'
      write(LER,*)'-slope_y[slope_y] -- Coefficient for plane     (0.0)'
      write(LER,*)'-zbias[zbias]     -- Coefficient for plane     (0.0)'
      write(LER,*)' '
      write(LER,*)'Usage:'
      write(LER,*)'       hangvz -map1[] -vz[] -O[]'
      write(LER,*)'              [ -z0[] -x0[] -y0[] -dz[] -dx[] -dy[]'
      write(LER,*)'                -nz[] -vtop[] -vbot[]'
      write(LER,*)'                -sm[] -zsm[] -rsm[] -map2[]'
      write(LER,*)'                -coef -slope_x[] -slope_y[] -zbias[]'
      write(LER,*)'                -V ]'
      write(LER,*)' '
      write(LER,*)'===================================================='
      
      return
      end

c -----------------  Subroutine -----------------------
c pick up command line arguments 

      subroutine cmdln
     :           ( map1file, map2file, velfile, otap,
     :             z0,x0,y0, dz,dx,dy, nz, vtop,vbot,
     :             usr_sm,usr_z,irsm, slope_x,slope_y,zbias,lcoef,
     :             name, verbos )

#include <f77/iounit.h>

      integer    argis, nz
      real       z0,x0,y0, dz,dx,dy, vtop,vbot, usr_sm,usr_z
      real       slope_x,slope_y,zbias
      integer    irsm
      character  map1file*(*),map2file*(*),velfile*(*)
      character  otap*(*), name*(*)
      logical    verbos,lcoef

      call argstr ('-map1', map1file, ' ', ' ') 
      call argstr ('-map2', map2file, ' ', ' ') 
      call argstr ('-vz', velfile, ' ', ' ') 
      call argstr ('-O', otap, ' ', ' ') 
      call argr4  ('-z0', z0,  0.0,  0.0)
      call argr4  ('-x0', x0,  0.0,  0.0)
      call argr4  ('-y0', y0,  0.0,  0.0)
      call argr4  ('-dz', dz, 10.0, 10.0)
      call argr4  ('-dx', dx, 25.0, 25.0)
      call argr4  ('-dy', dy, 25.0, 25.0)
      call argi4  ('-nz', nz, -1, -1)
      call argr4  ('-vtop', vtop, -1.0, -1.0)
      call argr4  ('-vbot', vbot, -1.0, -1.0)
      call argr4  ('-sm', usr_sm, -1.0, -1.0)
      call argr4  ('-zsm', usr_z, -1.0, -1.0)
      call argi4  ('-rsm',irsm, 25, 25)

      lcoef = .false.
      lcoef = (argis('-coef') .gt. 0)
      call argr4  ('-slope_x', slope_x, 0.0, 0.0)
      call argr4  ('-slope_y', slope_y, 0.0, 0.0)
      call argr4  ('-zbias'  ,   zbias, 0.0, 0.0)

      verbos = .false.
      verbos = (argis('-V') .gt. 0)

      if (usr_sm .gt. 1.0) usr_sm=1.0

c check for extraneous arguments and abort if found to
c catch all manner of user typo's

      call xtrarg (name, ler, .FALSE., .FALSE.)
      call xtrarg (name, lerr, .FALSE., .TRUE.)

      return
      end

c -----------------  Subroutine -----------------------

c verbal printout of pertinent program particulars


      subroutine verbal
     :       ( map1file,map2file,velfile,otap,
     :         nx1,ny1, nx2,ny2, nz_vel,
     :         z0,x0,y0, nz,nx,ny, dz,dx,dy,
     :         vtop,vbot, usr_sm,usr_z,
     :         slope_x,slope_y,zbias,lcoef,
     :         verbos )

#include <f77/iounit.h>

      character map1file*(*), map2file*(*), velfile*(*), otap*(*)
      integer   nx1,ny1, nx2,ny2, nz_vel, nz,nx,ny
      real      z0,x0,y0, dz,dx,dy, vtop,vbot, usr_sm,usr_z
      real      slope_x,slope_y,zbias
      logical   verbos,lcoef
      integer   len1,len2,len3,len4
      integer   nblen

      len1 = nblen(map1file)
      len2 = nblen(map2file)
      len3 = nblen(velfile)
      len4 = nblen(otap)

      write(LERR,*)' '
      write(LERR,*)' Input Line Header Parameters'
      write(LERR,*)' '
      write(LERR,*) ' Input map to hang Vz on = ', map1file(1:len1)
      write(LERR,*) '   Number of samples     = ', nx1
      write(LERR,*) '   Number of traces      = ', ny1
      write(LERR,*)' '
      if ( nmap .eq. 2 ) then
        write(LERR,*) ' Input map at base of Vz = ', map2file(1:len2)
        write(LERR,*) '   Number of samples     = ', nx2
        write(LERR,*) '   Number of traces      = ', ny2
        write(LERR,*)' '
      endif
      write(LERR,*) ' Vz trace                = ', velfile(1:len3)
      write(LERR,*) '   Number of samples     = ', nz_vel
      write(LERR,*)' '
      write(LERR,*) ' Output velocity volume  = ', otap(1:len4)
      write(LERR,*) '   origin.z   = ', z0
      write(LERR,*) '   origin.x   = ', x0
      write(LERR,*) '   origin.y   = ', y0
      write(LERR,*) '   size.z   (NumSmp)  = ', nz
      write(LERR,*) '   size.x   (NumTrc)  = ', nx
      write(LERR,*) '   size.y   (NumRec)  = ', ny
      write(LERR,*) '   delta.z  (Dz1000)/1000  = ', dz
      write(LERR,*) '   delta.x  (Dx1000)/1000  = ', dx
      write(LERR,*) '   delta.y  (Dy1000)/1000  = ', dy
      write(LERR,*)' '
      if ( vtop .gt. 0.0 ) then
        write(LERR,*) ' Fill velocity above top map   = ', vtop
      endif
      if ( nmap .eq. 2 .and. vbot .gt. 0.0 ) then
        write(LERR,*) ' Fill velocity below basal map = ', vbot
      endif
      write(LERR,*)' '
      write(LERR,*) ' Smoother value            = ', usr_sm
      write(LERR,*) ' Depth of smoothed surface = ', usr_z
      write(LERR,*)' '
      if ( lcoef ) then
        write(LERR,*) ' User supplied least-squares plane'
        write(LERR,*) '   slope_x  = ',slope_x
        write(LERR,*) '   slope_y  = ',slope_y
        write(LERR,*) '   zbias    = ',zbias
        write(LERR,*)' '
      endif
      write(LERR,*)' '
      if ( verbos ) then
        write(LERR,*) ' verbose printout requested'
        write(LERR,*)' '
      endif
      write(LERR,*)'================================================== '
      write(LERR,*)' '

      return
      end
