c...|....1....|....2....|....3....|....4....|....5....|....6....|....7..
C***********************************************************************
C Copyright 2001, Allied Geophysics, Inc. All Rights Reserved          *
C***********************************************************************
C Portions of this code and/or subroutines  used by this code are      *
C protected by the following copyright(s):                             *
C***********************************************************************
C                 copyright 2001, Amoco Production Company             *
C                             All Rights Reserved                      *
C                     an affiliate of BP America Inc.                  *
C***********************************************************************
c -----------------  Main Routine -----------------------
c
c mapextract - extract a map of the first occurence of a value in each
c              trace of a data volume
c
c  Program Description:
c
c     mapextract scans each trace of an input volume for a user defined
c     value. A map is created incorporating user overrides for the
c     origin.axis.1 and delta.axis.1 Traces can be scanned from front-
c     to-back or from back-to-front. In its original form, mapextract
c     looks for a specific value, but may soon allow detection of a
c     range of values or some other property of the input data. Places
c     in the map where no value can be found are filled with -emask,
c     an embedded mask.
c
c  Program Changes:
c    Oct 25, 2001 - original version
c    Oct 26, 2001 - Complete rewrite... buggered yesterdays file
c    Oct 30, 2001 - adopted emask as name for the embedded mask
c    Dec 28, 2001 - minor cleanup of code and man page
c    Jan 21, 2002 - implicit none in main and subs, defined indicated variables - PGAG
c                   
c

c get machine dependent parameters 

      implicit none

#include <f77/iounit.h>
#include <f77/lhdrsz.h>
#include <f77/sisdef.h>
#include <save_defs.h> 

c dimension standard USP variables 

      integer     itr(3*SZLNHD)
     
      integer     lbytes, nbytes, lbyout, obytes
      integer     luin, luout
      integer     argis, JERR

      character   ntap*255, otap*255, name*10

      logical     verbos

c Program Specific _ dynamic memory variables

      integer MapSize, TrcSize, InitialMem
      integer errcd1, errcd2, abort

      real    Map(2), Trc(2)

      pointer (ptr_Map, Map)
      pointer (ptr_Trc, Trc)

c Program Specific _ static memory variables

      integer map_index, i2,i3, id1, isamp
      integer n1_vol,n2_vol,n3_vol, n1_map,n2_map
      integer nblen, n1, n2, n3
      real    d1, val, z0,dz, emask
      character dir*(3)
      logical lfwd

c Variables for access to RecNum and TrcNum trace headers
      integer ifmt_RecNum,l_RecNum,ln_RecNum
      integer ifmt_TrcNum,l_TrcNum,ln_TrcNum

c Initialize variables

      data abort/0/
      data name/"MAPEXTRACT"/

c
c give command line help if requested
c
      if ( argis ( '-?' )    .gt. 0 .or. 
     :     argis ( '-h' )    .gt. 0 .or.
     :     argis ( '-help' ) .gt. 0 ) then
         call help(name)
         stop
      endif

c
c open printout file
c
#include <f77/open.h>

c
c get command line input parameters
c
      call cmdln ( ntap,otap, val, z0,dz, dir,
     :             emask, name, verbos )

      lfwd = .true.
      if (dir(1:3).eq.'rev') lfwd = .false.

c
c open input and output files
c
      call getln(luin, ntap,'r', 0)

c
c read the Line header info from inputs and check for inconsistencies
c
      call rtape(luin,itr,lbytes)
      if(lbytes.eq.0)then
        write(LER,*)name,
     :    ': no line header on input file ',ntap(1:nblen(ntap))
        write(LER,*)'FATAL'
        stop
      endif

      call saver(itr, 'NumSmp', n1_vol, LINHED)
      call saver(itr, 'NumTrc', n2_vol, LINHED)
      call saver(itr, 'NumRec', n3_vol, LINHED)
      call saver(itr, 'Dz1000', id1, LINHED)

      d1 = float(id1)*0.001
      if (dz.gt.0.0) d1 = dz

      n1_map = n2_vol
      n2_map = n3_vol

c
c print HLH to printout file
c
      call hlhprt (itr, lbytes, name, 4, LERR)

c
c verbose output of all pertinent information before processing begins
c
      call verbal ( ntap,otap, dir, val, z0,d1,
     :              n1,n2,n3, emask, verbos )

c
c dynamic memory allocation:
c
      MapSize = n1_map*n2_map*SZSMPD
      TrcSize = SZTRHD + SZSMPD*(max(n1_vol,n1_map))
      InitialMem = 2*MapSize+TrcSize

      call galloc (ptr_Map,MapSize,errcd1,abort)
      call galloc (ptr_Trc,TrcSize,errcd2,abort)
    
      if (errcd1.ne.0 .or. errcd2.ne.0) then

        write(LERR,*)' '
        write(LERR,*)
     :    name,': Unable to allocate workspace:',InitialMem,' bytes'
        write(LERR,*)name,': ABNORMAL Termination'
        write(LERR,*)' '

        write(LER,*)' '
        write(LER,*)
     :    name,': Unable to allocate workspace:',InitialMem,' bytes'
        write(LER,*)name,': ABNORMAL Termination'
        write(LER,*)' '

        call lbclos (luin)
        stop

      else

        write(LERR,*)' '
        write(LERR,*)name,': Allocating workspace:',InitialMem,' bytes'
        write(LERR,*)' '

      endif

c
c initialize Map() with emask
c
      call vfill(emask,Map,1,n1_map*n2_map)

c
c Loop over the input volume in search of val
c
      if(verbos) then
        write(LERR,*)' '
        write(LERR,*)name,': Starting to process ',n3_vol,
     :               ' records with ',n2_vol,' traces.'
        write(LERR,*)' '
      endif

      map_index = 0
      do i3 = 1,n3_vol
        do i2 = 1,n2_vol

c         read a trace
          nbytes = 0
          call rtape(luin,Trc,nbytes)
          if(nbytes.eq.0) then
            write(LERR,*)name,': Premature EOF on input at record ',i3,
     :                   ' trace ',i2
            call lbclos (luin)
            stop
          endif

c         search for the users value
          map_index = map_index + 1
          call find_val (Trc(ITHWP1),n1_vol,val,isamp,lfwd)
          if (isamp.gt.0) Map(map_index)=z0+d1*(isamp-1)

        enddo

        if(verbos) then
          write(LERR,*)name,':    Done with record ',i3,' of ',n3_vol
        endif

      enddo

c
c Prepare for output and write the map
c
c open output file
c
      call getln(luout, otap,'w', 1)

c
c modify line header to reflect new sizes
c
      call savew(itr, 'NumSmp', n1_map, LINHED)
      call savew(itr, 'NumTrc', n2_map, LINHED)
      call savew(itr, 'NumRec', 1, LINHED)

c
c save out hlh and line header
c
      call savhlh (itr, lbytes, lbyout)
      call wrtape (luout, itr, lbyout)

c
c Get the trace header info for RecNum and TrcNum so we can adjust
c output numbering
c
      call savelu('RecNum',ifmt_RecNum,l_RecNum,ln_RecNum,TRACEHEADER)
      call savelu('TrcNum',ifmt_TrcNum,l_TrcNum,ln_TrcNum,TRACEHEADER)

c
c Fix RecNum now since it will only be 1.
c
      call savew2(Trc,ifmt_RecNum,l_RecNum,ln_RecNum,1,TRACEHEADER)

c
c Number output bytes in a trace
c
      obytes = SZTRHD + SZSMPD*n1_map

c
c Now loop over output traces writing the map
c
      map_index = 1-n1_map
      do i2 = 1,n2_map
        map_index = map_index+n1_map
        call savew2(Trc,ifmt_TrcNum,l_TrcNum,ln_TrcNum,i2,TRACEHEADER)
        call vmov(Map(map_index),1,Trc(ITHWP1),1,n1_map)
        call wrtape(luout,Trc,obytes)
      enddo

c close data files 

      call lbclos (luin)
      call lbclos (luout)
      write(LERR,*)name,': Normal Termination'
      write(LER,*)name,': Normal Termination'
      write(LER,*)' '

      stop
      end

c -----------------  Subroutine -----------------------
c provide terse online help [detailed help goes in man page]

      subroutine help(name)

      character  name*(*)
      integer nblen, len1

#include <f77/iounit.h>

      len1 = nblen(name)

      write(LER,*)' '
      write(LER,*)'===================================================='
      write(LER,*)' '
      write(LER,*)' Command Line Arguments for ',name(1:len1)
      write(LER,*)'    Extract a map of the first occurence of a value'
      write(LER,*)'            in each trace of a data volume'
      write(LER,*)' '
      write(LER,*)'Input..................................... (default)'
      write(LER,*)' '
      write(LER,*)'-N[]      -- Input volume                   (stdin:)'
      write(LER,*)'-O[]      -- Output map                    (stdout:)'
      write(LER,*)' '
      write(LER,*)'-val[]    -- Value to scan for                (none)'
      write(LER,*)' '
      write(LER,*)'-z0[]     -- Z axis origin of volume           (0.0)'
      write(LER,*)'-dz[]     -- Delta Z override          (1e-3*Dz1000)'
      write(LER,*)' '
      write(LER,*)'-dir[]    -- Direction of scan                 (fwd)'
      write(LER,*)'               fwd - scan forward along each trace'
      write(LER,*)'               rev -      backward'
      write(LER,*)' '
      write(LER,*)'-emask[]  -- Value to indicate undefined    (-1e+37)'
      write(LER,*)'             regions in map. -emask is the value'
      write(LER,*)'             of the map if -val is not found'
      write(LER,*)' '
      write(LER,*)'Usage:'
      write(LER,*)'    mapextract -N[] -O[] -val[]'
      write(LER,*)'               -z0[] -dz[] -dir[] -emask[] -V'
      write(LER,*)' '
      write(LER,*)'===================================================='
      
      return
      end

c -----------------  Subroutine -----------------------
c pick up command line arguments 

      subroutine cmdln ( ntap,otap, val, z0,dz, dir,
     :                   emask, name, verbos )

      implicit none

#include <f77/iounit.h>

      integer    argis
      real       val, z0,dz, emask
      character  ntap*(*), otap*(*), dir*(*), name*(*)
      logical    verbos

      call argstr ('-N', ntap, ' ', ' ') 
      call argstr ('-O', otap, ' ', ' ') 
      call argstr ('-dir', dir, 'fwd', 'fwd') 

      call argr4  ('-val',val,-1.0e+37,-1.0e+37)

      call argr4  ('-z0', z0, 0.0, 0.0)
      call argr4  ('-dz', dz, 0.0, 0.0)

      call argr4  ('-emask',emask,-1.0e+37,-1.0e+37)

      verbos = (argis('-V') .gt. 0)

c
c now look for values passed in with alternative style
c
      call argstr ('in=', ntap, ntap, ntap) 
      call argstr ('out=', otap, otap, otap) 
      call argstr ('dir=', dir, dir, dir) 
      call argr4  ('val=', val, val, val)
      call argr4  ('z0=', z0, z0, z0)
      call argr4  ('dz=', dz, dz, dz)
      call argr4  ('emask=', emask, emask, emask)
      if (.not.verbos) verbos = (argis('verbose=') .gt. 0)

c
c verify that the user has supplied something for -val
c
      if ( val .eq. -1.0e+37 ) then
        write(LER,*)name,': Parameter -val is missing - FATAL'
        stop
      endif

c check for extraneous arguments and abort if found to
c catch all manner of user typo's

      call xtrarg (name, LER,  .FALSE., .FALSE.)
      call xtrarg (name, LERR, .FALSE., .TRUE.)

      return
      end

c -----------------  Subroutine -----------------------

c verbal printout of pertinent program particulars

      subroutine verbal
     :       ( ntap,otap, dir, val, z0,d1,
     :         n1,n2,n3, emask, verbos )

      implicit none

#include <f77/iounit.h>

      character ntap*(*), otap*(*), dir*(*)
      integer   n1,n2,n3
      real      val, z0,d1, emask
      logical   verbos
      integer   len1,len2
      integer   nblen

      len1 = nblen(ntap)
      len2 = nblen(otap)

      write(LERR,*)' '
      write(LERR,*)' Input Line Header Parameters'
      write(LERR,*)' '
      if(len1.gt.0) then
        write(LERR,*) ' Input data volume       = ', ntap(1:len1)
      else
        write(LERR,*) ' Input data volume       = stdin:'
      endif
      write(LERR,*) '   Number of samples     = ', n1
      write(LERR,*) '   Number of traces      = ', n2
      write(LERR,*) '   Number of records     = ', n3
      write(LERR,*) '   Sample axis origin    = ', z0
      write(LERR,*) '   Sample increment      = ', d1
      write(LERR,*)' '
      if(len2.gt.0) then
        write(LERR,*) ' Output map              = ', otap(1:len2)
      else
        write(LERR,*) ' Output map              = stdout:'
      endif
      write(LERR,*) '   Number of samples     = ', n2
      write(LERR,*) '   Number of traces      = ', n3
      write(LERR,*)' '
      write(LERR,*) ' Value to search for   val = ', val
      write(LERR,*) ' Direction of search   dir = ', dir
      write(LERR,*)' '
      if ( verbos ) then
        write(LERR,*) ' verbose printout requested'
        write(LERR,*)' '
      endif
      write(LERR,*)'================================================== '
      write(LERR,*)' '

      return
      end
