C***********************************************************************
C                 copyright 2001, Amoco Production Company             *
C                             All Rights Reserved                      *
C                     an affiliate of BP America Inc.                  *
C***********************************************************************
c 
c  routine to aid in calculation of bin center corner coordinates
c  for subset swath volumes of a large 3D survey.  If there is
c  not enough disk available to process the whole survey in a single
c  piece it is necessary to process in a number of swaths.  Each
c  swath bin center grid must fit exactly on the global grid for the
c  survey to make merging of the swaths post-stack a painless 
c  endevour.  This routine will calculate corner coordinates for any
c  number of sub-swaths which make this happen.

c Changes:

c October 4, 1999: - made routine double precision and corrected
c                    the length calculation for side 1-2 and side
c                    1-4 to be numdi + 1 and numli +1 times the 
c                    appropriate interval
c Garossino

      implicit none

#include <f77/iounit.h>
#include <f77/lhdrsz.h>

      integer NumRec, NumTrc, NumSmp, NumLI, NumDI, OverLap, i
      integer AvailableDisk, NumSwaths
      integer NumLI_PerSwath, argis, luxgraph, length, jerr, lenth

      real il_azimuth, cl_azimuth, ildm, cldm
      real SurveySize, DiskInBytes

      real*8 pi, deg2rad 
      real*8 theta, X_factor_12, Y_factor_12, X_factor_14, Y_factor_14
      real*8 length_12, length_14
      real*8 global_delta_X, global_delta_Y
      real*8 delta_X_12, delta_X_14
      real*8 delta_Y_12, delta_Y_14
      real*8 x1,  y1
      real*8 swath_x1, swath_y1, swath_x2, swath_y2, swath_x3, swath_y3
      real*8 swath_x4, swath_y4

      character c_swath*8, graphtap*255, name*12
      character*3 c_x1, c_x2, c_x3, c_x4, c_y1, c_y2, c_y3, c_y4

      logical verbos

c initialize variables

      data c_swath/'" Swath '/
      data name/'SWATH_DESIGN'/
      data c_x1/'-x1'/
      data c_x2/'-x2'/
      data c_x3/'-x3'/
      data c_x4/'-x4'/
      data c_y1/'-y1'/
      data c_y2/'-y2'/
      data c_y3/'-y3'/
      data c_y4/'-y4'/
      pi = 4.0 * datan ( dble(1.0) )
      deg2rad = pi/dble(180.)

c get command line help if necessary

      if ( argis ( '-?' ) .gt. 0 .or.
     :     argis ( '-h' ) .gt. 0 .or.
     :     argis ( '-help' ) .gt. 0 ) then
         call help()
         stop
      endif

c printout        

#include <f77/open.h>

c parse command line parameters

      call cmdln ( x1, y1, il_azimuth, cl_azimuth, NumRec, 
     :     NumTrc, NumSmp, NumLI, NumDI, OverLap, AvailableDisk, 
     :     ildm, cldm, graphtap, verbos )

c open xgraph file if requested

      if ( graphtap .ne. ' ' ) then
         length = lenth(graphtap)
         call alloclun ( luxgraph )
         open ( luxgraph, file=graphtap(1:length), status='unknown', 
     :        err=990)
      endif

c calculate the number of swaths required to process the data given the 
c SwathSize constraint

      SurveySize = float ( NumTrc * NumRec ) * 
     :     float ( SZTRHD + SZSMPD * NumSmp )
      DiskInBytes = float ( AvailableDisk ) * 1024. * 1024. * 1024.
      NumSwaths = nint(SurveySize / DiskInBytes) 
      if ( amod ( SurveySize, DiskInBytes ) .gt. 1.e-30 ) 
     :     NumSwaths = NumSwaths + 1
      
c determine number of LI's in each swath taking into account OverLap

      NumLI_PerSwath = (NumLI + (NumSwaths - 1 ) * OverLap ) / NumSwaths

      if ( mod ( (NumLI + (NumSwaths - 1 ) * OverLap ), NumSwaths ) 
     :     .gt. 0 ) NumLI_PerSwath = NumLI_PerSwath + 1

c echo key parameters to printout file

      call verbal ( x1, y1, il_azimuth, cl_azimuth, NumRec, 
     :     NumTrc, NumSmp, NumLI, NumDI, OverLap, AvailableDisk, 
     :     ildm, cldm, graphtap, NumSwaths, NumLI_PerSwath, verbos )

c based on in-line azimuth [il_azim] determine the x and y projection
c factors to be used to calculate orthogonal vertices.
      
      if ( il_azimuth .ge. 0.0 .and. il_azimuth .le. 90.0 ) then

C QUADRANT 1,  0 - 90 degrees

         theta = ( dble(90.0) - dble(il_azimuth) ) * deg2rad 
         
         X_factor_12 = dcos ( theta )
         X_factor_14 = dsin ( theta )
         Y_factor_12 = dsin ( theta )
         Y_factor_14 = dcos ( theta )
         
         if ( cl_azimuth .ge. 90. .and. cl_azimuth .le. 180. ) then
            Y_factor_14 = Y_factor_14 * ((-1.0))
         elseif ( cl_azimuth .ge. 270. .and. cl_azimuth .lt. 360. ) then
            X_factor_14 = X_factor_14 * ((-1.0))
         endif
         
      elseif ( il_azimuth .gt. 90.0 .and. il_azimuth .le. 180.0 ) then
            
C QUADRANT 2,  91 - 180 degrees 

         theta = ( dble(180.0) - dble(il_azimuth) ) * deg2rad 
         
         X_factor_12 = dsin ( theta )
         X_factor_14 = dcos ( theta )
         Y_factor_12 = dcos ( theta )
         Y_factor_14 = dsin ( theta )
         
         if ( cl_azimuth .ge. 0. .and. cl_azimuth .le. 90. ) then
            Y_factor_12 = Y_factor_12 * (-1.0)
         elseif ( cl_azimuth .gt. 180. .and. cl_azimuth .le. 270. ) then
            X_factor_14 = X_factor_14 * (-1.0)
            Y_factor_12 = Y_factor_12 * (-1.0)
            Y_factor_14 = Y_factor_14 * (-1.0)
         endif

      elseif ( il_azimuth .gt. 180.0 .and. il_azimuth .le. 270.0 ) then

C QUADRANT 3, 181 - 270 degrees

         theta = ( dble(270.0) - dble(il_azimuth) ) * deg2rad 
         
         X_factor_12 = dcos ( theta )
         X_factor_14 = dsin ( theta )
         Y_factor_12 = dsin ( theta )
         Y_factor_14 = dcos ( theta )
         
         if ( cl_azimuth .ge. 270. .and. cl_azimuth .le. 360. ) then
            X_factor_12 = X_factor_12 * dble(-1.0)
            X_factor_14 = X_factor_14 * dble(-1.0)
            Y_factor_12 = Y_factor_12 * dble(-1.0)
         elseif ( cl_azimuth .ge. 90. .and. cl_azimuth .le. 180. ) then
            X_factor_12 = X_factor_12 * dble(-1.0)
            Y_factor_12 = Y_factor_12 * dble(-1.0)
            Y_factor_14 = Y_factor_14 * dble(-1.0)
         endif
         
      elseif ( il_azimuth .gt. 270.0 .and. il_azimuth .lt. 360.0 ) then

c QUADRANT 4, 271 to 359 degrees

         theta = ( dble(360.0) - dble(il_azimuth) ) * deg2rad 
         
         X_factor_12 = dsin ( theta )
         X_factor_14 = dcos ( theta )
         Y_factor_12 = dcos ( theta )
         Y_factor_14 = dsin ( theta )
         
         if ( cl_azimuth .ge. 0. .and. cl_azimuth .le. 90. ) then
            X_factor_12 = X_factor_12 * dble(-1.0)
         elseif ( cl_azimuth .ge. 180. .and. cl_azimuth .le. 270. ) then
            X_factor_12 = X_factor_12 *dble (-1.0)
            X_factor_14 = X_factor_14 * dble(-1.0)
            Y_factor_14 = Y_factor_14 *dble (-1.0)
         endif
         
      else
         
         write(LERR,*)' Your in-line azimuth must be between'
         write(LERR,*)' zero and 360 degrees. i.e:'
         write(LERR,*)'  '
         write(LERR,*)'        0 <= azimuth < 360'
         write(LERR,*)'  '
         write(LERR,*)' Your in-line azimuth of ',il_azimuth
         write(LERR,*)' is not acceptable.  Please correct'
         write(LERR,*)' your command line entry and retry'
         write(LERR,*)'FATAL'
         write(LER,*)' '
         write(LER,*)'DSWATH:'
         write(LER,*)' Your in-line azimuth must be between'
         write(LER,*)' zero and 360 degrees. i.e:'
         write(LER,*)'  '
         write(LER,*)'        0 <= azimuth < 360'
         write(LER,*)'  '
         write(LER,*)' Your in-line azimuth of ',il_azimuth
         write(LER,*)' is not acceptable.  Please correct'
         write(LER,*)' your command line entry and retry'
         write(LERR,*)'FATAL'
         stop
      endif

c given the stated (x1,y1) coordinate and assuming the 1-2 side
c to be in the direction of the acquisition sail line build all
c required swath coordinates and report.

      length_12 = dfloat (NumDI) * dble(ildm)
      length_14 = dfloat(NumLI_PerSwath) * dble(cldm)
      
      do i = 1, NumSwaths

C LOCATE VERTEX 1

         if ( i .gt. 1 ) then

c project position of corner 1 for this swath

            global_delta_X = dfloat(i-1) * 
     :           dfloat(NumLI_PerSwath - OverLap ) * dble(cldm)  * 
     :           X_factor_14
            global_delta_Y = dfloat(i-1) * 
     :           dfloat(NumLI_PerSwath - OverLap ) * dble(cldm)  * 
     :           Y_factor_14
            swath_x1 = x1 + global_delta_X
            swath_y1 = y1 + global_delta_Y

         else

c use original corner 1

            swath_x1 = x1
            swath_y1 = y1

         endif

C LOCATE swath corner 2 

         delta_X_12 = length_12 * X_factor_12
         delta_Y_12 = length_12 * Y_factor_12
         swath_x2 = swath_x1 + delta_X_12
         swath_y2 = swath_y1 + delta_Y_12

c LOCATE swath corner 3

         delta_X_14 = length_14 * X_factor_14
         delta_Y_14 = length_14 * Y_factor_14
         swath_x3 = swath_x2 + delta_X_14
         swath_y3 = swath_y2 + delta_Y_14

c locate swath corner 4

         swath_x4 = swath_x1 + delta_X_14
         swath_y4 = swath_y1 + delta_Y_14

c report corner coordinates for swath

c to printout file in command line format

         write(LERR,*)' '
         write(LERR,*)' Swath ',i
         write(LERR,*)' '
         write(LERR,10) c_x1, swath_x1, c_y1, swath_y1
         write(LERR,10) c_x2, swath_x2, c_y2, swath_y2
         write(LERR,10) c_x3, swath_x3, c_y3, swath_y3
         write(LERR,10) c_x4, swath_x4, c_y4, swath_y4
 10      format(2(a3,f17.4,1x))

c to screen

         if ( verbos ) then
            write(LER,*)' '
            write(LER,*)' Swath ',i
            write(LER,*)' '
            write(LER,10) c_x1, swath_x1, c_y1, swath_y1
            write(LER,10) c_x2, swath_x2, c_y2, swath_y2
            write(LER,10) c_x3, swath_x3, c_y3, swath_y3
            write(LER,10) c_x4, swath_x4, c_y4, swath_y4
         endif

c to xgraph file

         if ( graphtap .ne. ' ' ) then

            write(luxgraph,'(/)')
            write(luxgraph,'(a8,i5)') c_swath, i
            write(luxgraph,*) swath_x1, swath_y1
            write(luxgraph,*) swath_x2, swath_y2
            write(luxgraph,*) swath_x3, swath_y3
            write(luxgraph,*) swath_x4, swath_y4
            write(luxgraph,*) swath_x1, swath_y1
         endif

      enddo


      write(LERR,*)' '
      write(LERR,*)' Normal Termination'
      write(LER,*)' '
      write(LER,*)' dswath: Normal Termination'
      stop

 990  continue
      
      write(LERR,*)' '      
      write(LERR,*)' Error opening Xgraph output file'
      write(LERR,*)' Check permissions/spelling and resubmit'
      write(LERR,*)'FATAL'      
      write(LER,*)' '      
      write(LER,*)'DSWATH:'      
      write(LER,*)' Error opening Xgraph output file'
      write(LER,*)' Check permissions/spelling and resubmit'
      write(LER,*)'FATAL'  
      stop
      end

         

