C***********************************************************************
C                 copyright 2001, Amoco Production Company             *
C                             All Rights Reserved                      *
C                     an affiliate of BP America Inc.                  *
C***********************************************************************
C**********************************************************************C
C
C     PROGRAM MODULE vfilt3d
C**********************************************************************C
C
C vfilt3d reads the frequency sliced data [ as output by fftpack ] , 
c transforms it, filters using a variety of velocity, azimuth and frequency
c parameters with tapers , and does the inverse transform.
C
C Original Written by D. A. Yanchak
c USP version and embellishments by P.G.A. Garossino
c
c Embelishments:
c 
c command line upgrades:
c
c     -v1     ...........             In all cases the taper occurs between
c     -v2    .           .            the 1 --> 2  and 3 --> 4 parameters with      
c     -v3   .             .           full pass or reject between 2 --> 3
c     -v4  .               .     
c          1  2         3  4
c
C
c     -a1     ...........             [v] specifies conic velocity
c     -a2    .           .            [a] aperture azimuth in kx,ky [0 is North]
c     -a3   .             .           [f] specifies temporal frequency
c     -a4  .               .     
c          1  2         3  4
c
C
c     -f1     ...........
c     -f2    .           .                  
c     -f3   .             .       
c     -f4  .               .     
C
c          1  2         3  4
c
c
c     -dt if present defines temporal sample interval in dataset going into fftpack
c
c     -dz if present indicates that input dataset is DEPTH and all other parameters
c         should be treated appropriately
c
c     -pass if present pass data as defined rather than reject.
c
c     if no variable is explicitly defined on the command line that option is left
c     wide open.  [i.e.  if no -a parameters ALL azimuths are passed or rejected.
c
c CHANGES:
c
c
c     Jun 9, 1998: fixed -dz option to use meters or feet not kilofeet.  This allows
c                  velocity parameterization to use 1 / tan(deg) with steepest angle
c                  from the horizontal going into v1 and shallowest into v4.
c     Garossino
c
c 
c
C
C     DECLARE VARIABLES
C
#include <save_defs.h>
#include <localsys.h>
#include <f77/iounit.h>
#include <f77/sisdef.h>
#include <f77/lhdrsz.h>

c Standard USP variable

      integer lhed(SZLNHD)
      integer luin, luout, nbytes, obytes, argis
      integer nsamp, nrec, ntrc, nfreq, iform 
      integer lbytes, lbyout
      integer nsampo, ntrco
      
      character   name * 7, ntap * 256, otap * 256

      logical verbos

c variables used with dynamic memory allocation

      integer errcd1, errcd2, abort, itemHeader, itemSpace

      integer headers

      real work

      pointer ( wkadr1, headers(200000) )
      pointer ( wkadr2, work(200000) )

c program dependant variables

      integer NextPowerOf2, size, JJ, KK, KJ, KI, KIt, KJt
      integer RecNum, l_RecNum, ifmt_RecNum, ln_RecNum
      integer trndx, hdrndx, ky
      integer ordfft

      real    dt, dx, dy, dz 
      real    v1, v2, v3, v4
      real    a1, a2, a3, a4
      real    f1, f2, f3, f4
      real    scal, pi, deltaF, factor, dkx, dky
      real    Taper, Ftaper

      logical pass

c initialize variables

      data  nbytes / 0 /
      data  lbytes / 0 /
      data  abort / 0 /
      data  name/'VFILT3D'/
      data  pass/.false./
      pi =  4. * atan(1.0)

C get help if requested

      if ( argis ( '-?' ) .gt. 0 .or. 
     :     argis ( '-h' ) .gt. 0 .or. 
     :     argis ( '-help' ) .gt. 0  ) then
         call help()
         stop
      endif

c open printout file

#include <f77/open.h>

C read command line parameters

      call cmdln ( ntap, otap, dt, dx, dy, dz, v1, v2, v3, v4, 
     :     a1, a2, a3, a4, f1, f2, f3, f4, pass, verbos )

c open input and output datasets

      call getln ( luin, ntap, 'r', 0 )
      call getln ( luout, otap, 'w', 1 )

c verify input dataset

      lbytes=0
      call rtape  ( luin, lhed, lbytes )
      if(lbytes .eq. 0) then
         write(LERR,*)'VFILT3D: no header read on unit ',ntap
         write(LERR,*)'FATAL'
         write(LERR,*)'Check existence/permissions of file & rerun'
         stop
      endif

      call savelu('RecNum',ifmt_RecNum,l_RecNum,ln_RecNum,TRACEHEADER)
  
      call saver(lhed, 'NumSmp', nsamp, LINHED)
      call saver(lhed, 'Format', iform, LINHED)
      call saver(lhed, 'NumTrc', ntrc , LINHED)
      call saver(lhed, 'NumRec', nrec , LINHED)
      call saver(lhed, 'OrNTRC', nslices, LINHED)
      call saver(lhed, 'UnitSc', unitsc, LINHED)
      if (unitsc .eq. 0.0) then
          write(LERR,*)'********************************************'
          write(LERR,*)'WARNING: sample unit scaler in LH = ',unitsc
          write(LERR,*)'         will set to .001 (millisec default)'
          write(LERR,*)'********************************************'
          unitsc = .001
          call savew(lhed, 'UnitSc', unitsc, LINHED)
      endif

      if ( dz .lt. 1e-32 ) dt = dt * unitsc

c define byte size of output data trace

      obytes = SZTRHD + SZSMPD * nsamp

c  find power of 2

      nfreq = nint ( 1.0 / (2. * dt ) )
      NextPowerOf2 = ordfft(nfreq)
      nfreq = 2 ** NextPowerOf2

c debug
c      write(LER,*)' nfreq = ',nfreq

      NextPowerOf2 = ordfft(nsamp)
      nsampo = 2 ** NextPowerOf2

      NextPowerOf2 = ordfft(ntrc)
      ntrco = 2 ** NextPowerOf2

      if(nsampo .gt. 8192) then
         write(LERR,*) 'VFILT3D: Trace length too long to transform'
         write(LERR,*) '          Be sure next power of 2 is less than'
         write(LERR,*) '          8192 or contact the USP guys at APR'
         write(LERR,*) '          for a program change'
         write(LERR,*) 'Fatal'  
         write(LERR,*) ' '
      
         write(LER,*) 'VFILT3D: Trace length too long to transform'
         write(LER,*) '          Be sure next power of 2 is less than'
         write(LER,*) '          8192 or contact the USP guys at APR'
         write(LER,*) '          for a program change'
         write(LER,*) 'Fatal'        
         write(LER,*) ' '
         stop
      endif
      
c dynamic memory allocation

      itemHeader = ntrco * ITRWRD * SZSMPD
      itemSpace = ntrco * nsampo * SZSMPD

      if ( ntrco .ge. ntrc ) then
         if ( nsampo .lt. nsamp ) itemSpace = ntrco * nsamp * SZSMPD
      else 
         if ( nsampo .ge. nsamp ) then
            itemHeader = ntrc * ITRWRD * SZSMPD
            itemSpace = ntrc * nsampo * SZSMPD
         else
            itemHeader = ntrc * ITRWRD * SZSMPD
            itemSpace = ntrc * nsamp * SZSMPD
         endif
      endif  
    
      call galloc (wkadr1, itemHeader, errcd1, abort)
      call galloc (wkadr2, itemSpace, errcd2, abort)

      if ( errcd1 .ne. 0 .or. errcd2 .ne. 0 ) then
         write(LERR,*)' '
         write(LERR,*)'Unable to allocate workspace:'
         write(LERR,*) itemHeader,' bytes'
         write(LERR,*) itemSpace,' bytes'
         write(LERR,*)' '
         go to 999
      else
         write(LERR,*)' '
         write(LERR,*)'Allocating workspace:'
         write(LERR,*) itemHeader,' bytes'
         write(LERR,*) itemSpace,' bytes'
         write(LERR,*)' '
      endif

c update HLH

      call hlhprt ( lhed , lbytes, name, 7, LERR )
  
c modify header as needed

      call savew ( lhed, 'NumSmp', nsamp, LINHED )
      call savew ( lhed, 'NumTrc', ntrc, LINHED )
      call savew ( lhed, 'NumRec', nrec, LINHED )
      call savhlh ( lhed, lbytes, lbyout )

C write out modified line header

      call wrtape(luout, lhed, lbyout)

c verbal printout

      call verbal( ntap, otap, nsamp, ntrc, nrec, v1, v2, v3, v4, a1, 
     :     a2, a3, a4, f1, f2, f3, f4, dt, dx, dy, dz, pass, verbos )

C input data and do fft2d

      scal = 1.0 / ( 2 * nsampo * ntrco )

c the following assumes that the input to this routine is 
c frequency slices from dc to Nyquist 

       deltaF = 1.0 / ( dt * float(nslices) )

c determine size of working array to initialize

      size = nsampo
      if ( nsamp .gt. nsampo) size = nsamp
      if ( ntrc .gt. ntrco ) then
         size = size * ntrc
      else
         size = size * ntrco
      endif

c start processing frequency slices

      DO JJ = 1, nrec

c clear workspace

         call vclr ( work, 1, size )

c load next record

         do KK = 1, ntrc
            trndx = (KK-1) * nsampo + 1
            hdrndx = (KK-1) * ITRWRD + 1

            nbytes = 0
            call rtape ( luin, lhed, nbytes )
            if(nbytes .eq. 0) then
               write(LERR,*)'End of file on input:'
               write(LERR,*)'  rec= ',JJ,'  trace= ',KK
               go to 999
            endif

c get frequency slice number from trace header of first trace only

            if ( KK .eq. 1 ) call saver2( lhed, ifmt_RecNum, l_RecNum, 
     :           ln_RecNum, RecNum, TRACEHEADER )

            call vmov ( lhed(ITHWP1), 1, work(trndx), 1, nsamp )
            call vmov ( lhed, 1, headers(hdrndx), 1, ITRWRD )
         enddo
            
c do the forward 2d fft and scale it

         call rfft2d ( work, nsampo, ntrco, 1 )
         call vsmul ( work, 1, scal, work, 1, nsampo*ntrco )

c here is the filtering part
c  calculate the frequency
c  remember that the first slice is frequency = 0
c                the second slice is frequency = nyquist
c  the frequency for pairs of slices (records) is the same

         ifreq = (RecNum-1) / 2
         freq = ifreq * deltaF
         if(RecNum.eq.2) freq = (nrec/2) * deltaF

c determine frequency taper 

         Ftaper = 1.0

         if ( abs(f1) .gt. 1.e-30 .or.
     :        abs(f2) .gt. 1.e-30 .or. 
     :        abs(f3) .gt. 1.e-30 .or. 
     :        abs(f4) .gt. 1.e-30 ) then

            call Single_Taper ( freq, f1, f2, f3, f4, 1, pi, Ftaper )
         endif

         do 4075  KJ = 1, ntrco
            do 4075  KI = 1, nsampo

               factor = 1.0

C calculate the value of kx, remember that there is a real
c and an imaginary part for each kx value

               trndx = (KJ-1) * nsampo + KI
               KIt = (KI-1) / 2
               dkx = float(KIt) * ( 1. / ( float(nsampo) * dx ) )

c  Calculate ky, there is only one column for each ky
c  value since the imaginary part follows the
c  real part (i.e. is in the same column but the next row.)

               ky = KJ - 1
               if ( ky .gt. ntrco/2 ) ky = (ntrco - ky)*(0.0 -1.0)
               dky = float(ky) * ( 1. / ( float(ntrco) * dy ) )

c the first two rows are different
c     if dkx = 0

               IF(KI .eq. 1) then

c dky = 0, let's keep this value

                  if(KJ .eq. 1) go to 4075

c if KJ=2 this is the nyquist value for ky

                  if(KJ .eq. 2) then
                     dky = -1. / ( float(ntrco) * dy  )
                  else

c in the KJ direction there is a real part followed by an imaginary
c  part up to half the Nyquist frequency in the ky direction.

                     KJt = (KJ-1) / 2
                     dky = float(KJt) * ( 1. / ( float(ntrco) * dy ) )
                  endif
               ENDIF

c if dkx = Nyquist

               IF(KI .eq. 2) then
                  dkx = -1. / ( float(nsampo) * dx )
                  if(KJ .eq. 1) dky = 0.0

c if KJ=2 this is the nyquist value for ky

                  if(KJ .eq. 2) then
                     dky = -1. / ( float(ntrco) * dy )
                  else
                     
c in the KJ direction there is a real part followed by an imaginary
c  part up to half the Nyquist frequency in the ky direction.

                     KJt = (KJ-1) / 2
                     dky = float(KJt) * ( 1. / ( float(ntrco) * dy ) )
                  endif
               ENDIF

c do nothing at kx = ky = 0

               if ( abs(dkx) .le. 1.e-30 .and. abs(dky) .le. 1.e-30 ) 
     :              go to 4075

               IF (v1 .gt. 0.0 .or.
     :              v2 .gt. 0.0 .or.
     :              v3 .gt. 0.0 .or.
     :              v4 .gt. 0.0 ) then
                  
c calculate the velocity 

                  vel = freq / (dkx**2 + dky**2)**.5
               else
                  vel = -999.99
               endif

c determine azimuth 

               if ( abs(a1) .gt. 1.e-30 .or.
     :              abs(a2) .gt. 1.e-30 .or.
     :              abs(a3) .gt. 1.e-30 .or.
     :              abs(a4) .gt. 1.e-30 ) then
                  
c determine azimuth of sample from origin, Azimuth at origin is flagged
c 999.99, if azimuth option is not specified and user wants radial
c symmetry Azimuth is flagged -999.99
                     
                  call GetAzimuth ( dkx, dky, pi, Azimuth )
               else
                  Azimuth = -999.99
               endif
               
c for the following if both vel and Azimuth are defined then both 
c boundary conditions must be examined, otherwise only one or the
c other need be considered.  If neither are set then the radial option
c is in place and Taper can be set to unity.

               if ( vel .gt. 0.0 .and. Azimuth .ge. 0.0 ) then
                  call Double_Taper ( vel, v1, v2, v3, v4, Azimuth, 
     :                 a1, a2, a3, a4, pass, pi, Taper )
               elseif ( vel .lt. 0.0 .and. Azimuth .ge. 0.0 ) then
                  call Single_Taper ( Azimuth, a1, a2, a3, a4, pass, 
     :                 pi, Taper )
               elseif ( vel .gt. 0.0 .and. Azimuth .lt. 0.0 ) then
                  call Single_Taper ( vel, v1, v2, v3, v4, pass, 
     :                 pi, Taper )
               else
                  Taper = 1.0
               endif
               
               factor = factor * Taper * Ftaper
               
c scale the data                  

 4075     work(trndx) = factor * work(trndx)

c Inverse fft2d 

         call rfft2d ( work, nsampo, ntrco, -1 )

c output the filtered data

         do KK = 1, ntrc
            trndx = (KK-1) * nsampo + 1
            hdrndx = (KK-1) * ITRWRD + 1
            call vmov ( headers(hdrndx), 1, lhed, 1, ITRWRD )
            call vmov ( work(trndx), 1, lhed(ITHWP1), 1, nsamp )
            call wrtape( luout, lhed, obytes)
         enddo

c go to next frequency slice

      ENDDO

c finished processing, close all attached files and terminate

      call lbclos ( luin )
      call lbclos ( luout )

      write(LERR,*)' Normal Termination'
      write(LER,*)'VFILT3D: Normal Termination'
      stop

 999  continue

c something abnormal occured, usually premature eof

      call lbclos ( luin )
      call lbclos ( luout )
      write(LERR,*)' Abormal Termination'
      write(LER,*)'VFILT3D: Abnormal Termination'
      stop
      END
