C***********************************************************************
C                 copyright 2001, Amoco Production Company             *
C                             All Rights Reserved                      *
C                     an affiliate of BP America Inc.                  *
C***********************************************************************
c**********************************************************************c
c
c     program module fkstrip
c
c**********************************************************************c
c
c fkstrip reads in seismic data one recors at a time
c applies a f-k strip filter to nail flat events
c and writes the resilts to an output file
c
c     this code works as is on both the cray2 & the sun
c
c**********************************************************************c
c
c     declare variables
 
#include <localsys.h>
#include <save_defs.h>
#include <f77/iounit.h>
#include <f77/lhdrsz.h>
#include <f77/sisdef.h>

      integer     itr ( 2*SZLNHD )
      integer     nsamp, nsi, ntrc, nrec, iform, lsf, iwst, iwed, ld
      integer     luin , luout, lbytes, nbytes, lbyout,obytes
      integer     lag, iws, iwe, itapr
      integer     irs,ire,ns,ne
      integer     argis

      integer     ifmt_WDepDP, l_WDepDP, ln_WDepDP, WDepDP
      integer     ifmt_HdrWrd, l_HdrWrd, ln_HdrWrd, iHdrWrd
      integer     ifmt_StaCor, l_StaCor, ln_StaCor, StaCor

      real        tri ( 2*SZLNHD ), sf ( 2*SZLNHD ), wt( 2*SZLNHD )

      character   ntap * 255, otap * 255, name*7, HdrWrd * 6

      logical     verbos, query, pass

c variables used in dynamic memory allocation 

      integer     itrh, errcd1, errcd2, errcdh, abort
      real        x, y
      pointer     (wkadr1, x(1))
      pointer     (wkadr2, y(1))
      pointer     (wkadrh, itrh(1))

c initialize variables

      data lbytes / 0 /
      data nbytes / 0 /
      data name/'FKSTRIP'/
      data pass/.false./
      data abort /0/

c-----
c     get online help if necessary
c-----
      query = ( argis ( '-?' ) .gt. 0 .or. argis ( '-h' ) .gt. 0)
      if ( query )then
            call help()
            stop
      endif

c-----
c     open printout files
c-----
#include <f77/open.h>

c-----
c     get command line parameters
c-----
      call gcmdln(ntap,otap,ns,ne,irs,ire,iwst,iwed,lsf,ex,aint,
     1            HdrWrd,vel,wb,itapr,pass,verbos)

c-----
c     get logical unit numbers for input and output
c-----
      call getln(luin , ntap,'r', 0)
      call getln(luout, otap,'w', 1)

c-----
c     read line header of input
c     save certain parameters
c-----
      call rtape   ( luin, itr, lbytes)
      if(lbytes .eq. 0) then
         write(LOT,*)'FKSTRIP: no header read from unit ',luin
         write(LOT,*)'FATAL'
         stop
      endif
      call saver(itr, 'NumSmp', nsamp, LINHED)
      call saver(itr, 'SmpInt', nsi  , LINHED)
      call saver(itr, 'NumTrc', ntrc , LINHED)
      call saver(itr, 'NumRec', nrec , LINHED)
      call saver(itr, 'Format', iform, LINHED)
      call saver(itr, 'UnitSc', unitsc, LINHED)
      if (unitsc .eq. 0.0) then
          write(LERR,*)'********************************************'
          write(LERR,*)'WARNING: sample unit scaler in LH = ',unitsc
          write(LERR,*)'         will set to .001 (millisec default)'
          write(LERR,*)'********************************************'
          unitsc = .001
          call savew(itr, 'UnitSc', unitsc, LINHED)
      endif

      if (ntrc .gt. 2*SZLNHD) then
         write(LERR,*)'Number of traces/rec too large'
         write(LERR,*)'Must be less than ',2*SZLNHD+1
         write(LERR,*)'You might try utop to subdivide recs'
         stop
      endif

c set up pointers to header values

      call savelu ('StaCor', ifmt_StaCor, l_StaCor, ln_StaCor, 
     :     TRACEHEADER )
      call savelu ('WDepDP', ifmt_WDepDP, l_WDepDP, ln_WDepDP, 
     :     TRACEHEADER)
      if (HdrWrd .ne. ' ')
     :call savelu (HdrWrd, ifmt_HdrWrd, l_HdrWrd, ln_HdrWrd, 
     :     TRACEHEADER)


      call hlhprt (itr, lbytes, name, 7, LERR)
c-----
c     ensure that command line values are compatible with data set
c-----
      call cmdchk(ns,ne,irs,ire,ntrc,nrec)

c-----
c     modify line header to reflect actual number of traces output
c     put command line string in historical line header
c-----
      nrecc=ire - irs+1
      jtr=ne-ns+1

c-----
c     verbose output of all pertinent information before
c     processing begins
c-----
            call verbal(nsamp, nsi, ntrc, nrec, iform, iwst,
     1                  iwed,lsf,ex,aint,itapr,ntap,otap,pass,
     2                  vel,wb,HdrWrd)
      ny = ntrc + lsf -1

c---------------------------------------------------
c  malloc only space we're going to use
      item1 = ntrc * nsamp
      item2 = ny   * nsamp
      itemh = ntrc * ITRWRD

      call galloc (wkadr1, item1*SZSMPD, errcd1, abort)
      call galloc (wkadr2, item2*SZSMPD, errcd2, abort)
      call galloc (wkadrh, itemh*SZSMPD, errcdh, abort)

      if ( errcd1 .ne. 0. .or.
     :     errcd2 .ne. 0. .or.
     :     errcdh .ne. 0. ) then
         write(LERR,*)' '
         write(LERR,*)'Unable to allocate workspace:'
         write(LERR,*) item1*SZSMPD,'  bytes'
         write(LERR,*) item2*SZSMPD,'  bytes'
         write(LERR,*) itemh*SZSMPD,'  bytes'
         write(LERR,*)' '
         write(LER,*)' '
         write(LER,*)'Unable to allocate workspace:'
         write(LER,*) item1*SZSMPD,'  bytes'
         write(LER,*) item2*SZSMPD,'  bytes'
         write(LER,*) itemh*SZSMPD,'  bytes'
         write(LER,*)' '
         go to 999
      else
         write(LERR,*)' '
         write(LERR,*)'Allocating workspace:'
         write(LERR,*) item1*SZSMPD,'  bytes'
         write(LERR,*) item2*SZSMPD,'  bytes'
         write(LERR,*) itemh*SZSMPD,'  bytes'
         write(LERR,*)' '
      endif
c-----

c___________________________________________________________________
c     pull off water column velocity
c___________________________________________________________________
      call saver(itr,'WatVel',ivw,LINEHEADER)
      if(wb .gt. 0. .and. ivw .eq. 0) then
         write(LERR,*) 'error in routine fkstrip!'
         write(LERR,*) 'application of filter to begin at ',
     1                  wb,' times the water depth'
         write(LERR,*) ' WatVel in line header = ',ivw
         write(LERR,*) ' run utop -h0WatVel=1480 or equivalent'
         call exit(666)
      endif
      vw = float (ivw)
      write(LERR,*) ' WatVel in line header = ',vw
      write(LERR,*) ' water bottom multiplier = ',wb

c-----
c     reduce times to points; compute data length, lag, 
c-----
      dt = real (nsi) * unitsc

      iws = iwst/nsi + 1
      iwe = iwed/nsi + 1
      if(iwed .eq. 0) iwe = nsamp
      ld = iwe - iws + 1
      lag = lsf/2

c-----
c     if data set contains records that have too few traces we need to read
c     them in larger bunches if possible
c-----
      IF (ntrc .lt. lag) THEN

         if (ns.eq.1 .and. ne.eq.ntrc .and. irs.eq.1 .and. ire.eq.nrec) 
     1        then

            nall = nrec*ntrc
            nall2 = nall/2

            do  ni = 5, 1, -1
               ne = ni * (lsf-1)
               if (ne .le. nall2) go to 5
            enddo

            write(LERR,*)' '
            write(LERR,*)'Data set does not have enough traces for'
            write(LERR,*)'fkstrip.  For the filter length= ',lsf
            write(LERR,*)'we need at least that many traces'
            stop

 5          continue

            ire = nall/ne
            ire = ire+1
            write(LERR,*)' '
            write(LERR,*)'In order to accomodate filter length= ',lsf
            write(LERR,*)'I will internally read your data in ',ne
            write(LERR,*)'trace bunches, ',ire,' bunches to the line'

         else

            write(LERR,*)' '
            write(LERR,*)'Data set is too small to use trace/record'
            write(LERR,*)'limitation.  Rerun removing -ns, -ne, -rs'
            write(LERR,*)'-re entreis from the command line'
            stop

         endif

      ENDIF

c-----
c     if necessary compute taper weights
c-----
      if (itapr .ne. 0) then
          itapr = float(ne*itapr)/100.
          den = float( itapr )
          do i = 1, itapr
             wt(i) = float( itapr - i + 1 )/den
          enddo
      endif

c-----
c     adjust line header for output number samples,
c     output line header
c-----
      call savhlh(itr,lbytes,lbyout)
      obytes = SZTRHD + SZSMPD * nsamp
      call wrtape ( luout, itr, lbyout  )

c-----
c     BEGIN PROCESSING
c     compute spatial filter
      call vnwt(ex, aint, lsf, sf)
      if(verbos) then
         write(LERR,*)'Filter Weights are'
         write(LERR,111) (sf(i),i=1,lsf)
 111     format(5e12.4)
      endif

c-----
c     read all or portion of record, filter, output data
c-----
c     skip unwanted records
c-----
      nbytes = obytes
      call recrw (1,irs-1,luin,ntrc,itr,luout, nbytes)
      if (nbytes  .eq. 0) go to 999

c-----
c     process desired trace records
c-----

      DO JJ = irs, ire

         nbytes = obytes
         call trcrw (JJ, 1, ns-1, luin, ntrc, itr, luout, nbytes)
         if (nbytes  .eq. 0) go to 999
         
         k = 0
         nlive = 0
         sumzw = 0.
         sumtw = 0.
c-----
c     get all or portion of a record
c-----
         DO KK = ns, ne
            nbytes = 0
            call rtape( luin, itr, nbytes)
            if(nbytes .eq. 0) then
               write(LERR,*)'Premature end of data at'
               write(LERR,*)'Rec= ',JJ,' Trc= ',KK
               write(LERR,*)'Continuing to run...'
               go to 510
            endif
            call vmov (itr(ITHWP1), 1, tri, 1, nsamp)
            k = k + 1
            call saver2 ( itr, ifmt_StaCor, l_StaCor, ln_StaCor, 
     :           StaCor, TRACEHEADER )
c-----
c     save data and trace headers
c-----
            if (StaCor .eq. 30000 ) then
               call vclr(tri,1,nsamp)
            else
               nlive = nlive + 1
               call saver2 ( itr, ifmt_WDepDP, l_WDepDP, 
     :              ln_WDepDP, WDepDP, TRACEHEADER )
               zw = float(WDepDP)
               sumzw = sumzw + zw
               if (HdrWrd .ne. ' ') then
               call saver2 ( itr, ifmt_HdrWrd, l_HdrWrd, 
     :              ln_HdrWrd, iHdrWrd, TRACEHEADER )
               tw = float(iHdrWrd)
               sumtw = sumtw + tw
               endif
            endif
            istrc = (k-1) * nsamp
            ishdr = (k-1) * ITRWRD

            do ii = 1, nsamp
               x(istrc+ii) = tri(ii)
            enddo

            call vmov (itr,1, itrh(ishdr+1),1,ITRWRD)
            
         ENDDO

 510     continue

c___________________________________________________________________
c        calculate start time of filter as function of water bottom
c___________________________________________________________________
         if(HdrWrd .eq. ' ' .AND. wb .ne. 0. .and. nlive .ne. 0) 
     1      then
            zwavg  = sumzw/float(nlive)
            tstart = 2. * wb * zwavg/vw
            ist    = tstart/dt
            ied    = iwe
            if (ist .ge. nsamp) then
               write(LERR,*)'FATAL HEART ATTACK IN fkstrip:'
               write(LERR,*)'start time= ',ist,' samples'
               write(LERR,*)'exceeds trace length= ',nsamp
               write(LERR,*)'using water velocity= ',vw
               write(LERR,*)'from line header'
               write(LERR,*)'and average water depth= ',zwavg
               write(LERR,*)'from trace headers.  Check all'
               write(LERR,*)'these values and correct with'
               write(LERR,*)'utop'
               goto 999
            endif
         elseif (HdrWrd .ne. 'WDepDP' .AND. HdrWrd .ne. ' ' .AND.
     1           nlive .ne. 0) then

               twavg = sumtw / float(nlive)
               iwst = twavg / float(nsi)
               ist  = iwst - iws + 1
               if (ist .le. 0) ist = 1
               ied  = iwst + iwe
               if (ied .gt. nsamp) ied = nsamp
               if (ist .ge. nsamp) then
                  write(LERR,*)'FATAL HEART ATTACK IN fkstrip:'
                  write(LERR,*)'start time= ',ist,' samples'
                  write(LERR,*)'exceeds trace length= ',nsamp
                  write(LERR,*)'using hdr word time = ',iHdrWrd
                  write(LERR,*)'and global start time= ',(iws-1)*nsi
                  goto 999
               endif
         else
            ist = iws
            ied = iwe
         endif
         
         
c-----
c     check to see if we really have enough traces in this
c     record to do a spatial filter
c-----
         IF (k .gt. lag) THEN
            
c***********************************************************************
c     filter the data
c-----

            ny = k + lsf - 1
            call sfold(k,ied,x,ist,lsf,sf,ny,y,nsamp)
            
c-----
c     take difference to get notch if reject specified
c-----
            if( .not.pass ) then
               
               do j = 1, k
                  kj = j + lag
                  lj = (j-1) * nsamp
                  lkj = (kj-1) * nsamp
                  do i = ist, ied
                     x(i+lj) = x(i+lj) - y(i+lkj)
                  enddo
               enddo
               
c-----
c     otherwise use pass option
c-----
            else

               do j = 1, k
                  lj = (j-1) * nsamp
                  kj = j + lag
                  lkj = (kj-1) * nsamp
                  do i = ist, ied
                     x(i+lj) = y(i+lkj)
                  enddo
               enddo
               
            endif

         ENDIF

c-----
c     get trace headers, update rec & trc numbers, and write out data
c     taper ends of spread if necessary
c-----
         DO KK = 1, k

            lk = (KK-1) * nsamp
            ishdr = (KK-1) * ITRWRD

            IF(itapr .eq. 0) THEN
               do ii = 1, nsamp
                  tri(ii) = x(ii+lk)
               enddo
            ELSE
               k2 = ne - itapr
               if(KK .le. itapr) then
                  do ii = 1, nsamp
                     tri(ii) = x(ii+lk) * wt(itapr-KK+1)
                  enddo
               elseif(KK .gt. k2) then
                  do ii = 1, nsamp
                     tri(ii) = x(ii+lk) * wt(KK-k2)
                  enddo
               else
                  do ii = 1, nsamp
                     tri(ii) = x(ii+lk)
                  enddo
               endif
            ENDIF
            
            call saver2 ( itr, ifmt_StaCor, l_StaCor, 
     :           ln_StaCor, StaCor, TRACEHEADER )
            if (StaCor .eq. 30000) call vclr (tri, 1, nsamp)
            call vmov (tri, 1, itr(ITHWP1), 1, nsamp)
            call vmov (itrh(ishdr+1),1,itr,1,ITRWRD)
            call wrtape( luout, itr, obytes)
            
         ENDDO

c-----
c     skip to end of present record; go get next record
c-----
         nbytes = obytes
         call trcrw (JJ, ne+1, ntrc, luin, ntrc, itr, luout, nbytes)
         if (nbytes .eq. 0) go to 999

      ENDDO

c------------------------
c  pass remainder of recs

      nbytes = obytes
      call recrw (ire+1, nrec, luin, ntrc, itr, luout, nbytes)
      if (nbytes .eq. 0) go to 999

c-----
c     Normal Termination: close data files
c-----
      call lbclos ( luin )
      call lbclos ( luout )
      write(LERR,*)'Normal Termination'
      write(LERR,*)'processed',nrec,' record(s)',
     :     ' with ',ntrc, ' traces'
      write(LER,*)'fkstrip: Normal Termination'
       stop

999   continue
c-----
c     Abormal Termination: close data files
c-----
      call lbclos ( luin )
      call lbclos ( luout )
      write(LERR,*)'Abnormal Termination'
      write(LERR,*)'processed',nrec,' record(s)',
     :     ' with ',ntrc, ' traces'
      write(LER,*)'fkstrip: Abnormal Termination'
      end


      subroutine help
#include <f77/iounit.h>

         write(LER,*)
     :'***************************************************************'
      write(LER,*)
     :'execute by typing fkstrip and a list of program parameters'
      write(LER,*)
     :'note that each parameter is proceeded by -a where "a" is '
      write(LER,*)
     :'a character(s) corresponding to some parameter.'
        write(LER,*)
     :'users enter the following parameters, or use the default values'
        write(LER,*)
     :' -N [ntap]    (no default)      : input data file name'
        write(LER,*)
     :' -O [otap]    (no default)      : output data file name'
       write(LER,*)' '
       write(LER,*)
     :' -ns[ns]      (default = first) : start process trace number'
       write(LER,*)
     :' -ne[ne]      (default = last)  : end process trace number'
       write(LER,*)
     :' -rs[irs]     (default = first) : start process record number'
       write(LER,*)
     :' -re[ire]     (default = last)  : process end record number'
       write(LER,*)' '
        write(LER,*)
     :' -s [iwst]    (default = 0ms)   : process start time (ms)'
        write(LER,*)
     :' -e [iwst]    (default = end)   : process end time (ms)'
c       write(LER,*)
c    :' -v [vel]     (default = flat)  : start time velocity adjust'
       write(LER,*)
     :' -hw [hdrwrd] (def=WDepDP)      : hdr wrd containing start time'
       write(LER,*)
     :' -b [wb]      (def=0.0)         : water bottom start time multipl
     :ier'
       write(LER,*)
     :'              wb*time to water bottom    (0.)'
       write(LER,*)
     :'              uses WatVel and WDepDP headers'
       write(LER,*)' '
        write(LER,*)
     :' -l [lsf]     (default = 31)    : length (points) of spatial filt
     :er'
        write(LER,*)
     :' -w [ex]      (default = 0)     : bessel weighting (0-3)'
        write(LER,*)
     :' -i [aint]    (default = .05)   : filter intercept'
        write(LER,*)
     :' -t [itapr]   (default = 0)     : number trace to taper each side
     :'
        write(LER,*)
     :' -P                  enhance rather than attenuate flat dips'
       write(LER,*)' '
         write(LER,*)
     :'usage:   fkstrip -N[ntap] -O[otap] -ns[ns] -ne[ne] -rs[irs] '
       write(LER,*)
     :'                 -s[iwst] -e[iwed] [ -v[vel] -b[wb] hw[] ]'
       write(LER,*)
     :'                 -l[lsf] -w[ex] -i[aint] -t[itapr] -V'
         write(LER,*)
     :'***************************************************************'
      return

      end

      subroutine gcmdln(ntap,otap,ns,ne,irs,ire,iwst,iwed,lsf,ex,aint,
     1                  HdrWrd,vel,wb,itapr,pass,verbos)
c-----
c     get command arguments
c
c     ntap  - c*255     input file name
c     otap  - c*255     output file name
c     ns    - i*4 starting trace index
c     ne    - i*4 ending trace index
c     irs   - i*4 starting record index
c     ire   - i*4 ending record index
c    iwst   - i*4 start window
c    iwed   - i*4 end window
c     lsf   - i*4 length of spatial filter
c     ex    - r*4 bessel weighting exponent
c    aint   - r*4 strip filter intercept
c   itapr   - i*4 percent taper applied to either side of spread
c    pass   - L   if true pass zero dip
c   verbos  - L   verbose output or not
c-----
#include <f77/iounit.h>
      character  ntap*(*), otap*(*), HdrWrd * 6
      integer *4 ns, ne, irs, ire,iwst,iwed,lsf,itapr
      real*4     ex, aint, wb, vel
      logical    verbos, pass
      integer    argis

            call argstr( '-N', ntap, ' ', ' ' )
            call argstr( '-O', otap, ' ', ' ' )
            call argstr( '-hw', HdrWrd, ' ', ' ' )
            call argi4 ( '-ns', ns ,   0  ,  0    )
            call argi4 ( '-ne', ne ,   0  ,  0    )
            call argi4 ( '-rs', irs ,   0  ,  0    )
            call argi4 ( '-re', ire ,   0  ,  0    )
            call argi4( '-s', iwst, 0, 0 )
            call argi4( '-e', iwed, 0, 0 )
            call argi4( '-l', lsf, 31, 31 )
            call argr4( '-w', ex, 0., 0. )
            call argr4( '-i', aint, .05, .05 )
            call argr4( '-v', vel, 999999., 999999. )
            call argr4( '-b', wb, 0.0, 0.0 )
            call argi4( '-t', itapr, 0, 0 )
            pass   = ( argis( '-P' ) .gt. 0 )
            verbos = ( argis( '-V' ) .gt. 0 )

            if(itapr .gt. 50) itapr = 50

      return
      end

      subroutine verbal(nsamp, nsi, ntrc, nrec, iform, iwst,
     1            iwed,lsf,ex,aint,itapr,ntap,otap,pass,
     2            vel,wb,HdrWrd)
#include <f77/iounit.h>
      integer nsamp, nsi, ntrc, nrec, iform, iwst,iwed,lsf,itapr
      real ex,aint
      character ntap*(*), otap*(*), HdrWrd * 6
      logical   pass

            write(LERR,*)' '
            write(LERR,*)' line header values after default check '
            write(LERR,*) ' # of samples/trace =  ', nsamp
            write(LERR,*) ' sample interval    =  ', nsi
            write(LERR,*) ' traces per record  =  ', ntrc
            write(LERR,*) ' records per line   =  ', nrec
            write(LERR,*) ' format of data     =  ', iform
            write(LERR,*) ' window start time  =  ', iwst
            write(LERR,*) ' window end time    =  ', iwed
c           write(LERR,*) ' start velocity     =  ', vel
            if (wb .ne. 0.0) then
            write(LERR,*) ' water bottom start time mult= ',wb
            endif
            if (HdrWrd .ne. ' ') then
            write(LERR,*) ' start time hdr word=  ', HdrWrd
            endif
            write(LERR,*) ' length of spatial filter = ',lsf
            write(LERR,*) ' besel weighting    =  ', ex
            write(LERR,*) ' strip intercept    =  ', aint
            write(LERR,*) ' spread taper       =  ', itapr,' traces'
            write(LERR,*) ' input data set name =  ', ntap
            write(LERR,*) ' output data set name=  ', otap
            if(pass)
     1      write(LERR,*) ' Will Pass 0-dip events only'
            if(.not.pass)
     1      write(LERR,*) ' Will Reject 0-dip events only'
            write(LERR,*)' '
            write(LERR,*)' '

      return
      end

