C***********************************************************************
C                 copyright 2001, Amoco Production Company             *
C                             All Rights Reserved                      *
C                     an affiliate of BP America Inc.                  *
C***********************************************************************
C***********************************************************************
C Written by P.G.A. Garossino and M.D. Bush, Cairo: September, 1988
C
C Technical Assistance supplied by:  S.L. Harris
C
C***********************************************************************
C Program TVD (Time Velocity Dip) was developed for use with the
C powerful multiple removal process CEP Demult described by D. Yanchak
C in the February 1988 RGTS minutes, "CEP Demult is a multiple
C suppression technique that exploits the differential moveout between
C primaries and multiples on Common End Point records.  A CEP record
C is a common source record or a common receiver record.".  In Cairo,
C the approach taken (RGTS September, 1988) was to derive a particular
C function of Time, Velocity and Dip (TVD) that, when used to move out
C the CEP record, would result in the separation of primary and
C multiple data into opposite quadrants of FK space.  The quadrant
C containing multiples was removed using an FK filter and the TVD
C function backed out allowing the continuation of conventional
C processing.  The derivation of the necessary TVD functions involved
C the processing and analysis of one Common Dip Analysis (CDA) panel
C every 25 shot/receiver points.  This phase typically took from 8 to
C 10 days per line for completion with 6 of those days requiring the
C full-time attention of the interpreter.  With program TVD it is now
C possible to complete the same analysis in 5 to 10 minutes.
C
C The basic building block of program TVD is the Time-Distance
C equation of a dipping interface:
C
C 1.    t(x) =((2h/v)**2 + (x/v)**2 + 4hxsin(d)/v**2)**0.5
C
C Since the expression 2h/v is equivalent to the arrival time at zero
C offset, t(0), equation 1 can be written:
C
C 2.    t(x) =(t(0)**2 + (x/v)**2 + 2t(0)xsin(d)/v)**0.5
C
C Equation 2 describes the hyperbolic trace of a dipping reflector
C with velocity (v) and dip (d) on a Common End-Point record.
C Equation 3 corresponds to the event as it would appear on a CDA
C panel corrected using velocity V and dip D.
C
C 3.    U(x) = (t(0)**2 + (x/v)**2 + 2t(0)xsin(d)/v)**0.5
C              - (t(0)**2 + (x/V)**2 + 2t(0)xsin(D)/V)**0.5
C
C Notice that in the case where both v = V and d = D the right side of
C equation 3 becomes zero.  Also notice that by varying V and D the
C expression U(xi) can be made positive or negative.  With equation 3
C then it is possible to derive the Velocity (V) and Dip (D)
C parameters required to drive any event into one or the other of the
C positive frequency quadrants within the FK domain.  To control the
C amount that the event is TWEAKed into either quadrant, equation 4
C was developed.
C
C 4.    TWEAK = (sum(i=1-10)U(xi)) / 10
C
C where:
C       xi = Gap + ((Nchan/10)i-1)Gint
C
C and:
C       Gap   =  Distance from source to near group
C       Nchan =  Number of channels on the recording system.
C       Gint  =  Group interval
C***********************************************************************
C DEFINE VARIABLES
C***********************************************************************

#include <f77/iounit.h>
#include <f77/lhdrsz.h>

      integer luvel, lu_mfile, length, argis, xsd_NumSegs
      integer lu_sred, lu_rred, lu_sgreen, lu_rgreen, lu_truedip
      integer Analysis_InitialShot, Analysis_ShotPoint
      integer Analysis_ShotIncrement, Analysis_ShotpointEnd 
      integer Model_NumShots, Model_InitialShot, i
      integer itype, fcnNumTrials, numVar, Model_SmoothingOrder
      integer fcnItrial, fcnIerr, domain, nchan, itemp
      integer Analysis_NumHorizons, tflag, xsd_Segment
      integer vel_NumFcns, count
      integer sp(SZLNHD), xsd_NumPoints(SZLNHD), vel_NumPoints(SZLNHD)

      real BoatDirection, Model_ShotIncrement 

      real*8 ShotInterval, GroupInterval, Gap, OffsetFactor
      real*8 fx, ax, tx, dt, dx, vstk, dip, Tweak
      real*8 fcnMin, fcnSlim
      real*8 negip, posip, negif, posif
      real*8 umin, umin2, pi, pi2, pid2, v0, Data_MaxTime
      real*8 c(SZLNHD), x(SZLNHD), t(SZLNHD), v(SZLNHD), d(SZLNHD)
      real*8 Var(SZLNHD), space(SZLNHD)
      real*8 dg(SZLNHD), vg(SZLNHD), uming(SZLNHD), vpg(SZLNHD)
      real*8 dpg(SZLNHD), tg(SZLNHD)
      real*8 dr(SZLNHD), vr(SZLNHD), uminr(SZLNHD), vpr(SZLNHD)
      real*8 dpr(SZLNHD), tr(SZLNHD)

      character vfile*256, mfile*256, c_domain*14, c_BoatDirection*2
      character sred*256, sgreen*256, rred*256, rgreen*256, truedip*256
      character line*256, name*3
      character*1 cg(SZLNHD), cr(SZLNHD)

      logical bail_out, Done, verbos

c variables used with dynamic memory allocation

      integer abort, errcd1, errcd2, errcd3, errcd4
      integer xsd_size, vel_size, vel_shotpoint

      real xsd_x_coords, xsd_horizon_times
      real vel_times, vel_velocities, vel_x_coords

      pointer ( memadr_xsd_x_coords, xsd_x_coords(1) )
      pointer ( memadr_xsd_horizon_times, xsd_horizon_times(1) )
      pointer ( memadr_vel_times, vel_times(1) )
      pointer ( memadr_vel_velocities, vel_velocities(1) )
      pointer ( memadr_vel_shotpoint, vel_shotpoint(1) )
      pointer ( memadr_vel_x_coords, vel_x_coords(1) )

c other dependencies

      external negif, nepgrd, posif, popgrd

c integer
c variable     meaning
c --------     ---------------------------------------------------------
c nchan      - number of channels on the cable
c domain     - a flag to identify the which domain to calculate the 
c              velocity, dip data:
c
c              domain = 0 for the shot domain
c              domain = 1 for the receiver domain
c
c Model_NumShots          - number of (shot point,x) pairs in the sp[], 
c                           and x[] arrays
c Analysis_InitialShot    - shot point location of initial analysis position.
c Analysis_ShotPoint      - shot point location of current analysis position.
c Analysis_ShotIncrement  - analysis increment in shot points.
c Analysis_ShotpointEnd   - contains the end of analysis shot point number
c xsd_NumSegs             - number of segments in xsd pickfile
c sp(i)                   - array of shot point values from evmig.prt file.
c Analysis_NumHorizons    - counter used to keep track of the number of 
c                           horizons that exist at any analysis location.
c bail_out       - will be .true. if either no velocity data exists for an
c                  analysis location, or, an interface block has no value
c                  at that location.
c i,j,k      - counters used for do loop control.
c prt        - used to check if the print flag is zero in the input file
c tflag      - set to 1 if tweak has to be adjusted during high tweak

c real*4
c variable     meaning
c --------     ---------------------------------------------------------
c BoatDirection   - flag to determine which direction the boat is steaming
c                   relative to positive x:
c                     BoatDirection = 1. shooting direction along +ve x
c                     BoatDirection = 0. shooting direction along -ve x
c Model_ShotIncrement   - used for oper shot point increment
c reqd       - used to allow free format entry of reqd.
c spi        - used to allow free format entry of Analysis_InitialShot.
c work1      - used to allow free format read of evmig.prt file.
c work2      - as above.

c real*8
c variable     meaning
c --------     ---------------------------------------------------------
c c(i)       - array of constants for attributes of primary and multiple
c              events:
c              multiple  c(1) = h
c                        c(2) = velocity
c                        c(3) = t(0)
c                        c(4) = dip (radians)
c              primary   c(5) = h
c                        c(6) = velocity
c                        c(7) = t(0)
c                        c(8) = dip (radians)
c              misc.     c(9) = reserved for x co-ordinate
c                        c(10) = group interval
c                        c(11) = gap
c                        c(12) = 2way time to multiple at analysis point
c                                ie(2*z/v)
c                        c(13) = 2way time to primary at analysis point
c                                ie(2*z/v)
c                        c(14) = not used at present
c                        c(15) = c(4)*bounce (radians)
c                        c(16) = shot interval --> ShotInterval
c                        c(20) = tweak in seconds
c                        c(21) = mult:-1 for green, +1 for red
c                        c(25) = percent of spread for interpolation
c                                used in dip calculation [OffsetFactor]
c x(i)       - array to contain x values of shot points from evmig.prt
c dt         - delta time across dip estimate at horizon.
c dx         - delta x for above.
c tx         - interpolated time of horizon at analysis location.
c t(i)       - array to contain function time.
c d(i)       - array to contain function dip.
c v(i)       - array to contain function velocity.
c vstk       - stacking velocity interpolated at horizon time and x.
c dip        - dip of horizon at analysis location. (radians)
c
c              green arrays
c
c dg(i)     - dip green
c vg(i)     - velocity green
c uming(i)  - tweak in milliseconds
c
c              red arrays
c
c dr(i)     - dip red
c vr(i)     - velocity red
c uminr(i)  - tweak in milliseconds
c
c fx         - function x co-ordinate.
c Var(i)     - Array of Variables for input to hyperbola calculation:
C              var(1) = velocity
c              var(2) = dip (radians)
c umin       - function minimum returned from subroutine

C Character Variables
c              green arrays
c cg(i)     - colour green
c
c              red arrays
c cr(i)     - colour red
c
C line       - Line number for analysis.

c ----- ----- PROGRAM START ----- -----

c give command line help if requested 

      if ( argis ( '-?' ) .gt. 0 .or.
     :     argis ( '-h' ) .gt. 0 .or.
     :     argis ( '-help' ) .gt. 0 )then
         call help()
         stop
      endif

c initialize variables

      numVar = 2
      fcnMin = 0.d0
      itype = -1
      fcnSlim = 1.e-9
      fcnNumTrials = 4
      pi=3.141592653589793
      pi2=pi*2
      pid2=pi/2
      tflag = 0
      name = 'TVD'

      call dzero ( SZLNHD, c )
      call dzero ( SZLNHD, Var )
      call dzero ( SZLNHD, t )
      call dzero ( SZLNHD, tr )
      call dzero ( SZLNHD, tg )
      call dzero ( SZLNHD, d )
      call dzero ( SZLNHD, v )
      call dzero ( SZLNHD, dr )
      call dzero ( SZLNHD, vr )
      call dzero ( SZLNHD, dg )
      call dzero ( SZLNHD, vg )
      call dzero ( SZLNHD, uminr )
      call dzero ( SZLNHD, uming )
      call dzero ( SZLNHD, space )

      do i = 1, SZLNHD
         cr(i)='Z'
         cg(i)='Z'
         sp(i) = 0
      enddo

c open printout file

#include <f77/open.h>

c parse command line parameters

      call cmdln( vfile, mfile, Model_NumShots, Model_InitialShot, 
     :     Model_ShotIncrement, Data_MaxTime,
     :     Analysis_InitialShot,  Analysis_ShotIncrement, 
     :     Analysis_ShotpointEnd, c_BoatDirection, ShotInterval, 
     :     GroupInterval, Gap, OffsetFactor, Tweak, nchan, c_domain, 
     :     Model_SmoothingOrder, v0, verbos )

c assign constants array entries based on command line input

      c(10) = GroupInterval
      c(11) = Gap
      c(16) = ShotInterval

c tweak is given in milliseconds on the command line and is converted
c to seconds here for universal application.  All [t] functions
c are converted to units of seconds in this routine

      Tweak = Tweak / 1000.d0
      Data_MaxTime = Data_MaxTime / 1000.d0
      c(20) = Tweak 
      c(25) = OffsetFactor

c  open tdfn velocity model 
 
      call alloclun (luvel)
      length = lenth( vfile )
      open ( luvel, file = vfile(1:length), status='old', err=990 )

c  open xsd model file [standard xsd pickfile]

      call alloclun ( lu_mfile )
      length = lenth(mfile)
      open ( lu_mfile, file = mfile(1:length), status='old', err=991 )

c open output tvd tweak function files labled with the associated
c domain be requested

      if ( c_domain .eq. 'shot' ) then

         domain = 0
         
c shot domain tvd output files

         sred = mfile(1:length)//'.sr'
         sgreen = mfile(1:length)//'.sg'

         call alloclun (lu_sred)
         call alloclun (lu_sgreen)

         length = lenth ( sred )
         open ( lu_sred, file = sred(1:length), status='unknown', 
     :        err=992 )

         length = lenth ( sgreen )
         open ( lu_sgreen, file = sgreen(1:length), status='unknown', 
     :        err=993 )

      elseif ( c_domain .eq. 'receiver' ) then

         domain = 1

c receiver domain tvd output files

         rred = mfile(1:length)//'.rr'
         rgreen = mfile(1:length)//'.rg'

         call alloclun (lu_rred)
         call alloclun (lu_rgreen)

         length = lenth ( rred )
         open ( lu_rred, file = rred(1:length), status='unknown', 
     :        err=994 )

         length = lenth ( rgreen )
         open ( lu_rgreen, file = rgreen(1:length), status='unknown', 
     :        err=995 )

      endif

c opening truedip tvd output file [this is the actual dip and velocity information
c at each pick at a given analysis location] 

      truedip = mfile(1:lenth(mfile))//'.truedip'
      call alloclun ( lu_truedip )
      length = lenth ( truedip )
      open (lu_truedip, file = truedip(1:length), status='unknown', 
     :     err = 996 )

c verbal printout of parameters and files

      call verbal ( mfile, vfile, Model_NumShots, 
     :     Model_InitialShot, Model_ShotIncrement, Model_SmoothingOrder,
     :     Data_MaxTime, Analysis_InitialShot, Analysis_ShotpointEnd, 
     :     Analysis_ShotIncrement, c_BoatDirection, ShotInterval, Gap, 
     :     GroupInterval, OffsetFactor, Tweak, nchan, c_domain, 
     :     v0, verbos )

c assign BoatDirection to keep correct sign on solution as a function 
c of the side of the [t-x] diagram used in the calculation.

      if ( c_BoatDirection .eq. 'lr' ) then

c shooting along +ve x

         BoatDirection = 1.0
      else

c shooting along -ve x

         BoatDirection = 0.0
      endif
 
c generate arrays for shotpoint indexing and associated horizontal distance
c in x.  In the new routine this will reference the xsd pickfile picked
c from a shot stack of the input dataset under consideration.  No option 
c will be given for picking the dip model from any other source....this
c keeps it easy...and in reality is all this is needed.  I need a shotpoint
c array set sp() to contain shotpoint indexes from 1 to Model_NumShots
c and an x array to contain distances calculated by considering the distance
c between traces on the shot stack that the model was picked from.

      call BuildShotArrays ( Model_NumShots, Model_ShotIncrement, 
     :     Model_InitialShot, ShotInterval, sp, x ) 

c reset default shot point limit to end of model 

      if ( Analysis_ShotpointEnd .lt. 1 )
     :     Analysis_ShotpointEnd = sp(Model_NumShots)

c determine if initial analysis shot point is out of range of model

      if ( Analysis_InitialShot .lt. sp(1) .or.
     :     Analysis_InitialShot .gt. sp(Model_NumShots) ) then
            write(LERR,*)' '
            write(LERR,*)'Shot point ',Analysis_InitialShot,
     :           ' out of range'
            write(LERR,*)'rerun with new starting analysis shot point'
            write(LERR,*)'FATAL'
            write(LERR,*)' '
            write(LERR,*)' '
            write(LERR,*)'TVD: '
            write(LERR,*)' Shot point ',Analysis_InitialShot,
     :           ' out of range'
            write(LERR,*)' rerun with new starting analysis shot point'
            write(LERR,*)'FATAL'
            write(LERR,*)' '
            stop
         endif
                      
c write function header to all requested output files

      length = lenth(mfile)
      line = mfile(1:length)

      call header ( c, nchan, BoatDirection, c_domain,  
     :     Analysis_InitialShot, Analysis_ShotIncrement, line, 
     :     lu_sred, lu_rred, lu_sgreen, lu_rgreen, lu_truedip )

c allocate memory to contain xsd pickfile structural dip model

c determine array sizes required

      call PickInit ( lu_mfile, xsd_NumSegs, xsd_NumPoints, 
     :     xsd_size )

c dynamically allocate memory

      call galloc ( memadr_xsd_x_coords, xsd_size*SZSMPD, 
     :     errcd1, abort )
      call galloc ( memadr_xsd_horizon_times, xsd_size*SZSMPD, 
     :     errcd2, abort )

c verify memory was obtained

      write(LERR,*) ' '
      write(LERR,*) ' Dynamic Memory Allocation '
      write(LERR,*) ' ------------------------- '
      write(LERR,*) ' '

      if ( errcd1 .ne. 0 .or.
     :     errcd2 .ne. 0 ) then
         write(LERR,*)' '
         write(LERR,*)'Unable to allocate  model workspace:'
         write(LERR,*) 2*xsd_size * SZSMPD, '  bytes'
         write(LERR,*)' '
         write(LER,*)' '
         write(LER,*)'Unable to allocate model workspace:'
         write(LER,*) 2*xsd_size * SZSMPD, '  bytes'
         write(LER,*)' '
         stop
      else
         write(LERR,*)' '
         write(LERR,*)'Allocating model workspace:'
         write(LERR,*) 2*xsd_size * SZSMPD, '  bytes'
         write(LERR,*)' '
      endif

c initialize memory

      call vclr ( xsd_x_coords, 1, xsd_size )
      call vclr ( xsd_horizon_times, 1, xsd_size )

c read dip model into memory

      call HorizonRead ( lu_mfile, xsd_NumSegs, xsd_NumPoints, 
     :     xsd_size, xsd_x_coords, xsd_horizon_times, ShotInterval, 
     :     Model_SmoothingOrder, verbos )

c determine memory requirements for velocity model, coming out of
c this routine the array vel_NumPoints[] contains the number of 
c tdfn cards per function.  Coming out of the read routine this
c array will hold the actual number of t,v entries per function
c location.

      call VelocityInit ( luvel, vel_NumFcns, vel_NumPoints, vel_size )

c account for v0 entry to be added later

      vel_size = vel_size + vel_NumFcns

c dynamically allocate memory for velocity model

      call galloc ( memadr_vel_times, vel_size * SZSMPD, errcd1, abort ) 
      call galloc ( memadr_vel_velocities, vel_size * SZSMPD, errcd2, 
     :     abort ) 
      call galloc ( memadr_vel_shotpoint, vel_size * SZSMPD, errcd3, 
     :     abort ) 
      call galloc ( memadr_vel_x_coords, vel_size * SZSMPD, errcd4, 
     :     abort ) 

c verify memory was available

      if ( errcd1 .ne. 0 .or.
     :     errcd2 .ne. 0 .or.
     :     errcd3 .ne. 0 .or.
     :     errcd4 .ne. 0 ) then
         write(LERR,*)' '
         write(LERR,*)'Unable to allocate velocity workspace:'
         write(LERR,*) 4*vel_size * SZSMPD, '  bytes'
         write(LERR,*)' '
         write(LER,*)' '
         write(LER,*)'Unable to allocate velocity workspace:'
         write(LER,*) 4*vel_size * SZSMPD, '  bytes'
         write(LER,*)' '
         stop
      else
         write(LERR,*)' '
         write(LERR,*)'Allocating velocity workspace:'
         write(LERR,*) 4*vel_size * SZSMPD, '  bytes'
         write(LERR,*)' '
      endif

c initialize memory

      call vclr ( vel_times, 1, vel_size )
      call vclr ( vel_velocities, 1, vel_size )
      call vclr ( vel_x_coords, 1, vel_size )
      do i = 1, vel_size
         vel_shotpoint(i) = 0
      enddo

c load velocity model to memory

      call VelocityRead ( luvel, vel_NumFcns, vel_size, v0,
     :     vel_NumPoints, vel_shotpoint, vel_times, vel_velocities )

c calculate velocity x coordinates

      call VelocityXcoord ( vel_NumFcns, vel_size, vel_NumPoints, 
     :     vel_shotpoint, vel_x_coords, ShotInterval ) 

c Calculate [t,v,d,colour,tweak] functions at requested analysis locations
      
      Analysis_ShotPoint = Analysis_InitialShot

      write(LERR,*)' '
      write(LERR,*)' '
      write(LERR,*)' Analysis Location Processing Report'
      write(LERR,*)' -----------------------------------'
      write(LERR,*)' '

45    continue

c reset the number of horizons at this analysis location

      Analysis_NumHorizons = 1

c initialize memory for red and green output arrays

      call dzero ( SZLNHD, tr )
      call dzero ( SZLNHD, tg )
      call dzero ( SZLNHD, dr )
      call dzero ( SZLNHD, vr )
      call dzero ( SZLNHD, dg )
      call dzero ( SZLNHD, vg )
      call dzero ( SZLNHD, uminr )
      call dzero ( SZLNHD, uming )
      call dzero ( SZLNHD, space )

c     calculate x for this analysis location

      call shotx ( Analysis_ShotPoint, sp, x, Model_NumShots, fx )
      
c calculate analysis offset ax as a function of side of [t-d] space
c being used.  This is useful if the dip model is very rugose with
c tight synclines or anticlynes where you want to use a single slope
c for a greater length of time toward the apexes to allow for  maximal
c filtering prior to a colour change......Garossino

      if ( domain .ne. ifix(BoatDirection) ) then
         ax = fx - ( ( dfloat( nchan - 1 ) * GroupInterval ) + Gap ) 
     :        * OffsetFactor
      else
         ax = fx + ( ( dfloat( nchan - 1 ) * GroupInterval ) + Gap ) 
     :        * OffsetFactor
      endif

c calculate t(x), dt, dx, slope along all horizons at this location
c go through all segments of the xsd pickfile looking for horizon
c segments at this location.

      DO 50  xsd_Segment = 1, xsd_NumSegs

c reset the bail out flag for this horizon
 
         bail_out = .false.

c if the difference between the actual function location and the 
c desired maximum offset percentage is small [by experimentation
c the 200 unit works well in practice for both metric and english
c units] then do not bother doing an offset estimate.

         if ( ( dabs( fx - ax) ) .le. 200. ) then

            call SlopeTime ( xsd_NumPoints, xsd_size, xsd_x_coords, 
     :     xsd_horizon_times, xsd_Segment, fx, tx, dt, dx, bail_out,    
     :     verbos )

         else

            call OffsetSlopeTime (xsd_NumPoints, xsd_size, xsd_x_coords, 
     :     xsd_horizon_times, xsd_Segment, fx, ax, tx, dt, dx, bail_out,    
     :     verbos )

         endif

c if no interface,blk exists at fx then bail_out is .true., go to next
c horizon and do not advance Analysis_NumHorizons

         if ( bail_out ) goto 50

c     calculate time, dip, vstk for the multiple/primary pair

         call DipVstack( fx, vel_NumFcns, vel_size, vel_NumPoints, 
     :     vel_times, vel_x_coords, vel_velocities, tx, dx, dt, 
     :     dip, vstk )

c assign horizon time, dip and velocity to arrays

         t( Analysis_NumHorizons ) = tx
         d( Analysis_NumHorizons ) = dip
         v( Analysis_NumHorizons ) = vstk

c set up for tweak dip,velocity determination.  must fill out the 
c constants entries in c() and Var() arrays for Mike Bush subroutines

         c(6) = vstk
         Var(1) = vstk

         c(8) = dip
         Var(2) = dip

         c(13) = tx

c call td to calculate h [perpendicular distance from the horizon] and
c the zero offset time.  These get stored in c(5) and c(7) respectively
c if the perpendicular distance to the horizon is less than the gap
c [stored in c(11)] then bail.

         call td ( c, nchan, bail_out )
         
         if ( bail_out ) goto 50

c routine to force a pick to be red or green
c mult is -ve for green picks as green picks have a primary time less
c that the corresponding multiple; mult is +ve for red picks

c***********************************************************************
C FOR BOAT>,S OR BOAT<, R USE NEGATIVE X
c
c ..............................
c               |
c               |
c    this       |
c   quadrant    |
c               |
c               |
c               |
c               |
c               |
c
c***********************************************************************

         IF ( ( ( c_BoatDirection .eq. 'lr' ) .and. 
     :        ( c_domain .eq. 'shot' ) )
     :        .or.
     :        ( ( c_BoatDirection .eq. 'rl' ) .and. 
     :        ( c_domain .eq. 'receiver' ) ) ) 
     :        then

c calculate vel and dip required for desired tweak at the current analysis
c location and horizon position

c set c(21) --> Mult to GREEN

         c(21) = 1.0

            call dfltpl ( numVar, Var, negif, nepgrd, fcnMin, itype, 
     :           fcnSlim, fcnNumTrials, space, fcnItrial, fcnIerr, c, 
     :           nchan )

c ----- determine if Var(1) is different than vstk -----
c       If so then flash warning to LOT, lower tweak by 25, set tweak
c       change flag and recall dfltpl with new tweak then retest.
c       Iterate until velocity is stable.  Velocity is unstable at
c       shallow horizons using large tweaks or at any horizon where
c       the tweak requested cannot be reached within 90 degrees of dip.

            do while ( abs( dnint( Var(1) ) - dnint(vstk) ) .gt. 0 ) 

               tflag = 1	

c adjust tweak entry in the constants array by 25 milliseconds up or
c down as required
	
               if ( c(20) .gt. 0.d0 ) c(20) = c(20) - dble(0.025)
               if ( c(20) .lt. 0.d0 ) c(20) = c(20) + dble(0.025)

               Var(1) = vstk

               write(lerr,*)' Adjusting',sngl(Tweak),' green tweak to',
     :              c(20), ' for horizon ', xsd_Segment
           
               call dfltpl ( numVar, Var, negif, nepgrd, fcnMin, itype, 
     :              fcnSlim, fcnNumTrials, space, fcnItrial, fcnIerr, c, 
     :              nchan )

            enddo

c if a tweak adjustment has occured reload the original tweak into
c the constants array [c(20)] and set the tweak flag back to zero

            if ( tflag .gt. 0 ) then
               tflag = 0
            endif

            umin = negip( c, var, nchan )

c  unfold Var2 greater than 2*pi radians

            itemp = nint( sngl( Var(2) / pi2 ) )
            Var(2) = Var(2) - dfloat(itemp) * pi2

            if ( ( umin * c(20) ) .gt. 0. ) then
               if ( Var(2) .gt. 0. ) then
                  Var(2) = pi - Var(2)
               else
                  Var(2) = -1.d0 * ( pi + Var(2) )
               endif
               umin = negip( c, var, nchan )
            endif

c     put value less than pi/2

            if ( dabs( Var(2) ) .ge. pid2 .and. Var(2) .gt. 0. ) 
     :           Var(2) = Var(2) - pi
            if ( dabs( Var(2) ) .ge. pid2 .and. Var(2) .lt. 0. ) 
     :           Var(2) = Var(2) + pi
            
            umin2 = negip( c, var, nchan )
         
            if ( dabs( umin - umin2 ) .gt. 10.e-4 ) then
               Var(2) = -1.d0 * Var(2)
               umin = negip( c, var, nchan )
            endif
            
         ELSE

c***********************************************************************
C FOR X> AND BOAT<,S OR BOAT>, R USE POSITIVE X
c
c ..............................
c               |
c               |
c               |    this
c               |  quadrant
c               |
c               |
c               |
c               |
c               |
c***********************************************************************

c set c(21) --> Mult to GREEN

         c(21) = -1.0

            call dfltpl ( numVar, Var, posif, popgrd, fcnMin, itype, 
     :           fcnSlim, fcnNumTrials, space, fcnItrial, fcnIerr, c, 
     :           nchan )

c ----- determine if Var(1) is different than vstk -----
c       If so then flash warning to LOT, lower tweak by 25, set tweak
c       change flag and recall dfltpl with new tweak then retest.
c       Iterate until velocity is stable. Velocity is unstable at
c       shallow horizons using large tweaks or at any horizon where
c       the tweak requested cannot be reached within 90 degrees of dip.

            do while ( abs( dnint( Var(1) ) - dnint(vstk) ) .gt. 0 ) 

               tflag = 1		
               
c adjust tweak entry in the constants array by 25 milliseconds up or
c down as required

               if ( c(20) .gt. 0.d0 ) c(20) = c(20) - dble(0.025)
               if ( c(20) .lt. 0.d0 ) c(20) = c(20) + dble(0.025)
               Var(1) = vstk
               
               write(LERR,*)' Adjusting',sngl(Tweak),' green tweak to',
     :              c(20), ' for horizon ', xsd_Segment
        
               call dfltpl ( numVar, Var, posif, popgrd, fcnMin, itype, 
     :              fcnSlim, fcnNumTrials, space, fcnItrial, fcnIerr, c, 
     :              nchan )

            enddo        

c if a tweak adjustment has occured reload the original tweak into
c the constants array [c(20)] and set the tweak flag back to zero

            if ( tflag .gt. 0 ) then
               c(20) = Tweak
               tflag = 0
            endif
    
            umin = posip( c, var, nchan )

c unfold Var2 greater than 2*pi radians

            itemp = nint( sngl( Var(2) / pi2 ) )
            Var(2) = Var(2) - dfloat(itemp) * pi2

            if ( ( umin * c(20) ) .gt. 0. ) then 

               if ( Var(2) .gt. 0. ) then
                  Var(2) = pi - Var(2)
               else
                  Var(2) = -1.d0 * ( pi + Var(2) )
               endif

               umin = posip( c, var, nchan ) 

            endif

c put value less than pi/2

            if ( dabs( Var(2) ) .ge. pid2 .and. Var(2) .gt. 0. )
     :           Var(2) = Var(2) - pi
            if ( dabs( Var(2) ) .ge. pid2 .and. Var(2) .lt. 0. )
     :           Var(2) = Var(2) + pi
            umin2 = posip( c, Var, nchan )

            if ( dabs( umin - umin2 ) .gt. 10.e-4 ) then
               Var(2) = -1.d0 * Var(2)
               umin = posip( c, Var, nchan )
            endif

         ENDIF

c change Var(2) from radians to degrees

         Var(2) = Var(2) * 180.d0 / pi

c assign values to green dip, velocity and color arrays

         dg(Analysis_NumHorizons) = Var(2)
         vg(Analysis_NumHorizons) = Var(1)
         cg(Analysis_NumHorizons) = 'g'
         tg(Analysis_NumHorizons) = c(13)


         uming(Analysis_NumHorizons) = umin
         vpg(Analysis_NumHorizons) = vstk

c assign dip in degrees

         dpg(Analysis_NumHorizons) = dip * 180.d0 / pi

c finished with green tweak
c reset Var() entries in preparation for red tweak calculation

         Var(1) = c(6)
         Var(2) = c(8)

C CALCULATE RED PARAMETERS

c***********************************************************************
C FOR BOAT>,S OR BOTH, OR BOAT<, R OR BOTH USE NEGATIVE X
c
c ..............................
c               |
c               |
c    this       |
c   quadrant    |
c               |
c               |
c               |
c               |
c               |
c
c***********************************************************************

      IF ( ( ( c_BoatDirection .eq. 'lr' ) .and. 
     :        ( c_domain .eq. 'shot' ) )
     :        .or.
     :        ( ( c_BoatDirection .eq. 'rl' ) .and. 
     :        ( c_domain .eq. 'receiver' ) ) ) 
     :        then

c set c(21) --> Mult to RED

         c(21) = -1.0

         call dfltpl ( numVar, Var, negif, nepgrd, fcnMin, itype, 
     :        fcnSlim, fcnNumTrials, space, fcnItrial, fcnIerr, c, 
     :        nchan )

c ----- determine if Var(1) is different than vstk -----
c       If so then flash warning to LOT, lower tweak by 25, set tweak
c       change flag and recall dfltpl with new tweak then retest.
c       Iterate until velocity is stable. Velocity is unstable at
c       shallow horizons using large tweaks or at any horizon where
c       the tweak requested cannot be reached within 90 degrees of dip.

         do while ( abs( dnint( Var(1) ) - dnint( VSTK ) ) .gt. 0 )
         
            tflag = 1		 
               
c adjust tweak entry in the constants array by 25 milliseconds up or
c down as required

            if ( c(20) .gt. 0.d0 ) c(20) = c(20) - dble(0.025)
            if ( c(20) .lt. 0.d0 ) c(20) = c(20) + dble(0.025)
            Var(1) = vstk

            write(LERR,*)' Adjusting',sngl(Tweak),' red tweak to',
     :           c(20), ' for horizon ', xsd_Segment
        
            call dfltpl ( numVar, Var, negif, nepgrd, fcnMin, itype, 
     :           fcnSlim, fcnNumTrials, space, fcnItrial, fcnIerr, c, 
     :           nchan )
         enddo

c if a tweak adjustment has occured reload the original tweak into
c the constants array [c(20)] and set the tweak flag back to zero

        if ( tflag .gt. 0 ) then
           c(20) = Tweak
            tflag = 0
         endif

         umin = negip( c, Var, nchan )

c unfold Var2 greater than 2*pi radians

         itemp = nint( sngl( Var(2) / pi2 ) )
         Var(2) = Var(2) - dfloat(itemp) * pi2

         if ( ( umin * c(20) ) .lt. 0. ) then

            if ( Var(2) .gt. 0. ) then
               Var(2) = pi - Var(2)
            else
               Var(2) = -1.d0 * ( pi + Var(2) )
            endif

            umin = negip( c, Var, nchan )

         endif

C put value less than pi/2

         if ( dabs( Var(2) ) .ge. pid2 .and. Var(2) .gt. 0. ) 
     :        Var(2) = Var(2) - pi
         if ( dabs( Var(2) ) .ge. pid2 .and. Var(2) .lt. 0. ) 
     :        Var(2) = Var(2) + pi

         umin2 = negip(c,Var,nchan)

         if ( dabs( umin - umin2 ) .gt. 10.e-4 ) then
            Var(2) = -1.d0 *Var(2)
            umin = negip( c, Var, nchan )
         endif

      ELSE

c***********************************************************************
C FOR X> AND BOAT<,S OR BOAT>, R USE POSITIVE X
c
c ..............................
c               |
c               |
c               |    this
c               |  quadrant
c               |
c               |
c               |
c               |
c               |
c***********************************************************************

c set c(21) --> Mult for RED
         
         c(21) = 1.d0

         call dfltpl ( numVar, Var, posif, popgrd, fcnMin, itype, 
     :        fcnSlim, fcnNumTrials, space, fcnItrial, fcnIerr, c, 
     :        nchan )

c ----- determine if Var(1) is different than vstk -----
c       If so then flash warning to LOT, lower tweak by 25, set tweak
c       change flag and recall dfltpl with new tweak then retest.
c       Iterate until velocity is stable. Velocity is unstable at
c       shallow horizons using large tweaks or at any horizon where
c       the tweak requested cannot be reached within 90 degrees of dip.

         do  while ( abs( dnint( Var(1) ) - dnint(VSTK) ) .gt. 0 )

            tflag = 1		
               
c adjust tweak entry in the constants array by 25 milliseconds up or
c down as required

            if ( C(20) .gt. 0.d0 ) C(20) = C(20) - dble(0.025)
            if ( C(20) .lt. 0.d0 ) C(20) = C(20) + dble(0.025)
            Var(1) = vstk

            write(LERR,*)' Adjusting',sngl(Tweak),' red tweak to',
     :           c(20), ' for horizon ', xsd_Segment
            
            call dfltpl ( numVar, Var, posif, popgrd, fcnMin, itype, 
     :           fcnSlim, fcnNumTrials, space, fcnItrial, fcnIerr, c, 
     :           nchan )
         enddo

c if a tweak adjustment has occured reload the original tweak into
c the constants array [c(20)] and set the tweak flag back to zero

         if ( tflag .gt. 0 ) then
            c(20) = Tweak
            tflag = 0
         endif

         umin = posip( c, Var, nchan )
      
c unfold var2 greater than 2*pi radians

         itemp =  nint( sngl( Var(2) / pi2 ) )
         Var(2) = Var(2) - dfloat(itemp) * pi2

         if ( ( umin * c(20) ) .lt. 0.d0 ) then

            if ( Var(2) .gt. 0.d0 ) then
               Var(2) = pi - Var(2)
            else
               Var(2) = -1.d0 * ( pi + Var(2) )
            endif

            umin = posip( c, Var, nchan )

         endif

c     put value less than pi/2

         if ( dabs ( Var(2) ) .ge. pid2 .and. Var(2) .gt. 0.d0 ) 
     :        Var(2) = Var(2) - pi
         if ( dabs ( Var(2) ) .ge. pid2 .and. Var(2) .lt. 0.d0 ) 
     :        Var(2) = Var(2) + pi

         umin2 = posip( c, Var, nchan )

         if ( dabs ( umin - umin2 ) .gt. 10.e-4 ) then
            Var(2) = -1.d0 * Var(2) 
            umin = posip( c, Var, nchan )
         endif

      ENDIF

C change Var(2),dr(Analysis_NumHorizons) to degrees

      Var(2) = Var(2) * 180.d0 / pi
      dr(Analysis_NumHorizons) = Var(2)
      vr(Analysis_NumHorizons) = Var(1)
      cr(Analysis_NumHorizons) = 'r'
      tr(Analysis_NumHorizons) = c(13)

      uminr(Analysis_NumHorizons) = umin
      vpr(Analysis_NumHorizons) = vstk
      dpr(Analysis_NumHorizons) = dip * 180.d0 / pi

c one horizon finished, advance horizon counter and get another

 49   Analysis_NumHorizons = Analysis_NumHorizons + 1

 50   CONTINUE

c determine actual number of horizons present by counting the number
c of consecutive nonzero vg[] entries

      Done = .false.
      count = 0
      Analysis_NumHorizons = 0

      do while ( .not. Done )

         count = count + 1

         if ( vg(count) .le. 0.d0 ) Done = .true.

      enddo

      Analysis_NumHorizons = count - 1

c end of all horizon calculations, continue processing with completed
c function for this analysis location

c Analysis_NumHorizons is required to be Analysis_NumHorizons+1 for the 
c first call to minmax so that
c the n=n-1 imbedded in the routine does not delete the Analysis_NumHorizons function
c from the analysis.  this is necessary since minmax adds 1 to Analysis_NumHorizons
c upon return and must be subtracted for additional calls to minmax
c
c I do not know what all this drivel is about but will sortit out
c at run time debug and see if I can clean this up

c the following if logic stops the program from output if no horizons
c have been found.  if not done the minmax routine will manufacture
c a tvd function for zero and 5.0 seconds and output it

      if ( Analysis_NumHorizons .ge. 2 ) then

c sort for out of sequence times

         call tvd_sort ( Analysis_NumHorizons, tr, dr, vr, uminr, dpr, 
     :        vpr, cr )
         call tvd_sort ( Analysis_NumHorizons, tg, dg, vg, uming, dpg, 
     :        vpg, cg )

c red minmax call

         analysis_numhorizons = analysis_numhorizons + 1

         call minmax ( tr, dr, vr, cr, uminr, Analysis_NumHorizons, dpr, 
     :        vpr, v0, Data_MaxTime )

c green minmax call

         analysis_numhorizons = analysis_numhorizons - 1

         call minmax ( tg, dg, vg, cg, uming, Analysis_NumHorizons, dpg,
     :        vpg, v0, Data_MaxTime )

c at this point one analysis location has been finished and it is now
c necessary to output the calculated functions to the appropriate files

c green

         call output ( tg, dg, vg, cg, Analysis_NumHorizons, domain, 
     :        Analysis_Shotpoint, line, c, x, nchan, BoatDirection, 
     :        uming, dpg, lu_sred, lu_rred, lu_sgreen, 
     :        lu_rgreen, lu_truedip  )


c red

         call output ( tr, dr, vr, cr, Analysis_NumHorizons, domain, 
     :        Analysis_Shotpoint, line, c, x, nchan, BoatDirection, 
     :        uminr, dpr, lu_sred, lu_rred, lu_sgreen, 
     :        lu_rgreen, lu_truedip   )


c advance shot point
c

      endif 
      
      if ( ( x(Model_NumShots) / dfloat( sp(Model_NumShots) ) )
     :     .lt. ( x(Model_NumShots) / dfloat(sp(1)) ) ) then

         Analysis_ShotPoint = Analysis_ShotPoint + Model_ShotIncrement

         if ( ( Analysis_ShotPoint .gt. Analysis_ShotpointEnd ) .or. 
     :        ( Analysis_ShotPoint .gt. sp(Model_NumShots) ) ) then
            write(lerr,*) ' '
            write(lerr,*) ' End of model reached'
            write(lerr,*) ' '
            goto 999
         endif
      endif
      
c go to next analysis location

      GOTO 45

c Error Messages

 990  continue
      write(LERR,*)' '
      write(LERR,*)' Error opening velocity model input file '
      write(LERR,*)' ',vfile(1:length)
      write(LERR,*)' check permissions/existance and resubmit'
      write(LERR,*)'FATAL'
      write(LERR,*)' '
      write(LER,*)' '
      write(LER,*)'TVD'
      write(LER,*)' Error opening velocity model input file '
      write(LER,*)' ',vfile(1:length)
      write(LER,*)' check permissions/existance and resubmit'
      write(LER,*)'FATAL'
      write(LER,*)' '
      write(LERR,*) 'Abnormal Termination'
      write(LER,*) 'tvd: abnormal termination'
      stop
     
 991  continue
      write(LERR,*)' '
      write(LERR,*)' Error opening xsd structural model input file '
      write(LERR,*)' ',mfile(1:length)
      write(LERR,*)' check permissions/existance and resubmit'
      write(LERR,*)'FATAL'
      write(LERR,*)' '
      write(LER,*)' '
      write(LER,*)'TVD'
      write(LER,*)' Error opening xsd structural model input file '
      write(LER,*)' ',mfile(1:length)
      write(LER,*)' check permissions/existance and resubmit'
      write(LER,*)'FATAL'
      write(LER,*)' '
      write(LERR,*) 'Abnormal Termination'
      write(LER,*) 'tvd: abnormal termination'
      stop
     
 992  continue
      write(LERR,*)' '
      write(LERR,*)' Error opening shot red tvd output file '
      write(LERR,*)' ',sred(1:length)
      write(LERR,*)' check permissions and resubmit'
      write(LERR,*)'FATAL'
      write(LERR,*)' '
      write(LER,*)' '
      write(LER,*)'TVD'
      write(LER,*)' Error opening shot red tvd output file '
      write(LER,*)' ',sred(1:length)
      write(LER,*)' check permissions and resubmit'
      write(LER,*)'FATAL'
      write(LER,*)' '
      write(LERR,*) 'Abnormal Termination'
      write(LER,*) 'tvd: abnormal termination'
      stop
     
 993  continue
      write(LERR,*)' '
      write(LERR,*)' Error opening shot green tvd output file '
      write(LERR,*)' ',sgreen(1:length)
      write(LERR,*)' check permissions and resubmit'
      write(LERR,*)'FATAL'
      write(LERR,*)' '
      write(LER,*)' '
      write(LER,*)'TVD'
      write(LER,*)' Error opening shot green tvd output file '
      write(LER,*)' ',sgreen(1:length)
      write(LER,*)' check permissions and resubmit'
      write(LER,*)'FATAL'
      write(LER,*)' '
      write(LERR,*) 'Abnormal Termination'
      write(LER,*) 'tvd: abnormal termination'
      stop
     
 994  continue
      write(LERR,*)' '
      write(LERR,*)' Error opening receiver red tvd output file '
      write(LERR,*)' ',rred(1:length)
      write(LERR,*)' check permissions and resubmit'
      write(LERR,*)'FATAL'
      write(LERR,*)' '
      write(LER,*)' '
      write(LER,*)'TVD'
      write(LER,*)' Error opening receiver red tvd output file '
      write(LER,*)' ',rred(1:length)
      write(LER,*)' check permissions and resubmit'
      write(LER,*)'FATAL'
      write(LER,*)' '
      write(LERR,*) 'Abnormal Termination'
      write(LER,*) 'tvd: abnormal termination'
      stop
     
 995  continue
      write(LERR,*)' '
      write(LERR,*)' Error opening receiver green tvd output file '
      write(LERR,*)' ',rgreen(1:length)
      write(LERR,*)' check permissions and resubmit'
      write(LERR,*)'FATAL'
      write(LERR,*)' '
      write(LER,*)' '
      write(LER,*)'TVD'
      write(LER,*)' Error opening receiver green tvd output file '
      write(LER,*)' ',rgreen(1:length)
      write(LER,*)' check permissions and resubmit'
      write(LER,*)'FATAL'
      write(LER,*)' '
      write(LERR,*) 'Abnormal Termination'
      write(LER,*) 'tvd: abnormal termination'
      stop
     
 996  continue
      write(LERR,*)' '
      write(LERR,*)' Error opening truedip tvd output file '
      write(LERR,*)' ',truedip(1:length)
      write(LERR,*)' check permissions and resubmit'
      write(LERR,*)'FATAL'
      write(LERR,*)' '
      write(LER,*)' '
      write(LER,*)'TVD'
      write(LER,*)' Error opening truedip tvd output file '
      write(LER,*)' ',truedip(1:length)
      write(LER,*)' check permissions and resubmit'
      write(LER,*)'FATAL'
      write(LER,*)' '
      write(LERR,*) 'Abnormal Termination'
      write(LER,*) 'tvd: abnormal termination'
      stop
     
 999  continue
      write(LERR,*) 'Normal Termination'
      write(LER,*) 'tvd: normal termination'
      STOP
      END
